/* Subroutines for insn-output.c for Texas Instruments MSP430 MCU
   Copyright (C) 2001, 2002 Free Software Foundation, Inc.
   Contributed by Dmitry Diky <diwil@mail.ru>
   
   This file is part of GNU CC. 
   GNU CC is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   GNU CC is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with GNU CC; see the file COPYING.  If not, write to
   the Free Software Foundation, 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */


#include "config.h"
#include "system.h"
#include "rtl.h"
#include "regs.h"
#include "hard-reg-set.h"
#include "real.h"
#include "insn-config.h"
#include "conditions.h"
#include "output.h"
#include "insn-attr.h"
#include "flags.h"
#include "reload.h"
#include "tree.h"
#include "expr.h"
#include "toplev.h"
#include "obstack.h"
#include "function.h"
#include "recog.h"
#include "tm_p.h"


/* This holds the last insn address.  */
static int last_insn_address = 0;

/* Commands count in the compiled file */
static int commands_in_file;

/* Commands in the functions prologues in the compiled file */
static int commands_in_prologues;

/* Commands in the functions epilogues in the compiled file */
static int commands_in_epilogues;

/* Prologue/Epilogue size in words */
static int prologue_size;
static int epilogue_size;

/* Size of all jump tables in the current function, in words.  */
static int jump_tables_size;


static int addenum_size = 0;
static int msp430_current_frame_offset = 0;

void msp430_instantinate_condition PARAMS ((rtx, rtx *, RTX_CODE));

int tstsi_needs_scratch_p PARAMS ((rtx, rtx *));
int tstsi_needs_half_scratch_p PARAMS ((rtx, rtx *));

void msp430_prepare_tstsi PARAMS ((rtx, rtx *));
void msp430_prepare_tsthi PARAMS ((rtx, rtx *));
void msp430_prepare_tstqi PARAMS ((rtx, rtx *));

int cmpsi_needs_scratch_p PARAMS ((rtx, rtx *));

void msp430_prepare_cmpsi PARAMS ((rtx, rtx *));
void msp430_prepare_cmphi PARAMS ((rtx, rtx *));
void msp430_prepare_cmpqi PARAMS ((rtx, rtx *));

int self_push PARAMS ((rtx));


/* shift helpers */
static const char * msp430_emit_ashlqi3_call PARAMS ((rtx, rtx *, int *));
static const char * msp430_emit_ashlhi3_call PARAMS ((rtx, rtx *, int *));
static const char * msp430_emit_ashlsi3_call PARAMS ((rtx, rtx *, int *));
static const char * msp430_emit_ashldi3_call PARAMS ((rtx, rtx *, int *));



static int hwmul_no_int_function_p PARAMS ((tree func));
static int interrupt_function_p PARAMS ((tree func));
static int msp430_naked_function_p  PARAMS ((tree func));
static int signal_function_p  PARAMS ((tree func));
static int wakeup_function_p  PARAMS ((tree func));
static int msp430_num_arg_regs PARAMS (( enum machine_mode mode, tree type ));
static int msp430_critical_function_p PARAMS ((tree func));

const char *msp430_init_stack="__stack";
const char *msp430_endup="__stop_progExec__";

int msp430_case_values_threshold = 30000;
int msp430_has_hwmul = 0;

const char *msp430_mcu_name = "msp430x110";

enum msp430_arch {
  MSP430_ISA_1    = 1,
  MSP430_ISA_2    = 2,
  MSP430_ISA_110   = 110,
  MSP430_ISA_11   = 11,
  MSP430_ISA_12   = 12,
  MSP430_ISA_13   = 13,
  MSP430_ISA_14   = 14,
  MSP430_ISA_41   = 41,
  MSP430_ISA_31   = 31,
  MSP430_ISA_32   = 32,
  MSP430_ISA_33   = 33,
  MSP430_ISA_43   = 43,
  MSP430_ISA_44   = 44
};

struct mcu_type_s
  {
    const char *name;
    enum msp430_arch arch;
    int has_hwmul;
  };

/* recently added:
1331, 1351, 435, 436, 437, 447, 448, 449 */

static struct mcu_type_s msp430_mcu_types[] =
    {
      {"msp1",	    MSP430_ISA_1,  0
      },
      {"msp2",	    MSP430_ISA_2,  1},
      {"msp430x1101",	    MSP430_ISA_110,0},
      {"msp430x1111",     MSP430_ISA_110,0},
      {"msp430x1121",     MSP430_ISA_110,0},
      {"msp430x1122",     MSP430_ISA_110,0},
      {"msp430x1132",     MSP430_ISA_110,0},


      {"msp430x110",      MSP430_ISA_11, 0},
      {"msp430x112",      MSP430_ISA_11, 0},

      {"msp430x122",      MSP430_ISA_12, 0},
      {"msp430x1222",     MSP430_ISA_12, 0},
      {"msp430x123",      MSP430_ISA_12, 0},
      {"msp430x1232",     MSP430_ISA_12, 0},

      {"msp430x133",      MSP430_ISA_13, 0},
      {"msp430x1331",	    MSP430_ISA_13, 0},
      {"msp430x135",      MSP430_ISA_13, 0},
      {"msp430x1351",     MSP430_ISA_13, 0},

      {"msp430x147",      MSP430_ISA_14, 1},
      {"msp430x148",      MSP430_ISA_14, 1},
      {"msp430x149",      MSP430_ISA_14, 1},

      {"msp430x412",      MSP430_ISA_41, 0},
      {"msp430x413",      MSP430_ISA_41, 0},

      {"msp430x311",      MSP430_ISA_31, 0},
      {"msp430x312",      MSP430_ISA_31, 0},
      {"msp430x313",      MSP430_ISA_31, 0},
      {"msp430x314",      MSP430_ISA_31, 0},
      {"msp430x315",      MSP430_ISA_31, 0},

      {"msp430x323",      MSP430_ISA_32, 0},
      {"msp430x325",      MSP430_ISA_32, 0},

      {"msp430x336",      MSP430_ISA_33, 1},
      {"msp430x337",      MSP430_ISA_33, 1},

      {"msp430x435",      MSP430_ISA_43, 0},
      {"msp430x436",      MSP430_ISA_43, 0},
      {"msp430x437",      MSP430_ISA_43, 0},

      {"msp430x447",      MSP430_ISA_44, 1},
      {"msp430x448",      MSP430_ISA_44, 1},
      {"msp430x449",      MSP430_ISA_44, 1},


      {NULL,0,0}
    };


void
msp430_override_options ()
{
  const struct mcu_type_s *t;

  for (t = msp430_mcu_types; t->name; t++)
    {
      if (strcmp (t->name, msp430_mcu_name) == 0)
        break;
    }

  if (!t->name)
    {
      error ("MCU %s not supported", msp430_mcu_name);
      fprintf (stderr, "Known MCU names:\n");
      for (t = msp430_mcu_types; t->name; t++)
        fprintf (stderr,"   %s\n", t->name);
      abort();
      return;
    }

  msp430_has_hwmul = t->has_hwmul || TARGET_HW_MUL ;

  if(TARGET_NO_HW_MUL) msp430_has_hwmul = 0;

  msp430_case_values_threshold = 8;	/* ? or there is a better value ? */

}

void
msp430_init_once()
{
  return;
}

static int reg_class_tab[16]={
  PC_REG,STACK_REGS,CG_REGS,CG_REGS,
  GENERAL_REGS,GENERAL_REGS,GENERAL_REGS,GENERAL_REGS,GENERAL_REGS,
  GENERAL_REGS,GENERAL_REGS,GENERAL_REGS,GENERAL_REGS,GENERAL_REGS,
  GENERAL_REGS,GENERAL_REGS /* r0 - r15 */
};


int
msp430_regno_ok_for_base_p(r)
int r;
{

  if(r == 2) return 0;
  if(r == 3) return 0;
  if(r < FIRST_PSEUDO_REGISTER && r > 0 ) return 1;
  if( reg_renumber
      && reg_renumber[r] < FIRST_PSEUDO_REGISTER
      && reg_renumber[r] > 0
      && reg_renumber[r] != 2
      && reg_renumber[r] != 3 ) return 1;

  return 0;

}

enum reg_class
msp430_regno_reg_class (r)
int r;
{
  if (r < FIRST_PSEUDO_REGISTER) return reg_class_tab[r];

  return NO_REGS;
}


enum reg_class
msp430_reg_class_from_letter  (c)
int c;
{
  switch (c)
    {
    case 'd' :
      return SP_REG;
    default  :
      break;
    }

  return NO_REGS;
}


/* Return non-zero if FUNC is a naked function.  */
static int
msp430_naked_function_p(func)
tree func;
{
  tree a;

  if (TREE_CODE (func) != FUNCTION_DECL) abort ();

  a = lookup_attribute ("naked", DECL_MACHINE_ATTRIBUTES (func));
  return a != NULL_TREE;
}


/* Return nonzero if FUNC is an interrupt function as specified
   by the "interrupt" attribute.  */

static int
interrupt_function_p(func)
tree func;
{
  tree a;

  if (TREE_CODE (func) != FUNCTION_DECL) abort ();

  a = lookup_attribute ("interrupt", DECL_MACHINE_ATTRIBUTES (func));
  return a != NULL_TREE;
}

static int
msp430_critical_function_p(func)
tree func;
{
  tree a;
  if (TREE_CODE (func) != FUNCTION_DECL) abort ();
  a = lookup_attribute ("critical", DECL_MACHINE_ATTRIBUTES (func));
  return a != NULL_TREE;

}

/* Return nonzero if FUNC is a function with the MSP430_HWMUL_NO_INT
   attribute.  */

static int
hwmul_no_int_function_p(func)
tree func;
{
  tree a;

  if (TREE_CODE (func) != FUNCTION_DECL) abort ();

  a = lookup_attribute ("hwmul_no_int", DECL_MACHINE_ATTRIBUTES (func));
  return a != NULL_TREE;
}

/* Return nonzero if current function should be a hwmul-no-int function.
   Takes into account target_flags and current function attributes */
int msp430_current_function_hwmul_no_int_function_p(void)
{
  int rval;

  if(!current_function_decl)
    return(0);

  rval = hwmul_no_int_function_p(current_function_decl);

  return(TARGET_HWMUL_NO_INT || rval);
}


/* Return nonzero if FUNC is a signal function as specified
   by the "signal" attribute.  */

static int
signal_function_p(func)
tree func;
{
  tree a;

  if (TREE_CODE (func) != FUNCTION_DECL) abort ();

  a = lookup_attribute ("signal", DECL_MACHINE_ATTRIBUTES (func));
  return a != NULL_TREE;
}


/* Return nonzero if FUNC is a  wakeup function as specified
   by the "wakeup" attribute.  */

static int
wakeup_function_p(func)
tree func;
{
  tree a;

  if (TREE_CODE (func) != FUNCTION_DECL) abort ();

  a = lookup_attribute ("wakeup", DECL_MACHINE_ATTRIBUTES (func));
  return a != NULL_TREE;
}


#define NOVECTOR	0xff

void
asm_declare_function_name(file, name, decl)
FILE *file;
char *name;
tree decl ATTRIBUTE_UNUSED;
{
  int interrupt_func_p;
  tree ss = 0;
  int vector = -1;
  int cfp = msp430_critical_function_p(current_function_decl);

  interrupt_func_p = interrupt_function_p (current_function_decl);

  if(interrupt_func_p)
    {
      /*
       * .global This_func1
       * .set vector11, This_func1
       * .type   This_func1,@function
       *
       */
      ss = lookup_attribute ("interrupt", DECL_MACHINE_ATTRIBUTES (current_function_decl));
      ss = TREE_VALUE (ss);
      if(ss)
        {
          ss = TREE_VALUE (ss);
          if(ss) vector = TREE_INT_CST_LOW(ss) ;

          if(vector != NOVECTOR) vector +=  0xffe0ul;

        }

      if(vector == -1)
      {
      	warning("No valid interrupt vector assigned to ISR `%s'.", name);
      }
      
      if((vector < 0xffe0 || vector > 0xfffe || (vector&1))
      	 && ( vector != NOVECTOR && vector != -1))
      {
      	warning("Interrupt vector 0x%x assigned to ISR `%s' does make sense.",
      			vector, name);
      }
      
      if(vector != NOVECTOR)
        {
          fprintf (file, ".global	vector_%04x\n",vector);
        }
      fprintf (file, "%s", TYPE_ASM_OP);
      assemble_name (file, name);
      putc (',', file);
      fprintf (file, TYPE_OPERAND_FMT, "function");
      putc ('\n', file);
      fprintf (file, "/***********************\n");
      fprintf (file, " * Interrupt %sRoutine `",
               (vector != NOVECTOR)?"Service ":"Sub-");
      assemble_name (file, name);
      fprintf (file, "' at 0x%04x\n", vector);
      fprintf (file, " ***********************/\n");

      if(vector != NOVECTOR)
        {
          fprintf (file, "vector_%04x:\n",vector);
        }

      ASM_OUTPUT_LABEL (file, name);
    }
  else
    {
      fprintf (file, "%s", TYPE_ASM_OP);
      assemble_name (file, name);
      putc (',', file);
      fprintf (file, TYPE_OPERAND_FMT, "function");
      putc ('\n', file);
      fprintf (file, "/***********************\n");
      fprintf (file, " * Function `");
      assemble_name (file, name);
      fprintf (file, "' %s\n ***********************/\n",
               cfp? "(OS critical)":"");
      ASM_OUTPUT_LABEL (file, name);
    }
}


static int
msp430_saved_regs_frame(void)
{
      int interrupt_func_p = interrupt_function_p (current_function_decl);
      int cfp = msp430_critical_function_p(current_function_decl);
      int leaf_func_p = leaf_function_p ();
      int offset = interrupt_func_p ? 0 : (cfp?2:0);
      int reg;
      
      for (reg = 4; reg < 16; ++reg)
        {
          if ( (!leaf_func_p && call_used_regs[reg] && (interrupt_func_p))
               || (regs_ever_live[reg] 
                   && (!call_used_regs[reg] || interrupt_func_p)))
            {
              offset += 2;
            }
        }
        
      return offset;
}

int
msp430_immediate_return()
{
	int ifp = interrupt_function_p (current_function_decl);
	int nfp = msp430_naked_function_p(current_function_decl);
	
	if(nfp && !ifp)
		return 1;
	else if(nfp && ifp)
		return 2;
	else if(msp430_current_frame_offset) 
		return 0;
	else if(ifp)
		return 2;
	else
		return 1;
	
	return 0;
}

/* Output function prologue */
void
function_prologue (file, size)
FILE *file;
int size;
{
  int i;
  int interrupt_func_p;
  int signal_func_p;
  int leaf_func_p;
  int main_p;
  int stack_reserve = 0;
  tree ss = 0;
  rtx x;
  const char *fnname;
  int offset;
  int cfp = msp430_critical_function_p(current_function_decl);


  x = DECL_RTL (current_function_decl);
  fnname = XSTR (XEXP(x,0) , 0);

  if(msp430_naked_function_p(current_function_decl))
    {

      fprintf (file, "/* prologue: naked */\n");
      fprintf (file, ".L__FrameSize_%s=0x%x\n",fnname, size);
      return;
    }

  interrupt_func_p = interrupt_function_p (current_function_decl);

  signal_func_p = signal_function_p (current_function_decl);
  leaf_func_p = leaf_function_p ();
  main_p = MAIN_NAME_P (DECL_NAME (current_function_decl));

  ss = lookup_attribute ("reserve", DECL_MACHINE_ATTRIBUTES (current_function_decl));
  if(ss)
    {
      ss = TREE_VALUE (ss);
      if(ss)
        {
          ss = TREE_VALUE (ss);
          if(ss) stack_reserve = TREE_INT_CST_LOW(ss);
          stack_reserve++;
          stack_reserve &= ~1;
        }
    }


  last_insn_address = 0;
  jump_tables_size = 0;
  prologue_size = 0;


  fprintf (file, "/* prologue: frame size = %d; addenum %d; alloca:%d, varargs:%d , fpr:%d*/\n",
           size,
           addenum_size,
           current_function_calls_alloca,
           current_function_varargs,
           frame_pointer_required_p());
  fprintf (file, ".L__FrameSize_%s=0x%x\n",fnname, size);


  offset = initial_elimination_offset(0,0)
           - 2
           + ((!interrupt_func_p
               && cfun->arg_offset_rtx
               && INTVAL(cfun->arg_offset_rtx) ) ?  addenum_size : 0);

  msp430_current_frame_offset = offset;
  
  fprintf (file, ".L__FrameOffset_%s=0x%x\n",fnname, offset);

  if (signal_func_p && interrupt_func_p)
    {
      prologue_size += 1;
      fprintf (file,"\teint\t; enable nested interrupt\n");
    }

  /* assume, that interrupt function do not accept params. */
  size += (!interrupt_func_p
           && cfun->arg_offset_rtx
           && INTVAL(cfun->arg_offset_rtx) ) ?  addenum_size : 0;

  if (main_p)
    {
      if(TARGET_NSI)
        {
          if(size || stack_reserve) fprintf (file,"\tsub\t#%d, r1\t", size+stack_reserve);
          if(frame_pointer_needed)
            {
              fprintf (file,"\tmov\tr1,r%d\n",FRAME_POINTER_REGNUM);
              prologue_size +=1;
            }

          if(size) prologue_size += 2;
          if(size==1 || size==2 || size == 4 || size == 8) prologue_size--;
        }
      else
        {
          fprintf (file,"\tmov\t#(%s-%d), r1\n", msp430_init_stack, size+stack_reserve);

          if(frame_pointer_needed)
            {
              fprintf (file,"\tmov\tr1,r%d\n",FRAME_POINTER_REGNUM);
              prologue_size +=1;
            }
          prologue_size += 2;
        }
    }
  else
    {

      for(i = 4; i<16; i++)
        {
          if((regs_ever_live[i]
              && (!call_used_regs[i]
                  || interrupt_func_p))
              ||
              (!leaf_func_p && (call_used_regs[i]
                                && (interrupt_func_p)))

            )
            {
              fprintf(file,"\tpush\tr%d\n", i);
              prologue_size += 1;
            }
        }

      if(!interrupt_func_p && cfp)
        {
          prologue_size += 3;
          fprintf (file,"\tpush\tr2\n");
          fprintf (file,"\tdint\n");
          if(!size)fprintf (file,"\tnop\n");
        }

      if(size)
        {
          /* The next is a hack... I do not undestand why, but if there
             ARG_POINTER_REGNUM and FRAME/STACK are different, 
             the compiler fails to compute corresponding
             displacement */
          if(!optimize && !optimize_size && regs_ever_live[ARG_POINTER_REGNUM])
            {
              int o = initial_elimination_offset(0,0) - size;
              fprintf (file,"\tmov\tr1, r%d\n", ARG_POINTER_REGNUM);
              fprintf (file,"\tadd\t#%d, r%d\n", o, ARG_POINTER_REGNUM);
              prologue_size += 2;
              if(o!=0 && o!=1 && o!=2 && o!=4 && o!=8) prologue_size += 1;

            }

          /* adjust frame ptr... */
          fprintf (file,"\tsub\t#%d, r1\t;	%d, fpn %d\n", (size+1)&~1, size, frame_pointer_needed );
          if(frame_pointer_needed)
            {
              fprintf (file,"\tmov\tr1,r%d\n",FRAME_POINTER_REGNUM);
              prologue_size += 1;
            }

          if(size==1 || size==2 || size==4 ||size==8)
            prologue_size += 1;
          else
            prologue_size += 2;
        }
    }


  fprintf (file, "/* prologue end (size=%d) */\n", prologue_size);

}


/* Output function epilogue */

void
function_epilogue (file, size)
FILE *file;
int size;
{
  int i;
  int interrupt_func_p;
  int leaf_func_p;
  int main_p;
  int wakeup_func_p;
  int function_size;
  int cfp = msp430_critical_function_p(current_function_decl);
  int still_return = 1;


  if(msp430_naked_function_p(current_function_decl))
    {
      fprintf (file, "/* epilogue: naked */\n");
      return;
    }

  fprintf (file, "/* epilogue: frame size=%d */\n", size);



  wakeup_func_p = wakeup_function_p (current_function_decl);
  interrupt_func_p = interrupt_function_p (current_function_decl);
  leaf_func_p = leaf_function_p ();
  main_p = MAIN_NAME_P (DECL_NAME (current_function_decl));
  function_size = (INSN_ADDRESSES (INSN_UID (get_last_insn ()))
                   - INSN_ADDRESSES (INSN_UID (get_insns ())));

  last_insn_address = 0;
  jump_tables_size = 0;
  epilogue_size = 0;

  size += (!interrupt_func_p
           && cfun->arg_offset_rtx
           && INTVAL(cfun->arg_offset_rtx) ) ?  addenum_size : 0;

  if (main_p)
    {
      if(size) fprintf (file,"\tadd\t#%d, r1\n",(size+1)&~1);
      epilogue_size += 2;
      if(size==1 || size==2 || size == 4 || size == 8) epilogue_size--;
      fprintf (file,  "\tbr\t#%s\n", msp430_endup);
      epilogue_size += 2;
    }
  else
    {
      if(size)
        {
          fprintf (file,"\tadd\t#%d, r1\n",(size+1)&~1);
          if(size==1 || size==2 || size==4 ||size==8)
            epilogue_size += 1;
          else
            epilogue_size += 2;

        }

      if(!interrupt_func_p && cfp && !main_p)
        {
          epilogue_size += 1;
          if(msp430_saved_regs_frame() == 2)
          {
          	fprintf (file,"\treti\n");
          	still_return = 0;
          }
          else
          	fprintf (file,"\tpop\tr2\n");
        }

      for(i = 15; i>=4; i--)
        {
          if((regs_ever_live[i]
              && (!call_used_regs[i]
                  || interrupt_func_p))
              ||
              (!leaf_func_p && (call_used_regs[i] && interrupt_func_p ))

            )
            {
              fprintf(file,"\tpop\tr%d\n", i);
              epilogue_size += 1;
            }
        }

      if (interrupt_func_p)
        {
          if(wakeup_func_p)
            {
              fprintf(file,   "\tbic\t#0xf0,0(r1)\n");
              epilogue_size += 3;
            }

          fprintf(file,"\treti\n");
          epilogue_size += 1;
        }
      else
        {
          if(still_return) fprintf (file, "\tret\n");
          epilogue_size += 1;
        }
    }

  fprintf (file, "/* epilogue end (size=%d) */\n", epilogue_size);
  fprintf (file, "/* function %s size %d (%d) */\n", current_function_name,
           prologue_size + function_size + epilogue_size, function_size);

  commands_in_file += prologue_size + function_size + epilogue_size;
  commands_in_prologues += prologue_size;
  commands_in_epilogues += epilogue_size;
}


/* Attempts to replace X with a valid
   memory address for an operand of mode MODE  */
/* FIXME: broken call */
rtx
legitimize_address (x, oldx, mode)
rtx x;
rtx oldx ATTRIBUTE_UNUSED;
enum machine_mode mode ATTRIBUTE_UNUSED;
{
  /*    if (GET_CODE (oldx) == MEM
              && GET_CODE (XEXP(oldx,0)) == PLUS
              && GET_CODE (XEXP(XEXP(oldx,0),0)) == MEM)
      {
          x = force_operand (oldx,0);
          return x;
      }
   
      return oldx;
  */  return x;
}

int
legitimate_address_p (mode, operand, strict)
enum machine_mode mode;
rtx operand;
int strict;
{
  rtx xfoob, x = operand;


  if (TARGET_ALL_DEBUG)
    {
      fprintf (stderr, "mode: (%s) %s %s %s %s:",
               GET_MODE_NAME(mode),
               strict ? "(strict)": "",
               reload_completed ? "(reload_completed)": "",
               reload_in_progress ? "(reload_in_progress)": "",
               reg_renumber ? "(reg_renumber)" : "");
      if (GET_CODE (x) == PLUS
          && REG_P (XEXP (x, 0))
          && GET_CODE (XEXP (x, 1)) == CONST_INT
          && reg_renumber
         )
        fprintf (stderr, "(r%d ---> r%d, const %d)", REGNO (XEXP (x, 0)),
                 true_regnum (XEXP (x, 0)), INTVAL(XEXP (x, 1)) );

      else if( GET_CODE (x) == REG && reg_renumber )
        fprintf (stderr, "(r%d ---> r%d)", REGNO(x), true_regnum(x));

      debug_rtx (x);
    }

  xfoob = XEXP (operand, 0);

  /* accept @Rn (Rn points to operand address ) */
  if (GET_CODE (operand) == REG
      && (strict ? REG_OK_FOR_BASE_STRICT_P (x)
          : REG_OK_FOR_BASE_NOSTRICT_P (x)))
    goto granted;

  /* accept address */
  if (CONSTANT_P (operand))
    goto granted;

  /* accept X(Rn) Rn + X points to operand address*/
  if (GET_CODE (operand) == PLUS
      && GET_CODE (XEXP (operand, 0)) == REG
      && CONSTANT_P (XEXP (operand, 1))
      && ( strict ? (REG_OK_FOR_BASE_STRICT_P(xfoob))
           : (REG_OK_FOR_BASE_NOSTRICT_P(xfoob))
         )
     )
    goto granted;

  if (TARGET_ALL_DEBUG) fprintf (stderr, "Address Failed\n");
  return 0;

granted:
  if (TARGET_ALL_DEBUG) fprintf (stderr, "Address granted\n");
  return 1;
}


void
print_operand_address (file, addr)
FILE *file;
rtx addr;
{
  /* hopefully will be never entered. */
  switch (GET_CODE (addr))
    {
    case REG:
      fprintf (file, "r%d", REGNO (addr) );
      return;
    case POST_INC:
      fprintf (file, "@r%d+", REGNO (addr) );
      return;
    case SYMBOL_REF:
    case LABEL_REF:
    case CONST:
      fprintf (file, "#");
      break;
    case CODE_LABEL:
      break;
    default:
      abort();
      fprintf (file, "&");
    }
  output_addr_const (file, addr);
}

void print_sub_operand PARAMS((FILE *,rtx,int));

const char *trim_array[] =
  {"llo","lhi","hlo","hhi"
  };



void
print_sub_operand(file, x, code)
FILE *file;
rtx x;
int code;
{

  if(GET_CODE (x) == SYMBOL_REF || GET_CODE (x) == LABEL_REF)
    {
      output_addr_const (file, x);
      return;
    }
  else if (GET_CODE (x) == CONST)
    {
      print_sub_operand (file, XEXP(x,0), code);
      return;
    }
  else if (GET_CODE (x) == PLUS)
    {
      print_sub_operand(file, XEXP(x,0), code);
      fprintf(file, "+");
      print_sub_operand(file, XEXP(x,1), code);
      return;
    }
  else if (GET_CODE (x) == CONST_INT)
    {
      fprintf (file, "%d", INTVAL (x));
      return ;
    }
  else
    abort();
}

void
print_operand (file, x, code)
FILE *file;
rtx x;
int code;
{
  int shift = 0;
  int ml = GET_MODE_SIZE (x->mode);
  int source_reg = 0;


  if(ml>1) ml = 2;

  if (code >= 'A' && code <= 'D') shift = code - 'A';

  if(code >='E' && code <= 'H')
    {
      shift = code - 'E';
      source_reg = 1;
    }

  if(code >='I' && code <= 'L')
    {
      ml = 1;
      shift = code - 'I';
    }

  if (REG_P (x))
    {
      fprintf (file, reg_names[true_regnum (x) + shift]);
    }
  else if (GET_CODE (x) == CONST_INT)
    {
      if(code!='F')fprintf (file, "#%s(%d)", trim_array[shift], INTVAL (x));
      else	fprintf (file, "%d", INTVAL (x));
    }
  else if (GET_CODE (x) == MEM)
    {
      rtx addr = XEXP (x,0);

      if(GET_CODE (addr) == POST_INC)
        {
          fprintf (file, "@r%d+",REGNO (addr) );
        }
      else if(GET_CODE (addr) == REG)
        {	/* for X(Rn) */
          if(shift || !source_reg)
            {
              if(shift)
                fprintf (file, "%d(r%d)", shift*ml , REGNO (addr) );
              else
                fprintf (file, "@r%d", REGNO (addr) );
            }
          else if(source_reg)
            {
              fprintf (file, "r%d", REGNO (addr)+shift );
            }
          else
            {
              fprintf (file, "@r%d", REGNO (addr) );
            }
        }
      else if(GET_CODE (addr) == SYMBOL_REF)
        {
          fprintf (file, "&");
          output_addr_const (file, addr);
          if(shift) fprintf (file, "+%d", shift*ml);
        }
      else if (GET_CODE (addr) == CONST || GET_CODE (addr) == CONST_INT)
        {
          fputc ('&', file);
          output_addr_const (file, addr);
          if(shift)fprintf (file, "+%d", shift*ml);
        }
      else if (GET_CODE (addr) == PLUS)
        {

          print_sub_operand(file,XEXP(addr,1), code);

          if(shift) fprintf (file, "+%d", shift*ml);

          if(REG_P(XEXP (addr,0)))
            fprintf(file, "(r%d)",REGNO(XEXP (addr,0)));
          else
            abort();
        }
      else if (GET_CODE (addr) == MEM)
        {
          fprintf(file, "@(Invalid addressing mode)");
          print_operand (file, addr, code);
        }
      else
        {
          fprintf(file,"Unknown operand. Please check.");
        }
    }
  else if(GET_CODE (x) == SYMBOL_REF )
    {
      fprintf (file, "#");
      output_addr_const (file, x);
      if(shift)fprintf (file, "+%d", shift*ml);
    }
  else if (GET_CODE (x) == CONST_DOUBLE)
    {
      if(GET_MODE (x) == VOIDmode )	/* FIXME: may be long long?? */
        {
          if(shift<2 )
            fprintf (file, "#%s(%d)", trim_array[shift], CONST_DOUBLE_LOW (x));
          else
            fprintf (file, "#%s(%d)", trim_array[shift-2], CONST_DOUBLE_HIGH (x));
        }
      else if( GET_MODE (x) == SFmode)
        {
          long val;
          REAL_VALUE_TYPE rv;
          REAL_VALUE_FROM_CONST_DOUBLE (rv, x);
          REAL_VALUE_TO_TARGET_SINGLE (rv, val);
          asm_fprintf (file, "#%s(0x%lx)", trim_array[shift], val);
        }
      else
        {
          fatal_insn ("Internal compiler bug. Unknown mode:", x);
        }
      ;
    }
  else
    print_operand_address (file, x);

}


/* mode for branch instruction */
int
msp430_jump_dist(x, insn)
rtx x;
rtx insn;
{
  int dest_addr = INSN_ADDRESSES (INSN_UID (GET_CODE (x) == LABEL_REF
                                  ? XEXP (x, 0) : x));
  int cur_addr = INSN_ADDRESSES (INSN_UID (insn));
  int jump_distance =  dest_addr - cur_addr;

  return jump_distance;
}



#define FIRST_CUM_REG 16

/* Initializing the variable cum for the state at the beginning
   of the argument list.  */

void
init_cumulative_args (cum, fntype, libname, indirect)
CUMULATIVE_ARGS *cum;
tree fntype;
rtx libname;
int indirect ATTRIBUTE_UNUSED;
{
  cum->nregs = 4;
  cum->regno = FIRST_CUM_REG;
  if (!libname)
    {
      int stdarg = (TYPE_ARG_TYPES (fntype) != 0
                    && (TREE_VALUE (tree_last (TYPE_ARG_TYPES (fntype)))
                        != void_type_node));
      if (stdarg) cum->nregs = 0;
    }
}


rtx
msp430_libcall_value (mode)
enum machine_mode mode;
{
  int offs = GET_MODE_SIZE (mode);
  offs >>= 1;
  if (offs < 1) offs = 1;
  return gen_rtx (REG, mode, (RET_REGISTER + 1 - offs ));

}

rtx
msp430_function_value (type, func)
tree type;
tree func ATTRIBUTE_UNUSED;
{
  unsigned int offs;
  if (TYPE_MODE (type) != BLKmode)
    return msp430_libcall_value (TYPE_MODE (type));

  offs = int_size_in_bytes (type);
  offs >>= 1;
  if (offs < 1) offs = 1;
  if (offs > 1 && offs < (GET_MODE_SIZE (SImode) >> 1))
    offs = GET_MODE_SIZE (SImode)>>1;
  else if(offs > (GET_MODE_SIZE (SImode)>>1) &&
          offs < (GET_MODE_SIZE (DImode)>>1))
    offs = GET_MODE_SIZE (DImode)>>1;

  return gen_rtx (REG, BLKmode, (RET_REGISTER + 1 - offs));
}


/* Returns the number of registers to allocate for a function argument.  */
static int
msp430_num_arg_regs(mode,type)
enum machine_mode mode;
tree type;
{
  int size;

  if (mode == BLKmode)
    size = int_size_in_bytes (type);
  else
    size = GET_MODE_SIZE (mode);

  if(size<2) size = 2;

  /* we do not care if argument is passed in odd register
     so, do not align the size ...
     BUT!!! even char argument passed in 16 bit register
     so, align the size */
  return ((size+1)&~1)>>1;
}

/* Controls whether a function argument is passed
   in a register, and which register. */

rtx
function_arg (cum, mode, type, named)
CUMULATIVE_ARGS *cum;
enum machine_mode mode;
tree type;
int named ATTRIBUTE_UNUSED;
{
  int regs = msp430_num_arg_regs(mode,type);

  if (cum->nregs && regs <= cum->nregs)
    return gen_rtx (REG, mode, cum->regno - regs);

  return NULL_RTX;
}

/* Update the summarizer variable CUM to advance past an argument
   in the argument list.  */

void
function_arg_advance (cum, mode, type, named)
CUMULATIVE_ARGS *cum;
enum machine_mode mode;
tree type;
int named ATTRIBUTE_UNUSED;
{
  int regs = msp430_num_arg_regs(mode,type);

  cum->nregs -= regs;
  cum->regno -= regs;

  if (cum->nregs <= 0)
    {
      cum->nregs = 0;
      cum->regno = FIRST_CUM_REG;
    }
}


int
nonimmediate_operand_msp430(op, mode)
rtx op;
enum machine_mode mode;
{
  int save_volatile_ok = volatile_ok;
  int niop = 0;

  if (!TARGET_NVWA) volatile_ok = 1;
  niop = nonimmediate_operand(op,mode);
  volatile_ok = save_volatile_ok;

  return niop;
}


int
memory_operand_msp430(op, mode)
rtx op;
enum machine_mode mode;
{
  int save_volatile_ok = volatile_ok;
  int mop = 0;

  if (!TARGET_NVWA) volatile_ok = 1;
  mop = memory_operand(op,mode);
  volatile_ok = save_volatile_ok;
  return mop;
}


int
general_operand_msp430(op, mode)
rtx op;
enum machine_mode mode;
{
  int save_volatile_ok = volatile_ok;
  int gop = 0;

  if (!TARGET_NVWA) volatile_ok = 1;
  gop = general_operand(op,mode);
  volatile_ok = save_volatile_ok;
  return gop;
}


int
halfnibble_integer(op, mode)
rtx op;
enum machine_mode mode;
{
  int hi, lo;
  int val;

  if(!const_int_operand(op, mode)) return 0;

  /* this integer is the one of form:
  	0xXXXX0000 or 0x0000XXXX,
  	where XXXX not one of -1,1,2,4,8 
  */
  val = INTVAL(op);
  hi = ((val&0xffff0000ul)>>16)&0xffff;
  lo = (val&0xffff);

  if(hi && lo) return 0;

  if(hi && hi!=0xffff && hi!=1 && hi!=2 && hi!=4 && hi!=8) return 1;
  if(lo && lo!=0xffff && lo!=1 && lo!=2 && lo!=4 && lo!=8) return 1;

  return 0;
}


int
halfnibble_constant(op, mode)
rtx op;
enum machine_mode mode;
{
  int hi, lo;
  int val;

  if(!const_int_operand(op, mode)) return 0;

  /* this integer is the one of form:
  	0xXXXX0000 or 0x0000XXXX,
  	where XXXX not one of -1,1,2,4,8 
  */
  val = INTVAL(op);
  hi = ((val&0xffff0000ul)>>16)&0x0000ffff;
  lo = (val&0x0000ffff);

  if((hi && lo) || (!hi && !lo)) return 0;

  if(hi==0xffff || hi==1 || hi==2 || hi==4 || hi==8) return 1;
  if(lo==0xffff || lo==1 || lo==2 || lo==4 || lo==8) return 1;

  if(!(hi&&lo)) return 1;

  return 0;
}


int
halfnibble_integer_shift(op, mode)
rtx op;
enum machine_mode mode;
{
  int hi, lo;
  int val;

  if(!immediate_operand(op, mode)) return 0;

  /* this integer is the one of form:
  	0xXXXX0000 or 0x0000XXXX,
  	where XXXX not one of -1,1,2,4,8 
  */
  val = 1<<INTVAL(op);
  hi = ((val&0xffff0000ul)>>16)&0x0000ffff;
  lo = (val&0x0000ffff);

  if(hi && lo) return 0;

  if(hi && hi!=0xffff && hi!=1 && hi!=2 && hi!=4 && hi!=8) return 1;
  if(lo && lo!=0xffff && lo!=1 && lo!=2 && lo!=4 && lo!=8) return 1;

  return 0;
}


int
halfnibble_constant_shift(op, mode)
rtx op;
enum machine_mode mode;
{
  int hi, lo;
  int val;

  if(!immediate_operand(op, mode)) return 0;

  /* this integer is the one of form:
  	0xXXXX0000 or 0x0000XXXX,
  	where XXXX not one of -1,1,2,4,8 
  */
  val = 1<<INTVAL(op);
  hi = ((val&0xffff0000ul)>>16)&0x0000ffff;
  lo = (val&0x0000ffff);

  if(hi && lo) return 0;

  if(hi && hi==0xffff && hi==1 && hi==2 && hi==4 && hi==8) return 1;
  if(lo && lo==0xffff && lo==1 && lo==2 && lo==4 && lo==8) return 1;

  return 0;
}


int
which_nibble(val)
int val;
{
  if(val&0xffff0000ul) return 1;
  return 0;
}


int
which_nibble_shift(val)
int val;
{
  if(val&0xffff0000ul) return 1;
  return 0;
}


int
extra_constraint (x, c)
rtx x;
int c;
{

  if(c == 'R')
    {
      if(GET_CODE (x) == MEM
          && GET_CODE (XEXP (x,0)) == REG)
        {
          rtx xx = XEXP (x, 0);
          int regno = REGNO (xx);
          if (regno >= 4 || regno == 1 ) return 1;
        }
    }
  else if(c == 'Q')
    {
      if(GET_CODE (x) == MEM
          && GET_CODE (XEXP (x,0)) == REG)
        {
          rtx xx = XEXP (x, 0);
          int regno = REGNO (xx);
          if (regno >= 4 || regno == 1) return 1;
        }

      if (GET_CODE (x) == MEM
          && GET_CODE (XEXP (x,0)) == PLUS
          && GET_CODE (XEXP (XEXP (x,0), 1)) == CONST_INT)
        {
          rtx xx = XEXP (XEXP (x,0), 0);
          int regno = REGNO (xx);
          if (regno >= 4 || regno == 1) return 1;
        }

      if (GET_CODE (x) == MEM
          && GET_CODE (XEXP (x,0)) == PLUS
          && REG_P (XEXP (XEXP (x,0), 0)) )
        {
          return 1;
        }

    }
  else if(c == 'S')
    {
      if(GET_CODE (x) == MEM
          && GET_CODE (XEXP (x,0)) == SYMBOL_REF)
        {
          return 1;
        }
    }

  return 0;
}

int
indexed_location(x)
rtx x;
{
  int r = 0;

  if(GET_CODE (x) == MEM
      && GET_CODE (XEXP (x,0)) == REG)
    {
      r = 1;
    }

  if (TARGET_ALL_DEBUG)
    {
      fprintf(stderr, "indexed_location %s: %s  \n",
              r?"granted":"failed",
              reload_completed? "reload completed":"reload in progress");
      debug_rtx(x);
    }

  return r;
}


int
zero_shifted(x)
rtx x;
{
  int r = 0;

  if (GET_CODE (x) == MEM &&
      GET_CODE (XEXP (x,0)) == REG
      && REGNO(XEXP (x,0)) != STACK_POINTER_REGNUM
      && REGNO(XEXP (x,0)) != FRAME_POINTER_REGNUM
      /* the following is Ok, cause we do not corrupt r4 within ISR */
      /*&& REGNO(XEXP (x,0)) != ARG_POINTER_REGNUM */ )
    {
      r = 1;
    }

  if (TARGET_ALL_DEBUG)
    {
      fprintf(stderr, "zero_shifted %s: %s  \n",
              r?"granted":"failed",
              reload_completed? "reload completed":"reload in progress");
      debug_rtx(x);
    }

  return r;
}



int
default_rtx_costs (X, code, outer_code)
rtx X ATTRIBUTE_UNUSED;
enum rtx_code code ;
enum rtx_code outer_code ATTRIBUTE_UNUSED;
{
  int cost=0;

  switch(code)
    {
    case SYMBOL_REF:
      cost = 1;
      break;
    case LABEL_REF:
      cost = 1;
      break;
    case MEM:
      cost += 1;
      break;
    case CONST_INT:
      cost = 1;
      break;
    case SIGN_EXTEND:
    case ZERO_EXTEND:
      cost += 2;
      break;
    default:
      break;
    }
  return cost;
}


void
order_regs_for_local_alloc ()
{
  unsigned int i;

  if(TARGET_REORDER)
    {
      reg_alloc_order[0] = 12;
      reg_alloc_order[1] = 13;
      reg_alloc_order[2] = 14;
      reg_alloc_order[3] = 15;
      for (i=4; i < 16;i++) reg_alloc_order[i] = 15-i;
    }
  else
    {
      for (i=0; i < 16;i++) reg_alloc_order[i] = 15-i;
    }

  return;
}




/* Output rtx VALUE as .byte to file FILE */

void
asm_output_char (file, value)
FILE *file;
rtx value;
{
  fprintf (file, "\t.byte\t");
  output_addr_const (file, value);
  fprintf (file, "\n");
}


/* Output VALUE as .byte to file FILE */

void
asm_output_byte (file, value)
FILE *file;
int value;
{
  fprintf (file, "\t.byte 0x%x\n", value & 0xff);
}


/* Output rtx VALUE as .word to file FILE */

void
asm_output_short (file, value)
FILE *file;
rtx value;
{
  fprintf (file, "\t.word ");
  output_addr_const (file, (value));
  fprintf (file, "\n");
}


/* Output real N to file FILE */

void
asm_output_float (file, n)
FILE *file;
REAL_VALUE_TYPE n;
{
  long val;
  char dstr[100];

  REAL_VALUE_TO_TARGET_SINGLE (n, val);
  REAL_VALUE_TO_DECIMAL (n, "%g", dstr);
  fprintf (file, "\t.long 0x%08lx\t/* %s */\n", val, dstr);
}

/* Sets section name for declaration DECL */

void
unique_section (decl, reloc)
tree decl;
int reloc ATTRIBUTE_UNUSED;
{
  int len;
  const char *name, *prefix;
  char *string;
  name = IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (decl));
  /* Strip off any encoding in name.  */
  STRIP_NAME_ENCODING (name, name);

  if (TREE_CODE (decl) == FUNCTION_DECL)
    {
      if (flag_function_sections)
        prefix = ".text.";
      else
        prefix = ".text";
    }
  else
    abort ();

  if (flag_function_sections)
    {
      len = strlen (name) + strlen (prefix);
      string = alloca (len + 1);
      sprintf (string, "%s%s", prefix, name);
      DECL_SECTION_NAME (decl) = build_string (len, string);
    }
}


/* Output section name to file FILE
   We make the section read-only and executable for a function decl,
   read-only for a const data decl, and writable for a non-const data decl.  */

void
asm_output_section_name(file, decl, name, reloc)
FILE *file;
tree decl;
const char *name;
int reloc ATTRIBUTE_UNUSED;
{
  fprintf (file, ".section %s, \"%s\", @progbits\n", name,
           decl && TREE_CODE (decl) == FUNCTION_DECL ? "ax" :
           decl && TREE_READONLY (decl) ? "a" : "aw");
}


/* The routine used to output NUL terminated strings.  We use a special
   version of this for most svr4 targets because doing so makes the
   generated assembly code more compact (and thus faster to assemble)
   as well as more readable, especially for targets like the i386
   (where the only alternative is to output character sequences as
   comma separated lists of numbers).   */

void
gas_output_limited_string(file, str)
FILE *file;
const char * str;
{
  const unsigned char *_limited_str = (unsigned char *) str;
  unsigned ch;
  fprintf (file, "%s\"", STRING_ASM_OP);
  for (; (ch = *_limited_str); _limited_str++)
    {
      int escape;
      switch (escape = ESCAPES[ch])
        {
        case 0:
          putc (ch, file);
          break;
        case 1:
          fprintf (file, "\\%03o", ch);
          break;
        default:
          putc ('\\', file);
          putc (escape, file);
          break;
        }
    }
  fprintf (file, "\"\n");
}

/* The routine used to output sequences of byte values.  We use a special
   version of this for most svr4 targets because doing so makes the
   generated assembly code more compact (and thus faster to assemble)
   as well as more readable.  Note that if we find subparts of the
   character sequence which end with NUL (and which are shorter than
   STRING_LIMIT) we output those using ASM_OUTPUT_LIMITED_STRING.  */

void
gas_output_ascii(file, str, length)
FILE * file;
const char * str;
size_t length;
{
  const unsigned char *_ascii_bytes = (const unsigned char *) str;
  const unsigned char *limit = _ascii_bytes + length;
  unsigned bytes_in_chunk = 0;
  for (; _ascii_bytes < limit; _ascii_bytes++)
    {
      const unsigned char *p;
      if (bytes_in_chunk >= 60)
        {
          fprintf (file, "\"\n");
          bytes_in_chunk = 0;
        }
      for (p = _ascii_bytes; p < limit && *p != '\0'; p++)
        continue;
      if (p < limit && (p - _ascii_bytes) <= (signed)STRING_LIMIT)
        {
          if (bytes_in_chunk > 0)
            {
              fprintf (file, "\"\n");
              bytes_in_chunk = 0;
            }
          gas_output_limited_string (file, (char*)_ascii_bytes);
          _ascii_bytes = p;
        }
      else
        {
          int escape;
          unsigned ch;
          if (bytes_in_chunk == 0)
            fprintf (file, "\t.ascii\t\"");
          switch (escape = ESCAPES[ch = *_ascii_bytes])
            {
            case 0:
              putc (ch, file);
              bytes_in_chunk++;
              break;
            case 1:
              fprintf (file, "\\%03o", ch);
              bytes_in_chunk += 4;
              break;
            default:
              putc ('\\', file);
              putc (escape, file);
              bytes_in_chunk += 2;
              break;
            }
        }
    }
  if (bytes_in_chunk > 0)
    fprintf (file, "\"\n");
}



/* If IDENTIFIER with arguments ARGS is a valid machine specific
   attribute for DECL return 1.
   Valid attributes:
   bootloader and infomemory are not declared
   signal  - make a function to be a hardware interrupt. After function
   		prologue interrupts are disabled;
   interrupt - make a function to be a hardware interrupt. After function
   		prologue interrupts are enabled;
   wakeup - full throttle CPU mode restored 
   		on function exit (in epilogue, just before ret(i) command)
   naked     - don't generate function prologue/epilogue and `ret' command.
   hwmul_no_int - assume no interrupts doing hwmul will occur durint function 
   critical - disables interrupts on func entry and re-enables on exit */

int
valid_machine_decl_attribute (decl, attributes, attr, args)
tree decl;
tree attributes ATTRIBUTE_UNUSED;
tree attr;
tree args ATTRIBUTE_UNUSED;
{
  if (is_attribute_p ("interrupt", attr)
      || is_attribute_p ("signal", attr)
      || is_attribute_p ("naked", attr)
      || is_attribute_p ("wakeup", attr)
      || is_attribute_p ("reserve", attr)
      || is_attribute_p ("hwmul_no_int",attr)
      || is_attribute_p ("critical",attr))
    return TREE_CODE (decl) == FUNCTION_DECL;

  return 0;
}


/* Outputs to the stdio stream FILE some
   appropriate text to go at the start of an assembler file.  */

void
asm_file_start (file)
FILE *file;
{
  output_file_directive (file, main_input_filename);
  fprintf (file, "\t.arch %s\n\n", msp430_mcu_name);

  if(msp430_has_hwmul)
    {
      fprintf (file, 	"/* Hardware multiplier registers: */\n"
               "__MPY=0x130\n"
               "__MPYS=0x132\n"
               "__MAC=0x134\n"
               "__MACS=0x136\n"
               "__OP2=0x138\n"
               "__RESLO=0x13a\n"
               "__RESHI=0x13c\n"
               "__SUMEXT=0x13e\n"
               "\n");

    }

  commands_in_file = 0;
  commands_in_prologues = 0;
  commands_in_epilogues = 0;
}

/* Outputs to the stdio stream FILE some
   appropriate text to go at the end of an assembler file.  */

void
asm_file_end (file)
FILE *file;
{
  fprintf (file,
           "\n"
           "/*********************************************************************\n"
           " * File %s: code size: %d words (0x%x)\n * incl. words in prologues: %d, epilogues: %d\n"
           " *********************************************************************/\n",
           main_input_filename,
           commands_in_file,
           commands_in_file,
           commands_in_prologues,
           commands_in_epilogues);
}

int
msp430_hard_regno_mode_ok(regno, mode)
int regno ATTRIBUTE_UNUSED;
enum machine_mode mode ATTRIBUTE_UNUSED;
{
  return 1;
}

int
frame_pointer_required_p ()
{
  return (current_function_calls_alloca
          /*            || current_function_args_info.nregs == 0 */
          || current_function_varargs);

  /* || get_frame_size () > 0); */
}

enum reg_class
preferred_reload_class (x, class)
rtx x ATTRIBUTE_UNUSED;
enum reg_class class;
{
  return class;
}



/* cfp minds the fact that the function may save r2 */
int
initial_elimination_offset (from, to)
int from;
int to;
{
  int reg;
  if (from == FRAME_POINTER_REGNUM && to == STACK_POINTER_REGNUM)
    return 0;
  else
    {
      int interrupt_func_p = interrupt_function_p (current_function_decl);
      int cfp = msp430_critical_function_p(current_function_decl);
      int leaf_func_p = leaf_function_p ();
      int offset = interrupt_func_p ? 0 : (cfp?2:0);

      for (reg = 4; reg < 16; ++reg)
        {
          if ( (!leaf_func_p && call_used_regs[reg] && (interrupt_func_p))
               || (regs_ever_live[reg] 
                   && (!call_used_regs[reg] || interrupt_func_p)))
            {
              offset += 2;
            }
        }
      return get_frame_size () + offset + 2;
    }
  return 0;
}


int
adjust_insn_length (insn, len)
rtx insn;
int len;
{

  rtx patt = PATTERN (insn);
  rtx set;

  set = single_set (insn);

  if (GET_CODE (patt) == SET)
    {
      rtx op[10];
      op[1] = SET_SRC (patt);
      op[0] = SET_DEST (patt);

      if (general_operand (op[1], VOIDmode)
          && general_operand (op[0], VOIDmode))
        {
          op[2] = SET_SRC (patt);
          switch (GET_MODE (op[0]))
            {
            case QImode:
            case HImode:
              if(indexed_location(op[1])) len--;
              break;

            case SImode:
            case SFmode:
              /* get length first */
              msp430_movesi_code(insn,op,&len);

              if(zero_shifted(op[1])
                  && regsi_ok_safe(op) )
                {
                  rtx reg = XEXP(op[1],0);
                  if( dead_or_set_p (insn, reg) )
                    len -= 1;
                }
              else if( !zero_shifted(op[1])
                       && indexed_location(op[1]) )
                {
                  len -= 1;
                }
              break;
            case DImode:
              msp430_movedi_code(insn,op,&len);
              if(zero_shifted(op[1])
                  && regdi_ok_safe(op) )
                {
                  rtx reg = XEXP(op[1],0);
                  if( dead_or_set_p (insn, reg) )
                    len -= 1;
                }
              else if( !zero_shifted(op[1])
                       && indexed_location(op[1]) )
                {
                  len -= 1;
                }
              break;

            default:
              break;
            }

          if(GET_CODE(op[2]) == CONST_INT)
            {
              if (GET_MODE (op[0]) == SImode)
                {
                  int x = INTVAL(op[2]);
                  int y = (x&0xffff0000ul)>>16;
                  x = x&0xffff;

                  if(x==0||x==1||x==2||x==4||x==8||x==0xffff) len--;
                  if(y==0||y==1||y==2||y==4||y==8||y==0xffff) len--;
                }
            }

          if(GET_CODE(op[2]) == CONST_DOUBLE)
            {
              if(GET_MODE (op[0]) == SFmode)
                {
                  long val;
                  int  y,x;
                  REAL_VALUE_TYPE rv;
                  REAL_VALUE_FROM_CONST_DOUBLE (rv, op[2]);
                  REAL_VALUE_TO_TARGET_SINGLE (rv, val);

                  y = (val&0xffff0000ul)>>16;
                  x = val&0xffff;
                  if(x==0||x==1||x==2||x==4||x==8||x==0xffff) len--;
                  if(y==0||y==1||y==2||y==4||y==8||y==0xffff) len--;
                }
              else
                {
                  int hi = CONST_DOUBLE_HIGH (op[2]);
                  int lo = CONST_DOUBLE_LOW (op[2]);
                  int x,y,z;

                  x = (hi&0xffff0000ul)>>16;
                  y =  hi&0xffff;
                  z = (lo&0xffff0000ul)>>16;
                  if(x==0||x==1||x==2||x==4||x==8||x==0xffff) len--;
                  if(y==0||y==1||y==2||y==4||y==8||y==0xffff) len--;
                  if(z==0||z==1||z==2||z==4||z==8||z==0xffff) len--;
                  z = lo&0xffff;
                  if(z==0||z==1||z==2||z==4||z==8||z==0xffff) len--;
                }
            }

          return len;
        }
      else if(GET_CODE (op[1]) == MULT)
        {
          rtx ops[10];
          ops[0] = op[0];
          ops[1] = XEXP (op[1],0);
          ops[2] = XEXP (op[1],1);

          if(GET_MODE (ops[0]) != SImode
              && GET_MODE (ops[0]) != SFmode
              && GET_MODE (ops[0]) != DImode)
            {
              if(indexed_location(ops[1])) len--;
              if(indexed_location(ops[2])) len--;
            }
        }
      else if(GET_CODE (op[1]) == PLUS
              || GET_CODE (op[1]) == MINUS
              || GET_CODE (op[1]) == AND
              || GET_CODE (op[1]) == IOR
              || GET_CODE (op[1]) == XOR
             )
        {
          rtx ops[10];
          ops[0] = op[0];
          ops[1] = XEXP (op[1],0);
          ops[2] = XEXP (op[1],1);

          switch (GET_MODE (ops[0]))
            {
            case QImode:
            case HImode:
              if(indexed_location(ops[2])) len--;
              break;
            case SImode:
            case SFmode:

              if(GET_CODE (op[1]) == PLUS) msp430_addsi_code(insn, ops, &len);
              if(GET_CODE (op[1]) == MINUS) msp430_subsi_code(insn, ops, &len);
              if(GET_CODE (op[1]) == AND) msp430_andsi_code(insn, ops, &len);
              if(GET_CODE (op[1]) == IOR) msp430_iorsi_code(insn, ops, &len);
              if(GET_CODE (op[1]) == XOR) msp430_xorsi_code(insn, ops, &len);


              if(zero_shifted(ops[2])
                  && regsi_ok_safe(ops) )
                {
                  rtx reg = XEXP(ops[2],0);
                  if( dead_or_set_p (insn, reg) )
                    len -= 1;
                }
              else if( !zero_shifted(ops[2])
                       && indexed_location(ops[2]) )
                {
                  len -= 1;
                }
              break;
            case DImode:

              if(GET_CODE (op[1]) == PLUS) msp430_adddi_code(insn, ops, &len);
              if(GET_CODE (op[1]) == MINUS) msp430_subdi_code(insn, ops, &len);
              if(GET_CODE (op[1]) == AND) msp430_anddi_code(insn, ops, &len);
              if(GET_CODE (op[1]) == IOR) msp430_iordi_code(insn, ops, &len);
              if(GET_CODE (op[1]) == XOR) msp430_xordi_code(insn, ops, &len);

              if(zero_shifted(ops[2])
                  && regdi_ok_safe(ops) )
                {
                  rtx reg = XEXP(ops[2],0);
                  if( dead_or_set_p (insn, reg) )
                    len -= 1;
                }
              else if( !zero_shifted(ops[2])
                       && indexed_location(ops[2]) )
                {
                  len -= 1;
                }
              break;

            default:
              break;
            }

          if(GET_MODE (ops[0]) == SImode)
            {
              if(GET_CODE(ops[2]) == CONST_INT)
                {
                  if(GET_CODE (op[1]) == AND)
                    {
                      msp430_emit_immediate_and2(insn, ops, &len);
                    }
                  else if(GET_CODE (op[1]) == IOR)
                    {
                      msp430_emit_immediate_ior2(insn, ops, &len);
                    }
                  else
                    {
                      if (GET_MODE (ops[0]) == SImode)
                        {
                          int x = INTVAL(ops[2]);
                          int y = (x&0xffff0000ul)>>16;
                          x = x&0xffff;

                          if(x==0||x==1||x==2||x==4||x==8||x==0xffff) len--;
                          if(y==0||y==1||y==2||y==4||y==8||y==0xffff) len--;
                        }
                    }
                }
            }

          if(GET_MODE (ops[0]) == SFmode || GET_MODE (ops[0]) == DImode)
            {
              if(GET_CODE(ops[2]) == CONST_DOUBLE)
                {

                  if(GET_CODE (op[1]) == AND)
                    {
                      msp430_emit_immediate_and4(insn, ops, &len);
                    }
                  else if(GET_CODE (op[1]) == IOR)
                    {
                      msp430_emit_immediate_ior4(insn, ops, &len);
                    }
                  else if(GET_MODE (ops[0]) == SFmode)
                    {
                      long val;
                      int  y,x;
                      REAL_VALUE_TYPE rv;
                      REAL_VALUE_FROM_CONST_DOUBLE (rv, ops[2]);
                      REAL_VALUE_TO_TARGET_SINGLE (rv, val);

                      y = (val&0xffff0000ul)>>16;
                      x = val&0xffff;
                      if(x==0||x==1||x==2||x==4||x==8||x==0xffff) len--;
                      if(y==0||y==1||y==2||y==4||y==8||y==0xffff) len--;
                    }
                  else
                    {
                      int hi = CONST_DOUBLE_HIGH (ops[2]);
                      int lo = CONST_DOUBLE_LOW (ops[2]);
                      int x,y,z;

                      x = (hi&0xffff0000ul)>>16;
                      y =  hi&0xffff;
                      z = (lo&0xffff0000ul)>>16;
                      if(x==0||x==1||x==2||x==4||x==8||x==0xffff) len--;
                      if(y==0||y==1||y==2||y==4||y==8||y==0xffff) len--;
                      if(z==0||z==1||z==2||z==4||z==8||z==0xffff) len--;
                    }
                }
            }

          return len;
        }
      else if(GET_CODE (op[1]) == NOT
              || GET_CODE (op[1]) == ABS
              || GET_CODE (op[1]) == NEG
             )
        {
          if(GET_MODE (op[0]) == HImode || GET_MODE (op[0]) == QImode)
            if(indexed_location(XEXP (op[1],0))) len--;
          /* consts handled by cpp */
          /* nothing... */
        }
      else if(GET_CODE (op[1]) == ZERO_EXTEND)
        {
          rtx ops[10];
          ops[0] = op[0];
          ops[1] = XEXP (op[1],0);

          if(GET_MODE (ops[1]) == QImode)
            {
              if(GET_MODE (ops[0]) == HImode) zeroextendqihi(insn, ops, &len);
              else if(GET_MODE (ops[0]) == SImode) zeroextendqisi(insn, ops, &len);
              else if(GET_MODE (ops[0]) == DImode) zeroextendqidi(insn, ops, &len);
            }
          else if(GET_MODE (ops[1]) == HImode)
            {
              if(GET_MODE (ops[0]) == SImode) zeroextendhisi(insn, ops, &len);
              else if(GET_MODE (ops[0]) == DImode) zeroextendhidi(insn, ops, &len);
            }
          else if(GET_MODE (ops[1]) == SImode)
            {
              if(GET_MODE (ops[1]) == DImode) zeroextendsidi(insn, ops, &len);
            }
        }
      else if(GET_CODE (op[1]) == SIGN_EXTEND)
        {
          rtx ops[10];
          ops[0] = op[0];  /* dest */
          ops[1] = XEXP (op[1],0); /* src */

          if(GET_MODE (ops[1]) == QImode)
            {
              if(GET_MODE (ops[0]) == HImode) signextendqihi(insn, ops, &len);
              else if(GET_MODE (ops[0]) == SImode) signextendqisi(insn, ops, &len);
              else if(GET_MODE (ops[0]) == DImode) signextendqidi(insn, ops, &len);
            }
          else if(GET_MODE (ops[1]) == HImode)
            {
              if(GET_MODE (ops[0]) == SImode) signextendhisi(insn, ops, &len);
              else if(GET_MODE (ops[0]) == DImode) signextendhidi(insn, ops, &len);
            }
          else if(GET_MODE (ops[1]) == SImode)
            {
              if(GET_MODE (ops[0]) == DImode) signextendsidi(insn, ops, &len);
            }
        }
      else if(GET_CODE (op[1]) == IF_THEN_ELSE)
        {
          ; /* FIXME: is this can be done with a help of
          		     machine dependend reorg? */
        }
      else if(GET_CODE(op[1]) == COMPARE)
        {
          rtx ops[4];
          ops[0] = XEXP (op[1],0);
          ops[1] = XEXP (op[1],1);

          if(GET_MODE (ops[0]) == HImode ) msp430_emit_cmphi(insn, ops, &len);
          if(GET_MODE (ops[0]) == QImode ) msp430_emit_cmpqi(insn, ops, &len);

        }
      else if(GET_CODE(op[0]) == CC0 && general_operand(op[1], VOIDmode))
        {
          rtx ops[4];
          ops[0] = op[1];
          if(GET_MODE (op[1]) == HImode ) msp430_emit_tsthi(insn, ops, &len);
          if(GET_MODE (op[1]) == QImode ) msp430_emit_tstqi(insn, ops, &len);
        }
      else if(GET_CODE(op[0]) == MEM
              && GET_CODE( XEXP (op[0],0)) == POST_DEC)
        {
          rtx ops[4];
          ops[0] = op[1];
          if(GET_MODE(op[0]) == QImode) msp430_pushqi(insn, ops, &len);
          if(GET_MODE(op[0]) == HImode) msp430_pushhi(insn, ops, &len);
          if(GET_MODE(op[0]) == SImode) msp430_pushsisf(insn, ops, &len);
          if(GET_MODE(op[0]) == DImode) msp430_pushdi(insn, ops, &len);
        }
    }

  if(set)
    {
      rtx op[10];
      op[1] = SET_SRC (set);
      op[0] = SET_DEST (set);

      if (GET_CODE (patt) == PARALLEL)
        {
          int l = XVECLEN (patt, 0); /* this == 2 for tests and == 3 for cmps */

          if(GET_CODE(op[0]) == CC0)
            {
              if(general_operand(op[1], VOIDmode) && l == 2)
                {
                  rtx ops[3];

                  ops[0] = op[1];
                  ops[1] = XEXP(XVECEXP (patt, 0, 1),0);	/* scratch */
                  if(GET_MODE (ops[0]) == SImode ) msp430_emit_tstsi(insn, ops, &len);
                }
              else if(GET_CODE(op[1]) == COMPARE)
                {
                  rtx ops[4];
                  ops[0] = XEXP (op[1],0);
                  ops[1] = XEXP (op[1],1);
                  ops[2] = XEXP(XVECEXP (patt, 0, 1),0);      /* scratch */
                  if(GET_MODE (ops[0]) == SImode ) msp430_emit_cmpsi(insn, ops, &len);
                }
            }
        }
      else if (GET_CODE (op[1]) == ASHIFT
               || GET_CODE (op[1]) == ASHIFTRT
               || GET_CODE (op[1]) == LSHIFTRT)
        {
          rtx ops[10];
          ops[0] = op[0];
          ops[1] = XEXP (op[1],0);
          ops[2] = XEXP (op[1],1);

          switch (GET_CODE (op[1]))
            {
            case ASHIFT:
              switch (GET_MODE (op[0]))
                {
                case QImode:
                  msp430_emit_ashlqi3(insn, ops,&len);
                  break;
                case HImode:
                  msp430_emit_ashlhi3(insn, ops,&len);
                  break;
                case SImode:
                  msp430_emit_ashlsi3(insn, ops,&len);
                  break;
                case DImode:
                  msp430_emit_ashldi3(insn, ops,&len);
                  break;
                default:
                  break;
                }
              break;

            case ASHIFTRT:
              switch (GET_MODE (op[0]))
                {
                case QImode:
                  msp430_emit_ashrqi3(insn, ops,&len);
                  break;
                case HImode:
                  msp430_emit_ashrhi3(insn, ops,&len);
                  break;
                case SImode:
                  msp430_emit_ashrsi3(insn, ops,&len);
                  break;
                case DImode:
                  msp430_emit_ashrdi3(insn, ops,&len);
                  break;
                default:
                  break;
                }
              break;

            case LSHIFTRT:
              switch (GET_MODE (op[0]))
                {
                case QImode:
                  msp430_emit_lshrqi3(insn, ops,&len);
                  break;
                case HImode:
                  msp430_emit_lshrhi3(insn, ops,&len);
                  break;
                case SImode:
                  msp430_emit_lshrsi3(insn, ops,&len);
                  break;
                case DImode:
                  msp430_emit_lshrdi3(insn, ops,&len);
                  break;
                default:
                  break;
                }
              break;

            default:
              break;
            }
        }
    }

  return len;
}


/* Output all insn addresses and their sizes into the assembly language
   output file.  This is helpful for debugging whether the length attributes
      in the md file are correct.
         Output insn cost for next insn.  */

void
final_prescan_insn (insn, operand, num_operands)
rtx insn, *operand ATTRIBUTE_UNUSED;
int num_operands ATTRIBUTE_UNUSED;
{
  int uid = INSN_UID (insn);

  if (TARGET_ALL_DEBUG)
    {
      fprintf (asm_out_file, "/*DEBUG: 0x%x\t\t%d\t%d */\n",
               INSN_ADDRESSES (uid),
               INSN_ADDRESSES (uid) - last_insn_address,
               rtx_cost (PATTERN (insn), INSN));
    }
  last_insn_address = INSN_ADDRESSES (uid);

  if (TARGET_RTL_DUMP)
    {
      fprintf (asm_out_file, "/* UID:%d ** ADDR:%d ** COST:%d *\n",
               uid,
               INSN_ADDRESSES (uid),
               rtx_cost (PATTERN (insn), INSN) );
      print_rtl_single (asm_out_file, insn);
      fprintf (asm_out_file, "*****************/\n");
    }
}


void
msp430_output_addr_vec_elt (stream, value)
FILE *stream;
int value;
{
  fprintf (stream, "\t.word	.L%d\n", value);
  jump_tables_size++;
}


/* Get rid of push registers which are scratches and not being used */
void
machine_dependent_reorg (first_insn)
rtx first_insn ATTRIBUTE_UNUSED;
{
  rtx insn, pattern, scratch;
  int reg_scratch_used[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
  int reg_scratch_requested[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
  int reg_really_used[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
  register int i;
  rtx u_reg[16];


  addenum_size = 0;

  for (i = 4; i<16;i++) u_reg[i] = gen_rtx(REG, HImode, i);

  for (insn = first_insn; insn; insn = NEXT_INSN (insn))
    {
      if (!(GET_CODE (insn) == INSN
            || GET_CODE (insn) == CALL_INSN
            || GET_CODE (insn) == JUMP_INSN)
          || !single_set (insn))
        continue;

      pattern = PATTERN (insn);

      scratch = 0;

      if (GET_CODE (pattern) == PARALLEL)
        {
          scratch = XEXP (XVECEXP (pattern, 0, 1), 0);
          pattern = XVECEXP (pattern, 0, 0);
        }

      /* find compare & test insns */
      if(GET_CODE (pattern) == SET
          && SET_DEST (pattern) == cc0_rtx)
        {
          RTX_CODE code = GET_CODE (SET_SRC (pattern));

          if (code == COMPARE)
            {
              rtx src = SET_SRC (pattern);
              rtx op[4];
              int s = 0;
              op[0] = XEXP(src,0);
              op[1] = XEXP(src,1);

              if(scratch)
                {
                  op[2] = scratch;

                  reg_scratch_requested[REGNO(op[2])] = 1;
                  reg_scratch_requested[REGNO(op[2])+1] = 1;

                  msp430_prepare_cmpsi(insn,op);

                  s = cmpsi_needs_scratch_p(insn, op);

                  if(s)
                    {
                      reg_scratch_used[REGNO(op[2])] = 1;
                      reg_scratch_used[REGNO(op[2])+1] = 1;
                    }
                }
              else
                {
                  if(GET_MODE(src)==HImode)
                    msp430_prepare_cmphi(insn,op);
                  else if(GET_MODE(src)==QImode)
                    msp430_prepare_cmpqi(insn,op);
                }
            }
          else if (general_operand (SET_SRC (pattern), VOIDmode))
            {
              if(scratch)
                {
                  rtx src = SET_SRC (pattern);
                  rtx op[3];
                  int s = 0, sh = 0;

                  op[0] = src;
                  op[1] = scratch;

                  s = tstsi_needs_scratch_p(insn, op);
                  sh = tstsi_needs_half_scratch_p(insn, op);

                  reg_scratch_requested[REGNO(op[1])] = 1;
                  if(!sh) reg_scratch_requested[REGNO(op[1])+1] = 1;

                  if(s || sh)
                    {
                      reg_scratch_used[REGNO(op[1])] = 1;
                      if(s)reg_scratch_used[REGNO(op[1])+1] = 1;
                    }

                  msp430_prepare_tstsi(insn,op);
                }
              else
                {
                  /* QI and HI modes. */
                  rtx src = SET_SRC (pattern);
                  rtx ops[3];

                  ops[0] = src;

                  if(GET_MODE(src)==HImode)
                    msp430_prepare_tsthi(insn,ops);
                  else if(GET_MODE(src)==QImode)
                    msp430_prepare_tstqi(insn,ops);
                }
            }
        }
    }

  if(!optimize && !optimize_size) return ;

  /* check if registers has gone within optimization*/
  for (insn = first_insn; insn; insn = NEXT_INSN (insn))
    {
      for(i=4;i<16;i++)
        {
          rtx reg = u_reg[i];

          if(!regs_ever_live[i] || reg_really_used[i] ) continue;

          if (INSN_P (insn)
              && (reg_overlap_mentioned_p (reg, PATTERN (insn))
                  || find_reg_fusage (insn, USE, reg)
                  || find_reg_fusage (insn, CLOBBER, reg)
                  || find_regno_note(insn, REG_UNUSED, i)))
            reg_really_used[i] = 1;
        }
    }

  for(i=4;i<16;i++)
    {
      /**** Performance hit: interrupt routines will save unused r12 - r15
      	if(!call_used_regs[i]
      	    && (regs_ever_live[i] != reg_really_used[i]))
      *****/
      if(regs_ever_live[i] != reg_really_used[i])
        {
          if(!call_used_regs[i]) addenum_size += 2;
          regs_ever_live[i] = reg_really_used[i];
        }
    }
}


int
test_hard_reg_class (class, x)
enum reg_class class;
rtx x;
{
  int regno = true_regnum (x);
  if (regno < 0) return 0;
  return TEST_HARD_REG_CLASS (class, regno);
}



/* Returns 1 if SCRATCH are safe to be allocated as a scratch
   registers (for a define_peephole2) in the current function.  */
/* UNUSED ... yet... */
int
msp430_peep2_scratch_safe (scratch)
rtx scratch;
{
  if ((interrupt_function_p (current_function_decl)
       || signal_function_p (current_function_decl))
      && leaf_function_p ())
    {
      int first_reg = true_regnum (scratch);
      int last_reg;
      int size = GET_MODE_SIZE (GET_MODE (scratch));
      int reg;

      size >>= 1;
      if(!size) size = 1;

      last_reg = first_reg + size - 1;

      for (reg = first_reg; reg <= last_reg; reg++)
        {
          if (!regs_ever_live[reg]) return 0;
        }
    }

  return 1;
}


/* Update the condition code in the INSN.  */

void
notice_update_cc (body, insn)
rtx body ATTRIBUTE_UNUSED;
rtx insn;
{
  rtx set;

  switch (get_attr_cc (insn))
    {
    case CC_NONE:
      /* Insn does not affect CC at all.  */

      /* Slightly complicated:
         If there is a sequence, which updates an RTL and
         this one is equal to cc_status.value1, we have to 'clobber' cc0.
         This applies to all 'mov' (mostly in HI mode),
         'andMi3_inv', 'ior' and truncates. Does not apply to branches */
      set = single_set (insn);
      if( set != 0 && cc_status.value1 != 0
          && (rtx_equal_p (SET_DEST (set), cc_status.value1)
              || reg_overlap_mentioned_p(SET_DEST (set), cc_status.value1)))
        {
          CC_STATUS_INIT;
        }

      break;

    case CC_SET_N:
      CC_STATUS_INIT;
      break;

    case CC_SET_ZN:
      set = single_set (insn);
      CC_STATUS_INIT;
      if (set)
        {
          cc_status.flags |= CC_NO_OVERFLOW;
          cc_status.value1 = SET_DEST (set);
        }
      break;

    case CC_SET_CZN:
      /* Insn sets the Z,N,C flags of CC to recog_operand[0].
         The V flag may or may not be known but that's ok because
         alter_cond will change tests to use EQ/NE.  */
      set = single_set (insn);
      CC_STATUS_INIT;
      if (set)
        {
          cc_status.value1 = SET_DEST (set);
          cc_status.flags |= CC_OVERFLOW_UNUSABLE;
        }
      break;

    case CC_COMPARE:
      set = single_set (insn);
      CC_STATUS_INIT;
      if (set)
        cc_status.value1 = SET_SRC (set);
      cc_status.value2 = SET_DEST (set);
      break;

    case CC_CLOBBER:
      /* Insn doesn't leave CC in a usable state.  */
      CC_STATUS_INIT;
      break;
    }
}



/*********************************************************************/

/*
     Next two for:
     (set (reg:xx)
     	  (mem:xx (reg:xx))
 
*/

int
regsi_ok_safe(operands)
rtx operands[];
{
  rtx dest = operands[0];
  rtx areg;
  int src_reg;
  int dst_reg;

  if(operands[2]) areg = XEXP (operands[2],0);
  else
    areg = XEXP (operands[1],0);

  if(GET_CODE(dest)==MEM)
    {
      dest = XEXP (operands[0],0);
      if(GET_CODE(dest)==PLUS && GET_CODE(XEXP (dest,0))==REG)
        {
          dest = XEXP (dest,0);
        }
      else if(GET_CODE(dest)==REG)
        {
          ; /* register */
        }
      else
        return 1;
    }


  if (REGNO (dest) >= FIRST_PSEUDO_REGISTER
      || REGNO (areg) >= FIRST_PSEUDO_REGISTER ) return 1;

  dst_reg = true_regnum (dest);
  src_reg = true_regnum (areg);
  if(dst_reg > src_reg || dst_reg+1<src_reg)
    {
      return 1;
    }
  return 0;
}

int
regsi_ok_clobber(operands)
rtx operands[];
{
  rtx dest = operands[0];
  rtx areg = XEXP (operands[2],0);
  int src_reg;
  int dst_reg;
  int regno = REGNO (dest);


  if(GET_CODE(dest)==MEM)
    {
      dest = XEXP (operands[0],0);
      if(GET_CODE(dest)==PLUS && GET_CODE(XEXP (dest,0))==REG)
        {
          dest = XEXP (dest,0);
        }
      else if(GET_CODE(dest)==REG)
        {
          ; /* register */
        }
      else
        return 1;
    }

  if (regno >= FIRST_PSEUDO_REGISTER
      || REGNO (areg) >= FIRST_PSEUDO_REGISTER ) return 1;

  dst_reg = true_regnum (dest);
  src_reg = true_regnum (areg);
  if(dst_reg+1 == src_reg) return 1;
  return 0;
}

int
regdi_ok_safe(operands)
rtx operands[];
{
  rtx dest = operands[0];
  rtx areg = XEXP (operands[2],0);
  int src_reg;
  int dst_reg;


  if(GET_CODE(dest)==MEM)
    {
      dest = XEXP (operands[0],0);
      if(GET_CODE(dest)==PLUS && GET_CODE(XEXP (dest,0))==REG)
        {
          dest = XEXP (dest,0);
        }
      else if(GET_CODE(dest)==REG)
        {
          ; /* register */
        }
      else
        return 1;
    }

  if (REGNO (dest) >= FIRST_PSEUDO_REGISTER
      || REGNO (areg) >= FIRST_PSEUDO_REGISTER ) return 1;

  dst_reg = true_regnum (dest);
  src_reg = true_regnum (areg);
  if(dst_reg > src_reg || dst_reg+3<src_reg)
    {
      return 1;
    }

  return 0;
}

int
regdi_ok_clobber(operands)
rtx operands[];
{
  rtx dest = operands[0];
  rtx areg = XEXP (operands[2],0);
  int src_reg;
  int dst_reg;
  int regno = REGNO (dest);

  if(GET_CODE(dest)==MEM)
    {
      dest = XEXP (operands[0],0);
      if(GET_CODE(dest)==PLUS && GET_CODE(XEXP (dest,0))==REG)
        {
          dest = XEXP (dest,0);
        }
      else if(GET_CODE(dest)==REG)
        {
          ; /* register */
        }
      else
        return 1;
    }

  if (regno >= FIRST_PSEUDO_REGISTER
      || REGNO (areg) >= FIRST_PSEUDO_REGISTER ) return 1;

  dst_reg = true_regnum (dest);
  src_reg = true_regnum (areg);
  if(dst_reg+3 == src_reg) return 1;
  return 0;
}


/***************** ARITHMETIC *******************/

int
emit_indexed_arith(insn, operands, m, cmd, iscarry)
rtx insn;
rtx operands[];
int m;
const char *cmd;
int iscarry;
{
  char template[256];
  register int i = 0;
  char *p;
  rtx reg = NULL;
  int len = m*2;
  rtx x = operands[0];
  int havestop = 0;

  if(operands[2])
    reg = XEXP (operands[2],0);
  else
    reg = XEXP (operands[1],0);

  if(GET_CODE(x) == REG)
    {
      int src;	//   = REGNO(reg);
      int dst = REGNO(x);

      if(!reg)
        {
          reg = XEXP (operands[1],0);
        }

      src = REGNO(reg);

      /* check if registers overlap */
      if(dst>src || (dst+m-1)<src)
        {
          ; /* fine !*/
        }
      else if( (dst+m-1) == src )
        {
          havestop = 1; /* worse */
        }
      else
        {
          /* cannot do reverse assigment */
          while(i< m)
            {
              p = (char *)(template + strlen(cmd));
              p += (i&&iscarry)?3:2;
              strcpy(template,cmd);
              strcat(template, (i&&iscarry)?"c\t%":"\t%");
              *p = 'A' + i;
              p++;
              *p = 0;
              strcat(template,"0, %");
              p += 2;
              *p = 'A' + i;
              p++;
              *p = 0;
              strcat(template,operands[2]?"2":"1");
              output_asm_insn (template, operands);
              i++;
            }
          return m*3;
        }
    }

  while(i< (m-havestop) )
    {
      p = template + strlen(cmd);

      strcpy(template,cmd);

      if(i && iscarry)
        {
          strcat(template,"c\t");
          p += 2;
        }
      else
        {
          strcat(template,"\t");
          p += 1;
        }
      strcat(template,operands[2]?"@%E2+, %":"@%E1+, %");
      p += 8;
      *p = 'A' + i;
      p++;
      *p = 0;
      strcat(template,"0");
      p += 1;

      /**** comments make thing worse!!! **************
              if(!i)
          {
                  if(operands[2])
                  {
                      strcat(template,"\t;\t%0 = %1 ");
                      strcat(template,cmd);
                      strcat(template," %2");
                  }
                  else if(operands[1])
                  {
                      strcat(template,"\t;\t%0 = ");
                      strcat(template,cmd);
                      strcat(template," %1");
                  }
                  else
                  {
                      strcat(template,"\t;\t");
                      strcat(template,cmd);
                      strcat(template," %0");
                  }
          }
      *************************************************/
      output_asm_insn (template, operands);
      i++;
    }

  if(havestop)
    {
      len++;
      p = template + strlen(cmd);
      strcpy(template,cmd);
      if(i && iscarry)
        {
          strcat(template,"c\t");
          p += 2;
        }
      else
        {
          strcat(template,"\t");
          p += 1;
        }
      strcat(template,operands[2]?"@%E2, %":"@%E1, %");
      p += 8;
      *p = 'A' + i;
      p++;
      *p = 0;
      strcat(template,"0	;	register won't die");
      p += 1;
      output_asm_insn (template, operands);
    }

  if(!dead_or_set_p (insn, reg) && !havestop)
    {
      len++;
      p = template + 3;
      strcpy(template,"sub");
      strcat(template,"\t#");
      p += 2;
      *p = '0' + m*2;
      p++;
      *p = 0;

      if(operands[2] )
        strcat(template,",    %E2	;	restore %E2");
      else
        strcat(template,",    %E1	;	restore %E1");
      output_asm_insn (template, operands);
    }

  return len;
}

int
sameoperand(operands, i)
rtx operands[];
int i;
{
  rtx dst = operands[0];
  rtx src = operands[i];

  return src->fld[0].rtx == dst->fld[0].rtx;
}

#define OUT_INSN(x,p,o) \
do {                            \
if(!x) output_asm_insn (p,o);   \
} while(0)



/************** MOV CODE *********************************/

const char *
movstrsi_insn(insn, operands, l)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{

  /* operands 0 and 1 are registers !!! */
  /* operand 2 is a cnt and not zero */
  output_asm_insn ("\n.Lmsn%=:",operands);
  output_asm_insn ("mov.b\t@%1+,0(%0)",operands);
  output_asm_insn ("inc\t%0",operands);
  output_asm_insn ("dec\t%2",operands);
  output_asm_insn ("jnz\t.Lmsn%=",operands);

  return "";
}


const char *
clrstrsi_insn(insn, operands, l)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{

  /* operand 0 is a register !!! */
  /* operand 1 is a cnt and not zero */
  output_asm_insn ("\n.Lcsn%=:",operands);
  output_asm_insn ("clr.b\t0(%0)	;	clr does not support @rn+",operands);
  output_asm_insn ("inc\t%0",operands);
  output_asm_insn ("dec\t%1",operands);
  output_asm_insn ("jnz\t.Lcsn%=",operands);
  return "";

}


const char *
movstrhi_insn(insn, operands, l)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{

  /* operands 0 and 1 are registers !!! */
  /* operand 2 is a cnt and not zero */
  output_asm_insn ("\n.Lmsn%=:",operands);
  output_asm_insn ("mov.b\t@%1+,0(%0)",operands);
  output_asm_insn ("inc\t%0",operands);
  output_asm_insn ("dec\t%2",operands);
  output_asm_insn ("jnz\t.Lmsn%=",operands);
  return "";

}


const char *
clrstrhi_insn(insn, operands, l)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{

  /* operand 0 is a register !!! */
  /* operand 1 is a cnt and not zero */
  output_asm_insn ("\n.Lcsn%=:",operands);
  output_asm_insn ("clr.b\t0(%0)	;	clr does not support @rn+",operands);
  output_asm_insn ("inc\t%0",operands);
  output_asm_insn ("dec\t%1",operands);
  output_asm_insn ("jnz\t.Lcsn%=",operands);
  return "";

}


const char *
emit_mov_memmemR(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[] ATTRIBUTE_UNUSED;
int *len ATTRIBUTE_UNUSED;
{
  return "bug";
}

const char *
emit_mov_memmemM(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[] ATTRIBUTE_UNUSED;
int *len ATTRIBUTE_UNUSED;
{
  return "bug";
}

const char *
emit_mov_memmemRHI(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[] ATTRIBUTE_UNUSED;
int *len ATTRIBUTE_UNUSED;
{
  return "bug";
}

const char *
emit_mov_memmemMHI(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[] ATTRIBUTE_UNUSED;
int *len ATTRIBUTE_UNUSED;
{
  return "bug";
}


const char *
emit_mov_memmemRSI(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[] ATTRIBUTE_UNUSED;
int *len ATTRIBUTE_UNUSED;
{
  return "bug";
}

const char *
emit_mov_memmemMSI(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[] ATTRIBUTE_UNUSED;
int *len ATTRIBUTE_UNUSED;
{
  return "bug";
}


int
msp430_emit_indexed_mov(insn, operands, m, cmd)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int m;
const char *cmd;
{
  char template[256];
  register int i = 0;
  char *p;
  rtx reg = XEXP (operands[1],0);
  int len = m*2;

  while(i<m)
    {
      p = template + strlen(cmd);

      strcpy(template,cmd);
      strcat(template,"\t");
      p += 1;
      strcat(template,"@%E1+, %");
      p += 8;
      *p = 'A' + i;
      p++;
      *p = 0;
      strcat(template,"0");
      p += 1;
      output_asm_insn (template, operands);
      i++;
    }

  if(!dead_or_set_p (insn, reg))
    {
      len++;
      p = template + 3;
      strcpy(template,"sub");
      strcat(template,"\t#");
      p += 2;
      *p = '0' + m*2;
      p++;
      *p = 0;
      strcat(template,",    %E1	;	restore %E1");
      output_asm_insn (template, operands);
    }

  return len;
}

const char *
msp430_emit_indexed_mov2(insn, operands, l)
rtx insn;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  msp430_emit_indexed_mov(insn, operands, 2, "mov");

  return "";
}

const char *
msp430_emit_indexed_mov4(insn, operands, l)
rtx insn;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  msp430_emit_indexed_mov(insn, operands, 4, "mov");
  return "";
}

const char *
movsisf_regmode(insn, operands, l)
rtx insn;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  rtx dest = operands[0];
  rtx src = operands[1];
  rtx areg = XEXP (src,0);
  int src_reg = true_regnum (areg);
  int dst_reg = true_regnum (dest);


  if(dst_reg > src_reg || dst_reg+1<src_reg)
    {
      output_asm_insn("mov\t@%E1+, %A0", operands);
      output_asm_insn("mov\t@%E1+, %B0", operands);
      if(!dead_or_set_p (insn, areg))
        {
          output_asm_insn("sub\t#4, %E1\t;\trestore %E1", operands);
        }
      return "";
    }
  else if(dst_reg+1 == src_reg)
    {
      output_asm_insn("mov\t@%E1+, %A0", operands);
      output_asm_insn("mov\t@%E1, %B0", operands);
      return "";
    }
  else
    {
      /* destination overlaps with source.
         so, update destination in reverse way */
      output_asm_insn("mov\t%B1, %B0", operands);
      output_asm_insn("mov\t@%E1, %A0", operands);
    }

  return "";	/* make compiler happy */

}


const char *
msp430_movesi_code(insn,operands,len)
rtx insn;
rtx operands[];
int *len;
{
  rtx op0 = operands[0];
  rtx op1 = operands[1];


  if(memory_operand(op0, VOIDmode)
      && memory_operand(op1, VOIDmode)
      && zero_shifted(op1) )
    {
      if(!len) msp430_emit_indexed_mov2(insn, operands, NULL);
      else *len = 5;
      return "";
    }
  else if (register_operand (op0, SImode)
           && memory_operand(op1, SImode)
           && zero_shifted(op1) )
    {
      if(!len) movsisf_regmode(insn,operands, NULL);
      else *len = 3;
      return "";
    }


  if(!len)
    {
      if(register_operand (op0, SImode)
          && register_operand(op1, SImode)
          && REGNO(op1) + 1 == REGNO(op0))
        {
          output_asm_insn("mov\t%B1, %B0", operands);
          output_asm_insn("mov\t%A1, %A0", operands);
        }
      else
        {
          output_asm_insn("mov\t%A1, %A0", operands);
          output_asm_insn("mov\t%B1, %B0", operands);
        }
    }
  else
    {
      *len = 2;	/* base length */

      if( register_operand (op0, VOIDmode) ) *len += 0;
      else if(memory_operand(op0, VOIDmode) ) *len += 2;

      if( register_operand (op1, VOIDmode) ) *len += 0;
      else if(memory_operand(op1, VOIDmode) ) *len += 2;
      else if(immediate_operand(op1, VOIDmode) ) *len += 2;
    }

  return "";
}



const char *
movdidf_regmode(insn, operands, l)
rtx insn;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  rtx dest = operands[0];
  rtx src = operands[1];
  rtx areg = XEXP (src,0);

  int src_reg = true_regnum (areg);
  int dst_reg = true_regnum (dest);


  if(dst_reg > src_reg || dst_reg+3<src_reg )
    {
      output_asm_insn("mov\t@%E1+, %A0", operands);
      output_asm_insn("mov\t@%E1+, %B0", operands);
      output_asm_insn("mov\t@%E1+, %C0", operands);
      output_asm_insn("mov\t@%E1+, %D0", operands);
      if(!dead_or_set_p (insn, areg))
        {
          output_asm_insn("sub\t#8, %E1\t;\trestore %E1", operands);
        }
    }
  else if(dst_reg+3 == src_reg)
    {
      output_asm_insn("mov\t@%E1+, %A0",operands);
      output_asm_insn("mov\t@%E1+, %B0",operands);
      output_asm_insn("mov\t@%E1+, %C0",operands);
      output_asm_insn("mov\t@%E1,  %D0	;	%E1 == %D0",operands);
    }
  else
    {
      /* destination overlaps with source.
         so, update destination in reverse way */
      output_asm_insn("mov\t%D1, %D0	; %E1 overlaps wit one of %A0 - %D0",operands);
      output_asm_insn("mov\t%C1, %C0",operands);
      output_asm_insn("mov\t%B1, %B0",operands);
      output_asm_insn("mov\t@%E1, %A0",operands);
    }

  return "";
}


const char *
msp430_movedi_code(insn,operands,len)
rtx insn;
rtx operands[];
int *len;
{
  rtx op0 = operands[0];
  rtx op1 = operands[1];


  if(memory_operand(op0, DImode)
      && memory_operand(op1, DImode)
      && zero_shifted(op1) )
    {
      if(!len) msp430_emit_indexed_mov4(insn, operands, NULL);
      else *len = 9;
      return "";
    }
  else if (register_operand (op0, DImode)
           && memory_operand(op1, DImode)
           && zero_shifted(op1) )
    {
      if(!len) movdidf_regmode(insn,operands, NULL);
      else *len = 5;
      return "";
    }

  if(!len)
    {
      if(register_operand (op0, SImode)
          && register_operand(op1, SImode)
          && REGNO(op1) + 3 == REGNO(op0))
        {
          output_asm_insn("mov\t%D1, %D0", operands);
          output_asm_insn("mov\t%C1, %C0", operands);
          output_asm_insn("mov\t%B1, %B0", operands);
          output_asm_insn("mov\t%A1, %A0", operands);
        }
      else
        {
          output_asm_insn("mov\t%A1, %A0", operands);
          output_asm_insn("mov\t%B1, %B0", operands);
          output_asm_insn("mov\t%C1, %C0", operands);
          output_asm_insn("mov\t%D1, %D0", operands);
        }
    }
  else
    {
      *len = 4;	/* base length */

      if( register_operand (op0, DImode) ) *len += 0;
      else if(memory_operand(op0, DImode) ) *len += 4;

      if( register_operand (op1, DImode) ) *len += 0;
      else if(memory_operand(op1, DImode) ) *len += 4;
      else if(immediate_operand(op1, DImode) ) *len += 4;
    }

  return "";
}




/**************	ADD CODE *********************************/


const char *
msp430_emit_indexed_add2(insn, operands, l)
rtx insn;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  emit_indexed_arith(insn, operands, 2, "add",1);

  return "";
}

const char *
msp430_emit_indexed_add4(insn, operands, l)
rtx insn;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  emit_indexed_arith(insn, operands, 4, "add",1);

  return "";
}


const char *
msp430_addsi_code(insn, operands, len)
rtx insn;
rtx operands[];
int *len;
{
  rtx op0 = operands[0];
  rtx op2 = operands[2];

  if( memory_operand(op2, SImode)
      && zero_shifted(operands[2])
      && regsi_ok_safe(operands) )
    {
      if(!len)
        msp430_emit_indexed_add2(insn, operands, NULL);
      else
        {
          if(memory_operand(op0, SImode) ) *len = 5;
          else if(register_operand(op0, SImode) ) *len = 3;
        }

      return "";
    }
  else if( memory_operand(op2, SImode)
           && zero_shifted(operands[2])
           && regsi_ok_clobber(operands) )
    {
      if(!len)
        {
          output_asm_insn("add\t@%E2+, %A0", operands);
          output_asm_insn("addc\t@%E2+, %B0", operands);
        }
      else
        {
          if(register_operand(op0, SImode) ) *len = 2;
          else if(memory_operand(op0, SImode)) *len = 4;
          else
            abort();
        }
      return "";
    }

  if(!len)
    {
      output_asm_insn("add\t%A2, %A0", operands);
      output_asm_insn("addc\t%B2, %B0", operands);
    }
  else
    {
      *len = 2;	/* base length */

      if( register_operand (op0, SImode) ) *len += 0;
      else if(memory_operand(op0, SImode) ) *len += 2;

      if( register_operand (op2, SImode) ) *len += 0;
      else if(memory_operand(op2, SImode) ) *len += 2;
      else if(immediate_operand(op2, SImode) )
        {
          int x = INTVAL(op2);
          if(x == -2 || x==-4 || x == -8)
            {
              *len += 1;
            }
          else
            *len += 2;
        }
    }

  return "";
}


const char *
msp430_adddi_code(insn, operands, len)
rtx insn;
rtx operands[];
int *len;
{
  rtx op0 = operands[0];
  rtx op2 = operands[2];

  if( memory_operand(op2, DImode)
      && zero_shifted(operands[2])
      && regdi_ok_safe(operands) )
    {
      if(!len)
        msp430_emit_indexed_add4(insn, operands, NULL);
      else
        {
          if(memory_operand(op0, DImode) ) *len = 9;
          else if(register_operand(op0, DImode) ) *len = 5;
        }

      return "";
    }
  else if( memory_operand(op2, DImode)
           && zero_shifted(operands[2])
           && regdi_ok_clobber(operands) )
    {
      if(!len)
        {
          output_asm_insn("add\t@%E2+, %A0", operands);
          output_asm_insn("addc\t@%E2+, %B0", operands);
          output_asm_insn("addc\t@%E2+, %C0", operands);
          output_asm_insn("addc\t@%E2+, %D0", operands);
        }
      else
        {
          if(register_operand(op0, DImode) ) *len = 4;
          else if(memory_operand(op0, DImode)) *len = 8;
          else
            abort();
        }
      return "";
    }

  if(!len)
    {
      output_asm_insn("add\t%A2, %A0", operands);
      output_asm_insn("addc\t%B2, %B0", operands);
      output_asm_insn("addc\t%C2, %C0", operands);
      output_asm_insn("addc\t%D2, %D0", operands);
    }
  else
    {
      *len = 4;	/* base length */

      if( register_operand (op0, DImode) ) *len += 0;
      else if(memory_operand(op0, DImode) ) *len += 4;

      if( register_operand (op2, DImode) ) *len += 0;
      else if(memory_operand(op2, DImode) ) *len += 4;
      else if(immediate_operand(op2, DImode) )
        {
          int x = INTVAL(op2);

          if(x == -2 || x == -4 || x == -8)
            *len += 0;
          else
            *len += 4;
        }
      else
        abort();
    }

  return "";
}




/**************	SUB CODE *********************************/


const char *
msp430_emit_indexed_sub2(insn, operands, l)
rtx insn;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  emit_indexed_arith(insn, operands, 2, "sub",1);

  return "";
}

const char *
msp430_emit_indexed_sub4(insn, operands, l)
rtx insn;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  emit_indexed_arith(insn, operands, 4, "sub",1);

  return "";
}



const char *
msp430_subsi_code(insn, operands, len)
rtx insn;
rtx operands[];
int *len;
{
  rtx op0 = operands[0];
  rtx op2 = operands[2];

  if( memory_operand(op2, SImode)
      && zero_shifted(operands[2])
      && regsi_ok_safe(operands) )
    {
      if(!len)
        msp430_emit_indexed_sub2(insn, operands, NULL);
      else
        {
          if(memory_operand(op0, SImode) ) *len = 5;
          else if(register_operand(op0, SImode) ) *len = 3;
        }

      return "";
    }
  else if( memory_operand(op2, SImode)
           && zero_shifted(operands[2])
           && regsi_ok_clobber(operands) )
    {
      if(!len)
        {
          output_asm_insn("sub\t@%E2+, %A0", operands);
          output_asm_insn("subc\t@%E2+, %B0", operands);
        }
      else
        {
          if(register_operand(op0, SImode) ) *len = 2;
          else if(memory_operand(op0, SImode)) *len = 4;
          else
            abort();
        }
      return "";
    }

  if(!len)
    {
      output_asm_insn("sub\t%A2, %A0", operands);
      output_asm_insn("subc\t%B2, %B0", operands);
    }
  else
    {
      *len = 2;	/* base length */

      if( register_operand (op0, SImode) ) *len += 0;
      else if(memory_operand(op0, SImode) ) *len += 2;

      if( register_operand (op2, SImode) ) *len += 0;
      else if(memory_operand(op2, SImode) ) *len += 2;
      else if(immediate_operand(op2, SImode) ) *len += 2;
    }

  return "";
}


const char *
msp430_subdi_code(insn, operands, len)
rtx insn;
rtx operands[];
int *len;
{
  rtx op0 = operands[0];
  rtx op2 = operands[2];

  if( memory_operand(op2, DImode)
      && zero_shifted(operands[2])
      && regdi_ok_safe(operands) )
    {
      if(!len)
        msp430_emit_indexed_sub4(insn, operands, NULL);
      else
        {
          if(memory_operand(op0, DImode) ) *len = 9;
          else if(register_operand(op0, DImode) ) *len = 5;
        }

      return "";
    }
  else if( memory_operand(op2, DImode)
           && zero_shifted(operands[2])
           && regdi_ok_clobber(operands) )
    {
      if(!len)
        {
          output_asm_insn("sub\t@%E2+, %A0", operands);
          output_asm_insn("subc\t@%E2+, %B0", operands);
          output_asm_insn("subc\t@%E2+, %C0", operands);
          output_asm_insn("subc\t@%E2+, %D0", operands);
        }
      else
        {
          if(register_operand(op0, DImode) ) *len = 4;
          else if(memory_operand(op0, DImode)) *len = 8;
          else
            abort();
        }
      return "";
    }

  if(!len)
    {
      output_asm_insn("sub\t%A2, %A0", operands);
      output_asm_insn("subc\t%B2, %B0", operands);
      output_asm_insn("subc\t%C2, %C0", operands);
      output_asm_insn("subc\t%D2, %D0", operands);
    }
  else
    {
      *len = 4;	/* base length */

      if( register_operand (op0, DImode) ) *len += 0;
      else if(memory_operand(op0, DImode) ) *len += 4;

      if( register_operand (op2, DImode) ) *len += 0;
      else if(memory_operand(op2, DImode) ) *len += 4;
      else if(immediate_operand(op2, DImode) ) *len += 4;
      else
        abort();
    }

  return "";
}




/**************	AND CODE *********************************/

const char *
msp430_emit_indexed_and2(insn, operands, l)
rtx insn;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  emit_indexed_arith(insn, operands, 2, "and",0);
  return "";
}

const char *
msp430_emit_indexed_and4(insn, operands, l)
rtx insn;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  emit_indexed_arith(insn, operands, 4, "and",0);
  return "";
}

const char *
msp430_emit_immediate_and2(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int v;
  int l = INTVAL(operands[2]);
  int r = REG_P(operands[0]);
  int list1 = ((~1) & 0xffff);
  int list2 = ((~2) & 0xffff);
  int list4 = ((~4) & 0xffff);
  int list8 = ((~8) & 0xffff);

  rtx op[4];

  op[0] = operands[0];
  op[1] = operands[1];
  op[2] = operands[2];

  /* check nibbles */

  v = (l)&0xffff;
  if(v!=0xffff)
    {
      if(v == list1 || v==list2 || v==list4 || v==list8)
        {
          op[2] = gen_rtx_CONST_INT(SImode,~v);
          OUT_INSN(len,"bic\t%A2, %A0", op);
          dummy++;
          if(!r) dummy++;
        }
      else
        {
          op[2] = gen_rtx_CONST_INT(SImode,v);
          OUT_INSN(len,"and\t%A2, %A0", op);
          dummy++;
          dummy++;
          if(!r) dummy++;
          if(v==0||v==1||v==2||v==4||v==8) dummy--;
        }
    }

  v = (l>>16)&0xffff;
  if(v!=0xffff)
    {
      if(v == list1 || v==list2 || v==list4 || v==list8)
        {
          op[2] = gen_rtx_CONST_INT(SImode,~v);
          OUT_INSN(len,"bic\t%A2, %B0", op);
          dummy++;
          if(!r) dummy++;
        }
      else
        {
          op[2] = gen_rtx_CONST_INT(SImode,v);
          OUT_INSN(len,"and\t%A2, %B0", op);
          dummy++;
          dummy++;
          if(!r) dummy++;
          if(v==0||v==1||v==2||v==4||v==8) dummy--;
        }
    }

  if(len) *len = dummy;
  return "";
}



const char *
msp430_emit_immediate_and4(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int v;
  int l = CONST_DOUBLE_LOW(operands[2]);
  int h = CONST_DOUBLE_HIGH(operands[2]);
  int r = REG_P(operands[0]);
  int list1 = ((~1) & 0xffff);
  int list2 = ((~2) & 0xffff);
  int list4 = ((~4) & 0xffff);
  int list8 = ((~8) & 0xffff);
  rtx op[4];

  op[0] = operands[0];
  op[1] = operands[1];
  op[2] = operands[2];

  /* check if operand 2 is really const_double */
  
  if(GET_CODE(operands[2]) == CONST_INT)
  {
  	l = INTVAL(operands[2]);
  	h = 0;
  }
  
  /* check nibbles */

  v = (l)&0xffff;
  if(v!=0xffff)
    {
      if(v == list1 || v==list2 || v==list4 || v==list8)
        {
          op[2] = gen_rtx_CONST_INT(SImode,~v);
          OUT_INSN(len,"bic\t%A2, %A0", op);
          dummy++;
          if(!r) dummy++;
        }
      else
        {
          op[2] = gen_rtx_CONST_INT(SImode,v);
          OUT_INSN(len,"and\t%A2, %A0", op);
          dummy++;
          dummy++;
          if(!r) dummy++;
          if(v==0||v==1||v==2||v==4||v==8) dummy--;
        }
    }

  v = (l>>16)&0xffff;
  if(v!=0xffff)
    {
      if(v == list1 || v==list2 || v==list4 || v==list8)
        {
          op[2] = gen_rtx_CONST_INT(SImode,~v);
          OUT_INSN(len,"bic\t%A2, %B0", op);
          dummy++;
          if(!r) dummy++;
        }
      else
        {
          op[2] = gen_rtx_CONST_INT(SImode,v);
          OUT_INSN(len,"and\t%A2, %B0", op);
          dummy++;
          dummy++;
          if(!r) dummy++;
          if(v==0||v==1||v==2||v==4||v==8) dummy--;
        }
    }


  v = (h)&0xffff;
  if(v!=0xffff)
    {
      if(v == list1 || v==list2 || v==list4 || v==list8)
        {
          op[2] = gen_rtx_CONST_INT(SImode,~v);
          OUT_INSN(len,"bic\t%A2, %C0", op);
          dummy++;
          if(!r) dummy++;
        }
      else
        {
          op[2] = gen_rtx_CONST_INT(SImode,v);
          OUT_INSN(len,"and\t%A2, %C0", op);
          dummy++;
          dummy++;
          if(!r) dummy++;
          if(v==0||v==1||v==2||v==4||v==8) dummy--;
        }
    }

  v = (h>>16)&0xffff;
  if(v!=0xffff)
    {
      if(v == list1 || v==list2 || v==list4 || v==list8)
        {
          op[2] = gen_rtx_CONST_INT(SImode,~v);
          OUT_INSN(len,"bic\t%A2, %D0", op);
          dummy++;
          if(!r) dummy++;
        }
      else
        {
          op[2] = gen_rtx_CONST_INT(SImode,v);
          OUT_INSN(len,"and\t%A2, %D0", op);
          dummy++;
          dummy++;
          if(!r) dummy++;
          if(v==0||v==1||v==2||v==4||v==8) dummy--;
        }
    }

  if(len) *len = dummy;
  return "";

}
const char *
msp430_andsi_code(insn, operands, len)
rtx insn;
rtx operands[];
int *len;
{
  rtx op0 = operands[0];
  rtx op2 = operands[2];

  if(nonimmediate_operand (op0, SImode)
      && immediate_operand(op2, SImode) )
    {
      if(!len) msp430_emit_immediate_and2(insn, operands, NULL);
      return "";
    }


  if( memory_operand(op2, SImode)
      && zero_shifted(operands[2])
      && regsi_ok_safe(operands) )
    {
      if(!len)
        msp430_emit_indexed_and2(insn, operands, NULL);
      else
        {
          if(memory_operand(op0, SImode) ) *len = 5;
          else if(register_operand(op0, SImode) ) *len = 3;
        }

      return "";
    }
  else if( memory_operand(op2, SImode)
           && zero_shifted(operands[2])
           && regsi_ok_clobber(operands) )
    {
      if(!len)
        {
          output_asm_insn("and\t@%E2+, %A0", operands);
          output_asm_insn("and\t@%E2+, %B0", operands);
        }
      else
        {
          if(register_operand(op0, SImode) ) *len = 2;
          else if(memory_operand(op0, SImode)) *len = 4;
          else
            abort();
        }
      return "";
    }

  if(!len)
    {
      output_asm_insn("and\t%A2, %A0", operands);
      output_asm_insn("and\t%B2, %B0", operands);
    }
  else
    {
      *len = 2;	/* base length */

      if( register_operand (op0, SImode) ) *len += 0;
      else if(memory_operand(op0, SImode) ) *len += 2;

      if( register_operand (op2, SImode) ) *len += 0;
      else if(memory_operand(op2, SImode) ) *len += 2;
      else if(immediate_operand(op2, SImode) ) *len += 2;
    }

  return "";
}


const char *
msp430_anddi_code(insn, operands, len)
rtx insn;
rtx operands[];
int *len;
{
  rtx op0 = operands[0];
  rtx op2 = operands[2];


  if(nonimmediate_operand (op0, DImode)
      && immediate_operand(op2, DImode) )
    {
      if(!len) msp430_emit_immediate_and4(insn, operands, NULL);
      return "";
    }



  if( memory_operand(op2, DImode)
      && zero_shifted(operands[2])
      && regdi_ok_safe(operands) )
    {
      if(!len)
        msp430_emit_indexed_and4(insn, operands, NULL);
      else
        {
          if(memory_operand(op0, DImode) ) *len = 9;
          else if(register_operand(op0, DImode) ) *len = 5;
        }

      return "";
    }
  else if( memory_operand(op2, DImode)
           && zero_shifted(operands[2])
           && regdi_ok_clobber(operands) )
    {
      if(!len)
        {
          output_asm_insn("and\t@%E2+, %A0", operands);
          output_asm_insn("and\t@%E2+, %B0", operands);
          output_asm_insn("and\t@%E2+, %C0", operands);
          output_asm_insn("and\t@%E2+, %D0", operands);
        }
      else
        {
          if(register_operand(op0, DImode) ) *len = 4;
          else if(memory_operand(op0, DImode)) *len = 8;
          else
            abort();
        }
      return "";
    }

  if(!len)
    {
      output_asm_insn("and\t%A2, %A0", operands);
      output_asm_insn("and\t%B2, %B0", operands);
      output_asm_insn("and\t%C2, %C0", operands);
      output_asm_insn("and\t%D2, %D0", operands);
    }
  else
    {
      *len = 4;	/* base length */

      if( register_operand (op0, DImode) ) *len += 0;
      else if(memory_operand(op0, DImode) ) *len += 4;

      if( register_operand (op2, DImode) ) *len += 0;
      else if(memory_operand(op2, DImode) ) *len += 4;
      else if(immediate_operand(op2, DImode) ) *len += 4;
      else
        abort();
    }

  return "";
}








/**************	IOR CODE *********************************/

const char *
msp430_emit_indexed_ior2(insn, operands, l)
rtx insn;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  emit_indexed_arith(insn, operands, 2, "bis",0);
  return "";
}

const char *
msp430_emit_indexed_ior4(insn, operands, l)
rtx insn;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  emit_indexed_arith(insn, operands, 4, "bis",0);
  return "";
}

const char *
msp430_emit_immediate_ior2(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int l = INTVAL(operands[2]);
  int r = REG_P(operands[0]);
  int v;


  v = l&0xffff;

  if(v)
    {
      OUT_INSN(len,"bis\t%A2,%A0",operands);
      dummy++;
      dummy++;
      if(v==0xffff||v==1||v==2||v==4||v==8)dummy--;
      if(!r) dummy++;
    }

  v = (l>>16)&0xffff;

  if(v)
    {
      OUT_INSN(len,"bis\t%B2,%B0",operands);
      dummy++;
      dummy++;
      if(v==0xffff||v==1||v==2||v==4||v==8)dummy--;
      if(!r) dummy++;
    }


  if(len) *len=dummy;
  return "";
}



const char *
msp430_emit_immediate_ior4(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int l = CONST_DOUBLE_LOW(operands[2]);
  int h = CONST_DOUBLE_HIGH(operands[2]);
  int r = REG_P(operands[0]);
  int v;

  if(GET_CODE(operands[2]) == CONST_INT)
  {
  	l = INTVAL(operands[2]);
  	h = 0;
  }

  v = l&0xffff;

  if(v)
    {
      OUT_INSN(len,"bis\t%A2,%A0",operands);
      dummy++;
      dummy++;
      if(v==0xffff||v==1||v==2||v==4||v==8) dummy--;
      if(!r) dummy++;
    }

  v = (l>>16)&0xffff;

  if(v)
    {
      OUT_INSN(len,"bis\t%B2,%B0",operands);
      dummy++;
      dummy++;
      if(v==0xffff||v==1||v==2||v==4||v==8) dummy--;
      if(!r) dummy++;
    }

  l = h;
  v = l&0xffff;

  if(v)
    {
      OUT_INSN(len,"bis\t%C2,%C0",operands);
      dummy++;
      dummy++;
      if(v==0xffff||v==1||v==2||v==4||v==8) dummy--;
      if(!r) dummy++;
    }

  v = (l>>16)&0xffff;

  if(v)
    {
      OUT_INSN(len,"bis\t%D2,%D0",operands);
      dummy++;
      dummy++;
      if(v==0xffff||v==1||v==2||v==4||v==8) dummy--;
      if(!r) dummy++;
    }

  if(len) *len=dummy;
  return "";
}


const char *
msp430_iorsi_code(insn, operands, len)
rtx insn;
rtx operands[];
int *len;
{
  rtx op0 = operands[0];
  rtx op2 = operands[2];


  if(nonimmediate_operand (op0, SImode)
      && immediate_operand(op2, SImode) )
    {
      if(!len) msp430_emit_immediate_ior2(insn, operands, NULL);
      return "";
    }



  if( memory_operand(op2, SImode)
      && zero_shifted(operands[2])
      && regsi_ok_safe(operands) )
    {
      if(!len)
        msp430_emit_indexed_ior2(insn, operands, NULL);
      else
        {
          if(memory_operand(op0, SImode) ) *len = 5;
          else if(register_operand(op0, SImode) ) *len = 3;
        }

      return "";
    }
  else if( memory_operand(op2, SImode)
           && zero_shifted(operands[2])
           && regsi_ok_clobber(operands) )
    {
      if(!len)
        {
          output_asm_insn("bis\t@%E2+, %A0", operands);
          output_asm_insn("bis\t@%E2+, %B0", operands);
        }
      else
        {
          if(register_operand(op0, SImode) ) *len = 2;
          else if(memory_operand(op0, SImode)) *len = 4;
          else
            abort();
        }
      return "";
    }

  if(!len)
    {
      output_asm_insn("bis\t%A2, %A0", operands);
      output_asm_insn("bis\t%B2, %B0", operands);
    }
  else
    {
      *len = 2;	/* base length */

      if( register_operand (op0, SImode) ) *len += 0;
      else if(memory_operand(op0, SImode) ) *len += 2;

      if( register_operand (op2, SImode) ) *len += 0;
      else if(memory_operand(op2, SImode) ) *len += 2;
      else if(immediate_operand(op2, SImode) ) *len += 2;
    }

  return "";
}


const char *
msp430_iordi_code(insn, operands, len)
rtx insn;
rtx operands[];
int *len;
{
  rtx op0 = operands[0];
  rtx op2 = operands[2];


  if(nonimmediate_operand (op0, DImode)
      && immediate_operand(op2, DImode) )
    {
      if(!len) msp430_emit_immediate_ior4(insn, operands, NULL);
      return "";
    }


  if( memory_operand(op2, DImode)
      && zero_shifted(operands[2])
      && regdi_ok_safe(operands) )
    {
      if(!len)
        msp430_emit_indexed_ior4(insn, operands, NULL);
      else
        {
          if(memory_operand(op0, DImode) ) *len = 9;
          else if(register_operand(op0, DImode) ) *len = 5;
        }

      return "";
    }
  else if( memory_operand(op2, DImode)
           && zero_shifted(operands[2])
           && regdi_ok_clobber(operands) )
    {
      if(!len)
        {
          output_asm_insn("bis\t@%E2+, %A0", operands);
          output_asm_insn("bis\t@%E2+, %B0", operands);
          output_asm_insn("bis\t@%E2+, %C0", operands);
          output_asm_insn("bis\t@%E2+, %D0", operands);
        }
      else
        {
          if(register_operand(op0, DImode) ) *len = 4;
          else if(memory_operand(op0, DImode)) *len = 8;
          else
            abort();
        }
      return "";
    }

  if(!len)
    {
      output_asm_insn("bis\t%A2, %A0", operands);
      output_asm_insn("bis\t%B2, %B0", operands);
      output_asm_insn("bis\t%C2, %C0", operands);
      output_asm_insn("bis\t%D2, %D0", operands);
    }
  else
    {
      *len = 4;	/* base length */

      if( register_operand (op0, DImode) ) *len += 0;
      else if(memory_operand(op0, DImode) ) *len += 4;

      if( register_operand (op2, DImode) ) *len += 0;
      else if(memory_operand(op2, DImode) ) *len += 4;
      else if(immediate_operand(op2, DImode) ) *len += 4;
      else
        abort();
    }

  return "";
}








/************************* XOR CODE *****************/

const char *
msp430_emit_indexed_xor2(insn, operands, l)
rtx insn;
rtx operands[];
int *l;
{
  int dummy = emit_indexed_arith(insn, operands, 2, "xor",0);
  if(!l) l = &dummy;
  *l = dummy;
  return "";
}

const char *
msp430_emit_indexed_xor4(insn, operands, l)
rtx insn;
rtx operands[];
int *l;
{
  int dummy = emit_indexed_arith(insn, operands, 4, "xor",0);
  if(!l) l = &dummy;
  *l = dummy;
  return "";
}


const char *
msp430_emit_indexed_xor2_3(insn, operands, l)
rtx insn;
rtx operands[];
int *l;
{
  int dummy;
  rtx x = operands[2];
  if(zero_shifted(x))
    {
      dummy = emit_indexed_arith(insn, operands, 2, "xor",0);
    }
  else
    {
      dummy = 6;
      output_asm_insn ("xor\t%A2, %A0", operands);
      output_asm_insn ("xor\t%B2, %B0", operands);
    }

  if(!l) l = &dummy;
  *l = dummy;
  return "";
}

const char *
msp430_emit_indexed_xor4_3(insn, operands, l)
rtx insn;
rtx operands[];
int *l;
{

  int dummy;
  rtx x = operands[2];
  if(zero_shifted(x))
    {
      dummy = emit_indexed_arith(insn, operands, 4, "xor",0);
    }
  else
    {
      dummy = 8;
      output_asm_insn ("xor\t%A2, %A0", operands);
      output_asm_insn ("xor\t%B2, %B0", operands);
      output_asm_insn ("xor\t%C2, %C0", operands);
      output_asm_insn ("xor\t%D2, %D0", operands);
    }

  if(!l) l = &dummy;
  *l = dummy;
  return "";
}



const char *
msp430_xorsi_code(insn, operands, len)
rtx insn;
rtx operands[];
int *len;
{
  rtx op0 = operands[0];
  rtx op2 = operands[2];

  if( memory_operand(op2, SImode)
      && zero_shifted(operands[2])
      && regsi_ok_safe(operands) )
    {
      if(!len)
        msp430_emit_indexed_xor2(insn, operands, NULL);
      else
        {
          if(memory_operand(op0, SImode) ) *len = 5;
          else if(register_operand(op0, SImode) ) *len = 3;
        }

      return "";
    }
  else if( memory_operand(op2, SImode)
           && zero_shifted(operands[2])
           && regsi_ok_clobber(operands) )
    {
      if(!len)
        {
          output_asm_insn("xor\t@%E2+, %A0", operands);
          output_asm_insn("xor\t@%E2+, %B0", operands);
        }
      else
        {
          if(register_operand(op0, SImode) ) *len = 2;
          else if(memory_operand(op0, SImode)) *len = 4;
          else
            abort();
        }
      return "";
    }

  if(!len)
    {

      if(immediate_operand(op2,SImode))
        {
          if(INTVAL(op2) & 0xfffful)
            output_asm_insn("xor\t%A2, %A0", operands);

          if(INTVAL(op2) & 0xffff0000ul)
            output_asm_insn("xor\t%B2, %B0", operands);
        }
      else
        {
          output_asm_insn("xor\t%A2, %A0", operands);
          output_asm_insn("xor\t%B2, %B0", operands);
        }

    }
  else
    {
      *len = 2;	/* base length */

      if( register_operand (op0, SImode) ) *len += 0;
      else if(memory_operand(op0, SImode) ) *len += 2;

      if( register_operand (op2, SImode) ) *len += 0;
      else if(memory_operand(op2, SImode) ) *len += 2;
      else if(immediate_operand(op2, SImode) )
        {
          if(INTVAL(op2) & 0xfffful) *len += 1;
          if(INTVAL(op2) & 0xffff0000ul) *len += 1;
        }
    }

  return "";
}


const char *
msp430_xordi_code(insn, operands, len)
rtx insn;
rtx operands[];
int *len;
{
  rtx op0 = operands[0];
  rtx op2 = operands[2];

  if( memory_operand(op2, DImode)
      && zero_shifted(operands[2])
      && regdi_ok_safe(operands) )
    {
      if(!len)
        msp430_emit_indexed_xor4(insn, operands, NULL);
      else
        {
          if(memory_operand(op0, DImode) ) *len = 9;
          else if(register_operand(op0, DImode) ) *len = 5;
        }

      return "";
    }
  else if( memory_operand(op2, DImode)
           && zero_shifted(operands[2])
           && regdi_ok_clobber(operands) )
    {
      if(!len)
        {
          output_asm_insn("xor\t@%E2+, %A0", operands);
          output_asm_insn("xor\t@%E2+, %B0", operands);
          output_asm_insn("xor\t@%E2+, %C0", operands);
          output_asm_insn("xor\t@%E2+, %D0", operands);
        }
      else
        {
          if(register_operand(op0, DImode) ) *len = 4;
          else if(memory_operand(op0, DImode)) *len = 8;
          else
            abort();
        }
      return "";
    }

  if(!len)
    {
      output_asm_insn("xor\t%A2, %A0", operands);
      output_asm_insn("xor\t%B2, %B0", operands);
      output_asm_insn("xor\t%C2, %C0", operands);
      output_asm_insn("xor\t%D2, %D0", operands);
    }
  else
    {
      *len = 4;	/* base length */

      if( register_operand (op0, DImode) ) *len += 0;
      else if(memory_operand(op0, DImode) ) *len += 4;

      if( register_operand (op2, DImode) ) *len += 0;
      else if(memory_operand(op2, DImode) ) *len += 4;
      else if(immediate_operand(op2, DImode) ) *len += 4;
      else
        abort();
    }

  return "";
}





/********* ABS CODE ***************************************/
const char *
msp430_emit_abssi(insn, operands,l)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  output_asm_insn ("tst\t%B0",operands);
  output_asm_insn ("jge\t.Lae%=",operands);
  output_asm_insn ("inv\t%A0", operands);
  output_asm_insn ("inv\t%B0", operands);
  output_asm_insn ("inc\t%A0", operands);
  output_asm_insn ("adc\t%B0", operands);
  output_asm_insn (".Lae%=:",operands);
  return "";
}

const char *
msp430_emit_absdi(insn, operands,l)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *l ATTRIBUTE_UNUSED;
{
  output_asm_insn ("tst\t%D0",operands);
  output_asm_insn ("jge\t.Lae%=",operands);
  output_asm_insn ("inv\t%A0", operands);
  output_asm_insn ("inv\t%B0", operands);
  output_asm_insn ("inv\t%C0", operands);
  output_asm_insn ("inv\t%D0", operands);
  output_asm_insn ("inc\t%A0", operands);
  output_asm_insn ("adc\t%B0", operands);
  output_asm_insn ("adc\t%C0", operands);
  output_asm_insn ("adc\t%D0", operands);
  output_asm_insn (".Lae%=:",operands);
  return "";
}



/***** SIGN EXTEND *********/

const char *
signextendqihi(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[0]) || indexed_location(operands[0]);

  if(!sameoperand(operands,1))
    {
      OUT_INSN (len, "mov.b\t%A1, %A0",operands);
      dummy = 3;
      if(indexed_location(operands[1])) dummy = 2;
      if(GET_CODE(operands[0]) == REG) dummy--;
      if(GET_CODE(operands[1]) == REG) dummy--;
    }

  OUT_INSN (len, "sxt\t%A0",operands);
  dummy += 2;

  if(zs || GET_CODE(operands[0]) == REG) dummy -= 1;

  if(len) *len = dummy;

  return "";
}

const char *
signextendqisi(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[0]) || indexed_location(operands[0]);

  if(!sameoperand(operands,1))
    {
      OUT_INSN (len, "mov.b\t%A1, %A0",operands);
      dummy = 3;
      if(indexed_location(operands[1])) dummy = 2;
      if(GET_CODE(operands[0]) == REG) dummy--;
      if(GET_CODE(operands[1]) == REG) dummy--;
    }


  OUT_INSN (len, "sxt\t%A0",operands);
  OUT_INSN (len, "mov\t%A0, %B0",operands);
  OUT_INSN (len, "rla\t%B0",operands);
  OUT_INSN (len, "subc\t%B0, %B0",operands);
  OUT_INSN (len, "inv\t%B0",operands);


  if(GET_CODE(operands[0]) == REG) dummy += 5;
  else if(zs) dummy += 10;
  else dummy += 12;

  if(len) *len = dummy;

  return "";
}


const char *
signextendqidi(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[0])|| indexed_location(operands[0]);

  if(!sameoperand(operands,1))
    {
      OUT_INSN (len, "mov.b\t%A1, %A0",operands);
      dummy = 3;
      if(indexed_location(operands[1])) dummy = 2;
      if(GET_CODE(operands[0]) == REG) dummy--;
      if(GET_CODE(operands[1]) == REG) dummy--;
    }

  OUT_INSN (len, "sxt\t%A0",operands);
  OUT_INSN (len, "mov\t%A0, %B0",operands);
  OUT_INSN (len, "rla\t%B0",operands);
  OUT_INSN (len, "subc\t%B0, %B0",operands);
  OUT_INSN (len, "inv\t%B0",operands);
  OUT_INSN (len, "mov\t%B0, %C0",operands);
  OUT_INSN (len, "mov\t%C0, %D0",operands);


  if(GET_CODE(operands[0]) == REG) dummy += 7;
  else if(zs) dummy += 16;
  else dummy += 18;

  if(len) *len = dummy;

  return "";
}




const char *
signextendhisi(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[0]) || indexed_location(operands[0]);

  if(!sameoperand(operands,1))
    {
      OUT_INSN (len, "mov\t%A1, %A0",operands);
      dummy = 3;
      if(indexed_location(operands[1])) dummy = 2;
      if(GET_CODE(operands[0]) == REG) dummy--;
      if(GET_CODE(operands[1]) == REG) dummy--;
    }


  OUT_INSN (len, "mov\t%A0, %B0",operands);
  OUT_INSN (len, "rla\t%B0",operands);
  OUT_INSN (len, "subc\t%B0, %B0",operands);
  OUT_INSN (len, "inv\t%B0",operands);

  if(GET_CODE(operands[0]) == REG) dummy += 4;
  else if(zs) dummy += 9;
  else dummy += 10;

  if(len) *len = dummy;

  return "";
}


const char *
signextendhidi(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[0]) || indexed_location(operands[0]);

  if(!sameoperand(operands,1))
    {
      OUT_INSN (len, "mov\t%A1, %A0",operands);
      dummy = 3;
      if(indexed_location(operands[1])) dummy = 2;
      if(GET_CODE(operands[0]) == REG) dummy--;
      if(GET_CODE(operands[1]) == REG) dummy--;
    }


  OUT_INSN (len, "mov\t%A0, %B0",operands);
  OUT_INSN (len, "rla\t%B0",operands);
  OUT_INSN (len, "subc\t%B0, %B0",operands);
  OUT_INSN (len, "inv\t%B0",operands);
  OUT_INSN (len, "mov\t%B0, %C0",operands);
  OUT_INSN (len, "mov\t%C0, %D0",operands);

  if(GET_CODE(operands[0]) == REG) dummy += 6;
  else if(zs) dummy += 13;
  else dummy += 14;

  if(len) *len = dummy;

  return "";
}


const char *
signextendsidi(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;

  if(!sameoperand(operands,1))
    {
      OUT_INSN (len, "mov\t%A1, %A0",operands);
      OUT_INSN (len, "mov\t%B1, %B0",operands);
      dummy = 6;
      if(indexed_location(operands[1])) dummy = 4;
      if(GET_CODE(operands[0]) == REG) dummy -= 2;
      if(GET_CODE(operands[1]) == REG) dummy -= 2;
    }

  OUT_INSN (len, "mov\t%B0, %C0",operands);
  OUT_INSN (len, "rla\t%C0",operands);
  OUT_INSN (len, "subc\t%C0, %C0",operands);
  OUT_INSN (len, "inv\t%C0",operands);
  OUT_INSN (len, "mov\t%C0, %D0",operands);

  if(GET_CODE(operands[0]) == REG) dummy += 5;
  else dummy += 13;

  if(len) *len = dummy;

  return "";
}





/**** ZERO EXTEND *****/

const char *
zeroextendqihi(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[1]) || indexed_location(operands[1]);

  if(!sameoperand(operands,1))
    {
      OUT_INSN (len, "mov.b\t%A1, %A0",operands);
      dummy = 3;
      if(zs) dummy = 2;
      if(GET_CODE(operands[0]) == REG) dummy--;
      if(GET_CODE(operands[1]) == REG) dummy--;
    }

  if(!REG_P(operands[0]))
    {
      OUT_INSN (len, "clr.b\t%J0",operands);
      dummy += 2;
    }

  if(len) *len = dummy;

  return "";
}

const char *
zeroextendqisi(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[1]) || indexed_location(operands[1]);

  if(!sameoperand(operands,1))
    {
      OUT_INSN (len, "mov.b\t%A1, %A0",operands);
      dummy = 3;
      if(zs) dummy = 2;
      if(GET_CODE(operands[0]) == REG) dummy--;
      if(GET_CODE(operands[1]) == REG) dummy--;
    }


  if(!REG_P(operands[0]))
    {
      OUT_INSN (len, "clr.b\t%J0",operands);
    }

  OUT_INSN (len, "clr\t%B0",operands);
  dummy += 2;
  if(GET_CODE(operands[0]) == REG) dummy--;

  if(len) *len = dummy;

  return "";
}


const char *
zeroextendqidi(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[1]) || indexed_location(operands[1]);

  if(!sameoperand(operands,1))
    {
      OUT_INSN (len, "mov.b\t%A1, %A0",operands);
      dummy = 3;
      if(zs) dummy = 2;
      if(GET_CODE(operands[0]) == REG) dummy--;
      if(GET_CODE(operands[1]) == REG) dummy--;
    }

  if(!REG_P(operands[0]))
    {
      OUT_INSN (len, "clr.b\t%J0",operands);
      dummy += 2;
    }

  dummy += 6;
  OUT_INSN (len, "clr\t%B0",operands);
  OUT_INSN (len, "clr\t%C0",operands);
  OUT_INSN (len, "clr\t%D0",operands);

  if(GET_CODE(operands[0]) == REG) *len -= 3;

  if(len) *len = dummy;

  return "";
}




const char *
zeroextendhisi(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[1]) || indexed_location(operands[1]);

  if(!sameoperand(operands,1))
    {
      OUT_INSN (len, "mov\t%A1, %A0",operands);
      dummy = 3;
      if(zs) dummy = 2;
      if(GET_CODE(operands[0]) == REG) dummy--;
      if(GET_CODE(operands[1]) == REG) dummy--;
    }

  OUT_INSN (len, "clr\t%B0",operands);
  dummy += 2;
  if(GET_CODE(operands[0]) == REG) dummy--;

  if(len) *len = dummy;

  return "";

}


const char *
zeroextendhidi(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[1]) || indexed_location(operands[1]);

  if(!sameoperand(operands,1))
    {
      OUT_INSN (len, "mov\t%A1, %A0",operands);
      dummy = 3;
      if(zs) dummy = 2;
      if(GET_CODE(operands[0]) == REG) dummy--;
      if(GET_CODE(operands[1]) == REG) dummy--;
    }

  dummy += 6;
  OUT_INSN (len, "clr\t%B0",operands);
  OUT_INSN (len, "clr\t%C0",operands);
  OUT_INSN (len, "clr\t%D0",operands);

  if(GET_CODE(operands[0]) == REG) dummy -= 3;

  if(len) *len = dummy;

  return "";
}


const char *
zeroextendsidi(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;

  if(!sameoperand(operands,1))
    {
      if(zero_shifted(operands[1]))
        {
          rtx reg = XEXP(operands[1],0);

          OUT_INSN (len, "mov\t@%E1+, %A0",operands);
          OUT_INSN (len, "mov\t@%E1+, %B0",operands);
          dummy = 4;
          if(GET_CODE(operands[0]) == REG) dummy -= 2;

          if(!dead_or_set_p (insn, reg))
            {
              OUT_INSN (len, "sub\t#4, %E1",operands);
              dummy += 1;
            }
        }
      else
        {
          OUT_INSN (len, "mov\t%A1, %A0",operands);
          OUT_INSN (len, "mov\t%B1, %B0",operands);
          dummy = 6;
          if(GET_CODE(operands[0]) == REG) dummy -= 2;
          if(GET_CODE(operands[1]) == REG) dummy -= 2;
          if(indexed_location(operands[1])) dummy--;
        }
    }

  dummy += 4;
  OUT_INSN (len, "clr\t%C0",operands);
  OUT_INSN (len, "clr\t%D0",operands);

  if(GET_CODE(operands[0]) == REG) dummy -= 2;

  if(len) *len = dummy;

  return "";
}



/*****************  TESTS AND JUMPS *********************/

RTX_CODE
followed_compare_condition (insn)
rtx insn;
{
  rtx next = next_real_insn (insn);
  RTX_CODE cond = UNKNOWN;

  if (next && GET_CODE (next) == JUMP_INSN)
    {
      rtx pat = PATTERN (next);
      rtx src = SET_SRC (pat);
      rtx t = XEXP (src, 0);
      cond = GET_CODE (t);
    }
  return cond;
}


void
msp430_instantinate_condition(insn, operands, cond)
rtx insn;
rtx operands[] ATTRIBUTE_UNUSED;
RTX_CODE cond;
{
  rtx next = next_real_insn (insn);
  rtx pat = PATTERN (next);
  rtx pattern = PATTERN (insn);
  rtx src,t;

  if (GET_CODE (pattern) == PARALLEL)
    pattern = XVECEXP (pattern, 0, 0);

  src = SET_SRC (pat);
  t = XEXP (src,0);
  PUT_CODE (t,cond);

  /*
      SET_SRC (pattern) = gen_rtx (NEG,
                                   GET_MODE (SET_SRC (pattern)),
                                   SET_SRC (pattern));
  */
  INSN_CODE (next) = -1;
}



/*
 * Tests for codes:
 *	GT,GTU,EQ,NE,LE clobbers source. Source is a register.
 * For LT,GE, GEU, LEU source any
 * Conditions reverting:
 	GT -> GE
 	GTU -> NE
 	LE -> LT
 	LEU -> EQ
 
 */

int
tstsi_needs_scratch_p(insn, operands)
rtx insn;
rtx operands[];
{
  RTX_CODE cond = followed_compare_condition (insn);

  if(cond == GT
      || cond == GTU
      || cond == LE
      || cond == LEU)
    {
      if(!REG_P(operands[0])) return 1;

      if( dead_or_set_p(insn, operands[0]) )
        {
          return 0;
        }
      else
        {
          return 1;
        }
    }

  return 0;
}

int
tstsi_needs_half_scratch_p(insn, operands)
rtx insn;
rtx operands[];
{
  RTX_CODE cond = followed_compare_condition (insn);

  if(cond == GTU
      || cond == LEU)
    {
      if(!REG_P(operands[0])) return 1;

      if( dead_or_set_p(insn, operands[0]) )
        {
          return 0;
        }
      else
        {
          return 1;
        }
    }

  return 0;

}


const char *
msp430_emit_tsthi(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int adl = ((GET_CODE(operands[0]) == REG)?0:1) + 1;

  OUT_INSN(len, "tst\t%A0", operands);
  dummy += adl;

  if(len) *len = dummy;
  return "";
}

void
msp430_prepare_tsthi(insn, operands)
rtx insn;
rtx operands[];
{
  RTX_CODE cond = followed_compare_condition (insn);
  rtx ops[10];

  switch(cond)
    {
    case GT:
      ops[1] = gen_rtx_CONST_INT(HImode,1);
      emit_insn_before( gen_cmphi(operands[0],ops[1]), insn);
      msp430_instantinate_condition(insn, operands, GE);
      delete_insn(insn);
      break;

    case LE:
      ops[1] = gen_rtx_CONST_INT(HImode,1);
      emit_insn_before( gen_cmphi(operands[0],ops[1]), insn);
      msp430_instantinate_condition(insn, operands, LT);
      delete_insn(insn);
      break;
    default:
      break;
    }

  return;
}


const char *
msp430_emit_tstqi(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int adl = ((GET_CODE(operands[0]) == REG)?0:1) + 1;

  OUT_INSN(len, "tst.b\t%A0", operands);
  dummy += adl;

  if(len) *len = dummy;
  return "";
}

void
msp430_prepare_tstqi(insn, operands)
rtx insn;
rtx operands[];
{
  RTX_CODE cond = followed_compare_condition (insn);
  rtx ops[10];

  switch(cond)
    {
    case GT:
      ops[1] = gen_rtx_CONST_INT(QImode,1);
      emit_insn_before( gen_cmpqi(operands[0],ops[1]), insn);
      msp430_instantinate_condition(insn, operands, GE);
      delete_insn(insn);
      break;

    case LE:
      ops[1] = gen_rtx_CONST_INT(QImode,1);
      emit_insn_before( gen_cmpqi(operands[0],ops[1]), insn);
      msp430_instantinate_condition(insn, operands, LT);
      delete_insn(insn);
      break;
    default:
      break;
    }
  return;
}



const char *
msp430_emit_tstsi(insn, operands, len)
rtx insn;
rtx operands[];
int *len;
{
  RTX_CODE cond = followed_compare_condition (insn);
  int dummy = 0;
  int adl = ((GET_CODE(operands[0]) == REG)?0:1) + 1;

  switch(cond)
    {
    case GT:
    case LE:
    case GTU:
    case LEU:
      /* these were removed on reorg pass*/
      abort();
      break;

      /* these do not clobber operand */
    case LT:
    case GE:
      OUT_INSN(len, "tst\t%B0",operands);
      dummy += adl;
      break;
    case GEU:		/* paranoia */
      OUT_INSN(len, "setc",operands);
      dummy += 1;
      break;
    case LTU:
      OUT_INSN(len, "clrc",operands);
      dummy += 1;
      break;

    case NE:
      {
        rtx next = next_real_insn(insn);
        rtx pat  = PATTERN (next);
        rtx src  = SET_SRC (pat);
        rtx labels[3];
        int dist;

        labels[0] = XEXP(src,1);

        /* This code will be called twice:
           at shorten branches pass and during insn match.
           First time is to compute insn length. At this time
           msp430_jump_dist() will return zero, and insn length
           might be worng. But it will be corrected later on final pass.
           During insn match everything seems to be Ok. */

        dist = msp430_jump_dist(labels[0],insn)-adl;

        OUT_INSN(len, "tst\t%A0", operands);
        dummy =  adl;

        if(dist < -511 || dist > 512)
          {
            OUT_INSN(len, "jeq\t+4",operands);
            OUT_INSN(len, "br\t#%l0", labels);
            dummy += 3;
          }
        else
          {
            OUT_INSN(len, "jne\t%l0", labels);
            dummy += 1;
          }
        OUT_INSN(len, "tst\t%B0", operands);
        dummy += adl;
      }
      break;

      /* Scratching is more expensive, than jump over insn. So,
         just tst bytes. */
    case EQ:
      OUT_INSN(len, "tst\t%A0", operands);
      if(adl==1) OUT_INSN(len, "jne\t+2",operands);
      if(adl==2) OUT_INSN(len, "jne\t+4",operands);
      OUT_INSN(len, "tst\t%B0", operands);
      dummy += 1 + 2*adl;
      break;

    default:
      abort();
    }

  if(len) *len = dummy;

  return "";
}

void
msp430_prepare_tstsi(insn,operands)
rtx insn;
rtx operands[];
{
  RTX_CODE cond = followed_compare_condition (insn);
  rtx ops[10];


  if(REGNO(operands[0]) == REGNO(operands[1]))
    {
      /* peephole. do nothing */

    }
  else if(tstsi_needs_half_scratch_p(insn, operands))
    {
      ops[0] = gen_lowpart(HImode, operands[1]);
      ops[1] = gen_lowpart(HImode, operands[0]);
      emit_insn_before( gen_movhi(ops[0],ops[1]), insn);
    }
  else if(tstsi_needs_scratch_p(insn, operands))
    {
      ops[0] = operands[1];
      ops[1] = operands[0];
      emit_insn_before( gen_movhi(ops[0],ops[1]), insn);
      operands[0] = operands[1];
    }
  else
    {
      operands[1] = operands[0];
    }

  switch(cond)
    {
    case GT:
      ops[2] = gen_rtx_CONST_INT(SImode,1);
      emit_insn_before( gen_subsi3(operands[0],operands[0],ops[2]), insn);
      msp430_instantinate_condition(insn, operands, GE);
      delete_insn(insn);
      break;

    case LE:
      ops[2] = gen_rtx_CONST_INT(SImode,1);
      emit_insn_before( gen_subsi3(operands[0],operands[0],ops[2]), insn);
      msp430_instantinate_condition(insn, operands, LT);
      delete_insn(insn);
      break;

      /* these two are paranoia */
    case GTU:
      ops[0] = gen_highpart(HImode, operands[1]);
      ops[1] = ops[0];
      ops[2] = gen_lowpart(HImode, operands[0]);
      emit_insn_before( gen_iorhi3(ops[0],ops[1],ops[2]),insn);
      emit_insn_before( gen_tsthi(ops[0]),insn);
      msp430_instantinate_condition(insn, operands, NE);
      delete_insn(insn);
      break;
    case LEU:
      ops[0] = gen_highpart(HImode, operands[1]);
      ops[1] = ops[0];
      ops[2] = gen_lowpart(HImode, operands[0]);
      emit_insn_before( gen_iorhi3(ops[0],ops[1],ops[2]),insn);
      emit_insn_before( gen_tsthi(ops[0]),insn);
      msp430_instantinate_condition(insn, operands, EQ);
      delete_insn(insn);
      break;
    default:
      break;
    }
  return;
}



int
cmpsi_needs_scratch_p(insn, operands)
rtx insn;
rtx operands[];
{
  rtx op0 = operands[0];
  RTX_CODE cond = followed_compare_condition (insn);

  if(cond == NE || cond == EQ) return 0;

  if(REG_P(op0) && dead_or_set_p(insn,op0))
    return 0;

  return 1;
}


void
msp430_prepare_cmpsi(insn, operands)
rtx insn;
rtx operands[];
{
  RTX_CODE cond = followed_compare_condition (insn);


#define SWO(o)	\
do {			\
rtx x = o[0];		\
o[0] = o[1];		\
o[1] = x;		\
} while(0);		\

if(cond == NE || cond == EQ) return;

  if(GET_CODE(operands[1]) != CONST_INT)
    {

      if(cond == GT
          || cond == LE
          || cond == GTU
          || cond == LEU)
        SWO(operands);

      if( cmpsi_needs_scratch_p(insn, operands) )
        {
          emit_insn_before(gen_movsi(operands[2],operands[0]),insn);
        }
      else
        {
          operands[2] = operands[0];
        }

      switch(cond)
        {
        case GE:
        case LT:
        case GEU:
        case LTU:
          break;
        case GT:
          msp430_instantinate_condition(insn, operands, LT);
          break;
        case LE:
          msp430_instantinate_condition(insn, operands, GE);
          break;
        case GTU:
          msp430_instantinate_condition(insn, operands, LTU);
          break;
        case LEU:
          msp430_instantinate_condition(insn, operands, GEU);
          break;
        default:
          abort();
        }
    }
  else
    {
      int val = INTVAL(operands[1]);

      /* const int operand */
      if( cmpsi_needs_scratch_p(insn, operands) )
        {
          emit_insn_before(gen_movsi(operands[2],operands[0]),insn);
        }
      else
        {
          operands[2] = operands[0];
        }

      if(cond == GT
          || cond == LE
          || cond == GTU
          || cond == LEU)
        {
          operands[1] = gen_rtx_CONST_INT(SImode,val+1);
        }


      switch(cond)
        {
        case GE:
        case LT:
        case GEU:
        case LTU:
          break;
        case GT:
          msp430_instantinate_condition(insn, operands, GE);
          break;
        case LE:
          msp430_instantinate_condition(insn, operands, LT);
          break;
        case GTU:
          msp430_instantinate_condition(insn, operands, GEU);
          break;
        case LEU:
          msp430_instantinate_condition(insn, operands, LTU);
          break;
        default:
          abort();

        }
    }

  emit_insn_before(gen_subsi3(operands[2],operands[2],operands[1]), insn);
  delete_insn(insn);
}


const char *
msp430_emit_cmpsi(insn, operands, len)
rtx insn;
rtx operands[];
int *len;
{
  int dummy = 0;
  RTX_CODE cond = followed_compare_condition (insn);
  int c1 = (GET_CODE(operands[1]) == CONST_INT);
  /*
     	'a' is 0 operand,
     	'b' is 1 operand
     	's' is a scratch
   	'c' is a constant
   
     The code mutations will be the following:
     1. Both operands are vars:
     	a.CC.b
     	
     	GT: b->s; s=s-a; GT->LT
     	GE: a->s; s=s-b; GE->GE
     	LT: a->s; s=s-b; LT->LT
     	LE: b->s; s=s-a; LE->GE	! can be done by swapping 'a' and 'b' and CC
     	EQ:
     	NE: either 'a' or 'b' can be moved to scratch and nothing much else...
     	GTU: b->s; s=s-a; GTU->LTU
     	GEU: a->s; s=s-b; GEU->GEU
     	LTU: a->s; s=s-b; LTU->LTU
     	LEU: b->s; s=s-a; LEU->GEU
     
     2. 'b' is constant
     	a.CC.c
     	
     	'a' always moved to scratch 
     	GT: s=s-(c+1); 	GT->GE
     	GE: s=s-c;	GE->GE
     	LT: s=s-c;	LT->LT
     	LE: s=s-(c+1);	LE->LT
     	NE:
     	EQ: do something!!!
     	GTU: s=s-(c+1); GTU->GEU
     	GEU: s=s-c;	GEU->GEU
     	LTU: s=s-c;	LTU->LTU
     	LEU: s=s-(c+1); LEU->LTU
   
   
     So, GT, LE, GTU and LEU always substituded by some else CC.
   
   *	...
   */

  switch(cond)
    {
    case GT:
    case GE:
    case LT:
    case LE:
    case GTU:
    case GEU:
    case LTU:
    case LEU:
      /* all these must be deleted ! */
      abort();
      break;
    case NE:
      {
        rtx next = next_real_insn(insn);
        rtx pat  = PATTERN (next);
        rtx src  = SET_SRC (pat);
        rtx labels[3];
        int dist;

        /* check if we can save 1 byte */
        if(indexed_location(operands[0])
            && !indexed_location(operands[1])
            && !c1)
          {
            rtx x = operands[0];
            operands[0] = operands[1];
            operands[1] = x;
          }

        labels[0] = XEXP(src,1);


        dummy =  3;
        if(indexed_location(operands[1]) || REG_P(operands[1]) ) dummy--;
        if(REG_P(operands[0])) dummy--;

        OUT_INSN(len, "cmp\t%A1, %A0", operands);

        dist = msp430_jump_dist(labels[0],insn) - dummy;

        if(dist < -511 || dist > 512)
          {
            OUT_INSN(len, "jeq\t+4",operands);
            OUT_INSN(len, "br\t#%l0", labels);
            dummy += 3;
          }
        else
          {
            OUT_INSN(len, "jne\t%l0", labels);
            dummy += 1;
          }
        OUT_INSN(len, "cmp\t%B1, %B0", operands);
        dummy += 3;
        if(REG_P(operands[0])) dummy--;
        if(REG_P(operands[1])) dummy--;

        if(c1)
          {
            int cval = INTVAL(operands[1]);
            int x = (cval)&0x0fffful;
            int y = (((unsigned long)(cval))&0xffff0000ul >> 16);
            if(x==0||x==1||x==2||x==4||x==8||x==0xffff) dummy--;
            if(y==0||y==1||y==2||y==4||y==8||y==0xffff) dummy--;
          }
      }
      break;
    case EQ:
      /* check if we can save 1 byte */
      if(indexed_location(operands[0])
          && !indexed_location(operands[1])
          && !c1)
        {
          rtx x = operands[0];
          operands[0] = operands[1];
          operands[1] = x;
        }

      if(!REG_P(operands[1])) dummy += 2;
      if(!REG_P(operands[0])) dummy += 2;

      OUT_INSN(len, "cmp\t%A1, %A0", operands);
      OUT_INSN(len, "jne\t.LcmpSIe%=",operands);
      OUT_INSN(len, "cmp\t%B1, %B0\n.LcmpSIe%=:", operands);

      dummy += 3;
      if(indexed_location(operands[1])) dummy--;

      if(c1)
        {
          int cval = INTVAL(operands[1]);
          int x = (cval)&0x0fffful;
          int y = (((unsigned long)(cval))&0xffff0000ul >> 16);
          if(x==0||x==1||x==2||x==4||x==8||x==0xffff) dummy--;
          if(y==0||y==1||y==2||y==4||y==8||y==0xffff) dummy--;
        }

      break;

    default:
      OUT_INSN(len, "what a condition???",operands);
      dummy = 1;
      break;
    }

  if(len) *len = dummy;
  return "";
}


void
msp430_prepare_cmphi(insn, operands)
rtx insn;
rtx operands[];
{
  RTX_CODE cond = followed_compare_condition (insn);
  int c1 = (GET_CODE(operands[1]) == CONST_INT);
  /* cause pointers have HImode, this is possible to obtain 
     operands like (const:hi (plus:hi (symbol_ref ("symbol")) (const_int)))
     so, lets check it as well */
  int c2 = (GET_CODE(operands[1]) == CONST);
  /* another one for just (symbol_ref ("symbol")),
     also, we take into account label_ref here */
  int c3 = (GET_CODE(operands[1]) == SYMBOL_REF || 
  		GET_CODE(operands[1]) == LABEL_REF);
  
  if(c1 || c2 || c3)
    {
    
      if(cond == GT
          || cond == LE
          || cond == GTU
          || cond == LEU)
      {
        if(c1)
        	operands[1] = gen_rtx_CONST_INT(HImode, INTVAL(operands[1])+1);
      	else if(c2)
      	{
      	  rtx in = XEXP(operands[1],0);
      	  rtx v = XEXP(in,1);
      	  v = gen_rtx_CONST_INT(HImode, INTVAL(v)+1);      	  
      	  XEXP( XEXP(operands[1],0),1 ) = v;
      	}
      	else /* c3 */
      	{
      		rtx cc = gen_rtx_CONST_INT(HImode, 1);
      		operands[1] = gen_rtx_CONST (HImode, 
      						gen_rtx_PLUS ( HImode, 
      								operands[1], 
      								cc));
      	}
      	
      }
      
      switch(cond)
        {
        case GT:
          msp430_instantinate_condition(insn, operands, GE);
          break;
        case LE:
          msp430_instantinate_condition(insn, operands, LT);
          break;
        case GTU:
          msp430_instantinate_condition(insn, operands, GEU);
          break;
        case LEU:
          msp430_instantinate_condition(insn, operands, LTU);
          break;
        default:
          break;
        }
    }
  else
    {
      /* try to win 1 byte and one cycle */
      if(indexed_location(operands[0]))
        {
          SWO(operands);
          cond = swap_condition(cond);
          msp430_instantinate_condition(insn, operands, cond);
        }

      /* try to win 2 cycles and 2 bytes (worse case) */
      switch(cond)
        {
        case GT:
          SWO(operands);
          msp430_instantinate_condition(insn, operands, LT);
          break;
        case LE:
          SWO(operands);
          msp430_instantinate_condition(insn, operands, GE);
          break;
        case LEU:
          SWO(operands);
          msp430_instantinate_condition(insn, operands, GEU);
          break;
        case GTU:
          SWO(operands);
          msp430_instantinate_condition(insn, operands, LTU);
          break;
        default:
          break;
        }
    }

  emit_insn_before(gen_cmphi(operands[0],operands[1]), insn);
  delete_insn(insn);

}

const char *
msp430_emit_cmphi(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int c1 = (GET_CODE(operands[1]) == CONST_INT);
  int cval;

#define LENGTH_CONST_H(val)                             \
{                                                       \
int x = (val)&0x0fffful;                                \
dummy += 3;                                             \
if(x==0||x==1||x==2||x==4||x==8||x==0xffff) dummy--;    \
if(REG_P(operands[0])) dummy--;				\
}

  if(c1) cval = (INTVAL(operands[1]))&0xfffful;

  if(c1)
    {
      LENGTH_CONST_H(cval);
      OUT_INSN(len, "cmp\t%A1, %A0",operands);
    }
  else
    {
      dummy += 3;
      if(indexed_location(operands[1])) dummy--;
      else if(REG_P(operands[1])) dummy--;
      if(REG_P(operands[0])) dummy--;
      OUT_INSN(len, "cmp\t%1, %0", operands);
    }

  if(len) *len = dummy;
  return "";

}



void
msp430_prepare_cmpqi(insn, operands)
rtx insn;
rtx operands[];
{
  RTX_CODE cond = followed_compare_condition (insn);
  int c1 = (GET_CODE(operands[1]) == CONST_INT);


  if(c1)
    {
      if(cond == GT
          || cond == LE
          || cond == GTU
          || cond == LEU)
        operands[1] = gen_rtx_CONST_INT(QImode, INTVAL(operands[1])+1);

      switch(cond)
        {
        case GT:
          msp430_instantinate_condition(insn, operands, GE);
          break;
        case LE:
          msp430_instantinate_condition(insn, operands, LT);
          break;
        case GTU:
          msp430_instantinate_condition(insn, operands, GEU);
          break;
        case LEU:
          msp430_instantinate_condition(insn, operands, LTU);
          break;
        default:
          break;
        }
    }
  else
    {
      /* try to win 1 byte and one cycle */
      if(indexed_location(operands[0]))
        {
          SWO(operands);
          cond = swap_condition(cond);
          msp430_instantinate_condition(insn, operands, cond);
        }

      /* try to win 2 cycles and 2 bytes */
      switch(cond)
        {
        case GT:
          SWO(operands);
          msp430_instantinate_condition(insn, operands, LT);
          break;
        case LE:
          SWO(operands);
          msp430_instantinate_condition(insn, operands, GE);
          break;
        case LEU:
          SWO(operands);
          msp430_instantinate_condition(insn, operands, GEU);
          break;
        case GTU:
          SWO(operands);
          msp430_instantinate_condition(insn, operands, LTU);
          break;
        default:
          break;
        }
    }

  emit_insn_before(gen_cmpqi(operands[0],operands[1]), insn);
  delete_insn(insn);
}

const char *
msp430_emit_cmpqi(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;
  int c1 = (GET_CODE(operands[1]) == CONST_INT);
  int cval;

#define LENGTH_CONST_Q(val)                             \
{                                                       \
int x = (val)&0x0fffful;                                \
dummy += 3;                                             \
if(x==0||x==1||x==2||x==4||x==8||x==0xffff) dummy--;    \
if(REG_P(operands[0])) dummy--;				\
}

  if(c1) cval = (INTVAL(operands[1]))&0xfffful;

  if(c1)
    {
      LENGTH_CONST_Q(cval);
      OUT_INSN(len, "cmp.b\t%A1, %A0",operands);
    }
  else
    {
      dummy += 3;
      if(indexed_location(operands[1])) dummy--;
      else if(REG_P(operands[1])) dummy--;
      if(REG_P(operands[0])) dummy--;
      OUT_INSN(len, "cmp.b\t%1, %0", operands);
    }

  if(len) *len = dummy;
  return "";

}



/******** jumps ************/

const char *
msp430_emit_blt0si(operands, len)
rtx operands[];
int len;
{
  output_asm_insn("tst\t%B2",operands);
  switch(len)
    {
    case 2:
      output_asm_insn("jl\t%0",operands);
      break;
    case 4:
      output_asm_insn("jge\t+4",operands);
      output_asm_insn("br\t#%0",operands);
      break;
    default:
      return "bug!!!";
    }

  return "";
}

const char *
msp430_emit_beq(operands, len)
rtx operands[];
int len;
{

  switch(len)
    {
    case 1:
    case 2:
      output_asm_insn("jeq\t%0",operands);
      break;
    case 3:
    case 4:
      output_asm_insn("jne\t+4",operands);
      output_asm_insn("br\t#%0",operands);
      break;
    default:
      return "bug!!!";
    }

  return "";
}


const char *
msp430_emit_bne(operands, len)
rtx operands[];
int len;
{

  switch(len)
    {
    case 1:
    case 2:
      output_asm_insn("jne\t%0",operands);
      break;
    case 3:
    case 4:
      output_asm_insn("jeq\t+4",operands);
      output_asm_insn("br\t#%0",operands);
      break;
    default:
      return "bug!!!";
    }

  return "";
}



const char *
msp430_emit_bgt(operands, len)
rtx operands[];
int len;
{

  switch(len)
    {
    case 2:
      output_asm_insn("jeq\t+2",operands);
      output_asm_insn("jge\t%0",operands);

      break;
    case 4:
      output_asm_insn("jeq\t+6",operands);
      output_asm_insn("jl\t+4",operands);
      output_asm_insn("br\t#%0",operands);
      break;
    default:
      return "bug!!!";
    }

  return "";
}



const char *
msp430_emit_bgtu(operands, len)
rtx operands[];
int len;
{

  switch(len)
    {
    case 2:
      output_asm_insn("jeq\t+2",operands);
      output_asm_insn("jhs\t%0",operands);

      break;
    case 4:
      output_asm_insn("jeq\t+6",operands);
      output_asm_insn("jlo\t+4",operands);
      output_asm_insn("br\t#%0",operands);
      break;
    default:
      return "bug!!!";
    }

  return "";
}





const char *
msp430_emit_blt(operands, len)
rtx operands[];
int len;
{

  switch(len)
    {
    case 1:
    case 2:
      output_asm_insn("jl\t%0	;	%0",operands);
      break;
    case 3:
    case 4:
      output_asm_insn("jge\t+4",operands);
      output_asm_insn("br\t#%0",operands);
      break;
    default:
      return "bug!!!";
    }

  return "";
}



const char *
msp430_emit_bltu(operands, len)
rtx operands[];
int len;
{

  switch(len)
    {
    case 1:
    case 2:
      output_asm_insn("jlo\t%0",operands);
      break;
    case 3:
    case 4:
      output_asm_insn("jhs\t+4",operands);
      output_asm_insn("br\t#%0",operands);
      break;
    default:
      return "bug!!!";
    }

  return "";
}


const char *
msp430_emit_bge(operands, len)
rtx operands[];
int len;
{

  switch(len)
    {
    case 1:
    case 2:
      output_asm_insn("jge\t%l0",operands);
      break;
    case 3:
    case 4:
      output_asm_insn("jl\t+4",operands);
      output_asm_insn("br\t#%0",operands);
      break;
    default:
      return "bug!!!";
    }

  return "";
}

const char *
msp430_emit_bgeu(operands, len)
rtx operands[];
int len;
{

  switch(len)
    {
    case 1:
    case 2:
      output_asm_insn("jhs\t%l0",operands);
      break;
    case 3:
    case 4:
      output_asm_insn("jlo\t+4",operands);
      output_asm_insn("br\t#%0",operands);
      break;
    default:
      return "bug!!!";
    }

  return "";
}



const char *
msp430_emit_ble(operands, len)
rtx operands[];
int len;
{

  switch(len)
    {
    case 2:
      output_asm_insn("jeq\t%0",operands);
      output_asm_insn("jl\t%0",operands);
      break;
    case 4:
      output_asm_insn("jeq\t+2",operands);
      output_asm_insn("jge\t+4",operands);
      output_asm_insn("br\t#%0",operands);
      break;
    default:
      return "bug!!!";
    }

  return "";
}


const char *
msp430_emit_bleu(operands, len)
rtx operands[];
int len;
{

  switch(len)
    {
    case 2:
      output_asm_insn("jeq\t%0",operands);
      output_asm_insn("jlo\t%0",operands);
      break;
    case 4:
      output_asm_insn("jeq\t+2",operands);
      output_asm_insn("jhs\t+4",operands);
      output_asm_insn("br\t#%0",operands);
      break;
    default:
      return "bug!!!";
    }

  return "";
}





/******* COMMON SHIFT CODE ***************/


int
is_shift_better_in_reg(operands)
rtx operands[];
{
  rtx x = operands[0];
  rtx cnt = operands[2];
  int size = GET_MODE_SIZE(x->mode);
  int icnt = -1;
  int r = 0;

  if( !optimize ) return 0;

  if (GET_CODE (cnt) == CONST_INT)
    icnt = INTVAL(cnt);
  else
    return 1;

  switch(size)
    {
    case 1:
      if(icnt!=1
          && icnt !=2
          && icnt !=7) r = 1;
      break;
    case 2:
      if(icnt!=1
          && icnt !=2
          && icnt !=8
          && icnt !=15) r = 2;
      break;
    case 4:
      if(icnt!=1
          && icnt!=2
          && icnt!=8
          && icnt!=16
          && icnt!=24
          && icnt!=31 ) r = 4;
      break;
    case 8:
      if(icnt!=1
          && icnt!=2
          && icnt!=16
          && icnt!=32
          && icnt!=48
          && icnt!= 63) r = 8;
      break;
    }

  return r;
}




static int set_len PARAMS ((rtx,int,int));

/* for const operand2 and for SI, DI modes.*/
static int
set_len(x, bl, sc)
rtx x;	/* operand0 */
int bl;	/* base length in assumption of memory operand */
int sc; /* shift count */
{
  int dummy;
  int zs = zero_shifted(x);
  int size = GET_MODE_SIZE(x->mode);
  int sshi = 0;

  if(size==4) sshi = 1;
  else if(size==8) sshi = 2;

  if(size == 1) size++;

  if(GET_CODE(x) == REG ) dummy = (bl>>1)-sshi;	/* bl / 2 is not fully correct */
  else if(zs) dummy = bl - (size>>1) + 1;
  else if(indexed_location(x)) dummy = bl-1;
  else dummy = bl;

  return dummy*sc;
}



static int set_ren PARAMS ((rtx,int,int));

/* for const operand2 and for SI, DI modes.*/
static int
set_ren(x, bl, sc)
rtx x;	/* operand0 */
int bl;	/* base length in assumption of memory operand */
int sc; /* shift count */
{
  int dummy;

  bl *= sc;
  if(GET_CODE(x) == REG ) dummy = bl/2;
  else if(indexed_location(x)) dummy = bl-sc;
  else dummy = bl;
  return dummy;
}


static int set_rel PARAMS ((rtx,int,int));

/* for const operand2 and for SI, DI modes.*/
static int
set_rel(x, bl, sc)
rtx x;	/* operand0 */
int bl;	/* base length in assumption of memory operand */
int sc; /* shift count */
{
  int dummy;

  bl *= sc;
  if(GET_CODE(x) == REG ) dummy = bl/2;
  else if(indexed_location(x)) dummy = bl-sc;
  else dummy = bl;
  dummy += sc;
  return dummy;
}



#define INST_THRESHOLD  16

int
msp430_emit_shift_cnt(set_len_fun,pattern, insn, operands, len, lsc)
int (*set_len_fun) (rtx,int,int);
const char *pattern;
rtx insn;
rtx operands[];
int *len;
int lsc;
{
  int scratch = (GET_CODE (PATTERN (insn)) == PARALLEL);
  rtx cnt = operands[2];
  rtx op[10];
  int used_reg0 = -1;
  int used_reg2 = -1;
  int havemove  =  1;
  int havepush  =  0;
  int i;
  int dummy = 0;

  op[0] = operands[0];
  op[1] = operands[1];
  op[2] = operands[2];
  op[3] = operands[3];

  if (GET_CODE (op[2]) == CONST_INT)
    {
      int shiftpos = INTVAL(cnt);
      if(optimize_size)
        {
          if( shiftpos*lsc< INST_THRESHOLD + lsc)
            {
              /* do inline anyway cause an actual output is smaller */

              while(shiftpos--)
                {
                  OUT_INSN (len, pattern, op);
                  dummy += (set_len_fun)(op[0], lsc, 1);
                }
              if(len) *len = dummy;
              return 0;
            }
        }
      else if(optimize == 3)
        {
          /* inline much faster */
          if(len) *len = shiftpos*lsc;
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, op);
              dummy += (set_len_fun)(op[0], lsc, 1);
            }
          if(len) *len = dummy;
          return 0;
        }
    }


  /* find suitable register first */
  /* check if counter is a register */
  if(GET_CODE (op[2]) == REG && REGNO(op[2]) != 1)
    {
      if(dead_or_set_p (insn, op[2]))
        {
          op[3] = op[2];
          havemove = 0;
        }
      else
        used_reg2 = REGNO(op[2]);
    }

  if(GET_CODE (op[2]) == MEM
      && GET_CODE (XEXP(op[2],0)) == REG
      && REGNO(XEXP(op[2],0)) != 1   )
    {
      used_reg2 = REGNO( XEXP(op[2],0));
    }

  if(GET_CODE (op[0]) == REG && REGNO(op[0]) != 1)
    {
      used_reg0 = REGNO(op[0]);
    }

  if(GET_CODE (op[0]) == MEM
      && GET_CODE (XEXP(op[0],0)) == REG)
    {
      used_reg0 = REGNO( XEXP(op[0],0));
    }

  if(havemove && !scratch )
    {
      for(i = 15; i>4; i--)
        {
          int r = regs_ever_live[i];

          if(i != used_reg0
              && i != used_reg2
              && !r )
            break;
        }

      /* this will be difficult to adjust frame pointer if we mark
         register as used and push it on stack. However, if we push 
         and pop it locally, the stack will survive. */

      if(i == 9)
        {
          havepush  =  1;
          op[3] = gen_rtx (REG, VOIDmode,9);
        }
      else
        {
          havepush  =  1;
          op[3] = gen_rtx (REG, VOIDmode,i);
        }
    }


  /* generate code : */
  /* operand 3 is a register */
  if(havepush)
    {
      dummy += 1;
      OUT_INSN (len, "push\t%3",op);
    }

  if(havemove)
    {
      OUT_INSN (len, "mov\t%2, %3",op);

      if(GET_CODE(op[2])==CONST_INT)
        {
          int v = INTVAL(op[2]);

          if(v == 0
              || v == 1
              || v == 2
              || v == 4
              || v == 8
            )
            dummy++;
          else
            dummy += 2;
        }
      else if(indexed_location(op[2]) || REG_P(op[2]))
        {
          dummy++;
        }
      else
        {
          dummy += 2;
        }
    }

  if(GET_CODE (op[2]) != CONST_INT)
    {
      OUT_INSN (len, "tst\t%3",op);
      dummy++;
    }
  OUT_INSN (len, "jz\t.Lsend%=\n.Lsst%=:",op);
  OUT_INSN (len, pattern, op);
  OUT_INSN (len, "dec\t%3",op);
  OUT_INSN (len, "jnz\t.Lsst%=\n.Lsend%=:",op);
  dummy += (set_len_fun)(op[0],lsc,1) + 3;

  if(havepush)
    {
      dummy++;
      OUT_INSN (len, "pop\t%3",op);
    }

  if(len) *len = dummy;
  return 0;
}


/* <<<<<<<<<<<<< SHIFT LEFT CODE <<<<<<<<<<<<<<<<<     */

const char *
msp430_emit_ashlqi3(insn, operands,len)
rtx insn;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[0]) || indexed_location(operands[0]);
  const char *pattern;
  int shiftpos;

  if(TARGET_CALL_SHIFT)
    return msp430_emit_ashlqi3_call(insn, operands,len);

  if(zs)
    pattern = "rla.b\t@%E0";
  else
    pattern = "rla.b\t%A0";

  if(GET_CODE (operands[2]) == CONST_INT)
    {
      shiftpos =INTVAL (operands[2]);

      switch (shiftpos)
        {
        default:
          if(zs)
            OUT_INSN (len, "clr.b\t@%E0", operands);
          else
            OUT_INSN (len, "clr.b\t%A0", operands);
          dummy = 2;
          if(REG_P(operands[0]) ) dummy >>= 1;
          break;

        case 0:		/* paranoia setting */
          dummy = 0;
          break;

        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += set_len(operands[0],3,1);
            }
          break;

        case 7:
          if(zs)
            {
              OUT_INSN (len, "rra.b\t%0", operands);
              OUT_INSN (len, "clr.b\t%0", operands);
              OUT_INSN (len, "rrc.b\t%0", operands);
              dummy = 5;
            }
          else
            {
              OUT_INSN (len, "rra.b\t%0", operands);
              OUT_INSN (len, "clr.b\t%0", operands);
              OUT_INSN (len, "rrc.b\t%0", operands);
              dummy = 6;
              if(REG_P(operands[0]) ) dummy = 3;
            }

          break;
        }

      if(len) *len = dummy;
      return "";

    }
  else
    {
      msp430_emit_shift_cnt(set_len,pattern, insn, operands, len, 3 );
    }

  return "";
}


static const char *
msp430_emit_ashlqi3_call(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;

  if (*operands
      && (GET_CODE (operands[0]) == CONST_INT
          || (operands[2]
              && GET_CODE (operands[2]) == CONST_INT)))
    {
      int x;

      if(!len)
        x = INTVAL (operands[0]);
      else
        x = INTVAL (operands[2]);


      if(x == 1)
        {
          OUT_INSN (len, "rla.b\tr15", operands);
        }
      else if(x == 7)
        {
          OUT_INSN (len, "call\t#__lshiftqi_1_6", operands);

        }
      else if(x > 0 && x < 7)
        {
          char str[256];
          snprintf(str,255,"call\t#__lshiftqi_1_6+%d",12 - x*2);
          OUT_INSN (len, str, operands);
        }
      else
        {
          OUT_INSN (len, "clr\tr15", operands);
        }

      dummy = 1;
    }
  else
    {
      OUT_INSN (len, "call\t#__lshiftqi_cnt", operands);
      dummy = 1;
    }

  if(len) *len = dummy;
  return "";

}

const char *
msp430_emit_ashlhi3(insn, operands,len)
rtx insn;
rtx operands[];
int *len;
{
  int dummy=0;
  int zs;
  const char *pattern;
  int shiftpos;

  if( REG_NOTES(insn)
      && XEXP(REG_NOTES(insn),0)
      && GET_CODE(XEXP(REG_NOTES(insn),0)) == MULT) goto sskip;

  if(TARGET_CALL_SHIFT)
    return msp430_emit_ashlhi3_call(insn, operands,len);
sskip:
  zs = zero_shifted(operands[0]) || indexed_location(operands[0]);

  if(zs)
    pattern = "rla\t@%E0";
  else
    pattern = "rla\t%A0";

  if(GET_CODE (operands[2]) == CONST_INT)
    {
      shiftpos = INTVAL (operands[2]);

      switch (shiftpos)
        {
        case 0:		/* paranoia setting */
          dummy = 0;
          break;

        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += set_len(operands[0],3,1);
            }
          break;

        case 8:
        case 9:
        case 10:
        case 11:
        case 12:
        case 13:
        case 14:
          if(zs)
            {
              dummy = 3;
              OUT_INSN (len, "and.b\t#0xffff, %A0",operands);
              OUT_INSN (len, "swpb\t@%E0",operands);
            }
          else
            {
              dummy = 4;
              OUT_INSN (len, "and.b\t#0xffff, %A0",operands);
              OUT_INSN (len, "swpb\t%A0",operands);
              if( REG_P(operands[0]) ) dummy = 2;
            }


          shiftpos -= 8;
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += set_len(operands[0],3,1);
            }
          break;

        case 15:
          if(zs)
            {
              OUT_INSN (len, "rra\t%0", operands);
              OUT_INSN (len, "clr\t%0", operands);
              OUT_INSN (len, "rrc\t%0", operands);
              dummy = 5;
            }
          else
            {
              OUT_INSN (len, "rra\t%0", operands);
              OUT_INSN (len, "clr\t%0", operands);
              OUT_INSN (len, "rrc\t%0", operands);
              dummy = 6;
              if(REG_P(operands[0]) ) dummy = 3;
            }

          break;


        default:

          OUT_INSN (len, "clr\t%A0", operands);
          dummy = 2;
          if(REG_P(operands[0]) ) dummy = 1;
          break;
        }

      if(len) *len = dummy;
      return "";
    }
  else
    {
      msp430_emit_shift_cnt(set_len,pattern, insn, operands, len,3);
    }

  return "";
}



static const char *
msp430_emit_ashlhi3_call(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;

  if (*operands
      && (GET_CODE (operands[0]) == CONST_INT
          || (operands[2]
              && GET_CODE (operands[2]) == CONST_INT)))
    {
      int x;

      if(!len)
        x = INTVAL (operands[0]);
      else
        {
          x = INTVAL (operands[2]);
        }

      /* structure indices better be calculated inline.
         Therefore for hi mode when shift count < 3
         perform shifts inline. The index hopfully will be 
         a register. */

      if(x == 1)
        {
          OUT_INSN (len, "rla\tr15", operands);
        }
      else if(x == 2)
        {
          OUT_INSN (len, "rla\tr15", operands);
          OUT_INSN (len, "rla\tr15", operands);
          dummy += 1;
        }
      else if(x == 3)
        {
          OUT_INSN (len, "rla\tr15", operands);
          OUT_INSN (len, "rla\tr15", operands);
          OUT_INSN (len, "rla\tr15", operands);
          dummy += 2;
        }
      else if(x == 15)
        {
          OUT_INSN (len, "call\t#__lshifthi_15", operands);

        }
      else if(x > 2 && x < 8)
        {
          char str[256];
          snprintf(str,255,"call\t#__lshifthi_1_7+%d",14 - x*2);
          OUT_INSN (len, str, operands);
        }
      else if(x==8)
        {
          OUT_INSN (len, "call\t#__lshifthi_8", operands);
        }
      else if(x>8 && x<15)
        {
          char str[256];
          OUT_INSN (len, "call\t#__lshifthi_8", operands);
          x -= 8;
          snprintf(str,255,"call\t#__lshifthi_1_7+%d",14 - x*2);
          OUT_INSN (len, str, operands);
          dummy += 1;
        }
      else
        {
          OUT_INSN (len, "clr\tr15", operands);
        }

      dummy += 1;
    }
  else
    {
      OUT_INSN (len, "call\t#__lshifthi_cnt", operands);
      dummy = 1;
    }

  if(len) *len = dummy;
  return "";

}


const char *
msp430_emit_ashlsi3(insn, operands,len)
rtx insn;
rtx operands[];
int *len;
{

  int dummy = 0;
  int zs;
  const char *pattern;

  if(TARGET_CALL_SHIFT)
    return msp430_emit_ashlsi3_call(insn, operands,len);

  zs = zero_shifted(operands[0]);

  if(zs)
    pattern = "add\t@%E0+, -2(%E0)\n\taddc\t@%E0+, -2(%E0)\n\tsub\t#4, %E0";
  else
    pattern = "rla\t%A0\n\trlc\t%B0";


  if(GET_CODE (operands[2]) == CONST_INT)
    {
      int shiftpos = INTVAL(operands[2]);

      switch(shiftpos)
        {

        case 0:
          dummy = 0;
          break;
        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += set_len(operands[0],6,1);
            }
          break;

        case 8:
        case 9:
        case 10:
        case 11:
        case 12:
        case 13:
        case 14:
        case 15:

          if(zs || indexed_location(operands[0]))
            {
              OUT_INSN (len, "xor.b\t@%E0, %B0", operands);
              OUT_INSN (len, "xor\t@%E0, %B0", operands);
              OUT_INSN (len, "swpb\t%B0", operands);
              OUT_INSN (len, "and.b\t#-1, %A0", operands);
              OUT_INSN (len, "swpb\t@%E0", operands);
              dummy = 9;
            }
          else
            {
              OUT_INSN (len, "xor.b\t%A0, %B0", operands);
              OUT_INSN (len, "xor\t%A0, %B0", operands);
              OUT_INSN (len, "swpb\t%B0", operands);
              OUT_INSN (len, "and.b\t#-1, %A0", operands);
              OUT_INSN (len, "swpb\t%A0", operands);
              dummy = 12;
              if(REG_P(operands[0]) ) dummy = 5;
            }

          shiftpos -= 8;

          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += set_len(operands[0],6,1);
            }

          if(len) *len = dummy;
          return "";

          break;

        case 16:
        case 17:
        case 18:
        case 19:
        case 20:
        case 21:
        case 22:
        case 23:

          if(zs || indexed_location(operands[0]) )
            {
              OUT_INSN (len, "mov\t@%E0, %B0", operands);
              OUT_INSN (len, "clr\t%A0",  operands);
              dummy = 4;
            }
          else
            {
              OUT_INSN (len, "mov\t%A0, %B0", operands);
              OUT_INSN (len, "clr\t%A0",  operands);
              dummy = 5;
              if(REG_P(operands[0]) ) dummy = 3;
            }

          shiftpos -= 16;
          while(shiftpos--)
            {
              OUT_INSN (len, pattern,operands);
              dummy += set_len(operands[0],6,1);
            }

          if(len) *len = dummy;
          return "";
          break;

        case 24:
        case 25:
        case 26:
        case 27:
        case 28:
        case 29:
        case 30:
          if(zs || indexed_location(operands[0]) )
            {
              OUT_INSN (len, "mov.b\t@%E0,%B0",operands);
              OUT_INSN (len, "swpb\t%B0",operands);
              OUT_INSN (len, "clr\t@%E0",operands);
              dummy = 6;
            }
          else
            {
              OUT_INSN (len, "mov.b\t%A0,%B0",operands);
              OUT_INSN (len, "swpb\t%B0",operands);
              OUT_INSN (len, "clr\t%A0",operands);
              dummy = 8;
              if(GET_CODE(operands[0]) == REG) dummy = 3;
            }

          shiftpos -= 24;
          while(shiftpos--)
            {
              OUT_INSN (len, pattern,operands);
              dummy += set_len(operands[0],6,1);
            }

          if(len) *len = dummy;
          return "";

          break;

        case 31:
          if(zs || indexed_location(operands[0]))
            {
              OUT_INSN (len, "rra\t@%E0",operands);
              OUT_INSN (len, "clr\t%A0",operands);
              OUT_INSN (len, "clr\t%B0",operands);
              OUT_INSN (len, "rrc\t%B0",operands);
              dummy = 9;

            }
          else
            {
              OUT_INSN (len, "rra\t%A0",operands);
              OUT_INSN (len, "clr\t%A0",operands);
              OUT_INSN (len, "clr\t%B0",operands);
              OUT_INSN (len, "rrc\t%B0",operands);
              dummy = 10;
              if(REG_P(operands[0])) dummy = 4;
            }

          if(len) *len = dummy;
          return "";
          break;

        default:
          OUT_INSN (len, "clr\t%A0",operands);
          OUT_INSN (len, "clr\t%B0",operands);
          if(len) *len = set_len(operands[0],6,1);
          return "";
          break;

        } /* switch */

      if(len) *len = dummy;
      return "";

    }
  else
    msp430_emit_shift_cnt(set_len,pattern, insn, operands, len, 8);

  return "";

}


static const char *
msp430_emit_ashlsi3_call(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;

  if (*operands
      && (GET_CODE (operands[0]) == CONST_INT
          || (operands[2]
              && GET_CODE (operands[2]) == CONST_INT)))
    {
      int x;

      if(!len)
        x = INTVAL (operands[0]);
      else
        {
          x = INTVAL (operands[2]);
        }


      if(x == 1)
        {
          OUT_INSN (len, "rla\tr14", operands);
          OUT_INSN (len, "rlc\tr15", operands);
          dummy += 1;
        }
      else if(x==8)
        {
          OUT_INSN (len, "call\t#__lshiftsi_8", operands);
        }
      else if(x == 16)
        {
          OUT_INSN (len, "call\t#__lshiftsi_16", operands);

        }
      else if(x == 24)
        {
          OUT_INSN (len, "call\t#__lshiftsi_8", operands);
          OUT_INSN (len, "call\t#__lshiftsi_16", operands);
          dummy += 1;
        }
      else if(x == 31)
        {
          OUT_INSN (len, "call\t#__lshiftsi_31", operands);
        }
      else if(x > 0 && x < 8)
        {
          char str[256];
          snprintf(str,255,"call\t#__lshiftsi_1_7+%d",28 - x*4);
          OUT_INSN (len, str, operands);
        }
      else if(x>8 && x<15)
        {
          char str[256];
          OUT_INSN (len, "call\t#__lshiftsi_8", operands);
          x -= 8;
          snprintf(str,255,"call\t#__lshiftsi_1_7+%d",28 - x*4);
          OUT_INSN (len, str, operands);
          dummy += 1;
        }
      else if(x > 16 && x < 24)
        {
          char str[256];
          OUT_INSN (len, "call\t#__lshiftsi_16", operands);
          x -= 16;
          /* if x > 16 no r14 has to be shifted */
          snprintf(str,255,"call\t#__lshifthi_1_7+%d",14 - x*2);
          OUT_INSN (len, str, operands);
          dummy += 1;
        }
      else if(x > 24 && x < 31)
        {
          char str[256];
          OUT_INSN (len, "call\t#__lshiftsi_8", operands);
          OUT_INSN (len, "call\t#__lshiftsi_16", operands);
          x -= 24;
          snprintf(str,255,"call\t#__lshifthi_1_7+%d",14 - x*2);
          OUT_INSN (len, str, operands);
          dummy += 2;
        }
      else
        {
          OUT_INSN (len, "clr\tr14", operands);
          OUT_INSN (len, "clr\tr15", operands);
          dummy += 1;
        }

      dummy += 1;
    }
  else
    {
      OUT_INSN (len, "call\t#__lshiftsi_cnt", operands);
      dummy = 1;
    }

  if(len) *len = dummy;
  return "";

}



const char *
msp430_emit_ashldi3(insn, operands,len)
rtx insn;
rtx operands[];
int *len;
{

  int dummy = 0;
  int zs;
  const char *pattern;

  if(TARGET_CALL_SHIFT)
    return msp430_emit_ashldi3_call(insn, operands,len);


  zs = zero_shifted(operands[0]);

  if(zs)
    pattern = "add\t@%E0+,-2(%E0)\n\taddc\t@%E0+,-2(%E0)\n\taddc\t@%E0+,-2(%E0)\n\taddc\t@%E0+,-2(%E0)\n\tsub\t#8,%E0";
  else
    pattern = "rla\t%A0\n\trlc\t%B0\n\trlc\t%C0\n\trlc\t%D0";

  if(GET_CODE (operands[2]) == CONST_INT)
    {
      int shiftpos = INTVAL(operands[2]);

      switch(shiftpos)
        {
        case 0:
          dummy = 0;
          if(len) *len = dummy;
          break;

        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
        case 8:
        case 9:
        case 10:
        case 11:
        case 12:
        case 13:
        case 14:
        case 15:
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += set_len(operands[0],12,1);
            }
          if(len) *len = dummy;
          break;

        case 16:
        case 17:
        case 18:
        case 19:
        case 20:
        case 21:
        case 22:
        case 23:
          if(zs || indexed_location(operands[0]))
            {
              dummy = 10;
              OUT_INSN (len, "mov\t%C0, %D0",operands);
              OUT_INSN (len, "mov\t%B0, %C0",operands);
              OUT_INSN (len, "mov\t@%E0, %B0",operands);
              OUT_INSN (len, "clr\t@%E0",operands);
            }
          else
            {
              dummy = 11;
              OUT_INSN (len, "mov\t%C0, %D0",operands);
              OUT_INSN (len, "mov\t%B0, %C0",operands);
              OUT_INSN (len, "mov\t%A0, %B0",operands);
              OUT_INSN (len, "clr\t%A0",operands);

            }
          if(GET_CODE(operands[0]) == REG) dummy = 4;
          shiftpos -= 16;
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += set_len(operands[0],12,1);
            }
          if(len) *len = dummy;
          break;

        case 24:
        case 25:
        case 26:
        case 27:
        case 28:
        case 29:
        case 30:
        case 31:
          if(zs)
            {
              dummy = 8;
              OUT_INSN (len, "mov\t@%E0, %D0",operands);
              OUT_INSN (len, "clr\t%A0",operands);
              OUT_INSN (len, "clr\t%B0",operands);
              OUT_INSN (len, "clr\t%C0",operands);

            }
          else
            {
              dummy = 9;
              OUT_INSN (len, "mov\t%A0, %D0",operands);
              OUT_INSN (len, "clr\t%A0",operands);
              OUT_INSN (len, "clr\t%B0",operands);
              OUT_INSN (len, "clr\t%C0",operands);
            }
          if(GET_CODE(operands[0]) == REG) dummy = 4;

          shiftpos -= 16;
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += set_len(operands[0],12,1);
            }

          if(len) *len = dummy;
          break;

        case 32:
        case 33:
        case 34:
        case 35:
        case 36:
        case 37:
        case 38:
        case 39:
        case 40:
        case 41:
        case 42:
        case 43:
        case 44:
        case 45:
        case 46:
        case 47:

          if(zs)
            {
              OUT_INSN (len, "mov\t@%E0+, %C0",operands);
              OUT_INSN (len, "mov\t@%E0+, %D0",operands);
              OUT_INSN (len, "sub\t#4, %E0",operands);
              OUT_INSN (len, "clr\t%A0",operands);
              OUT_INSN (len, "clr\t%B0",operands);
              dummy = 9;
            }
          else
            {
              dummy = 10;
              OUT_INSN (len, "mov\t%A0, %C0",operands);
              OUT_INSN (len, "mov\t%B0, %D0",operands);
              OUT_INSN (len, "clr\t%A0",operands);
              OUT_INSN (len, "clr\t%B0",operands);
            }
          if(GET_CODE(operands[0]) == REG) dummy = 4;

          shiftpos -= 32;
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += set_len(operands[0],12,1);
            }

          if(len) *len = dummy;
          break;

        case 48:
        case 49:
        case 50:
        case 51:
        case 52:
        case 53:
        case 54:
        case 55:
        case 56:
        case 57:
        case 58:
        case 59:
        case 60:
        case 61:
        case 62:
          if(zs)
            {
              dummy = 8;
              OUT_INSN (len, "mov\t@%E0, %D0",operands);
              OUT_INSN (len, "clr\t%A0",operands);
              OUT_INSN (len, "clr\t%B0",operands);
              OUT_INSN (len, "clr\t%C0",operands);
            }
          else
            {
              dummy = 9;
              OUT_INSN (len, "mov\t%A0, %D0",operands);
              OUT_INSN (len, "clr\t%A0",operands);
              OUT_INSN (len, "clr\t%B0",operands);
              OUT_INSN (len, "clr\t%C0",operands);
            }

          shiftpos -= 48;
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += set_len(operands[0],12,1);
            }

          if(GET_CODE(operands[0]) == REG) dummy = 4;
          if(len) *len = dummy;

          break;

        case 63:
          if(zs || indexed_location(operands[0]))
            {
              OUT_INSN (len, "rra\t@%E0",operands);
              OUT_INSN (len, "clr\t%A0",operands);
              OUT_INSN (len, "clr\t%B0",operands);
              OUT_INSN (len, "clr\t%C0",operands);
              OUT_INSN (len, "clr\t%D0",operands);
              OUT_INSN (len, "rrc\t%D0",operands);
              dummy = 11;
            }
          else
            {
              OUT_INSN (len, "rra\t%A0",operands);
              OUT_INSN (len, "clr\t%A0",operands);
              OUT_INSN (len, "clr\t%B0",operands);
              OUT_INSN (len, "clr\t%C0",operands);
              OUT_INSN (len, "clr\t%D0",operands);
              OUT_INSN (len, "rrc\t%D0",operands);
              dummy = 12;
              if(REG_P(operands[0])) dummy = 6;
            }

          if(len) *len = dummy;

          break;	/* make compiler happy */

        default:
          OUT_INSN (len, "clr\t%A0",operands);
          OUT_INSN (len, "clr\t%B0",operands);
          OUT_INSN (len, "clr\t%C0",operands);
          OUT_INSN (len, "clr\t%D0",operands);
          dummy = 8;
          if(zs) dummy--;
          if(REG_P(operands[0])) dummy = 4;

          if(len) *len = dummy;

        } /* switch */

      return "";
    }
  else
    msp430_emit_shift_cnt(set_len,pattern, insn, operands, len, 12);

  return "";	/* make compiler happy */
}

static const char *
msp430_emit_ashldi3_call(insn, operands,len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int dummy = 0;

  if (*operands
      && (GET_CODE (operands[0]) == CONST_INT
          || (operands[2]
              && GET_CODE (operands[2]) == CONST_INT)))
    {
      int x;

      if(!len)
        x = INTVAL (operands[0]);
      else
        {
          x = INTVAL (operands[2]);
        }

      if(x == 1)
        {
          OUT_INSN (len, "rla\tr12", operands);
          OUT_INSN (len, "rlc\tr13", operands);
          OUT_INSN (len, "rlc\tr14", operands);
          OUT_INSN (len, "rlc\tr15", operands);
          dummy += 3;
        }
      else if(x == 16)
        {
          OUT_INSN (len, "call\t#__lshiftdi_16", operands);

        }
      else if(x == 32)
        {
          OUT_INSN (len, "call\t#__lshiftdi_32", operands);
        }
      else if(x == 40)
        {
          OUT_INSN (len, "call\t#__lshiftdi_32", operands);
          OUT_INSN (len, "call\t#__lshiftsi_8", operands);
          dummy += 1;
        }
      else if(x == 48)
        {
          OUT_INSN (len, "call\t#__lshiftdi_32", operands);
          OUT_INSN (len, "call\t#__lshiftsi_16", operands);
          dummy += 1;
        }
      else if(x == 63)
        {
          OUT_INSN (len, "call\t#__lshiftdi_63", operands);
        }
      else if(x > 0 && x < 15)
        {
          char str[256];
          snprintf(str,255,"call\t#__lshiftdi_1_15+%d",56 - x*8);
          OUT_INSN (len, str, operands);
        }
      else if(x > 16 && x < 32)
        {
          char str[256];
          OUT_INSN (len, "call\t#__lshiftdi_16", operands);
          x -= 16;
          /* if x > 16 no r14 has to be shifted */
          snprintf(str,255,"call\t#__lshifthi_1_15_lhi+%d",42 - x*6);
          OUT_INSN (len, str, operands);
          dummy += 1;
        }
      else if(x > 32 && x < 40)
        {
          char str[256];
          OUT_INSN (len, "call\t#__lshiftdi_32", operands);
          x -= 32;
          snprintf(str,255,"call\t#__lshiftsi_1_7+%d",28 - x*4);
          OUT_INSN (len, str, operands);
          dummy += 1;
        }
      else if(x>40 && x<48)
        {
          char str[256];
          OUT_INSN (len, "call\t#__lshiftdi_32", operands);
          OUT_INSN (len, "call\t#__lshiftsi_8", operands);
          x -= 40;
          snprintf(str,255,"call\t#__lshiftsi_1_7+%d",28 - x*4);
          OUT_INSN (len, str, operands);
          dummy += 2;
        }
      else if(x>48 && x<63)
        {
          char str[256];
          OUT_INSN (len, "call\t#__lshiftdi_32", operands);
          OUT_INSN (len, "call\t#__lshiftsi_16", operands);
          x -= 48;
          snprintf(str,255,"call\t#__lshifthi_1_7+%d",14 - x*2);
          dummy += 2;
        }
      else
        {
          OUT_INSN (len, "clr\tr12", operands);
          OUT_INSN (len, "clr\tr13", operands);
          OUT_INSN (len, "clr\tr14", operands);
          OUT_INSN (len, "clr\tr15", operands);
          dummy += 3;
        }

      dummy += 1;
    }
  else
    {
      OUT_INSN (len, "call\t#__lshiftdi_cnt", operands);
      dummy = 1;
    }

  if(len) *len = dummy;
  return "";
}



/**/


/********* SHIFT RIGHT CODE ***************************************/

const char *
msp430_emit_ashrqi3(insn, operands,len)
rtx insn;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[0]) || indexed_location(operands[0]);
  const char *pattern;
  int shiftpos;

  if(zs)
    pattern = "rra.b\t@%E0";
  else
    pattern = "rra.b\t%A0";

  if(GET_CODE (operands[2]) == CONST_INT)
    {

      shiftpos = INTVAL (operands[2]);

      switch (shiftpos)
        {
        case 0:		/* paranoia setting */
          dummy = 0;
          break;

        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += 2;
            }
          break;

        case 7:
          if(zs)
            {
              OUT_INSN (len, "sxt\t@%E0", operands);
              OUT_INSN (len, "swpb\t@%E0", operands);
              OUT_INSN (len, "and.b\t#-1, %A0", operands);
              dummy = 4;
            }
          else
            {
              OUT_INSN (len, "sxt\t%A0", operands);
              OUT_INSN (len, "swpb\t%A0", operands);
              OUT_INSN (len, "mov.b\t#-1, %A0", operands);
              dummy = 6;
            }
          if(REG_P(operands[0])) dummy = 3;
          if(len) *len = dummy;
          return "";

          break;

        default:
          OUT_INSN (len, "clr.b\t%A0", operands);
          dummy = 2;
          if(REG_P(operands[0])) dummy = 1;
        }

      if(len) *len = dummy;
      return "";

    }
  else
    {
      msp430_emit_shift_cnt(set_ren,pattern, insn, operands, len,2);
    }

  return "";
}


const char *
msp430_emit_ashrhi3(insn, operands,len)
rtx insn;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[0]) || indexed_location(operands[0]);
  const char *pattern;
  int shiftpos;

  if(zs)
    pattern = "rra\t@%E0";
  else
    pattern = "rra\t%A0";

  if(GET_CODE (operands[2]) == CONST_INT)
    {
      shiftpos = INTVAL (operands[2]);

      switch (shiftpos)
        {
        case 0:		/* paranoia setting */
          dummy = 0;
          break;

        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += 2;
            }
          if(zs || REG_P(operands[0])) dummy >>= 1;
          break;

        case 8:
        case 9:
        case 10:
        case 11:
        case 12:
        case 13:
        case 14:
          if(zs)
            {
              OUT_INSN (len, "swpb\t@%E0", operands);
              OUT_INSN (len, "sxt\t@%E0", operands);
              dummy = 2;
            }
          else
            {
              OUT_INSN (len, "swpb\t%A0", operands);
              OUT_INSN (len, "sxt\t%A0", operands);
              dummy = 4;
              if(REG_P(operands[0])) dummy = 2;
            }
          shiftpos -= 8;
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += (zs || REG_P(operands[0])) ? 1 : 2;
            }
          break;

        case 15:
          if(zs)
            {
              OUT_INSN (len, "swpb\t@%E0", operands);
              OUT_INSN (len, "sxt\t@%E0", operands);
              OUT_INSN (len, "swpb\t@%E0", operands);
              OUT_INSN (len, "swpb\t@%E0", operands);
              dummy = 4;
            }
          else
            {
              OUT_INSN (len, "swpb\t%A0", operands);
              OUT_INSN (len, "sxt\t%A0", operands);
              OUT_INSN (len, "swpb\t%A0", operands);
              OUT_INSN (len, "sxt\t%A0", operands);
              dummy = 8;
            }
          if(REG_P(operands[0])) dummy = 4;
          break;

        default:
          OUT_INSN (len, "clr\t%A0", operands);
          dummy = 2;
          if(REG_P(operands[0])) dummy = 1;
        }

      if(len) *len = dummy;
      return "";
    }
  else
    {
      msp430_emit_shift_cnt(set_ren,pattern, insn, operands, len, 2);
    }

  return "";
}

const char *
msp430_emit_ashrsi3(insn, operands,len)
rtx insn;
rtx operands[];
int *len;
{

  int dummy = 0;
  const char *pattern;
  int zs = zero_shifted(operands[0]);

  pattern = "rra\t%B0\n\trrc\t%A0";

  if(GET_CODE (operands[2]) == CONST_INT)
    {
      int shiftpos = INTVAL(operands[2]);

      switch(shiftpos)
        {
        case 0:
          dummy = 0;
          break;
        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += set_ren(operands[0],4,1);
            }
          break;

        case 8:
        case 9:
        case 10:
        case 11:
        case 12:
        case 13:
        case 14:
        case 15:
          OUT_INSN (len, "swpb\t%A0",operands);
          OUT_INSN (len, "swpb\t%B0", operands);
          OUT_INSN (len, "xor.b\t%B0, %A0", operands);
          OUT_INSN (len, "xor\t%B0, %A0", operands);
          OUT_INSN (len, "sxt\t%B0", operands);
          dummy = 12;

          if(REG_P(operands[0])) dummy = 5;
          shiftpos -= 8;
          while(shiftpos--)
            {
              OUT_INSN (len, pattern, operands);
              dummy += set_ren(operands[0],4,1);
            }
          break;

        case 16:
        case 17:
        case 18:
        case 19:
        case 20:
        case 21:
        case 22:
        case 23:
          OUT_INSN (len, "mov\t%B0, %A0", operands);
          OUT_INSN (len, "bit\t#0x8000, %B0",operands);
          OUT_INSN (len, "jz\t.Lsrc%=",operands);
          OUT_INSN (len, "bis\t#0xffff, %B0",  operands);
          OUT_INSN (len, "jmp\t.Lsre%=\n.Lsrc%=:",operands);
          OUT_INSN (len, "clr\t%B0\n.Lsre%=:",  operands);
          dummy = 12;

          if(GET_CODE(operands[0]) == REG) dummy = 7;

          shiftpos -= 16;
          while(shiftpos--)
            {
              OUT_INSN (len, "rra\t%A0", operands);
              dummy += 2;
              if(GET_CODE(operands[0]) == REG || zs) dummy--;
            }

          break;

        case 24:
        case 25:
        case 26:
        case 27:
        case 28:
        case 29:
        case 30:
          OUT_INSN (len, "swpb\t%B0", operands);
          OUT_INSN (len, "sxt\t%B0", operands);
          OUT_INSN (len, "mov\t%B0, %A0",operands);
          OUT_INSN (len, "swpb\t%B0", operands);
          OUT_INSN (len, "sxt\t%B0", operands);
          dummy = 11;

          if(GET_CODE(operands[0]) == REG) dummy = 5;

          shiftpos -= 24;
          while(shiftpos--)
            {
              OUT_INSN (len, "rra\t%A0", operands);
              dummy += 2;
              if(GET_CODE(operands[0]) == REG || zs) dummy--;
            }
          break;

        case 31:
          OUT_INSN (len, "tst\t%B0",operands);
          OUT_INSN (len, "mov\t#-1,%B0",operands);
          OUT_INSN (len, "mov\t#-1,%A0",operands);
          if(GET_CODE(operands[0]) == REG)
            OUT_INSN (len, "jn\t+4",operands);
          else
            OUT_INSN (len, "jn\t+8",operands);
          OUT_INSN (len, "clr\t%A0",operands);
          OUT_INSN (len, "clr\t%B0",operands);
          dummy = 11;
          if(GET_CODE(operands[0]) == REG) dummy = 6;
          break;

        default:
          dummy = 0;	/* leave it alone!!! */
          break;

        } /* switch */

      if(len) *len = dummy;
      return "";
    }
  else
    msp430_emit_shift_cnt(set_ren,pattern, insn, operands, len, 4);

  return "";

}


const char *
msp430_emit_ashrdi3(insn, operands,len)
rtx insn;
rtx operands[];
int *len;
{

  int dummy = 0;
  const char *pattern;


  pattern = "rra\t%D0\n\trrc\t%C0\n\trrc\t%B0\n\trrc\t%A0";


  if(GET_CODE (operands[2]) == CONST_INT)
    {
      int shiftpos = INTVAL(operands[2]);

      switch(shiftpos)
        {
        case 0:
          dummy = 0;
          break;
        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
        case 8:
        case 9:
        case 10:
        case 11:
        case 12:
        case 13:
        case 14:
        case 15:
          while(shiftpos--)
            {
              OUT_INSN (len, pattern,operands);
              dummy += set_ren(operands[0],8,1);
            }
          break;

        case 16:
        case 17:
        case 18:
        case 19:
        case 20:
        case 21:
        case 22:
        case 23:
        case 24:
        case 25:
        case 26:
        case 27:
        case 28:
        case 29:
        case 30:
        case 31:

          OUT_INSN (len, "mov\t%B0, %A0",operands);
          OUT_INSN (len, "mov\t%C0, %B0",operands);
          OUT_INSN (len, "mov\t%D0, %C0",operands);
          OUT_INSN (len, "swpb\t%D0",operands);
          OUT_INSN (len, "sxt\t%D0",operands);
          OUT_INSN (len, "swpb\t%D0",operands);
          OUT_INSN (len, "sxt\t%D0",operands);

          dummy = 17;
          if(GET_CODE(operands[0]) == REG) dummy = 7;
          shiftpos -= 16;
          while(shiftpos--)
            {
              OUT_INSN (len, "rra\t%C0\n\trrc\t%B0\n\trrc\t%A0" ,operands);
              dummy += set_ren(operands[0],6,1);
            }

          break;

        case 32:
        case 33:
        case 34:
        case 35:
        case 36:
        case 37:
        case 38:
        case 39:
        case 40:
        case 41:
        case 42:
        case 43:
        case 44:
        case 45:
        case 46:
        case 47:
          OUT_INSN (len, "mov\t%C0, %A0",operands);
          OUT_INSN (len, "mov\t%D0, %B0",operands);
          OUT_INSN (len, "swpb\t%D0",operands);
          OUT_INSN (len, "sxt\t%D0",operands);
          OUT_INSN (len, "swpb\t%D0",operands);
          OUT_INSN (len, "sxt\t%D0",operands);
          OUT_INSN (len, "mov\t%D0, %C0",operands);
          dummy = 17;
          if(GET_CODE(operands[0]) == REG) dummy = 8;
          shiftpos -= 32;
          while(shiftpos--)
            {
              OUT_INSN (len, "rra\t%B0\n\trrc\t%A0",operands);
              dummy += set_ren(operands[0],4,1);
            }
          break;

        case 48:
        case 49:
        case 50:
        case 51:
        case 52:
        case 53:
        case 54:
        case 55:
        case 56:
        case 57:
        case 58:
        case 59:
        case 60:
        case 61:
        case 62:
          OUT_INSN (len, "mov\t%D0, %A0",operands);
          OUT_INSN (len, "swpb\t%D0",operands);
          OUT_INSN (len, "sxt\t%D0",operands);
          OUT_INSN (len, "swpb\t%D0",operands);
          OUT_INSN (len, "sxt\t%D0",operands);
          OUT_INSN (len, "mov\t%D0, %C0",operands);
          OUT_INSN (len, "mov\t%D0, %B0",operands);
          dummy = 17;
          if(GET_CODE(operands[0]) == REG) dummy = 7;
          shiftpos -= 48;
          while(shiftpos--)
            {
              OUT_INSN (len, "rra\t%A0",operands);
              dummy += set_ren(operands[0],2,1);
            }
          break;

        case 63:
          OUT_INSN (len, "swpb\t%D0",operands);
          OUT_INSN (len, "sxt\t%D0",operands);
          OUT_INSN (len, "swpb\t%D0",operands);
          OUT_INSN (len, "sxt\t%D0",operands);
          OUT_INSN (len, "mov\t%D0, %C0",operands);
          OUT_INSN (len, "mov\t%D0, %B0",operands);
          OUT_INSN (len, "mov\t%D0, %A0",operands);
          dummy = 17;
          if(GET_CODE(operands[0]) == REG) dummy = 7;
          break;

        default:
          dummy = 0;

        } /* case */

      if(len) *len = dummy;
      return "";
    }
  else
    msp430_emit_shift_cnt(set_ren,pattern, insn, operands, len, 8);
  return "";
}



/********* LOGICAL SHIFT RIGHT CODE ***************************************/

const char *
msp430_emit_lshrqi3(insn, operands,len)
rtx insn;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[0]) || indexed_location(operands[0]);
  const char *pattern;
  const char *second_pat;
  int shiftpos;

  if(zs)
    {
      pattern = "clrc\n\trrc.b\t@%E0";
      second_pat = "rra.b\t@%E0";
    }
  else
    {
      pattern = "clrc\n\trrc.b\t%A0";
      second_pat = "rra.b\t%A0";
    }

  if(GET_CODE (operands[2]) == CONST_INT)
    {

      shiftpos = INTVAL (operands[2]);

      if(shiftpos!=7 && shiftpos)
        {
          OUT_INSN (len, pattern, operands);
          dummy += set_rel(operands[0],2,1);
          shiftpos--;
        }

      switch (shiftpos)
        {
        case 0:
          break;

        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:

          while(shiftpos--)
            {
              OUT_INSN (len, second_pat, operands);
              dummy += set_rel(operands[0],2,1) - 1;
            }

          break;

        case 7:
          if(zs)
            {
              OUT_INSN (len, "rla.b\t@%E0", operands);
              OUT_INSN (len, "clr.b\t%A0", operands);
              OUT_INSN (len, "rlc.b\t@%E0", operands);
              dummy = 4;
            }
          else
            {
              OUT_INSN (len, "rla.b\t%A0", operands);
              OUT_INSN (len, "clr.b\t%A0", operands);
              OUT_INSN (len, "rlc.b\t%A0", operands);
              dummy = 6;
            }
          if(REG_P(operands[0])) dummy = 3;
          break;

        default:
          OUT_INSN (len, "clr.b\t%A0", operands);
          dummy = 2;
          if(REG_P(operands[0])) dummy = 1;
          break;
        }

      if(len) *len = dummy;
    }
  else
    {
      msp430_emit_shift_cnt(set_rel,pattern, insn, operands, len,2);
    }

  return "";
}


const char *
msp430_emit_lshrhi3(insn, operands,len)
rtx insn;
rtx operands[];
int *len;
{
  int dummy = 0;
  int zs = zero_shifted(operands[0]) || indexed_location(operands[0]);
  const char *pattern;
  const char *second_pat;
  int shiftpos;

  if(zs)
    {
      pattern = "clrc\n\trrc\t@%E0";
      second_pat = "rra\t@%E0";
    }
  else
    {
      pattern = "clrc\n\trrc\t%A0";
      second_pat = "rra\t%A0";
    }

  if(GET_CODE (operands[2]) == CONST_INT)
    {
      shiftpos = INTVAL (operands[2]);

      if(shiftpos < 8 && shiftpos)
        {
          OUT_INSN (len, pattern, operands);
          dummy += set_rel(operands[0],2,1);
          shiftpos--;
        }

      switch (shiftpos)
        {

        case 0:
          break;

        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:

          while(shiftpos--)
            {
              OUT_INSN (len, second_pat, operands);
              dummy += set_rel(operands[0],2,1) - 1;
            }

          break;

        case 8:
        case 9:
        case 10:
        case 11:
        case 12:
        case 13:
        case 14:

          if(zs)
            {
              OUT_INSN (len, "swpb\t@%E0",operands);
              OUT_INSN (len, "and.b\t#-1, %A0",operands);
              dummy = 3;
            }
          else
            {
              OUT_INSN (len, "swpb\t%A0",operands);
              OUT_INSN (len, "and.b\t#-1, %A0",operands);
              dummy = 4;
            }
          if(REG_P(operands[0])) dummy = 2;
          shiftpos -= 8;
          while(shiftpos--)
            {
              OUT_INSN (len, second_pat, operands);
              dummy += set_rel(operands[0],2,1) - 1;
            }
          break;

        case 15:

          if(zs)
            {
              OUT_INSN (len, "rla\t@%E0", operands);
              OUT_INSN (len, "clr\t@%E0", operands);
              OUT_INSN (len, "rlc\t@%E0", operands);
              dummy = 3;
            }
          else
            {
              OUT_INSN (len, "rla\t%A0", operands);
              OUT_INSN (len, "clr\t%A0", operands);
              OUT_INSN (len, "rlc\t%A0", operands);
              dummy = 6;
            }
          if(REG_P(operands[0])) dummy = 3;
          break;

        default:
          OUT_INSN (len, "clr\t%A0", operands);
          dummy = 2;
          if(REG_P(operands[0])) dummy = 1;
          break;
        }

      if(len) *len = dummy;
      return "";
    }
  else
    {
      msp430_emit_shift_cnt(set_rel,pattern, insn, operands, len,2);
    }

  return "";

}


const char *
msp430_emit_lshrsi3(insn, operands,len)
rtx insn;
rtx operands[];
int *len;
{
  const char *pattern;
  int dummy = 0;
  int zs = zero_shifted(operands[0]) || indexed_location(operands[0]);
  const char *second_pat = "rra\t%B0\n\trrc\t%A0";

  pattern = "clrc\n\trrc\t%B0\n\trrc\t%A0";


  if(GET_CODE (operands[2]) == CONST_INT)
    {
      int shiftpos = INTVAL(operands[2]);

      if(shiftpos < 8 && shiftpos)
        {
          OUT_INSN (len, pattern, operands);
          /* This function was underestimating the length by 1 for shifts from
             1 to 7.  I added one here - Max */
          dummy += set_rel(operands[0],2,1) + 1;
          shiftpos--;
        }

      switch(shiftpos)
        {
        case 0:
          break;

        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
          while(shiftpos--)
            {
              OUT_INSN (len, second_pat, operands);
              dummy += set_rel(operands[0],4,1)-1;
            }

          break;

        case 8:
        case 9:
        case 10:
        case 11:
        case 12:
        case 13:
        case 14:
        case 15:
          OUT_INSN (len, "swpb\t%A0",operands);
          OUT_INSN (len, "swpb\t%B0", operands);
          OUT_INSN (len, "xor.b\t%B0, %A0", operands);
          OUT_INSN (len, "xor\t%B0, %A0", operands);
          OUT_INSN (len, "and.b\t#-1, %B0", operands);
          dummy = 12;

          if(REG_P(operands[0])) dummy = 5;
          shiftpos -= 8;
          while(shiftpos--)
            {
              OUT_INSN (len, second_pat, operands);
              dummy += set_rel(operands[0],4,1) - 1;
            }
          break;

        case 16:
        case 17:
        case 18:
        case 19:
        case 20:
        case 21:
        case 22:
        case 23:
          OUT_INSN (len, "mov\t%B0, %A0", operands);
          OUT_INSN (len, "clr\t%B0",  operands);
          dummy = 5;
          if(REG_P(operands[0])) dummy = 2;

          shiftpos -= 16;
          if(shiftpos)
            {
              OUT_INSN (len, "clrc\n\trrc\t%A0",operands);
              dummy += 2;
              if(!zs && !REG_P(operands[0])) dummy++;
              shiftpos--;
            }

          while(shiftpos--)
            {
              OUT_INSN (len, "rra\t%A0", operands);
              dummy += 1;
              if(!zs && !REG_P(operands[0])) dummy++;
            }
          break;

        case 24:
        case 25:
        case 26:
        case 27:
        case 28:
        case 29:
        case 30:
          OUT_INSN (len, "mov\t%B0, %A0", operands);
          OUT_INSN (len, "clr\t%B0",  operands);
          OUT_INSN (len, "swpb\t%A0",  operands);
          OUT_INSN (len, "and.b\t#-1, %A0", operands);
          dummy = 9;
          if(REG_P(operands[0])) dummy = 4;
          if(indexed_location(operands[0])) dummy -= 1;
          shiftpos -= 24;

          while(shiftpos--)
            {
              OUT_INSN (len, "rra\t%A0", operands);
              dummy += 1;
              if(!zs && !REG_P(operands[0])) dummy++;
            }
          break;

        case 31:
          OUT_INSN (len, "rla\r%B0", operands);
          OUT_INSN (len, "clr\t%B0",  operands);
          OUT_INSN (len, "clr\t%A0",  operands);
          OUT_INSN (len, "rlc\t%A0",  operands);
          dummy = 8;
          if(REG_P(operands[0])) dummy = 4;
          if(indexed_location(operands[0])) dummy -= 1;
          break;

        default:
          OUT_INSN (len, "clr\t%B0",  operands);
          OUT_INSN (len, "clr\t%A0",  operands);
          dummy = 4;
          if(REG_P(operands[0])) dummy = 2;
          break;

        } /* switch */

      if(len) *len = dummy;
      return "";
    }
  else
    msp430_emit_shift_cnt(set_rel,pattern, insn, operands, len, 4);

  return "";

}


const char *
msp430_emit_lshrdi3(insn, operands,len)
rtx insn;
rtx operands[];
int *len;
{

  int dummy = 0;
  const char *pattern;
  int zs = zero_shifted(operands[0]) || indexed_location(operands[0]);
  const char *secondary_pat = "rra\t%D0\n\trrc\t%C0\n\trrc\t%B0\n\trrc\t%A0";

  pattern = "clrc\n\trrc\t%D0\n\trrc\t%C0\n\trrc\t%B0\n\trrc\t%A0";

  if(GET_CODE (operands[2]) == CONST_INT)
    {
      int shiftpos = INTVAL(operands[2]);

      if(shiftpos < 16 && shiftpos)
        {
          OUT_INSN (len, pattern, operands);
          dummy += set_rel(operands[0],2,1);
          shiftpos--;
        }

      switch(shiftpos)
        {
        case 0:
          break;

        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
        case 8:
        case 9:
        case 10:
        case 11:
        case 12:
        case 13:
        case 14:
        case 15:
          while(shiftpos--)
            {
              OUT_INSN (len, secondary_pat, operands);
              dummy += set_rel(operands[0],8,1) - 1;
            }

          break;

        case 16:
        case 17:
        case 18:
        case 19:
        case 20:
        case 21:
        case 22:
        case 23:
        case 24:
        case 25:
        case 26:
        case 27:
        case 28:
        case 29:
        case 30:
        case 31:
          OUT_INSN (len, "mov\t%B0, %A0",operands);
          OUT_INSN (len, "mov\t%C0, %B0",operands);
          OUT_INSN (len, "mov\t%D0, %C0",operands);
          OUT_INSN (len, "clr\t%D0",operands);
          dummy = 11;
          if(REG_P(operands[0])) dummy = 4;
          shiftpos -= 16;

          if(shiftpos)
            {
              OUT_INSN (len, secondary_pat,operands);
              dummy += set_rel(operands[0],8,1)-1;
              shiftpos--;
            }

          while(shiftpos--)
            {
              OUT_INSN (len, "rra\t%C0\n\trrc\t%B0\n\trrc\t%A0", operands);
              if(REG_P(operands[0]))
                dummy = 3;
              else
                dummy += 6;
              if(zs) dummy--;
            }

          break;

        case 32:
        case 33:
        case 34:
        case 35:
        case 36:
        case 37:
        case 38:
        case 39:
        case 40:
        case 41:
        case 42:
        case 43:
        case 44:
        case 45:
        case 46:
        case 47:
          OUT_INSN (len, "mov\t%C0, %A0",operands);
          OUT_INSN (len, "mov\t%D0, %B0",operands);
          OUT_INSN (len, "clr\t%C0",operands);
          OUT_INSN (len, "clr\t%D0",operands);

          dummy = 10;
          if(GET_CODE(operands[0]) == REG) dummy = 4;

          shiftpos -= 32;

          if(shiftpos)
            {
              OUT_INSN (len,"clrc\n\trrc\t%B0,rrc\t%A0",operands);
              if(REG_P(operands[0]))
                dummy += 3;
              else
                dummy += 5;
              if(zs) dummy--;
              shiftpos--;
            }

          while(shiftpos--)
            {
              OUT_INSN (len,"rra\t%B0,rrc\t%A0",operands);
              if(REG_P(operands[0]))
                dummy += 2;
              else
                dummy += 4;
              if(zs) dummy--;
            }
          break;

        case 48:
        case 49:
        case 50:
        case 51:
        case 52:
        case 53:
        case 54:
        case 55:
        case 56:
        case 57:
        case 58:
        case 59:
        case 60:
        case 61:
        case 62:
          OUT_INSN (len, "mov\t%D0, %A0",operands);
          OUT_INSN (len, "clr\t%B0",operands);
          OUT_INSN (len, "clr\t%C0",operands);
          OUT_INSN (len, "clr\t%D0",operands);
          dummy = 9;
          if(GET_CODE(operands[0]) == REG) dummy = 4;
          shiftpos -= 48;

          if(shiftpos)
            {
              OUT_INSN (len,"clrc\n\trrc\t%A0",operands);
              if(REG_P(operands[0]) || zs )
                dummy += 2;
              else
                dummy += 3;

              shiftpos--;
            }

          while(shiftpos--)
            {
              OUT_INSN (len, "rra\t%A0", operands);
              if(REG_P(operands[0]) || zs )
                dummy++;
              else
                dummy += 2;
            }
          break;

        case 63:

          OUT_INSN (len, "rla\t%D0",operands);
          OUT_INSN (len, "clr\t%D0",operands);
          OUT_INSN (len, "clr\t%C0",operands);
          OUT_INSN (len, "clr\t%B0",operands);
          OUT_INSN (len, "clr\t%A0",operands);
          OUT_INSN (len, "rlc\t%A0",operands);
          if(REG_P(operands[0]))
            dummy += 6;
          else
            dummy += 13;

          if(zs) dummy--;
          break;

        default:
          msp430_emit_shift_cnt(set_rel,pattern, insn, operands, len, 8);
        } /* case */

      if(len) *len = dummy;
    }
  else
    msp430_emit_shift_cnt(set_rel,pattern, insn, operands, len, 8);

  return "";
}



/* is something like 'push x(r1)' or 'push @r1' */

int
self_push(x)
rtx x;
{
  rtx c;
  rtx r;

  if(GET_CODE(x) != MEM) return 0;

  c = XEXP(x,0);

  if(REG_P(c) && REGNO(c) == 1) return 1;

  if(GET_CODE(c) == PLUS)
    {
      r = XEXP(c,0);
      if(REG_P(r) && REGNO(r) == 1) return 1;
    }
  return 0;
}

/* difficult pushes.
   if planets are not aligned, the combiner does not allocate
   r4 as a frame pointer. Instead, it uses stack pointer for frame.
   If there is a va_arg call and non-register local var has to be passed 
   as a function parameter, the push X(r1) in SI, SF and DI modes will
   corrupt passed var. The following mind this fact */


const char *
msp430_pushqi(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int sp = self_push(operands[0]);
  int dummy = 0;

  if(sp)
    {
      rtx c;
      c = XEXP(operands[0],0);
      if(REG_P(c))
        OUT_INSN (len, "push.b\t2(%E0)",operands);
      else
        OUT_INSN (len, "push.b\t2+%A0",operands);
      dummy = 2;
    }
  else
    {
      OUT_INSN (len, "push.b\t%A0",operands);
      dummy = 2;

      if(GET_CODE(operands[0])==CONST_INT)
        {
          int cval = INTVAL(operands[0]);
          int x = (cval)&0x0fffful;
          if(x==0||x==1||x==2||x==4||x==8||x==0xffff) dummy--;

        }
      else if(GET_CODE(operands[0])==REG)
        dummy--;
      else if(GET_CODE(operands[0])==MEM
              && REG_P(XEXP(operands[0],0)) )
        dummy--;
    }

  return "";
}


const char *
msp430_pushhi(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int sp = self_push(operands[0]);
  int dummy = 0;

  if(sp)
    {
      rtx c;
      c = XEXP(operands[0],0);
      if(REG_P(c))
        OUT_INSN (len, "push\t2(%E0)",operands);
      else
        OUT_INSN (len, "push\t2+%A0",operands);
      dummy = 2;
    }
  else
    {
      OUT_INSN (len, "push\t%A0",operands);
      dummy = 2;

      if(GET_CODE(operands[0])==CONST_INT)
        {
          int cval = INTVAL(operands[0]);
          int x = (cval)&0x0fffful;
          if(x==0||x==1||x==2||x==4||x==8||x==0xffff) dummy--;

        }
      else if(GET_CODE(operands[0])==REG)
        dummy--;
      else if(GET_CODE(operands[0])==MEM
              && REG_P(XEXP(operands[0],0)) )
        dummy--;
    }
  if(len) *len = dummy;
  return "";
}

const char *
msp430_pushsisf(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int sp = self_push(operands[0]);
  int dummy = 0;

  if( !sp)
    {
      OUT_INSN (len, "push\t%B0",operands);
      OUT_INSN (len, "push\t%A0",operands);
      dummy = 4;
      if(indexed_location(operands[0])) dummy--;
      if(REG_P (operands[0])) dummy -= 2;
      if(GET_CODE(operands[0])==CONST_INT)
        {
          int cval = INTVAL(operands[0]);
          int x = (cval)&0x0fffful;
          int y = (((unsigned long)(cval))&0xffff0000ul >> 16);
          if(x==0||x==1||x==2||x==4||x==8||x==0xffff) dummy--;
          if(y==0||y==1||y==2||y==4||y==8||y==0xffff) dummy--;
        }
      else if( GET_CODE(operands[0])==CONST_DOUBLE
               && GET_MODE (operands[0]) == SFmode)
        {
          long val;
          int  y,x;
          REAL_VALUE_TYPE rv;
          REAL_VALUE_FROM_CONST_DOUBLE (rv, operands[0]);
          REAL_VALUE_TO_TARGET_SINGLE (rv, val);

          y = (val&0xffff0000ul)>>16;
          x = val&0xffff;
          if(x==0||x==1||x==2||x==4||x==8||x==0xffff) dummy--;
          if(y==0||y==1||y==2||y==4||y==8||y==0xffff) dummy--;
        }
    }
  else
    {
      OUT_INSN (len, "push\t2+%B0",operands);
      OUT_INSN (len, "push\t2+%B0",operands);
      dummy = 4;
    }

  if(len) *len = dummy;

  return "";
}


const char *
msp430_pushdi(insn, operands, len)
rtx insn ATTRIBUTE_UNUSED;
rtx operands[];
int *len;
{
  int sp = self_push(operands[0]);
  int dummy = 0;

  if( !sp)
    {
      OUT_INSN (len, "push\t%D0",operands);
      OUT_INSN (len, "push\t%C0",operands);
      OUT_INSN (len, "push\t%B0",operands);
      OUT_INSN (len, "push\t%A0",operands);

      dummy = 8;
      if(indexed_location(operands[0])) dummy--;
      if(REG_P (operands[0])) dummy -= 4;
      if(GET_CODE(operands[0])==CONST_DOUBLE)
        {
          int hi = CONST_DOUBLE_HIGH (operands[0]);
          int lo = CONST_DOUBLE_LOW (operands[0]);
          int x,y,z;

          x = (hi&0xffff0000ul)>>16;
          y =  hi&0xffff;
          z = (lo&0xffff0000ul)>>16;
          if(x==0||x==1||x==2||x==4||x==8||x==0xffff) dummy--;
          if(y==0||y==1||y==2||y==4||y==8||y==0xffff) dummy--;
          if(z==0||z==1||z==2||z==4||z==8||z==0xffff) dummy--;
          z = lo&0xffff;
          if(z==0||z==1||z==2||z==4||z==8||z==0xffff) dummy--;
        }
    }
  else
    {
      OUT_INSN (len, "push\t2+%D0",operands);
      OUT_INSN (len, "push\t2+%D0",operands);
      OUT_INSN (len, "push\t2+%D0",operands);
      OUT_INSN (len, "push\t2+%D0",operands);
      dummy = 8;
    }

  if(len) *len = dummy;

  return "";
}

int
dead_or_set_in_peep(which, insn, x)
int which;
rtx insn;
rtx x;
{
  rtx r;

  while(which--) insn = next_real_insn(insn);

  /* something weared is going on here.
     so, it needs to be reconsidered a lot!!! */

  if(!REG_P(x)) return 0;
  r = find_regno_note(insn, REG_DEAD, REGNO(x));

  return r?1:0;

}













