#include <stdio.h>
#include <stdlib.h>
#include <Fourier.h>
#include <cmdLineParser.h>
#include <image.h>
#include "gridIO.h"

using namespace std;
using namespace Util;
using namespace Image;

CmdLineParameter< string > Input( "in" ) , Pattern( "pattern" ) , Output( "out" );
CmdLineReadable Double( "double" );

CmdLineReadable* params[] =
{
	&Input , &Output , &Pattern , &Double ,
	NULL
};

void ShowUsage( const string &ex )
{
	cout << "Usage " << ex << ":" << endl;
	cout << "\t --" << Input.name   << " <input file>"    << endl;
	cout << "\t --" << Pattern.name << " <pattern image>" << endl;
	cout << "\t --" << Output.name  << " <output image>"  << endl;
	cout << "\t[--" << Double.name  << "]"                << endl;
}

template< typename Real >
void Execute( void )
{
	SquareGrid< Real > input[CHANNEL_COUNT] , pattern[CHANNEL_COUNT] , mask;

	// Read in the red, green, and blue components of the image
	ReadImage( Input.value , input );
	int res = input[0].resolution();

	// Read in the red, green, and blue components of the pattern
	SquareGrid< Real > temp[3];
	ReadImage( Pattern.value , temp );
	int pRes = temp[0].resolution();

	// Check that the pattern is smaller than the image
	if( res<=pRes ) THROW( "Pattern must be smaller than image: %d < %d" , pRes , res );

	// Copy the pattern to a re-centered, zero-padded image, and set the mask
	mask.resize( res );
	for( int c=0 ; c<CHANNEL_COUNT ; c++ )
	{
		pattern[c].resize( res );
		for( int i=0 ; i<pRes ; i++ ) for( int j=0 ; j<pRes ; j++ )
		{
			pattern[c](i-pRes/2,j-pRes/2) = temp[c](i,j);
			mask(i-pRes/2,j-pRes/2) = 1;
		}
	}

	/*********************************/
	/* Do the patterm matching here. */
	/*********************************/
	WARN( "functionality not implemented" );


	// The average value of (x-y)*(x-y) with x,y in [0,1] is 1/6.
	// Multiplying this by the square norm of the mask gives us the expected square L2-difference of two random signals on the mask.
	// We divide the moving square L2-difference by this value to get a function that should roughly be in the range [0,1]
	Real scale = (Real)mask.squareNorm()/6;

	// Normalize by the expected error
	for( int c=0 ; c<CHANNEL_COUNT ; c++ ) for( int i=0 ; i<res ; i++ ) for( int j=0 ; j<res ; j++ ) input[c](i,j) /= scale;

	// Invert the image so that brighter spots correspond to better matches
	for( int c=0 ; c<CHANNEL_COUNT ; c++ ) for( int i=0 ; i<res ; i++ ) for( int j=0 ; j<res ; j++ ) input[c](i,j) = (Real)( 1. - input[c](i,j) );

	// Write the image out to file
	WriteImage( Output.value , input );
}

int main( int argc , char *argv[] )
{
	CmdLineParse( argc-1 , argv+1 , params );
	if( !Input.set || !Pattern.set || !Output.set ){ ShowUsage( argv[0] ) ; return EXIT_FAILURE; }
	try
	{
		if( Double.set ) Execute< double >();
		else             Execute< float  >();
	}
	catch( Exception e )
	{
		cout << e.what() << endl;
		return EXIT_FAILURE;
	}
	return EXIT_SUCCESS;
}