#include "image.h"
#include "SphericalGrid.h"

#ifndef GRID_IO_INCLUDED
#define GRID_IO_INCLUDED

enum
{
	CHANNEL_RED ,
	CHANNEL_GREEN ,
	CHANNEL_BLUE ,
	CHANNEL_COUNT
};

///////////////////////////
// Function declarations //
///////////////////////////
template< typename Real >
inline void ReadImage( std::string fileName , SquareGrid< Real > rgb[CHANNEL_COUNT] );

template< typename Real >
inline void WriteImage( std::string fileName , const SquareGrid< Real > rgb[CHANNEL_COUNT] );

template< class Real >
inline void DrawCircularArray( std::string fileName , const CircularArray< Real > &values , int res );

template< class Real >
inline void DrawHistogram( std::string fileName , const CircularArray< Real > &values , int res );

template< typename Real >
inline void ReadSphericalGrid( std::string fileName , SphericalGrid< Real > &grid );

template< typename Real >
inline void WriteSphericalGrid( std::string fileName , const SphericalGrid< Real > &grid );

//////////////////////////
// Function definitions //
//////////////////////////
template< typename Real >
inline void ReadImage( std::string fileName , SquareGrid< Real > rgb[CHANNEL_COUNT] )
{
	Image::Image32 img;
	img.read( fileName );

	int w = img.width();
	int h = img.height();
	int res = std::max< int >( w , h );

	for( int c=0 ; c<CHANNEL_COUNT ; c++ ) rgb[c].resize( res );

	for( int i=0 ; i<w ; i++ ) for( int j=0 ; j<h ; j++ )
	{
		Image::Pixel32 p = img(i,j);
		rgb[0](i,j) =(Real)( p.r / 255. );
		rgb[1](i,j) =(Real)( p.g / 255. );
		rgb[2](i,j) =(Real)( p.b / 255. );
	}
}

template< typename Real >
inline void WriteImage( std::string fileName , const SquareGrid< Real > rgb[CHANNEL_COUNT] )
{
	if( rgb[0].resolution()!=rgb[1].resolution() || rgb[0].resolution()!=rgb[2].resolution() ) THROW( "RGB resolutions differ: %d %d %d" , rgb[0].resolution() , rgb[1].resolution() , rgb[2].resolution() );

	Image::Image32 img;
	int res = rgb[0].resolution();
	img.setSize( res , res );

	for( int i=0 ; i<res ; i++ ) for( int j=0 ; j<res ; j++ )
	{
		Image::Pixel32 p;
		p.r = (unsigned char)( std::max< Real >( 0 , std::min< Real >( 255 , rgb[0](i,j) * 255 ) ) );
		p.g = (unsigned char)( std::max< Real >( 0 , std::min< Real >( 255 , rgb[1](i,j) * 255 ) ) );
		p.b = (unsigned char)( std::max< Real >( 0 , std::min< Real >( 255 , rgb[2](i,j) * 255 ) ) );
		img(i,j) = p;
	}

	img.write( fileName );
}

template< class Real >
inline void DrawCircularArray( std::string fileName , const CircularArray< Real > &values , int res )
{
	Image::Image32 img;
	img.setSize( res , res );
	Real c = (Real)( res / 2. );

	for( int i=0 ; i<res ; i++ )
	{
		Real x = (Real)( i-c );
		for( int j=0 ; j<res ; j++ )
		{
			Real y = (Real)( j-c );
			Real r = (Real)sqrt( x*x + y*y )/c;
			Real theta = (Real)atan2( -y , x );
			Image::Pixel32 p;
			if( r<values( (Real)( theta / (2.*PI) * values.resolution() ) ) ) p.r = p.g = p.b = 0;
			else                                                              p.r = p.g = p.b = 255;
			img(i,j) = p;
		}
	}

	img.write( fileName );
}

template< class Real >
inline void DrawHistogram( std::string fileName , const CircularArray< Real > &values , int res )
{
	Image::Image32 img;
	img.setSize( res , res );
	Real c = (Real)( res/2. );

	for( int i=0 ; i<res ; i++ )
	{
		int idx = (int)( ( (Real)i ) / res*values.resolution() );
		for( int j=0 ; j<res ; j++ )
		{
			Real y = (Real)( 1. - ( (Real)j )/res );
			Image::Pixel32 p;
			if( y<values(idx) ) p.r = p.g = p.b = 0;
			else                p.r = p.g = p.b = 255;
			img(i,j) = p;
		}
	}

	img.write( fileName );
}

template< typename Real >
inline void ReadSphericalGrid( std::string fileName , SphericalGrid< Real > &grid )
{
	SphericalGrid< float > _grid;
	if( !_grid.read( fileName.c_str() ) ) THROW( "failed to read spherical grid: %s" , fileName.c_str() );
	grid.resize( _grid.resolution() );
	for( int i=0 ; i<_grid.resolution() ; i++ ) for( int j=0 ; j<_grid.resolution() ; j++ ) grid(i,j) = (Real)_grid(i,j);
}

template< typename Real >
inline void WriteSphericalGrid( std::string fileName , const SphericalGrid< Real > &grid )
{
	SphericalGrid< float > _grid;
	_grid.resize( grid.resolution() );
	for( int i=0 ; i<_grid.resolution() ; i++ ) for( int j=0 ; j<_grid.resolution() ; j++ ) _grid(i,j) = (float)grid(i,j);
	if( !_grid.write( fileName.c_str() ) ) THROW( "failed to write spherical grid: %s" , fileName.c_str() );
}


#endif // GRID_IO_INCLUDED