#ifndef SPHERICAL_GRID_INCLUDED
#define SPHERICAL_GRID_INCLUDED

#ifndef PI
#define PI 3.1415926535897932384
#endif 

// This templated class represents a spherical function, where index (i,j) corresponds to the
// point on the sphere with spherical coordinates:
//			theta =2PI*i/r
//			phi   = PI*(2j+1)/(2r)
//			(theta,phi) -> [cos(theta)*sin(phi),cos(phi),sin(theta)*sin(phi)]
// where r is the resolution of the sampling.
template< class Real >
class SphericalGrid
{
protected:
	Real* values;
	int res;
	static void Transform(const Real matrix[3][3],const Real* in,Real* out);
public:
	SphericalGrid( void );
	~SphericalGrid( void );

	// Returns the dimension of the array
	int resolution( void ) const;
	// Allocates memory for the array
	int resize(int resolution);

	// Clears the values of the array to 0
	void clear( void );

	// Returns a reference to the indexed array element
	Real &operator() ( int x , int y );
	const Real &operator() ( int x , int y ) const { return const_cast< SphericalGrid * >(this)->operator()(x,y); }
	Real *operator[] ( int x );
	const Real *operator[] ( int x ) const { return const_cast< SphericalGrid * >(this)->operator[](x); }
	// Returns the linear interpolation of the value at the specified index
	Real operator() ( Real x , Real y );
	Real operator() ( Real x , Real y ) const { return const_cast< SphericalGrid * >(this)->operator()(x,y); }
	// Returns the linear interpolation of the value at the specified spherical point
	Real operator()( const Real axis[3] );
	Real operator()( const Real axis[3] ) const { return const_cast< SphericalGrid * >(this)->operator()(axis); }

	// Returns the square of the L2-norm of the array elements
	Real squareNorm( void ) const;

	// Reads in an array from the specified file
	int read(const char* fileName);
	int read(FILE* fp);

	// Writes out the array to the specified file
	int write(const char* fileName) const;
	int write(FILE* fp) const;

	// Sets the (x,y,z) coordinates of the spherical point indexed by (theta,phi)
	static void SetCoordinates(Real theta,Real phi,Real coords[3]);
	// Sets the (theta,phi) coordinates of the spherical point (x,y,z)
	static void SetCoordinates(const Real coords[3],Real& theta,Real& phi);

	// Sets the (x,y,z) coordinates of the spherical point indexed by (i,j)
	void setCoordinates(Real i,Real j,Real coords[3]) const;
	// Sets the (i,j) coordinates of the spherical point (x,y,z)
	void setCoordinates(const Real coords[3],Real& i,Real& j) const;

	// Returns the square of the L2-difference between two spherical grids
	static Real SquareDifference(const SphericalGrid& g1,const SphericalGrid& g2);

	// Returns the dot-product of two spherical grids
	static Real Dot(const SphericalGrid& g1,const SphericalGrid& g2);

	// Rotates a spherical grid
	static void Rotate(const SphericalGrid& in,const Real rotation[3][3],SphericalGrid& out);
};
#include "SphericalGrid.inl"
#endif // SPHERICAL_GRID_INCLUDED

