#include <stdio.h>
#include <stdlib.h>
#include <limits>
#include "Fourier.h"
#include "SphericalGrid.h"
#include "cmdLineParser.h"
#include "gridIO.h"

using namespace std;
using namespace Util;

CmdLineParameter< string > Source( "source" ) , Target( "target" ) , Output( "out" );
CmdLineReadable Double( "double" );

CmdLineReadable* params[] =
{
	&Source , &Target , &Output , &Double ,
	NULL
};

void ShowUsage( const string &ex )
{
	cout << "Usage " << ex << ":" << endl;
	cout << "\t --" << Source.name << " <source spherical grid>"  << endl;
	cout << "\t --" << Target.name << " <target spherical grid>"  << endl;
	cout << "\t --" << Output.name << " <aligned spherical grid>" << endl;
	cout << "\t[--" << Double.name                         << "]" << endl;
}

template< typename Real >
void Execute( void )
{
	SphericalGrid< Real > source , target , out;
	RotationGrid< Real > euler;
	FourierKeyS2< Real > sourceKey , targetKey;
	FourierKeySO3< Real > keySO3;
	HarmonicTransform< Real > hForm;
	WignerTransform< Real > wForm;
	Real matrix[3][3];

	// Read in the input spherical function
	ReadSphericalGrid( Source.value , source );
	ReadSphericalGrid( Target.value , target );
	if( source.resolution()!=target.resolution() ) THROW( "Source and target resolutions differ: %d != %d" , source.resolution() , target.resolution() );

	// Allocate memory
	keySO3.resize( source.resolution() );

	// Compute the spherical harmonic transform
	hForm.ForwardFourier( source , sourceKey );
	hForm.ForwardFourier( target , targetKey );

	/*******************************************************************************************************/
	/* Compute the aligning rotation here.                                                                 */
	/* Perform the correlation here and find the aligning rotation, writing it into the variable "matrix". */
	/* Note that if you have the coordinates of the optimal correlation, you can invoke the function:      */
	/*     RotationGrid::setCoordinates( Real i , Real j , Real k , Real matrix[3][3] ) const;             */
	/* to set the "matrix" entries from the 3D indices (i,j,k)                                             */
	/*******************************************************************************************************/
	WARN( "functionality not implemented" );


	// Rotate the source to register it with the target and write it out to a file
	SphericalGrid< Real >::Rotate( source , matrix , out );
	WriteSphericalGrid( Output.value , out );
}

int main( int argc , char* argv[] )
{
	CmdLineParse( argc-1 , argv+1 , params );
	if( !Source.set || !Target.set || !Output.set ){ ShowUsage( argv[0] ) ; return EXIT_FAILURE; }
	try
	{
		if( Double.set ) Execute< double >();
		else             Execute< float  >();
	}
	catch( Exception e )
	{
		cout << e.what() << endl;
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}

