#include "Fortune.h"
#include "Util/Visualization.h"

#ifndef M_PI
#define M_PI		3.14159265358979323846
#endif // M_PI

struct VoronoiDiagram2DVisualization : public Misha::Viewable< VoronoiDiagram2DVisualization >
{
	static const unsigned int RESOLUTION;

	double maximumAnimationHeight;
	bool verbose;
	unsigned int selectedSite;
	Fortune::VertexData selectedVertex;
	bool showCircumcircles , showDeletions , showFullParabolas , showVoronoi;
	Fortune::State fortuneState;
	double stepSize;
	VoronoiDiagram2DVisualization( const std::vector< Geometry::Point2i > &sites );
	void advance( double height );
	void advance( void );
	double dotRadius;

	Geometry::Point2d select( int x , int y );

	void idle( void );
	void keyboardFunc( unsigned char key , int x , int y );
	void specialFunc( int key, int x, int y );
	void display( void );
	void mouseFunc( int button , int state , int x , int y );
	void motionFunc( int x , int y );
	void passiveMotionFunc( int x , int y );
	Geometry::Point2d SiteToScreen( Geometry::Point2d s );
	Geometry::Point2d SiteToScreen( Geometry::Point2i s );
	Geometry::Point2d ScreenToSite( Geometry::Point2d s );
	static void   IncreaseDotRadiusCallBack( VoronoiDiagram2DVisualization *v , const char * ){ v->dotRadius *= 1.1; }
	static void   DecreaseDotRadiusCallBack( VoronoiDiagram2DVisualization *v , const char * ){ v->dotRadius /= 1.1; }
	static void         SetStepSizeCallBack( VoronoiDiagram2DVisualization *v , const char *prompt ){ v->stepSize = atof(prompt); }
	static void ToggleCircumCirclesCallBack( VoronoiDiagram2DVisualization *v , const char * ){ v->showCircumcircles = !v->showCircumcircles; }
	static void     ToggleDeletionsCallBack( VoronoiDiagram2DVisualization *v , const char * ){ v->showDeletions = !v->showDeletions; }
	static void ToggleFullParabolasCallBack( VoronoiDiagram2DVisualization *v , const char * ){ v->showFullParabolas = !v->showFullParabolas; }
	static void      ToggleVoronoiCallBack( VoronoiDiagram2DVisualization *v , const char * ){ v->showVoronoi = !v->showVoronoi; }
	static void             AdvanceCallBack( VoronoiDiagram2DVisualization *v , const char * ){ v->advance( Fortune::BeachLineElement::Height + v->stepSize ); }
	static void  AdvanceToNextEventCallBack( VoronoiDiagram2DVisualization *v , const char * ){ v->advance(); }
	static void      PrintBeachLineCallBack( VoronoiDiagram2DVisualization *v , const char * ){ std::cout << v->fortuneState.beachLine << std::endl; }
	static void      PrintEventListCallBack( VoronoiDiagram2DVisualization *v , const char * ){ std::cout << v->fortuneState.eventList << std::endl; }
	static void           MakeVideoCallBack( VoronoiDiagram2DVisualization *v , const char *prompt )
	{
		char header[512];
		float maxY;
		if( sscanf( prompt , " %s | %f " , header , &maxY )==2 )
		{
			v->showHelp = v->showInfo = v->showFPS = false;
			v->maximumAnimationHeight = maxY;
			v->setVideo( header , (int)ceil( ( maxY-Fortune::BeachLineElement::Height ) / v->stepSize ) , false );
		}
	}

protected:
	std::vector< Geometry::Point2i > _sites;
	Geometry::Point2d _center;
	double _radius;
	char _positionLine[512] , _stepSizeLine[512] , _sweepHeightLine[512] , _selectedLine[512];
};
const unsigned int VoronoiDiagram2DVisualization::RESOLUTION = 100;

Geometry::Point2d VoronoiDiagram2DVisualization::SiteToScreen( Geometry::Point2d s )
{
	Geometry::Point2d p;
	p[0] = ( s[0] - _center[0] ) / _radius;
	p[1] = ( s[1] - _center[1] ) / _radius;
	return p;
}
Geometry::Point2d VoronoiDiagram2DVisualization::SiteToScreen( Geometry::Point2i s )
{
	Geometry::Point2d p;
	p[0] = ( s[0] - _center[0] ) / _radius;
	p[1] = ( s[1] - _center[1] ) / _radius;
	return p;
}
Geometry::Point2d VoronoiDiagram2DVisualization::ScreenToSite( Geometry::Point2d s )
{
	Geometry::Point2d p;
	p[0] = s[0] * _radius + _center[0];
	p[1] = s[1] * _radius + _center[1];
	return p;
}

VoronoiDiagram2DVisualization::VoronoiDiagram2DVisualization( const std::vector< Geometry::Point2i > &sites ) : _sites(sites) , fortuneState(sites)
{
	maximumAnimationHeight = -std::numeric_limits<double>::infinity();
	verbose = false;
	selectedSite = -1;
	selectedVertex.radius = -1;
	showCircumcircles = false;
	showDeletions = true;
	showFullParabolas = false;
	showVoronoi = true;

	Geometry::Point2i min , max;
	min = max = _sites[0];
	for( int i=0 ; i<_sites.size() ; i++ ) for( int j=0 ; j<2 ; j++ ) min[j] = std::min< int >( min[j] , _sites[i][j] ) , max[j] = std::max< int >( max[j] , _sites[i][j] );
	_center[0] = ( min[0] + max[0] ) / 2.;
	_center[1] = ( min[1] + max[1] ) / 2.;
	_radius = std::max< int >( max[0]-min[0] , max[1]-min[1] ) / 2.;
	dotRadius = (double)_radius/50;
	_radius *= 1.5;

	stepSize = 0.1;

	callBacks.push_back( KeyboardCallBack( this , '[' , KeyboardCallBack::Modifiers() , "decrease dot radius" , DecreaseDotRadiusCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , ']' , KeyboardCallBack::Modifiers() , "increase dot radius" , IncreaseDotRadiusCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , ' ' , KeyboardCallBack::Modifiers() , "advance to next event" , AdvanceToNextEventCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , '+' , KeyboardCallBack::Modifiers() , "advance" , AdvanceCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , 's' , KeyboardCallBack::Modifiers() , "set step size" , "Step-size" , SetStepSizeCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , 'p' , KeyboardCallBack::Modifiers() , "print beach line" , PrintBeachLineCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , 'P' , KeyboardCallBack::Modifiers() , "print event list" , PrintEventListCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , 'c' , KeyboardCallBack::Modifiers() , "toggle circumcircles" , ToggleCircumCirclesCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , 'C' , KeyboardCallBack::Modifiers() , "toggle deletions" , ToggleDeletionsCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , 'f' , KeyboardCallBack::Modifiers() , "toggle full parabolas" , ToggleFullParabolasCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , 'v' , KeyboardCallBack::Modifiers() , "toggle Voronoi" , ToggleVoronoiCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , 'V' , KeyboardCallBack::Modifiers() , "make video" , "Header | Max-Y" , MakeVideoCallBack ) );

	info.push_back( _selectedLine );
	info.push_back( _positionLine );
	info.push_back( _stepSizeLine );
	info.push_back( _sweepHeightLine);
	sprintf( _positionLine , "Position: " );
	sprintf( _selectedLine , "Selected: " );
}

void VoronoiDiagram2DVisualization::advance( double height )
{
	if( height<=Fortune::BeachLineElement::Height ) ERROR_OUT( "Cannot move sweep-line backward" );
	while( fortuneState.eventList.size() && ( *fortuneState.eventList.begin() ).eventPosition()[1]<height ) fortuneState.processNextEvent( verbose );
	Fortune::BeachLineElement::Height = height;
}

void VoronoiDiagram2DVisualization::advance( void )
{
	if( fortuneState.eventList.size() ) fortuneState.processNextEvent( verbose );
}

void VoronoiDiagram2DVisualization::display( void )
{
	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	float ar = (float)screenWidth/(float)screenHeight , ar_r = 1.f/ar;
	if( screenWidth>screenHeight ) glOrtho( -ar , ar , -1 , 1 , -1.5 , 1.5 );
	else                           glOrtho( -1 , 1 , -ar_r , ar_r , -1.5 , 1.5 );

	glMatrixMode( GL_MODELVIEW );
	glLoadIdentity();

	double xMin , xMax;
	{
		Geometry::Point2d p;
		p[0] = -1.5;
		p = ScreenToSite( p );
		xMin = p[0];
		p[0] =  1.5;
		p = ScreenToSite( p );
		xMax = p[0];
	}

	auto DrawDisk = [&]( Geometry::Point2d c , double r , unsigned int res )
	{
		glBegin( GL_POLYGON );
		for( unsigned int i=0 ; i<res ; i++ )
		{
			double angle = (2.*M_PI*i) / res;
			Geometry::Point2d p;
			p[0] = c[0] + cos( angle ) * r;
			p[1] = c[1] + sin( angle ) * r;
			p = SiteToScreen( p );
			glVertex2d( p[0] , p[1] );
		}
		glEnd();
	};

	auto DrawCircle = [&]( Geometry::Point2d c , double r , unsigned int res )
	{
		glBegin( GL_LINE_LOOP );
		for( unsigned int i=0 ; i<res ; i++ )
		{
			double angle = (2.*M_PI*i) / res;
			Geometry::Point2d p;
			p[0] = c[0] + cos( angle ) * r;
			p[1] = c[1] + sin( angle ) * r;
			p = SiteToScreen( p );
			glVertex2d( p[0] , p[1] );
		}
		glEnd();
	};

	// Draw the sweep line
	{
		glLineWidth(2);
		glColor3f( 0 , 0 , 0 );
		glBegin( GL_LINES );
		{
			Geometry::Point2d p;
			p[0] = xMin , p[1] = Fortune::BeachLineElement::Height;
			p = SiteToScreen( p );
			glVertex2d( p[0] , p[1] );
		}
		{
			Geometry::Point2d p;
			p[0] = xMax , p[1] = Fortune::BeachLineElement::Height;
			p = SiteToScreen( p );
			glVertex2d( p[0] , p[1] );
		}
		glEnd();
	}

	// Draw the beach line
	{
		int count = 0;
		for( auto iter=fortuneState.beachLine.begin() ; iter!=fortuneState.beachLine.end() ; iter++ )
		{
			if( count%2 ) glColor3f( 1 , 0 , 0 );
			else          glColor3f( 0 , 1 , 0 );
			auto next=iter;
			next++;
			if( next!=fortuneState.beachLine.end() )
			{
				Fortune::BeachLine::Arc arc( iter->endPoint , next->endPoint );
				Polynomial1D< 2 > P = Fortune::ParabolicFront( iter->endPoint.rightFace->data.site , Fortune::BeachLineElement::Height );
				double start = arc.leftEndPoint() , end = arc.rightEndPoint();
				if( end>xMin || start<xMax )
				{
					start = std::max< double >( start , xMin );
					end = std::min< double >( end , xMax );
					if( showFullParabolas )
					{
						glLineWidth(2);
						glBegin( GL_LINE_STRIP );
						static const int _RESOLUTION = RESOLUTION * 100;
						for( int i=0 ; i<=_RESOLUTION ; i++ )
						{
							Geometry::Point2d p;
							p[0] = xMin + ( ( xMax - xMin ) * i ) / _RESOLUTION;
							p[1] = P( p[0] );
							p = SiteToScreen( p );
							glVertex2d( p[0] , p[1] );
						}
						glEnd();
					}
					if( selectedSite!=-1 && iter->endPoint.rightFace->data.site[0]==_sites[selectedSite][0] && iter->endPoint.rightFace->data.site[1]==_sites[selectedSite][1] )
					{
						glLineWidth(4);
						glBegin( GL_LINE_STRIP );
						static const int _RESOLUTION = RESOLUTION * 100;
						for( int i=0 ; i<=_RESOLUTION ; i++ )
						{
							Geometry::Point2d p;
							p[0] = xMin + ( ( xMax - xMin ) * i ) / _RESOLUTION;
							p[1] = P( p[0] );
							p = SiteToScreen( p );
							glVertex2d( p[0] , p[1] );
						}
						glEnd();
					}
					else
					{
						glLineWidth(2);
						glBegin( GL_LINE_STRIP );
						for( int i=0 ; i<=RESOLUTION ; i++ )
						{
							Geometry::Point2d p;
							p[0] = start + ( ( end - start ) * i ) / RESOLUTION;
							p[1] = P( p[0] );
							p = SiteToScreen( p );
							glVertex2d( p[0] , p[1] );
						}
						glEnd();
					}
				}
			}
			count++;
		}
	}

	// Draw the Voronoi diagram
	if( showVoronoi )
	{
		glColor3f( 0 , 0 , 1 );
		if( showDeletions )
		{
			for( int i=0 ; i<fortuneState.vertices.size() ; i++ ) DrawDisk( fortuneState.vertices[i]->data.vertex , dotRadius , 64 );
			glLineWidth(1);
			for( auto iter=fortuneState.eventList.begin() ; iter!=fortuneState.eventList.end() ; iter++ ) if( iter->type==Fortune::Event::DELETION )
			{
				Geometry::Point2d center = iter->deletion.center();
				if( !fortuneState.beachLine.isFinalized( center ) ) DrawCircle( center , dotRadius , 64 );
			}
		}
		if( showCircumcircles )
		{
			for( int i=0 ; i<fortuneState.vertices.size() ; i++ ) DrawCircle( fortuneState.vertices[i]->data.vertex , fortuneState.vertices[i]->data.radius , 128 );
			for( auto iter=fortuneState.eventList.begin() ; iter!=fortuneState.eventList.end() ; iter++ ) if( iter->type==Fortune::Event::DELETION )
			{
				Geometry::Point2d center = iter->deletion.center();
				if( !fortuneState.beachLine.isFinalized( center ) ) DrawCircle( center , iter->deletion.radius() , 128 );
			}
		}
		glLineWidth(2);
		if( selectedVertex.radius>0 ) DrawCircle( selectedVertex.vertex , selectedVertex.radius , 128 );


		glLineWidth(2);
		for( int i=0 ; i<fortuneState.halfEdges.size() ; i++ )
		{
			glBegin( GL_LINES );
			if( fortuneState.halfEdges[i]->startVertex && fortuneState.halfEdges[i]->opposite->startVertex )
			{
				{
					Geometry::Point2d p = fortuneState.halfEdges[i]->startVertex->data.vertex;
					p = SiteToScreen( p );
					glVertex2d( p[0] , p[1] );
				}
				{
					Geometry::Point2d p = fortuneState.halfEdges[i]->opposite->startVertex->data.vertex;
					p = SiteToScreen( p );
					glVertex2d( p[0] , p[1] );
				}
			}
			else if( fortuneState.halfEdges[i]->startVertex )
			{
				Fortune::BeachLineElement::EndPoint e1( fortuneState.halfEdges[i]->face , fortuneState.halfEdges[i]->opposite->face , NULL , NULL );
				Fortune::BeachLineElement::EndPoint e2( fortuneState.halfEdges[i]->opposite->face , fortuneState.halfEdges[i]->face , NULL , NULL );
				auto i1 = fortuneState.beachLine.find( e1 );
				auto i2 = fortuneState.beachLine.find( e2 );
				if( i1==fortuneState.beachLine.end() && i2==fortuneState.beachLine.end() ) WARN( "Neither end-point on beach line" );
				else if( i1==fortuneState.beachLine.end() )
				{
					{
						Geometry::Point2d p = fortuneState.halfEdges[i]->startVertex->data.vertex;
						p = SiteToScreen( p );
						glVertex2d( p[0] , p[1] );
					}
					{
						Geometry::Point2d p = i2->endPoint.position();
						p = SiteToScreen( p );
						glVertex2d( p[0] , p[1] );
					}
				}
				else if( i2==fortuneState.beachLine.end() )
				{
					{
						Geometry::Point2d p = fortuneState.halfEdges[i]->startVertex->data.vertex;
						p = SiteToScreen( p );
						glVertex2d( p[0] , p[1] );
					}
					{
						Geometry::Point2d p = i1->endPoint.position();
						p = SiteToScreen( p );
						glVertex2d( p[0] , p[1] );
					}
				}
				else WARN( "both end-points on beach line" );
			}
			else if( !fortuneState.halfEdges[i]->opposite->startVertex )
			{
				Fortune::BeachLineElement::EndPoint e1( fortuneState.halfEdges[i]->face , fortuneState.halfEdges[i]->opposite->face , NULL , NULL );
				Fortune::BeachLineElement::EndPoint e2( fortuneState.halfEdges[i]->opposite->face , fortuneState.halfEdges[i]->face , NULL , NULL );
				auto i1 = fortuneState.beachLine.find( e1 );
				auto i2 = fortuneState.beachLine.find( e2 );
				if( i1==fortuneState.beachLine.end() || i2==fortuneState.beachLine.end() ) WARN( "End-points not on beach line" );
				{
					Geometry::Point2d p = i1->endPoint.position();
					p = SiteToScreen( p );
					glVertex2d( p[0] , p[1] );
				}
				{
					Geometry::Point2d p = i2->endPoint.position();
					p = SiteToScreen( p );
					glVertex2d( p[0] , p[1] );
				}
			}
			glEnd();
		}
	}

	// Draw the _sites
	{
		glColor3f( 0 , 0 , 0 );
		for( int i=0 ; i<_sites.size() ; i++ )
		{
			Geometry::Point2d p;
			p[0] = _sites[i][0] , p[1] = _sites[i][1];
			DrawDisk( p , dotRadius , 16 );
		}
	}


	sprintf( _stepSizeLine , "Step-size: %.3f" , stepSize );
	sprintf( _sweepHeightLine , "Sweep height: %.3f" , Fortune::BeachLineElement::Height );
}

Geometry::Point2d VoronoiDiagram2DVisualization::select( int x , int  y )
{
	GLdouble modelViewMatrix[16];
	GLdouble projectionMatrix[16];
	GLint viewPort[4];

	glGetDoublev( GL_MODELVIEW_MATRIX , modelViewMatrix );
	glGetDoublev( GL_PROJECTION_MATRIX , projectionMatrix );
	glGetIntegerv( GL_VIEWPORT , viewPort );

	Geometry::Point2d p;
	{
		double _x , _y , _z;
		gluUnProject( x , screenHeight-y , 1. , modelViewMatrix , projectionMatrix , viewPort , &_x , &_y , &_z );
		p[0] = _x , p[1] = _y;
		p = ScreenToSite( p );
	}
	return p;
}

void VoronoiDiagram2DVisualization::mouseFunc( int button , int state , int x , int y )
{
}

void VoronoiDiagram2DVisualization::motionFunc( int x , int y )
{
}
void VoronoiDiagram2DVisualization::passiveMotionFunc( int x , int y )
{
	Geometry::Point2d p = select( x , y );
	sprintf( _positionLine , "Position: ( %f , %f )" , p[0] , p[1] );
	auto SquareSiteDistance = [&]( unsigned int i ){ return ( p[0] - _sites[i][0] ) * ( p[0] - _sites[i][0] ) + ( p[1] - _sites[i][1] ) * ( p[1] - _sites[i][1] ); };
	auto SquareVertexDistance = [&]( Geometry::Point2d q ){ return ( p[0] - q[0] ) * ( p[0] - q[0] ) + ( p[1] - q[1] ) * ( p[1] - q[1] ); };

	selectedSite = -1;
	selectedVertex.radius = -1;

	for( unsigned int i=0 ; i<_sites.size() ; i++ ) if( selectedSite==-1 || SquareSiteDistance( i )<SquareSiteDistance( selectedSite ) ) selectedSite = i;
	if( SquareSiteDistance( selectedSite )>4*dotRadius*dotRadius ) selectedSite = -1;

	for( unsigned int i=0 ; i<fortuneState.vertices.size() ; i++ ) if( selectedVertex.radius<0 || SquareVertexDistance( fortuneState.vertices[i]->data.vertex )<SquareVertexDistance( selectedVertex.vertex ) ) selectedVertex = fortuneState.vertices[i]->data;
	for( auto iter=fortuneState.eventList.begin() ; iter!=fortuneState.eventList.end() ; iter++ ) if( iter->type==Fortune::Event::DELETION )
	{
		Geometry::Point2d center = iter->deletion.center();
		if( !fortuneState.beachLine.isFinalized( center ) )
			if( selectedVertex.radius<0 || SquareVertexDistance( center )<SquareVertexDistance( selectedVertex.vertex ) ) selectedVertex.vertex = center , selectedVertex.radius = iter->deletion.radius();
	}
	if( selectedVertex.radius>0 && SquareVertexDistance( selectedVertex.vertex )>4*dotRadius*dotRadius ) selectedVertex.radius = -1;

	if( selectedSite!=-1 ) sprintf( _selectedLine , "Selected: ( %d , %d )" , _sites[selectedSite][0] , _sites[selectedSite][1] );
	else if( selectedVertex.radius>0 ) sprintf( _selectedLine , "Selected: ( %f , %f ) %f" , selectedVertex.vertex[0] , selectedVertex.vertex[1] , selectedVertex.radius );
	else sprintf( _selectedLine , "Selected: ");

	glutPostRedisplay();
}

void VoronoiDiagram2DVisualization::idle( void )
{
	if( !promptCallBack )
	{
		if( Fortune::BeachLineElement::Height<maximumAnimationHeight )
		{
			advance( Fortune::BeachLineElement::Height + stepSize );
			glutPostRedisplay();
		}
	}
}

void VoronoiDiagram2DVisualization::keyboardFunc( unsigned char key , int x , int y )
{
}

void VoronoiDiagram2DVisualization::specialFunc( int key, int x, int y )
{
	float stepSize = 10.f / ( screenWidth + screenHeight );
	if( glutGetModifiers()&GLUT_ACTIVE_CTRL ) stepSize /= 16;
	float panSize = stepSize*2 , scaleSize = stepSize*2;

	switch( key )
	{
		case Misha::KEY_PGUP:       _radius /= 1.1               ; break;
		case Misha::KEY_PGDN:       _radius *= 1.1               ; break;
		case Misha::KEY_LEFTARROW:  _center[0] += _radius * .025 ; break;
		case Misha::KEY_RIGHTARROW: _center[0] -= _radius * .025 ; break;
		case Misha::KEY_UPARROW:    _center[1] -= _radius * .025 ; break;
		case Misha::KEY_DOWNARROW:  _center[1] += _radius * .025 ; break;
	}
	glutPostRedisplay();
}