#include <stdio.h>
#include <stdlib.h>
#include <omp.h>
#include "Util/CmdLineParser.h"
#include "Util/Geometry.h"
#include "Util/Ply.h"
#include "Util/Timer.h"
#include "Util/Exceptions.h"
#include "VoronoiDiagram2DViewer.inl"

CmdLineParameter< std::string > In( "in" ) , Out( "out" );
CmdLineParameter< float > StepSize( "stepSize" , 0.1f );
CmdLineReadable ASCII( "ascii" ) , Verbose( "verbose" );
CmdLineReadable* params[] = { &In , &Out , &StepSize , &ASCII , &Verbose , NULL };

void ShowUsage( const char* ex )
{
	printf( "Usage %s:\n" , ex );
	printf( "\t --%s <input sites>\n" , In.name.c_str() );
	printf( "\t[--%s <output delaunay triangulation>\n" , Out.name.c_str() );
	printf( "\t[--%s <step size>=%f]\n" , StepSize.name.c_str() , StepSize.value );
	printf( "\t[--%s]\n" , ASCII.name.c_str() );
	printf( "\t[--%s]\n" , Verbose.name.c_str() );
}

int main( int argc , char* argv[] )
{
	CmdLineParse( argc-1 , argv+1 , params );
	if( !In.set ){ ShowUsage( argv[0] ) ; return EXIT_FAILURE; }

	std::vector< Geometry::Point2i > sites;
	{
		typedef PLY::VertexData< int , PLY::VertexPosition< 2 , int > > PlyVertex;
		std::vector< PlyVertex > _sites;

		int fileType;
		PLY::Read< PlyVertex >( In.value , _sites , NULL , NULL , PlyVertex::ReadProperties() , NULL , PlyVertex::ReadNum , fileType );
		sites.resize( _sites.size() );
		for( int i=0 ; i<sites.size() ; i++ ) sites[i] = _sites[i].data<0>();
	}

	if( sites.size()<3 ) std::cout << "[ERROR] Need at least three points" << std::endl , exit( 0 );
	

	if( Out.set )
	{
		Timer timer;
		std::vector< Geometry::Triangle > triangles = Fortune::State::Delaunay( sites , Verbose.set );
		std::cout << "Got Delaunay Triangulation: " << timer << std::endl;
		std::cout << "Vertices / Triangles: " << sites.size() << " / " << triangles.size() << std::endl;

		typedef PLY::VertexData< float , PLY::VertexPosition< 3 , float > > PlyVertex;

		std::vector< PlyVertex > _sites( sites.size() );
		for( int i=0 ; i<sites.size() ; i++ )
		{
			_sites[i].data<0>()[0] = (float) sites[i][0];
			_sites[i].data<0>()[1] = (float) sites[i][1];
			_sites[i].data<0>()[2] = 0.f;
		}
		PLY::WriteTriangles( Out.value , _sites , triangles , ASCII.set ? PLY_ASCII : PLY_BINARY_NATIVE );
	}
	else
	{
		std::cout << "+--------------------------------------------------------+" << std::endl;
		std::cout << "| Interface Controls:                                    |" << std::endl;
		std::cout << "|    [Arrow keys]:                translate              |" << std::endl;
		std::cout << "|    [Page up]:                   zoom in                |" << std::endl;
		std::cout << "|    [Page down]:                 zoom out               |" << std::endl;
		std::cout << "|    [Space]:                     advance to next event  |" << std::endl;
		std::cout << "|    '+':                         advance the sweep line |" << std::endl;
		std::cout << "|    'h':                         toggle help            |" << std::endl;
		std::cout << "+--------------------------------------------------------+" << std::endl;

		VoronoiDiagram2DVisualization vd( sites );
		vd.showHelp = false;
		vd.verbose = Verbose.set;
		vd.stepSize = StepSize.value;
		vd.showFPS = false;

		char windowName[1024];
		sprintf( windowName , "Fortune's algorithm viewer: %d" , (int)sites.size() );
		VoronoiDiagram2DVisualization::Viewer::Run( &vd , argc , argv , windowName );
	}
	return EXIT_SUCCESS;

}