namespace Fortune
{
	// Given a point q and a sweep line height h, the parabola P(x) of points equidistant to q and the line y=h satisfies:
	//		|| ( x , P(x) ) - q ||^2 = || P(x) - h ||^2
	// <=>	( x-q[0] )^2 + ( P(x)-q[1] )^2 = ( P(x)-h )^2
	// <=>	( x-q[0] )^2 + P^2(x) - 2 P(x) * q[1] + q[1]^2 = P^2(x) - 2 P(x) * h + h^2
	// <=>	( x-q[0] )^2 - 2 P(x) * q[1] + q[1]^2 = - 2 P(x) * h + h^2
	// <=>	2 P(x) * ( h - q[1] ) = h^2 - q[1]^2 - ( x - q[0] )^2
	// <=>  P(x) = ( h^2 - q[1]^2 - ( x - q[0] )^2 ) / ( 2 * ( h-q[1] ) )
	// <=>  P(x) = ( h^2 - q[1]^2 - x^2 + 2 x * q[0] - q[0]^2 ) / ( 2 * ( h-q[1] ) )

	Polynomial1D< 2 > ParabolicFront( Geometry::Point2i site , double height )
	{
		Polynomial1D< 2 > p;
		p[0] = height * height - site[1]*site[1] - site[0] * site[0];
		p[1] = 2 * site[0];
		p[2] = -1;
		return p / ( 2. *  ( height - site[1] ) );
	}

	//////////////////////
	// Event::Insertion //
	//////////////////////
	Event::Insertion::Insertion( F *f ) : face(f) {}
	Geometry::Point2d Event::Insertion::eventPosition( void ) const
	{
		if( !face ) ERROR_OUT( "NULL face" );
		Geometry::Point2d p;

		////////////////////////////////
		// You need to implement this //
		////////////////////////////////
		WARN_ONCE( "Method not supported" );

		return p;
	}
	bool Event::Insertion::operator == ( const Event::Insertion &i ) const { return face==i.face; }
	bool Event::Insertion::operator != ( const Event::Insertion &i ) const { return !( (*this)==i ); }

	/////////////////////
	// Event::Deletion //
	/////////////////////
	Event::Deletion::Deletion( F *f1 , F *f2 , F *f3 ) : face1(f1) , face2(f2) , face3(f3)
	{
		if( !face1 || !face2 || !face3 ) ERROR_OUT( "Null face" );

		////////////////////////////////
		// You need to implement this //
		////////////////////////////////
		// Compute the center and radius of the circle circumscribing the sites pointed to by the faces
		WARN_ONCE( "Method not supported" );
	}

	Geometry::Point2d Event::Deletion::center( void ) const{ return _center; }
	double Event::Deletion::radius( void ) const { return _radius; }
	Geometry::Point2d Event::Deletion::eventPosition( void ) const
	{
		Geometry::Point2d p = _center;
		p[1] += _radius;
		return p;
	}
	bool Event::Deletion::operator == ( const Event::Deletion &d ) const { return face1==d.face1 && face2==d.face2 && face3==d.face3; }
	bool Event::Deletion::operator != ( const Event::Deletion &d ) const { return !( (*this)==d ); }

	///////////
	// Event //
	///////////
	Event::Event( F *face ) : type( INSERTION ) , insertion( face ) {}
	Event::Event( F *face1 , F *face2 , F *face3 ) : type( DELETION ) , deletion( face1 , face2 , face3 ){ }
	Event::Event( Insertion i ) : type( INSERTION ) , insertion(i){ }
	Event::Event( Deletion d ) : type( DELETION ) , deletion(d){ }
	bool Event::Compare( const Event &e1 , const Event &e2 )
	{
		Geometry::Point2d p1 = e1.eventPosition() , p2 = e2.eventPosition();
		if( p1[1]!=p2[1] ) return p1[1]<p2[1];
		else               return p1[0]<p2[0];
	}

	Geometry::Point2d Event::eventPosition( void ) const
	{
		if     ( type==INSERTION ) return insertion.eventPosition();
		else if( type==DELETION  ) return deletion.eventPosition();
		else ERROR_OUT( "unknown event type" );
		return Geometry::Point2d();
	}

	bool Event::operator == ( const Event &e ) const
	{
		if( type!=e.type ) return false;
		else if( type==INSERTION ) return insertion==e.insertion;
		else                       return deletion==e.deletion;
	}
	bool Event::operator !=( const Event &e ) const { return !( (*this)==e); }
	std::ostream &operator << ( std::ostream &os , Event::Insertion e ){ return os << e.face->data.site; }
	std::ostream &operator << ( std::ostream &os , Event::Deletion d ){ return os << d.face1->data.site << " " << d.face2->data.site << " " << d.face3->data.site << " @ " << d.center() << " , " << d.radius(); }
	std::ostream &operator << ( std::ostream &os , Event e )
	{
		if     ( e.type==Event::INSERTION ) return os << "Insertion @ " << e.insertion.eventPosition() << ": " << e.insertion;
		else if( e.type==Event::DELETION  ) return os << "Deletion  @ " << e.deletion.eventPosition() << ": " << e.deletion;
		else return os << "Unknown";
		return os;
	};

	////////////////////////////////
	// BeachLineElement::EndPoint //
	////////////////////////////////
	// An object of type BeachLineElement represents one of the two points of intersections of two parabolic arcs
	BeachLineElement::EndPoint::EndPoint( void ) : leftFace(NULL) , rightFace(NULL) , leftHalfEdge(NULL) , rightHalfEdge(NULL) {}
	BeachLineElement::EndPoint::EndPoint( F *lf , F *rf , HE *lhe , HE *rhe ) : leftFace(lf) , rightFace(rf) , leftHalfEdge(lhe) , rightHalfEdge(rhe) {}

	double BeachLineElement::EndPoint::operator()( void ) const
	{
		if     ( !leftFace && !rightFace ) ERROR_OUT( "No sites" );
		else if( !leftFace               ) return -std::numeric_limits<double>::infinity();
		else if(              !rightFace ) return  std::numeric_limits<double>::infinity();
		else
		{
			////////////////////////////////
			// You need to implement this //
			////////////////////////////////
			WARN_ONCE( "Method undefined" );
		}
	}
	Geometry::Point2d BeachLineElement::EndPoint::position( void ) const
	{
		Geometry::Point2d p;
		if     ( !leftFace && !rightFace ) ERROR_OUT( "Both sides cannot be infinite" );
		else if( !leftFace )
		{
			p[0] = -std::numeric_limits<double>::infinity();
			p[1] =  std::numeric_limits<double>::infinity();
		}
		else if(              !rightFace )
		{
			p[0] =  std::numeric_limits<double>::infinity();
			p[1] =  std::numeric_limits<double>::infinity();
		}
		else
		{
			////////////////////////////////
			// You need to implement this //
			////////////////////////////////
			// Compute the position of the intersection of the two arcs
			WARN_ONCE( "Method not supported" );
		}
		return p;
	}

	bool BeachLineElement::EndPoint::operator == ( const BeachLineElement::EndPoint &ep ) const { return leftFace==ep.leftFace && rightFace==ep.rightFace; }
	bool BeachLineElement::EndPoint::operator != ( const BeachLineElement::EndPoint &ep ) const { return !( (*this)==ep ); }

	/////////////////////////////////
	// BeachLineElement::_Position //
	/////////////////////////////////
	BeachLineElement::_XPosition::_XPosition( double _x ) : x(_x) {}
	double BeachLineElement::_XPosition::operator()( void ) const { return x; }

	//////////////////////
	// BeachLineElement //
	//////////////////////
	BeachLineElement::BeachLineElement( EndPoint e ) : endPoint(e) , _type(_END_POINT) {}
	BeachLineElement::BeachLineElement( _XPosition x ) : _xPosition(x) , _type(_X_POSITION) {}

	double BeachLineElement::operator()( void ) const
	{
		if( _type==_END_POINT ) return endPoint();
		else                    return _xPosition();
	}

	bool BeachLineElement::Compare( const BeachLineElement &e1 , const BeachLineElement &e2 ){ return e1() < e2(); };

	double BeachLineElement::Height;
	std::ostream &operator << ( std::ostream &os , const BeachLineElement::EndPoint &e )
	{
		os << "[ ";
		if( e.leftFace ) os << e.leftFace->data.site;
		else             os << "NULL";
		os << " , ";
		if( e.rightFace ) os << e.rightFace->data.site;
		else              os << "NULL";
		os << " ]{ " << e() << " }";
		return os;
	}
	std::ostream &operator << ( std::ostream &os , const BeachLineElement::_XPosition &x ){ return os << x.x; }
	std::ostream &operator << ( std::ostream &os , const BeachLineElement &e )
	{
		if( e._type==BeachLineElement::_X_POSITION ) return os << e._xPosition;
		else                                         return os << e.endPoint;
	}

	///////////////
	// EventList //
	///////////////
	EventList::EventList( void ) : std::set< Event , EventComparator >( Event::Compare ){}

	void EventList::insertSite( F *face ){ insert( Event( face ) ); }

	bool EventList::insertVertex( const BeachLine &beachLine , F *f1 , F *f2 , F *f3 )
	{
		if( !f1 || !f2 || !f3 ) return false;
		Event::Deletion deletion( f1 , f2 , f3 );
		Geometry::Point2d center = deletion.center();
		// Don't bother inserting the deletion event if it's position has already been finalized
		if( !beachLine.isFinalized( center ) ) return insert( deletion ).second;
		else return false;
	}

	Event EventList::pop( void )
	{
		auto iter = begin();
		Event e = *iter;
		erase( iter );
		return e;
	}

	std::ostream &operator << ( std::ostream &os , const EventList &eventList )
	{
		os << "Event list:" << std::endl;
		for( auto iter=eventList.begin() ; iter!=eventList.end() ; iter++ ) os << "\t" << *iter << std::endl;
		return os;
	}

	////////////////////
	// BeachLine::Arc //
	////////////////////
	BeachLine::Arc::Arc( BeachLineElement::EndPoint l , BeachLineElement::EndPoint r ) : leftEndPoint(l) , rightEndPoint(r){}
	BeachLine::Arc::Arc( void ){}
	bool BeachLine::Arc::operator == ( const BeachLine::Arc &arc ) const { return leftEndPoint==arc.leftEndPoint && rightEndPoint==arc.rightEndPoint; }
	bool BeachLine::Arc::operator != ( const BeachLine::Arc &arc ) const { return !( (*this)==arc ); }


	///////////////
	// BeachLine //
	///////////////
	BeachLine::BeachLine( void ) : std::set< BeachLineElement , BeachLineElementComparator >( BeachLineElement::Compare ){}

	void BeachLine::sanityCheck( void ) const
	{
		for( auto iter=begin() ; iter!=end() ; iter++ )
		{
			if( iter->_type!=BeachLineElement::_END_POINT ) ERROR_OUT( "Beach line should only contain end-points of arcs @ " , BeachLineElement::Height );
			if( !iter->endPoint.leftFace && !iter->endPoint.rightFace ) ERROR_OUT( "Both faces cannot be NULL" );
			auto next = std::next( iter );
			if( next!=end() )
			{
				if( iter->endPoint.rightFace!=next->endPoint.leftFace ) ERROR_OUT( "Inconsistent arc @ " , BeachLineElement::Height );
				if( (*iter)()>(*next)() ) ERROR_OUT( "Out of order @ " , BeachLineElement::Height );
			}
		}
	}

	BeachLine::Arc BeachLine::findArc( double x ) const
	{
		Arc arc;
		auto next = upper_bound( BeachLineElement::_XPosition( x ) );
		auto prev = std::prev( next );
		if( next==end() || next==begin() ) ERROR_OUT( "Expected interior arc" );
		arc.leftEndPoint = prev->endPoint;
		arc.rightEndPoint = next->endPoint;
		return arc;
	}

	bool BeachLine::isFinalized( Geometry::Point2d p ) const
	{
		////////////////////////////////
		// You need to implement this //
		////////////////////////////////
		// Determine if p[1] is below the beach-line at p[0]
		WARN_ONCE( "Method not supported" );
		return false;
	}

	void BeachLine::insert( BeachLine::Arc arc )
	{
		// Insert the left edge
		{
			auto iterator_success_pair = insert( arc.leftEndPoint );
			if( !iterator_success_pair.second ) ERROR_OUT( "Failed to insert left edge into beach line" );
		}
		// Insert the right edge
		{
			auto iterator_success_pair = insert( arc.rightEndPoint );
			if( !iterator_success_pair.second ) ERROR_OUT( "Failed to insert right edge into beach line" );
		}
	}
	std::ostream &operator << ( std::ostream &os , const BeachLine::Arc &arc )
	{
		return os << "[ " << arc.leftEndPoint << " , "  << arc.rightEndPoint << " ]";
	}
	std::ostream &operator << ( std::ostream &os , const BeachLine &beachLine )
	{
		os << "Beach line ( " << BeachLineElement::Height << " )" << std::endl;
		for( auto iter=beachLine.begin() ; iter!=beachLine.end() ; iter++ ) os << "\t" << *iter << std::endl;
		return os;
	}

	///////////
	// State //
	///////////
	State::State( const std::vector< Geometry::Point2i > &sites )
	{
		vertices.clear();
		halfEdges.clear();
		faces.clear();
		eventList.clear();
		beachLine.clear();

		int minY = sites[0][1];
		faces.resize( sites.size() );
		for( int i=0 ; i<sites.size() ; i++ )
		{
			faces[i] = new F();
			faces[i]->data.site = sites[i];
			faces[i]->data.index = i;
			eventList.insertSite( faces[i] );
			minY = std::min< int >( minY , sites[i][1] );
		}
		BeachLineElement::Height = minY - 0.4999999999999;
	}

	void State::processNextEvent( bool verbose )
	{
		if( eventList.empty() ){ WARN_ONCE( "Event list is empty" ) ; return; }
		Event e = eventList.pop();

		////////////////////////////////
		// You need to implement this //
		////////////////////////////////
		// Process the very first insertion event
		if( !beachLine.size() )
		{
			if( e.type!=Event::INSERTION ) ERROR_OUT( "Expected insertion" );

			////////////////////////////////
			// You need to implement this //
			////////////////////////////////
			WARN_ONCE( "Method not supported" );
		}
		// Process subsequent insertion/deletion event
		else
		{
			if( e.type==Event::INSERTION )
			{
				////////////////////////////////
				// You need to implement this //
				////////////////////////////////
				WARN_ONCE( "Method not supported" );
			}
			else
			{
				////////////////////////////////
				// You need to implement this //
				////////////////////////////////
				WARN_ONCE( "Method not supported" );
			}
		}
		if( verbose ) beachLine.sanityCheck();
	}

	std::vector< Geometry::Triangle > State::Delaunay( const std::vector< Geometry::Point2i > &sites , bool verbose )
	{
		State state( sites );
		if( verbose ) std::cout << state.eventList << std::endl;
		while( !state.eventList.empty() ) state.processNextEvent( verbose );
		std::vector< Geometry::Triangle > triangles( state.vertices.size() );
		for( int i=0 ; i<state.vertices.size() ; i++ )
		{
			int count = 0;
			auto ProcessFunctor = [&]( const HE * he )
			{
				if( count>=3 ) ERROR_OUT( "Expected a triangle: " , count );
				triangles[i][count++] = he->face->data.index;
			};
			state.vertices[i]->processHalfEdges( ProcessFunctor );
		}
		return triangles;
	}
}
