#ifndef FORTUNE_INCLUDED
#define FORTUNE_INCLUDED

#include <algorithm>
#include <functional>
#include <set>
#include <limits>
#include "Util/Geometry.h"
#include "Util/HalfEdge.h"
#include "Util/Exceptions.h"
#include "Util/Polynomial.h"

namespace Fortune
{
	struct Event;
	struct BeachLineElement;
	struct EventList;
	struct BeachLine;
	struct State;

	/** Returns the quadratic polynomial corresponding to the parabolic front associated to a site and a given height of the sweep-line */
	Polynomial1D< 2 > ParabolicFront( Geometry::Point2i site , double height );

	/** Data that we want the half-edge data structure to store with each vertex */
	struct VertexData  { Geometry::Point2d vertex ; double radius; };

	/** Data that we want the half-edge data structure to store with each half-edge */
	struct HalfEdgeData{ };

	/** Data that we want the half-edge data structure to store with each face */
	struct FaceData    { int index ; Geometry::Point2i site; };

	/** Simplified names for the half-edge vertices */
	typedef typename HalfEdge< VertexData , HalfEdgeData , FaceData >::V  V;

	/** Simplified names for the half-edge half-edges */
	typedef typename HalfEdge< VertexData , HalfEdgeData , FaceData >::HE HE;

	/** Simplified names for the half-edge faces */
	typedef typename HalfEdge< VertexData , HalfEdgeData , FaceData >::F  F;

	/** This class represents an event within the event list */
	struct Event
	{
		/** An object representing an insertion event */
		struct Insertion
		{
			/** A pointer to the Voronoi face being inserted */
			F *face;

			/** A constructor for the insertion event */
			Insertion( F *f );

			/** The position at which the insertion event happens */
			Geometry::Point2d eventPosition( void ) const;

			/** An operator determining if two insertion events are the same */
			bool operator == ( const Insertion &i ) const;

			/** An operator determining if two insertion events are different */
			bool operator != ( const Insertion &i ) const;
		};

		/** An object representing a deletion event */
		struct Deletion
		{
			/** Pointers to the three Voronoi faces creating the deletion event */
			F *face1 , *face2 , *face3;

			/** A constructor for the deletion event */
			Deletion( F *f1 , F *f2 , F *f3 );

			/** The position at which the deletion event happens */
			Geometry::Point2d eventPosition( void ) const;

			/** The circumcenter of the circule circumscribing the three sites */
			Geometry::Point2d center( void ) const;

			/** The radius of the circule circumscribing the three sites */
			double radius( void ) const;

			/** An operator determining if two deletion events are the same */
			bool operator == ( const Deletion &d ) const;

			/** An operator determining if two deletion events are different */
			bool operator != ( const Deletion &d ) const;
		protected:
			double _radius;
			Geometry::Point2d _center;
			friend std::ostream &operator << ( std::ostream & , Deletion );
		};

		enum Type{ INSERTION , DELETION };

		/** The type of this event */
		Type type;

		/** The actual insertion/deletion data */
		union
		{
			Insertion insertion;
			Deletion deletion;
		};

		/** A constructor generating an insertion event */
		Event( F * face );

		/** A constructor generating a deletion event */
		Event( F *face1 , F *face2 , F *face3 );

		/** A constructor generating an insertion event */
		Event( Insertion i );

		/** A constructor generating a deletion event */
		Event( Deletion d );

		/** An operator comparing two events */
		static bool Compare( const Event &e1 , const Event &e2 );

		/** The position at which the event happens */
		Geometry::Point2d eventPosition( void ) const;

		/** An operator determining if two events are the same */
		bool operator == ( const Event &e ) const;

		/** An operator determining if two events are different */
		bool operator != ( const Event &e ) const;
	};

	/** An operator for streaming out insertion events */
	std::ostream &operator << ( std::ostream &os , Event::Insertion e );

	/** An operator for streaming out deletion events */
	std::ostream &operator << ( std::ostream &os , Event::Deletion e );

	/** An operator for streaming out events */
	std::ostream &operator << ( std::ostream &os , Event e );

	/** This class represents one of the two points of intersections of two parabolic arcs */
	struct BeachLineElement
	{
		static double Height;

		/** This class represents the end-point of an arc, and is what is stored in the beach line */
		struct EndPoint
		{
			/** The left and right faces defining the two intersecting parabolic arcs */
			F *leftFace , *rightFace;

			/** The half-edges of the Voronoi diagram associated with the end-point */ 
			HE *leftHalfEdge , *rightHalfEdge;

			/** The default constructor initializing the end-point */
			EndPoint( void );

			/** The constructor initializing the end-point from the left and right faces */
			EndPoint( F *lf , F *rf , HE *lhe , HE *rhe );

			/** The position of the intersection of the two parabolic fronts (as a function of the static "Height" variable */
			Geometry::Point2d position( void ) const;

			/** The x-coordinate of the position of intersection of the two parabolic fronts (as a function of the static "Height" variable */
			double operator()( void ) const;

			/** An operator determining if two end-points are the same */
			bool operator == ( const EndPoint &ep ) const;

			/** An operator determining if two end-points are different */
			bool operator != ( const EndPoint &ep ) const;
		};

	protected:
		/** This object represents an x-coordinate and is used for searching for an arc within a beach-line */
		struct _XPosition
		{
			double x;
			_XPosition( double _x );
			double operator()( void ) const;
		};
		friend std::ostream &operator<<( std::ostream & , const _XPosition & );
		friend std::ostream &operator<<( std::ostream & , const BeachLineElement & );
		friend struct BeachLine;

		enum _Type{ _END_POINT , _X_POSITION };

		/** The type of this end-point */
		_Type _type;
	public:

		/** The actual EndPoint/_Position data */
		union
		{
			EndPoint endPoint;
			_XPosition _xPosition;
		};

		/** A constructor generating an end-point event */
		BeachLineElement( EndPoint e );

		/** A constructor generating an x-position event */
		BeachLineElement( _XPosition x );

		/** The x-coordinate of the beach line element */
		double operator()( void ) const;

		/** An operator comparing two beach-line elements */
		static bool Compare( const BeachLineElement &e1 , const BeachLineElement &e2 );
	};

	/** An operator for streaming out end-point elements */
	std::ostream &operator << ( std::ostream &os , const BeachLineElement::EndPoint &e );

	/** An operator for streaming out x-position elements */
	std::ostream &operator << ( std::ostream &os , const BeachLineElement::_XPosition &p );

	/** An operator for streaming out beach-line elements */
	std::ostream &operator << ( std::ostream &os , const BeachLineElement &e );


	/** The type of function used for comparing events */
	typedef std::function< bool ( const Event & , const Event & ) > EventComparator;

	/** This class represents the list of events to be processed/considered by the sweep-line algorithm */
	struct EventList : public std::set< Event , EventComparator >
	{
		/** The default constructor, which calls the constructor for the underlying std::set object */
		EventList( void );

		/** This method add an insertion event into the event list */
		void insertSite( F *face );

		/** This method add a deletion event into the event list */
		bool insertVertex( const BeachLine &beachLine , F *f1 , F *f2 , F *f3 );

		/** This method returns and removes the next event in the list */
		Event pop( void );
	};

	/** An operator for streaming out event-list */
	std::ostream &operator << ( std::ostream &os , const EventList &eventList );


	/** The type of function used for comparing beach-line elements */
	typedef std::function< bool ( const BeachLineElement & , const BeachLineElement & ) > BeachLineElementComparator;

	/** This class represents the current beach line as defined by BeachLineElement::Height */
	struct BeachLine : public std::set< BeachLineElement , BeachLineElementComparator >
	{
		/** This structure represents an arc between two consecutive end-points on the beach-line */
		struct Arc
		{
			/** The two end-points of the arc */
			BeachLineElement::EndPoint leftEndPoint , rightEndPoint;

			/** The constructor generating an arc from its two end-points */
			Arc( BeachLineElement::EndPoint l , BeachLineElement::EndPoint r );

			/** An operator determining if two arcs are the same */
			bool operator == ( const Arc &arc ) const;

			/** An operator determining if two arcs are different */
			bool operator != ( const Arc &arc ) const;

		protected:
			Arc( void );
			friend BeachLine;
		};

		using std::set< BeachLineElement , BeachLineElementComparator >::insert;

		/** The default constructor, which calls the constructor for the underlying std::set object */
		BeachLine( void );

		/** This method performs a sanity check validating that the beach-line is self-consistent */
		void sanityCheck( void ) const;

		/** This method returns the arc in the beach line sitting over the position x */
		Arc findArc( double x ) const;

		/** This method determines if a point has already been finalized by the current sweep-line */
		bool isFinalized( Geometry::Point2d p ) const;

		/** This method inserts an arc into the beach-line */
		void insert( Arc arc );
	};

	/** An operator for streaming out a beach-line arc */
	std::ostream &operator << ( std::ostream &os , const BeachLine::Arc &arc );

	/** An operator for streaming out the beach-line */
	std::ostream &operator << ( std::ostream &os , const BeachLine &beachLine );

	/** This object represents the state within the course of implementing Fortune's algorithm */
	struct State
	{
		/** The event-list */
		EventList eventList;

		/** The beach-line */
		BeachLine beachLine;

		/** The vertices of the Voronoi diagram , constructed up to the current height of the sweep line */
		std::vector< Fortune::V * > vertices;

		/** The half-edges of the Voronoi diagram , constructed up to the current height of the sweep line */
		std::vector< Fortune::HE * > halfEdges;

		/** The faces of the Voronoi diagram */
		std::vector< Fortune::F * > faces;

		/** The constructor of the state, given the list of sites */
		State( const std::vector< Geometry::Point2i > &sites );

		/** This method processes the next event in the event-list */
		void processNextEvent( bool verbose=false );

		/** This method computes the Delaunay triangulation of the set of sites by running Fortune's algorithm to completion and then computing the dual of the Voronoi diagram */
		static std::vector< Geometry::Triangle > Delaunay( const std::vector< Geometry::Point2i > &sites , bool verbose=false );
	};
}
#include "Fortune.inl"
#endif // FORTUNE_INCLUDED