/* -*- C++ -*-
Copyright (c) 2019, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

#ifndef PLY_VERTEX_DATA_INCLUDED
#define PLY_VERTEX_DATA_INCLUDED

#include <vector>
#include "Geometry.h"
#include "PlyFile.h"

namespace PLY
{
	template< unsigned int Dim , typename Real > struct VertexPosition;
	template< unsigned int Dim , typename Real > struct VertexNormal;
	template< unsigned int Dim , typename Real > struct VertexTexture;
	template<                    typename Real > struct VertexColor;
	template<                    typename Real > struct VertexValue;

	// Generic class overloading binary operators for the vertex data 
	// The Data type should have:
	// -- binary operators for vectors
	// -- static int ReadNum
	// -- static int WriteNum
	// -- static ValidPlyProperties( const bool* ) method
	// -- static const PlyProperty* ReadProperties( void )
	// -- static const PlyProperty* WriteProperties( void )

	template< typename Real , typename Data >
	struct _VertexData
	{
		typedef Data type;
		Data plyData;

		// Overloaded binary operators 
		_VertexData& operator += ( const _VertexData& p ){ plyData += p.plyData ; return *this; }
		_VertexData& operator -= ( const _VertexData& p ){ plyData -= p.plyData ; return *this; }
		_VertexData& operator *= ( Real s )              { plyData *= s ; return *this; }
		_VertexData& operator /= ( Real s )              { plyData /= s ; return *this; }
		_VertexData  operator +  ( const _VertexData& p ) const { _VertexData _p = *this ; _p += p ; return _p; }
		_VertexData  operator -  ( const _VertexData& p ) const { _VertexData _p = *this ; _p -= p ; return _p; }
		_VertexData  operator *  ( Real s )               const { _VertexData _p = *this ; _p *= s ; return _p; }
		_VertexData  operator /  ( Real s )               const { _VertexData _p = *this ; _p /= s ; return _p; }

		// Returns access to the data
		Data &data( void ){ return plyData; }
		const Data &data( void ) const { return plyData; }

		_VertexData( void ){}
		_VertexData( const Data &data ) : plyData(data){}
	};

	// The vertex position
	template< unsigned int Dim , typename Real >
	struct VertexPosition : public _VertexData< Real , Geometry::Point< Dim , Real > >
	{
		static const int ReadNum = Dim;
		static const int WriteNum = Dim;
		static const PlyProperty* ReadProperties( void ){ return _Properties; }
		static const PlyProperty* WriteProperties( void ){ return _Properties; }
		static bool ValidReadProperties( const bool* flags ){ for( int d=0 ; d<Dim ; d++ ) if( !flags[d] ) return false ; return true ; }

		VertexPosition( void ){}
		VertexPosition( const typename _VertexData< Real , Geometry::Point< Dim , Real > >::type &data ) : _VertexData< Real , Geometry::Point< Dim , Real > >::plyData(data){}
	protected:
		static const PlyProperty _Properties[];
	};

	template<>
	const PlyProperty VertexPosition< 2 , int >::_Properties[] =
	{
		PlyProperty( "x" , PLY_FLOAT , PLY_INT , int( offsetof( VertexPosition , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "y" , PLY_FLOAT , PLY_INT , int( offsetof( VertexPosition , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
	};

	template<>
	const PlyProperty VertexPosition< 2 , float >::_Properties[] =
	{
		PlyProperty( "x" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexPosition , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "y" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexPosition , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
	};

	template<>
	const PlyProperty VertexPosition< 2 , double >::_Properties[] =
	{
		PlyProperty( "x" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexPosition , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "y" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexPosition , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
	};

	template<>
	const PlyProperty VertexPosition< 3 , int >::_Properties[] =
	{
		PlyProperty( "x" , PLY_FLOAT , PLY_INT , int( offsetof( VertexPosition , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "y" , PLY_FLOAT , PLY_INT , int( offsetof( VertexPosition , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "z" , PLY_FLOAT , PLY_INT , int( offsetof( VertexPosition , plyData.coordinates[2] ) ) , 0 , 0 , 0 , 0 ) ,
	};

	template<>
	const PlyProperty VertexPosition< 3 , float >::_Properties[] =
	{
		PlyProperty( "x" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexPosition , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "y" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexPosition , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "z" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexPosition , plyData.coordinates[2] ) ) , 0 , 0 , 0 , 0 ) ,
	};

	template<>
	const PlyProperty VertexPosition< 3 , double >::_Properties[] =
	{
		PlyProperty( "x" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexPosition , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "y" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexPosition , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "z" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexPosition , plyData.coordinates[2] ) ) , 0 , 0 , 0 , 0 ) ,
	};

	// The vertex normal
	template< unsigned int Dim , typename Real >
	struct VertexNormal : public _VertexData< Real , Geometry::Point< Dim , Real > >
	{
		static const int ReadNum = Dim;
		static const int WriteNum = Dim;
		static const PlyProperty* ReadProperties( void ){ return _Properties; }
		static const PlyProperty* WriteProperties( void ){ return _Properties; }
		static bool ValidReadProperties( const bool* flags ){ for( int d=0 ; d<Dim ; d++ ) if( !flags[d] ) return false ; return true ; }

		VertexNormal( void ){}
		VertexNormal( const typename _VertexData< Real , Geometry::Point< Dim , Real > >::type &data ) : _VertexData< Real , Geometry::Point< Dim , Real > >::plyData(data){}
	protected:
		static const PlyProperty _Properties[];
	};

	template<>
	const PlyProperty VertexNormal< 2 , float >::_Properties[] =
	{
		PlyProperty( "nx" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexNormal , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "ny" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexNormal , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
	};

	template<>
	const PlyProperty VertexNormal< 2 , double >::_Properties[] =
	{
		PlyProperty( "nx" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexNormal , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "ny" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexNormal , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
	};

	template<>
	const PlyProperty VertexNormal< 3 , float >::_Properties[] =
	{
		PlyProperty( "nx" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexNormal , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "ny" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexNormal , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "nz" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexNormal , plyData.coordinates[2] ) ) , 0 , 0 , 0 , 0 ) ,
	};

	template<>
	const PlyProperty VertexNormal< 3 , double >::_Properties[] =
	{
		PlyProperty( "nx" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexNormal , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "ny" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexNormal , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "nz" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexNormal , plyData.coordinates[2] ) ) , 0 , 0 , 0 , 0 ) ,
	};

	// The vertex texture
	template< unsigned int Dim , typename Real >
	struct VertexTexture : public _VertexData< Real , Geometry::Point< Dim , Real > >
	{
		static const int ReadNum = Dim;
		static const int WriteNum = Dim;
		static const PlyProperty* ReadProperties( void ){ return _Properties; }
		static const PlyProperty* WriteProperties( void ){ return _Properties; }
		static bool ValidReadProperties( const bool* flags ){ for( int d=0 ; d<Dim ; d++ ) if( !flags[d] ) return false ; return true ; }

		VertexTexture( void ){}
		VertexTexture( const typename _VertexData< Real , Geometry::Point< Dim , Real > >::type &data ) : _VertexData< Real , Geometry::Point< Dim , Real > >::plyData(data){}
	protected:
		static const PlyProperty _Properties[];
	};
	template<>
	const PlyProperty VertexTexture< 1 , float >::_Properties[] =
	{
		PlyProperty( "u" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexTexture , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
	};
	template<>
	const PlyProperty VertexTexture< 1 , double >::_Properties[] =
	{
		PlyProperty( "u" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexTexture , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
	};
	template<>
	const PlyProperty VertexTexture< 2 , float >::_Properties[] =
	{
		PlyProperty( "u" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexTexture , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "v" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexTexture , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
	};
	template<>
	const PlyProperty VertexTexture< 2 , double >::_Properties[] =
	{
		PlyProperty( "u" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexTexture , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "v" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexTexture , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
	};
	template<>
	const PlyProperty VertexTexture< 3 , float >::_Properties[] =
	{
		PlyProperty( "u" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexTexture , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "v" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexTexture , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "w" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexTexture , plyData.coordinates[2] ) ) , 0 , 0 , 0 , 0 ) ,
	};
	template<>
	const PlyProperty VertexTexture< 3 , double >::_Properties[] =
	{
		PlyProperty( "u" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexTexture , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "v" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexTexture , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "w" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexTexture , plyData.coordinates[2] ) ) , 0 , 0 , 0 , 0 ) ,
	};

	// The vertex color
	template< class Real >
	struct VertexColor : public _VertexData< Real , Geometry::Point< 3 , Real > >
	{
		static const int ReadNum = 6;
		static const int WriteNum = 3;
		static const PlyProperty* ReadProperties( void ){ return _Properties; }
		static const PlyProperty* WriteProperties( void ){ return _Properties; }
		static bool ValidReadProperties( const bool* flags ){ for( int d=0 ; d<3 ; d++ ) if( !flags[d] && !flags[d+3] ) return false ; return true ; }

		VertexColor( void ){}
		VertexColor( const typename _VertexData< Real , Geometry::Point< 3 , Real > >::type &data ) : _VertexData< Real , Geometry::Point< 3 , Real > >::plyData(data){}
	protected:
		static const PlyProperty _Properties[];
	};

	template<>
	const PlyProperty VertexColor< float >::_Properties[] =
	{
		PlyProperty( "red"   , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexColor , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "green" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexColor , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "blue"  , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexColor , plyData.coordinates[2] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "r"     , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexColor , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "g"     , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexColor , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "b"     , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexColor , plyData.coordinates[2] ) ) , 0 , 0 , 0 , 0 ) ,
	};
	template<>
	const PlyProperty VertexColor< double >::_Properties[] =
	{
		PlyProperty( "red"   , PLY_DOUBLE , PLY_DOUBLE , int( offsetof( VertexColor , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) , 
		PlyProperty( "green" , PLY_DOUBLE , PLY_DOUBLE , int( offsetof( VertexColor , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "blue"  , PLY_DOUBLE , PLY_DOUBLE , int( offsetof( VertexColor , plyData.coordinates[2] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "r"     , PLY_DOUBLE , PLY_DOUBLE , int( offsetof( VertexColor , plyData.coordinates[0] ) ) , 0 , 0 , 0 , 0 ) , 
		PlyProperty( "g"     , PLY_DOUBLE , PLY_DOUBLE , int( offsetof( VertexColor , plyData.coordinates[1] ) ) , 0 , 0 , 0 , 0 ) ,
		PlyProperty( "b"     , PLY_DOUBLE , PLY_DOUBLE , int( offsetof( VertexColor , plyData.coordinates[2] ) ) , 0 , 0 , 0 , 0 ) ,
	};

	// The vertex value
	template< class Real >
	struct VertexValue : public _VertexData< Real , Real >
	{
		static const int ReadNum = 1;
		static const int WriteNum = 1;
		static const PlyProperty* ReadProperties( void ){ return _Properties; }
		static const PlyProperty* WriteProperties( void ){ return _Properties; }
		static bool ValidReadProperties( const bool* flags ){ if( !flags[0] ) return false ; return true ; }

		VertexValue( void ){}
		VertexValue( const typename _VertexData< Real , Real >::type &data ) : _VertexData< Real , Real >::plyData(data){}
	public:
		static const PlyProperty _Properties[];
	};

	template<>
	const PlyProperty VertexValue< float >::_Properties[] =
	{
		PlyProperty( "value" , PLY_FLOAT , PLY_FLOAT , int( offsetof( VertexValue , plyData ) ) , 0 , 0 , 0 , 0 ) , 
	};

	template<>
	const PlyProperty VertexValue< double >::_Properties[] =
	{
		PlyProperty( "value" , PLY_FLOAT , PLY_DOUBLE , int( offsetof( VertexValue , plyData ) ) , 0 , 0 , 0 , 0 ) , 
	};

	// The vertex data combination
	template< typename Real , typename ... Data >
	struct VertexData : public _VertexData< Real , std::tuple< Data ... > >
	{
		typedef std::tuple< Data ... > MultiData;
		using _VertexData< Real , MultiData >::plyData;
		template< unsigned int I > using DataType = typename std::tuple_element< I , MultiData >::type;

		VertexData& operator += ( const VertexData& p ){ _add<0>( p ) ; return *this; }
		VertexData& operator -= ( const VertexData& p ){ _sub<0>( p ) ; return *this; }
		VertexData& operator *= ( Real s )                  { _mul<0>( s ) ; return *this; }
		VertexData& operator /= ( Real s )                  { _div<0>( s ) ; return *this; }
		VertexData  operator +  ( const VertexData& p ) const { VertexData _p = *this ; _p += p ; return _p; }
		VertexData  operator -  ( const VertexData& p ) const { VertexData _p = *this ; _p -= p ; return _p; }
		VertexData  operator *  ( Real s )                   const { VertexData _p = *this ; _p *= s ; return _p; }
		VertexData  operator /  ( Real s )                   const { VertexData _p = *this ; _p /= s ; return _p; }

		template< unsigned int I > using _DataType = typename std::tuple_element< I , MultiData >::type::type;

		MultiData &data( void ){ return plyData; }
		const MultiData &data( void ) const { return plyData; }
		template< unsigned int I >       _DataType< I > &data( void )       { return std::get< I >( plyData ).plyData; }
		template< unsigned int I > const _DataType< I > &data( void ) const { return std::get< I >( plyData ).plyData; }

		VertexData( void ){}
		VertexData( const MultiData &data ) : _VertexData< Real , std::tuple< Data ... > >::plyData(data){}
		template< typename ... Types > VertexData( const Types & ... types ){ static_assert( sizeof...(Types)==sizeof...(Data) , "Number of arguments does not match number of data types" ); _set< 0 >( types ... ); }

	private:
		template< unsigned int I , typename Type , typename ... Types > void _set( const Type &type , const Types & ... types ){ data< I >() = type ; _set< I+1 >( types ... ); }
		template< unsigned int I , typename Type                      > void _set( const Type &type                           ){ data< I >() = type ;                           }

		template< unsigned int I > static constexpr typename std::enable_if< I!=sizeof...(Data) , int >::type _TotalReadNum( void ){ return DataType< I >::ReadNum + _TotalReadNum< I+1 >(); }
		template< unsigned int I > static constexpr typename std::enable_if< I==sizeof...(Data) , int >::type _TotalReadNum( void ){ return 0; }
		template< unsigned int I > static constexpr typename std::enable_if< I!=sizeof...(Data) , int >::type _TotalWriteNum( void ){ return DataType< I >::WriteNum + _TotalWriteNum< I+1 >(); }
		template< unsigned int I > static constexpr typename std::enable_if< I==sizeof...(Data) , int >::type _TotalWriteNum( void ){ return 0; }
	public:
		static const int ReadNum = _TotalReadNum<0>();
		static const int WriteNum = _TotalWriteNum<0>();
		static PlyProperty* ReadProperties( void ){ _SetReadProperties<0>( _ReadProperties ) ; return _ReadProperties; }
		static PlyProperty* WriteProperties( void ){ _SetWriteProperties<0>( _WriteProperties ) ; return _WriteProperties; }

		static bool ValidReadProperties( const bool* flags ){ return _ValidReadProperties<0>( flags ) ; }
		template< unsigned int I > static bool ValidReadProperties( const bool* flags ){ return DataType< I >::ValidReadProperties( flags + _ReadOffset< I >() ); }
	protected:
		static PlyProperty _ReadProperties[];
		static PlyProperty _WriteProperties[];
	private:
		// Gives the offset to the I-th element
		template< unsigned int I > static typename std::enable_if< I==0 , unsigned int >::type _ReadOffset( void ){ return 0; }
		template< unsigned int I > static typename std::enable_if< I!=0 , unsigned int >::type _ReadOffset( void ){ return DataType< I-1 >::ReadNum + _ReadOffset< I-1 >(); }

		template< unsigned int I > typename std::enable_if< I!=sizeof...(Data) >::type _add( const VertexData& p ){ std::get< I >( plyData ) += std::get< I >( p.plyData ) ; _add< I+1 >( p ); }
		template< unsigned int I > typename std::enable_if< I==sizeof...(Data) >::type _add( const VertexData& p ){ }
		template< unsigned int I > typename std::enable_if< I!=sizeof...(Data) >::type _sub( const VertexData& p ){ std::get< I >( plyData ) -= std::get< I >( p.plyData ) ; _sub< I+1 >( p ); }
		template< unsigned int I > typename std::enable_if< I==sizeof...(Data) >::type _sub( const VertexData& p ){ }
		template< unsigned int I > typename std::enable_if< I!=sizeof...(Data) >::type _mul( Real s ){ std::get< I >( plyData ) *= s ; _mul< I+1 >( s ); }
		template< unsigned int I > typename std::enable_if< I==sizeof...(Data) >::type _mul( Real s ){ }
		template< unsigned int I > typename std::enable_if< I!=sizeof...(Data) >::type _div( Real s ){ std::get< I >( plyData ) /= s ; _div< I+1 >( s ); }
		template< unsigned int I > typename std::enable_if< I==sizeof...(Data) >::type _div( Real s ){ }

		template< unsigned int I > static typename std::enable_if< I!=sizeof...(Data) >::type _SetReadProperties( PlyProperty* ReadProperties )
		{
			for( int d=0 ; d<DataType< I >::ReadNum ; d++ )
			{
				ReadProperties[d] = DataType< I >::ReadProperties()[d];
				VertexData temp;
				const typename std::tuple_element< I , MultiData >::type& temp_data = std::get< I >( temp.plyData );
				ReadProperties[d].offset += (int)( (size_t)&temp_data - (size_t)&temp );
			}
			_SetReadProperties< I+1 >( ReadProperties + DataType< I >::ReadNum );
		}
		template< unsigned int I > static typename std::enable_if< I==sizeof...(Data) >::type _SetReadProperties( PlyProperty* ReadProperties ){ }
		template< unsigned int I > static typename std::enable_if< I!=sizeof...(Data) >::type _SetWriteProperties( PlyProperty* WriteProperties )
		{
			for( int d=0 ; d<DataType< I >::WriteNum ; d++ )
			{
				WriteProperties[d] = DataType< I >::WriteProperties()[d];
				VertexData temp;
				const typename std::tuple_element< I , MultiData >::type& temp_data = std::get< I >( temp.plyData );
				WriteProperties[d].offset += (int)( (size_t)&temp_data - (size_t)&temp );
			}
			_SetWriteProperties< I+1 >( WriteProperties + DataType< I >::WriteNum );
		}
		template< unsigned int I > static typename std::enable_if< I==sizeof...(Data) >::type _SetWriteProperties( PlyProperty* WriteProperties ){ }

		template< unsigned int I > static typename std::enable_if< I!=sizeof...(Data) , bool >::type _ValidReadProperties( const bool* flags ){ return DataType< I >::ValidReadProperties( flags ) && _ValidReadProperties< I+1 >( flags + std::tuple_element< I , MultiData >::type::ReadNum ); }
		template< unsigned int I > static typename std::enable_if< I==sizeof...(Data) , bool >::type _ValidReadProperties( const bool* flags ){ return true; }
	};
	template< typename Real , typename ... Data > PlyProperty VertexData< Real , Data ... >::_ReadProperties[ VertexData< Real , Data ... >::ReadNum==0 ? 1 : VertexData< Real , Data ... >::ReadNum ];
	template< typename Real , typename ... Data > PlyProperty VertexData< Real , Data ... >::_WriteProperties[ VertexData< Real , Data ... >::WriteNum==0 ? 1 : VertexData< Real , Data ... >::WriteNum ];
}
#endif // PLY_VERTEX_DATA_INCLUDED