/* -*- C++ -*-
Copyright (c) 2019, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

namespace PLY
{
	struct Face
	{
		int nr_vertices;
		int *vertices;
	};
	const PlyProperty FaceProps[] =
	{
		PlyProperty( "vertex_indices" , PLY_INT , PLY_INT , offsetof( Face , vertices ) , 1 , PLY_INT, PLY_INT , offsetof( Face , nr_vertices ) ) ,
	};

	struct Edge{ int v1 , v2; };
	const PlyProperty EdgeProps[] =
	{ 
		{ "v1" , PLY_INT , PLY_INT , (int)offsetof( Edge , v1 ) , 0 , 0 , 0 , 0 },
		{ "v2" , PLY_INT , PLY_INT , (int)offsetof( Edge , v2 ) , 0 , 0 , 0 , 0 }
	};

	// Read
	inline void ReadHeader( std::string fileName , const PlyProperty *properties , int propertyNum , bool *readFlags , int &file_type )
	{
		std::vector< std::string > elist;
		float version;

		PlyFile *ply = PlyFile::Read( fileName , elist , file_type , version );
		if( !ply ) THROW( "could not create read ply file: " , fileName );

		for( int i=0 ; i<(int)elist.size() ; i++ ) if( elist[i]=="vertex" ) for( int j=0 ; j<propertyNum ; j++ ) if( readFlags ) readFlags[j] = ply->get_property( elist[i] , &properties[j] )!=0;

		delete ply;
	}

	inline void ReadHeader( std::string fileName , const PlyProperty *properties , int propertyNum , bool *readFlags )
	{
		int file_type;
		ReadHeader( fileName , properties , propertyNum , readFlags , file_type );
	}

	template< class Vertex >
	void Read
	(
		std::string fileName ,
		std::vector< Vertex >& vertices , 
		std::vector< Geometry::Segment >* edges ,
		std::vector< std::vector< unsigned int > >* polygons ,
		const PlyProperty* vertexProperties ,
		bool* vertexPropertiesFlag ,
		int vertexPropertyNum ,
		int& file_type ,
		std::vector< std::string > *comments
	)
	{
		float version;
		std::vector< std::string > elist;

		PlyFile *ply = PlyFile::Read( fileName , elist , file_type , version );
		if( !ply ) THROW( "could not create read ply file: " , fileName );

		if( comments )
		{
			comments->reserve( comments->size() + ply->comments.size() );
			for( int i=0 ; i<ply->comments.size() ; i++ ) comments->push_back( ply->comments[i] );
		}
		for( int i=0 ; i<elist.size() ; i++ )
		{
			std::string &elem_name = elist[i];
			int num_elems;
			std::vector< PlyProperty * > plist = ply->get_element_description( elem_name , num_elems );
			if( !num_elems ) continue;
			else if( !plist.size() )
			{
				delete ply;
				THROW( "could not get element description for: " , elem_name );
			}
			if( elem_name=="vertex" )
			{
				for( int i=0 ; i<vertexPropertyNum ; i++)
				{
					int hasProperty = ply->get_property( elem_name , &vertexProperties[i] );
					if( vertexPropertiesFlag ) vertexPropertiesFlag[i] = (hasProperty!=0);
				}
				vertices.resize( num_elems );
				for( int j=0 ; j<num_elems ; j++ ) ply->get_element( (void *)&vertices[j] );
			}
			else if( elem_name=="face" && polygons )
			{
				ply->get_property( elem_name , &FaceProps[0] );
				polygons->resize( num_elems );
				for( int j=0 ; j<num_elems ; j++ )
				{
					Face ply_face;
					ply->get_element( (void *)&ply_face );
					(*polygons)[j].resize( ply_face.nr_vertices );
					for( int k=0 ; k<ply_face.nr_vertices ; k++ ) (*polygons)[j][k] = ply_face.vertices[k];
					free( ply_face.vertices );
				}  // for, read faces
			}  // if face
			else if( elem_name=="edge" && edges )
			{
				ply->get_property( elem_name , &EdgeProps[0] );
				ply->get_property( elem_name , &EdgeProps[1] );
				edges->resize( num_elems );
				for( int j=0 ; j<num_elems ; j++ )
				{
					Edge ply_edge;
					ply->get_element( (void*)&ply_edge );
					(*edges)[j][0] = ply_edge.v1 , (*edges)[j][1] = ply_edge.v2;
				}
			}
			else ply->get_other_element( elem_name , num_elems );

			for( int j=0 ; j<plist.size() ; j++ ) delete plist[j];
		}  // for each type of element
		delete ply;
	}

	template< class Vertex >
	void ReadVertices
	(
		std::string fileName ,
		std::vector< Vertex > &vertices ,
		const PlyProperty *vertexProperties ,
		bool* vertexPropertiesFlag ,
		int vertexPropertyNum ,
		int& file_type ,
		std::vector< std::string > *comments
	)
	{
		return Read( fileName , vertices , NULL , NULL , vertexProperties , vertexPropertiesFlag , vertexPropertyNum , file_type , comments );
	}

	template< class Vertex >
	void ReadTriangles
	(
		std::string fileName ,
		std::vector< Vertex > &vertices ,
		std::vector< Geometry::Triangle > &triangles ,
		const PlyProperty *vertexProperties ,
		bool* vertexPropertiesFlag ,
		int vertexPropertyNum ,
		int& file_type ,
		std::function< Geometry::Point3d (Vertex) > VertexToPointFunctor ,
		std::vector< std::string > *comments
	)
	{
		std::vector< std::vector< unsigned int > > polygons;
		ReadPolygons( fileName , vertices , polygons , vertexProperties , vertexPropertiesFlag , vertexPropertyNum , file_type , comments );
		std::vector< Geometry::Point3d > poly;
		std::vector< Geometry::Triangle > tris;

		triangles.clear();
		for( unsigned int i=0 ; i<polygons.size() ; i++ )
		{
			if( polygons[i].size()!=3 ) ERROR_OUT( "Not a triangle face: " , polygons[i].size() );
			else
			{
				Geometry::Triangle tri;
				for( int j=0 ; j<3 ; j++ ) tri[j] = polygons[i][j];
				triangles.push_back( tri );
			}
		}
	}

	template< class Vertex >
	void ReadPolygons
	(
		std::string fileName ,
		std::vector< Vertex > &vertices ,
		std::vector< std::vector< unsigned int > > &polygons ,
		const PlyProperty *properties ,
		bool *readFlags ,
		int propertyNum ,
		int &file_type ,
		std::vector< std::string > *comments
	)
	{
		std::vector< std::string > elist;
		float version;

		PlyFile *ply = PlyFile::Read( fileName , elist , file_type , version );
		if( !ply ) THROW( "could not create read ply file: " , fileName );

		if( comments )
		{
			comments->reserve( comments->size() + ply->comments.size() );
			for( int i=0 ; i<(int)ply->comments.size() ; i++ ) comments->push_back( ply->comments[i] );
		}

		for( int i=0 ; i<(int)elist.size() ; i++ )
		{
			std::string &elem_name = elist[i];
			int num_elems;
			std::vector< PlyProperty * > plist = ply->get_element_description( elem_name , num_elems );
			if( !num_elems ) continue;
			else if( !plist.size() )
			{
				delete ply;
				THROW( "could not get element description for: " , elem_name );
			}
			if( elem_name=="vertex" )
			{
				for( int i=0 ; i<propertyNum ; i++)
				{
					int hasProperty = ply->get_property( elem_name , &properties[i] );
					if( readFlags ) readFlags[i] = (hasProperty!=0);
				}
				vertices.resize( num_elems );
				for( int j=0 ; j<num_elems ; j++ ) ply->get_element( (void *)&vertices[j] );
			}
			else if( elem_name=="face" )
			{
				ply->get_property( elem_name , &FaceProps[0] );
				polygons.resize( num_elems );
				for( int j=0 ; j<num_elems ; j++ )
				{
					Face ply_face;
					ply->get_element( (void *)&ply_face );
					polygons[j].resize( ply_face.nr_vertices );
					for( int k=0 ; k<ply_face.nr_vertices ; k++ ) polygons[j][k] = ply_face.vertices[k];
					free( ply_face.vertices );
				}  // for, read faces
			}  // if face
			else ply->get_other_element( elem_name , num_elems );

			for( int j=0 ; j<(int)plist.size() ; j++ ) delete plist[j];
		}  // for each type of element

		delete ply;
	}
	template< class Vertex , class Polygon >
	int ReadPolygons
	(
		std::string fileName ,
		std::vector< Vertex >& vertices ,
		std::vector< Polygon >& polygons ,
		PlyProperty*  vertexProperties , bool*  vertexPropertiesFlag , int  vertexPropertyNum ,
		PlyProperty* polygonProperties , bool* polygonPropertiesFlag , int polygonPropertyNum ,
		int& file_type,
		std::vector< std::string > *comments
	)
	{
		std::vector< std::string > elist = { std::string( "vertex" ) , std::string( "face" ) };
		float version;

		PlyFile *ply = PlyFile::Read( fileName , elist , file_type , version );
		if(!ply) return 0;

		if( comments )
		{
			comments->reserve( comments->size() + ply->comments.size() );
			for( int i=0 ; i<ply->comments.size() ; i++ ) comments->push_back( ply->comments[i] );
		}

		for( int i=0 ; i<elist.size() ; i++ )
		{
			std::string &elem_name = elist[i];
			int num_elems;
			std::vector< PlyProperty * > plist = ply->get_element_description( elem_name , num_elems );
			if( !plist.size() )
			{
				delete ply;
				return 0;
			}		
			if( elem_name=="vertex" )
			{
				for( int i=0 ; i<vertexPropertyNum ; i++)
				{
					int hasProperty = ply->get_property( elem_name , &vertexProperties[i] );
					if( vertexPropertiesFlag ) vertexPropertiesFlag[i] = (hasProperty!=0);
				}
				vertices.resize( num_elems );
				for( size_t j=0 ; j<num_elems ; j++ ) ply->get_element( (void *)&vertices[j] );
			}
			else if( elem_name=="face" )
			{
				for( int i=0 ; i<polygonPropertyNum ; i++ )
				{
					int hasProperty = ply->get_property( elem_name , &polygonProperties[i] );
					if( polygonPropertiesFlag ) polygonPropertiesFlag[i] = (hasProperty!=0);
				}
				polygons.resize( num_elems );
				for( size_t j=0 ; j<num_elems ; j++ ) ply->get_element( (void *)&polygons[j] );
			}
			else ply->get_other_element( elem_name , num_elems );

			for( int j=0 ; j<plist.size() ; j++ ) delete plist[j];
		}

		delete ply;
		return 1;
	}

	template< class Vertex >
	void ReadTetrahedra
	(
		std::string fileName ,
		std::vector< Vertex > &vertices ,
		std::vector< Geometry::Tetrahedron > &tetrahedra ,
		const PlyProperty *vertexProperties ,
		bool* vertexPropertiesFlag ,
		int vertexPropertyNum ,
		int& file_type ,
		std::function< Geometry::Point3d (Vertex) > VertexToPointFunctor ,
		std::vector< std::string > *comments
	)
	{
		std::vector< std::vector< unsigned int > > polygons;
		ReadPolygons( fileName , vertices , polygons , vertexProperties , vertexPropertiesFlag , vertexPropertyNum , file_type , comments );

		for( int i=0 ; i<polygons.size() ; i++ ) if( polygons[i].size()!=4 ) ERROR_OUT( "Expected polygon with four vertices" );
		tetrahedra.resize( polygons.size() );
		for( unsigned int i=0 ; i<polygons.size() ; i++ ) for( int j=0 ; j<4 ; j++ ) tetrahedra[i][j] = polygons[i][j];
	}

	// Write
	template< class Vertex >
	void Write
	(
		std::string fileName ,
		const std::vector< Vertex > &vertices , 
		const std::vector< Geometry::Segment > *edges , 
		const std::vector< std::vector< unsigned int > > *polygons,
		const PlyProperty *vertexProperties ,
		int vertexPropertyNum ,
		int file_type ,
		const std::vector< std::string > *comments
	)
	{
		int nr_vertices =            (int) vertices.size()    ;
		int nr_edges    = edges    ? (int)   edges->size() : 0;
		int nr_faces    = polygons ? (int)polygons->size() : 0;
		float version;
		std::vector< std::string > elist = { std::string( "vertex" ) , std::string( "edge" ) , std::string( "face" ) };

		PlyFile *ply = PlyFile::Write( fileName , elist , file_type , version );
		if( !ply ) THROW( "could not create write ply file: " , fileName );

		//
		// describe vertex, edge, and face properties
		//
		{
			ply->element_count( "vertex", nr_vertices );
			for( int i=0 ; i<vertexPropertyNum ; i++ ) ply->describe_property( "vertex" , &vertexProperties[i] );
		}
		{
			ply->element_count( "edge" , nr_edges );
			ply->describe_property( "edge" , &EdgeProps[0] );
			ply->describe_property( "edge" , &EdgeProps[1] );
		}
		{
			ply->element_count( "face" , nr_faces );
			ply->describe_property( "face" , &FaceProps[0] );
		}

		// Write in the comments
		if( comments ) for( int i=0 ; i<comments->size() ; i++ ) ply->put_comment( (*comments)[i] );

		ply->header_complete();

		// write vertices
		ply->put_element_setup( "vertex" );
		for( int i=0 ; i<nr_vertices ; i++ ) ply->put_element( (void*)&vertices[i] );

		// write edges
		if( nr_edges )
		{
			Edge ply_edge;
			ply->put_element_setup( "edge" );
			for( int i=0 ; i<nr_edges ; i++ )
			{
				ply_edge.v1 = (*edges)[i][0] , ply_edge.v2 = (*edges)[i][1];
				ply->put_element( (void*)&ply_edge );
			}
		}

		// write faces
		if( nr_faces )
		{
			Face ply_face;
			int maxFaceVerts=3;
			ply_face.nr_vertices = 3;
			ply_face.vertices = new int[3];

			ply->put_element_setup( "face" );
			for( int i=0 ; i<nr_faces ; i++ )
			{
				int face_size = (int)(*polygons)[i].size();
				if( face_size>maxFaceVerts )
				{
					delete[] ply_face.vertices;
					maxFaceVerts = face_size;
					ply_face.vertices = new int[face_size];
				}
				ply_face.nr_vertices = face_size;
				for( int j=0 ; j<ply_face.nr_vertices ; j++ ) ply_face.vertices[j] = (*polygons)[i][j];
				ply->put_element( (void*)&ply_face );
			}
			delete[] ply_face.vertices;
		}
		delete ply;
	}

	template< class Vertex >
	void WriteVertices
	(
		std::string fileName ,
		const std::vector< Vertex > &vertices ,
		const PlyProperty *properties ,
		int propertyNum ,
		int file_type ,
		const std::vector< std::string > *comments
	)
	{
		int nr_vertices = int(vertices.size());
		float version;
		std::vector< std::string > elem_names = { std::string( "vertex" ) };
		PlyFile *ply = PlyFile::Write( fileName , elem_names , file_type , version );
		if( !ply ) THROW( "could not create write ply file: " , fileName );

		//
		// describe vertex and face properties
		//
		ply->element_count( "vertex", nr_vertices );
		for( int i=0 ; i<propertyNum ; i++ ) ply->describe_property( "vertex" , &properties[i] );

		// Write in the comments
		if( comments ) for( int i=0 ; i<comments->size() ; i++ ) ply->put_comment( (*comments)[i] );
		ply->header_complete();

		// write vertices
		ply->put_element_setup( elem_names[0] );
		for( int i=0 ; i<(int)vertices.size() ; i++ ) ply->put_element( (void *)&vertices[i] );

		delete ply;
	}

	template< class Vertex >
	void WriteTriangles
	(
		std::string fileName ,
		const std::vector< Vertex > &vertices ,
		const std::vector< Geometry::Triangle > &triangles ,
		const PlyProperty *properties ,
		int propertyNum ,
		int file_type ,
		const std::vector< std::string > *comments
	)
	{
		std::vector< std::vector< unsigned int > > polygons( triangles.size() );
		for( int i=0 ; i<triangles.size() ; i++ )
		{
			polygons[i].resize( 3 );
			for( int j=0 ; j<3 ; j++ ) polygons[i][j] = triangles[i][j];
		}
		WritePolygons( fileName , vertices , polygons , properties , propertyNum , file_type , comments );
	}

	template< class Vertex >
	void WritePolygons
	(
		std::string fileName ,
		const std::vector< Vertex > &vertices ,
		const std::vector< std::vector< unsigned int > > &polygons ,
		const PlyProperty *properties ,
		int propertyNum ,
		int file_type ,
		const std::vector< std::string > *comments
	)
	{
		int nr_vertices = int(vertices.size());
		int nr_faces = int(polygons.size());
		float version;
		std::vector< std::string > elem_names = { std::string( "vertex" ) , std::string( "face" ) };
		PlyFile *ply = PlyFile::Write( fileName , elem_names , file_type , version );
		if( !ply ) THROW( "could not create write ply file: " , fileName );

		//
		// describe vertex and face properties
		//
		ply->element_count( "vertex", nr_vertices );
		for( int i=0 ; i<propertyNum ; i++ ) ply->describe_property( "vertex" , &properties[i] );
		ply->element_count( "face" , nr_faces );
		ply->describe_property( "face" , &FaceProps[0] );

		// Write in the comments
		if( comments ) for( int i=0 ; i<comments->size() ; i++ ) ply->put_comment( (*comments)[i] );
		ply->header_complete();

		// write vertices
		ply->put_element_setup( elem_names[0] );
		for( int i=0 ; i<(int)vertices.size() ; i++ ) ply->put_element( (void *)&vertices[i] );

		// write faces
		Face ply_face;
		int maxFaceVerts = 3;
		ply_face.nr_vertices = maxFaceVerts;
		ply_face.vertices = new int[ maxFaceVerts ];

		ply->put_element_setup( elem_names[1] );
		for( int i=0 ; i<nr_faces ; i++ )
		{
			if( (int)polygons[i].size()>maxFaceVerts )
			{
				delete[] ply_face.vertices;
				maxFaceVerts = (int)polygons[i].size();
				ply_face.vertices=new int[ maxFaceVerts ];
			}
			ply_face.nr_vertices = (int)polygons[i].size();
			for( int j=0 ; j<ply_face.nr_vertices ; j++ ) ply_face.vertices[j] = polygons[i][j];
			ply->put_element( (void *)&ply_face );
		}
		delete[] ply_face.vertices;
		delete ply;
	}

	template< class Vertex >
	void WriteTetrahedra
	(
		std::string fileName ,
		const std::vector< Vertex > &vertices ,
		const std::vector< Geometry::Tetrahedron > &tetrahedra ,
		const PlyProperty *properties ,
		int propertyNum ,
		int file_type ,
		const std::vector< std::string > *comments
	)
	{
		std::vector< std::vector< unsigned int > > polygons( tetrahedra.size() );
		for( int i=0 ; i<tetrahedra.size() ; i++ )
		{
			polygons[i].resize( 4 );
			for( int j=0 ; j<4 ; j++ ) polygons[i][j] = tetrahedra[i][j];
		}
		WritePolygons( fileName , vertices , polygons , properties , propertyNum , file_type , comments );
	}

	inline int DefaultFileType( void ){ return PLY_ASCII; }
}
