/* -*- C++ -*-
Copyright (c) 2019, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/


#ifndef PLY_INCLUDED
#define PLY_INCLUDED

#include <vector>
#include <string>
#include <functional>
#include "PlyFile.h"
#include "PlyVertexData.h"
#include "Geometry.h"
#include "Exceptions.h"

namespace PLY
{
	int DefaultFileType( void );

	// PLY read functionality
	void ReadHeader( std::string fileName , const PlyProperty *properties , int propertyNum , bool *readFlags );

	void ReadHeader( std::string fileName , const PlyProperty *properties , int propertyNum , bool *readFlags , int &file_type );

	template< class Vertex >
	void Read( std::string fileName , std::vector< Vertex > &vertices , std::vector< Geometry::Segment > *edges , std::vector< std::vector< unsigned int > > *polygons , const PlyProperty *vertexProperties , bool *vertexPropertiesFlag , int vertexPropertyNum , int &file_type , std::vector< std::string > *comments=NULL );

	template< class Vertex >
	void ReadVertices( std::string fileName , std::vector< Vertex > &vertices , const PlyProperty *vertexProperties , bool *vertexPropertiesFlag , int vertexPropertyNum , int &file_type , std::vector< std::string > *comments=NULL );

	template< class Vertex >
	void ReadTriangles( std::string fileName , std::vector< Vertex > &vertices , std::vector< Geometry::Triangle > &triangles , const PlyProperty *vertexProperties , bool *vertexPropertiesFlag , int vertexPropertyNum , int &file_type , std::function< Geometry::Point3d (Vertex) > VertexToPointFunctor , std::vector< std::string > *comments=NULL );

	template< class Vertex >
	void ReadPolygons( std::string fileName , std::vector< Vertex > &vertices , std::vector< std::vector< unsigned int > > &polygons , const PlyProperty *properties , bool *readFlags , int propertyNum , int &file_type , std::vector< std::string > *comments=NULL );

	template< class Vertex >
	void ReadTetrahedra( std::string fileName , std::vector< Vertex > &vertices , std::vector< Geometry::Tetrahedron > &tetrahedra , const PlyProperty *vertexProperties , bool *vertexPropertiesFlag , int vertexPropertyNum , int &file_type , std::function< Geometry::Point3d (Vertex) > VertexToPointFunctor , std::vector< std::string > *comments=NULL );

	// PLY write functionality
	template< class Vertex >
	void Write( std::string fileName , const std::vector< Vertex > &vertices , const std::vector< Geometry::Segment > *edges , const std::vector< std::vector< unsigned int > > *polygons , const PlyProperty *vertexProperties , int vertexPropertyNum , int file_type , const std::vector< std::string > *comments=NULL );

	template< class Vertex >
	void WriteVertices( std::string fileName , const std::vector< Vertex > &vertices , const PlyProperty *vertexProperties , int vertexPropertyNum , int file_type , const std::vector< std::string > *comments=NULL );

	template< class Vertex >
	void WriteTriangles( std::string fileName , const std::vector< Vertex > &vertices , const std::vector< Geometry::Triangle > &triangles , const PlyProperty *properties , int propertyNum , int file_type , const std::vector< std::string > *comments=NULL );

	template< class Vertex >
	void WritePolygons( std::string fileName , const std::vector< Vertex > &vertices , const std::vector< std::vector< unsigned int > > &polygons , const PlyProperty *properties , int propertyNum , int file_type , const std::vector< std::string > *comments=NULL );

	template< class Vertex >
	void WriteTetrahedra( std::string fileName , const std::vector< Vertex > &vertices , const std::vector< Geometry::Tetrahedron > &tetrahedra , const PlyProperty *properties , int propertyNum , int file_type , const std::vector< std::string > *comments=NULL );

	// Read/write specializations for PLY::VertexData< Real , VertexPosition< 3 , Real > ... >

	template< typename Real , typename ... AdditionalVertexData >
	void Read( std::string fileName , std::vector< VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > > &vertices , std::vector< Geometry::Segment > *edges , std::vector< std::vector< unsigned int > > *polygons , bool *vertexPropertiesFlag , int &file_type , std::vector< std::string > *comments=NULL )
	{
		typedef VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > PlyVertex;
		return Read( fileName , vertices , edges , polygons , PlyVertex::ReadProperties() , vertexPropertiesFlag , PlyVertex::ReadNum , file_type , comments );
	}
	template< typename Real , typename ... AdditionalVertexData >
	void Write( std::string fileName , const std::vector< VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > > &vertices , const std::vector< Geometry::Segment > *edges , const std::vector< std::vector< unsigned int > > *polygons , int file_type , const std::vector< std::string > *comments=NULL )
	{
		typedef VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > PlyVertex;
		return Write( fileName , vertices , edges , polygons , PlyVertex::WriteProperties() , PlyVertex::WriteNum , file_type , comments );
	}

	template< typename Real , typename ... AdditionalVertexData >
	void ReadVertices( std::string fileName , std::vector< VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > > &vertices , bool *vertexPropertiesFlag , int &file_type , std::vector< std::string > *comments=NULL )
	{
		typedef VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > PlyVertex;
		return ReadVertices< PlyVertex >( fileName , vertices , PlyVertex::ReadProperties() , vertexPropertiesFlag , PlyVertex::ReadNum , file_type , comments );
	}
	template< typename Real , typename ... AdditionalVertexData >
	void WriteVertices( std::string fileName , const std::vector< VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > > &vertices , int file_type , const std::vector< std::string > *comments=NULL )
	{
		typedef VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > PlyVertex;
		return WriteVertices( fileName , vertices , PlyVertex::WriteProperties() , PlyVertex::WriteNum , file_type , comments );
	}

	template< typename Real , typename ... AdditionalVertexData >
	void ReadTriangles( std::string fileName , std::vector< VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > > &vertices , std::vector< Geometry::Triangle > &triangles , bool *vertexPropertiesFlag , int &file_type , std::vector< std::string > *comments=NULL )
	{
		typedef VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > PlyVertex;
		return ReadTriangles< PlyVertex >( fileName , vertices , triangles , PlyVertex::ReadProperties() , vertexPropertiesFlag , PlyVertex::ReadNum , file_type , comments );
	}
	template< typename Real , typename ... AdditionalVertexData >
	void WriteTriangles( std::string fileName , const std::vector< VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > > &vertices , const std::vector< Geometry::Triangle > &triangles , int file_type , const std::vector< std::string > *comments=NULL )
	{
		typedef VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > PlyVertex;
		return WriteTriangles( fileName , vertices , triangles , PlyVertex::WriteProperties() , PlyVertex::WriteNum , file_type , comments );
	}

	template< typename Real , typename ... AdditionalVertexData >
	void ReadPolygons( std::string fileName , std::vector< VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > > &vertices , std::vector< std::vector< unsigned int > > &polygons , bool *readFlags , int &file_type , std::vector< std::string > *comments=NULL )
	{
		typedef VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > PlyVertex;
		return ReadPolygons( fileName , vertices , polygons , PlyVertex::ReadProperties() , readFlags , PlyVertex::ReadNum , file_type , comments );
	}
	template< typename Real , typename ... AdditionalVertexData >
	void WritePolygons( std::string fileName , const std::vector< VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > > &vertices , const std::vector< std::vector< unsigned int > > &polygons , int file_type , const std::vector< std::string > *comments=NULL )
	{
		typedef VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > PlyVertex;
		return WritePolygons( fileName , vertices , polygons , PlyVertex::WriteProperties() , PlyVertex::WriteNum , file_type , comments );
	}

	template< typename Real , typename ... AdditionalVertexData >
	void ReadTetrahedra( std::string fileName , std::vector< VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > > &vertices , std::vector< Geometry::Tetrahedron > &tetrahedra , bool *vertexPropertiesFlag , int &file_type , std::vector< std::string > *comments=NULL )
	{
		typedef VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > PlyVertex;
		return ReadTetrahedra< PlyVertex >( fileName , vertices , tetrahedra , PlyVertex::ReadProperties() , vertexPropertiesFlag , PlyVertex::ReadNum , file_type , comments );
	}
	template< typename Real , typename ... AdditionalVertexData >
	void WriteTetrahedra( std::string fileName , const std::vector< VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > > &vertices , const std::vector< Geometry::Tetrahedron > &tetrahedra , int file_type , const std::vector< std::string > *comments=NULL )
	{
		typedef VertexData< Real , VertexPosition< 3 , Real > , AdditionalVertexData ... > PlyVertex;
		return WriteTetrahedra( fileName , vertices , tetrahedra , PlyVertex::WriteProperties() , PlyVertex::WriteNum , file_type , comments );
	}
}

#include "Ply.inl"

#endif // PLY_INCLUDED
