#ifndef GEOMETRY_INCLUDED
#define GEOMETRY_INCLUDED

#include <iostream>

namespace Geometry
{
	/** A class representing a point in Dim-dimensions using CType to represent the coordinates */
	template< unsigned int Dim , typename CType >
	struct Point
	{
		/** The coordinates of the point */
		CType coordinates[Dim];

		/** An operator returning a reference the i-th coordinate */
		CType &operator[] ( unsigned int i );

		/** An operator returning a const reference the i-th coordinate */
		const CType &operator[] ( unsigned int i ) const;

		/** An operator determining if two points are the same */
		bool operator == ( const Point &p ) const;

		/** An operator determining if two points are different */
		bool operator != ( const Point &p ) const;
	};

	/** An operator for streaming out points */
	template< unsigned int Dim , class CType >
	std::ostream &operator << ( std::ostream &os , const Point< Dim , CType > &p );

	/** A class representing the indices of a Dim-dimensional simplex */
	template< unsigned int Dim >
	struct Simplex
	{
		/** The indices of the simplex */
		unsigned int indices[Dim+1];

		/** An operator returning a reference the i-th index */
		unsigned int &operator[] ( unsigned int i );

		/** An operator returning a const reference the i-th index */
		const unsigned int &operator[] ( unsigned int i ) const;
	};

	/** An operator for streaming out simplices */
	template< unsigned int Dim >
	std::ostream &operator << ( std::ostream &os , const Simplex< Dim > &s );

	/** Simplified names for some standard points and simplices */
	typedef Point< 2 , int > Point2i;
	typedef Point< 3 , int > Point3i;
	typedef Point< 2 , float > Point2f;
	typedef Point< 3 , float > Point3f;
	typedef Point< 2 , double > Point2d;
	typedef Point< 3 , double > Point3d;
	typedef Simplex< 1 > Segment;
	typedef Simplex< 2 > Triangle;
	typedef Simplex< 3 > Tetrahedron;

	/** A class representing the bisector of two points. */
	struct Bisector
	{
		/** Twice the position of midpoint */
		Geometry::Point2i midpoint2;
		/** The direction of the bisector */
		Geometry::Point2i direction;

		/** Default constructor */
		Bisector( void );

		/** Constructor for the bisector of two points */
		Bisector( Geometry::Point2i p1 , Geometry::Point2i p2 );

		/** Computes the intersection of two bisectors, output with floating point precision */
		static Geometry::Point2d Intersection( Bisector b1 , Bisector b2 );

		/** Computes the circumcenter and radius of the circumcircle passing through three points, in floating point precision */
		static std::pair< Geometry::Point2d , double > CircumcenterAndRadius( Geometry::Point2i p1 , Geometry::Point2i p2 , Geometry::Point2i p3 );
	};

	/** An operator for streaming out bisectors */
	std::ostream &operator << ( std::ostream &os , const Bisector &b );
};

#include "Geometry.inl"
#endif // GEOMETRY_INCLUDED