#include <stdio.h>
#include <stdlib.h>
#include <unordered_set>
#include "Util/CmdLineParser.h"
#include "Util/Geometry.h"
#include "Util/Timer.h"
#include "Util/Ply.h"

////////////////////////////

cmdLineParameter< char* > Out( "out" );
cmdLineParameter< int > Count( "count" ) , RandomSeed( "sRand" , 0 ) , Resolution( "res" , 1024 );
cmdLineReadable ForVisualization( "viewable" );
cmdLineReadable* params[] = { &Count , &Out , &Resolution , &RandomSeed , &ForVisualization , NULL };

void ShowUsage( const char* ex )
{
	printf( "Usage %s:\n" , ex );
	printf( "\t --%s <input vertex count>\n" , Count.name );
	printf( "\t[--%s <output 2D triangulation>]\n" , Out.name );
	printf( "\t[--%s <random seed>=%d]\n" , RandomSeed.name , RandomSeed.value );
	printf( "\t[--%s <grid resolution>=%d]\n" , Resolution.name , Resolution.value );
	printf( "\t[--%s]\n" , ForVisualization.name );
}
// Command line parsing info
////////////////////////////

using namespace Geometry;

void RandomPoints( std::vector< Point2i >& points , int count , int seed , int res )
{
	srand( seed );

	// Add distinct random points in a disk
	points.reserve( count );
	Point2i center;
	center[0] = center[1] = res/2;
	long long r = res / 2;
	std::unordered_set< long long > usedPoints;
	while( points.size()<count )
	{
		Point2i p;
		p[0] = rand() % res , p[1] = rand() % res;
		{
			long long d[] = { center[0] - p[0] , center[1] - p[1] };
			if( d[0]*d[0] + d[1]*d[1]>r*r ) continue;
		}
		long long key = ( ( long long )p[0] ) << 32 | ( ( long long )p[1] );
		if( usedPoints.find( key )==usedPoints.end() ) points.push_back( p ) , usedPoints.insert( key );
	}
}

template< class CType >
void Delaunay( std::vector< Point< 2 , CType > >& points , std::vector< Point< 2 , CType > >& dVertices , std::vector< Triangle >& dTriangles )
{
	/////////////////////////////
	// You need to implement this
	/////////////////////////////
}

int main( int argc , char* argv[] )
{
	PlyProperty Point2iProperties[] =
	{
		{ "x" , PLY_INT , PLY_INT , int( offsetof( Point2i , coordinates[0] ) ) , 0 , 0 , 0 , 0 } ,
		{ "y" , PLY_INT , PLY_INT , int( offsetof( Point2i , coordinates[1] ) ) , 0 , 0 , 0 , 0 }
	};
	PlyProperty Point3fProperties[] =
	{
		{ "x" , PLY_FLOAT , PLY_FLOAT , int( offsetof( Point3f , coordinates[0] ) ) , 0 , 0 , 0 , 0 } ,
		{ "y" , PLY_FLOAT , PLY_FLOAT , int( offsetof( Point3f , coordinates[1] ) ) , 0 , 0 , 0 , 0 } ,
		{ "z" , PLY_FLOAT , PLY_FLOAT , int( offsetof( Point3f , coordinates[2] ) ) , 0 , 0 , 0 , 0 }
	};


	cmdLineParse( argc-1 , argv+1 , params );
	if( !Count.set )
	{
		ShowUsage( argv[0] );
		return EXIT_FAILURE;
	}

	std::vector< Point< 2 , int > > points , dVertices;
	std::vector< Triangle > dTriangles;
	{
		Timer t;
		RandomPoints( points , Count.value , RandomSeed.value , Resolution.value );
		printf( "Got random points: %.2f(s)\n" , t.elapsed() );
	}
	{
		Timer t;
		Delaunay( points , dVertices , dTriangles );
		printf( "Computed Deluanay Triangulation %d -> %d in %.2f (s)\n" , Count.value , (int)dVertices.size() , t.elapsed() );
	}

	if( Out.set )
	{
		if( ForVisualization.set )
		{
			std::vector< Point3f > _dVertices( dVertices.size() );
			std::vector< std::vector< unsigned int > > _dTriangles( dTriangles.size() );
			for( int i=0 ; i<dTriangles.size() ; i++ )
			{
				_dTriangles[i].resize( 3 );
				for( int j=0 ; j<3 ; j++ ) _dTriangles[i][j] = dTriangles[i][j];
			}
			for( int i=0 ; i<dVertices.size() ; i++ )
			{
				_dVertices[i][0] = (float) dVertices[i][0];
				_dVertices[i][1] = (float) dVertices[i][1];
				_dVertices[i][2] = 0.f;
			}
			PLY::Write( Out.value , _dVertices , _dTriangles , Point3fProperties , 3 , PLY_ASCII );
		}
		else PLY::Write( Out.value , dVertices , NULL , &dTriangles , NULL , Point2iProperties , 2 , PLY_ASCII );
	}

	return EXIT_SUCCESS;
}

