#include <stdio.h>
#include <stdlib.h>
#include <unordered_set>
#include "Util/CmdLineParser.h"
#include "Util/Geometry.h"
#include "Util/Timer.h"
#include "Util/Ply.h"

////////////////////////////
// Command line parsing info
enum
{
	ALGORITHM_NAIVE ,
	ALGORITHM_GIFT_WRAP ,
	ALGORITHM_INCREMENTAL ,
	ALGORITHM_COUNT
};
const char* AlgorithmNames[] = { "Naive" , "Gift-Wrap" , "Incremental" };

cmdLineParameter< char* > Out( "out" );
cmdLineParameter< int > Count( "count" ) , AlgorithmType( "aType" , ALGORITHM_NAIVE ) , RandomSeed( "sRand" , 0 ) , Resolution( "res" , 1024 );
cmdLineReadable ForVisualization( "viewable" );
cmdLineReadable* params[] = { &Count , &Out , &Resolution , &AlgorithmType , &RandomSeed , &ForVisualization , NULL };

void ShowUsage( const char* ex )
{
	printf( "Usage %s:\n" , ex );
	printf( "\t --%s <input vertex count>\n" , Count.name );
	printf( "\t[--%s <output 3D triangulation>]\n" , Out.name );
	printf( "\t[--%s <algorithm type>=%d]\n" , AlgorithmType.name , AlgorithmType.value );
	for( int i=0 ; i<ALGORITHM_COUNT ; i++ ) printf( "\t\t%d] %s\n" , i , AlgorithmNames[i] );
	printf( "\t[--%s <random seed>=%d]\n" , RandomSeed.name , RandomSeed.value );
	printf( "\t[--%s <grid resolution>=%d]\n" , Resolution.name , Resolution.value );
	printf( "\t[--%s]\n" , ForVisualization.name );
}
// Command line parsing info
////////////////////////////

using namespace Geometry;

long long Volume6( const Point3i p[4] )
{
	long long matrix[3][3];
	for( int i=0 ; i<3 ; i++ ) for( int j=0 ; j<3 ; j++ ) matrix[i][j] = p[i+1][j]-p[0][j];
	long long det = 0;
	for( int i=0 ; i<3 ; i++ )
	{
		long long _det = matrix[(i+1)%3][1] * matrix[(i+2)%3][2] - matrix[(i+2)%3][1] * matrix[(i+1)%3][2];
		det += matrix[i][0] * _det;
	}
	return det;
}

long long Volume6( Point3i p1 , Point3i p2 , Point3i p3 , Point3i p4 )
{
	Point3i p[] = { p1 , p2 , p3 , p4 };
	return Volume6( p );
}

long long SquaredArea( const Point3i p[3] )
{
	Point3i d[2];
	for( int i=0 ; i<3 ; i++ ) d[0][i] = p[1][i]-p[0][i] , d[1][i] = p[2][i]-p[0][i];
	Point3i n;
	n[0] = d[0][1]*d[1][2] - d[0][2] * d[1][1];
	n[1] = d[0][2]*d[1][0] - d[0][0] * d[1][2];
	n[2] = d[0][0]*d[1][1] - d[0][1] * d[1][0];
	long long a = 0;
	for( int i=0 ; i<3 ; i++ ) a += (long long)n[i] * (long long)n[i];
	return a;
}

void RandomPoints( std::vector< Point3i >& points , int count , int seed , int res )
{
	const int MAX_EXP= ( sizeof(long long) * 8 ) / 3;
	if( res>(1<<MAX_EXP) ) fprintf( stderr , "[WARNING] Maximum resolution is: %d\n" , 1<<MAX_EXP ) , res = 1<<MAX_EXP;

	srand( seed );

	// Add distinct random points in a disk
	points.reserve( count );
	Point3i center;
	center[0] = center[1] = center[2] = res/2;
	long long r = res / 2;
	std::unordered_set< long long > usedPoints;
	while( points.size()<count )
	{
		Point3i p;
		p[0] = rand() % res , p[1] = rand() % res , p[2] = rand() % res;
		{
			long long d[] = { center[0] - p[0] , center[1] - p[1] , center[2] - p[2] };
			if( d[0]*d[0] + d[1]*d[1] + d[2]*d[2] >r*r ) continue;
		}
		long long key = ( ( ( long long )p[0] ) << ( 2 * MAX_EXP ) ) | ( ( ( long long )p[1] ) << MAX_EXP ) | ( ( long long )p[2] );
		if( usedPoints.find( key )==usedPoints.end() ) points.push_back( p ) , usedPoints.insert( key );
	}
}

template< class CType >
void NaiveAlgorithm( std::vector< Point< 3 , CType > >& points , std::vector< Point< 3 , CType > >& hullV , std::vector< Triangle >& hullF )
{
	// [WARNING] This implementation assumes that no four points are co-planar.
	auto ValidTriangle = [&]( Triangle t )
	{
		for( int i=0 ; i<points.size() ; i++ )
		{
			if( i!=t[0] && i!=t[1] && i!=t[2] )
			{
				long long v6 = Volume6( points[ t[0] ] , points[ t[1] ] , points[ t[2] ] , points[i] );
				if( v6>0 ) return false;
			}
		}
		return true;
	};

	for( int i=0 ; i<points.size() ; i++ ) for( int j=0 ; j<i ; j++ ) for( int k=0 ; k<j ; k++ )
	{
		Triangle t1 , t2;
		t1[0] = i , t1[1] = j , t1[2] = k;
		t2[0] = i , t2[1] = k , t2[2] = j;
		if( ValidTriangle( t1 ) ) hullF.push_back( t1 );
		if( ValidTriangle( t2 ) ) hullF.push_back( t2 );
	}

	std::vector< unsigned int > vMap( points.size() , -1 );
	for( int i=0 ; i<hullF.size() ; i++ )
	{
		for( int j=0 ; j<3 ; j++ )
		{
			unsigned int idx;
			if( vMap[ hullF[i][j] ]!=-1 ) idx = vMap[ hullF[i][j] ];
			else
			{
				idx = (unsigned int)hullV.size();
				hullV.push_back( points[ hullF[i][j] ] );
				vMap[ hullF[i][j] ] = idx;
			}
			hullF[i][j] = idx;
		}
	}
}

template< class CType >
void GiftWrapAlgorithm( std::vector< Point< 3 , CType > >& points , std::vector< Point< 3 , CType > >& hullV , std::vector< Triangle >& hullF )
{
	/////////////////////////////
	// You need to implement this
	/////////////////////////////
}
template< class CType >
void IncrementalAlgorithm( std::vector< Point< 3 , CType > >& points , std::vector< Point< 3 , CType > >& hullV , std::vector< Triangle >& hullF )
{
	/////////////////////////////
	// You need to implement this
	/////////////////////////////
}

int main( int argc , char* argv[] )
{
	PlyProperty Point3iProperties[] =
	{
		{ "x" , PLY_INT , PLY_INT , int( offsetof( Point3i , coordinates[0] ) ) , 0 , 0 , 0 , 0 } ,
		{ "y" , PLY_INT , PLY_INT , int( offsetof( Point3i , coordinates[1] ) ) , 0 , 0 , 0 , 0 } ,
		{ "z" , PLY_INT , PLY_INT , int( offsetof( Point3i , coordinates[2] ) ) , 0 , 0 , 0 , 0 }
	};
	PlyProperty Point3fProperties[] =
	{
		{ "x" , PLY_FLOAT , PLY_FLOAT , int( offsetof( Point3f , coordinates[0] ) ) , 0 , 0 , 0 , 0 } ,
		{ "y" , PLY_FLOAT , PLY_FLOAT , int( offsetof( Point3f , coordinates[1] ) ) , 0 , 0 , 0 , 0 } ,
		{ "z" , PLY_FLOAT , PLY_FLOAT , int( offsetof( Point3f , coordinates[2] ) ) , 0 , 0 , 0 , 0 }
	};


	cmdLineParse( argc-1 , argv+1 , params );
	if( !Count.set )
	{
		ShowUsage( argv[0] );
		return EXIT_FAILURE;
	}

	std::vector< Point< 3 , int > > points , hullVertices;
	std::vector< Triangle > hullFaces;
	{
		Timer t;
		RandomPoints( points , Count.value , RandomSeed.value , Resolution.value );
		printf( "Got random points: %.2f (s)\n" , t.elapsed() );
	}
	{
		Timer t;
		switch( AlgorithmType.value )
		{
			case ALGORITHM_NAIVE:             NaiveAlgorithm( points , hullVertices , hullFaces ) ; break;
			case ALGORITHM_GIFT_WRAP:      GiftWrapAlgorithm( points , hullVertices , hullFaces ) ; break;
			case ALGORITHM_INCREMENTAL: IncrementalAlgorithm( points , hullVertices , hullFaces ) ; break;
			default: fprintf( stderr , "[ERROR] Unrecognized algorithm type: %d\n" , AlgorithmType.value ) , exit( 0 );
		}
		printf( "Computed hull %d -> %d in %.2f (s)\n" , Count.value , (int)hullVertices.size() , t.elapsed() );
	}

	if( Out.set )
	{
		if( ForVisualization.set )
		{
			std::vector< std::vector< unsigned int > > _hullFaces( hullFaces.size() );
			std::vector< Point3f > _hullVertices( hullVertices.size() );
			for( int i=0 ; i<hullFaces.size() ; i++ )
			{
				_hullFaces[i].resize( 3 );
				for( int j=0 ; j<3 ; j++ ) _hullFaces[i][j] = hullFaces[i][j];
			}
			for( int i=0 ; i<hullVertices.size() ; i++ )
			{
				_hullVertices[i][0] = (float) hullVertices[i][0];
				_hullVertices[i][1] = (float) hullVertices[i][1];
				_hullVertices[i][2] = (float) hullVertices[i][2];
			}
			PLY::Write( Out.value , _hullVertices , _hullFaces , Point3fProperties , 3 , PLY_ASCII );
		}
		else PLY::Write( Out.value , hullVertices , NULL , &hullFaces , NULL , Point3iProperties , 3 , PLY_ASCII );
	}

	return EXIT_SUCCESS;
}

