#include <stdio.h>
#include <stdlib.h>
#include <sstream>
#include <limits>
#include "Fourier.h"
#include "SphericalGrid.h"
#include "cmdLineParser.h"
#include "gridIO.h"

using namespace std;
using namespace Util;

CmdLineParameter< string > Input( "in" ) , OutputHeader( "out" );
CmdLineParameter< int > Rotational( "rot" );
CmdLineReadable Reflective( "ref" ) , Axial( "axial" ) , Double( "double" );

CmdLineReadable* params[] =
{
	&Input , &OutputHeader , &Rotational , &Reflective , &Axial , &Double ,
	NULL
};

void ShowUsage( const string &ex )
{
	cout << "Usage " << ex << ":" << endl;
	cout << "\t --" << Input.name  << " <input spherical grid>"           << endl;
	cout << "\t --" << OutputHeader.name << " <output header>"            << endl;
	cout << "\t[--" << Rotational.name << " <rotational symmetry index>"  << endl;
	cout << "\t[--" << Reflective.name                             << "]" << endl;
	cout << "\t[--" << Axial.name                                  << "]" << endl;
	cout << "\t[--" << Double.name                                 << "]" << endl;
}

template< typename Real >
void Execute( void )
{
	SphericalGrid< Real > in , sym;
	FourierKeyS2< Real > keyS2;
	HarmonicTransform< Real > hForm;

	// Read in the input spherical function
	ReadSphericalGrid( Input.value , in );

	// Kill off the 0th order frequency component and normalize so that the function has unit norm.
	hForm.ForwardFourier( in , keyS2 );
	keyS2(0,0) = 0;
	hForm.InverseFourier( keyS2 , in );
	Real l = (Real)sqrt( in.squareNorm() );
	for( int i=0 ; i<in.resolution() ; i++ ) for( int j=0 ; j<in.resolution() ; j++ ) in(i,j) /= l;

	// Allocate memory
	sym.resize( in.resolution() );

	// Compute the spherical harmonic transform
	hForm.ForwardFourier( in , keyS2 );


	if( Reflective.set )
	{
		/*******************************************************************************************/
		/* Compute the reflective symmetry here.                                                   */
		/* Set the reflective symmetry descriptor values into the SphericalGrid "sym".             */
		/* You may want to consider using the method:                                              */
		/*     void SphericalGrid< Real >::setCoordinate( Real i , Real j , Real axis[3] ) const;  */
		/* to get the axis associated with a particular bin of the spherical grid, and the method: */
		/*     Real RotationGrid< Real >::operator()( const Real axis[3] , Real angle );           */
		/* for sampling the rotation grid at rotation parametrized by the axis/angle.              */
		/*******************************************************************************************/
		WARN( "functionality not implemented" );


		// Print out the maximal measure of reflective symmetry
		Real maxSymmetry = -std::numeric_limits< Real >::infinity();
		for( int i=0 ; i<sym.resolution() ; i++ ) for( int j=0 ; j<sym.resolution() ; j++ )  maxSymmetry = std::max< Real >( maxSymmetry , sym(i,j) );
		cout << "Reflective symmetry: " << maxSymmetry << endl;

		// Write out the symmetry descriptor
		stringstream outputNameStream;
		outputNameStream << OutputHeader.value << ".ref.sgrid";
		string outputName;
		outputNameStream >> outputName;
		WriteSphericalGrid( outputName , sym );
	}


	if( Rotational.set || Axial.set )
	{
		if( Rotational.set )
		{
			for( int k=2 ; k<=Rotational.value ; k++ )
			{
				/**************************************************************************************/
				/* Compute the k-fold rotational symmetry here.                                       */
				/* Set the k-fold rotational symmetry descriptor values into the SphericalGrid "sym". */
				/**************************************************************************************/
				WARN( "functionality not implemented" );


				// Print out the maximal measure of rotational symmetry
				Real maxSymmetry = -std::numeric_limits< Real >::infinity();
				for( int i=0 ; i<sym.resolution() ; i++ ) for( int j=0 ; j<sym.resolution() ; j++ )  maxSymmetry = std::max< Real >( maxSymmetry , sym(i,j) );
				cout << "Rotational symmetry[" << k << "]: " << maxSymmetry << endl;

				// Write out the symmetry descriptor
				stringstream outputNameStream;
				outputNameStream << OutputHeader.value << ".rot." << k << ".sgrid";
				string outputName;
				outputNameStream >> outputName;
				WriteSphericalGrid( outputName , sym );
			}
		}
		if( Axial.set )
		{
			/*************************************************************************************/
			/* Compute the axial rotational symmetry here.                                       */
			/* Set the axial rotational symmetry descriptor values into the SphericalGrid "sym". */
			/*************************************************************************************/
			WARN( "functionality not implemented" );


			// Print out the maximal measure of axial symmetry
			Real maxSymmetry = -std::numeric_limits< Real >::infinity();
			for( int i=0 ; i<sym.resolution() ; i++ ) for( int j=0 ; j<sym.resolution() ; j++ )  maxSymmetry = std::max< Real >( maxSymmetry , sym(i,j) );
			cout << "Axial symmetry: " << maxSymmetry << endl;

			// Write out the symmetry descriptor
			stringstream outputNameStream;
			outputNameStream << OutputHeader.value << ".axial.sgrid";
			string outputName;
			outputNameStream >> outputName;
			WriteSphericalGrid( outputName , sym );
		}
	}
}

int main( int argc , char *argv[] )
{
	CmdLineParse( argc-1 , argv+1 , params );
	if( !Input.set || !OutputHeader.set ){ ShowUsage( argv[0] ) ; return EXIT_FAILURE; }
	try
	{
		if( Double.set ) Execute< double >();
		else             Execute< float  >();
	}
	catch( Exception e )
	{
		cout << e.what() << endl;
		return EXIT_FAILURE;
	}
	return EXIT_SUCCESS;
}

