#ifndef ROTATION_GRID_INCLUDED
#define ROTATION_GRID_INCLUDED

#ifndef PI
#define PI 3.1415926535897932384
#endif 


// This templated class represents a function on the group of rotations, where index (i,j,k) corresponds
// to the rotation with Euler coefficients:
//			theta =2PI*j/r
//			phi   =PI*(2i+1)/(2r)
//			psi   =2PI*k/r
// where r is the resolution of the sampling.

template< class Real >
class RotationGrid
{
public:
	int res;
	Real* values;
	RotationGrid( void );
	~RotationGrid( void );

	// Returns the dimension of the array
	int resolution( void ) const;
	// Allocates memory for the array
	int resize( int res );

	// Clears the values of the array to 0
	void clear( void );

	// Returns a reference to the indexed array element
	Real& operator() ( int x , int y , int z );
	const Real &operator() ( int x , int y , int z ) const { return const_cast< RotationGrid * >(this)->operator()(x,y,z); }

	// Returns the linear interpolation of the value at the specified index
	Real operator()( Real x , Real y , Real z );
	Real operator()( Real x , Real y , Real z ) const { return const_cast< RotationGrid * >(this)->operator()(x,y,z); } 

	// Returns the linear interpolation of the value at the specified rotation
	Real operator()( const Real matrix[3][3] );
	Real operator()( const Real matrix[3][3] ) const { return const_cast< RotationGrid * >(this)->operator()(matrix); } 

	// Returns the linear interpolation of the value at the specified axis/angle rotation
	Real operator()( const Real axis[3] , Real angle );
	Real operator()( const Real axis[3] , Real angle ) const { return const_cast< RotationGrid * >(this)->operator()(axis,angle); } 

	// Returns the square of the L2-norm of the array elements
	Real squareNorm( void ) const;

	// Reads in an array from the specified file
	int read(const char* fileName);
	int read(FILE* fp);

	// Writes out the array to the specified file
	int write(const char* fileName) const;
	int write(FILE* fp) const;

	// Sets the matrix coordinates of the Euler angle (theta,phi,psi)
	static void SetCoordinates( Real theta , Real phi , Real psi , Real matrix[3][3] );
	// Sets the coordinates (theta,phi,psi) associated to the specified rotation matrix
	static void SetCoordinates( const Real matrix[3][3] , Real& theta , Real& phi , Real& psi );
	// Sets the matrix coordinates of the the rotation about the specified axis with the specified angle
	static void SetCoordinates( const Real axis[3] , Real angle , Real matrix[3][3] );

	// Sets the matrix coordinates of the Euler angle indexed by (i,j,k)
	void setCoordinates( Real i , Real j , Real k , Real matrix[3][3] ) const;
	// Sets the coordinates (i,j,k) associated to the specified rotation matrix
	void setCoordinates( const Real matrix[3][3] , Real& i , Real& j , Real& k ) const;

	// Returns the square of the L2-difference between two arrays
	static Real SquareDifference(const RotationGrid& g1,const RotationGrid& g2);

	// Returns the dot-product of two arrays
	static Real Dot(const RotationGrid& g1,const RotationGrid& g2);
};
#include "RotationGrid.inl"
#endif // ROTATION_GRID_INCLUDED

