#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "FFTW/fftw3.h"
#include <math.h>

//////////////////
// FourierKey1D //
//////////////////
template< class Real > FourierKey1D< Real >::FourierKey1D( void ){
	dim=res=0;
	values=NULL;
}
template< class Real > FourierKey1D< Real >::~FourierKey1D( void ){
	if(values){delete[] values;}
	values=NULL;
	dim=res=0;
}
template< class Real > int FourierKey1D< Real >::read(const char* fileName){
	FILE* fp=fopen(fileName,"rb");
	if(!fp){return 0;}
	int r=read(fp);
	fclose(fp);
	return r;
}
template< class Real > int FourierKey1D< Real >::write(const char* fileName) const{
	FILE* fp=fopen(fileName,"wb");
	if(!fp){return 0;}
	int w=write(fp);
	fclose(fp);
	return w;
}
template< class Real > int FourierKey1D< Real >::read(FILE* fp){
	int resolution,r;
	r=int(fread(&resolution,sizeof(int),1,fp));
	if(!r){return 0;}
	resize(resolution);
	r=int(fread(values,sizeof(Complex< Real >),dim,fp));
	if(r==dim){return 1;}
	else{return 0;}
}
template< class Real > int FourierKey1D< Real >::write(FILE* fp) const {
	int w;
	w=int(fwrite(&res,sizeof(int),1,fp));
	if(!w){return 0;}
	w=int(fwrite(values,sizeof(Complex< Real >),dim,fp));
	if(w==dim){return 1;}
	else{return 0;}
}
template< class Real > int FourierKey1D< Real >::size( void ) const{return dim;}
template< class Real > int FourierKey1D< Real >::resolution( void ) const{return res;}
template< class Real > int FourierKey1D< Real >::resize(int resolution,int clr){
	int d=FourierTransform< Real >::BandWidth(resolution);
	if(resolution<0){return 0;}
	else if(d!=dim){
		if(values){delete[] values;}
		values=NULL;
		dim=0;
		res=0;
		if(d){
			values=new Complex< Real >[d];
			if(!values){return 0;}
			else{dim=d;}
		}
	}
	res=resolution;
	if(clr){clear();}
	return 1;
}
template< class Real > void FourierKey1D< Real >::clear( void ){if(dim){memset(values,0,sizeof(Complex< Real >)*dim);}}
template< class Real > Complex< Real > &FourierKey1D< Real >::operator() (int i){return values[i];}
template< class Real > const Complex< Real > &FourierKey1D< Real >::operator() (int i) const {return values[i];}
template< class Real > Real FourierKey1D< Real >::squareNorm( void ) const{ return Dot( *this , *this ).r; }
template< class Real > Real FourierKey1D< Real >::SquareDifference( const FourierKey1D& g1 , const FourierKey1D& g2 ){ return g1.squareNorm() + g2.squareNorm() - 2*Dot(g1,g2).r; }
template< class Real > Complex< Real > FourierKey1D< Real >::Dot( const FourierKey1D& g1 , const FourierKey1D& g2 )
{
	Complex< Real > dot;
	if( g1.res!=g2.res ) fprintf( stderr , "Could not compare arrays of different sizes: %d != %d\n" , g1.dim , g2.dim ) , exit(0);
#ifdef NORMALIZED_FFT
	Real n = 1;
#else // !NORMALIZED_FFT
	Real n = (Real)( 1 /( 2.0 * PI ) );
#endif // NORMALIZED_FFT
	dot += g1.values[0] * g2.values[0].conjugate();
	for( int i=1 ; i<g1.dim-1 ; i++ ) dot += g1.values[i] * g2.values[i].conjugate() * 2;
	if( g1.res & 1 ) dot += g1.values[g1.dim-1] * g2.values[g1.dim-1].conjugate() * 2;
	else             dot += g1.values[g1.dim-1] * g2.values[g1.dim-1].conjugate();
	return dot * n;
}

//////////////////////
// FourierTransform //
//////////////////////
template<>
int FourierTransform<float>::ForwardFourier( const CircularArray< float >& g , FourierKey1D< float >& key )
{
	if(key.resolution()!=g.resolution()){key.resize(g.resolution());}
	fftwf_plan plan = fftwf_plan_dft_r2c_1d( g.resolution() , const_cast< float * >( &g(0) ) , (fftwf_complex*)&key(0) , FFTW_PRESERVE_INPUT | FFTW_ESTIMATE );
	fftwf_execute(plan);
	fftwf_destroy_plan(plan);
#ifdef NORMALIZED_FFT
	float n = (float)( sqrt( 2.0*PI )/g.resolution() );
	for( int i=0 ; i<key.size() ; i++ ) key(i) *= n;
#else // !NORMALIZED_FFT
	float n=float(1.0/(2.0*PI))*g.resolution();
	for( int i=0 ; i<key.size() ; i++ ) key(i) /= n;
#endif // NORMALIZED_FFT
	return 1;
}
template<>
int FourierTransform<double>::ForwardFourier( const CircularArray< double >& g , FourierKey1D< double >& key )
{
	if(key.resolution()!=g.resolution()){key.resize(g.resolution(),1);}
	fftw_plan plan = fftw_plan_dft_r2c_1d( g.resolution() , const_cast< double * >( &g(0) ) , (fftw_complex*)&key(0) , FFTW_PRESERVE_INPUT | FFTW_ESTIMATE );
	fftw_execute(plan);
	fftw_destroy_plan(plan);
#ifdef NORMALIZED_FFT
	double n = sqrt( 2.0*PI )/g.resolution();
	for( int i=0 ; i<key.size() ; i++ ) key(i) *= n;
#else // !NORMALIZED_FFT
	double n = 1.0/(2.0*PI)*g.resolution();
	for( int i=0 ; i<key.size() ; i++ ) key(i) /= n;
#endif // NORMALIZED_FFT
	return 1;
}
template< class Real >
int FourierTransform< Real >::ForwardFourier( const CircularArray< Real >& g,FourierKey1D< Real >& key){
	fprintf(stderr,"Only float and double precision FFTs suppored\n");
	return 0;
}
template<>
int FourierTransform< float >::InverseFourier( const FourierKey1D< float >& key , CircularArray< float >& g )
{
	if(key.resolution()!=g.resolution()){g.resize(key.resolution());}
	fftwf_plan plan = fftwf_plan_dft_c2r_1d( g.resolution() , (fftwf_complex*)(&key(0)) , &g(0) , FFTW_PRESERVE_INPUT | FFTW_ESTIMATE );
	fftwf_execute(plan);
	fftwf_destroy_plan(plan);
#ifdef NORMALIZED_FFT
	float n = (float)( 1/sqrt( 2*PI ) );
	for( int i=0 ; i<g.resolution() ; i++ ) g(i) *= n;
#else // !NORMALIZED_FFT
	float n=float(2.0*PI);
	for( int i=0 ; i<g.resolution() ; i++ ) g(i) /= n;
#endif // NORMALIZED_FFT
	return 1;
}
template<>
int FourierTransform<double>::InverseFourier( const FourierKey1D< double >& key , CircularArray< double >& g )
{
	if(key.resolution()!=g.resolution()){g.resize(key.resolution());}
	fftw_plan plan = fftw_plan_dft_c2r_1d( g.resolution() , (fftw_complex*)&key(0) , &g(0) , FFTW_PRESERVE_INPUT | FFTW_ESTIMATE );
	fftw_execute(plan);
	fftw_destroy_plan(plan);
#ifdef NORMALIZED_FFT
	// To transform the normalized Fourier coefficients to the un-normalized Fourier coefficients, we need to multiply by
	//		N / sqrt( 2 * Pi )
	// To get the full inverse fourier transform we need to multiply the results of the FFTW by:
	//		1/N
	double n = 1/sqrt( 2*PI );
	for( int i=0 ; i<g.resolution() ; i++ ) g(i) *= n;
#else // !NORMALIZED_FFT
	double n=2.0*PI;
	for( int i=0 ; i<g.resolution() ; i++ ) g(i) /= n;
#endif // NORMALIZED_FFT
	return 1;
}
template< class Real >
int FourierTransform< Real >::InverseFourier( const FourierKey1D< Real >& key , CircularArray< Real >& g )
{
	fprintf(stderr,"Only float and double precision FFTs suppored\n");
	return 0;
}
