#include <stdio.h>
#include <stdlib.h>
#include <algorithm>
#include <unordered_map>
#include <Eigen/Sparse>

#ifdef _OPENMP
#include <omp.h>
#else // !_OPENMP
int  omp_get_num_procs  ( void ){ return 1; }
int  omp_get_max_threads( void ){ return 1; }
int  omp_get_thread_num ( void ){ return 0; }
void omp_set_num_threads( int  ){ }
void omp_set_num_threads( int  ){ }
#endif // _OPENMP

#include "Util/Ply.h"
#include "Util/Geometry.h"
#include "Util/CmdLineParser.h"
#include "Util/Timer.h"

CmdLineParameter< char* > In( "in" ) , Out( "out" );
CmdLineParameter< float > Smoothness( "smooth" , 1e-7f );
CmdLineParameter< int > Iters( "iters" , 1 );
CmdLineReadable CMCF( "cmcf" ) , Verbose( "verbose" ) , ASCII( "ascii" );

void Usage( const char* ex )
{
	printf( "Usage %s:\n" , ex );
	printf( "\t --%s <input mesh>\n" , In.name );
	printf( "\t[--%s <output mesh>]\n" , Out.name );
	printf( "\t[--%s <iters>=%d]\n" , Iters.name , Iters.value );
	printf( "\t[--%s <smoothness weight>=%g]\n" , Smoothness.name , Smoothness.value );
	printf( "\t[--%s]\n" , Verbose.name );
	printf( "\t[--%s]\n" , CMCF.name );
	printf( "\t[--%s]\n" , ASCII.name );
}
CmdLineReadable* cmdLineParameters[] = { &In , &Out , &Iters , &Smoothness , &Verbose , &CMCF , &ASCII , NULL };

float MakeUnitArea( const std::vector< TriangleIndex >& triangles , std::vector< PlyVertex< float > >& vertices )
{
	// Rescale the mesh so that it has unit area and return the scaling factor that was applied
	return 1.f;
}

typedef Eigen::SimplicialLLT< Eigen::SparseMatrix< double > > Eigen_Solver;
typedef Eigen::VectorXd Eigen_Vector;
typedef Eigen::SparseMatrix< double > Eigen_Matrix;

void SetMassMatrix( const std::vector< TriangleIndex >& triangles , const std::vector< PlyVertex< float > >& vertices , Eigen_Matrix& M )
{
	M.resize( (int)vertices.size() , (int)vertices.size() );
	std::vector< Eigen::Triplet< double > > triplets;

	// Add the matrix entries into the vectors
	// ...

	// Compute the matrices from the lists of entries
	M.setFromTriplets( triplets.begin() , triplets.end() );
}
void SetStiffnessMatrix( const std::vector< TriangleIndex >& triangles , const std::vector< PlyVertex< float > >& vertices , Eigen_Matrix& S )
{
	S.resize( (int)vertices.size() , (int)vertices.size() );
	std::vector< Eigen::Triplet< double > > triplets;

	// Add the matrix entries into the vectors
	// ...

	// Compute the matrices from the lists of entries
	S.setFromTriplets( triplets.begin() , triplets.end() );
}

int main( int argc , char* argv[] )
{
	// Read the command line arguments
	CmdLineParse( argc-1 , argv+1 , cmdLineParameters );

	// Check that an input mesh has been specified
	if( !In.set )
	{
		Usage( argv[0] );
		return EXIT_FAILURE;
	}

	float scale = 1.f;

	// Storage for the vertices and triangles of the mesh
	std::vector< PlyVertex< float > > vertices;
	std::vector< TriangleIndex > triangles;

	{
		// Flags for tracking which ply parameters were read
		PlyReadTriangles( In.value , vertices , triangles , PlyVertex< float >::ReadProperties , NULL , PlyVertex< float >::ReadComponents );
		scale = MakeUnitArea( triangles , vertices );
		printf( "Read mesh: %d vertices , %d triangles\n" , (int)vertices.size() , (int)triangles.size() );
	}

	Eigen_Matrix M , S;
	for( int i=0 ; i<Iters.value ; i++ )
	{
		Timer t;
		// Get the mass and stiffness matrices
		// Create the solver
		// Define the right hand side
		// Solve the linear system

		printf( "Iteration[%d]: %.2f(s)\n" , i+1 , t.elapsed() );
	}

	if( Out.set )
	{
#pragma omp parallel for
		for( int i=0 ; i<vertices.size() ; i++ ) vertices[i].position /= scale;
		PlyWriteTriangles( Out.value , vertices , triangles , PlyVertex< float >::ReadProperties , PlyVertex< float >::ReadComponents , ASCII.set ? PLY_ASCII : PLY_BINARY_NATIVE );
	}
	return EXIT_SUCCESS;
}