/*
Copyright (c) 2009, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "Util.h"
#include "RangeGrid.h"
#include "Ply.h"
#include "Mesh.h"
#include "Octree.h"
#include "SparseMatrix.h"
#include "CmdLineParser.h"
#include "MemoryUsage.h"
#include "PoissonMeshData.h"
#include "KDtree.h"

#define SCALE_FACTOR 1.1f
#define MEMORY_ALLOCATOR_BLOCK_SIZE 1<<12

template< class Vertex >
struct KDMesh
{
	Point3D< float > DoP;
	std::vector< Vertex > v;
	std::vector< TriangleIndex > f;
	mutable std::vector< std::vector< int > > af;
	KDtree *kd;

	KDMesh() : kd( NULL ) {}
	~KDMesh() { if (kd) delete kd; }

	void build_kd(void)
	{
		if (kd) delete kd;
		kd = new KDtree( &v[0].point[0] , v.size() , sizeof( Vertex ) );
	}

	void need_adjacent_faces() const
	{
		if( af.size() == v.size() ) return;

		af.clear();
		af.resize( v.size() );
		for( size_t i = 0 ; i<f.size() ; i++ )
			for( int j=0 ; j<3 ; j++ ) 
				af[ f[i][j] ].push_back((int) i);
	}
	int closest_vertex( Point3D< float > p ) const
	{
		assert( kd );
		float maxDist2 = -1;
		Point3D< float > *nearest = (Point3D<float> *) kd->closest_to_pt((const float *) &p[0], maxDist2 );
		if( nearest ) return (int)( ( size_t( nearest ) - size_t( &v[0].point[0] ) ) / sizeof( Vertex ) );
		else return -1;
	}
	Point3D< double > normal( int face ) const
	{
		return Point3D< double >::CrossProduct( Point3D< double >( v[ f[face][1] ] ) - Point3D< double >( v[ f[face][0] ] ) , Point3D< double >( v[ f[face][2] ] ) - Point3D< double >( v[ f[face][0] ] ) );
	}
	void setDirectionOfProjection( void )
	{
		std::vector< Point3D< double > > normals;
		DoP *= 0;
		normals.resize( v.size() );
		for( int i=0 ; i<v.size() ; i++ ) normals[i] *= 0;
		for( int i=0 ; i<f.size() ; i++ )
		{
			Point3D< float > normal = Point3D< float >( this->normal( i ) );
			for( int j=0 ; j<3 ; j++ ) normals[ f[i][j] ] += normal;
		}
		for( int i=0 ; i<v.size() ; i++ )
		{
			double w = Length( normals[i] );
			if( w ) DoP += normals[i] / w / w;
		}
	}
};

cmdLineString In( "in" ) , Out( "out" ) , Scans( "scans" ) , XForms( "xForms" );
cmdLineInt Depth( "depth" , 8 ) , Iters( "iters" , 10 ) , Subdivide( "subdivide" , 0 ) , Cycles( "cycles" , 1 ) , MonteCarlo( "monteCarlo" , 0 );
cmdLineFloat Scale( "scale" , SCALE_FACTOR ) , CleanTree( "cutOff" , 0.00001f ) , IMultiplier( "iMultiplier" , 3.f ) , VWeight( "vWeight" , 0.f ) , GWeight( "gWeight" , 1.f ) , GrazingFallOff( "grazingFallOff" , 2.f );
cmdLineReadable Verbose( "verbose" ) , FullVerbose( "fullVerbose" ) , ASCII( "ascii" ) , Double( "double" ) , UseKD( "useKD" ) , Progress( "progress" ) , Primal( "primal" ) , NoSolver( "noSolver" ) , NoTriangleSamples( "noTSamples" ) , VertexSamples( "vSamples" ) , NoRefinedSampling( "noRSampling" );

cmdLineReadable* params[]=
{
	&In , &Out , &Scans , &XForms,
	&Depth , &Iters , &Subdivide , &Cycles , &MonteCarlo ,
	&Scale , &CleanTree , &IMultiplier , &VWeight , &GWeight , &GrazingFallOff ,
	&Verbose , &FullVerbose , &ASCII , &Double , &UseKD , &Progress , &Primal , &NoSolver , &NoTriangleSamples , &VertexSamples , &NoRefinedSampling ,
};

void ShowUsage(char* ex)
{
	printf("Usage: %s\n",ex);
	printf( "\t --%s <input triangle mesh>\n" ,			In.name);
	printf( "\t --%s <input range scans>\n" ,			Scans.name);
	printf( "\t --%s <ouput triangle mesh>\n" ,			Out.name);
	printf( "\t[--%s]\n" , UseKD.name );
	printf( "\t[--%s <range grid transformations>\n" ,	XForms.name);

	printf("\t[--%s <maximum reconstruction depth>=%d]\n",Depth.name,Depth.value);
	printf("\t\t Running at depth d corresponds to solving on a 2^d x 2^d x 2^d\n");
	printf("\t\t voxel grid.\n");

	printf("\t[--%s <solver iterations>=%d]\n",Iters.name,Iters.value);
	printf("\t\t The number of solver iterations within each resolution\n");

	printf("\t[--%s <cycles>=%d]\n" , Cycles.name , Cycles.value );
	printf("\t\t The number of solver cycles to be performed\n");

	printf( "\t[--%s <integration samples>]\n" , MonteCarlo.name );
	printf( "\t\t Specifies the number of point samples that are to be used\n" );
	printf( "\t\t to approximate the integrals defining the system coefficients.\n" );

	printf( "\t[--%s <value weight>=%f]\n" , VWeight.name , VWeight.value );
	printf("\t[--%s <subdivision iterations>=%d]\n",Subdivide.name,Subdivide.value);

	printf("\t[--%s <scale factor>=%f]\n",Scale.name,Scale.value);
	printf("\t\t Specifies the factor of the bounding cube that the input\n");
	printf("\t\t samples should fit into.\n");

	printf( "\t[--%s <fall-off exponent>=%f]\n" , GrazingFallOff.name , GrazingFallOff.value );
	printf( "\t\t Specifies the exponent used to modulate the fall-off at grazing angles\n" );

	printf( "\t[--%s]\n" , NoSolver.name );
	printf( "\t[--%s]\n" , ASCII.name );
	printf( "\t[--%s]\n" , Double.name );
	printf( "\t[--%s]\n" , Primal.name );
	printf( "\t[--%s]\n" , Verbose.name );
	printf( "\t[--%s]\n" , Progress.name );

	printf( "\t[--%s]\n" , VertexSamples.name );
	printf( "\t[--%s]\n" , NoTriangleSamples.name );
	printf( "\t[--%s]\n" , NoRefinedSampling.name );
}

template< class Real , class Vertex >
void GetRangeFunctions( const std::vector<Vertex>& vertices , const std::vector< TriangleIndex >& triangles , 
					    const Point3D<double>& center , double scale , double confidence ,
						const RangeGrid<PlyOrientedColorVertex,Real> *rGrids , int gridCount ,
						std::vector< LinearFunction3D< Point3D< Real > > >& functions )
{
	ProgressBar* bar=NULL;
	typedef typename RangeGrid<PlyOrientedColorVertex,Real>::Sample GridSample;
	GridSample* gridSamples = new GridSample[gridCount];
	GridSample vSamples[3];

	XForm4x4<Real> xForm		= XForm4x4<Real>::Identity();
	XForm4x4<Real>* xForms		= new XForm4x4<Real>	[gridCount];
	XForm4x4<Real>* iXForms		= new XForm4x4<Real>	[gridCount];
	Point3D<Real>* centers		= new Point3D<Real>		[gridCount];
	// Transformations:
	// 1] Unit-cube to 3D global:					3D -> 3D	xForm
	// 2] Unit-cube to camera coordinates:			3D -> 3D	iXForms		C
	// 3] Camera coordinates to scan grid:			3D -> 2D	 xforms		D
	// 6] Camera centers in 3D global:				3D			centers

	for(int c=0;c<3;c++)
	{
		xForm(3,c) = Real( center[c] );
		xForm(c,c) = Real( scale );
	}

	if( Progress.set ) bar = new ProgressBar( 20 , gridCount , "Transforming Grids" );
	for( int i=0 ; i<gridCount ; i++ )
	{
		if( Progress.set ) bar->update();
		iXForms[i]	= rGrids[i].xForm.inverse()*xForm;
		xForms[i]	= rGrids[i].backProjectionXForm( false );
		centers[i]	= rGrids[i].xForm( Point3D<Real>( ) );
	}
	if( Progress.set ) delete bar;

	functions.resize( triangles.size() );

	if( Progress.set ) bar = new ProgressBar( 20 , triangles.size() , "Setting Functions" );
	for( int i=0 ; i<triangles.size() ; i++ )
	{
		if( Progress.set ) bar->update( );
		functions[i] *= 0;
		// Get the index of the grid closest to the triangle center
		Point3D< Real > verts[] = { Point3D< Real >( vertices[ triangles[i][0] ] ) , Point3D< Real >( vertices[ triangles[i][1] ] ) , Point3D< Real >( vertices[ triangles[i][2] ] ) };
		Point3D< double > dVerts[] = { Point3D< double >( vertices[ triangles[i][0] ] ) , Point3D< double >( vertices[ triangles[i][1] ] ) , Point3D< double >( vertices[ triangles[i][2] ] ) };
		Point3D< Real > tCenter = ( verts[0] + verts[1] + verts[2] ) / 3;
		int idx = -1;
		{
			Point3D< Real > p = xForm( tCenter );							// Unit-cube -> 3D global
			for( int j=0 ; j<gridCount ; j++ )
			{
				Point3D< Real > q = xForms[j]( iXForms[j]( tCenter ) );		// Unit-cube -> grid
				double x = q[0] / q[2];
				double y = q[1] / q[2];
				gridSamples[j] = rGrids[j].sample( x , y );					// Sample the scan at the projected position (in 3D global)
			}
			double dist = 0;
			for( int j=0 ; j<gridCount ; j++ )	// Iterate over the valid back-projected samples
												// in all the scans and find the closest one
				if( gridSamples[j].validSample )
				{
					Real temp = Point3D< Real >::SquareDistance( p , gridSamples[j].v.point );
					double len = sqrt( Point3D<Real>::SquareNorm( centers[j]-Point<Real, 3>(p) ) * Point3D<Real>::SquareNorm( gridSamples[j].v.normal ) );
					double dot = Point3D<Real>::Dot( centers[j]-Point<Real, 3>(p) ,gridSamples[j].v.normal ) / len;
					if( confidence>0 && dot>0 ) temp /= Real( pow( dot , confidence ) );
					if( idx==-1 || temp<dist ) idx = j , dist = temp;
				}
		}

		if( idx>=0 )
		{
			bool validVertices = true;
			for( int j=0 ; j<3 ; j++ )
			{
				Point3D< Real > p = xForm( verts[j] );							// Unit-cube -> 3D global
				Point3D< Real > q = xForms[idx]( iXForms[idx]( verts[j] ) );	// Unit-cube -> grid
				double x = q[0] / q[2];
				double y = q[1] / q[2];
				vSamples[j] = rGrids[idx].sample( x , y );			// Sample the scan at the projected position (in 3D global)
				validVertices &= vSamples[j].validSample;
			}
#if PAN_FIX
			if( !validVertices ) functions[i].offset = -Point3D< Real >( gridSamples[idx].v.color );
#else // !PAN_FIX
			if( !validVertices ) functions[i].offset =  Point3D< Real >( gridSamples[idx].v.color );
#endif // PAN_FIX
			else
			{
				Point3D< double > values[3];
				for( int j=0 ; j<3 ; j++ ) values[j] = Point3D< double >( vSamples[j].v.color );
				functions[i] = LinearFunction3D< Point3D< Real > >( LinearFunction3D< Point3D< double > >::GetInterpolant( dVerts , values ) );
				for( int j=0 ; j<3 ; j++ ) for( int k=0 ; k<3 ; k++ ) if( functions[i].gradients[j][k]!=functions[i].gradients[j][k] )
				{
					printf( "Bad gradients[%d]...\n", i );
					functions[i].gradients *= 0;
#if PAN_FIX
					functions[i].offset = -Point3D< Real >( values[0] + values[1] + values[2] ) / 3;
#else // !PAN_FIX
					functions[i].offset =  Point3D< Real >( values[0] + values[1] + values[2] ) / 3;
#endif // PAN_FIX
				}
			}
		}
	}
	if( Progress.set ) delete bar;
	delete[] gridSamples;
	delete[] xForms;
	delete[] iXForms;
	delete[] centers;
}


// Returns the closes face on the mesh when the nearest vertex is known
template< class Vertex >
static int closest_point_on_face( 
								 const KDMesh< Vertex > &m ,		// The Mesh
								 int vert ,							// Index of the nearest point to pin
								 const Point3D<float> &pIn ,		// The point of interest
								 Vertex &vOut ,						// The point on the triangle that is closest
								 float &a0 , float &a1 , float &a2	// The barycentric coordinates of the point on the face
								 )
{
	if( vert<0 ) return -1;
	int f = -1;
	float dst = 0;
	bool found_inside = false;
	float a[3];
	m.need_adjacent_faces();

	for( size_t i=0; i<m.af[vert].size() ; i++ )			// Iterate over all faces adjacent to the nearest vertex
	{
		const TriangleIndex &face = m.f[m.af[vert][i]];		// The current face of interest
		BarycentricCoordinates( pIn , m.v[ face[0] ].point , m.v[ face[1] ].point , m.v[ face[2] ].point , a[0] , a[1] , a[2] );

		float sum = 0;
		for( int d=0 ; d<3 ; d++ )
		{
			if( a[d]<0 ) a[d] = 0;
			sum += a[d];
		}
		for( int d=0 ; d<3 ; d++ ) a[d] /= sum;
		Vertex v = m.v[face[0]] * a[0] + m.v[face[1]] * a[1] +  m.v[face[2]] * a[2];
		float dist = Point3D<float>::SquareNorm( Point3D< float >( v ) - pIn);
		if (f < 0 || dist < dst )
		{
			f = (int) m.af[vert][i];
			dst = dist;
			vOut = v;
			a0 = a[0] , a1 = a[1] , a2 = a[2];
		}
	}
	return f;
}
template< class Vertex >
static int closest_point_on_face( 
								 const KDMesh< Vertex > &m ,		// The Mesh
								 int vert ,							// Index of the nearest point to pin
								 const Point3D<float> &pIn ,		// The point of interest
								 float &a0 , float &a1 , float &a2	// The barycentric coordinates of the point on the face
								 )
{
	Vertex vOut;
	return closest_point_on_face( m , vert , pIn , vOut , a0 , a1 , a2 );
}

// Returns the closes face on the mesh when the nearest vertex is unknown
template< class Vertex >
static int closest_point_on_face( 
								 const KDMesh< Vertex > &m ,		// The Mesh
								 const Point3D<float> &pIn ,		// The point of interest
								 Vertex &vOut ,						// The point on the triangle that is closest
								 float &a0 , float &a1 , float &a2	// The barycentric coordinates of the point on the face
						)
{
	return closest_point_on_face( m , m.closest_vertex(pIn) , pIn , vOut , a0 , a1 , a2 );
}
template< class Vertex >
static int closest_point_on_face( 
								 const KDMesh< Vertex > &m ,		// The Mesh
								 const Point3D<float> &pIn ,		// The point of interest
								 float &a0 , float &a1 , float &a2	// The barycentric coordinates of the point on the face
						)
{
	Vertex vOut;
	return closest_point_on_face( m , pIn , vOut , a0 , a1 , a2 );
}
template< class Vertex >
struct MeshSample
{
	int mesh , f;
	Vertex v;
	float a0 , a1 , a2;
};
template< class Real , class Vertex , class MeshVertex >
void GetMeshFunctions(
					  const std::vector< Vertex >& vertices , const std::vector< TriangleIndex >& triangles , 
					  const Point3D< double >& center , double scale , double confidence ,
					  const KDMesh< MeshVertex >* rMeshes , int gridCount ,
					  std::vector< LinearFunction3D< Point3D< Real > > >& functions )
{
	MeshSample<MeshVertex>* meshSamples = new MeshSample<MeshVertex>[gridCount];
	MeshSample<MeshVertex> vSamples[3];

	XForm4x4< Real > xForm = XForm4x4< Real >::Identity();

	for( int c=0 ; c<3 ; c++ )
	{
		xForm( 3 , c ) = Real( center[c] );
		xForm( c , c ) = Real( scale );
	}
	functions.resize( triangles.size() );

	ProgressBar* bar=NULL;
	if( Progress.set ) bar = new ProgressBar( 20 , triangles.size() , "Setting Functions" );
	for( int t=0 ; t<triangles.size() ; t++ )
	{
		if( Progress.set ) bar->update( );
		functions[t] *= 0;
		// Get the index of the grid closest to the triangle center
		Point3D< Real > verts[]    = { Point3D< Real   >( vertices[ triangles[t][0] ] ) , Point3D< Real   >( vertices[ triangles[t][1] ] ) , Point3D< Real   >( vertices[ triangles[t][2] ] ) };
		Point3D< double > dVerts[] = { Point3D< double >( vertices[ triangles[t][0] ] ) , Point3D< double >( vertices[ triangles[t][1] ] ) , Point3D< double >( vertices[ triangles[t][2] ] ) };

		Point3D< Real > tCenter = ( verts[0] + verts[1] + verts[2] ) / 3;	// The center of the triangle , defining the mesh of interest
		int idx = -1;
		{
			Point3D< Real > p = xForm( tCenter );
			// Find the closest point on each mesh to the center of the triangle
			for( int g=0 ; g<gridCount ; g++ )
				meshSamples[g].f = closest_point_on_face( rMeshes[g] , Point3D< float >( p ) , meshSamples[g].v , meshSamples[g].a0 , meshSamples[g].a1 , meshSamples[g].a2 );
			double dist = 0;
			for( int g=0 ; g<gridCount ; g++ )	// Iterate over the valid back-projected samples
												// in all the scans and find the closest one
				if( meshSamples[g].f!=-1 )
				{
					Real temp = Point3D< Real >::SquareDistance( p , meshSamples[g].v.point );
					Point3D< float > n = Point3D< float >( rMeshes[g].normal( meshSamples[g].f ) );
					double len = sqrt( Point3D<Real>::SquareNorm( rMeshes[g].DoP ) * Point3D<Real>::SquareNorm( n ) );
					double dot = Point3D<Real>::Dot( rMeshes[g].DoP , n ) / len;
					if( confidence>0 && dot>0 ) temp /= Real( pow( dot , confidence ) );
					if( idx==-1 || temp<dist ) idx = g , dist = temp;
				}
		}
		if( idx>=0 )
		{
			bool validVertices = true;
			int f = meshSamples[idx].f;
			for( int v=0 ; v<3 ; v++ )
			{
				Point3D< Real > p = xForm( verts[v] );
				vSamples[v].f = closest_point_on_face( rMeshes[idx] , Point3D< float >( p ) , vSamples[v].v , vSamples[v].a0 , vSamples[v].a1 , vSamples[v].a2 );
if( vSamples[v].a0!=vSamples[v].a0 || vSamples[v].a1!=vSamples[v].a1 || vSamples[v].a2!=vSamples[v].a2 )
{
	printf( "Bad barycentric coordinates: %f %f %f\n" , vSamples[v].a0 , vSamples[v].a1 , vSamples[v].a2 );
	vSamples[v].a0 = vSamples[v].a1 = vSamples[v].a2 = 1./3;
}

				if( vSamples[v].f==-1 ) validVertices=false;
			}
#if PAN_FIX
			if( !validVertices ) functions[t].offset = -Point3D< Real >( meshSamples[idx].v.color );
#else // !PAN_FIX
			if( !validVertices ) functions[t].offset =  Point3D< Real >( meshSamples[idx].v.color );
#endif // PAN_FIX
			else
			{
				Point3D< double > values[3];
				for( int v=0 ; v<3 ; v++ ) values[v] = Point3D< double >( vSamples[v].v.color );
				functions[t] = LinearFunction3D< Point3D< Real > >( LinearFunction3D< Point3D< double > >::GetInterpolant( dVerts , values ) );
				for( int j=0 ; j<3 ; j++ ) for( int k=0 ; k<3 ; k++ ) if( functions[t].gradients[j][k]!=functions[t].gradients[j][k] )
				{
					printf( "Bad gradients[%d]...\n", t );
					printf( "\t[%d %d %d] %f %f %f\t%f %f %f\n" , idx , vSamples[0].mesh , vSamples[0].f , dVerts[0][0] , dVerts[0][1] , dVerts[0][2] , values[0][0] , values[0][1] , values[0][2] );
					printf( "\t[%d %d %d] %f %f %f\t%f %f %f\n" , idx , vSamples[0].mesh , vSamples[1].f , dVerts[1][0] , dVerts[1][1] , dVerts[1][2] , values[1][0] , values[1][1] , values[1][2] );
					printf( "\t[%d %d %d] %f %f %f\t%f %f %f\n" , idx , vSamples[0].mesh , vSamples[2].f , dVerts[2][0] , dVerts[2][1] , dVerts[2][2] , values[2][0] , values[2][1] , values[2][2] );

					functions[t].gradients *= 0;
#if PAN_FIX
					functions[t].offset = -Point3D< Real >( values[0] + values[1] + values[2] ) / 3;
#else // !PAN_FIX
					functions[t].offset =  Point3D< Real >( values[0] + values[1] + values[2] ) / 3;
#endif // PAN_FIX
				}
			}
		}
	}
	if( Progress.set ) delete bar;

	delete[] meshSamples;
}

template< class Real , class Vertex>
void GetFunctions( const std::vector< Vertex >& vertices , const std::vector< TriangleIndex >& triangles ,
				   int gridCount , char** gridNames , char** xFormNames , Point3D< double > center , double scale , double confidence ,
				   std::vector< LinearFunction3D< Point3D< Real > > >& functions )
{
	if( UseKD.set )
	{
		int fileType;
		KDMesh< PlyColorVertex >* rMeshes = new KDMesh< PlyColorVertex >[gridCount];
		ProgressBar* bar=NULL;
		if( Progress.set ) bar = new ProgressBar( 20 , gridCount , "Setting kD-tree" );
		for (int idx = 0 ; idx<gridCount ; idx++ )
		{
			if( Progress.set ) bar->update();
			PlyReadTriangles( gridNames[idx] , rMeshes[idx].v , rMeshes[idx].f , PlyColorVertex::ReadProperties , PlyColorVertex::ReadComponents , fileType );
printf( "%d -> " , rMeshes[idx].f.size() );
for( int i=rMeshes[idx].f.size()-1 ; i>=0 ; i-- )
{
	double area = TriangleArea( rMeshes[idx].v , rMeshes[idx].f[i] );
	if( area<1e-15 )
	{
//		fprintf( stderr , "Degenerate Triangle: %d in %s (%f)\n" , i , gridNames[idx] , area );
		rMeshes[idx].f[i] = rMeshes[idx].f.back();
		rMeshes[idx].f.pop_back();
	}
}
printf( "%d\n" , rMeshes[idx].f.size() );
			if ( xFormNames )
			{
				FILE* fp = fopen( xFormNames[idx] , "r" );
				XForm4x4< Real > xForm;
				if( fp )
				{
					for( int j = 0; j < 4; j++ )
						for(int i = 0; i < 4; i++)
						{
							float d;
							if( fscanf( fp , " %f " , &d )!=1 ) fprintf( stderr , "Failed to read matrix coefficient ( %d , %d ) from %s\n" , i , j , xFormNames[idx] ) , exit( 0 );
							xForm.coords[i][j] = d;
						}
					fclose(fp);
				}
				else
				{
					xForm = XForm4x4< Real >::Identity( );
					fprintf( stderr , "Failed to read xForm from %s\n" , xFormNames[idx] );
				}
				for( size_t i = 0; i < rMeshes[idx].v.size(); i++) rMeshes[idx].v[i] = rMeshes[idx].v[i].xForm( xForm );
			}
			rMeshes[idx].build_kd();
			rMeshes[idx].setDirectionOfProjection();
		}
		// WARNING: Have to add a function for computing the meshes 
		if( Progress.set ) delete bar;
 		GetMeshFunctions( vertices , triangles , center , scale , confidence , rMeshes , gridCount , functions );
		delete[] rMeshes;
	}
	else
	{
		RangeGrid<PlyOrientedColorVertex,Real> *rGrids = new RangeGrid< PlyOrientedColorVertex , Real >[gridCount];
		ProgressBar* bar=NULL;
		if( Progress.set ) bar = new ProgressBar( 20 , gridCount , "Reading Grids" );
		for(int i=0;i<gridCount;i++)
		{
			if( Progress.set ) bar->update();
			if( xFormNames ) rGrids[i].Read( gridNames[i] , xFormNames[i] );
			else			 rGrids[i].Read( gridNames[i] );
		}
		if( Progress.set ) delete bar;
		GetRangeFunctions( vertices , triangles , center , scale , confidence , rGrids , gridCount , functions );
		delete[] rGrids;
	}
}
template< class Real , bool Primal >
int Execute(int argc,char* argv[])
{
	int paramNum=sizeof(params)/sizeof(cmdLineReadable*);
	if( FullVerbose.set )	Verbose.set=true;
	int verbose = 0;
	if( FullVerbose.set )	verbose |= MeshOctree< Real , Primal >::FULL_VERBOSE;
	if( Verbose.set )		verbose |= MeshOctree< Real , Primal >::VERBOSE;

	int commentNum=0;
	char **comments;

	comments=new char*[paramNum+8];
	for(int i=0;i<paramNum+8;i++){comments[i]=new char[1024];}

	DumpOutput( Verbose.set , comments[commentNum++] , "Running Texture Stitcher\n" );
	char valueString[1024];
	for(int i=0;i<paramNum;i++)
		if(params[i]->set)
		{
			params[i]->writeValue(valueString);
			DumpOutput( Verbose.set , comments[commentNum++] , "\t--%s %s\n" , params[i]->name , valueString );
		}

	double t;
	double tt=Time();
	int fileType;
	Point3D<double> center;
	double scale;
	std::vector< PlyOrientedVertex > vertices;
	std::vector< TriangleIndex > triangles;
	MeshOctree< Real , Primal > tree;
	char **scanNames,**xFormNames=NULL;
	int scanCount,xFormCount=0;


	if( !In.set || !Out.set || !Scans.set )
	{
		ShowUsage(argv[0]);
		return 0;
	}
	VWeight.value *= 1<<( 2*Depth.value );

	scanNames=ReadWords( Scans.value , scanCount );
	if( !scanNames )
	{
		fprintf( stderr , "Failed to read grid names from: %s\n" , Scans.value );
		return EXIT_FAILURE;
	}
	if( XForms.set )
	{
		xFormNames = ReadWords( XForms.value , xFormCount );
		if( !xFormNames )
		{
			fprintf( stderr , "Failed to read xForm names from: %s\n" , XForms.value );
			return EXIT_FAILURE;
		}
		if( scanCount!=xFormCount )
		{
			fprintf( stderr , "Grid count and xForm count differ: %d != %d\n" , scanCount , xFormCount );
			return EXIT_FAILURE;
		}
	}

	// Read in the mesh
	{
		PlyReadTriangles( In.value , vertices , triangles , PlyVertex::ReadProperties , PlyVertex::ReadComponents , fileType );
		TrimHangingVertices( vertices , triangles );
		// Remove zero-area triangles
		for( int i=(int)triangles.size()-1 ; i>=0 ; i-- )
		{
			Point3D< double > normal = Point3D< double >::CrossProduct( vertices[triangles[i][1]]-vertices[triangles[i][0]],vertices[triangles[i][2]]-vertices[triangles[i][0]] );
			double len = sqrt( Point3D< double >::SquareNorm( normal ) );
			if( len==0 )
			{
				triangles[i] = triangles[triangles.size()-1];
				triangles.pop_back( );
			}
		}
		FitVertices< double , PlyOrientedVertex >( vertices , Scale.value , center , scale );
		// Set the normals
		SetVertexNormals( vertices , triangles );

		// It could also happen that the same triangle appears twice with opposite orientation, which could be very bad!
		// One manifestation of this is a triangle whose (normalized) vertex normal is undefined. We remove all such triangles.
		for( int i=0 ; i<vertices.size() ; i++ )
		{
			if( vertices[i].normal[0]!=vertices[i].normal[0] || vertices[i].normal[1]!=vertices[i].normal[1] || vertices[i].normal[2]!=vertices[i].normal[2] || !Point3D< float >::SquareNorm( vertices[i].normal ) )
			{
				for( int j=(int)triangles.size()-1 ; j>=0 ; j-- )
					if( triangles[j][0]==i || triangles[j][1]==i || triangles[j][2]==i )
					{
						triangles[j] = triangles[triangles.size()-1];
						triangles.pop_back();
					}
			}
		}
		for( int s=0 ; s<Subdivide.value ; s++ ) SubdivideTriangle( vertices , triangles );
	}
	// Not quite sure how this could happen after sub-division if the original triangle was non-singular...
	for( int i=(int)triangles.size()-1 ; i>=0 ; i-- )
	{
		Point3D< double > normal = Point3D< double >::CrossProduct( vertices[triangles[i][1]]-vertices[triangles[i][0]],vertices[triangles[i][2]]-vertices[triangles[i][0]] );
		double len = Length( normal );
		if( len==0 )
		{
			triangles[i] = triangles[triangles.size()-1];
			triangles.pop_back( );
		}
	}
	// Set up the octree
	{
		t=Time();
		if( MonteCarlo.value>0 ) 
		{
			std::vector< OrientedPoint3D< Real > > samples;
			std::vector< int > indices;
			bool useVertSamples = VertexSamples.set , useTriSamples = !NoTriangleSamples.set;
			int count , vCount=0 , tCount=0;
			std::vector< OrientedPoint3D< Real > > subSamples;
			std::vector< int > subIndices , vIndices;
			Real alpha , vAlpha , tAlpha;
			std::vector< Real > areas;

			vIndices.resize( vertices.size() );
			for( int i=0 ; i<triangles.size() ; i++ ) for( int c=0 ; c<3 ; c++ ) vIndices[ triangles[i][c] ] = i;

			if( useVertSamples ) vCount = (int)vertices.size();
			if( useTriSamples )
			{
				if( NoRefinedSampling.set ) tCount = (int)triangles.size();
				else                        tCount = (int)triangles.size() * 4;
			}
			if( MonteCarlo.value<vCount+tCount) count = 0;
			else						        count = MonteCarlo.value - vCount - tCount;
			indices.resize( vCount + tCount + count );
			samples.resize( vCount + tCount + count );
			if( useVertSamples )
			{
				vCount = 0;
				SetVertexAreas( vertices, triangles , areas );
				for( int i=0 ; i<vertices.size() ; i++ )
				{
					double l = Length( Point3D< double >( vertices[i].normal ) );
					if( l!=l ) continue;
					samples[ vCount ].position = Point3D< Real >( vertices[i].point  );
					samples[ vCount ].normal   = Point3D< Real >( vertices[i].normal ) * areas[i];
					indices[ vCount ] = vIndices[ i ];
					vCount++;
				}
			}
			if( useTriSamples )
			{
				tCount = 0;
				for( int i=0 ; i<triangles.size() ; i++ )
				{
					Point3D< double > verts[] = { Point3D< double >( vertices[ triangles[i][0] ] ) , Point3D< double >( vertices[ triangles[i][1] ] ) , Point3D< double >( vertices[ triangles[i][2] ] ) };
					Point3D< double > normal = Point3D< double >::CrossProduct( verts[1] - verts[0] , verts[2] - verts[0] );
					double l = Length( Point3D< double >( normal ) );
					if( l!=l ) continue;
					samples[ vCount+tCount ].position = Point3D< Real >( ( verts[0] + verts[1] + verts[2] ) / 3 );
					samples[ vCount+tCount ].normal   = Point3D< Real >( normal );
					indices[ vCount+tCount ] = i;
					tCount++;
					if( !NoRefinedSampling.set )
					{
						samples[ vCount+tCount ].position = Point3D< Real >( ( 4*verts[0] + verts[1] + verts[2] ) / 6 );
						samples[ vCount+tCount ].normal   = Point3D< Real >( normal );
						indices[ vCount+tCount ] = i;
						tCount++;
						samples[ vCount+tCount ].position = Point3D< Real >( ( verts[0] + 4*verts[1] + verts[2] ) / 6 );
						samples[ vCount+tCount ].normal   = Point3D< Real >( normal );
						indices[ vCount+tCount ] = i;
						tCount++;
						samples[ vCount+tCount ].position = Point3D< Real >( ( verts[0] + verts[1] + 4*verts[2] ) / 6 );
						samples[ vCount+tCount ].normal   = Point3D< Real >( normal );
						indices[ vCount+tCount ] = i;
						tCount++;
					}
				}
			}
			if( MonteCarlo.value<vCount+tCount ) count = 0;
			else                                 count = MonteCarlo.value - vCount - tCount;
			indices.resize( vCount + tCount + count );
			samples.resize( vCount + tCount + count );


			if( count )
			{
				SampleTriangles< Real , PlyOrientedVertex >( vertices , triangles, count , subSamples , subIndices );
				for( int i=0 ; i<count ; i++ )
				{
					samples[vCount+tCount+i].position = subSamples[ i ].position;
					samples[vCount+tCount+i].normal   = subSamples[ i ].normal;
					indices[vCount+tCount+i] = subIndices[ i ];
				}
			}
			alpha  = Real(  count ) / ( vCount + tCount + count );
			vAlpha = Real( vCount ) / ( vCount + tCount + count );
			tAlpha = Real( tCount ) / ( vCount + tCount + count );

			for( int i=0 ; i<vCount ; i++ ) samples[              i].normal *= vAlpha;
			for( int i=0 ; i<tCount ; i++ ) samples[       vCount+i].normal *= tAlpha;
			for( int i=0 ; i< count ; i++ ) samples[tCount+vCount+i].normal *=  alpha;
			if( Verbose.set ) printf( "Monte-Carlo samples used: %d\n" , count + tCount + vCount );
			tree.setTree( samples , indices , Depth.value , CleanTree.value , Progress.set );
		}
		else tree.setTree( vertices , triangles , Depth.value , CleanTree.value , Progress.set );
		DumpOutput( Verbose.set , comments[commentNum++],"#             Tree set in: %9.1f (s), %9.1f (MB)\n" , Time()-t , tree.maxMemoryUsage );
		if( Verbose.set )
		{
			printf( "Dimension/Leaves/Nodes: %d / %d / %d\n" , tree.dimension(Depth.value) , tree.tree.leaves() , tree.tree.nodes() );
			printf( "    Vertices/Triangles: %d / %d\n"  , (int)vertices.size() , (int)triangles.size() );
			if( Primal ) printf( "            Tree Depth: %d\n" , tree.tree.maxDepth()-1 );
			else         printf( "            Tree Depth: %d\n" , tree.tree.maxDepth()   );
		}
	}

	std::vector<PlyColorVertex> outVertices;

	if( NoSolver.set )
	{
		std::vector< LinearFunction3D< Point3D< Real > > > functions;
		GetFunctions( vertices , triangles , scanCount , scanNames , xFormNames , center , scale , GrazingFallOff.value , functions );
		outVertices.resize( vertices.size() );
		for( int i=0 ; i<vertices.size() ; i++ ) outVertices[i].point = vertices[i].point*float( scale ) + Point3D< float >( center );
		for( int i=0 ; i<triangles.size() ; i++ )
			for( int j=0 ; j<3 ; j++ )
				outVertices[ triangles[i][j] ].color = Point3D< float >( functions[i]( vertices[ triangles[i][j] ].point ) );
	}
	else
	{
		Point3D< Real > inAverage;
		std::vector< Point3D< Real > > X;
		{
			Vector< Point3D< Real > > _X , _B;
			double bTime = Time();
			{
				std::vector< LinearFunction3D< Point3D< Real > > > functions;
				GetFunctions( vertices , triangles , scanCount , scanNames , xFormNames , center , scale , GrazingFallOff.value , functions );

				for( int i=0 ; i<triangles.size() ; i++ )
				{
					Point3D< Real > normal = Point3D< Real >::CrossProduct( vertices[triangles[i][1]]-vertices[triangles[i][0]] , vertices[triangles[i][2]]-vertices[triangles[i][0]] );
					functions[i] = functions[i].fitToHyperplane( vertices[triangles[i][0]].point , normal );
				}

				inAverage = tree.template getPreciseIntegral< Point3D< Real > , Point3D< double > >( functions ) / tree.getArea();
				std::vector< Point3D< Real > > B;
				if( !GWeight.value )
					tree.template getPreciseDotVector< Point3D< Real > , Point3D< double > , LinearFunction3D< Point3D< Real > > >( functions , Depth.value , B , Progress.set );
				else if( !VWeight.value )
				{
					std::vector< ConstantFunction< Gradient3D< Point3D< Real > > > > gradients;
					gradients.resize( functions.size() );
					for( int i=0 ; i<functions.size() ; i++ ) gradients[i].value = functions[i].gradients;
					tree.template getPreciseGradientDotGradientVector< Point3D< Real > , Point3D< double > , ConstantFunction< Gradient3D< Point3D< Real > > > >( gradients , Depth.value , B , Progress.set , false );
				}
				else
					tree.template getPreciseConstrainedGradientDotGradientVector< Point3D< Real > , Point3D< double > , LinearFunction3D< Point3D< Real > > >( functions , VWeight.value , GWeight.value , Depth.value , B , Progress.set , false );
				_B.Resize( B.size() );
				for( int i=0 ; i<B.size() ; i++ ) _B[i] = B[i];
			}
			bTime = Time()-bTime;
			double bMem = tree.MemoryUsage();

			SparseMatrix< Real > M;
			double mTime = Time();
			if		( !GWeight.value )	tree.getDotProductMatrix( Depth.value , M , Progress.set );
			else if ( !VWeight.value )	tree.getLaplacianMatrix ( Depth.value , M , Progress.set );
			else						tree.getConstrainedLaplacianMatrix( Depth.value , M , VWeight.value , GWeight.value , Progress.set );
			mTime = Time()-mTime;
			double mMem = tree.MemoryUsage();

			double sTime = Time();
			double normB , normR , normX , vectorSetTime , matrixSetTime , solveTime;
			tree.solveLinearSystem( M , _B , _X , 0 , Iters.value , Cycles.value , 1.f , false , verbose , false , false , normB , normR , normX , vectorSetTime , matrixSetTime , solveTime , IMultiplier.value );
			sTime = Time()-sTime;
			double sMem = tree.MemoryUsage();
			if( Verbose.set )
			{
				printf("Set initial constraints in: %9.1f (s), %9.1f (MB)\n" , bTime , bMem );
				printf("     Set initial matrix in: %9.1f (s), %9.1f (MB), %d entries\n" , mTime , mMem , M.Entries() );
			}
			DumpOutput( Verbose.set , comments[commentNum++] , "Solver Time (vector set / matrix set / solve):\t%f / %f / %f\n" , vectorSetTime , matrixSetTime , solveTime );
			DumpOutput( Verbose.set , comments[commentNum++] , "Solver Error:\t%f -> %f = %f\n" , sqrt(normB) , sqrt(normR) , sqrt( normR / normB ) );

			X.resize( _X.Dimensions() );
			for( int i=0 ; i<_X.Dimensions() ; i++ ) X[i] = _X[i];
			DumpOutput( Verbose.set , comments[commentNum++] , "#     Solved system in: %9.1f (s), %9.1f (MB)\n" , sTime , sMem );

			std::vector< Point3D< Real > > values;
			outVertices.resize( vertices.size() );
			for( int i=0 ; i<vertices.size() ; i++ ) outVertices[i].point = vertices[i].point;
			tree.getValues( outVertices , X , values );
			Point3D< Real > outAverage = tree.template getPreciseCoefficientIntegral< Point3D< Real > , Point3D< double > >( X , Depth.value ) / tree.getArea();

			Point3D< Real > offset;
			if( !VWeight.value ) offset = inAverage - outAverage;
			else offset *= 0;

			for( int i=0 ; i<outVertices.size() ; i++ )
			{
				outVertices[i].color = Point3D<float>( values[i] + offset );
				outVertices[i].point = outVertices[i].point*float( scale ) + Point3D< float >( center );
				for( int c=0 ; c<3 ; c++ )
					if		( outVertices[i].color[c]<0   )	outVertices[i].color[c] =   0;
					else if ( outVertices[i].color[c]>255 )	outVertices[i].color[c] = 255;
			}
		}
	}
	if( ASCII.set ) fileType = PLY_ASCII;
	PlyWriteTriangles( Out.value , outVertices , triangles , PlyColorVertex::WriteProperties , PlyColorVertex::WriteComponents , fileType , comments , commentNum );
	return 1;
}

int main( int argc , char* argv[] )
{
	srand( 0 );
	int paramNum = sizeof( params ) / sizeof( cmdLineReadable* );
	cmdLineParse( argc-1 , &argv[1] , paramNum , params , 0 );

	if( Primal.set )
		if( Double.set )	Execute< double , true >( argc , argv );
		else				Execute< float  , true >( argc , argv );
	else
		if( Double.set )	Execute< double , false >( argc , argv );
		else				Execute< float  , false >( argc , argv );
	return EXIT_SUCCESS;
 }