/*

 Header for PLY polygon files.
 
  - Greg Turk, March 1994
  
   A PLY file contains a single polygonal _object_.
   
	An object is composed of lists of _elements_.  Typical elements are
	vertices, faces, edges and materials.
	
	 Each type of element for a given object has one or more _properties_
	 associated with the element type.  For instance, a vertex element may
	 have as properties three floating-point values x,y,z and three unsigned
	 chars for red, green and blue.
	 
	  ---------------------------------------------------------------
	  
	   Copyright (c) 1994 The Board of Trustees of The Leland Stanford
	   Junior University.  All rights reserved.   
	   
		Permission to use, copy, modify and distribute this software and its   
		documentation for any purpose is hereby granted without fee, provided   
		that the above copyright notice and this permission notice appear in   
		all copies of this software and that you do not sell the software.   
		
		 THE SOFTWARE IS PROVIDED "AS IS" AND WITHOUT WARRANTY OF ANY KIND,   
		 EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY   
		 WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.   
		 
*/

#ifndef PLY_INCLUDED
#define PLY_INCLUDED
#include <vector>
#include "Geometry.h"
#include "PlyFile.h"
#include "Grid.h"
#include "Algebra.h"

class PlyVertex : public VectorSpace< float , PlyVertex >
{
public:
	//////////////////////////
	// Vector space methods //
	void Add	(const PlyVertex& p)	{	point+=p.point;	}
	void Scale	(float s)				{	point*=s;		}
	//////////////////////////

	const static int ReadComponents=3;
	const static int WriteComponents=3;
	static PlyProperty ReadProperties[];
	static PlyProperty WriteProperties[];

	Point3D<float> point;

	operator Point3D< float >& ()				{ return point; }
	operator const Point3D<float>& () const		{ return point; }
	template< class Real > operator Point3D< Real > ( ) const { return Point3D< Real >( point ); }
	PlyVertex(void)								{point.coords[0]=point.coords[1]=point.coords[2]=0;}
	PlyVertex(const Point3D<float>& p)			{point=p;}

	template<class Real>
	PlyVertex xForm(const XForm4x4<Real>& xForm) const
	{
		PlyVertex v;
		v.point  = xForm ( point );
		return v;
	}
};
PlyProperty PlyVertex::ReadProperties[]=
{
	{"x", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyVertex,point.coords[0])), 0, 0, 0, 0},
	{"y", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyVertex,point.coords[1])), 0, 0, 0, 0},
	{"z", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyVertex,point.coords[2])), 0, 0, 0, 0}
};
PlyProperty PlyVertex::WriteProperties[]=
{
	{"x", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyVertex,point.coords[0])), 0, 0, 0, 0},
	{"y", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyVertex,point.coords[1])), 0, 0, 0, 0},
	{"z", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyVertex,point.coords[2])), 0, 0, 0, 0}
};
class PlyValueVertex : public VectorSpace< float , PlyValueVertex >
{
public:
	//////////////////////////
	// Vector space methods //
	void Add	(const PlyValueVertex& p)	{	point+=p.point;	value+=p.value;	}
	void Scale	(float s)					{	point*=s;		value*=s;		}
	//////////////////////////
	const static int ReadComponents=4;
	const static int WriteComponents=4;
	static PlyProperty ReadProperties[];
	static PlyProperty WriteProperties[];

	Point3D<float> point;
	float value;

	operator Point3D<float>& ()				{return point;}
	operator const Point3D<float>& () const	{return point;}
	template< class Real > operator Point3D< Real > ( ) const { return Point3D< Real >( point ); }
	PlyValueVertex(void)					{point.coords[0]=point.coords[1]=point.coords[2]=0;value=0;}
	PlyValueVertex(const Point3D<float>& p)	{point=p;}

	template<class Real>
	PlyValueVertex xForm(const XForm4x4<Real>& xForm) const
	{
		PlyValueVertex v;
		v.value=value;
		v.point  = xForm ( point );
		return v;
	}
};
PlyProperty PlyValueVertex::ReadProperties[]=
{
	{"x", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyValueVertex,point.coords[0])), 0, 0, 0, 0},
	{"y", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyValueVertex,point.coords[1])), 0, 0, 0, 0},
	{"z", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyValueVertex,point.coords[2])), 0, 0, 0, 0},
	{"value", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyValueVertex,value)), 0, 0, 0, 0}
};
PlyProperty PlyValueVertex::WriteProperties[]=
{
	{"x", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyValueVertex,point.coords[0])), 0, 0, 0, 0},
	{"y", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyValueVertex,point.coords[1])), 0, 0, 0, 0},
	{"z", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyValueVertex,point.coords[2])), 0, 0, 0, 0},
	{"value", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyValueVertex,value)), 0, 0, 0, 0}
};

class PlyOrientedVertex : public VectorSpace< float , PlyOrientedVertex >
{
public:
	//////////////////////////
	// Vector space methods //
	void Add	(const PlyOrientedVertex& p)	{	point+=p.point;	normal+=p.normal;	}
	void Scale	(float s)						{	point*=s;		normal*=s;			}
	//////////////////////////
	const static int ReadComponents=6;
	const static int WriteComponents=6;
	static PlyProperty ReadProperties[];
	static PlyProperty WriteProperties[];

	Point3D<float> point,normal;

	operator Point3D<float>& ()					{return point;}
	operator const Point3D<float>& () const		{return point;}
	template< class Real > operator Point3D< Real > ( ) const { return Point3D< Real >( point ); }
	PlyOrientedVertex(void)						{point.coords[0]=point.coords[1]=point.coords[2]=normal.coords[0]=normal.coords[1]=normal.coords[2]=0;}
	PlyOrientedVertex(const Point3D<float>& p)	{point=p;}

	template<class Real>
	PlyOrientedVertex xForm(const XForm4x4<Real>& xForm) const
	{
		XForm3x3<Real> temp=Matrix<Real,3,3>(xForm);
		PlyOrientedVertex v;
		v.point  = xForm ( point );
		v.normal = temp * normal;
		return v;
	}
};
PlyProperty PlyOrientedVertex::ReadProperties[]=
{
	{"x", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedVertex,point.coords[0])), 0, 0, 0, 0},
	{"y", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedVertex,point.coords[1])), 0, 0, 0, 0},
	{"z", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedVertex,point.coords[2])), 0, 0, 0, 0},
	{"nx", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedVertex,normal.coords[0])), 0, 0, 0, 0},
	{"ny", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedVertex,normal.coords[1])), 0, 0, 0, 0},
	{"nz", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedVertex,normal.coords[2])), 0, 0, 0, 0}
};
PlyProperty PlyOrientedVertex::WriteProperties[]=
{
	{"x", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedVertex,point.coords[0])), 0, 0, 0, 0},
	{"y", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedVertex,point.coords[1])), 0, 0, 0, 0},
	{"z", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedVertex,point.coords[2])), 0, 0, 0, 0},
	{"nx", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedVertex,normal.coords[0])), 0, 0, 0, 0},
	{"ny", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedVertex,normal.coords[1])), 0, 0, 0, 0},
	{"nz", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedVertex,normal.coords[2])), 0, 0, 0, 0}
};

class PlyColorVertex : public VectorSpace< float , PlyColorVertex >
{
public:
	//////////////////////////
	// Vector space methods //
	void Add	(const PlyColorVertex& p)	{	point+=p.point;	color+=p.color;	}
	void Scale	(float s)					{	point*=s;		color*=s;		}
	//////////////////////////
	const static int ReadComponents=9;
	const static int WriteComponents=6;
	static PlyProperty ReadProperties[];
	static PlyProperty WriteProperties[];

	Point3D<float> point,color;

	operator Point3D<float>& ()					{return point;}
	operator const Point3D<float>& () const		{return point;}
	template< class Real > operator Point3D< Real > ( ) const { return Point3D< Real >( point ); }
	PlyColorVertex(void)						{point.coords[0]=point.coords[1]=point.coords[2]=0,color.coords[0]=color.coords[1]=color.coords[2]=0;}
	PlyColorVertex(const Point3D<float>& p)		{point=p;}

	template<class Real>
	PlyColorVertex xForm(const XForm4x4<Real>& xForm) const
	{
		PlyColorVertex v;
		v.color.coords[0]=color.coords[0];
		v.color.coords[1]=color.coords[1];
		v.color.coords[2]=color.coords[2];
		v.point  = xForm ( point );
		return v;
	}
};
PlyProperty PlyColorVertex::ReadProperties[]=
{
	{"x", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyColorVertex,point.coords[0])), 0, 0, 0, 0},
	{"y", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyColorVertex,point.coords[1])), 0, 0, 0, 0},
	{"z", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyColorVertex,point.coords[2])), 0, 0, 0, 0},
	{"red",		PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyColorVertex,color.coords[0])),0, 0, 0, 0},
	{"green",	PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyColorVertex,color.coords[1])),0, 0, 0, 0},
	{"blue",	PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyColorVertex,color.coords[2])),0, 0, 0, 0},
	{"diffuse_red",		PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyColorVertex,color.coords[0])),0, 0, 0, 0},
	{"diffuse_green",	PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyColorVertex,color.coords[1])),0, 0, 0, 0},
	{"diffuse_blue",	PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyColorVertex,color.coords[2])),0, 0, 0, 0},
};
PlyProperty PlyColorVertex::WriteProperties[]=
{
	{"x", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyColorVertex,point.coords[0])), 0, 0, 0, 0},
	{"y", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyColorVertex,point.coords[1])), 0, 0, 0, 0},
	{"z", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyColorVertex,point.coords[2])), 0, 0, 0, 0},
	{"red",		PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyColorVertex,color.coords[0])),0, 0, 0, 0},
	{"green",	PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyColorVertex,color.coords[1])),0, 0, 0, 0},
	{"blue",	PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyColorVertex,color.coords[2])),0, 0, 0, 0}
};

class PlyOrientedColorVertex : public VectorSpace< float , PlyOrientedColorVertex >
{
public:
	//////////////////////////
	// Vector space methods //
	void Add	(const PlyOrientedColorVertex& p)	{	point+=p.point;	normal+=p.normal;	color+=p.color;	}
	void Scale	(float s)							{	point*=s;		normal*=s;			color*=s;		}
	//////////////////////////
	const static int ReadComponents=15;
	const static int WriteComponents=12;
	static PlyProperty ReadProperties[];
	static PlyProperty WriteProperties[];

	Point3D<float> point,normal,color;

	operator Point3D<float>& ()					{return point;}
	operator const Point3D<float>& () const		{return point;}
	template< class Real > operator Point3D< Real > ( ) const { return Point3D< Real >( point ); }
	PlyOrientedColorVertex(void)				{point.coords[0]=point.coords[1]=point.coords[2]=normal.coords[0]=normal.coords[1]=normal.coords[2]=0,color.coords[0]=color.coords[1]=color.coords[2]=0;}
	PlyOrientedColorVertex(const Point3D<float>& p)	{point=p;}

	template<class Real>
	PlyOrientedColorVertex xForm(const XForm4x4<Real>& xForm) const
	{
		XForm3x3<Real> temp=XForm3x3<Real>(Matrix<Real,3,3>(xForm));
		PlyOrientedColorVertex v;
		v.color.coords[0]=color.coords[0];
		v.color.coords[1]=color.coords[1];
		v.color.coords[2]=color.coords[2];
		v.point  = xForm ( point );
		v.normal = temp *normal;
		return v;
	}
};
PlyProperty PlyOrientedColorVertex::ReadProperties[]=
{
	{"x", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,point.coords[0])), 0, 0, 0, 0},
	{"y", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,point.coords[1])), 0, 0, 0, 0},
	{"z", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,point.coords[2])), 0, 0, 0, 0},
	{"nx", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,normal.coords[0])), 0, 0, 0, 0},
	{"ny", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,normal.coords[1])), 0, 0, 0, 0},
	{"nz", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,normal.coords[2])), 0, 0, 0, 0},
	{"red",		PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,color.coords[0])),	0, 0, 0, 0},
	{"green",	PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,color.coords[1])),	0, 0, 0, 0},
	{"blue",	PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,color.coords[2])),	0, 0, 0, 0},
	{"diffuse_red",		PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,color.coords[0])),	0, 0, 0, 0},
	{"diffuse_green",	PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,color.coords[1])),	0, 0, 0, 0},
	{"diffuse_blue",	PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,color.coords[2])),	0, 0, 0, 0}
};
PlyProperty PlyOrientedColorVertex::WriteProperties[]=
{
	{"x", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,point.coords[0])), 0, 0, 0, 0},
	{"y", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,point.coords[1])), 0, 0, 0, 0},
	{"z", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,point.coords[2])), 0, 0, 0, 0},
	{"nx", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,normal.coords[0])), 0, 0, 0, 0},
	{"ny", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,normal.coords[1])), 0, 0, 0, 0},
	{"nz", PLY_FLOAT, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,normal.coords[2])), 0, 0, 0, 0},
	{"red",		PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,color.coords[0])),	0, 0, 0, 0},
	{"green",	PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,color.coords[1])),	0, 0, 0, 0},
	{"blue",	PLY_UCHAR, PLY_FLOAT, int(offsetof(PlyOrientedColorVertex,color.coords[2])),	0, 0, 0, 0},
};






int PlyDefaultFileType(void){return PLY_ASCII;}

template<class Vertex>
int PlyReadPoints(char* fileName,
				  std::vector<Vertex>& vertices,
				  PlyProperty* properties,int propertyNum,
				  int& file_type,
				  char*** comments=NULL,int* commentNum=NULL);
template<class Vertex>
int PlyWritePoints(char* fileName,
				   const std::vector<Vertex>& vertices,
				   PlyProperty* properties,int propertyNum,
				   int file_type,
				   char** comments=NULL,const int& commentNum=0);

template<class Vertex>
int PlyReadPolygons(char* fileName,
					std::vector<Vertex>& vertices,std::vector<std::vector<int> >& polygons,
					PlyProperty* properties,int propertyNum,
					int& file_type,
					char*** comments=NULL,int* commentNum=NULL);
template< class Vertex >
int PlyReadTriangles( char* fileName ,
					  std::vector<Vertex>& vertices , std::vector< TriangleIndex >& triangles ,
					  PlyProperty* properties , int propertyNum ,
					  int& file_type ,
					  char*** comments=NULL , int* commentNum=NULL );
template<class Vertex>
int PlyWritePolygons(char* fileName,
					 const std::vector<Vertex>& vertices,const std::vector<std::vector<int> >& polygons,
					 PlyProperty* properties,int propertyNum,
					 int file_type,
					 char** comments=NULL,const int& commentNum=0);
template<class Vertex>
int PlyWriteTriangles( char* fileName ,
					   const std::vector<Vertex>& vertices , const std::vector< TriangleIndex >& triangles ,
					   PlyProperty* properties , int propertyNum ,
					   int file_type ,
					   char** comments=NULL , const int& commentNum=0);

template<class Vertex>
int PlyReadGrid(char* fileName,
				std::vector<Vertex>& vertices,Grid<int>& grid,
				PlyProperty* properties,int propertyNum,
				int& file_type,
				char*** comments=NULL,int* commentNum=NULL);
#include "Ply.inl"
#endif // PLY_INCLUDED
