/***************************************************************************************************
 * File:         $URL: https://svn.bolitho.us/Library/Trunk/Src/Memory/Md5.cpp $
 * Author:       $Author: BOLITHO\matthew $
 * Revision:     $Rev: 1631 $
 * Last Updated: $Date: 2008-09-15 07:52:23 -0700 (Mon, 15 Sep 2008) $
 * 
 * 
 * Copyright (c) 2004-2007, Matthew G Bolitho
 * All rights reserved.
 *
 *
 * Microsoft Reference License (Ms-RL)
 *   
 * This license governs use of the accompanying software. If you use the software, you accept 
 * this license. If you do not accept the license, do not use the software.
 *
 * 1. Definitions
 *  - The terms "reproduce," "reproduction" and "distribution" have the same meaning here as under 
 *    U.S. copyright law.
 *  - "You" means the licensee of the software.
 *  - "Your company" means the company you worked for when you downloaded the software.
 *  - "Reference use" means use of the software within your company as a reference, in read only 
 *    form, for the sole purposes of debugging your products, maintaining your products, or 
 *    enhancing the interoperability of your products with the software, and specifically excludes 
 *    the right to distribute the software outside of your company.
 *  - "Licensed patents" means any Licensor patent claims which read directly on the software as 
 *    distributed by the Licensor under this license.
 * 
 * 2. Grant of Rights
 *  (A) Copyright Grant- Subject to the terms of this license, the Licensor grants you a non-transferable, 
 *      non-exclusive, worldwide, royalty-free copyright license to reproduce the software for reference use.
 *  (B) Patent Grant- Subject to the terms of this license, the Licensor grants you a non-transferable,
 *      non-exclusive, worldwide, royalty-free patent license under licensed patents for reference use.
 * 
 * 3. Limitations
 *  (A) No Trademark License - This license does not grant you any rights to use the Licensor's name
 *      logo, or trademarks.
 *  (B) If you begin patent litigation against the Licensor over patents that you think may apply 
 *      to the software (including a cross-claim or counterclaim in a lawsuit), your license to the 
 *      software ends automatically.
 *  (C) The software is licensed "as-is." You bear the risk of using it. The Licensor gives no express 
 *      warranties, guarantees or conditions. You may have additional consumer rights under your local 
 *      laws which this license cannot change. To the extent permitted under your local laws, the 
 *      Licensor excludes the implied warranties of merchantability, fitness for a particular purpose 
 *      and non-infringement.
 *
 ***************************************************************************************************/




#include "CodeLibrary.hpp"

#include "Memory/Memory.hpp"

using namespace Bolitho;


/* F, G, H and I are basic MD5 functions. */
#define F(x, y, z) (((x) & (y)) | ((~x) & (z)))
#define G(x, y, z) (((x) & (z)) | ((y) & (~z)))
#define H(x, y, z) ((x) ^ (y) ^ (z))
#define I(x, y, z) ((y) ^ ((x) | (~z)))

/* ROTATE_LEFT rotates x left n bits.*/
#define ROTATE_LEFT(x, n) (((x) << (n)) | ((x) >> (32-(n))))

/* FF, GG, HH, and II transformations for rounds 1, 2, 3, and 4. */
#define FF(a, b, c, d, x, s, ac)                                                                  \
{                                                                                                 \
  (a) += F ((b), (c), (d)) + (x) + (UINT32)(ac);                                                  \
  (a) = ROTATE_LEFT ((a), (s));                                                                   \
  (a) += (b);                                                                                     \
}                                                                                                 \

#define GG(a, b, c, d, x, s, ac)                                                                  \
{                                                                                                 \
  (a) += G ((b), (c), (d)) + (x) + (UINT32)(ac);                                                  \
  (a) = ROTATE_LEFT ((a), (s));                                                                   \
  (a) += (b);                                                                                     \
}                                                                                                 \

#define HH(a, b, c, d, x, s, ac)                                                                  \
{                                                                                                 \
  (a) += H ((b), (c), (d)) + (x) + (UINT32)(ac);                                                  \
  (a) = ROTATE_LEFT ((a), (s));                                                                   \
  (a) += (b);                                                                                     \
}                                                                                                 \

#define II(a, b, c, d, x, s, ac)                                                                  \
{                                                                                                 \
  (a) += I ((b), (c), (d)) + (x) + (UINT32)(ac);                                                  \
  (a) = ROTATE_LEFT ((a), (s));                                                                   \
  (a) += (b);                                                                                     \
}                                                                                                 \

#define S11 7
#define S12 12
#define S13 17
#define S14 22
#define S21 5
#define S22 9
#define S23 14
#define S24 20
#define S31 4
#define S32 11
#define S33 16
#define S34 23
#define S41 6
#define S42 10
#define S43 15
#define S44 21


static unsigned char Padding[64] = 
{
  0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
//-----------------------------------------------------------------------------------------------------------------------------------------------------
static VOID Decode(UINT32* output, CONST BYTE* input, SIZE_T len)
{
  SIZE_T i, j;

  for (i = 0, j = 0; j < len; i++, j += 4)
    output[i] = ((UINT32)input[j]) | (((UINT32)input[j+1]) << 8) | (((UINT32)input[j+2]) << 16) | (((UINT32)input[j+3]) << 24);
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
static VOID Encode (BYTE* output, UINT32* input, SIZE_T len)
{
  unsigned int i, j;
  for (i = 0, j = 0; j < len; i++, j += 4) 
  {
    output[j]   = (BYTE)(input[i] & 0xff);
    output[j+1] = (BYTE)((input[i] >> 8) & 0xff);
    output[j+2] = (BYTE)((input[i] >> 16) & 0xff);
    output[j+3] = (BYTE)((input[i] >> 24) & 0xff);
  }
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
ImplementRuntimeType(Bolitho,Md5,Object);
//-----------------------------------------------------------------------------------------------------------------------------------------------------
Md5::Md5()
{
  Reset();
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Md5::Reset()
{
  m_Count = 0;

  m_Hash.A = 0x67452301;
  m_Hash.B = 0xefcdab89;
  m_Hash.C = 0x98badcfe;
  m_Hash.D = 0x10325476;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
String Md5::ToString() CONST
{
  Md5 Hash(*this);
  return Hash.Value().ToString();
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
String Md5Hash::ToString() CONST
{
  UINT32 AA = A,BB = B,CC = C,DD = D;
  SwapByteOrder(&AA);
  SwapByteOrder(&BB);
  SwapByteOrder(&CC);
  SwapByteOrder(&DD);

  return String::Format(TEXT("%08x%08x%08x%08x"), AA, BB, CC, DD);
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
Md5Hash Md5::Value() CONST
{
  Md5 Hash(*this);
  Hash.Finish();
  return Hash.m_Hash;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Md5::Update(LPCVOID pData, SIZE_T N)
{
  SIZE_T index = (SIZE_T)((m_Count >> 3) & 0x3F);

  /* Update number of bits */
  m_Count += N*8;

  SIZE_T partLen = 64 - index;
  SIZE_T i = 0;

  /* Transform as many times as possible. */
  if (N >= partLen) 
  {
    CopyMemory(&m_Buffer[index], pData, partLen);
    Md5Transform(m_Buffer);

    for (i = partLen; i + 63 < N; i += 64)
      Md5Transform ((LPCBYTE)pData + i);
    index = 0;
  }
  else
    i = 0;

  /* Buffer remaining input */
  CopyMemory(&m_Buffer[index], (LPCBYTE)pData + i, N-i);
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Md5::Md5Transform(LPCVOID pData)
{
  UINT32 a = m_Hash.A, b = m_Hash.B, c = m_Hash.C, d = m_Hash.D, x[16];

  Decode(x, (LPCBYTE)pData, 64);

  /* Round 1 */
  FF (a, b, c, d, x[ 0], S11, 0xd76aa478); /* 1 */
  FF (d, a, b, c, x[ 1], S12, 0xe8c7b756); /* 2 */
  FF (c, d, a, b, x[ 2], S13, 0x242070db); /* 3 */
  FF (b, c, d, a, x[ 3], S14, 0xc1bdceee); /* 4 */
  FF (a, b, c, d, x[ 4], S11, 0xf57c0faf); /* 5 */
  FF (d, a, b, c, x[ 5], S12, 0x4787c62a); /* 6 */
  FF (c, d, a, b, x[ 6], S13, 0xa8304613); /* 7 */
  FF (b, c, d, a, x[ 7], S14, 0xfd469501); /* 8 */
  FF (a, b, c, d, x[ 8], S11, 0x698098d8); /* 9 */
  FF (d, a, b, c, x[ 9], S12, 0x8b44f7af); /* 10 */
  FF (c, d, a, b, x[10], S13, 0xffff5bb1); /* 11 */
  FF (b, c, d, a, x[11], S14, 0x895cd7be); /* 12 */
  FF (a, b, c, d, x[12], S11, 0x6b901122); /* 13 */
  FF (d, a, b, c, x[13], S12, 0xfd987193); /* 14 */
  FF (c, d, a, b, x[14], S13, 0xa679438e); /* 15 */
  FF (b, c, d, a, x[15], S14, 0x49b40821); /* 16 */

 /* Round 2 */
  GG (a, b, c, d, x[ 1], S21, 0xf61e2562); /* 17 */
  GG (d, a, b, c, x[ 6], S22, 0xc040b340); /* 18 */
  GG (c, d, a, b, x[11], S23, 0x265e5a51); /* 19 */
  GG (b, c, d, a, x[ 0], S24, 0xe9b6c7aa); /* 20 */
  GG (a, b, c, d, x[ 5], S21, 0xd62f105d); /* 21 */
  GG (d, a, b, c, x[10], S22,  0x2441453); /* 22 */
  GG (c, d, a, b, x[15], S23, 0xd8a1e681); /* 23 */
  GG (b, c, d, a, x[ 4], S24, 0xe7d3fbc8); /* 24 */
  GG (a, b, c, d, x[ 9], S21, 0x21e1cde6); /* 25 */
  GG (d, a, b, c, x[14], S22, 0xc33707d6); /* 26 */
  GG (c, d, a, b, x[ 3], S23, 0xf4d50d87); /* 27 */
  GG (b, c, d, a, x[ 8], S24, 0x455a14ed); /* 28 */
  GG (a, b, c, d, x[13], S21, 0xa9e3e905); /* 29 */
  GG (d, a, b, c, x[ 2], S22, 0xfcefa3f8); /* 30 */
  GG (c, d, a, b, x[ 7], S23, 0x676f02d9); /* 31 */
  GG (b, c, d, a, x[12], S24, 0x8d2a4c8a); /* 32 */

  /* Round 3 */
  HH (a, b, c, d, x[ 5], S31, 0xfffa3942); /* 33 */
  HH (d, a, b, c, x[ 8], S32, 0x8771f681); /* 34 */
  HH (c, d, a, b, x[11], S33, 0x6d9d6122); /* 35 */
  HH (b, c, d, a, x[14], S34, 0xfde5380c); /* 36 */
  HH (a, b, c, d, x[ 1], S31, 0xa4beea44); /* 37 */
  HH (d, a, b, c, x[ 4], S32, 0x4bdecfa9); /* 38 */
  HH (c, d, a, b, x[ 7], S33, 0xf6bb4b60); /* 39 */
  HH (b, c, d, a, x[10], S34, 0xbebfbc70); /* 40 */
  HH (a, b, c, d, x[13], S31, 0x289b7ec6); /* 41 */
  HH (d, a, b, c, x[ 0], S32, 0xeaa127fa); /* 42 */
  HH (c, d, a, b, x[ 3], S33, 0xd4ef3085); /* 43 */
  HH (b, c, d, a, x[ 6], S34,  0x4881d05); /* 44 */
  HH (a, b, c, d, x[ 9], S31, 0xd9d4d039); /* 45 */
  HH (d, a, b, c, x[12], S32, 0xe6db99e5); /* 46 */
  HH (c, d, a, b, x[15], S33, 0x1fa27cf8); /* 47 */
  HH (b, c, d, a, x[ 2], S34, 0xc4ac5665); /* 48 */

  /* Round 4 */
  II (a, b, c, d, x[ 0], S41, 0xf4292244); /* 49 */
  II (d, a, b, c, x[ 7], S42, 0x432aff97); /* 50 */
  II (c, d, a, b, x[14], S43, 0xab9423a7); /* 51 */
  II (b, c, d, a, x[ 5], S44, 0xfc93a039); /* 52 */
  II (a, b, c, d, x[12], S41, 0x655b59c3); /* 53 */
  II (d, a, b, c, x[ 3], S42, 0x8f0ccc92); /* 54 */
  II (c, d, a, b, x[10], S43, 0xffeff47d); /* 55 */
  II (b, c, d, a, x[ 1], S44, 0x85845dd1); /* 56 */
  II (a, b, c, d, x[ 8], S41, 0x6fa87e4f); /* 57 */
  II (d, a, b, c, x[15], S42, 0xfe2ce6e0); /* 58 */
  II (c, d, a, b, x[ 6], S43, 0xa3014314); /* 59 */
  II (b, c, d, a, x[13], S44, 0x4e0811a1); /* 60 */
  II (a, b, c, d, x[ 4], S41, 0xf7537e82); /* 61 */
  II (d, a, b, c, x[11], S42, 0xbd3af235); /* 62 */
  II (c, d, a, b, x[ 2], S43, 0x2ad7d2bb); /* 63 */
  II (b, c, d, a, x[ 9], S44, 0xeb86d391); /* 64 */

  m_Hash.A += a;
  m_Hash.B += b;
  m_Hash.C += c;
  m_Hash.D += d;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Md5::Finish()
{
  BYTE bits[8];
  SIZE_T index, padLen;

  /* Save number of bits */
  Encode(bits, (UINT32*)&m_Count, 8);

  /* Pad out to 56 mod 64. */
  index = (SIZE_T)((m_Count >> 3) & 0x3f);
  padLen = (index < 56) ? (56 - index) : (120 - index);
  
  Update(Padding, padLen);
  Update(bits, 8);
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
