/***************************************************************************************************
 * File:         $URL: https://svn.bolitho.us/Library/Trunk/Src/Graphics/Image.cpp $
 * Author:       $Author: matthew $
 * Revision:     $Rev: 1744 $
 * Last Updated: $Date: 2008-12-27 13:09:28 -0800 (Sat, 27 Dec 2008) $
 * 
 * 
 * Copyright (c) 2004-2007, Matthew G Bolitho
 * All rights reserved.
 *
 *
 * Microsoft Reference License (Ms-RL)
 *   
 * This license governs use of the accompanying software. If you use the software, you accept 
 * this license. If you do not accept the license, do not use the software.
 *
 * 1. Definitions
 *  - The terms "reproduce," "reproduction" and "distribution" have the same meaning here as under 
 *    U.S. copyright law.
 *  - "You" means the licensee of the software.
 *  - "Your company" means the company you worked for when you downloaded the software.
 *  - "Reference use" means use of the software within your company as a reference, in read only 
 *    form, for the sole purposes of debugging your products, maintaining your products, or 
 *    enhancing the interoperability of your products with the software, and specifically excludes 
 *    the right to distribute the software outside of your company.
 *  - "Licensed patents" means any Licensor patent claims which read directly on the software as 
 *    distributed by the Licensor under this license.
 * 
 * 2. Grant of Rights
 *  (A) Copyright Grant- Subject to the terms of this license, the Licensor grants you a non-transferable, 
 *      non-exclusive, worldwide, royalty-free copyright license to reproduce the software for reference use.
 *  (B) Patent Grant- Subject to the terms of this license, the Licensor grants you a non-transferable,
 *      non-exclusive, worldwide, royalty-free patent license under licensed patents for reference use.
 * 
 * 3. Limitations
 *  (A) No Trademark License - This license does not grant you any rights to use the Licensor's name
 *      logo, or trademarks.
 *  (B) If you begin patent litigation against the Licensor over patents that you think may apply 
 *      to the software (including a cross-claim or counterclaim in a lawsuit), your license to the 
 *      software ends automatically.
 *  (C) The software is licensed "as-is." You bear the risk of using it. The Licensor gives no express 
 *      warranties, guarantees or conditions. You may have additional consumer rights under your local 
 *      laws which this license cannot change. To the extent permitted under your local laws, the 
 *      Licensor excludes the implied warranties of merchantability, fitness for a particular purpose 
 *      and non-infringement.
 *
 ***************************************************************************************************/

#include "CodeLibrary.hpp"
#include "Debug/Trace.hpp"
#include "Graphics/Image.hpp"

using namespace Bolitho;
using namespace Bolitho::IO;
using namespace Bolitho::Graphics;

//-----------------------------------------------------------------------------------------------------------------------------------------------------
ImplementRuntimeType(Bolitho::Graphics, ImageException, UserException);
ImplementRuntimeType(Bolitho::Graphics, Image, Object);
//-----------------------------------------------------------------------------------------------------------------------------------------------------

//-----------------------------------------------------------------------------------------------------------------------------------------------------
#if defined(USE_LUA)
Lua::Bind::Scope ImageFormat::GenerateBindings(LPCTSTR Name)
{
  using namespace Lua::Bind;

  return Class<Bolitho::Graphics::ImageFormat>(Name)
    .Enum("ImageFormat")
    [
      Value("UNKNOWN", ImageFormat::UNKNOWN),
      Value("JPEG", ImageFormat::JPEG),
      Value("BMP", ImageFormat::BMP),
      Value("TIFF", ImageFormat::TIFF),
      Value("PNG", ImageFormat::PNG),
      Value("WDP", ImageFormat::WDP)
    ];
}
#endif
//-----------------------------------------------------------------------------------------------------------------------------------------------------

//-----------------------------------------------------------------------------------------------------------------------------------------------------
#if defined(USE_LUA)
Lua::Bind::Scope ImageBoundary::GenerateBindings(LPCTSTR Name)
{
  using namespace Lua::Bind;

  return Class<Bolitho::Graphics::ImageBoundary>(Name)
    .Enum("ImageBoundary")
    [
      Value("NONE", ImageBoundary::NONE),
      Value("CLAMP", ImageBoundary::CLAMP),
      Value("ZERO", ImageBoundary::ZERO),
      Value("PERIODIC", ImageBoundary::PERIODIC),
      Value("REFLECTIVE", ImageBoundary::REFLECTIVE)
    ];
}
#endif
//-----------------------------------------------------------------------------------------------------------------------------------------------------


//-----------------------------------------------------------------------------------------------------------------------------------------------------
Image::Image()
{
  m_Width = 0;
  m_Height = 0;
  m_pPixelData = 0;
  m_PixelFormat = PixelFormat::UNKNOWN;
  m_Boundary = ImageBoundary::NONE;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
Image::Image(UINT Width, UINT Height, DWORD PixelFormat)
{
  m_Width = 0;
  m_Height = 0;
  m_pPixelData = 0;
  m_PixelFormat = PixelFormat::UNKNOWN;
  m_Boundary = ImageBoundary::NONE;

  Resize(Width, Height, PixelFormat);
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
Image::Image(UINT Width, UINT Height, DWORD PixelFormat, LPCVOID pPixelData)
{
  m_Width = 0;
  m_Height = 0;
  m_pPixelData = 0;
  m_PixelFormat = PixelFormat::UNKNOWN;
  m_Boundary = ImageBoundary::NONE;

  Resize(Width, Height, PixelFormat);
  CopyMemory(m_pPixelData, pPixelData, m_Width*m_Height*GetPixelSize());
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
Image::Image(CONST Image& I)
{
  m_Width = 0;
  m_Height = 0;
  m_pPixelData = 0;
  m_PixelFormat = PixelFormat::UNKNOWN;
  m_Boundary = ImageBoundary::NONE;

  Resize(I.m_Width, I.m_Height, I.m_PixelFormat);
  CopyMemory(m_pPixelData, I.m_pPixelData, m_Width*m_Height*GetPixelSize());
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
Image::Image(LPCTSTR Filename)
{
  m_Width = 0;
  m_Height = 0;
  m_pPixelData = 0;
  m_PixelFormat = PixelFormat::UNKNOWN;
  m_Boundary = ImageBoundary::NONE;

  Load(Filename);
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
Image::Image(LPCTSTR Filename, DWORD PixelFormat)
{
  m_Width = 0;
  m_Height = 0;
  m_pPixelData = 0;
  m_PixelFormat = PixelFormat::UNKNOWN;
  m_Boundary = ImageBoundary::NONE;

  Load(Filename, PixelFormat);
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
Image::~Image()
{
  if (m_pPixelData)
    FreeMemory(m_pPixelData);
  m_pPixelData = 0;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
Image& Image::operator = (CONST Image& I)
{
  Resize(I.m_Width, I.m_Height, I.m_PixelFormat);
  CopyMemory(m_pPixelData, I.m_pPixelData, m_Width*m_Height*GetPixelSize());
  return *this;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::Resize(UINT Width, UINT Height, DWORD PixelFormat)
{
  UINT MinWidth = Min(Width, m_Width);
  UINT MinHeight = Min(Height, m_Height);

  UINT AllocationSize = Width*Height*GetPixelSize(PixelFormat);
  UINT OldStride = m_Width * GetPixelSize(m_PixelFormat);
  UINT NewStride = Width * GetPixelSize(PixelFormat);

  if (AllocationSize > 0)
  {
    LPVOID pPixelData = AllocateMemory(AllocationSize);
    
    if (m_pPixelData)
    {
      for (UINT i=0; i<MinHeight; i++)
       ConvertScanLine((LPBYTE)pPixelData + i*NewStride, PixelFormat, (LPCBYTE)m_pPixelData + i*OldStride, m_PixelFormat, MinWidth);
      FreeMemory(m_pPixelData);
    }

    m_pPixelData = pPixelData;
    m_Width = Width;
    m_Height = Height;
    m_PixelFormat = PixelFormat;
  }
  else
  {
    if (m_pPixelData)
      FreeMemory(m_pPixelData);
    m_pPixelData = 0;
    m_Width = Width;
    m_Height = Height;
    m_PixelFormat = PixelFormat;
  }

}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::Resize(UINT Width, UINT Height)
{
  Resize(Width, Height, m_PixelFormat);
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
Image Image::GetSubImage(UINT x, UINT y, UINT Width, UINT Height)
{
  Image I(*this);
  I.Crop(x, y, Width, Height);
  return I;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::Crop(UINT x, UINT y, UINT Width, UINT Height)
{
  if (x + Width > m_Width)
    Width = m_Width - x;
  if (y + Height > m_Height)
    Height = m_Height - y;
  if (x >= m_Width)
    Width = 0;
  if (y >= m_Height)
    Height = 0;

  UINT AllocationSize = Width*Height*GetPixelSize(m_PixelFormat);
  UINT OldStride = m_Width * GetPixelSize(m_PixelFormat);
  UINT NewStride = Width * GetPixelSize(m_PixelFormat);

  if (AllocationSize > 0)
  {
    LPVOID pPixelData = AllocateMemory(AllocationSize);

    if (m_pPixelData)
    {
      for (UINT i=0; i<Height; i++)
        CopyMemory((LPBYTE)pPixelData + i*NewStride , (LPCBYTE)m_pPixelData + (i+y)*OldStride + x*GetPixelSize(m_PixelFormat), Width*GetPixelSize(m_PixelFormat));
      FreeMemory(m_pPixelData);
      m_pPixelData = pPixelData;
    }
    m_Width = Width;
    m_Height = Height;
  }
  else
  {
    m_pPixelData = 0;
    m_Width = 0;
    m_Height = 0;
  }
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::SetPixelFormat(DWORD PixelFormat)
{
  if (m_pPixelData == 0 || m_PixelFormat == PixelFormat)
  {
    m_PixelFormat = PixelFormat;
    return;
  }

  LPVOID pNewPixelData = AllocateMemory(m_Width*m_Height*GetPixelSize(PixelFormat));

  for (UINT i=0; i<m_Height; i++)
    ConvertScanLine((LPBYTE)pNewPixelData + i*m_Width*GetPixelSize(PixelFormat), PixelFormat, (LPBYTE)m_pPixelData + i*m_Width*GetPixelSize(m_PixelFormat), m_PixelFormat, m_Width);

  FreeMemory(m_pPixelData);
  m_pPixelData = pNewPixelData;
  m_PixelFormat = PixelFormat;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
#if defined(USE_LUA)
Lua::Bind::Scope Image::GenerateBindings(LPCTSTR Name)
{
  using namespace Lua::Bind;

  return 
    Class<Image>(Name)
      .Def(Constructor<>())
      .Def(Constructor<UINT,UINT,DWORD>())
      .Def(Constructor<LPCTSTR>())

      .Property("Width", &Image::Width)
      .Property("Height", &Image::Height)
      .Property("PixelFormat", &Image::GetPixelFormat)
      .Property("PixelSize", (UINT(Image::*)() CONST)&Image::GetPixelSize) 
      .Property("PixelChannelCount", (UINT(Image::*)() CONST)&Image::GetPixelChannels)

      .Def("SetPixelFormat", &Image::SetPixelFormat)

      .Def("Resize", (VOID(Image::*)(UINT, UINT))&Image::Resize)
      .Def("Resize", (VOID(Image::*)(UINT, UINT, DWORD))&Image::Resize)

      .Def("Load", (VOID(Image::*)(LPCTSTR))&Image::Load)
      .Def("Load", (VOID(Image::*)(LPCTSTR, DWORD))&Image::Load)
      
      .Def("Save", (VOID(Image::*)(LPCTSTR) CONST)&Image::Save)
      .Def("Save", (VOID(Image::*)(LPCTSTR, DWORD) CONST)&Image::Save)
      .Def("Save", (VOID(Image::*)(LPCTSTR, DWORD, DWORD) CONST)&Image::Save)
      
      .Def("Crop", &Image::Crop)
      .Def("GetSubImage", &Image::GetSubImage)

      .Scope[ Def("GetNativePixelFormat", &Image::GetImagePixelFormat) ]
      .Scope[ Def("GetPixelSize", (UINT(*)(DWORD))&Image::GetPixelSize) ]
      .Scope[ Def("GetPixelChannelCount", (UINT(*)(DWORD))&Image::GetPixelChannels) ]

      .Scope[ Def("Capture", (Image(*)(UINT, UINT, UINT, UINT, UINT))&Image::CaptureImage) ]
      .Scope[ Def("Capture", (Image(*)(LPCTSTR, UINT, UINT, UINT, UINT))&Image::CaptureImage) ]

    ;
  
}
#endif
//-----------------------------------------------------------------------------------------------------------------------------------------------------

#if defined(PLATFORM_WINDOWS)

#include "Graphics/StillImageCapture.hpp"
#include "Com/Guid.hpp"
#include "Com/ComException.hpp"
#include "Com/ComPointer.hpp"
using namespace Bolitho::Com;

#include <wincodec.h>
#pragma comment(lib, "windowscodecs.lib")
//-----------------------------------------------------------------------------------------------------------------------------------------------------
Image Image::CaptureImage(UINT DeviceOrdinal, UINT Width, UINT Height, UINT FrameCount, UINT FrameDelay)
{
  Array<StringPair> DeviceList = StillImageCapture::EnumerateCaptureDevices();
  if (DeviceOrdinal < DeviceList.Length())
    return CaptureImage(DeviceList[DeviceOrdinal].First, Width, Height, FrameCount, FrameDelay);
  else
    throw InvalidOperationException(0, "There is no device with the given ordinal");
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
Image Image::CaptureImage(LPCTSTR DeviceName, UINT Width, UINT Height, UINT FrameCount, UINT FrameDelay)
{
  StillImageCapture C(DeviceName, Width, Height);
  Image I(Width, Height, PixelFormat::R32FG32FB32FA32F);

  //if (Parameters.Contains("Exposure"))
  //  C.SetExposure(GetParameter<LONG>(Parameters, "Exposure"));

  C.Capture(FrameDelay, FrameCount, TRUE);
  
  Buffer<FLOAT>& AccumulationBuffer = C.GetAccumulationBuffer();
  UINT k=0;
  for (UINT i=I.Height()-1; i<I.Height(); i--)
  {
    for (UINT j=0; j<I.Width(); j++)
    {
      I.Pixel<ColorRGBA128F>(i,j)[2] = AccumulationBuffer[k++] / 255.0f;
      I.Pixel<ColorRGBA128F>(i,j)[1] = AccumulationBuffer[k++] / 255.0f;
      I.Pixel<ColorRGBA128F>(i,j)[0] = AccumulationBuffer[k++] / 255.0f;
      I.Pixel<ColorRGBA128F>(i,j)[3] = 1.0f;
    }
  }

  return I;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------


#define TryFormat(x)                                                            \
{                                                                               \
  for (UINT i=0; i<PixelFormatCount; i++)                                       \
  if (IsEqualGUID(pPixelFormats[i], x))                                         \
  return x;                                                                     \
}

//-----------------------------------------------------------------------------------------------------------------------------------------------------
WICPixelFormatGUID ChooseBestPixelFormat(DWORD PixelFormat, UINT PixelFormatCount, WICPixelFormatGUID* pPixelFormats)
{
  switch (PixelFormat)
  {
  case PixelFormat::R8G8B8A8:
    TryFormat(GUID_WICPixelFormat32bppBGRA);
    TryFormat(GUID_WICPixelFormat32bppBGR);
    TryFormat(GUID_WICPixelFormat24bppRGB);
    TryFormat(GUID_WICPixelFormat24bppBGR);
    break;
  case PixelFormat::R8G8B8X8:
    TryFormat(GUID_WICPixelFormat32bppBGR);
    TryFormat(GUID_WICPixelFormat24bppRGB);
    TryFormat(GUID_WICPixelFormat24bppBGR);
    TryFormat(GUID_WICPixelFormat32bppBGRA);
    break;
  case PixelFormat::R8G8B8:
    TryFormat(GUID_WICPixelFormat24bppRGB);
    TryFormat(GUID_WICPixelFormat24bppBGR);
    TryFormat(GUID_WICPixelFormat32bppBGR);
    TryFormat(GUID_WICPixelFormat32bppBGRA);
    break;

  case PixelFormat::B8G8R8A8:
    TryFormat(GUID_WICPixelFormat32bppBGRA);
    TryFormat(GUID_WICPixelFormat32bppBGR);
    TryFormat(GUID_WICPixelFormat24bppBGR);
    TryFormat(GUID_WICPixelFormat24bppRGB);
    break;
  case PixelFormat::B8G8R8X8:
    TryFormat(GUID_WICPixelFormat32bppBGR);
    TryFormat(GUID_WICPixelFormat24bppBGR);
    TryFormat(GUID_WICPixelFormat24bppRGB);
    TryFormat(GUID_WICPixelFormat32bppBGRA);
    break;
  case PixelFormat::B8G8R8:
    TryFormat(GUID_WICPixelFormat24bppBGR);
    TryFormat(GUID_WICPixelFormat24bppRGB);
    TryFormat(GUID_WICPixelFormat32bppBGR);
    TryFormat(GUID_WICPixelFormat32bppBGRA);
    break;

  case PixelFormat::R32FG32FB32FA32F:
    TryFormat(GUID_WICPixelFormat128bppRGBAFloat);
    TryFormat(GUID_WICPixelFormat32bppBGRA);
    TryFormat(GUID_WICPixelFormat32bppBGR);
    TryFormat(GUID_WICPixelFormat24bppBGR);
    TryFormat(GUID_WICPixelFormat24bppRGB);
    break;
  case PixelFormat::R32FG32FB32FX32:
    TryFormat(GUID_WICPixelFormat128bppRGBAFloat);
    TryFormat(GUID_WICPixelFormat32bppBGR);
    TryFormat(GUID_WICPixelFormat32bppBGRA);
    TryFormat(GUID_WICPixelFormat24bppBGR);
    TryFormat(GUID_WICPixelFormat24bppRGB);
    break;
  case PixelFormat::R32FG32FB32F:
    TryFormat(GUID_WICPixelFormat128bppRGBAFloat);
    TryFormat(GUID_WICPixelFormat24bppBGR);
    TryFormat(GUID_WICPixelFormat24bppRGB);
    TryFormat(GUID_WICPixelFormat32bppBGR);
    TryFormat(GUID_WICPixelFormat32bppBGRA);
    break;

  case PixelFormat::R8:
    TryFormat(GUID_WICPixelFormat8bppGray);
    TryFormat(GUID_WICPixelFormat32bppBGRA);
    TryFormat(GUID_WICPixelFormat24bppBGR);
    TryFormat(GUID_WICPixelFormat24bppRGB);
    break;
  case PixelFormat::R32F:
    TryFormat(GUID_WICPixelFormat32bppGrayFloat);
    TryFormat(GUID_WICPixelFormat128bppRGBAFloat);
    TryFormat(GUID_WICPixelFormat8bppGray);
    TryFormat(GUID_WICPixelFormat32bppBGRA);
    TryFormat(GUID_WICPixelFormat24bppBGR);
    TryFormat(GUID_WICPixelFormat24bppRGB);
    break;
  }

  return GUID_WICPixelFormatUndefined;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------

#define MatchFormat(x, y)                                                       \
{                                                                               \
  if (IsEqualGUID(Format, x))                                                   \
  return y;                                                                     \
}

//-----------------------------------------------------------------------------------------------------------------------------------------------------
DWORD ChooseBestPixelFormat(WICPixelFormatGUID& Format)
{
  MatchFormat(GUID_WICPixelFormat128bppRGBAFloat, PixelFormat::R32FG32FB32FA32F);
  MatchFormat(GUID_WICPixelFormat24bppBGR, PixelFormat::B8G8R8);
  MatchFormat(GUID_WICPixelFormat24bppRGB, PixelFormat::R8G8B8);
  MatchFormat(GUID_WICPixelFormat32bppBGR, PixelFormat::B8G8R8X8);
  MatchFormat(GUID_WICPixelFormat32bppBGRA, PixelFormat::B8G8R8A8);
  MatchFormat(GUID_WICPixelFormat32bppGrayFloat, PixelFormat::R32F);
  MatchFormat(GUID_WICPixelFormat8bppGray, PixelFormat::R8);

  return PixelFormat::UNKNOWN;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
UINT GetPixelFormatStride(UINT Width, WICPixelFormatGUID& Format)
{
  if (IsEqualGUID(Format, GUID_WICPixelFormat32bppBGRA))
    return (Width * 32 + 7) / 8;
  if (IsEqualGUID(Format, GUID_WICPixelFormat32bppBGR))
    return (Width * 32 + 7) / 8;
  if (IsEqualGUID(Format, GUID_WICPixelFormat24bppBGR))
    return (Width * 24 + 7) / 8;
  if (IsEqualGUID(Format, GUID_WICPixelFormat128bppRGBAFloat))
    return (Width * 16);
  if (IsEqualGUID(Format, GUID_WICPixelFormat8bppGray))
    return Width;

  Assert(FALSE);
  return 0;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::Save(LPCTSTR Filename, DWORD FileFormat, DWORD PixelFormat) CONST
{
  if (FileFormat == ImageFormat::UNKNOWN)
  {
    if (Path::ExtensionMatches(Filename, TEXT("PNG")))
      FileFormat = ImageFormat::PNG;
    if (Path::ExtensionMatches(Filename, TEXT("BMP")))
      FileFormat = ImageFormat::BMP;
    if (Path::ExtensionMatches(Filename, TEXT("JPEG")) || Path::ExtensionMatches(Filename, TEXT("JPG")))
      FileFormat = ImageFormat::JPEG;
    if (Path::ExtensionMatches(Filename, TEXT("TIFF")))
      FileFormat = ImageFormat::TIFF;
    if (Path::ExtensionMatches(Filename, TEXT("WDP")))
      FileFormat = ImageFormat::WDP;
    if (Path::ExtensionMatches(Filename, TEXT("FLT")))
      FileFormat = ImageFormat::FLT;
    if (Path::ExtensionMatches(Filename, TEXT("PPM")))
      FileFormat = ImageFormat::PPM;
  }

  if (FileFormat == ImageFormat::FLT)
    return SaveFLT(Filename);
  if (FileFormat == ImageFormat::PPM)
    return SavePPM(Filename);
  if (FileFormat == ImageFormat::PPMBINARY)
    return SavePPMBinary(Filename);

  ComPtr<IWICImagingFactory> pFactory;
  ComPtr<IWICBitmapEncoder> pEncoder;
  ComPtr<IWICBitmapFrameEncode> pBitmapFrame;
  ComPtr<IPropertyBag2> pPropertyBag;
  ComPtr<IWICStream> pStream;

  VerifyComResult(pFactory.CreateInstance(CLSID_WICImagingFactory));
  VerifyComResult(pFactory->CreateStream(&pStream));
  VerifyComResult(pStream->InitializeFromFilename(StringW(Filename), GENERIC_WRITE));

  switch(FileFormat)
  {
  case ImageFormat::BMP:
    VerifyComResult(pFactory->CreateEncoder(GUID_ContainerFormatBmp, NULL, &pEncoder)); break;
  case ImageFormat::PNG:
    VerifyComResult(pFactory->CreateEncoder(GUID_ContainerFormatPng, NULL, &pEncoder)); break;
  case ImageFormat::JPEG:
    VerifyComResult(pFactory->CreateEncoder(GUID_ContainerFormatJpeg, NULL, &pEncoder)); break;
  case ImageFormat::TIFF:
    VerifyComResult(pFactory->CreateEncoder(GUID_ContainerFormatTiff, NULL, &pEncoder)); break;
  case ImageFormat::WDP:
    VerifyComResult(pFactory->CreateEncoder(GUID_ContainerFormatWmp, NULL, &pEncoder)); break;
  };
  
  VerifyComResult(pEncoder->Initialize(pStream, WICBitmapEncoderNoCache));
  
  ComPtr<IWICBitmapEncoderInfo> pEncoderInfo;
  VerifyComResult(pEncoder->GetEncoderInfo(&pEncoderInfo));

  UINT PixelFormatCount = 0;
  VerifyComResult(pEncoderInfo->GetPixelFormats(0, 0, &PixelFormatCount));
  GUID* pPixelFormats = new GUID[PixelFormatCount];
  VerifyComResult(pEncoderInfo->GetPixelFormats(PixelFormatCount, pPixelFormats, &PixelFormatCount));
  WICPixelFormatGUID FormatGUID = ChooseBestPixelFormat(m_PixelFormat, PixelFormatCount, pPixelFormats);
  DWORD OutFormat = ChooseBestPixelFormat(FormatGUID);
  delete[] pPixelFormats;

  VerifyComResult(pEncoder->CreateNewFrame(&pBitmapFrame, &pPropertyBag));
  VerifyComResult(pBitmapFrame->Initialize(pPropertyBag));
  VerifyComResult(pBitmapFrame->SetSize(m_Width, m_Height));
  VerifyComResult(pBitmapFrame->SetPixelFormat(&FormatGUID));
  UINT Stride = GetPixelFormatStride(m_Width, FormatGUID);

  BYTE* pScanLine = new BYTE[Stride];
  for (UINT i=0; i<m_Height; i++)
  {
    ConvertScanLine(pScanLine, OutFormat, (LPBYTE)m_pPixelData + i*m_Width*GetPixelSize(m_PixelFormat), m_PixelFormat, m_Width);
    VerifyComResult(pBitmapFrame->WritePixels(1, Stride, Stride, pScanLine));
  }
  delete[] pScanLine;

  VerifyComResult(pBitmapFrame->Commit());
  VerifyComResult(pEncoder->Commit());
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::Load(LPCTSTR Filename, DWORD SpecifiedPixelFormat)
{
  ComPtr<IWICImagingFactory> pFactory;
  ComPtr<IWICBitmapDecoder> pDecoder;
  ComPtr<IWICBitmapFrameDecode> pBitmapFrame;

  VerifyComResult(pFactory.CreateInstance(CLSID_WICImagingFactory));
  HRESULT hr = pFactory->CreateDecoderFromFilename(StringW(Filename), NULL, GENERIC_READ, WICDecodeMetadataCacheOnDemand, &pDecoder);
  if (FAILED(hr))
  {
    if (hr == 0x80070002)
      throw IO::FileException(this, String::Format("The file \"%s\" cannot be found", Filename));
    else
      throw ComException(this, hr);
  }

  VerifyComResult(pDecoder->GetFrame(0, &pBitmapFrame));  

  UINT Width, Height;
  VerifyComResult(pBitmapFrame->GetSize(&Width, &Height));
  WICPixelFormatGUID FormatGUID;
  VerifyComResult(pBitmapFrame->GetPixelFormat(&FormatGUID));

  DWORD InputPixelFormat = ChooseBestPixelFormat(FormatGUID);
  DWORD InternalPixelFormat = SpecifiedPixelFormat;
  if (InternalPixelFormat == PixelFormat::UNKNOWN)
    InternalPixelFormat = InputPixelFormat;

  Resize(Width, Height, InternalPixelFormat);

  UINT Stride = GetPixelFormatStride(m_Width, FormatGUID);
  BYTE* pScanLine = new BYTE[Stride];
  for (UINT i=0; i<m_Height; i++)
  {
    WICRect rc;
    rc.X = 0;
    rc.Y = i;
    rc.Width = m_Width;
    rc.Height = 1;
    
    VerifyComResult(pBitmapFrame->CopyPixels(&rc, Stride, Stride, pScanLine));

    ConvertScanLine((LPBYTE)m_pPixelData + i*m_Width*GetPixelSize(InternalPixelFormat), InternalPixelFormat, pScanLine, InputPixelFormat, m_Width);
  }
  delete[] pScanLine;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
DWORD Image::GetImagePixelFormat(LPCTSTR Filename)
{
  ComPtr<IWICImagingFactory> pFactory;
  ComPtr<IWICBitmapDecoder> pDecoder;
  ComPtr<IWICBitmapFrameDecode> pBitmapFrame;

  VerifyComResultStatic(pFactory.CreateInstance(CLSID_WICImagingFactory));
  VerifyComResultStatic(pFactory->CreateDecoderFromFilename(StringW(Filename), NULL, GENERIC_READ, WICDecodeMetadataCacheOnDemand, &pDecoder));
  VerifyComResultStatic(pDecoder->GetFrame(0, &pBitmapFrame));  

  WICPixelFormatGUID FormatGUID;
  VerifyComResultStatic(pBitmapFrame->GetPixelFormat(&FormatGUID));

  return ChooseBestPixelFormat(FormatGUID);
}
#endif
//-----------------------------------------------------------------------------------------------------------------------------------------------------

//-----------------------------------------------------------------------------------------------------------------------------------------------------
#if defined(PLATFORM_POSIX)
VOID Image::Save(LPCTSTR Filename, DWORD FileFormat, DWORD PixelFormat) CONST
{
  if (FileFormat == ImageFormat::UNKNOWN)
  {
    if (Path::ExtensionMatches(Filename, TEXT("PNG")))
      FileFormat = ImageFormat::PNG;
    if (Path::ExtensionMatches(Filename, TEXT("BMP")))
      FileFormat = ImageFormat::BMP;
    if (Path::ExtensionMatches(Filename, TEXT("JPEG")) || Path::ExtensionMatches(Filename, TEXT("JPG")))
      FileFormat = ImageFormat::JPEG;
    if (Path::ExtensionMatches(Filename, TEXT("TIFF")))
      FileFormat = ImageFormat::TIFF;
    if (Path::ExtensionMatches(Filename, TEXT("WDP")))
      FileFormat = ImageFormat::WDP;
    if (Path::ExtensionMatches(Filename, TEXT("FLT")))
      FileFormat = ImageFormat::FLT;
    if (Path::ExtensionMatches(Filename, TEXT("PPM")))
      FileFormat = ImageFormat::PPM;
    if (Path::ExtensionMatches(Filename, TEXT("PBM")))
      FileFormat = ImageFormat::PPMBINARY;
    if (Path::ExtensionMatches(Filename, TEXT("EXR")))
      FileFormat = ImageFormat::EXR;
  }

  if (FileFormat == ImageFormat::FLT)
    return SaveFLT(Filename);
  if (FileFormat == ImageFormat::PPM)
    return SavePPM(Filename);
  if (FileFormat == ImageFormat::PPMBINARY)
    return SavePPMBinary(Filename);
  if (FileFormat == ImageFormat::BMP)
    return SaveBMP(Filename);
  if (FileFormat == ImageFormat::PNG)
    return SavePNG(Filename);
  if (FileFormat == ImageFormat::JPEG)
    return SaveJPEG(Filename);
  //if (FileFormat == ImageFormat::EXR)
  //  return SaveEXR(Filename);

  throw InvalidOperationException(this, "Unknown FileFormat");
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::Load(LPCTSTR Filename, DWORD SpecifiedPixelFormat)
{
  DWORD FileFormat = ImageFormat::UNKNOWN;

  if (Path::ExtensionMatches(Filename, TEXT("PNG")))
    FileFormat = ImageFormat::PNG;
  if (Path::ExtensionMatches(Filename, TEXT("BMP")))
    FileFormat = ImageFormat::BMP;
  if (Path::ExtensionMatches(Filename, TEXT("JPEG")) || Path::ExtensionMatches(Filename, TEXT("JPG")))
    FileFormat = ImageFormat::JPEG;
  if (Path::ExtensionMatches(Filename, TEXT("TIFF")))
    FileFormat = ImageFormat::TIFF;
  if (Path::ExtensionMatches(Filename, TEXT("WDP")))
    FileFormat = ImageFormat::WDP;
  if (Path::ExtensionMatches(Filename, TEXT("FLT")))
    FileFormat = ImageFormat::FLT;
  if (Path::ExtensionMatches(Filename, TEXT("PPM")))
    FileFormat = ImageFormat::PPM;
  if (Path::ExtensionMatches(Filename, TEXT("PBM")))
    FileFormat = ImageFormat::PPMBINARY;
  if (Path::ExtensionMatches(Filename, TEXT("EXR")))
    FileFormat = ImageFormat::EXR;
  
  //if (FileFormat == ImageFormat::FLT)
  //  return LoadFLT(Filename);
  //if (FileFormat == ImageFormat::PPM)
  //  return LoadPPM(Filename);
  //if (FileFormat == ImageFormat::PPMBINARY)
  //  return LoadPPMBinary(Filename);
  if (FileFormat == ImageFormat::BMP)
    return LoadBMP(Filename);
  if (FileFormat == ImageFormat::PNG)
    return LoadPNG(Filename);
  if (FileFormat == ImageFormat::JPEG)
    return LoadJPEG(Filename);
  //if (FileFormat == ImageFormat::EXR)
  //  return SaveEXR(Filename);

  throw InvalidOperationException(this, "Unknown FileFormat");
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------

//-----------------------------------------------------------------------------------------------------------------------------------------------------
#pragma pack(push,1)
struct BITMAPINFOHEADER
{
  DWORD  biSize; 
  INT    biWidth; 
  INT    biHeight; 
  WORD   biPlanes; 
  WORD   biBitCount; 
  DWORD  biCompression; 
  DWORD  biSizeImage; 
  INT    biXPelsPerMeter; 
  INT    biYPelsPerMeter; 
  DWORD  biClrUsed; 
  DWORD  biClrImportant; 
};
//-----------------------------------------------------------------------------------------------------------------------------------------------------
struct BITMAPFILEHEADER 
{ 
  WORD    bfType; 
  DWORD   bfSize; 
  WORD    bfReserved1; 
  WORD    bfReserved2; 
  DWORD   bfOffBits; 
};
#pragma pack(pop)
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::LoadBMP(LPCTSTR Filename)
{
  BITMAPFILEHEADER FileHeader;
  BITMAPINFOHEADER InfoHeader;

  Ptr<File> pFile = File::Open(Filename, FileMode::OPEN, FileAccess::READ);

  pFile->Read(&FileHeader, sizeof(FileHeader));
  pFile->Read(&InfoHeader, sizeof(InfoHeader));

  Resize(InfoHeader.biWidth, InfoHeader.biHeight);
  pFile->Seek(FileHeader.bfOffBits);


  UINT BMPStride = 0;
  DWORD BitmapPixelFormat = 0;
  if (InfoHeader.biBitCount == 24)
  {
    BitmapPixelFormat = PixelFormat::B8G8R8;
    BMPStride = 4 * ((3 * InfoHeader.biWidth + 3) / 4);
  }
  else if (InfoHeader.biBitCount == 32)
  {
    BitmapPixelFormat = PixelFormat::R8G8B8X8;
    BMPStride = InfoHeader.biWidth * 4;
  }
  else
    throw InvalidFormatException(this, "Bitmap must be 24bbp or 32 bpp");

  UINT Stride = InfoHeader.biWidth * GetPixelSize(m_PixelFormat);
  LPBYTE pBuffer = new BYTE[BMPStride];
  LPBYTE pPixelData = (LPBYTE)m_pPixelData;

  for (UINT i=0; i<InfoHeader.biHeight; i++)
  {
    pFile->Read(pBuffer, BMPStride);
    ConvertScanLine(pPixelData, m_PixelFormat, pBuffer, BitmapPixelFormat, m_Width);
    pPixelData += Stride;
  }
  
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::SaveBMP(LPCTSTR Filename) CONST
{
  BITMAPFILEHEADER FileHeader;
  BITMAPINFOHEADER InfoHeader;

  Ptr<File> pFile = File::Open(Filename, FileMode::CREATE, FileAccess::WRITE);
  
  UINT BMPStride = 4 * ((3 * m_Width + 3) / 4);
  UINT BitmapSize = sizeof(FileHeader) + sizeof(InfoHeader) + BMPStride * m_Height;

  ZeroMemory(&FileHeader, sizeof(FileHeader));
  FileHeader.bfType = 19778;
  FileHeader.bfSize = BitmapSize;
  FileHeader.bfOffBits = sizeof(FileHeader) + sizeof(InfoHeader);

  ZeroMemory(&InfoHeader, sizeof(InfoHeader));
  InfoHeader.biSize = sizeof(InfoHeader);
  InfoHeader.biWidth = m_Width;
  InfoHeader.biHeight = m_Height;
  InfoHeader.biPlanes = 1;
  InfoHeader.biBitCount = 24;
  InfoHeader.biSizeImage = BMPStride * m_Height;
  InfoHeader.biXPelsPerMeter = 2835;
  InfoHeader.biYPelsPerMeter = 2835;

  pFile->Write(&FileHeader, sizeof(FileHeader));
  pFile->Write(&InfoHeader, sizeof(InfoHeader));

  UINT Stride = InfoHeader.biWidth * GetPixelSize(m_PixelFormat);
  LPBYTE pBuffer = new BYTE[BMPStride];
  LPBYTE pPixelData = (LPBYTE)m_pPixelData;

  for (UINT i=0; i<InfoHeader.biHeight; i++)
  {
    ConvertScanLine(pBuffer, PixelFormat::B8G8R8, pPixelData, m_PixelFormat, m_Width);
    pPixelData += Stride;
    pFile->Write(pBuffer, BMPStride);
  }
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------

//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::LoadPNG(LPCTSTR Filename)
{
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::LoadJPEG(LPCTSTR Filename)
{
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::SavePNG(LPCTSTR Filename) CONST
{
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::SaveJPEG(LPCTSTR Filename) CONST
{
} 
#endif
//-----------------------------------------------------------------------------------------------------------------------------------------------------


//-----------------------------------------------------------------------------------------------------------------------------------------------------
DWORD Image::GetNearestFloatingPointPixelFormat(DWORD PixelFormat)
{
  return PixelFormat & 0xf0ffffff | 0x0500000;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
DWORD Image::GetNearestBytePixelFormat(DWORD PixelFormat)
{
  return PixelFormat & 0xf0ffffff | 0x0100000;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::SaveFLT(LPCTSTR Filename) CONST
{
  DWORD PixelFormat = GetNearestFloatingPointPixelFormat(m_PixelFormat);
  UINT Channels = GetPixelChannels(PixelFormat);
  UINT Stride = GetPixelChannels(m_PixelFormat) * m_Width;

  FLOAT* pScanLine = new FLOAT[m_Width * Channels];
  BinaryStream B(File::Open(Filename, FileMode::OPENORCREATE, FileAccess::WRITE));
  B << m_Width;
  B << m_Height;
  B << Channels;

  for (UINT i=0; i<m_Height; i++)
  {
    ConvertScanLine(pScanLine, PixelFormat, (LPCBYTE)m_pPixelData + i*Stride, m_PixelFormat, m_Width);
    B.Write(pScanLine, sizeof(FLOAT) * m_Width * Channels);
  }

  delete[] pScanLine;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::SavePPMBinary(LPCTSTR Filename) CONST
{
  DWORD PixelFormat = PixelFormat::R8G8B8;
  UINT Channels = GetPixelChannels(PixelFormat);
  UINT Stride = GetPixelChannels(m_PixelFormat) * m_Width;

  BYTE* pScanLine = new BYTE[m_Width * Channels];
  
  Ptr<File> pFile = File::Open(Filename, FileMode::OPENORCREATE, FileAccess::WRITE);

  TextStream T(pFile);
  T.WriteLine("P6");
  T.WriteLine("%d %d", m_Width, m_Height);
  T.WriteLine("255");

  BinaryStream B(pFile);
  for (UINT i=0; i<m_Height; i++)
  {
    ConvertScanLine(pScanLine, PixelFormat, (LPCBYTE)m_pPixelData + i*Stride, m_PixelFormat, m_Width);
    B.Write(pScanLine, m_Width * Channels);
  }

  delete[] pScanLine;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------
VOID Image::SavePPM(LPCTSTR Filename) CONST
{
  DWORD PixelFormat = PixelFormat::R8G8B8;
  UINT Channels = GetPixelChannels(PixelFormat);
  UINT Stride = GetPixelChannels(m_PixelFormat) * m_Width;

  BYTE* pScanLine = new BYTE[m_Width * Channels];
  
  Ptr<File> pFile = File::Open(Filename, FileMode::OPENORCREATE, FileAccess::WRITE);

  TextStream T(pFile);
  T.WriteLine("P6");
  T.WriteLine("%d %d", m_Width, m_Height);
  T.WriteLine("255");

  BinaryStream B(pFile);
  for (UINT i=0; i<m_Height; i++)
  {
    ConvertScanLine(pScanLine, PixelFormat, (LPCBYTE)m_pPixelData + i*Stride, m_PixelFormat, m_Width);
    for (UINT j=0; j<m_Width; j++)
      T.Write("%d %d %d ", pScanLine[3*j], pScanLine[3*j+1], pScanLine[3*j+2]);
    T.WriteLine("");
  }

  delete[] pScanLine;
}
//-----------------------------------------------------------------------------------------------------------------------------------------------------


//-----------------------------------------------------------------------------------------------------------------------------------------------------
#pragma optimize("", off)
VOID Image::ConvertScanLine(LPVOID pOutData, DWORD OutPixelFormat, LPCVOID pInData, DWORD InPixelFormat, UINT Width)
{
if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R8)
  CopyMemory(pOutData, pInData, Width*1);

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[1*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[2*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[2*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 1] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[2*i + 1] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[2*i + 1] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[3*i + 2] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((0) << 24);
    pTypedOutData[3*i + 2] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[3*i + 0] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((0) << 24);
    pTypedOutData[3*i + 0] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R16)
  CopyMemory(pOutData, pInData, Width*2);

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[2*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[2*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[2*i + 1] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 1] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[2*i + 1] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[3*i + 2] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((0) << 16);
    pTypedOutData[3*i + 2] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[3*i + 0] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((0) << 16);
    pTypedOutData[3*i + 0] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R32)
  CopyMemory(pOutData, pInData, Width*4);

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[2*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[2*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[2*i + 1] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[2*i + 1] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[3*i + 2] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[3*i + 2] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[3*i + 0] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[3*i + 0] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[1*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R32F)
  CopyMemory(pOutData, pInData, Width*4);

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[2*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[2*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[2*i + 1] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[2*i + 1] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 1] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[3*i + 2] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[3*i + 2] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[3*i + 2] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0.0f;
    pTypedOutData[3*i + 2] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 0.0f;
    pTypedOutData[4*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 2] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[3*i + 0] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[3*i + 0] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[3*i + 0] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0.0f;
    pTypedOutData[3*i + 0] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 0.0f;
    pTypedOutData[4*i + 3] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 1] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[2*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::A8R8)
  CopyMemory(pOutData, pInData, Width*2);

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 1]) << 8);
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 1]) << 24);
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 1]) << 8);
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 24);
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[2*i + 1]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[3*i + 2] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((0) << 24);
    pTypedOutData[3*i + 2] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[2*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[2*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[2*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[2*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[2*i + 1];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[2*i + 1]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[3*i + 0] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[2*i + 1]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((0) << 24);
    pTypedOutData[3*i + 0] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[2*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[2*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[2*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[2*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::A8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::A16R16)
  CopyMemory(pOutData, pInData, Width*4);

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 1]) << 16);
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 16);
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[3*i + 2] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((0) << 16);
    pTypedOutData[3*i + 2] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[2*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[2*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[3*i + 0] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[2*i + 1];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[2*i + 1]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((0) << 16);
    pTypedOutData[3*i + 0] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[2*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[2*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::A16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::A32R32)
  CopyMemory(pOutData, pInData, Width*8);

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[3*i + 2] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[3*i + 2] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[3*i + 0] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[3*i + 0] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[2*i + 1];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::A32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[2*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 1]) * 255);
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::A32FR32F)
  CopyMemory(pOutData, pInData, Width*8);

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 1]) * 255);
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[2*i + 1]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[3*i + 2] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[3*i + 2] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[3*i + 2] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[3*i + 1] = 0.0f;
    pTypedOutData[3*i + 2] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[2*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 0.0f;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[2*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 2] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[2*i + 1]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[3*i + 0] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[3*i + 0] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[3*i + 0] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[2*i + 1];
    pTypedOutData[3*i + 1] = 0.0f;
    pTypedOutData[3*i + 0] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[2*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 0.0f;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 1] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[2*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::A32FR32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 1];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[2*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 0];
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 0]) << 8);
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 0]) << 24);
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R8A8)
  CopyMemory(pOutData, pInData, Width*2);

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 0]) << 8);
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 24);
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[2*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[2*i + 0]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[3*i + 2] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((0) << 24);
    pTypedOutData[3*i + 2] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[2*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[2*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[2*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[2*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[2*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[2*i + 0]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[3*i + 0] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[2*i + 0]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((0) << 24);
    pTypedOutData[3*i + 0] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[2*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[2*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[2*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[2*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[2*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 0];
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 0]) << 16);
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R16A16)
  CopyMemory(pOutData, pInData, Width*4);

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 16);
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[3*i + 2] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[2*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((0) << 16);
    pTypedOutData[3*i + 2] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[2*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[2*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[3*i + 0] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[2*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[2*i + 0]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((0) << 16);
    pTypedOutData[3*i + 0] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[2*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[2*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[2*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[2*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 0];
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R32A32)
  CopyMemory(pOutData, pInData, Width*8);

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[3*i + 2] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[3*i + 2] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[2*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[3*i + 0] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[3*i + 0] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[2*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[2*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[2*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[2*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[2*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 0]) * 255);
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 0];
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 0]) * 255);
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R32FA32F)
  CopyMemory(pOutData, pInData, Width*8);

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[2*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[2*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[2*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[2*i + 0]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[3*i + 2] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[3*i + 2] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[3*i + 2] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[2*i + 0];
    pTypedOutData[3*i + 1] = 0.0f;
    pTypedOutData[3*i + 2] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[2*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 0.0f;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[2*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 2] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[2*i + 0]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[3*i + 0] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[3*i + 0] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[3*i + 0] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[2*i + 0];
    pTypedOutData[3*i + 1] = 0.0f;
    pTypedOutData[3*i + 0] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[2*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[2*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 0.0f;
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[2*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 1] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[2*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[2*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[2*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[2*i + 0];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[1*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[1*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[1*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 1];
    pTypedOutData[2*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 1]) << 8);
    pTypedOutData[2*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 1]) << 24);
    pTypedOutData[2*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[2*i + 1] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 1]) << 8);
    pTypedOutData[2*i + 1] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 1]) << 24);
    pTypedOutData[2*i + 1] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::X8R8)
  CopyMemory(pOutData, pInData, Width*2);

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[1*i + 1]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[3*i + 2] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[1*i + 1]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((0) << 24);
    pTypedOutData[3*i + 2] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[1*i + 1];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[1*i + 1]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[3*i + 0] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[1*i + 1]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((0) << 24);
    pTypedOutData[3*i + 0] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::X8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[1*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[1*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
    pTypedOutData[2*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 1];
    pTypedOutData[2*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 1]) << 16);
    pTypedOutData[2*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
    pTypedOutData[2*i + 1] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[2*i + 1] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 1]) << 16);
    pTypedOutData[2*i + 1] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::X16R16)
  CopyMemory(pOutData, pInData, Width*4);

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[3*i + 2] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[1*i + 1]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((0) << 16);
    pTypedOutData[3*i + 2] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[3*i + 0] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[1*i + 1];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[1*i + 1]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((0) << 16);
    pTypedOutData[3*i + 0] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::X16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[1*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
    pTypedOutData[2*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
    pTypedOutData[2*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 1];
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
    pTypedOutData[2*i + 1] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
    pTypedOutData[2*i + 1] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::X32R32)
  CopyMemory(pOutData, pInData, Width*8);

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[3*i + 2] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[3*i + 2] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[3*i + 0] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[3*i + 0] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[1*i + 1];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::X32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[1*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[1*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 1]) * 255);
    pTypedOutData[2*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
    pTypedOutData[2*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 1];
    pTypedOutData[2*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 1]) * 255);
    pTypedOutData[2*i + 1] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
    pTypedOutData[2*i + 1] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[2*i + 1] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::X32R32F)
  CopyMemory(pOutData, pInData, Width*8);

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[1*i + 1]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[3*i + 2] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[3*i + 2] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[3*i + 2] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[3*i + 1] = 0.0f;
    pTypedOutData[3*i + 2] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 0.0f;
    pTypedOutData[4*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 2] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[1*i + 1]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[3*i + 0] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[3*i + 0] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[3*i + 0] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[1*i + 1];
    pTypedOutData[3*i + 1] = 0.0f;
    pTypedOutData[3*i + 0] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 0.0f;
    pTypedOutData[4*i + 3] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 1] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::X32R32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 1];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[1*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[2*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[2*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 1] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[2*i + 1] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[2*i + 1] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R8X8)
  CopyMemory(pOutData, pInData, Width*2);

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[3*i + 2] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((0) << 24);
    pTypedOutData[3*i + 2] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[3*i + 0] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((0) << 24);
    pTypedOutData[3*i + 0] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((0) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((0) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((0) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((0) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((0) << 24);
    pTypedOutData[4*i + 0] = (UINT)((0) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[2*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[2*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[2*i + 1] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 1] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[2*i + 1] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R16X16)
  CopyMemory(pOutData, pInData, Width*4);

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[3*i + 2] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((0) << 16);
    pTypedOutData[3*i + 2] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[3*i + 0] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((0) << 16);
    pTypedOutData[3*i + 0] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((0) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((0) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((0) << 16);
    pTypedOutData[4*i + 0] = (UINT)((0) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[2*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[2*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[2*i + 1] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[2*i + 1] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R32X32)
  CopyMemory(pOutData, pInData, Width*8);

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[3*i + 2] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[3*i + 2] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 3] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 2] = 0;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[3*i + 0] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[3*i + 0] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0;
    pTypedOutData[3*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0;
    pTypedOutData[4*i + 1] = 0;
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((0) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((0) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((0) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((0) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0;
    pTypedOutData[4*i + 0] = 0;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[1*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((0) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((0) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[1*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[2*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[2*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[2*i + 1] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[2*i + 1] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[2*i + 1] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[1*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[1*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[1*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R32FX32)
  CopyMemory(pOutData, pInData, Width*8);

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[3*i + 2] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[3*i + 2] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[3*i + 2] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0.0f;
    pTypedOutData[3*i + 2] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 0.0f;
    pTypedOutData[4*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 3] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 2] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[3*i + 0] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[3*i + 0] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[3*i + 0] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[3*i + 1] = 0.0f;
    pTypedOutData[3*i + 0] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 0.0f;
    pTypedOutData[4*i + 3] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 2] = 0.0f;
    pTypedOutData[4*i + 1] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[1*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((0.0f) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((0.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[1*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((0.0f) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((0.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[1*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((0.0f) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((0.0f) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[1*i + 0];
    pTypedOutData[4*i + 1] = 0.0f;
    pTypedOutData[4*i + 0] = 0.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[2*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[2*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 1] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[2*i + 1] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[2*i + 1] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R8G8B8)
  CopyMemory(pOutData, pInData, Width*3);

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[2*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[2*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[2*i + 1] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 1] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[2*i + 1] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R16G16B16)
  CopyMemory(pOutData, pInData, Width*6);

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[2*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[2*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[2*i + 1] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[2*i + 1] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R32G32B32)
  CopyMemory(pOutData, pInData, Width*12);

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[3*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[2*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[2*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[2*i + 1] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[2*i + 1] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 1] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R32FG32FB32F)
  CopyMemory(pOutData, pInData, Width*12);

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[4*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[4*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[4*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[4*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[4*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::A8R8G8B8)
  CopyMemory(pOutData, pInData, Width*4);

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 3]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 3]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 3];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 3]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 3]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[4*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[4*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 3];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 3]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 3]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 3]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 3]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::A8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[4*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[4*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[4*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::A16R16G16B16)
  CopyMemory(pOutData, pInData, Width*8);

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 3]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 3];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 3]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[4*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 3];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 3]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 3]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::A16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[4*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::A32R32G32B32)
  CopyMemory(pOutData, pInData, Width*16);

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 3];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 3];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::A32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[4*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[4*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[4*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 3]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
  CopyMemory(pOutData, pInData, Width*16);

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 3]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 3];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[4*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 3]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 3];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 3]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::A32FR32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[4*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 0];
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 0]) << 8);
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 0]) << 24);
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 0];
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 0]) << 8);
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 24);
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[4*i + 0]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[4*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R8G8B8A8)
  CopyMemory(pOutData, pInData, Width*4);

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[4*i + 0]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[4*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[4*i + 0]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[4*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 2]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 2]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R8G8B8A8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 0];
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 0]) << 16);
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 0];
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 16);
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R16G16B16A16)
  CopyMemory(pOutData, pInData, Width*8);

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[4*i + 0]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[4*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 2]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R16G16B16A16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 0];
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 0];
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R32G32B32A32)
  CopyMemory(pOutData, pInData, Width*16);

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R32G32B32A32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[4*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[4*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[4*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[4*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 0]) * 255);
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 0];
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 0]) * 255);
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 0];
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[4*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[4*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[4*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[4*i + 0]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[4*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
  CopyMemory(pOutData, pInData, Width*16);

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[4*i + 0]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[4*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[4*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[3*i + 0] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 2]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 2];
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[4*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[4*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[4*i + 1]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[4*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[4*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[4*i + 1]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[4*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[4*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[4*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[4*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R32FG32FB32FA32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[4*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[4*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[4*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[3*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[3*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[2*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[2*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[2*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[2*i + 1] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[2*i + 1] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[2*i + 1] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 1]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 1]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 3]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 3]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 3]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 3]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 3]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 3]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::X8R8G8B8)
  CopyMemory(pOutData, pInData, Width*4);

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 3]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 3]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 3]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 3]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 3]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 3]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 3]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 3]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::X8R8G8B8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 3]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[3*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[2*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[2*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[2*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[2*i + 1] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[2*i + 1] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[2*i + 1] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 1]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 3]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 3]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 3]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 3]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 3]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 3]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::X16R16G16B16)
  CopyMemory(pOutData, pInData, Width*8);

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 3]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 3]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 3]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 3]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 3]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 3]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 3]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 3]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::X16R16G16B16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 3]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[2*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[2*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[2*i + 1] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[2*i + 1] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 3]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 3]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 3]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 3]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 3]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 3]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::X32R32G32B32)
  CopyMemory(pOutData, pInData, Width*16);

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 3]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 3]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 3]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 3]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 3]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 3]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 3]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 3]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::X32R32G32B32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 3]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[3*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[2*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[2*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[2*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[2*i + 1] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[2*i + 1] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[2*i + 1] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 1]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 1];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 3]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 3]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 3]) * 4294967295u);
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 3]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 3]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 3]) * 4294967295u);
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 3] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::X32R32FG32FB32F)
  CopyMemory(pOutData, pInData, Width*16);

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 3]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 3]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 3]) * 4294967295u);
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 3]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 3]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 3]) * 4294967295u);
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 3];
    pTypedOutData[4*i + 3] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 3]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 3]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 3]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::X32R32FG32FB32F)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 3];
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[2*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[2*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 1] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[2*i + 1] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[2*i + 1] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R8G8B8X8)
  CopyMemory(pOutData, pInData, Width*4);

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R8G8B8X8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[2*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[2*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[2*i + 1] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 1] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[2*i + 1] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R16G16B16X16)
  CopyMemory(pOutData, pInData, Width*8);

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R16G16B16X16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[2*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[2*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[2*i + 1] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[2*i + 1] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R32G32B32X32)
  CopyMemory(pOutData, pInData, Width*16);

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R32G32B32X32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[3*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[2*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[2*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[2*i + 1] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[2*i + 1] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[2*i + 1] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 0]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
  CopyMemory(pOutData, pInData, Width*16);

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 0] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) * 255);
    pTypedOutData[4*i + 3] = (BYTE)((1.0f) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
    pTypedOutData[4*i + 3] = (USHORT)((1.0f) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 3] = 1.0f;
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) * 255);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) * 255);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) * 255);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) * 65535u);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) * 65535u);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) * 65535u);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) * 4294967295u);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) * 4294967295u);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) * 4294967295u);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::R32FG32FB32FX32)
{
  FLOAT* pTypedInData = (FLOAT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[2*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[2*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[2*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[2*i + 1] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[2*i + 1] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[2*i + 1] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 2]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::B8G8R8)
  CopyMemory(pOutData, pInData, Width*3);

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 0] = 255;
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 3] = 255;
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
    pTypedOutData[4*i + 3] = (USHORT)((255) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
    pTypedOutData[4*i + 3] = (UINT)((255) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((255) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 2]) << 8);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) << 8);
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 0]) << 8);
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 24);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 24);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 24);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::B8G8R8)
{
  BYTE* pTypedInData = (BYTE*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 255.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 255.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 255.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[2*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[2*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[2*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[2*i + 1] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[2*i + 1] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[2*i + 1] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32R32G32B32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32FG32FB32F && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8X8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16X16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32X32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FX32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16 && InPixelFormat == PixelFormat::B16G16R16)
  CopyMemory(pOutData, pInData, Width*6);

if (OutPixelFormat == PixelFormat::B32G32R32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[3*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[3*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32F && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8B8G8R8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::A16B16G16R16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 0] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::A32B32G32R32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::A32FB32FG32FR32F && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8A8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
    pTypedOutData[4*i + 3] = (BYTE)((65535u) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16A16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 3] = 65535u;
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32A32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
    pTypedOutData[4*i + 3] = (UINT)((65535u) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FA32F && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((65535u) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8B8G8R8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::X16B16G16R16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::X32B32G32R32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::X32B32FG32FR32F && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::B8G8R8X8 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 2]) >> 8);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 8);
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 0]) >> 8);
  }
}

if (OutPixelFormat == PixelFormat::B16G16R16X16 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::B32G32R32X32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (UINT)((pTypedInData[3*i + 2]) << 16);
    pTypedOutData[4*i + 1] = (UINT)((pTypedInData[3*i + 1]) << 16);
    pTypedOutData[4*i + 0] = (UINT)((pTypedInData[3*i + 0]) << 16);
  }
}

if (OutPixelFormat == PixelFormat::B32FG32FR32FX32 && InPixelFormat == PixelFormat::B16G16R16)
{
  USHORT* pTypedInData = (USHORT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 2]) / 65535.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 65535.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 0]) / 65535.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R32F && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[1*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[2*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[2*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[2*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32F && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[2*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8A8 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[2*i + 1] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16A16 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[2*i + 1] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32A32 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[2*i + 1] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FA32F && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[2*i + 1] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::X32R32 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::X32R32F && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8X8 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16X16 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32X32 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = pTypedInData[3*i + 2];
  }
}

if (OutPixelFormat == PixelFormat::R32FX32 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[2*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[3*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[3*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[3*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[3*i + 2] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[3*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[3*i + 2] = pTypedInData[3*i + 0];
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32F && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[3*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[3*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[3*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::A8R8G8B8 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 0] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::A16R16G16B16 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 0] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::A32R32G32B32 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 3] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 0] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::A32FR32FG32FB32F && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 0] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::R8G8B8A8 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((4294967295u) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::R16G16B16A16 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  USHORT* pTypedOutData = (USHORT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (USHORT)((pTypedInData[3*i + 2]) >> 16);
    pTypedOutData[4*i + 1] = (USHORT)((pTypedInData[3*i + 1]) >> 16);
    pTypedOutData[4*i + 2] = (USHORT)((pTypedInData[3*i + 0]) >> 16);
    pTypedOutData[4*i + 3] = (USHORT)((4294967295u) >> 16);
  }
}

if (OutPixelFormat == PixelFormat::R32G32B32A32 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  UINT* pTypedOutData = (UINT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = pTypedInData[3*i + 2];
    pTypedOutData[4*i + 1] = pTypedInData[3*i + 1];
    pTypedOutData[4*i + 2] = pTypedInData[3*i + 0];
    pTypedOutData[4*i + 3] = 4294967295u;
  }
}

if (OutPixelFormat == PixelFormat::R32FG32FB32FA32F && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  FLOAT* pTypedOutData = (FLOAT*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 0] = (FLOAT)((pTypedInData[3*i + 2]) / 4294967295.0f);
    pTypedOutData[4*i + 1] = (FLOAT)((pTypedInData[3*i + 1]) / 4294967295.0f);
    pTypedOutData[4*i + 2] = (FLOAT)((pTypedInData[3*i + 0]) / 4294967295.0f);
    pTypedOutData[4*i + 3] = (FLOAT)((4294967295u) / 4294967295.0f);
  }
}

if (OutPixelFormat == PixelFormat::X8R8G8B8 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
  BYTE* pTypedOutData = (BYTE*)pOutData;
  for (UINT i=0; i<Width; i++)
  {
    pTypedOutData[4*i + 1] = (BYTE)((pTypedInData[3*i + 2]) >> 24);
    pTypedOutData[4*i + 2] = (BYTE)((pTypedInData[3*i + 1]) >> 24);
    pTypedOutData[4*i + 3] = (BYTE)((pTypedInData[3*i + 0]) >> 24);
  }
}

if (OutPixelFormat == PixelFormat::X16R16G16B16 && InPixelFormat == PixelFormat::B32G32R32)
{
  UINT* pTypedInData = (UINT*)pInData;
 