#ifndef VISUALIZATION_INCLUDED
#define VISUALIZATION_INCLUDED
#include <GL/glew.h>
#include <GL/glut.h>
#include <vector>
#include <Util/JPEG.h>
#include <UTIL/Array.h>
#include <UTIL/geometry.h>

#define KEY_UPARROW		101
#define KEY_DOWNARROW	103
#define KEY_LEFTARROW	100
#define KEY_RIGHTARROW	102
#define KEY_PGUP		104
#define KEY_PGDN		105
#define KEY_CTRL_C        3
#define KEY_BACK_SPACE    8
#define KEY_ENTER        13
#define KEY_ESC          27

template< class Real >
void HSV2RGB( const Real* _hsv , Real* rgb )
{
	Real hsv[] = { _hsv[0] , _hsv[1] , _hsv[2] };
	int i;
	Real f, p, q, t;
	if( hsv[1] == 0 )
	{
		rgb[0] = rgb[1] = rgb[2] = hsv[2];
		return;
	}
	else if( hsv[1]<0 )
	{
		fprintf( stderr , "[ERROR] Saturation can't be negative\n" );
		return;
	}
	while( hsv[0]<0 ) hsv[0] += 2. * M_PI;
	hsv[0] /= M_PI / 3.;
	i = (int)floor( hsv[0] );
	f = (Real)(hsv[0] - i);
	p = (Real)(hsv[2] * ( 1 - hsv[1] ));
	q = (Real)(hsv[2] * ( 1 - hsv[1] * f ));
	t = (Real)(hsv[2] * ( 1 - hsv[1] * ( 1 - f ) ));
	switch( i ) {
		case 0:  rgb[0] = hsv[2] , rgb[1] = t ,      rgb[2] = p      ; break;
		case 1:  rgb[0] = q ,      rgb[1] = hsv[2] , rgb[2] = p      ; break;
		case 2:  rgb[0] = p ,      rgb[1] = hsv[2] , rgb[2] = t      ; break;
		case 3:  rgb[0] = p ,      rgb[1] = q ,      rgb[2] = hsv[2] ; break;
		case 4:  rgb[0] = t ,      rgb[1] = p ,      rgb[2] = hsv[2] ; break;
		default: rgb[0] = hsv[2] , rgb[1] = p ,      rgb[2] = q      ;
	}
}

class Camera
{
	void _setRight( void )
	{
		right = Point3D< double >::CrossProduct( forward , up );
		right /= Length( right );
	}
	void rotatePoint( Point3D< double > axis , double angle , Point3D< double > center )
	{
		Point3D< double > p , r , f , u;
		Point3D< double > v[3];
		double c , s;
		double d[3];

		v[2] = axis/Length( axis );

		v[0] = Point3D< double >::CrossProduct( v[2] , Point3D< double >( 1 , 0 , 0 ) );
		if( Point3D< double >::SquareNorm( v[0] )<.001) v[0] = Point3D< double >::CrossProduct( v[2] , Point3D< double >( 0 , 1 , 0 ) );
		v[0] /= Length( v[0] );
		v[1] = Point3D< double >::CrossProduct( v[2] , v[0] );
		v[1] /= Length( v[1] );

		c = cos(angle);
		s = sin(angle);

		p = position-center;
		for( int j=0 ; j<3 ; j++ ) d[j] = Point3D< double >::Dot( p , v[j] );

		position = v[2]*d[2] + v[0]*(d[0]*c+d[1]*s) + v[1]*(-d[0]*s+d[1]*c) + center;

		for( int j=0 ; j<3 ; j++ )
		{
			r[j] = Point3D< double >::Dot(   right , v[j] );
			f[j] = Point3D< double >::Dot( forward , v[j] );
			u[j] = Point3D< double >::Dot(      up , v[j] );
		}

		r = v[2]*r[2]+v[0]*(r[0]*c+r[1]*s)+v[1]*(-r[0]*s+r[1]*c);
		f = v[2]*f[2]+v[0]*(f[0]*c+f[1]*s)+v[1]*(-f[0]*s+f[1]*c);
		u = v[2]*u[2]+v[0]*(u[0]*c+u[1]*s)+v[1]*(-u[0]*s+u[1]*c);

		forward	= f / Length(f);
		right	= r / Length(r);
		up		= u / Length(u);

		_setRight();
	}

public:
	Point3D< double > position , forward , up , right;

	Camera( void )
	{
		position = Point3D< double >( 0 , 0 , 0 );
		forward  = Point3D< double >( 0 , 0 , 1 );
		up       = Point3D< double >( 0 , 1 , 0 );
		_setRight();
	}
	Camera( Point3D< double > p , Point3D< double > f , Point3D< double > u )
	{
		position = p , forward = f , up = u;
		_setRight();
	}
	void draw( void )
	{
		glMatrixMode( GL_MODELVIEW );        
		glLoadIdentity();
		gluLookAt(
			position[0] ,position[1] , position[2] ,
			position[0]+forward[0] , position[1]+forward[1] , position[2]+forward[2] ,
			up[0] , up[1] , up[2]
		);
	}

	void translate( Point3D< double > t ){ position += t; }
	void rotateUp     ( double angle , Point3D< double > p=Point3D< double >() ){ rotatePoint( up      , angle , p ); }
	void rotateRight  ( double angle , Point3D< double > p=Point3D< double >() ){ rotatePoint( right   , angle , p ); }
	void rotateForward( double angle , Point3D< double > p=Point3D< double >() ){ rotatePoint( forward , angle , p ); }

	Point2D< double > project( Point3D< double > p , bool orthographic )
	{
		p -= position;
		double x = Point3D< double >::Dot( p , right ) , y = Point3D< double >::Dot( p , up ) , z = Point3D< double >::Dot( p , forward );
		if( orthographic ) return Point2D< double >( x , y );
		else               return Point2D< double >( x/z , y/1 );
	}
};

struct Visualization
{
	int frameCount , videoSampleRate;
	int screenWidth , screenHeight;
	void *font , *promptFont;
	int fontHeight , promptFontHeight;
	bool showHelp , showInfo , videoMode;
	int infoOffset;
	void (*promptCallBack)( Visualization* , const char* );
	char promptString[1024];
	int promptLength;
	char* snapshotName;
	bool flushImage;

	struct KeyboardCallBack
	{
		static const char* SpecialKeyNames[];
		static const char* SpecialKeyName( int key );

		char key;
		char prompt[1024];
		char description[1024];
		void (*callBackFunction)( Visualization* , const char* );
		bool special;
		Visualization* visualization;
		KeyboardCallBack( Visualization* visualization , int key , const char* description , void (*callBackFunction)( Visualization* , const char* ) , bool special=false )
		{
			this->special = special;
			this->visualization = visualization;
			this->key = key;
			strcpy( this->description , description );
			prompt[0] = 0;
			this->callBackFunction = callBackFunction;
		}
		KeyboardCallBack( Visualization* visualization , int key , const char* description , const char* prompt , void ( *callBackFunction )( Visualization* , const char* ) , bool special=false )
		{
			this->special = special;
			this->visualization = visualization;
			this->key = key;
			strcpy( this->description , description );
			strcpy( this->prompt , prompt );
			this->callBackFunction = callBackFunction;
		}
	};

	std::vector< KeyboardCallBack > keyboardCallBacks;
	struct InfoCallBack
	{
		void* param;
		void (*callBackFunction)( Visualization* , void* );
		InfoCallBack( void (*callBackFunction)( Visualization* , void* ) , void* param = NULL ){ this->param = param , this->callBackFunction = callBackFunction; }
	};
	std::vector< InfoCallBack > infoCallBacks;
	Visualization( void )
	{
		keyboardCallBacks.push_back( KeyboardCallBack( this , KEY_ESC    , "" , QuitCallBack ) );
		keyboardCallBacks.push_back( KeyboardCallBack( this , KEY_CTRL_C , "" , QuitCallBack ) );
		keyboardCallBacks.push_back( KeyboardCallBack( this , 'h' , "toggle help" , ToggleHelpCallBack ) );
		keyboardCallBacks.push_back( KeyboardCallBack( this , 'H' , "toggle info" , ToggleInfoCallBack ) );
#if !MINIMAL_UI
		keyboardCallBacks.push_back( KeyboardCallBack( this , 'I' , "save video" , "Ouput header" , SetVideoCallBack ) );
#endif // !MINIMAL_UI
		keyboardCallBacks.push_back( KeyboardCallBack( this , 'i' , "save frame buffer" , "Ouput image" , SetFrameBufferCallBack ) );
		frameCount = 0;
		videoSampleRate = 1;
		snapshotName = NULL;
		flushImage = false;
		videoMode = false;
		screenWidth = screenHeight = 512;
		font = GLUT_BITMAP_HELVETICA_12;
		fontHeight = 12;
		promptFont = GLUT_BITMAP_TIMES_ROMAN_24;
		promptFontHeight = 24;
		showInfo = true;
		showHelp = true;
		promptCallBack = NULL;
		strcpy( promptString , "" );
		promptLength = 0;
	}
	virtual void idle( void ){;}
	virtual void keyboardFunc( unsigned char key , int x , int y ){;}
	virtual void specialFunc( int key, int x, int y ){;}
	virtual void display( void ){;}
	virtual void mouseFunc( int button , int state , int x , int y ){;}
	virtual void motionFunc( int x , int y ){;}

	void Idle        ( void );
	void KeyboardFunc( unsigned char key , int x , int y );
	void SpecialFunc ( int key, int x, int y );
	void Display     ( void );
	void Reshape     ( int w , int h );
	void MouseFunc   ( int button , int state , int x , int y );
	void MotionFunc  ( int x , int y );

	static void           QuitCallBack( Visualization*   , const char* ){ exit( 0 ); }
	static void     ToggleInfoCallBack( Visualization* v , const char* ){ v->showInfo = !v->showInfo; }
	static void     ToggleHelpCallBack( Visualization* v , const char* ){ v->showHelp = !v->showHelp; }
	static void SetFrameBufferCallBack( Visualization* v , const char* prompt )
	{
		if( prompt )
		{
			v->snapshotName = new char[ strlen(prompt)+1 ];
			strcpy( v->snapshotName , prompt );
			v->flushImage = true;
		}
	}
	static void SetVideoCallBack( Visualization* v , const char* prompt )
	{
		if( prompt )
		{
			if( !strlen( prompt ) )
			{
				if( v->snapshotName ) delete[] v->snapshotName;
				v->snapshotName = NULL;
			}
			else
			{
				int position = -1;
				for( int i=0 ; i<strlen(prompt) ; i++ ) if( prompt[i]==':' )
				{
					if( position==-1 ) position = i;
					else
					{
						fprintf( stderr , "[WARNING] SetVideoCallBack: Multiple \':\' separators in: %s\n" , prompt );
						return;
					}
					if( position>0 && position<strlen(prompt) ) v->videoSampleRate = atoi( prompt+position+1 );
				}
				v->videoMode = true;
				v->snapshotName = new char[ strlen(prompt)+1 ];
				v->frameCount = 0;
				strcpy( v->snapshotName , prompt );
				if( position>0 ) v->snapshotName[position] = 0;
				v->flushImage = true;
			}
		}
	}

	static void WriteLeftString( int x , int y , void* font , const char* format , ... );
	static int StringWidth( void* font , const char* format , ... );
	void writeLeftString( int x , int y , const char* format , ... ) const;
	void writeRightString( int x , int y , const char* format , ... ) const;
	void addInfoString( const char* format , ... );

	void saveFrameBuffer( const char* fileName , int whichBuffer=GL_BACK );
};
struct VisualizationViewer
{
	static Visualization* visualization;
	static void Idle        ( void );
	static void KeyboardFunc( unsigned char key , int x , int y );
	static void SpecialFunc ( int key, int x, int y );
	static void Display     ( void );
	static void Reshape     ( int w , int h );
	static void MouseFunc   ( int button , int state , int x , int y );
	static void MotionFunc  ( int x , int y );
};
const char* Visualization::KeyboardCallBack::SpecialKeyNames[] = { "F1" , "F2" , "F3" , "F4" , "F5" , "F6" , "F7" , "F8" , "F9" , "F10" , "F11" , "F12" , "Left" , "Up" , "Right" , "Down" , "Page-Up" , "Page-Down" , "Home" , "End" , "Insert" };
const char* Visualization::KeyboardCallBack::SpecialKeyName( int key )
{
	switch( key )
	{
	case GLUT_KEY_F1:        return SpecialKeyNames[ 0];
	case GLUT_KEY_F2:        return SpecialKeyNames[ 1];
	case GLUT_KEY_F3:        return SpecialKeyNames[ 2];
	case GLUT_KEY_F4:        return SpecialKeyNames[ 3];
	case GLUT_KEY_F5:        return SpecialKeyNames[ 4];
	case GLUT_KEY_F6:        return SpecialKeyNames[ 5];
	case GLUT_KEY_F7:        return SpecialKeyNames[ 6];
	case GLUT_KEY_F8:        return SpecialKeyNames[ 7];
	case GLUT_KEY_F9:        return SpecialKeyNames[ 8];
	case GLUT_KEY_F10:       return SpecialKeyNames[ 9];
	case GLUT_KEY_F11:       return SpecialKeyNames[10];
	case GLUT_KEY_F12:       return SpecialKeyNames[11];
	case GLUT_KEY_LEFT:      return SpecialKeyNames[12];
	case GLUT_KEY_UP:        return SpecialKeyNames[13];
	case GLUT_KEY_RIGHT:     return SpecialKeyNames[14];
	case GLUT_KEY_DOWN:      return SpecialKeyNames[15];
	case GLUT_KEY_PAGE_UP:   return SpecialKeyNames[16];
	case GLUT_KEY_PAGE_DOWN: return SpecialKeyNames[17];
	case GLUT_KEY_HOME:      return SpecialKeyNames[18];
	case GLUT_KEY_END:       return SpecialKeyNames[19];
	case GLUT_KEY_INSERT:    return SpecialKeyNames[20];
	default: return NULL;
	}
}
Visualization* VisualizationViewer::visualization = NULL;
void VisualizationViewer::Idle( void ){ visualization->Idle(); }
void VisualizationViewer::KeyboardFunc( unsigned char key , int x , int y ){ visualization->KeyboardFunc( key , x , y ); }
void VisualizationViewer::SpecialFunc( int key , int x , int y ){ visualization->SpecialFunc( key , x ,  y ); }
void VisualizationViewer::Display( void ){ visualization->Display(); }
void VisualizationViewer::Reshape( int w , int h ){ visualization->Reshape( w , h ); }
void VisualizationViewer::MouseFunc( int button , int state , int x , int y ){ visualization->MouseFunc( button , state , x , y ); }
void VisualizationViewer::MotionFunc( int x , int y ){ visualization->MotionFunc( x , y ); }
void Visualization::Reshape( int w , int h )
{
	screenWidth = w , screenHeight = h;
	glViewport( 0 , 0 , screenWidth , screenHeight );
}
void Visualization::MouseFunc( int button , int state , int x , int y ){ mouseFunc( button , state , x , y ); }
void Visualization::MotionFunc( int x , int y ){ motionFunc( x , y );}
void Visualization::Idle( void )
{
	if( !promptCallBack )
	{
		if( snapshotName )
		{
			if( flushImage )
			{
				flushImage = false;
				glutPostRedisplay();
				return;
			}
			else
			{
				if( videoMode )
				{
					if( !(frameCount % videoSampleRate ) )
					{
						char fileName[512];
						sprintf( fileName, "%s.%d.jpg" , snapshotName , frameCount );
						saveFrameBuffer( fileName , GL_FRONT );
					}
				}
				else
				{
					saveFrameBuffer( snapshotName , GL_FRONT );
					delete[] snapshotName;
					snapshotName = NULL;
				}
			}
		}
		frameCount++;
		idle();
	}
}
void Visualization::KeyboardFunc( unsigned char key , int x , int y )
{
	if( promptCallBack )
	{
		size_t len = strlen( promptString );
		if( key==KEY_BACK_SPACE )
		{
			if( len>promptLength ) promptString[len-1] = 0;
		}
		else if( key==KEY_ENTER )
		{
			promptCallBack( this , promptString+promptLength );
			promptString[0] = 0;
			promptLength = 0;
			promptCallBack = NULL;
		}
		else if( key==KEY_CTRL_C )
		{
			promptString[0] = 0;
			promptLength = 0;
			promptCallBack = NULL;
		}
		else if( key>=32 && key<=126 ) // ' ' to '~'
		{
			promptString[ len ] = key;
			promptString[ len+1 ] = 0;
		}
		glutPostRedisplay();
		return;
	}
	switch( key )
	{
	case KEY_CTRL_C:
		exit( 0 );
		break;
	default:
		for( int i=0 ; i<keyboardCallBacks.size() ; i++ ) if( !keyboardCallBacks[i].special && keyboardCallBacks[i].key==key )
		{
			if( strlen( keyboardCallBacks[i].prompt ) )
			{
				sprintf( promptString , "%s: " , keyboardCallBacks[i].prompt );
				promptLength = int( strlen( promptString ) );
				promptCallBack = keyboardCallBacks[i].callBackFunction;
			}
			else (*keyboardCallBacks[i].callBackFunction)( this , NULL );
			break;
		}
	}
	keyboardFunc( key , x , y );
	glutPostRedisplay();
}

void Visualization::SpecialFunc( int key , int x , int y )
{
	for( int i=0 ; i<keyboardCallBacks.size() ; i++ ) if( keyboardCallBacks[i].special && keyboardCallBacks[i].key==key )
	{
		if( strlen( keyboardCallBacks[i].prompt ) )
		{
			sprintf( promptString , "%s: " , keyboardCallBacks[i].prompt );
			promptLength = int( strlen( promptString ) );
			promptCallBack = keyboardCallBacks[i].callBackFunction;
		}
		else (*keyboardCallBacks[i].callBackFunction)( this , NULL );
	}
	specialFunc( key , x , y );
}
void Visualization::Display( void )
{
	glClearColor( 1 , 1 , 1 , 1 );
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

	display();

	int offset = fontHeight/2;
	if( showHelp )
	{
		int y = offset;
		for( int i=0 ; i<keyboardCallBacks.size() ; i++ ) if( strlen( keyboardCallBacks[i].description ) )
			if( !keyboardCallBacks[i].special )
				if( keyboardCallBacks[i].key!=9 ) writeRightString( 10 , y , "\'%c\': %s" , keyboardCallBacks[i].key , keyboardCallBacks[i].description ) , y += fontHeight + offset;
				else                              writeRightString( 10 , y , "\"tab\": %s" , keyboardCallBacks[i].description ) , y += fontHeight + offset;
			else writeRightString( 10 , y , "\"%c\": %s" , KeyboardCallBack::SpecialKeyName( keyboardCallBacks[i].key ) , keyboardCallBacks[i].description ) , y += fontHeight + offset;
	}
	if( showInfo )
	{
		infoOffset = fontHeight/2;
		for( int i=0 ; i<infoCallBacks.size() ; i++ ) infoCallBacks[i].callBackFunction( this , infoCallBacks[i].param );
	}
	if( strlen( promptString ) )
	{
		void* _font = font;
		int _fontHeight = fontHeight;
		font = promptFont;
		fontHeight = promptFontHeight;

		int sw = StringWidth ( font , promptString );
		glColor4f( 1.f , 1.f , 1.f , 0.5 );
		glEnable( GL_BLEND );
		glBlendFunc( GL_SRC_ALPHA , GL_ONE_MINUS_SRC_ALPHA );
		glBegin( GL_QUADS );
		{
			glVertex2f(     0 , screenHeight              );
			glVertex2f( sw+20 , screenHeight              );
			glVertex2f( sw+20 , screenHeight-fontHeight*2 );
			glVertex2f(     0 , screenHeight-fontHeight*2 );
		}
		glEnd();
		glDisable( GL_BLEND );
		glColor4f( 0 , 0 , 0 , 1 );
		glLineWidth( 2.f );
		glBegin( GL_LINE_LOOP );
		{
			glVertex2f(     0 , screenHeight              );
			glVertex2f( sw+20 , screenHeight              );
			glVertex2f( sw+20 , screenHeight-fontHeight*2 );
			glVertex2f(     0 , screenHeight-fontHeight*2 );
		}
		glEnd();
		writeLeftString( 10 , screenHeight-fontHeight-fontHeight/2 , promptString );
		font = _font;
		fontHeight = _fontHeight;
	}
	glutSwapBuffers();
}

void Visualization::WriteLeftString( int x , int y , void* font , const char* format , ... )
{
	static char str[1024];
	{
		va_list args;
		va_start( args , format );
		vsprintf( str , format , args );
		va_end( args );
	}

	GLint vp[4];

	glGetIntegerv( GL_VIEWPORT , vp );
	glPushMatrix();
	glLoadIdentity();
	glMatrixMode( GL_PROJECTION );
	glPushMatrix();
	glLoadIdentity();
	glOrtho( vp[0] , vp[2] , vp[1] , vp[3] , 0 , 1 );

	glDisable( GL_DEPTH_TEST );
	glDisable( GL_LIGHTING );
	glColor4f( 0 , 0 , 0 , 1 );
	glRasterPos2f( x , y  );
	int len = int( strlen( str ) );
	for( int i=0 ; i<len ; i++ ) glutBitmapCharacter( font , str[i] );
	glPopMatrix();
	glMatrixMode( GL_MODELVIEW );
	glPopMatrix();
}
void Visualization::addInfoString( const char* format , ... )
{
	static char str[1024];
	{
		va_list args;
		va_start( args , format );
		vsprintf( str , format , args );
		va_end( args );
	}
	writeLeftString( 10 , infoOffset , "%s" , str ) , infoOffset += (3*fontHeight)/2;
}

int Visualization::StringWidth( void* font , const char* format , ... )
{
	static char str[1024];
	{
		va_list args;
		va_start( args , format );
		vsprintf( str , format , args );
		va_end( args );
	}
	return glutBitmapLength( font , (unsigned char*) str );
}
void Visualization::writeLeftString( int x , int y , const char* format , ... ) const
{
	static char str[1024];
	{
		va_list args;
		va_start( args , format );
		vsprintf( str , format , args );
		va_end( args );
	}
	WriteLeftString( x , y , font , str );
}
void Visualization::writeRightString( int x , int y , const char* format , ... ) const
{
	static char str[1024];
	{
		va_list args;
		va_start( args , format );
		vsprintf( str , format , args );
		va_end( args );
	}
	WriteLeftString( screenWidth-x-glutBitmapLength( font , (unsigned char*) str ) , y , font , str );
}
void Visualization::saveFrameBuffer( const char* fileName , int whichBuffer )
{
	Pointer( float ) pixels = AllocPointer< float >( sizeof(float) * 3 * screenWidth * screenHeight );
	Pointer( unsigned char ) _pixels = AllocPointer< unsigned char >( sizeof(unsigned char) * 3 * screenWidth * screenHeight );
	glReadBuffer( whichBuffer );
	glReadPixels( 0 , 0 , screenWidth , screenHeight , GL_RGB , GL_FLOAT , pixels );
	for( int j=0 ; j<screenHeight ; j++ ) for( int i=0 ; i<screenWidth ; i++ ) for( int c=0 ; c<3 ; c++ )
	{
		int ii = int( pixels[ c + i * 3 + ( screenHeight - 1 - j ) * screenWidth * 3 ]*256 );
		if( ii<  0 ) ii =   0;
		if( ii>255 ) ii = 255;
		_pixels[ c + i * 3 + j * screenWidth * 3 ] = (unsigned char)ii;
	}
	FreePointer( pixels );
	JPEGWriteColor( fileName , _pixels , screenWidth , screenHeight , 95 );
	FreePointer( _pixels );

	printf( "Wrote image to: %s\n" , fileName );
}

#endif // VISUALIZATION_INCLUDED