#include <Util/SoRWaveSimulation.h>
#include <Visualization/Visualization.h>
#ifdef _WIN32
#include <Windows.h>
#include <Psapi.h>
#endif // _WIN32

#define USE_VBO 1

template< class Real >
struct SoRWaveVisualization : public Visualization
{
protected:
	std::vector< Point3D< float > > _verticesNormalsAndColors;
	Point3D< float > *_vertices , *_normals , *_colors;
	std::vector< float > _cumulativeAreas;
public:
	static const int SAMPLE_SPACING = 1;
	enum
	{
		SELECT_UP ,
		SELECT_DOWN ,
	};

	int threads;
	int steps;
	float stepSize , waveSpeed , dampingFactor , elasticity , dropWidth , dropHeight;
	RegularGridFEM::GridType gridType;
	int advanceCount;
	double runningTime , fps , time;
	std::vector< std::pair< int , int > > selectionPoints;
	std::vector< std::pair< Point3D< float > , float > > sourcePoints;
	float sourceFrequency;
	bool useRain;
	std::vector< TriangleIndex > triangles;
	std::vector< Point3D< float > > vertices , normals;
	Camera camera;
	float zoom;
#if USE_VBO
	GLuint vbo , ebo;
#endif // USE_VBO
	GLfloat lightAmbient[4] , lightDiffuseF[4] , lightDiffuseB[4] , lightSpecular[4] , shapeSpecular[4] , shapeSpecularShininess;
	int oldX , oldY , newX , newY;
	float imageZoom , imageOffset[2];
	bool rotating , scaling , selecting;
	int count;
	bool useLight , showEdges , showBoundary;
	bool showColor;
	int resX , resY;
	SoRWaveSimulation< Real >* waveSimulation;

	SoRWaveVisualization( void );
#if USE_DIRECT_SOLVER
	void init( int resX , int resY , RegularGridFEM::GridType gridType , ConstPointer( Point2D <double > ) curve , bool conicalGeometry , bool useDirect , Real theta );
#else // !USE_DIRECT_SOLVER
	void init( int resX , int resY , RegularGridFEM::GridType gridType , ConstPointer( Point2D <double > ) curve , bool conicalGeometry , Real theta );
#endif // USE_DIRECT_SOLVER
	void getSelectionPoints( const std::vector< std::pair< int , int > >& inPoints , std::vector< Point3D< float > >& outPoints , float spacing );

	void idle( void );
	void keyboardFunc( unsigned char key , int x , int y );
	void specialFunc( int key, int x, int y );
	void initMesh( void );
	Point3D< float > pointSelect( float x , float y );
	void pointSelect( const std::vector< std::pair< float , float > >& screenPoints , std::vector< Point3D< float > >& worldPoints );
	void display( void );
	void mouseFunc( int button , int state , int x , int y );
	void motionFunc( int x , int y );

	static void SetInfoCallBack( Visualization* v , void* param );
	static void           ResetCallBack( Visualization*v , const char* ){ ( (SoRWaveVisualization*)v)->waveSimulation->reset() , ( (SoRWaveVisualization*)v)->count = 0 , ( (SoRWaveVisualization*)v)->sourcePoints.clear(); }
	static void    ResetSourcesCallBack( Visualization*v , const char* ){ ( (SoRWaveVisualization*)v)->waveSimulation->resetSources() , ( (SoRWaveVisualization*)v)->sourcePoints.clear(); }
	static void      ToggleRainCallBack( Visualization*v , const char* ){ ( (SoRWaveVisualization*)v)->useRain = !( (SoRWaveVisualization*)v)->useRain; }
	static void     ToggleColorCallBack( Visualization*v , const char* ){ ( (SoRWaveVisualization*)v)->showColor = !( (SoRWaveVisualization*)v)->showColor; }
	static void     ToggleLightCallBack( Visualization*v , const char* ){ ( (SoRWaveVisualization*)v)->useLight = !( (SoRWaveVisualization*)v)->useLight; }
	static void     ToggleEdgesCallBack( Visualization*v , const char* ){ ( (SoRWaveVisualization*)v)->showEdges = !( (SoRWaveVisualization*)v)->showEdges; }
	static void  ToggleBoundaryCallBack( Visualization*v , const char* ){ ( (SoRWaveVisualization*)v)->showBoundary = !( (SoRWaveVisualization*)v)->showBoundary; }
	static void     TogglePauseCallBack( Visualization*v , const char* )
	{
		SoRWaveVisualization *_v = (SoRWaveVisualization*)v;
		if     ( _v->advanceCount==0 ) _v->advanceCount = -1;
		else if( _v->advanceCount< 0 ) _v->advanceCount =  0;
	}
	static void        AdvanceCallBack( Visualization* v , const char* ){ if( ( (SoRWaveVisualization*)v )->advanceCount>=0 ) ( (SoRWaveVisualization*)v )->advanceCount++; }
	// Call-backs that take an argument
	static void    AdvanceMultipleCallBack( Visualization* v , const char* prompt )
	{
		SoRWaveVisualization* _v = (SoRWaveVisualization*)v;
		if( _v->advanceCount>=0 )
		{
			_v->advanceCount += atoi(prompt);
			if( _v->advanceCount<0 ) _v->advanceCount = 0;
		}
	}
	static void      SetDropHeightCallBack( Visualization* v , const char* prompt ){ ( (SoRWaveVisualization*)v )->dropHeight = (float)atof( prompt ); }
	static void       SetDropWidthCallBack( Visualization* v , const char* prompt ){ ( (SoRWaveVisualization*)v )->dropWidth = (float)atof( prompt ); }
	static void SetSourceFrequencyCallBack( Visualization* v , const char* prompt ){ ( (SoRWaveVisualization*)v )->sourceFrequency = ( (SoRWaveVisualization*)v )->waveSimulation->sourceFrequency = atof( prompt ); }
	static void      SetElasticityCallBack( Visualization* v , const char* prompt )
	{
		SoRWaveVisualization* _v = (SoRWaveVisualization*)v;
		_v->elasticity = (float)atof( prompt );
		_v->waveSimulation->resetParameters( _v->stepSize , _v->waveSpeed , _v->dampingFactor , _v->elasticity );
	}
	static void   SetDampingFactorCallBack( Visualization* v , const char* prompt )
	{
		SoRWaveVisualization* _v = (SoRWaveVisualization*)v;
		_v->dampingFactor = (float)atof( prompt );
		_v->waveSimulation->resetParameters( _v->stepSize , _v->waveSpeed , _v->dampingFactor , _v->elasticity );
	}
	static void       SetWaveSpeedCallBack( Visualization* v , const char* prompt )
	{
		SoRWaveVisualization* _v = (SoRWaveVisualization*)v;
		_v->waveSpeed = (float)atof( prompt );
		_v->waveSimulation->resetParameters( _v->stepSize , _v->waveSpeed , _v->dampingFactor , _v->elasticity );
	}
	static void    SetStepSizeCallBack( Visualization*v , const char* prompt )
	{
		SoRWaveVisualization* _v = (SoRWaveVisualization*)v;
		_v->stepSize = (float)atof(prompt);
		_v->waveSimulation->resetParameters( _v->stepSize , _v->waveSpeed , _v->dampingFactor , _v->elasticity );
	}
	static void       SetStepsCallBack( Visualization*v , const char* prompt ){ ( (SoRWaveVisualization*)v )->steps = atoi( prompt ); }


	bool setPosition( int x , int y , Point3D< double >& p );
	bool setPosition( int x , int y , Point3D< float >& p );
};
template< class Real >
SoRWaveVisualization< Real >::SoRWaveVisualization( void )
{
	showColor = true;
	useRain = false;
	advanceCount = -1;
	threads = 1;
	steps = 1;
	stepSize = 1.f;
	waveSpeed = 1.f;
	dampingFactor = 0.f;
	elasticity = 0.f;
	waveSimulation = NULL;
	resX = resY = 0;
	runningTime=0. , fps=0. , time=0;
	zoom = 1.05f;
	lightAmbient [0] = lightAmbient [1] = lightAmbient [2] = 0.25f , lightAmbient [3] = 1.f;
	lightDiffuseF[0] = lightDiffuseF[1] = lightDiffuseF[2] = 0.70f , lightDiffuseF[3] = 1.f;
	lightDiffuseB[0] = lightDiffuseB[1] = lightDiffuseB[2] = 0.35f , lightDiffuseB[3] = 1.f;
	lightSpecular[0] = lightSpecular[1] = lightSpecular[2] = 1.00f , lightSpecular[3] = 1.f;
	shapeSpecular[0] = shapeSpecular[1] = shapeSpecular[2] = 1.00f , shapeSpecular[3] = 1.f;
	shapeSpecularShininess = 128.;
	oldX , oldY , newX , newY;
	imageZoom = 1.f , imageOffset[0] = imageOffset[1] = 0.f;
	sourceFrequency = 0.1f;
	rotating = false , scaling = false , selecting = false;
	count = 0;
	useLight = true;
	showEdges = false;
	showBoundary = true;

	keyboardCallBacks.push_back( KeyboardCallBack( this , 'e' , "toggle edges" , ToggleEdgesCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'b' , "toggle boundary" , ToggleBoundaryCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'C' , "toggle color" , ToggleColorCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'L' , "toggle light" , ToggleLightCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'r' , "toggle rain" , ToggleRainCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '+' , "advanace" , AdvanceCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '=' , "advanace multiple" , "Advance Steps" , AdvanceMultipleCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , ' ' , "toggle pause" , TogglePauseCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '4' , "set elasticity" , "Elasticity" , SetElasticityCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '3' , "set damping factor" , "Damping factor" , SetDampingFactorCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '2' , "set wave speed" , "Wave speed" , SetWaveSpeedCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '1' , "set step size" , "Step size" , SetStepSizeCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '0' , "set steps" , "Steps" , SetStepsCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'f' , "reset sources" , ResetSourcesCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'F' , "set source frequency" , "Source frequency" , SetSourceFrequencyCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'P' , "set drop width" , "Drop width" , SetDropWidthCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'p' , "set drop height" , "Drop height" , SetDropHeightCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'R' , "reset" , ResetCallBack ) );
	infoCallBacks.push_back( InfoCallBack( SetInfoCallBack , this ) );
}
template< class Real >
bool SoRWaveVisualization< Real >::setPosition( int x , int y , Point3D< double >& p )
{
	double _x =(double)x / screenWidth - 0.5 , _y = 1. - (double)y/screenHeight - 0.5;
	_x *= 2. , _y *= 2;
	_x *= zoom , _y *= zoom;
	double r = _x*_x + _y*_y;
	if( r<1 ){	p = camera.forward * ( -sqrt( 1-r ) ) + camera.right * _x + camera.up * _y; return true; }
	return false;
}
template< class Real >
bool SoRWaveVisualization< Real >::setPosition( int x , int y , Point3D< float >& p )
{
	Point3D< double > _p;
	bool ret = setPosition( x , y , _p );
	p = Point3D< float >( (float)_p[0] , (float)_p[1] , (float)_p[2] );
	return ret;
}

template< class Real >
#if USE_DIRECT_SOLVER
void SoRWaveVisualization< Real >::init( int rX , int rY , RegularGridFEM::GridType gridType , ConstPointer( Point2D <double > ) curve , bool conicalGeometry , bool useDirect , Real theta )
#else // !USE_DIRECT_SOLVER
void SoRWaveVisualization< Real >::init( int rX , int rY , RegularGridFEM::GridType gridType , ConstPointer( Point2D <double > ) curve , bool conicalGeometry , Real theta )
#endif // USE_DIRECT_SOLVER
{
	this->gridType = gridType;
	if( waveSimulation ) delete waveSimulation;
#if USE_DIRECT_SOLVER
	waveSimulation = new SoRWaveSimulation< Real >( rX , rY , gridType , curve , conicalGeometry , theta , useDirect , stepSize , waveSpeed , dampingFactor , elasticity , threads );
#else // !USE_DIRECT_SOLVER
	waveSimulation = new SoRWaveSimulation< Real >( rX , rY , gridType , curve , conicalGeometry , theta , stepSize , waveSpeed , dampingFactor , elasticity , threads );
#endif // USE_DIRECT_SOLVER
	waveSimulation->sorParam->resolution( resX , resY );
	waveSimulation->sourceFrequency = sourceFrequency;
	_verticesNormalsAndColors.resize( 0 );
	_vertices = _normals = _colors = NULL;
	vertices.resize( 0 ) , normals.resize( 0 );
}

template< class Real >
void SoRWaveVisualization< Real >::mouseFunc( int button , int state , int x , int y )
{
	newX = x ; newY = y;

	rotating = scaling = selecting = false;
	if( glutGetModifiers()!=GLUT_ACTIVE_CTRL && state==GLUT_DOWN )
	{
		if( button==GLUT_LEFT_BUTTON  ) selectionPoints.clear() , selectionPoints.push_back( std::pair< int , int >(x,y) ) , selecting = true;
		else if( button==GLUT_RIGHT_BUTTON )
		{
			Point3D< float > p = pointSelect( (float)x , (float)y );
			{
				int nearestI = 0 , nearestJ = 0;
				float nearestL = -1;
				int bands = waveSimulation->sorParam->bands();
				for( int i=0 ; i<resX ; i++ ) for( int j=0 ; j<bands ; j++ )
				{
					Point3D< float > q = waveSimulation->sorParam->template position< float >( i+0.5 , j+0.5 );
					float l = (p-q).squareNorm();
					if( nearestL<0 || l<nearestL ) nearestL = l , nearestI = i , nearestJ = j;
				}
				p = waveSimulation->sorParam->template position< float >( nearestI+0.5 , nearestJ+0.5 );
			}
			sourcePoints.push_back( std::pair< Point3D< float > , float >( p , dropHeight*64 ) );
		}
	}
	else if( button==GLUT_LEFT_BUTTON  ) rotating = true;
	else if( button==GLUT_RIGHT_BUTTON ) scaling = true;
}
template< class Real >
void SoRWaveVisualization< Real >::motionFunc( int x , int y )
{
	oldX = newX , oldY = newY , newX = x , newY = y;

	int imageSize = std::min< int >( screenWidth , screenHeight );
	float rel_x = (newX - oldX) / (float)imageSize * 2;
	float rel_y = (newY - oldY) / (float)imageSize * 2;
	float pRight = -rel_x , pUp = rel_y;
	float sForward = rel_y*4;
	float rRight = rel_y , rUp = rel_x;

	if     ( rotating ) camera.rotateUp( rUp ) , camera.rotateRight( rRight );
	else if( scaling  ) zoom *= (float)pow( 0.9 , (double)sForward );
	else if( selecting ) selectionPoints.push_back( std::pair< int , int >(x,y) );
	glutPostRedisplay();
}

template< class Real >
void SoRWaveVisualization< Real >::getSelectionPoints( const std::vector< std::pair< int , int > >& inPoints , std::vector< Point3D< float > >& outPoints , float spacing )
{
	std::vector< std::pair< float , float > > newSelectionPoints;
	int sz = (int)inPoints.size();
	for( int i=0 ; i<sz-1 ; i++ )
	{
		double x0 = selectionPoints[i?(i-1):i].first , y0 = selectionPoints[i?(i-1):i].second;
		double x1 = selectionPoints[i  ].first , y1 = selectionPoints[i  ].second;
		double x2 = selectionPoints[i+1].first , y2 = selectionPoints[i+1].second;
		double x3 = selectionPoints[(i<sz-2)?(i+2):(i+1)].first , y3 = selectionPoints[(i<sz-2)?(i+2):(i+1)].second;
		double l = (x1-x2)*(x1-x2) + (y1-y2)*(y1-y2);
		{
			l = sqrt(l);
			int steps = (int)ceil( l/spacing )+1;
			for( int j=0 ; j<steps ; j++ )
			{
				double t = ((float)j) / steps;
				double a0 = -1./6 * t * t * t + 1./2 * t * t - 1./2 * t + 1./6;
				double a1 =  1./2 * t * t * t -        t * t            + 2./3;
				double a2 = -1./2 * t * t * t + 1./2 * t * t + 1./2 * t + 1./6;
				double a3 =  1./6 * t * t * t;
				double x = x0*a0 + x1*a1 + x2*a2 + x3*a3;
				double y = y0*a0 + y1*a1 + y2*a2 + y3*a3;
				newSelectionPoints.push_back( std::pair< float , float >( (float)x , (float)y ) );
			}
		}
	}
	newSelectionPoints.push_back( std::pair< float , float >( (float)selectionPoints.back().first , (float)selectionPoints.back().second ) );
	pointSelect( newSelectionPoints , outPoints );
}
template< class Real >
void SoRWaveVisualization< Real >::idle( void )
{
	if( !promptCallBack )
	{
		time = Time();
		int resX = waveSimulation->resX , resY = waveSimulation->resY;
		int bands = waveSimulation->sorParam->bands();
		int dimX = waveSimulation->sorParam->gridType().xPeriodic() ? resX : resX-1;
		static std::vector< Real > _scratch;
		_scratch.resize( bands * dimX );
		if( useRain )
		{
			if( _cumulativeAreas.size() && advanceCount )
			{
				unsigned int fCount = waveSimulation->sorParam->faces();
				memset( &_scratch[0] , 0 , sizeof(Real) * _scratch.size() );
				Point3D< float > selectedPoint;
				{
					double r = Random< double >();
					unsigned int fMin = 0 , fMax = fCount-1 , fIdx = ( fMax + fMin ) / 2;
					while( fMax-fMin>1 )
					{
						if( _cumulativeAreas[fIdx]<r ) fMin = fIdx;
						else fMax = fIdx;
						fIdx = ( fMax + fMin ) / 2;
					}
					unsigned int x , y;
					waveSimulation->sorParam->faceIndices( fIdx , x , y );
					selectedPoint = waveSimulation->sorParam->template position< Real >( (Real)( x+0.5 ) , (Real)( y+0.5 ) );
				}
				float cutOff = dropWidth;
				float cutOff2 = cutOff * cutOff;
				Real height = dropHeight;
#pragma omp parallel for num_threads( threads )
				for( int j=0 ; j<bands ; j++ ) for( int i=0 ; i<dimX ; i++ )
				{
					Point3D< float > p = waveSimulation->sorParam->template position< float >( i+0.5 , j+0.5 );
					_scratch[j*dimX+i] = (Real)exp( -(p-selectedPoint).squareNorm() / (cutOff*cutOff) ) * height;
				}
				waveSimulation->addHeightOffset( GetPointer( _scratch ) );
			}
		}
		else if( selectionPoints.size() && !selecting )
		{
			float cutOff = dropWidth;
			float cutOff2 = cutOff * cutOff;
			std::vector< Point3D< float > > points;
			getSelectionPoints( selectionPoints , points , SAMPLE_SPACING );
			if( points.size() )
			{
				Point3D< float > dirToCamera = Point3D< float >( camera.forward );
				int directionSign=1;

				int nearestI = 0 , nearestJ = 0;
				if( points.size() )
				{
					float nearestL = -1;
					for( int i=0 ; i<dimX ; i++ ) for( int j=0 ; j<bands ; j++ )
					{
						Point3D< float > p = waveSimulation->sorParam->template position< float >( i+0.5 , j+0.5 );
						float l = (p-points[0]).squareNorm();
						if( nearestL<0 || l<nearestL ) nearestL = l , nearestI = i , nearestJ = j;
					}
					directionSign = Point3D< float >::Dot( dirToCamera , waveSimulation->sorParam->template normal< float >( nearestI+0.5 , nearestJ+0.5 ) ) > 0 ? 1 : -1;
				}
				selectionPoints.clear();

				memset( &_scratch[0] , 0 , sizeof(Real) * _scratch.size() );

				std::vector< Point3D< float > > tangents;
				if( points.size()>1 ) tangents.resize( points.size()-1 );
				for( int i=1 ; i<points.size() ; i++ )
				{
					Point3D< float > d = points[i] - points[i-1];
					tangents[i-1] = Point3D< float >( d / Length(d) );
				}
//				Real height = dropHeight;
				Real height = dropHeight * 4;
#pragma omp parallel for num_threads( threads )
				for( int j=0 ; j<bands ; j++ ) for( int i=0 ; i<dimX ; i++ )
				{
					int idx = -1;
					double minDist2;
					Point3D< float > p = waveSimulation->sorParam->template position< float >( i+0.5 , j+0.5 );
					Point3D< float > n = waveSimulation->sorParam->template normal  < float >( i+0.5 , j+0.5 );
					bool visible = ( Point3D< float >::Dot( n , dirToCamera )*directionSign )>0;

					for( int k=0 ; k<points.size()-1 ; k++ )
					{
						Point3D< float > d1 = p - points[k] , d2 = p-points[k+1];
						float dist2;
						if( Point3D< float >::Dot( d1 , tangents[k] )<0 ) dist2 = d1.squareNorm();
						else if( Point3D< float >::Dot( d2 , tangents[k] )>0 ) dist2 = d2.squareNorm();
						else dist2 = ( d1 - tangents[k] * Point3D< float >::Dot( d1 , tangents[k] ) ).squareNorm();
						if( idx==-1 || dist2<minDist2 ) minDist2 = dist2 , idx = k;
					}
					if( points.size()==1 ) minDist2 = ( p - points[0] ).squareNorm() , idx = 0 , visible = true;
					if( visible ) _scratch[j*dimX+i] = (Real)exp( -minDist2 / cutOff2 ) * height;
					else _scratch[j*dimX+i] = (Real)0.;
				}
				waveSimulation->addHeightOffset( GetPointer( _scratch ) );
			}
		}
		if( sourcePoints.size() )
		{
			float cutOff = dropWidth;
			float cutOff2 = cutOff * cutOff;
			Point3D< float > dirToCamera = Point3D< float >( camera.forward );
			memset( &_scratch[0] , 0 , sizeof(Real) * _scratch.size() );

#pragma omp parallel for num_threads( threads )
			for( int j=0 ; j<bands ; j++ ) for( int i=0 ; i<dimX ; i++ )
			{
				Point3D< float > p = waveSimulation->sorParam->template position< float >( i+0.5 , j+0.5 );
				for( int s=0 ; s<sourcePoints.size() ; s++ ) _scratch[j*dimX+i] += (Real)exp( -(p-sourcePoints[s].first).squareNorm() / (cutOff*cutOff) ) * sourcePoints[s].second;
			}
			sourcePoints.clear();
			waveSimulation->addHeightSource( GetPointer( _scratch ) );
		}
		if( advanceCount ) waveSimulation->advance( steps );
		if( advanceCount>0 ) advanceCount--;
		time = Time()-time;
		count++;
		glutPostRedisplay();
	}
}

template< class Real >
void SoRWaveVisualization< Real >::keyboardFunc( unsigned char key , int x , int y )
{
	switch( key )
	{
	case 'q': camera.rotateUp( -M_PI/128 ) ; break;
	case 'Q': camera.rotateUp( -M_PI/  4 ) ; break;
	case 'w': camera.rotateUp(  M_PI/128 ) ; break;
	case 'W': camera.rotateUp(  M_PI/  4 ) ; break;
	case 'a': camera.rotateRight( -M_PI/128 ) ; break;
	case 'A': camera.rotateRight( -M_PI/  4 ) ; break;
	case 'z': camera.rotateRight(  M_PI/128 ) ; break;
	case 'Z': camera.rotateRight(  M_PI/  4 ) ; break;
	case 's': camera.rotateForward( -M_PI/128 ) ; break;
	case 'S': camera.rotateForward( -M_PI/  4 ) ; break;
	case 'x': camera.rotateForward(  M_PI/128 ) ; break;
	case 'X': camera.rotateForward(  M_PI/  4 ) ; break;
	}
}

template< class Real >
void SoRWaveVisualization< Real >::specialFunc( int key, int x, int y )
{
	float stepSize = 10.f / ( screenWidth + screenHeight );
	if( glutGetModifiers()&GLUT_ACTIVE_CTRL ) stepSize /= 16;
	float panSize = stepSize , scaleSize = stepSize*2;

	switch( key )
	{
	case KEY_UPARROW:    camera.translate(  camera.forward*scaleSize ) ; break;
	case KEY_DOWNARROW:  camera.translate( -camera.forward*scaleSize ) ; break;
	case KEY_LEFTARROW:  camera.translate(  camera.right * panSize ) ; break;
	case KEY_RIGHTARROW: camera.translate( -camera.right * panSize ) ; break;
	case KEY_PGUP:       camera.translate( -camera.up    * panSize ) ; break;
	case KEY_PGDN:       camera.translate(  camera.up    * panSize ) ; break;
	}
	glutPostRedisplay();
}

template< class Real >
void SoRWaveVisualization< Real >::initMesh( void )
{
	unsigned int vCount = waveSimulation->sorParam->vertices() , fCount = waveSimulation->sorParam->faces();
	_verticesNormalsAndColors.resize( vCount*3 );
	_vertices = &_verticesNormalsAndColors[vCount*0];
	_normals  = &_verticesNormalsAndColors[vCount*1];
	_colors   = &_verticesNormalsAndColors[vCount*2];
	vertices.resize( vCount ) , normals.resize( vCount );
	triangles.clear();
	_cumulativeAreas.resize( fCount );
	for( int i=0 ; i<(int)vCount ; i++ )
	{
		vertices[i] = waveSimulation->sorParam->template vertexPosition< float >( i );
		normals[i] = Point3D< float >();
	}
	{
		unsigned int faceV[4];
		TriangleIndex tri;
		double areaSum = 0;
		for( int i=0 ; i<(int)fCount ; i++ )
		{
			int c = waveSimulation->sorParam->faceVertices( i , faceV );
			Point3D< float > n;
			for( int j=0 ; j<c ; j++ ) n += Point3D< float >::CrossProduct( vertices[ faceV[j] ] , vertices[ faceV[(j+1)%c] ] );
			for( int j=0 ; j<c ; j++ ) normals[ faceV[j] ] += n;
			if( c==3 ){ tri[0] = faceV[0] , tri[1] = faceV[1] , tri[2] = faceV[2] ; triangles.push_back( tri ); }
			else
			{
				tri[0] = faceV[0] , tri[1] = faceV[1] , tri[2] = faceV[3] ; triangles.push_back( tri );
				tri[0] = faceV[2] , tri[1] = faceV[3] , tri[2] = faceV[1] ; triangles.push_back( tri );
			}
			unsigned int x , y;
			waveSimulation->sorParam->faceIndices( i , x , y );
			_cumulativeAreas[i] = waveSimulation->sorParam->area( y );
			areaSum += _cumulativeAreas[i];
		}
		for( int i=0 ; i<(int)fCount ; i++ ) _cumulativeAreas[i] /= (float)areaSum;
		for( int i=0 ; i<(int)fCount ; i++ ) _cumulativeAreas[i] += _cumulativeAreas[i-1];
	}
	for( int i=0 ; i<(int)vCount ; i++ ) normals[i] /= Length( normals[i] );

#if USE_VBO
	glGenBuffers( 1 , &vbo );
	glBindBuffer( GL_ARRAY_BUFFER , vbo );
	glBufferData( GL_ARRAY_BUFFER , 3 * vCount * sizeof( Point3D< float > ) , &_verticesNormalsAndColors[0] , GL_DYNAMIC_DRAW );
	glBindBuffer( GL_ARRAY_BUFFER , 0 );

	glGenBuffers( 1 , &ebo );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , ebo );
	glBufferData( GL_ELEMENT_ARRAY_BUFFER , triangles.size() * sizeof( int ) * 3 , &triangles[0] , GL_STATIC_DRAW );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , 0 );
#endif // USE_VBO
}
template< class Real >
Point3D< float > SoRWaveVisualization< Real >::pointSelect( float x , float y )
{
	Point3D< float > out;
	Pointer( float ) depthBuffer = AllocPointer< float >( sizeof(float) * screenWidth * screenHeight );
	glReadPixels( 0 , 0 , screenWidth , screenHeight , GL_DEPTH_COMPONENT , GL_FLOAT , depthBuffer );
	float ar = (float)screenWidth/(float)screenHeight ;
	float _screenWidth , _screenHeight;
	if( screenWidth>screenHeight ) _screenWidth =  screenWidth * ar , _screenHeight = screenHeight;
	else                           _screenWidth =  screenWidth , _screenHeight = screenHeight / ar;
	{
		double _x =(double)x/screenWidth - 0.5 , _y = 1. - (double)y/screenHeight - 0.5 , _z;
		if( screenWidth>screenHeight ) _x *= zoom*ar , _y *= zoom;
		else                           _x *= zoom , _y *= zoom/ar;
		_x *= 2. , _y *= 2;
		int x1 = (int)floor(x) , y1 = (int)floor(y) , x2 = x1+1 , y2 = y1+1;
		float dx = x-x1 , dy = y-y1;
		x1 = std::max< int >( 0.f , std::min< int >( x1 , screenWidth -1 ) );
		y1 = std::max< int >( 0.f , std::min< int >( y1 , screenHeight-1 ) );
		x2 = std::max< int >( 0.f , std::min< int >( x2 , screenWidth -1 ) );
		y2 = std::max< int >( 0.f , std::min< int >( y2 , screenHeight-1 ) );
		_z =
			depthBuffer[ (screenHeight-1-y1)*screenWidth+x1 ] * (1.f-dx) * (1.f-dy) +
			depthBuffer[ (screenHeight-1-y1)*screenWidth+x2 ] * (    dx) * (1.f-dy) +
			depthBuffer[ (screenHeight-1-y2)*screenWidth+x1 ] * (1.f-dx) * (    dy) +
			depthBuffer[ (screenHeight-1-y2)*screenWidth+x2 ] * (    dx) * (    dy) ;
		if( _z<1 ) out = Point3D< float >( camera.forward * ( -1.5 + 3. * _z ) + camera.right * _x + camera.up * _y );
	}
	FreePointer( depthBuffer );
	return out;
}
template< class Real >
void SoRWaveVisualization< Real >::pointSelect( const std::vector< std::pair< float , float > >& screenPoints , std::vector< Point3D< float > >& worldPoints )
{
	Pointer( float ) depthBuffer = AllocPointer< float >( sizeof(float) * screenWidth * screenHeight );
	glReadPixels( 0 , 0 , screenWidth , screenHeight , GL_DEPTH_COMPONENT , GL_FLOAT , depthBuffer );
	float ar = (float)screenWidth/(float)screenHeight ;
	float _screenWidth , _screenHeight;
	if( screenWidth>screenHeight ) _screenWidth =  screenWidth * ar , _screenHeight = screenHeight;
	else                           _screenWidth =  screenWidth , _screenHeight = screenHeight / ar;

	worldPoints.resize( 0 );
	worldPoints.reserve( screenPoints.size() );
	for( int i=0 ; i<screenPoints.size() ; i++ )
	{
		if( !i || screenPoints[i]!=screenPoints[i-1] )
		{
			float x = screenPoints[i].first , y = screenPoints[i].second;
			double _x =(double)x/screenWidth - 0.5 , _y = 1. - (double)y/screenHeight - 0.5 , _z;
			if( screenWidth>screenHeight ) _x *= zoom*ar , _y *= zoom;
			else                           _x *= zoom , _y *= zoom/ar;
			_x *= 2. , _y *= 2;
			int x1 = (int)floor(x) , y1 = (int)floor(y) , x2 = x1+1 , y2 = y1+1;
			float dx = x-x1 , dy = y-y1;
			x1 = std::max< int >( 0.f , std::min< int >( x1 , screenWidth -1 ) );
			y1 = std::max< int >( 0.f , std::min< int >( y1 , screenHeight-1 ) );
			x2 = std::max< int >( 0.f , std::min< int >( x2 , screenWidth -1 ) );
			y2 = std::max< int >( 0.f , std::min< int >( y2 , screenHeight-1 ) );
			_z =
				depthBuffer[ (screenHeight-1-y1)*screenWidth+x1 ] * (1.f-dx) * (1.f-dy) +
				depthBuffer[ (screenHeight-1-y1)*screenWidth+x2 ] * (    dx) * (1.f-dy) +
				depthBuffer[ (screenHeight-1-y2)*screenWidth+x1 ] * (1.f-dx) * (    dy) +
				depthBuffer[ (screenHeight-1-y2)*screenWidth+x2 ] * (    dx) * (    dy) ;
			if( _z<1 ) worldPoints.push_back( Point3D< float >( camera.forward * ( -1.5 + 3. * _z ) + camera.right * _x + camera.up * _y ) );
		}
	}
	FreePointer( depthBuffer );
}
template< class Real >
void SoRWaveVisualization< Real >::display( void )
{
	static std::vector< Point3D< float > > faceNormals;
	int bands = waveSimulation->sorParam->bands();
	int vCount = waveSimulation->sorParam->vertices() , fCount = waveSimulation->sorParam->faces();
	RegularGridFEM::GridType gridType = waveSimulation->sorParam->gridType();
	if( _verticesNormalsAndColors.size()!=(3*vCount) ) initMesh();
	faceNormals.resize( fCount );

	float max = dropHeight / 2.f;
	max *= 4.f;
	Point3D< float > red( 1.f , 0.f , 0.f ) , blue( 0.f , 0.f , 1.f ) , gray( 0.5f , 0.5f , 0.5f );
	int dimX = waveSimulation->sorParam->gridType().xPeriodic() ? resX : resX-1;

#pragma omp parallel for num_threads( threads )
	for( int i=0 ; i<fCount ; i++ )
	{
		unsigned int faceV[4] , x , y;
		waveSimulation->sorParam->faceIndices( i , x , y );
		int c = waveSimulation->sorParam->faceVertices( i , faceV );
		Point3D< float > n;
		if( c==3 ) faceNormals[i] = Point3D< float >::CrossProduct( _vertices[faceV[1]]-_vertices[faceV[0]] , _vertices[faceV[2]]-_vertices[faceV[0]] );
		else if( c==4 )
		{
			// n = v[0] x v[1] + v[1] x v[2] + v[2] x v[3] + v[3] x v[0]
			//   = (v[0]-v[2]) x v[1] + (v[2]-v[0]) x v[3]
			//   = (v[0]-v[2]) x (v[1]-v[3])
			Point3D< float > v02 = _vertices[ faceV[0] ] - _vertices[ faceV[2] ];
			Point3D< float > v13 = _vertices[ faceV[1] ] - _vertices[ faceV[3] ];
			faceNormals[i] = Point3D< float >::CrossProduct( v02 , v13 );
		}
		else
		{
			for( int j=0 ; j<c ; j++ ) n += Point3D< float >::CrossProduct( _vertices[ faceV[j] ] , _vertices[ faceV[(j+1)%c] ] );
			faceNormals[i] = n;
		}
	}
#pragma omp parallel for num_threads( threads )
	for( int i=0 ; i<vCount ; i++ )
	{
		unsigned int x , y;
		waveSimulation->sorParam->vertexIndices( i , x , y );
		float h = waveSimulation->heightOffset( x , y );
		_vertices[i] = vertices[i] + normals[i] * h;
		_normals[i] = Point3D< float >();
		h /= max;
		h = std::max< float >( -1.f , std::min< float >( 1.f , h ) );
		bool useRed = h<0;
		h = (float)sqrt( fabs(h) );
		if( useRed ) _colors[i] = red  * h + gray * (1-h);
		else         _colors[i] = blue * h + gray * (1-h);

		if( y==0 && gridType.yPole0() )
		{
			Point3D< float > n;
			for( int j=0 ; j<dimX ; j++ ) n += faceNormals[j];
			_normals[i] = n;
		}
		else if( y==0 && ( gridType.yDirichlet0() || gridType.yNeumann0() ) )
		{
			int y1 = (y+bands-1) % bands , x1 = (x+dimX-1) % dimX;
			_normals[i] = faceNormals[y*dimX+x] + faceNormals[y*dimX+x1];
		}
		else if( y==resY-1 && gridType.yPole1() )
		{
			Point3D< float > n;
			for( int j=0 ; j<dimX ; j++ ) n += faceNormals[(bands-1)*dimX+j];
			_normals[i] = n;
		}
		else if( y==resY-1 && ( gridType.yDirichlet1() || gridType.yNeumann1() ) )
		{
			int y1 = (y+bands-1) % bands , x1 = (x+dimX-1) % dimX;
			_normals[i] = faceNormals[y1*dimX+x] + faceNormals[y1*dimX+x1];
		}
		else
		{
			int y1 = (y+bands-1) % bands , x1 = (x+dimX-1) % dimX;
			_normals[i] = faceNormals[y*dimX+x] + faceNormals[y*dimX+x1] + faceNormals[y1*dimX+x] + faceNormals[y1*dimX+x1];
		}
	}
	glEnable( GL_NORMALIZE );

#if USE_VBO
	glBindBuffer( GL_ARRAY_BUFFER , vbo );
	if( showColor ) glBufferData( GL_ARRAY_BUFFER , sizeof( Point3D< float > ) * 3 * vCount , &_verticesNormalsAndColors[0] , GL_DYNAMIC_DRAW );
	else            glBufferData( GL_ARRAY_BUFFER , sizeof( Point3D< float > ) * 2 * vCount , &_verticesNormalsAndColors[0] , GL_DYNAMIC_DRAW );
	glBindBuffer( GL_ARRAY_BUFFER , 0 );
#endif // USE_VBO

	glDisable( GL_CULL_FACE );
	glDisable( GL_DEPTH_TEST );
	glDisable( GL_LIGHTING );

	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	float ar = (float)screenWidth/(float)screenHeight , ar_r = 1.f/ar;
	if( screenWidth>screenHeight ) glOrtho( -ar*zoom , ar*zoom , -zoom , zoom , -1.5 , 1.5 );
	else                           glOrtho( -zoom , zoom , -ar_r*zoom , ar_r*zoom , -1.5 , 1.5 );
	camera.position = Point3D< double >( 0. , 0. , 0 );
	glMatrixMode( GL_MODELVIEW );

	glLoadIdentity();
	camera.draw();

	GLfloat lPosition[4];

	{
		Point3D< float > d = camera.up + camera.right - camera.forward*5;
		lPosition[0] = d[0] , lPosition[1] = d[1] , lPosition[2] = d[2];
	}
	lPosition[3] = 0.0;
	glLightModeli( GL_LIGHT_MODEL_LOCAL_VIEWER , GL_FALSE );
	glLightModeli( GL_LIGHT_MODEL_TWO_SIDE , GL_TRUE );
	glLightfv( GL_LIGHT0 , GL_AMBIENT , lightAmbient );
	glLightfv( GL_LIGHT0 , GL_DIFFUSE , lightDiffuseF );
	glLightfv( GL_LIGHT0 , GL_SPECULAR , lightSpecular );
	glLightfv( GL_LIGHT0 , GL_POSITION , lPosition );
	glEnable( GL_LIGHT0 );

	if( useLight ) glEnable ( GL_LIGHTING );
	else           glDisable( GL_LIGHTING );
	glColorMaterial( GL_FRONT_AND_BACK , GL_AMBIENT_AND_DIFFUSE );
	glEnable( GL_COLOR_MATERIAL );

	glEnable( GL_DEPTH_TEST );
	glMaterialfv( GL_FRONT_AND_BACK , GL_SPECULAR  , shapeSpecular );
	glMaterialf ( GL_FRONT_AND_BACK , GL_SHININESS , shapeSpecularShininess );

#if USE_VBO
	glBindBuffer( GL_ARRAY_BUFFER , vbo );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , ebo );
	glEnableClientState( GL_VERTEX_ARRAY );
	glVertexPointer( 3 , GL_FLOAT , 0 , (GLubyte*)NULL + sizeof( Point3D< float > ) * vCount * 0 );
	glNormalPointer(     GL_FLOAT , 0 , (GLubyte*)NULL + sizeof( Point3D< float > ) * vCount * 1 );
	if( showColor )
	{
		glColorPointer ( 3 , GL_FLOAT , 0 , (GLubyte*)NULL + sizeof( Point3D< float > ) * vCount * 2 );
		glEnableClientState( GL_COLOR_ARRAY ); 
	}
	else glColor3f( 0.5f , 0.5f , 0.5f );
	glEnableClientState( GL_NORMAL_ARRAY );
	glDrawElements( GL_TRIANGLES , (GLsizei)(triangles.size()*3) , GL_UNSIGNED_INT , NULL );
	glDisableClientState( GL_COLOR_ARRAY );
	glDisableClientState( GL_NORMAL_ARRAY );
	glBindBuffer( GL_ARRAY_BUFFER , 0 );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , 0 );
#else // !USE_VBO
	glBegin( GL_TRIANGLES );
	if( !showColor ) glColor3f( 0.5f , 0.5f, 0.5f );
	for( int i=0 ; i<triangles.size() ; i++ ) for( int j=0 ; j<3 ; j++ )
	{
		glVertex3fv( (GLfloat*)( _vertices + triangles[i][j] ) );
		glNormal3fv( (GLfloat*)( _normals  + triangles[i][j] ) );
		if( showColor ) glColor3fv( (GLfloat*)( _colors + triangles[i][j] ) );
	}
	glEnd();
#endif // USE_VBO

	if( showEdges )
	{
		GLint src , dst;
		glGetIntegerv( GL_BLEND_SRC , &src );
		glGetIntegerv( GL_BLEND_DST , &dst );
		Point3D< float > f = camera.forward / 256;
		glPushMatrix();
		glTranslatef( -f[0] , -f[1] , -f[2] );
		glColor3f( 0.125 , 0.125 , 0.125 );
		glBlendFunc( GL_SRC_ALPHA , GL_ONE_MINUS_SRC_ALPHA );
		glEnable( GL_BLEND );
		glEnable( GL_LINE_SMOOTH );
		glLineWidth( 0.25f );
		glPolygonMode( GL_FRONT_AND_BACK , GL_LINE );

		{
			unsigned int vCount = waveSimulation->sorParam->vertices() , fCount = waveSimulation->sorParam->faces();
			unsigned int faceV[4];
			glBegin( GL_QUADS );
			for( int i=0 ; i<(int)fCount ; i++ )
			{
				int c = waveSimulation->sorParam->faceVertices( i , faceV );
				if( c==3 )
				{
					glVertex3f( _vertices[faceV[0]][0] , _vertices[faceV[0]][1] , _vertices[faceV[0]][2] );
					glVertex3f( _vertices[faceV[1]][0] , _vertices[faceV[1]][1] , _vertices[faceV[1]][2] );
					glVertex3f( _vertices[faceV[2]][0] , _vertices[faceV[2]][1] , _vertices[faceV[2]][2] );
					glVertex3f( _vertices[faceV[2]][0] , _vertices[faceV[2]][1] , _vertices[faceV[2]][2] );
				}
				else
				{
					glVertex3f( _vertices[faceV[0]][0] , _vertices[faceV[0]][1] , _vertices[faceV[0]][2] );
					glVertex3f( _vertices[faceV[1]][0] , _vertices[faceV[1]][1] , _vertices[faceV[1]][2] );
					glVertex3f( _vertices[faceV[2]][0] , _vertices[faceV[2]][1] , _vertices[faceV[2]][2] );
					glVertex3f( _vertices[faceV[3]][0] , _vertices[faceV[3]][1] , _vertices[faceV[3]][2] );
				}
			}
			glEnd();
		}

		glPolygonMode( GL_FRONT_AND_BACK , GL_FILL );
		glDisable( GL_LINE_SMOOTH );
		glPopMatrix();
		glDisable( GL_BLEND );
		glBlendFunc( src , dst );
	}
	if( showBoundary && ( gridType.yDirichlet0() || gridType.yDirichlet1() || gridType.yNeumann0() || gridType.yNeumann1() || gridType.xDirichlet() || gridType.xNeumann() ) )
	{
		glDisable( GL_LIGHTING );
		glLineWidth( 3.f );
		glColor3f( 0.f , 0.f , 0.f );
		Point2D< float > p;

		if( gridType.yDirichlet0() || gridType.yNeumann0() )
		{
			if( gridType.xPeriodic() ) glBegin( GL_LINE_LOOP );
			else glBegin( GL_LINE_STRIP );
			for( int i=0 ; i<resX ; i++ )
			{
				Point3D< float > p = _vertices[ waveSimulation->sorParam->vertexIndex( i , 0 ) ];
				glVertex3f( p[0] , p[1] , p[2] );
			}
			glEnd();
		}

		if( gridType.yDirichlet1() || gridType.yNeumann1() )
		{
			if( gridType.xPeriodic() ) glBegin( GL_LINE_LOOP );
			else glBegin( GL_LINE_STRIP );
			for( int i=0 ; i<resX ; i++ )
			{
				Point3D< float > p = _vertices[ waveSimulation->sorParam->vertexIndex( i , resY-1 ) ];
				glVertex3f( p[0] , p[1] , p[2] );
			}
			glEnd();
		}
		if( !gridType.xPeriodic() )
		{
			glBegin( GL_LINE_STRIP );
			for( int j=0 ; j<resY ; j++ )
			{
				Point3D< float > p = _vertices[ waveSimulation->sorParam->vertexIndex( 0 , j ) ];
				glVertex3f( p[0] , p[1] , p[2] );
			}
			glEnd();
			glBegin( GL_LINE_STRIP );
			for( int j=0 ; j<resY ; j++ )
			{
				Point3D< float > p = _vertices[ waveSimulation->sorParam->vertexIndex( resX-1 , j ) ];
				glVertex3f( p[0] , p[1] , p[2] );
			}
			glEnd();
		}
	}
	if( count==0 ) runningTime = Time();
	if( (count%50)==49 ){ double t = Time() ; fps = 50./( t-runningTime ) , runningTime = t; }
}
template< class Real >
void SoRWaveVisualization< Real >::SetInfoCallBack( Visualization* v , void* param )
{
	const SoRWaveVisualization< Real >* sor = ( const SoRWaveVisualization< Real >* )param;

	v->addInfoString( "[fps=%.2f] Time: %.3f(s) = %.3f + ..." , sor->fps , sor->time , sor->waveSimulation->solveTime );
	v->addInfoString( "Elasticity: %g\n" , sor->elasticity );
	v->addInfoString( "Damping Factor: %g\n" , sor->dampingFactor );
	v->addInfoString( "Wave Speed: %g\n" , sor->waveSpeed );
	v->addInfoString( "Step Size: %g\n" , sor->stepSize );
	v->addInfoString( "Steps: %d\n" , sor->steps );
	v->addInfoString( "Source Frequency: %f\n" , sor->sourceFrequency );
	v->addInfoString( "Drop Height: %f\n" , sor->dropHeight );
	v->addInfoString( "Drop Width: %f\n" , sor->dropWidth );
	if( !sor->gridType.xPeriodic() ) v->addInfoString( "Grid: %d x %d %s / %s %g\n" , sor->resX , sor->resY , sor->gridType.xName() , sor->gridType.yName() , sor->waveSimulation->sorParam->angleOfRevolution() / ( 2. * PI ) * 360. );
	else                             v->addInfoString( "Grid: %d x %d %s / %s\n"    , sor->resX , sor->resY , sor->gridType.xName() , sor->gridType.yName() );
}
