#include <Util/SoRFlowSimulation.h>
#include <Util/Timer.h>
#include <Visualization/Visualization.h>

Point3D< float > RandomColor( void )
{
	static int colorCount = 0;
	colorCount++;
	if( !(colorCount%10) ) return Point3D< float >( -1.f , -1.f , -1.f );
	if( !((colorCount+5)%10) ) return Point3D< float >( 1.f , 1.f , 1.f );
	return Point3D< float >( Random< float >() , Random< float >() , Random< float >() ) * 2.f - Point3D< float >( 1.f , 1.f , 1.f );
}

template< class Real >
struct SoRFlowVisualization : public Visualization
{
	static const int SELECT_BUFFER_SIZE = 1<<15;
	static const int SAMPLE_SPACING = 1;
	enum
	{
		ADVANCE_ONCE ,
		ADVANCE_INFINITE ,
		ADVANCE_NONE
	};

	int threads;
	float maxStepSize;
	int subSteps;
	float vorticityScale;
	float stepSize;
	float vectorDensity;
	RegularGridFEM::GridType gridType;
	int advanceCount;
	bool useVorticity;
	bool ccwVorticity;
	double runningTime , fps , time;
	Timer startSelectionTimer;
	std::vector< std::pair< int , int > > selectionPoints2D;
	std::vector< Point3D< float > > selectionPoints3D;
	float inkWidth;
	float viscosity;
	float disolve;
	std::vector< TriangleIndex > triangles;
	std::vector< Point3D< float > > verticesNormalsAndColors;
	Point3D< float > *vertices , *normals , *colors;
	Camera camera;
	float zoom;
	GLuint vbo , ebo;
	GLfloat lightAmbient[4] , lightDiffuse[4] , lightSpecular[4] , shapeSpecular[4] , shapeSpecularShininess;
	int oldX , oldY , newX , newY;
	float imageZoom , imageOffset[2];
	bool rotating , scaling , selecting , addingParticles;
	int count;
	bool useLight , showEdges , showVectors , showColor;
	float vectorScale;
	bool showBoundary;
	int resX , resY;
	SoRFlowSimulation< Real >* flowSimulation;
	bool project , advect , harmonic;
	std::vector< Point2D< double > > particles;

	SoRFlowVisualization( void );
#if USE_DIRECT_SOLVER
	void init( int resX , int resY , RegularGridFEM::GridType gridType , ConstPointer( Point2D <double > ) curve , bool conicalGeometry , bool useDirect , Real theta );
#else // !USE_DIRECT_SOLVER
	void init( int resX , int resY , RegularGridFEM::GridType gridType , ConstPointer( Point2D <double > ) curve , bool conicalGeometry , Real theta );
#endif // USE_DIRECT_SOLVER

	void idle( void );
	void keyboardFunc( unsigned char key , int x , int y );
	void specialFunc( int key, int x, int y );
	void initMesh( void );
	int select( int x , int y );
	void select( const std::vector< std::pair< float , float > >& in , std::vector< Point3D< float > >& out );
	void display( void );
	void mouseFunc( int button , int state , int x , int y );
	void motionFunc( int x , int y );

	static void SetInfoCallBack( Visualization* v , void* param );
	static void           ResetCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->flowSimulation->reset() , ( (SoRFlowVisualization*)v)->count = 0 , ( (SoRFlowVisualization*)v)->particles.clear() , ( (SoRFlowVisualization*)v)->selectionPoints3D.clear(); }
	static void     ToggleLightCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->useLight = !( (SoRFlowVisualization*)v)->useLight; }
	static void ToggleAdvectionCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->advect = !( (SoRFlowVisualization*)v)->advect; }
	static void     ToggleColorCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->showColor = !( (SoRFlowVisualization*)v)->showColor; }
	static void     ToggleEdgesCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->showEdges = !( (SoRFlowVisualization*)v)->showEdges; }
	static void   ToggleVectorsCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->showVectors = !( (SoRFlowVisualization*)v)->showVectors; }
	static void  ToggleBoundaryCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->showBoundary = !( (SoRFlowVisualization*)v)->showBoundary; }
	static void   ToggleProjectCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->project = !( (SoRFlowVisualization*)v)->project; }
	static void     TogglePauseCallBack( Visualization* v , const char* )
	{
		SoRFlowVisualization *_v = (SoRFlowVisualization*)v;
		if     ( _v->advanceCount==0 ) _v->advanceCount = -1;
		else if( _v->advanceCount< 0 ) _v->advanceCount =  0;
	}
	static void             AdvanceCallBack( Visualization* v , const char* ){ if( ( (SoRFlowVisualization*)v )->advanceCount>=0 ) ( (SoRFlowVisualization*)v )->advanceCount++; }
	static void     AdvanceMultipleCallBack( Visualization* v , const char* prompt )
	{
		SoRFlowVisualization* _v = (SoRFlowVisualization*)v;
		if( _v->advanceCount>=0 )
		{
			_v->advanceCount += atoi(prompt);
			if( _v->advanceCount<0 ) _v->advanceCount = 0;
		}
	}

	static void          ToggleAdvectCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->advect = !( (SoRFlowVisualization*)v)->advect; }
	static void       ToggleVorticityCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->useVorticity = !( (SoRFlowVisualization*)v)->useVorticity; }
	static void   DecreaseVectorScaleCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->vectorScale /= 1.1f; }
	static void   IncreaseVectorScaleCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->vectorScale *= 1.1f; }
	static void DecreaseVectorDensityCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->vectorDensity /= 1.1f; }
	static void IncreaseVectorDensityCallBack( Visualization* v , const char* ){ ( (SoRFlowVisualization*)v)->vectorDensity *= 1.1f; }
	// Call-backs that take an argument
	static void SetSubStepsCallBack( Visualization*v , const char* prompt ){ ( (SoRFlowVisualization*)v)->subSteps = (float)atof(prompt); }
	static void SetDisolveRateCallBack( Visualization*v , const char* prompt ){ ( (SoRFlowVisualization*)v)->disolve = (float)atof(prompt); }
	static void SetKinematicViscosityCallBack( Visualization* v , const char* prompt )
	{
		SoRFlowVisualization* _v = (SoRFlowVisualization*)v;
		_v->viscosity = (float)atof(prompt);
#if USE_DIRECT_SOLVER
		if( _v->flowSimulation->pSolver ) _v->flowSimulation->pSolver->resetDiffusion( (Real)( _v->viscosity * _v->stepSize ) );
		else if( _v->flowSimulation->sorSolver ) _v->flowSimulation->sorSolver->resetDiffusion( (Real)( _v->viscosity * _v->stepSize ) );
#else // !USE_DIRECT_SOLVER
		if( _v->flowSimulation->sorSolver ) _v->flowSimulation->sorSolver->resetDiffusion( (Real)( _v->viscosity * _v->stepSize ) );
#endif // USE_DIRECT_SOLVER
	}
	static void    SetStepSizeCallBack( Visualization*v , const char* prompt )
	{
		SoRFlowVisualization* _v = (SoRFlowVisualization*)v;
		_v->stepSize = (float)atof(prompt);
		if( _v->flowSimulation->sorSolver ) _v->flowSimulation->sorSolver->resetDiffusion( (Real)( _v->viscosity * _v->stepSize ) );
	}
	static void SetMaxStepSizeCallBack( Visualization*v , const char* prompt ){ ( (SoRFlowVisualization*)v )->maxStepSize = (float)atof(prompt); }
	bool setPosition( int x , int y , Point3D< double >& p );
	bool setPosition( int x , int y , Point3D< float >& p );
};
template< class Real >
SoRFlowVisualization< Real >::SoRFlowVisualization( void )
{
	advanceCount = -1;
	vorticityScale = 5e-1;
	project = advect = true;
	threads = 1;
	inkWidth = 0.1f;
	viscosity = (float)1e-3;
	disolve = 0.99f;
	stepSize = 1.f;
	vectorDensity = 1.f/16.f;
	useVorticity = false;
	flowSimulation = NULL;
	resX = resY = 0;
	runningTime=0. , fps=0. , time=0;
	zoom = 1.05f;
	lightAmbient [0] = lightAmbient [1] = lightAmbient [2] = 0.25f , lightAmbient [3] = 1.f;
	lightDiffuse [0] = lightDiffuse [1] = lightDiffuse [2] = 0.70f , lightDiffuse [3] = 1.f;
	lightSpecular[0] = lightSpecular[1] = lightSpecular[2] = 1.00f , lightSpecular[3] = 1.f;
	shapeSpecular[0] = shapeSpecular[1] = shapeSpecular[2] = 1.00f , shapeSpecular[3] = 1.f;
	shapeSpecularShininess = 128.;
	oldX , oldY , newX , newY;
	imageZoom = 1.f , imageOffset[0] = imageOffset[1] = 0.f;
	rotating = false , scaling = false , selecting = false , addingParticles = false;
	count = 0;
	useLight = true;
	showEdges = false;
	showVectors = false;
	showColor = true;
	vectorScale = 1.f;
	showBoundary = true;

	keyboardCallBacks.push_back( KeyboardCallBack( this , '{' , "decrease vector density" , DecreaseVectorDensityCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '}' , "increase vector density" , IncreaseVectorDensityCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '[' , "decrease vector size" , DecreaseVectorScaleCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , ']' , "increase vector size" , IncreaseVectorScaleCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'e' , "toggle edges" , ToggleEdgesCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'v' , "toggle vectors" , ToggleVectorsCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'b' , "toggle boundary" , ToggleBoundaryCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'C' , "toggle color" , ToggleColorCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'L' , "toggle light" , ToggleLightCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'd' , "toggle advection" , ToggleAdvectionCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '+' , "advanace" , AdvanceCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '=' , "advanace multiple" , "Advance Steps" , AdvanceMultipleCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , ' ' , "toggle pause" , TogglePauseCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '4' , "set sub-steps" , "Sub-Steps" , SetSubStepsCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '3' , "set disolve rate" , "Disolve Rate" , SetDisolveRateCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '2' , "set kinematic viscosity" , "Kinematic Viscosity" , SetKinematicViscosityCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '1' , "set max step size" , "Max step size" , SetMaxStepSizeCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , '0' , "set step size" , "Step size" , SetStepSizeCallBack ) );
	keyboardCallBacks.push_back( KeyboardCallBack( this , 'R' , "reset" , ResetCallBack ) );
	infoCallBacks.push_back( InfoCallBack( SetInfoCallBack , this ) );
}
template< class Real >
bool SoRFlowVisualization< Real >::setPosition( int x , int y , Point3D< double >& p )
{
	double _x =(double)x / screenWidth - 0.5 , _y = 1. - (double)y/screenHeight - 0.5;
	_x *= 2. , _y *= 2;
	_x *= zoom , _y *= zoom;
	double r = _x*_x + _y*_y;
	if( r<1 )
	{
		p = camera.forward * ( -sqrt( 1-r ) ) + camera.right * _x + camera.up * _y;
		return true;
	}
	return false;
}
template< class Real >
bool SoRFlowVisualization< Real >::setPosition( int x , int y , Point3D< float >& p )
{
	Point3D< double > _p;
	bool ret = setPosition( x , y , _p );
	p = Point3D< float >( (float)_p[0] , (float)_p[1] , (float)_p[2] );
	return ret;
}

template< class Real >
#if USE_DIRECT_SOLVER
void SoRFlowVisualization< Real >::init( int rX , int rY , RegularGridFEM::GridType gridType , ConstPointer( Point2D <double > ) curve , bool conicalGeometry , bool useDirect , Real theta )
#else // !USE_DIRECT_SOLVER
void SoRFlowVisualization< Real >::init( int rX , int rY , RegularGridFEM::GridType gridType , ConstPointer( Point2D <double > ) curve , bool conicalGeometry , Real theta )
#endif // USE_DIRECT_SOLVER
{
	this->gridType = gridType;
	if( flowSimulation ) delete flowSimulation;
#if USE_DIRECT_SOLVER
	flowSimulation = new SoRFlowSimulation< Real >( rX , rY , gridType , curve , conicalGeometry , theta , useDirect , stepSize , viscosity , threads );
#else // !USE_DIRECT_SOLVER
	flowSimulation = new SoRFlowSimulation< Real >( rX , rY , gridType , curve , conicalGeometry , theta , stepSize , viscosity , threads );
#endif // USE_DIRECT_SOLVER
	flowSimulation->sorParam->resolution( resX , resY );
	verticesNormalsAndColors.resize( 0 );
}

template< class Real >
void SoRFlowVisualization< Real >::mouseFunc( int button , int state , int x , int y )
{
	newX = x ; newY = y;

	selecting = rotating = scaling = addingParticles = false;

	if( glutGetModifiers()!=GLUT_ACTIVE_CTRL && state==GLUT_DOWN )
	{
		startSelectionTimer.reset();
		selectionPoints2D.clear() , selectionPoints2D.push_back( std::pair< int , int >(x,y) );
		if( glutGetModifiers()==GLUT_ACTIVE_SHIFT ) addingParticles = true;
		else selecting = true;
		if( button==GLUT_RIGHT_BUTTON ) ccwVorticity = false;
		else                            ccwVorticity = true;
	}
	else
		if     ( button==GLUT_LEFT_BUTTON  ) rotating = true;
		else if( button==GLUT_RIGHT_BUTTON ) scaling = true;

	if( !selecting && selectionPoints2D.size() )
	{
		selectionPoints3D.clear();
		std::vector< std::pair< float , float > > _selectionPoints2D;
		int sz = (int)selectionPoints2D.size();
		for( int i=0 ; i<sz-1 ; i++ )
		{
			double x0 = selectionPoints2D[i?(i-1):i].first            , y0 = selectionPoints2D[i?(i-1):i].second;
			double x1 = selectionPoints2D[i  ].first                  , y1 = selectionPoints2D[i  ].second;
			double x2 = selectionPoints2D[i+1].first                  , y2 = selectionPoints2D[i+1].second;
			double x3 = selectionPoints2D[(i<sz-2)?(i+2):(i+1)].first , y3 = selectionPoints2D[(i<sz-2)?(i+2):(i+1)].second;
			double l = (x1-x2)*(x1-x2) + (y1-y2)*(y1-y2);
			{
				l = sqrt(l);
				int steps = (int)ceil( l/SAMPLE_SPACING )+1;
				for( int j=0 ; j<steps ; j++ )
				{
					double t = ((float)j) / steps;
					double a0 = -1./6 * t * t * t + 1./2 * t * t - 1./2 * t + 1./6;
					double a1 =  1./2 * t * t * t -        t * t            + 2./3;
					double a2 = -1./2 * t * t * t + 1./2 * t * t + 1./2 * t + 1./6;
					double a3 =  1./6 * t * t * t;
					double x = x0*a0 + x1*a1 + x2*a2 + x3*a3;
					double y = y0*a0 + y1*a1 + y2*a2 + y3*a3;
					_selectionPoints2D.push_back( std::pair< float , float >( (float)x , (float)y ) );
				}
			}
		}
		_selectionPoints2D.push_back( std::pair< float , float >( (float)selectionPoints2D.back().first , (float)selectionPoints2D.back().second ) );
		select( _selectionPoints2D , selectionPoints3D );
		selectionPoints2D.clear();
	}
}
template< class Real >
void SoRFlowVisualization< Real >::motionFunc( int x , int y )
{
	oldX = newX , oldY = newY , newX = x , newY = y;

	int imageSize = std::min< int >( screenWidth , screenHeight );
	float rel_x = (newX - oldX) / (float)imageSize * 2;
	float rel_y = (newY - oldY) / (float)imageSize * 2;
	float pRight = -rel_x , pUp = rel_y;
	float sForward = rel_y*4;
	float rRight = rel_y , rUp = rel_x;

	if     ( rotating ) camera.rotateUp( rUp ) , camera.rotateRight( rRight );
	else if( scaling  ) zoom *= (float)pow( 0.9 , (double)sForward );
	else if( selecting ) selectionPoints2D.push_back( std::pair< int , int >(x,y) );
	glutPostRedisplay();
}
#ifdef _WIN32
#include <Windows.h>
#include <Psapi.h>
#endif // _WIN32

template< class Real >
void SoRFlowVisualization< Real >::idle( void )
{
	if( !promptCallBack )
	{
		time = Time();
		if( selectionPoints3D.size() && !selecting )
		{
			Point3D< float > inkColor = RandomColor();
			double dt = startSelectionTimer.elapsed();
			if( !dt ) dt = 1.;
			int resX = flowSimulation->resX , resY = flowSimulation->resY;
			int fDimX = flowSimulation->sorParam->gridType().xPeriodic() ? resX : resX-1;
			float cutOff = inkWidth * zoom;
			float cutOff2 = cutOff * cutOff;
			Point3D< float > dirToCamera = Point3D< float >( camera.forward );
			int directionSign=1;
			int bands = flowSimulation->sorParam->bands();
			if( selectionPoints3D.size() )
			{
				int nearestI = 0 , nearestJ = 0;
				float nearestL = -1;
				for( int j=0 ; j<bands ; j++ ) for( int i=0 ; i<fDimX ; i++ )
				{
					Point3D< float > p = flowSimulation->sorParam->template position< float >( i+0.5 , j+0.5 );
					float l = (p-selectionPoints3D[0]).squareNorm();
					if( nearestL<0 || l<nearestL ) nearestL = l , nearestI = i , nearestJ = j;
				}
				directionSign = Point3D< float >::Dot( dirToCamera , flowSimulation->sorParam->template normal< float >( nearestI+0.5 , nearestJ+0.5 ) ) > 0 ? 1 : -1;
			}

			static std::vector< Point2D< Real > > _scratchVF;
			static std::vector< Point3D< Real > > _scratchInk;
			_scratchVF.resize( bands * fDimX ) , _scratchInk.resize( bands * fDimX );
#pragma omp parallel for
			for( int i=0 ; i<_scratchVF.size() ; i++ ) _scratchVF[i] = Point2D< Real >() , _scratchInk[i] = Point3D< Real >();
			static std::vector< Real > _scratchVorticity;
			_scratchVorticity.resize( bands * fDimX );
#pragma omp parallel for
			for( int i=0 ; i<_scratchVorticity.size() ; i++ ) _scratchVorticity[i] = (Real)0;

			if( selectionPoints3D.size()>1 )
			{
				std::vector< Point3D< float > > tangents( selectionPoints3D.size()-1 );
				float l = 0;
				for( int i=1 ; i<selectionPoints3D.size() ; i++ )
				{
					float _l = Length< float >( selectionPoints3D[i]-selectionPoints3D[i-1] );
					tangents[i-1] = Point3D< float >( ( selectionPoints3D[i]-selectionPoints3D[i-1] ) / _l );
					l += _l;
				}
				if( dt ) l /= dt;
				else l=0;
#pragma omp parallel for num_threads( threads )
				for( int j=0 ; j<bands ; j++ ) for( int i=0 ; i<fDimX ; i++ )
				{
					int idx = -1;
					double minDist;
					Point3D< float > p , n , dX , dY;
					p = flowSimulation->sorParam->template position< float >( i+0.5 , j+0.5 );
					n = flowSimulation->sorParam->template normal  < float >( i+0.5 , j+0.5 ) , n /= Length( n );
					flowSimulation->sorParam->template cotangents  < float >( i+0.5 , j+0.5 , dX , dY );
					SquareMatrix< Real , 2 > I;
					I(0,0) = dX.squareNorm() , I(1,1) = dY.squareNorm();
					I(0,1) = I(1,0) = Point3D< Real >::Dot( dX , dY );
					I = I.inverse();

					for( int k=0 ; k<selectionPoints3D.size()-1 ; k++ )
					{
						Point3D< float > d1 = p - selectionPoints3D[k] , d2 = p-selectionPoints3D[k+1];
						float dist;
						if( Point3D< float >::Dot( d1 , tangents[k] )<0 ) dist = d1.squareNorm();
						else if( Point3D< float >::Dot( d2 , tangents[k] )>0 ) dist = d2.squareNorm();
						else dist = ( d1 - tangents[k] * Point3D< float >::Dot( d1 , tangents[k] ) ).squareNorm();
						if( idx==-1 || dist<minDist ) minDist = dist , idx = k;
					}
					if( idx!=-1 )
					{
						float e = (float)exp( -minDist / (cutOff*cutOff) );
						bool visible = Point3D< float >::Dot( n , dirToCamera )*directionSign>0;
						if( useVorticity )
						{
							if( ccwVorticity ) _scratchVorticity[j*fDimX+i] =  e * vorticityScale / 5;
							else               _scratchVorticity[j*fDimX+i] = -e * vorticityScale / 5;
						}
						else if( visible )
						{
							_scratchInk[j*fDimX+i] = inkColor * e;
							Point3D< float > _v = tangents[idx];
							_v *= e * l / 5;
							Point3D< Real > v( _v[0] , _v[1] , _v[2] );
							{
								Point2D< Real > c( Point3D< Real >::Dot( v , dX ) , Point3D< Real >::Dot( v , dY ) );
								c = I * c;
								Point2D< Real > _c = c;
								_c = -flowSimulation->sorParam->rotate90( i+0.5 , j+0.5 , _c );
								c = Point2D< Real >( _c[0] , _c[1] );
								_scratchVF[j*fDimX+i] = Point2D< Real >( c );
							}
						}
					}
				}
			}	
			else if( selectionPoints3D.size() )
			{
				if( addingParticles )
				{
					if( ccwVorticity )
					{
						int nearestI = 0 , nearestJ = 0;
						float nearestL = -1;
						for( int j=0 ; j<bands ; j++ ) for( int i=0 ; i<fDimX ; i++ )
						{
							Point3D< float > p = flowSimulation->sorParam->template position< float >( i+0.5 , j+0.5 );
							float l = (p-selectionPoints3D[0]).squareNorm();
							if( nearestL<0 || l<nearestL ) nearestL = l , nearestI = i , nearestJ = j;
						}
						particles.push_back( Point2D< double >( nearestI+0.5 , nearestJ+0.5 ) );
					}
					else if( particles.size() ) particles.pop_back();
				}
				else
				{
#pragma omp parallel for num_threads( threads )
					for( int j=0 ; j<bands ; j++ ) for( int i=0 ; i<fDimX ; i++ )
					{
						Point3D< float > p , n , dX , dY;
						p = flowSimulation->sorParam->template position< float >( i+0.5 , j+0.5 );
						n = flowSimulation->sorParam->template normal  < float >( i+0.5 , j+0.5 ) , n /= Length( n );
						flowSimulation->sorParam->template cotangents  < float >( i+0.5 , j+0.5 , dX , dY );
						SquareMatrix< Real , 2 > I;
						I(0,0) = dX.squareNorm() , I(1,1) = dY.squareNorm();
						I(0,1) = I(1,0) = Point3D< Real >::Dot( dX , dY );
						I = I.inverse();

						Point3D< float > q = selectionPoints3D[0];
						double minDist;
						float sign;
						if( useVorticity ) minDist = (p-q).squareNorm();
						else
						{
							sign = Point3D< float >::Dot(p,q) * 10;
							minDist = Point3D< float >::Dot( p , q ) * Point3D< float >::Dot( p , q );
						}
						float e = (float)exp( -minDist / (cutOff*cutOff) );
						bool visible = ( Point3D< float >::Dot( n , dirToCamera )*directionSign )>0;
						if( useVorticity )
						{
							if( ccwVorticity ) _scratchVorticity[j*fDimX+i] =  e * vorticityScale;
							else               _scratchVorticity[j*fDimX+i] = -e * vorticityScale;
						}
						else if( visible )
						{
							_scratchInk[j*fDimX+i] = inkColor * e;
							Point3D< float > _v = Point3D< float >::CrossProduct( p , q );
							_v *= e / 5.f;
							Point3D< Real > v( _v[0]*sign , _v[1]*sign , _v[2]*sign );
							{
								v = Point3D< Real >::CrossProduct( n , v );
								Point2D< Real > c( Point3D< Real >::Dot( v , dX ) , Point3D< Real >::Dot( v , dY ) );
								c = I * c;
								_scratchVF[j*fDimX+i] = Point2D< Real >( c );
							}
						}
					}
				}
			}
			if( !addingParticles )
			{
				if( useVorticity ) flowSimulation->vorticity.setFromFaceValues( GetPointer( _scratchVorticity ) , true , threads );
				else               flowSimulation->vf.setFromFaceValues( ( ConstPointer( Real ) )GetPointer( _scratchVF ) , true , threads );
				flowSimulation->ink.setFromFaceValues( GetPointer( _scratchInk ) , true , 1 );
			}
			addingParticles = false;
			selectionPoints3D.clear();
		}
		if( advanceCount )
		{
			if( !useVorticity )
#pragma omp parallel for num_threads( threads )
				for( int i=0 ; i<(int)flowSimulation->ink.dim() ; i++ ) flowSimulation->ink()[i] *= (Real)disolve;
			flowSimulation->advance( useVorticity , project , harmonic , advect , stepSize , maxStepSize , subSteps );
		}
		if( advanceCount>0 ) advanceCount--;
		time = Time()-time;
		count++;
		glutPostRedisplay();
	}
}

template< class Real >
void SoRFlowVisualization< Real >::keyboardFunc( unsigned char key , int x , int y )
{
	switch( key )
	{
	case 'q': camera.rotateUp( -M_PI/128 ) ; break;
	case 'Q': camera.rotateUp( -M_PI/  4 ) ; break;
	case 'w': camera.rotateUp(  M_PI/128 ) ; break;
	case 'W': camera.rotateUp(  M_PI/  4 ) ; break;
	case 'a': camera.rotateRight( -M_PI/128 ) ; break;
	case 'A': camera.rotateRight( -M_PI/  4 ) ; break;
	case 'z': camera.rotateRight(  M_PI/128 ) ; break;
	case 'Z': camera.rotateRight(  M_PI/  4 ) ; break;
	case 's': camera.rotateForward( -M_PI/128 ) ; break;
	case 'S': camera.rotateForward( -M_PI/  4 ) ; break;
	case 'x': camera.rotateForward(  M_PI/128 ) ; break;
	case 'X': camera.rotateForward(  M_PI/  4 ) ; break;
	}
}

template< class Real >
void SoRFlowVisualization< Real >::specialFunc( int key, int x, int y )
{
	float stepSize = 10.f / ( screenWidth + screenHeight );
	if( glutGetModifiers()&GLUT_ACTIVE_CTRL ) stepSize /= 16;
	float panSize = stepSize , scaleSize = stepSize*2;

	switch( key )
	{
	case KEY_UPARROW:    camera.translate(  camera.forward*scaleSize ) ; break;
	case KEY_DOWNARROW:  camera.translate( -camera.forward*scaleSize ) ; break;
	case KEY_LEFTARROW:  camera.translate(  camera.right * panSize ) ; break;
	case KEY_RIGHTARROW: camera.translate( -camera.right * panSize ) ; break;
	case KEY_PGUP:       camera.translate( -camera.up    * panSize ) ; break;
	case KEY_PGDN:       camera.translate(  camera.up    * panSize ) ; break;
	}
	glutPostRedisplay();
}

template< class Real >
void SoRFlowVisualization< Real >::initMesh( void )
{
	unsigned int vCount = flowSimulation->sorParam->vertices() , fCount = flowSimulation->sorParam->faces();
	verticesNormalsAndColors.resize( vCount*3 );
	vertices = &verticesNormalsAndColors[vCount*0];
	normals  = &verticesNormalsAndColors[vCount*1];
	colors   = &verticesNormalsAndColors[vCount*2];
	triangles.clear();
	for( int i=0 ; i<(int)vCount ; i++ )
	{
		vertices[i] = flowSimulation->sorParam->template vertexPosition< float >( i );
		normals[i] = Point3D< float >();
	}
	{
		unsigned int faceV[4];
		TriangleIndex tri;
		for( int i=0 ; i<(int)fCount ; i++ )
		{
			int c = flowSimulation->sorParam->faceVertices( i , faceV );
			Point3D< float > n;
			for( int j=0 ; j<c ; j++ ) n += Point3D< float >::CrossProduct( vertices[ faceV[j] ] , vertices[ faceV[(j+1)%c] ] );
			for( int j=0 ; j<c ; j++ ) normals[ faceV[j] ] += n;
			if( c==3 )
			{
				tri[0] = faceV[0] , tri[1] = faceV[1] , tri[2] = faceV[2] ; triangles.push_back( tri );
			}
			else
			{
				tri[0] = faceV[0] , tri[1] = faceV[1] , tri[2] = faceV[3] ; triangles.push_back( tri );
				tri[0] = faceV[2] , tri[1] = faceV[3] , tri[2] = faceV[1] ; triangles.push_back( tri );
			}
		}
	}
	for( int i=0 ; i<(int)vCount ; i++ ) normals[i] /= Length( normals[i] );

	glGenBuffers( 1 , &vbo );
	glBindBuffer( GL_ARRAY_BUFFER , vbo );
	glBufferData( GL_ARRAY_BUFFER , 3 * vCount * sizeof( Point3D< float > ) , &verticesNormalsAndColors[0] , GL_DYNAMIC_DRAW );
	glBindBuffer( GL_ARRAY_BUFFER , 0 );

	glGenBuffers( 1 , &ebo );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , ebo );
	glBufferData( GL_ELEMENT_ARRAY_BUFFER , triangles.size() * sizeof( int ) * 3 , &triangles[0] , GL_STATIC_DRAW );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , 0 );
}
template< class Real >
void SoRFlowVisualization< Real >::select( const std::vector< std::pair< float , float > >& in , std::vector< Point3D< float > >& out )
{
	Pointer( float ) depthBuffer = AllocPointer< float >( sizeof(float) * screenWidth * screenHeight );
	glReadPixels( 0 , 0 , screenWidth , screenHeight , GL_DEPTH_COMPONENT , GL_FLOAT , depthBuffer );
	float ar = (float)screenWidth/(float)screenHeight ;
	float _screenWidth , _screenHeight;
	if( screenWidth>screenHeight ) _screenWidth =  screenWidth * ar , _screenHeight = screenHeight;
	else                           _screenWidth =  screenWidth , _screenHeight = screenHeight / ar;
	for( int i=0 ; i<in.size() ; i++ )
	{
		if( !i || in[i]!=in[i-1] )
		{
			float x = in[i].first , y = in[i].second;
			double _x =(double)x/screenWidth - 0.5 , _y = 1. - (double)y/screenHeight - 0.5 , _z;
			if( screenWidth>screenHeight ) _x *= zoom*ar , _y *= zoom;
			else                           _x *= zoom , _y *= zoom/ar;
			_x *= 2. , _y *= 2;
			int x1 = (int)floor(x) , y1 = (int)floor(y) , x2 = x1+1 , y2 = y1+1;
			float dx = x-x1 , dy = y-y1;
			x1 = std::max< int >( 0.f , std::min< int >( x1 , screenWidth -1 ) );
			y1 = std::max< int >( 0.f , std::min< int >( y1 , screenHeight-1 ) );
			x2 = std::max< int >( 0.f , std::min< int >( x2 , screenWidth -1 ) );
			y2 = std::max< int >( 0.f , std::min< int >( y2 , screenHeight-1 ) );
			_z =
				depthBuffer[ (screenHeight-1-y1)*screenWidth+x1 ] * (1.f-dx) * (1.f-dy) +
				depthBuffer[ (screenHeight-1-y1)*screenWidth+x2 ] * (    dx) * (1.f-dy) +
				depthBuffer[ (screenHeight-1-y2)*screenWidth+x1 ] * (1.f-dx) * (    dy) +
				depthBuffer[ (screenHeight-1-y2)*screenWidth+x2 ] * (    dx) * (    dy) ;
			if( _z<1 ) out.push_back( Point3D< float >( camera.forward * ( -1.5 + 3. * _z ) + camera.right * _x + camera.up * _y ) );
		}
	}
	FreePointer( depthBuffer );
}
template< class Real >
int SoRFlowVisualization< Real >::select( int x , int y )
{
	GLint vp[4];
	glGetIntegerv( GL_VIEWPORT , vp );
	static GLuint selectBuffer[SELECT_BUFFER_SIZE];
	int hits=0 , id;
	bool set=false;

	glSelectBuffer( SELECT_BUFFER_SIZE , selectBuffer );
	glRenderMode( GL_SELECT );

	glMatrixMode( GL_PROJECTION ) , glPushMatrix();
	glLoadIdentity( );
	gluPickMatrix( x , vp[3]-y , 1 , 1 , vp );
	float ar = (float)screenWidth/(float)screenHeight , ar_r = 1.f/ar;
	if( screenWidth>screenHeight ) glOrtho( -ar*zoom , ar*zoom , -zoom , zoom , -1.5 , 1.5 );
	else                           glOrtho( -zoom , zoom , -ar_r*zoom , ar_r*zoom , -1.5 , 1.5 );
	camera.position = Point3D< double >( 0. , 0. , 0 );

	glMatrixMode( GL_MODELVIEW ) , glPushMatrix();
	camera.draw();

	glInitNames();
	glPushName( ~0 );
	for( int t=0 ; t<triangles.size() ; t++ )
	{
		const TriangleIndex& tri = triangles[t];
		glLoadName( t );
		glBegin( GL_TRIANGLES );
		for( int i=0 ; i<3 ; i++ ) glVertex3f( vertices[tri[i]][0] , vertices[tri[i]][1] , vertices[tri[i]][2] );
		glEnd();
	}

	glMatrixMode( GL_MODELVIEW )  , glPopMatrix();
	glMatrixMode( GL_PROJECTION ) , glPopMatrix();
	glFlush();

	hits = glRenderMode( GL_RENDER );
	if( hits>0 )
	{
		int idx=0;
		unsigned int depth , tDepth;
		for( int j=0 ; j<hits ; j++ )
		{
			int count=selectBuffer[idx];
			idx++;
			tDepth=selectBuffer[idx];
			idx+=2;
			if( count && (!set || tDepth<depth) )
			{
				depth = tDepth;
				id = selectBuffer[idx];
				set = true;
			}
			idx += count;
		}
	}
	return set ? id : -1;
}
template< class Real >
void SoRFlowVisualization< Real >::display( void )
{
	int vCount = flowSimulation->sorParam->vertices() , fCount = flowSimulation->sorParam->faces();
	RegularGridFEM::GridType gridType = flowSimulation->sorParam->gridType();
	if( verticesNormalsAndColors.size()!=(3*vCount) ) initMesh();
	if( useVorticity )
	{
		float maxVorticity = vorticityScale / 100;
		maxVorticity = vorticityScale;
#pragma omp parallel for num_threads( threads )
		for( int i=0 ; i<vCount ; i++ )
		{
			unsigned int x , y;
			flowSimulation->sorParam->vertexIndices( i , x , y );
			float v = ( flowSimulation->vorticity.sample(x,y) ) / maxVorticity;
			float scale = 0.5f + cos( v * 2. * M_PI * 3. ) / 2.f;
			if( v<0 ) colors[i] = Point3D< float >( 0.5f , 0.5f , 0.5f ) - Point3D< float >(  1.f , -1.f , -1.f ) * v;
			else      colors[i] = Point3D< float >( 0.5f , 0.5f , 0.5f ) + Point3D< float >( -1.f , -1.f ,  1.f ) * v;
			colors[i] *= scale;
		}
	}
	else
	{
#pragma omp parallel for num_threads( threads )
		for( int i=0 ; i<vCount ; i++ )
		{
			unsigned int x , y;
			flowSimulation->sorParam->vertexIndices( i , x , y );
			Point3D< float > clr;
			for( int k=0 ; k<3 ; k++ ) clr[k] = std::max< float >( -1.f , std::min< float >( flowSimulation->ink.sample(x,y)[k] , 1.f ) );
			clr *= 10.f/2.f;
			clr += Point3D< float >( 0.5f , 0.5f , 0.5f );
			colors[i] = clr;
		}
	}
	glBindBuffer( GL_ARRAY_BUFFER , vbo );
	glBufferSubData( GL_ARRAY_BUFFER , sizeof( Point3D< float > ) * 2 * vCount , sizeof( Point3D< float > ) * vCount , colors );
	glBindBuffer( GL_ARRAY_BUFFER , 0 );

	glDisable( GL_CULL_FACE );
	glDisable( GL_DEPTH_TEST );
	glDisable( GL_LIGHTING );

	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	float ar = (float)screenWidth/(float)screenHeight , ar_r = 1.f/ar;
	if( screenWidth>screenHeight ) glOrtho( -ar*zoom , ar*zoom , -zoom , zoom , -1.5 , 1.5 );
	else                           glOrtho( -zoom , zoom , -ar_r*zoom , ar_r*zoom , -1.5 , 1.5 );
	camera.position = Point3D< double >( 0. , 0. , 0 );
	glMatrixMode( GL_MODELVIEW );

	glLoadIdentity();
	camera.draw();

	GLfloat lPosition[4];

	{
		Point3D< float > d = camera.up + camera.right - camera.forward*5;
		lPosition[0] = d[0] , lPosition[1] = d[1] , lPosition[2] = d[2];
	}
	lPosition[3] = 0.0;
	glLightModeli( GL_LIGHT_MODEL_LOCAL_VIEWER , GL_FALSE );
	glLightModeli( GL_LIGHT_MODEL_TWO_SIDE , GL_TRUE );
	glLightfv( GL_LIGHT0 , GL_AMBIENT , lightAmbient );
	glLightfv( GL_LIGHT0 , GL_DIFFUSE , lightDiffuse );
	glLightfv( GL_LIGHT0 , GL_SPECULAR , lightSpecular );
	glLightfv( GL_LIGHT0 , GL_POSITION , lPosition );
	glEnable( GL_LIGHT0 );
	if( useLight ) glEnable ( GL_LIGHTING );
	else           glDisable( GL_LIGHTING );
	glColorMaterial( GL_FRONT_AND_BACK , GL_AMBIENT_AND_DIFFUSE );
	glEnable( GL_COLOR_MATERIAL );

	glEnable( GL_DEPTH_TEST );
	glMaterialfv( GL_FRONT_AND_BACK , GL_SPECULAR  , shapeSpecular );
	glMaterialf ( GL_FRONT_AND_BACK , GL_SHININESS , shapeSpecularShininess );

	glBindBuffer( GL_ARRAY_BUFFER , vbo );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , ebo );
	glEnableClientState( GL_VERTEX_ARRAY );
	glVertexPointer( 3 , GL_FLOAT , 0 , (GLubyte*)NULL + sizeof( Point3D< float > ) * vCount * 0 );
	glNormalPointer(     GL_FLOAT , 0 , (GLubyte*)NULL + sizeof( Point3D< float > ) * vCount * 1 );
	glColorPointer ( 3 , GL_FLOAT , 0 , (GLubyte*)NULL + sizeof( Point3D< float > ) * vCount * 2 );
	if( showColor ) glEnableClientState( GL_COLOR_ARRAY ); 
	else glColor3f( 0.5f , 0.5f , 0.5f );
	glEnableClientState( GL_NORMAL_ARRAY );
	glDrawElements( GL_TRIANGLES , (GLsizei)(triangles.size()*3) , GL_UNSIGNED_INT , NULL );
	glDisableClientState( GL_COLOR_ARRAY );
	glDisableClientState( GL_NORMAL_ARRAY );
	glBindBuffer( GL_ARRAY_BUFFER , 0 );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , 0 );

	if( showEdges )
	{
		GLint src , dst;
		glGetIntegerv( GL_BLEND_SRC , &src );
		glGetIntegerv( GL_BLEND_DST , &dst );
		Point3D< float > f = camera.forward / 256;
		glPushMatrix();
		glTranslatef( -f[0] , -f[1] , -f[2] );
		glColor3f( 0.125 , 0.125 , 0.125 );
		glBlendFunc( GL_SRC_ALPHA , GL_ONE_MINUS_SRC_ALPHA );
		glEnable( GL_BLEND );
		glEnable( GL_LINE_SMOOTH );
		glLineWidth( 0.25f );
		glPolygonMode( GL_FRONT_AND_BACK , GL_LINE );
		glBindBuffer( GL_ARRAY_BUFFER , vbo );
		glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , ebo );
		glEnableClientState( GL_VERTEX_ARRAY );
		glVertexPointer( 3 , GL_FLOAT , 0 , NULL );
		glDrawElements( GL_TRIANGLES , (GLsizei)(triangles.size()*3) , GL_UNSIGNED_INT , NULL );
		glBindBuffer( GL_ARRAY_BUFFER , 0 );
		glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , 0 );
		glPolygonMode( GL_FRONT_AND_BACK , GL_FILL );
		glDisable( GL_LINE_SMOOTH );
		glPopMatrix();
		glDisable( GL_BLEND );
		glBlendFunc( src , dst );
	}
	if( showBoundary && ( gridType.yDirichlet0() || gridType.yDirichlet1() || gridType.yNeumann0() || gridType.yNeumann1() || !gridType.xPeriodic() ) )
	{
		glDisable( GL_LIGHTING );
		glLineWidth( 3.f );
		glColor3f( 0.f , 0.f , 0.f );
		Point2D< float > p;

		if( gridType.xPeriodic() )
		{
			if( gridType.yDirichlet0() || gridType.yNeumann0() )
			{
				glBegin( GL_LINE_LOOP );
				for( int i=0 ; i<resX ; i++ )
				{
					Point3D< float > p = flowSimulation->sorParam->template position< float >( i , 0 );
					glVertex3f( p[0] , p[1] , p[2] );
				}
				glEnd();
			}

			if( gridType.yDirichlet1() || gridType.yNeumann1() )
			{
				glBegin( GL_LINE_LOOP );
				for( int i=0 ; i<resX ; i++ )
				{
					Point3D< float > p = flowSimulation->sorParam->template position< float >( i , resY-1 );
					glVertex3f( p[0] , p[1] , p[2] );
				}
				glEnd();
			}
		}
		else
		{
			if( gridType.yDirichlet0() || gridType.yNeumann0() )
			{
				glBegin( GL_LINE_STRIP );
				for( int i=0 ; i<resX ; i++ )
				{
					Point3D< float > p = flowSimulation->sorParam->template position< float >( i , 0 );
					glVertex3f( p[0] , p[1] , p[2] );
				}
				glEnd();
			}

			if( gridType.yDirichlet1() || gridType.yNeumann1() )
			{
				glBegin( GL_LINE_STRIP );
				for( int i=0 ; i<resX ; i++ )
				{
					Point3D< float > p = flowSimulation->sorParam->template position< float >( i , resY-1 );
					glVertex3f( p[0] , p[1] , p[2] );
				}
				glEnd();
			}
			glBegin( GL_LINE_STRIP );
			for( int j=0 ; j<resY ; j++ )
			{
				Point3D< float > p = flowSimulation->sorParam->template position< float >( 0 , j );
				glVertex3f( p[0] , p[1] , p[2] );
			}
			glEnd();
			glBegin( GL_LINE_STRIP );
			for( int j=0 ; j<resY ; j++ )
			{
				Point3D< float > p = flowSimulation->sorParam->template position< float >( resX-1 , j );
				glVertex3f( p[0] , p[1] , p[2] );
			}
			glEnd();
		}
	}
	if( showVectors )
	{
		int fDimX = gridType.xPeriodic() ? resX : resX-1;
		int bands = flowSimulation->sorParam->bands();
		static std::vector< Real > random;
		static bool firstTime = true;
		if( firstTime )
		{
			srand( 0 );
			random.resize( bands*fDimX );
			if( bands*fDimX != fCount ) fprintf( stderr , "[ERROR] Sanity check fail\n" ) , exit( 0 );
			for( int i=0 ; i<bands*fDimX ; i++ ) random[i] = (Real)Random< double >();
			firstTime = false;
		}
//		Point3D< float > f = camera.forward / 256;
		Point3D< float > f = camera.forward / 128;
		glDisable( GL_LIGHTING );
		glPushMatrix();
		glTranslatef( -f[0] , -f[1] , -f[2] );
		double delta = flowSimulation->sorParam->area() / fCount / vectorDensity;

		glBegin( GL_TRIANGLES );
		for( int j=0 ; j<bands ; j++ )
		{
			double a = flowSimulation->sorParam->area( j );
			for( int i=0 ; i<fDimX ; i++ ) 
			{
				double r = random[j*fDimX+i];
				if( a>r*delta )
				{
					Point3D< Real > dX , dY;
					Point3D< Real > p = flowSimulation->sorParam->template position< Real >( i+0.5 , j+0.5 );
					Point3D< Real > normal = flowSimulation->sorParam->template normal< Real >( i+0.5 , j+0.5 ) ; normal /= Length( normal );
					std::pair< Real , Real > v = flowSimulation->vf.sample( i+0.5 , j+0.5 );
					Point2D< Real > _v( v.first , v.second );
					_v = flowSimulation->sorParam->rotate90( i+0.5 , j+0.5 , _v );
					flowSimulation->sorParam->cotangents( i+0.5 , j+0.5 , dX , dY );
					Point3D< Real > d = ( dX * _v[0] + dY * _v[1] ) * vectorScale;
					Point3D< Real > n = Point3D< Real >::CrossProduct( d , normal );
					n /= vectorDensity * 256.f;
					glColor3f( 1.0f , 1.0f , 1.0f ) , glVertex3f( p[0]+d[0] , p[1]+d[1] , p[2]+d[2] );
					glColor3f( 0.0f , 0.0f , 0.0f ) , glVertex3f( p[0]-n[0] , p[1]-n[1] , p[2]-n[2] ) , glVertex3f( p[0]+n[0] , p[1]+n[1] , p[2]+n[2] );
				}
			}
		}
		glEnd();
		glPopMatrix();
	}
	if( particles.size() )
	{
		glDisable( GL_LIGHTING );
		glColor3f( 0.f , 0.f , 0.f );
		GLUquadric* quadric = gluNewQuadric();

		for( int i=0 ; i<particles.size() ; i++ )
		{
			flowSimulation->sorParam->template advectForward< Real >( flowSimulation->vf , true , particles[i] , stepSize , maxStepSize , subSteps );
			Point3D< Real > p = flowSimulation->sorParam->template position< Real >( particles[i][0] , particles[i][1] );
			glPushMatrix();
			glTranslatef( p[0] , p[1] , p[2] );
			gluSphere( quadric , 0.02 , 12 , 6 );
			glPopMatrix();
		}
		gluDeleteQuadric( quadric );
	}
	////////////
	if( count==0 ) runningTime = Time();
	if( (count%50)==49 ){ double t = Time() ; fps = 50./( t-runningTime ) , runningTime = t; }
}
template< class Real >
void SoRFlowVisualization< Real >::SetInfoCallBack( Visualization* v , void* param )
{
	const SoRFlowVisualization< Real >* sor = ( const SoRFlowVisualization< Real >* )param;

	v->addInfoString( "[fps=%.2f] Time: %.2f(s) = %.2f + %.2f + %.2f(s) + %.2f(s)" , sor->fps , sor->time , sor->flowSimulation->divTime , sor->flowSimulation->projectTime , sor->flowSimulation->gradTime , sor->flowSimulation->advectTime );
	v->addInfoString( "Average Advection Steps: %f" , sor->flowSimulation->averageAdvectSamples );
	v->addInfoString( "Viscosity: %f" , sor->viscosity );
	v->addInfoString( "Disolve: %f" , sor->disolve );
	v->addInfoString( "Projection: %s" , sor->project ? "on" : "off" );
	v->addInfoString( "Advection: %s"  , sor->advect  ? "on" : "off" );
	v->addInfoString( "Vorticity: %s" , sor->useVorticity ? "on" : "off" , sor->stepSize );
	v->addInfoString( "Max Step Size: %f\n" , sor->maxStepSize );
	v->addInfoString( "Step Size: %f\n" , sor->stepSize );
}
