#include "Util/Curve.h"
#include "Visualization/Visualization.h"

float DistanceToSegment( Point2D< float > p , Point2D< float > q1 , Point2D< float > q2 )
{
	Point2D< float > d = q2 - q1;
	p -= q1;
	if( Point2D< float >::Dot( p , d )<0 ) return (float)sqrt( p.squareNorm() );
	else if( Point2D< float >::Dot( p , d )>d.squareNorm() ) return (float)sqrt( (p-d).squareNorm() );
	else return (float)sqrt( ( p - d * Point2D< float >::Dot( p , d ) / d.squareNorm() ).squareNorm() );
}

struct GeneratingCurveVisualization : public Visualization
{
	static const int SAMPLE_SPACING = 8;
	Camera camera;
	Point2D< float > translate;
	float zoom;
	int startX , startY , selected , res;
	bool linear , invert;
	Curve< float >& curve;
	GeneratingCurveVisualization( Curve< float >& curve ) : curve( curve )
	{
		zoom = 1.05f;
		selected = -1;
		linear = false;
#if !MINIMAL_UI
		keyboardCallBacks.push_back( Visualization::KeyboardCallBack( this , 'l' , "toggle linear" , ToggleLinearCallBack ) );
		keyboardCallBacks.push_back( Visualization::KeyboardCallBack( this , 'o' , "output curve" , "Curve Name" , OutputCurveCallBack ) );
#endif // !MINIMAL_UI
		keyboardCallBacks.push_back( Visualization::KeyboardCallBack( this , 'c' , "toggle circular" , ToggleCircularCallBack ) );
#if !MINIMAL_UI
		keyboardCallBacks.push_back( Visualization::KeyboardCallBack( this , 'r' , "set resolution" , "Resolution" , SetResolutionCallBack ) );
#endif // !MINIMAL_UI
	}
	void init( int res , int curveType )
	{
		this->res = res;
		curve.type = curveType;
		curve.points.resize( 0 );
		if( curveType==Curve< float >::CURVE_CLOSED )
		{
			curve.addPoint( Point2D< float >( 0.25f, -0.75f ) );
			curve.addPoint( Point2D< float >( 1.75f, -0.75f ) );
			curve.addPoint( Point2D< float >( 1.75f,  0.75f ) );
			curve.addPoint( Point2D< float >( 0.25f,  0.75f ) );
		}
		else
		{
			curve.addPoint( Point2D< float >(  0.75f, -0.75f ) );
			curve.addPoint( Point2D< float >(  0.75f,  0.75f ) );
		}
	}
	Point2D< float > screenToWorld( Point2D< float > p ) const;
	Point2D< float > worldToScreen( Point2D< float > p ) const;

	void keyboardFunc( unsigned char key , int x , int y );
	void specialFunc( int key, int x, int y );
	int select( int x , int y );
	void display( void );
	void mouseFunc( int button , int state , int x , int y );
	void motionFunc( int x , int y );

	static void      FullResetCallBack( Visualization* v , const char* ){ ( (GeneratingCurveVisualization*)v )->curve.reset(); }
	static void ToggleCircularCallBack( Visualization* v , const char* )
	{
		int type = ( (GeneratingCurveVisualization*)v )->curve.type;
		type = ( type + 1 ) % Curve< float >::CURVE_COUNT;
		( (GeneratingCurveVisualization*)v )->init( ( (GeneratingCurveVisualization*)v )->res , type );
		glutPostRedisplay();
	}
	static void  SetResolutionCallBack( Visualization* v , const char* res ){ ( (GeneratingCurveVisualization*)v )->res = atoi(res) , glutPostRedisplay(); }
	static void   ToggleLinearCallBack( Visualization* v , const char* ){( (GeneratingCurveVisualization*)v )->linear = !( (GeneratingCurveVisualization*)v )->linear; }
	static void    OutputCurveCallBack( Visualization* v , const char* fileName ){ ( (GeneratingCurveVisualization*)v )->curve.write( fileName ); }
};
Point2D< float > GeneratingCurveVisualization::screenToWorld( Point2D< float > p ) const
{
	float ar = (float)screenWidth/(float)screenHeight;

	p[0] = p[0]/screenWidth - 0.5f , p[1] = 1.f - p[1]/screenHeight - 0.5f;
	if( screenWidth>screenHeight ) p[0] *= zoom*ar , p[1] *= zoom;
	else                           p[0] *= zoom , p[1] *= zoom/ar;
	p[0] *= 2 , p[1] *= 2;
	if( curve.type!=Curve< float >::CURVE_CLOSED_REFLECTED && curve.type!=Curve< float >::CURVE_OPEN_REFLECTED ) p[0] += 1.f;
	return p;
}
Point2D< float > GeneratingCurveVisualization::worldToScreen( Point2D< float > p ) const
{
	float ar = (float)screenWidth/(float)screenHeight;

	if( curve.type!=Curve< float >::CURVE_CLOSED_REFLECTED && curve.type!=Curve< float >::CURVE_OPEN_REFLECTED ) p[0] -= 1.f;
	p[0] /= 2 , p[1] /= 2;
	if( screenWidth>screenHeight ) p[0] /= zoom*ar , p[1] /= zoom;
	else                           p[0] /= zoom , p[1] /= zoom/ar;
	p[0] = ( p[0] + 0.5f ) * screenWidth , p[1] = -( p[1] - 0.5f ) * screenHeight;
	return p;
}
void GeneratingCurveVisualization::mouseFunc( int button , int state , int x , int y )
{
	Point2D< float > p( (float)x , (float)y );
	if( state==GLUT_DOWN )
	{
		selected = -1;
		startX = x+100 , startY = y+100;
		bool found = false;
		invert = false;
		// Check if we are on a point
		{
			int idx = -1;
			float d = 10;
			for( int i=0 ; i<curve.size() ; i++ )
			{
				Point2D< float > q = worldToScreen( curve[i] );
				float _d = (float)sqrt( (p-q).squareNorm() );
				if( idx==-1 || _d<d ) d = _d , idx = i;
			}
			invert = ( ( curve.type==Curve< float >::CURVE_CLOSED_REFLECTED || curve.type==Curve< float >::CURVE_OPEN_REFLECTED ) && idx>=curve.points.size() );
			if( d<SAMPLE_SPACING )
			{
				startX = x , startY = y;
				if( invert ) selected = curve.size()-1-idx;
				else selected = idx;
				found = true;
			}
		}
		// Check if we are on a segment
		if( !found )
		{
			int idx = -1;
			float d = 10;
			for( int i=0 ; i<(int)curve.size() ; i++ )
			{
				float _d = DistanceToSegment( p , worldToScreen( curve[i] ) , worldToScreen( curve[i+1] ) );
				if( idx==-1 || _d<d ) d = _d , idx = i+1;
			}
			invert = ( ( curve.type==Curve< float >::CURVE_CLOSED_REFLECTED || curve.type==Curve< float >::CURVE_OPEN_REFLECTED ) && idx>=curve.points.size() );
			if( d<SAMPLE_SPACING )
			{
				if( invert ) selected = curve.size()-idx;
				else selected = idx;
				Point2D< float > p = screenToWorld( Point2D< float >( (float)x , (float)y ) );
				if( invert && (selected==0 || selected==curve.points.size() ) )
				{
					float n1 = ( curve[selected-1]-p ).squareNorm() , n2 = ( curve[selected]-p ).squareNorm();
					if( selected==0 && n2<n1 ) invert = false;
					else if( selected==curve.points.size() && n1<n2 ) invert = false;
				}
				if( invert ) p[0] = -p[0];
				curve.addPoint( p , selected ) , found = true;
			}
		}
	}
	else if( selected!=-1 && x==startX && y==startY ) curve.deletePoint( selected );
	glutPostRedisplay();
}
void GeneratingCurveVisualization::motionFunc( int x , int y )
{
	if( selected!=-1 )
	{
		curve.points[ selected ] = screenToWorld( Point2D< float >( (float)x  , (float)y ) );
		if( invert ) curve.points[ selected ][0] = -curve.points[selected][0];
	}
	glutPostRedisplay();
}
void GeneratingCurveVisualization::keyboardFunc( unsigned char key , int x , int y )
{
}

void GeneratingCurveVisualization::specialFunc( int key, int x, int y )
{
}

void GeneratingCurveVisualization::display( void )
{
	glDisable( GL_DEPTH_TEST );
	glDisable( GL_LIGHTING );

	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	float ar = (float)screenWidth/(float)screenHeight , ar_r = 1.f/ar;
	if( screenWidth>screenHeight )
		if( !Curve< float >::Reflected( curve.type ) ) glOrtho( -ar*zoom+1.0 , ar*zoom+1.0 , -zoom , zoom , -1.5 , 1.5 );
		else glOrtho( -ar*zoom , ar*zoom , -zoom , zoom , -1.5 , 1.5 );
	else
		if( !Curve< float >::Reflected( curve.type ) ) glOrtho( -zoom+1.0 , zoom+1.0 , -ar_r*zoom , ar_r*zoom , -1.5 , 1.5 );
		else                 glOrtho( -zoom , zoom , -ar_r*zoom , ar_r*zoom , -1.5 , 1.5 );

	glMatrixMode( GL_MODELVIEW );
	camera.draw();

	// Draw the y-axis
	glColor3f( 1.f , 0.f , 0.f );
	glBegin( GL_LINES );
	if( Curve< float >::Reflected( curve.type) ) glVertex2f( -1.f , 0.f ) , glVertex2f( 1.f ,  0.f ) , glVertex2f( 0.f , -1.f ) , glVertex2f( 0.f ,  1.f );
	else glVertex2f(  0.f , 0.f ) , glVertex2f( 2.f ,  0.f ) , glVertex2f( 0.f , -1.f ) , glVertex2f( 0.f ,  1.f );
	glEnd();
	if( curve.size() )
	{
		// Draw Lines
		glColor3f( 0.5f , 0.5f , 0.5f );
		glLineWidth( 1.f );
		glLineStipple( 1, 0x00FF );
		glEnable( GL_LINE_STIPPLE );
		glBegin( Curve< float >::Closed( curve.type ) ? GL_LINE_LOOP : GL_LINE_STRIP );
		for( int i=0 ; i<curve.size() ; i++ )
		{
			Point2D< float > p = curve[i];
			glVertex2f( p[0] , p[1] );
		}
		glEnd();
		glDisable( GL_LINE_STIPPLE );

		// Draw Curves
		glColor3f( 0.125f , 0.125f , 0.125f );
		glLineWidth( 2.f );
		glBegin( Curve< float >::Closed( curve.type ) ? GL_LINE_LOOP : GL_LINE_STRIP );
		for( int i=0 ; i<res ; i++ )
		{
			double t = (double)i / ( Curve< float >::Closed( curve.type ) ? res : res-1 );
			Point2D< float > p = curve( t , linear );
			glVertex2f( p[0] , p[1] );
		}
		glEnd();
		glDisable( GL_LINE_STIPPLE );

		// Draw Points
		glColor3f( 0.f , 0.f , 0.f );
		glPointSize( SAMPLE_SPACING );
		glBegin( GL_POINTS );
		for( int i=0 ; i<curve.size() ; i++ )
		{
			Point2D< float > p = curve[i];
			glVertex2f( p[0] , p[1] );
		}
		glEnd();
		glColor3f( 1.f , 1.f , 1.f );
		glPointSize( SAMPLE_SPACING/2 );
		glBegin( GL_POINTS );
		if( Curve< float >::Reflected( curve.type ) )
			for( int i=(int)curve.points.size() ; i<curve.size() ; i++ )
			{
				Point2D< float > p = curve[i];
				glVertex2f( p[0] , p[1] );
			}
		glEnd();
	}
}

