#ifndef SOR_FLOW_FIELD_INCLUDED
#define SOR_FLOW_FIELD_INCLUDED
#include <Util/CmdLineParser.h>
#include <Util/Geometry.h>
#include <Util/SparseMatrix.h>
#include <Util/Array.h>
#include <Util/Polynomial.h>
#include "RegularGridFEM.h"

// [WARNING]
// In implementing Neumann boundary conditions, there are two ways one can go.
// One can either define as though the domain has twice the size and the values are reflected.
// This gives a Laplacian akin to:
// |  2 -2  0              |
// | -1  2 -1              |
// |          .            |
// |            .          |
// |              -1  2 -1 |
// |               0 -2  2 |
// Alternatively, one can define it over the clipped domain, getting a Laplacian akin to:
// |  1 -1  0              |
// | -1  2 -1              |
// |          .            |
// |            .          |
// |              -1  2 -1 |
// |               0 -1  1 |
// In principal, the Fourier approach implies the first interpretation.
// However, this is problematic because the matrix is not symmetric.
// (The Fourier solver doesn't care because it doesn't see the lack of symmetry and works for non-symmetric matrices in any case. This could kill a direct solver however.)
// So instead, we define everything in terms of the second formulation.
// This means that in order to get the Fourier solution to work, we need to double the value of the constraints on the Neumann boundaries.
// This is managed in the solver when SoRPoissonSolver::solve is called, but SoRPoissonSolver::solveSpectral assumes that either this was already
// taken care of before, or that there is a pre-multiplication by the mass matrix.


struct ConeData
{
	double ratio;
	union { double r0 , width; };
	union { double r1 , height; };
	double a0 , a1 , b0;
};
struct TrapData{ double width0 , width1 , height , reflectCos , reflectSin; };
template< int D1 , int D2 , class Real=double >
struct Stencil
{
	Real values[D1][D2];
	Real* operator[] ( int x ) { return values[x]; }
	const Real* operator[] ( int x ) const { return values[x]; }
};
struct ValueStencil{ Stencil< 3 , 3 > stencil; };
struct DerivativeStencil{ Stencil< 2 , 3 > hStencil ; Stencil< 3 , 2 > vStencil; };


// When describing vector fields, we consider two different bases:
// Global:
//		These are the partial derivatives of the parameterization 
//		They are the representation for public methods
// Local:
//		These are either the coordinate axes (for trapezoidal represntations) or normalized angular and radial (for conical representation)
//		They are the representation for protected methods
//		[NOTE] This basis does not preserve the orientation...
struct SoRParameterization
{
protected:
	double _angleOfRevolution;
	bool _conicalGeometry;
	RegularGridFEM::GridType _gridType;
	int _resX , _resY;
	Pointer( Point2D< double > ) _samples;

	Pointer( TrapData ) _trapData;
	Pointer( ConeData ) _coneData;
	Pointer(      ValueStencil ) _mStencils;
	Pointer(      ValueStencil ) _sStencils;
	Pointer( DerivativeStencil ) _dStencils;
	Pointer( DerivativeStencil ) _drStencils;

	enum{ _L=0 , _R=1 , _B=2 , _T=3 };
	static inline bool __V( int e ){ return e==_L || e==_R; }
	static inline bool __H( int e ){ return e==_T || e==_B; }
	enum{ _B_L=0 , _B_R=1 , _T_L=2 , _T_R=3 };
	static inline bool __B( int e ){ return e==_B_L || e==_B_R; }
	static inline bool __T( int e ){ return e==_T_L || e==_T_R; }
	static inline bool __L( int e ){ return e==_T_L || e==_B_L; }
	static inline bool __R( int e ){ return e==_T_R || e==_B_R; }

	double _theta( double x , double y ) const;
	struct MetricRoot { Polynomial< 1 > x_dx , y_dx ; double y_dy; };
	MetricRoot _metricRoot( int b ) const;
	SquareMatrix< double , 2 > _metricRoot( int b , double x , double y ) const;
	template< class Real > Point2D< Real > _globalToLocal( int b , double x , double y , Point2D< Real > d ) const;
	template< class Data , class Real > std::pair< Data , Data > _globalToLocal( int b , double x , double y , std::pair< Data , Data > d ) const;
	template< class Real > Point2D< Real > _localToGlobal( int b , double x , double y , Point2D< Real > d ) const;
	template< class Data , class Real > std::pair< Data , Data > _localToGlobal( int b , double x , double y , std::pair< Data , Data > d ) const;
	Polynomial< 1 > _area( int b ) const;
	double _vIntegral( int b , int type1 , int type2 , const Polynomial< 1 >& area ) const;
	double _dIntegral( int b , int type1 , int type2 , bool rotate1 , bool rotate2 , const MetricRoot& metricRoot ) const;
	void __mStencil( int j , Stencil< 3 , 3 >& stencil ) const;
	void __sStencil( int j , Stencil< 3 , 3 >& stencil ) const;
	template< int DType=RegularGridFEM::DERIVATIVE_BOTH >
	void __dStencils( int j , Stencil< 2 , 3 >& hStencil , Stencil< 3 , 2 >& vStencil , bool rotate ) const;
	void _dStencil( int b , double stencil[4][4] ) const;

	enum
	{
		X0_INTERSECTION ,
		X1_INTERSECTION ,
		Y0_INTERSECTION ,
		Y1_INTERSECTION ,
		NO_INTERSECTION
	};
	int _trapezoidIntersect( double& s , double tWidth , double bWidth , double height , double x , double y , double dx , double dy , int invalidIsectType ) const;
	void _circleIntersect( double& s , double radius , double y , double dx , double dy , bool onCircle ) const;
	void _parameterToTrapezoid( int j , double& x , double& y ) const;
	void _trapezoidToParameter( int j , double& x , double& y ) const;
	void _trapReflectTangent( int j , int isectType , double& dx , double& dy ) const;
	void _trapGeodesic( double& x , double& y , double& dx , double& dy , double len ) const;
	void _coneGeodesic( double& x , double& y , double& dx , double& dy , double len ) const;
	void _geodesic( double& x , double& y , double& dx , double& dy ) const;

	template< class Data , class Real , int DType >
	void _div_curl( const RegularGridFEM::template Derivative< Data , Real , DType >& v , RegularGridFEM::template Signal< Data , Real >& div , bool rotate , int threads ) const;

	template< class Real > void _poissonFrequencySystems( BandedMatrix< Real , 1   >* mass , BandedMatrix< Real , 1   >* stiffness , unsigned int minFrequency , unsigned int maxFrequency , int threads ) const;

	template< class Data , class Real > void _assertValidity( const RegularGridFEM::template Signal    < Data , Real >& f , const char* methodName ) const;
	template< class Data , class Real , int DType > void _assertValidity( const RegularGridFEM::template Derivative< Data , Real , DType >& f , const char* methodName ) const;
	void _init( int resX , int resY , RegularGridFEM::GridType gridType , bool conincalGeometry , ConstPointer( Point2D< double > ) samples , double theta );
public:
	SoRParameterization( FILE* fp );
	SoRParameterization( int resX , int resY , RegularGridFEM::GridType gridType , bool conincalGeometry , ConstPointer( Point2D< double > ) samples , double angleOfRevolution=0 );
	~SoRParameterization( void );
	void resolution( int& resX , int& resY ) const;
	unsigned int resX( void ) const { return _resX; }
	unsigned int resY( void ) const { return _resY; }
	double angleOfRevolution( void ) const { return _angleOfRevolution; }
	RegularGridFEM::GridType gridType( void ) const { return _gridType; }
	bool conicalGeometry( void ) const { return _conicalGeometry; }
	double area( int j ) const;
	double area( void ) const;
	template< class Real > Point3D< Real > position( double x , double y ) const;
	// [WARNING] Note that this value is not be well defined when x or y are integers, as the derivatives may be discontinuous
	template< class Real > void tangents( double x , double y , Point3D< Real >& dX , Point3D< Real >& dY ) const;
	template< class Real > void cotangents( double x , double y , Point3D< Real >& dX , Point3D< Real >& dY ) const;
	template< class Real > SquareMatrix< Real , 2 > metricTensor( double x , double y ) const;
	template< class Real > Point3D< Real > normal( double x , double y ) const;
	template< class Real > Point2D< Real > rotate90( double x , double y , Point2D< Real > v ) const;
	double area( double x , double y ) const;

	unsigned int vertices( void ) const;
	unsigned int bands( void ) const;
	unsigned int faces( void ) const;
	unsigned int faceVertices( unsigned int face , unsigned int* vertices ) const;
	template< class Real > Point3D< Real > vertexPosition( unsigned int vertex ) const;
	void faceIndices( unsigned int face , unsigned int& x0 , unsigned int& y0 ) const;
	void vertexIndices( unsigned int vertex , unsigned int& x , unsigned int& y ) const;
	unsigned int vertexIndex( unsigned int x , unsigned int y ) const;

	// [NOTE] The derivative is expressed in terms of the global frame.
	void geodesic( double& x , double& y , double& dx , double& dy ) const;

	template< class Data , class Real >
	void screenedLaplacian( const RegularGridFEM::template Signal< Data , Real >& in , RegularGridFEM::template Signal< Data , Real >& out , double mWeight , double sWeight , bool add=false , int threads=1 ) const;
	template< class Data , class Real >
	void gradient( const RegularGridFEM::template Signal< Data , Real >& sf , RegularGridFEM::template Derivative< Data , Real >& vf , int threads=1 ) const;
	template< class Data , class Real , int DType >
	void divergence( const RegularGridFEM::template Derivative< Data , Real , DType >& v , RegularGridFEM::template Signal< Data , Real >& div , int threads=1 ) const;
	template< class Data , class Real >
	void curl( const RegularGridFEM::template Derivative< Data , Real >& v , RegularGridFEM::template Signal< Data , Real >& crl , int threads=1 ) const;
	template< class Real > double dotProduct( const RegularGridFEM::template Signal< Real , Real >& s1 , const RegularGridFEM::template Signal< Real , Real >& s2 , int threads=1 ) const;
	template< class Real > double dotProduct( const RegularGridFEM::template Derivative< Real , Real >& d1 , const RegularGridFEM::template Derivative< Real , Real >& d2 , int threads=1 ) const;
	template< class Real > double squareNorm( const RegularGridFEM::template Signal< Real , Real >& s , int threads=1 ) const { return dotProduct( s , s , threads ); }
	template< class Real > double squareNorm( const RegularGridFEM::template Derivative< Real , Real >& d , int threads=1 ) const { return dotProduct( d , d , threads ); }
	template< class Real > double squareDifference( const RegularGridFEM::template Signal< Real , Real >& s1 , const RegularGridFEM::template Signal< Real , Real >& s2 , int threads=1 ) const { return squareNorm(s1,threads) + squareNorm(s2,threads) - 2. * dotProduct(s1,s2,threads ); }
	template< class Real > double squareDifference( const RegularGridFEM::template Derivative< Real , Real >& d1 , const RegularGridFEM::template Derivative< Real , Real >& d2 , int threads=1 ) const { return squareNorm(d1,threads) + squareNorm(d2,threads) - 2. * dotProduct(d1,d2,threads ); }

	template< class Real > void poissonFrequencySystem ( BandedMatrix< Real , 1 >& mass , BandedMatrix< Real , 1 >& stiffness ,                    int    frequency , int threads=1 ) const;
	template< class Real > void poissonFrequencySystems( BandedMatrix< Real , 1 >* mass , BandedMatrix< Real , 1 >* stiffness , int minFrequency , int maxFrequency , int threads=1 ) const;
	template< class Real > void poissonSystem( SparseMatrix< Real , int >& mass , SparseMatrix< Real , int >& stiffness , int threads=1 ) const;

	template< class VectorData , class SignalData , class Real >
	double advectBackward( const RegularGridFEM::template Derivative< Real , Real >& vf , bool rotate ,
		RegularGridFEM::template Derivative< VectorData , Real >* inOutFlow ,  RegularGridFEM::template Signal< SignalData , Real >* inOutF ,
		Real delta , Real maxStepSize , int subSteps , int threads=1 ) const;

	template< class VectorData , class SignalData , class Real >
	double advectForward( const RegularGridFEM::template Derivative< Real , Real >& vf , bool rotate ,
		RegularGridFEM::template Derivative< VectorData , Real >* inOutFlow , RegularGridFEM::template Signal< SignalData , Real >* inOutF ,
		Real delta , Real maxStepSize , int subSteps , int threads=1 ) const;

	template< class Real >
	int advectForward( const RegularGridFEM::template Derivative< Real , Real >& vf , bool rotate , Point2D< double >& position , Real delta , Real maxStepSize , int subSteps ) const;

	template< class Real >
	int harmonics( RegularGridFEM::template Derivative< Real , Real >& h1 , RegularGridFEM::template Derivative< Real , Real >& h2 ) const;

	template< class Data , class Real > void dual( const RegularGridFEM::template Signal< Data , Real >& in , RegularGridFEM::template Signal< Data , Real >& out , int threads=1 ) const;
	template< class Data , class Real > void dual( const RegularGridFEM::template Derivative< Data , Real >& in , RegularGridFEM::template Derivative< Data , Real >& out , int threads=1 ) const;

	template< class Real > void sanityCheckSystem( int tests , int threads=1 , double cutOff=0. ) const;

	template< class Data , class Real > static void   ToDoubleCoveringConstraints( Pointer( Data ) constraints , int resX , int resY , RegularGridFEM::GridType gridType , int threads=1 );
	template< class Data , class Real > static void FromDoubleCoveringConstraints( Pointer( Data ) constraints , int resX , int resY , RegularGridFEM::GridType gridType , int threads=1 );

	void write( FILE* fp ) const;
};
#include "SoRMetric.inl"
#endif // SOR_FLOW_FIELD_INCLUDED