/*
Copyright (c) 2013, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <float.h>
#include <algorithm>

#ifndef MOD
#define MOD( a , mod ) ( ( (a)<0 ? (mod) - ( (-a)%(mod) ) : (a) ) % (mod) )
#endif // MOD
#ifndef MOD_F
#define MOD_F( a , mod ) ( (a)/(mod) - floor((a)/(mod)) ) * (mod)
#endif // MOD_F
#ifndef CLAMP
#define CLAMP( a , mn , mx ) ( std::max< int >( (mn) , std::min< int >( (mx) , (a) ) ) )
#endif // CLAMP

////////////////////
// RegularGridFEM //
////////////////////


inline void RegularGridFEM::RemapVertexIndex( int& x , int& y , unsigned int resX , unsigned int resY , GridType gridType , bool& reflectX , bool& reflectY , bool& negate )
{
	unsigned int _resY;
	reflectX = reflectY = negate = false;
	// Modulus:
	// Periodic -> res
	// Dirichlet-Dirichlet / Neumann-Neumann / Pole-Pole -> 2*(res-1)
	// Pole-Dirichlet / Pole-Neumann / Dirichlet-Pole / Neumann-Pole -> 4*(res-1)

	switch( gridType.yType() )
	{
	case BoundaryType::PERIODIC:
		y = MOD( y , resY );
		break;
	case BoundaryType::DIRICHLET_DIRICHLET:
	case BoundaryType::NEUMANN_NEUMANN:
		y = MOD( y , 2*(resY-1) );
		break;
	case BoundaryType::POLE_POLE:
		y = MOD( y , 2*(resY-1) );
		if( y>(int)resY-1 )
		{
			y = 2*(resY-1) - y;
#if MISHA_CODE
			if( resX & 1 ) fprintf( stderr , "[ERROR] RegularGridFEM::RemapVertexIndex: Resolution is not even: %d\n" , resX ) , exit( 0 );
#endif // MISHA_CODE
			x += resX / 2;
			reflectY = !reflectY;
		}
		break;
	case BoundaryType::POLE_DIRICHLET:
	case BoundaryType::POLE_NEUMANN:
		y += resY-1;
		_resY = 2 * (resY-1) + 1;	// The resolution of the sphere after boundary reflection
		y = MOD( y , 2 * (_resY-1) );
		if( y>(int)_resY-1 )
		{
			y = 2*(_resY-1) - y;
#if MISHA_CODE
			if( resX & 1 ) fprintf( stderr , "[ERROR] RegularGridFEM::RemapVertexIndex: Resolution is not even: %d\n" , resX ) , exit( 0 );
#endif // MISHA_CODE
			x += resX /2;
			reflectY = !reflectY;
		}
		y -= resY-1;
		break;
	case BoundaryType::DIRICHLET_POLE:
	case BoundaryType::NEUMANN_POLE:
		_resY = 2 * (resY-1) + 1;	// The resolution of the sphere after boundary reflection
		y = MOD( y , 2 * (_resY-1) );
		if( y>(int)_resY-1 )
		{
			y = 2*(_resY-1) - y;
#if MISHA_CODE
			if( resX & 1 ) fprintf( stderr , "[ERROR] RegularGridFEM::RemapVertexIndex: Resolution is not even: %d\n" , resX ) , exit( 0 );
#endif // MISHA_CODE
			x += resX /2;
			reflectY = !reflectY;
		}
		break;
	default: fprintf( stderr , "[ERROR] RegularGridFEM::RemapIndex: Unrecognized y-type: %d\n" , gridType.yType() ) , exit( 0 );
	}
	if( y>(int)resY-1 )
	{
		y = 2 * ( resY - 1 ) - y;
		reflectY = !reflectY;
		if( gridType.yDirichlet0() || gridType.yDirichlet1() ) negate = !negate;
	}
	if( y<0 )
	{
		y = -y;
		reflectY = !reflectY;
		if( gridType.yDirichlet0() || gridType.yDirichlet1() ) negate = !negate;
	}
	switch( gridType.xType() )
	{
	case BoundaryType::PERIODIC:
		x = MOD( x , resX );
		break;
	case BoundaryType::DIRICHLET_DIRICHLET:
	case BoundaryType::NEUMANN_NEUMANN:
		x = MOD( x , 2*(resX-1) );
		break;
	default: fprintf( stderr , "[ERROR] RegularGridFEM::RemapIndex: Unrecognized x-type: %d\n" , gridType.xType() ) , exit( 0 );
	}
	if( x>(int)resX-1 )
	{
		x = 2 * ( resX - 1 ) - x;
		reflectX = !reflectX;
		if( gridType.xDirichlet() ) negate = !negate;
	}
}
inline void RegularGridFEM::RemapFaceIndex( int& x , int& y , unsigned int resX , unsigned int resY , GridType gridType , bool& reflectX , bool& reflectY , bool& negate )
{
	unsigned int _resY;
	reflectX = reflectY = negate = false;
	// Modulus:
	// Periodic -> res
	// Dirichlet-Dirichlet / Neumann-Neumann / Pole-Pole -> 2*(res-1)
	// Pole-Dirichlet / Pole-Neumann / Dirichlet-Pole / Neumann-Pole -> 4*(res-1)

	switch( gridType.yType() )
	{
	case BoundaryType::PERIODIC:
		y = MOD( y , resY );
		break;
	case BoundaryType::DIRICHLET_DIRICHLET:
	case BoundaryType::NEUMANN_NEUMANN:
		y = MOD( y , 2*(resY-1) );
		break;
	case BoundaryType::POLE_POLE:
		y = MOD( y , 2*(resY-1) );
		if( y>=(int)resY-1 )
		{
			y = 2*(resY-1) - y -1;
#if MISHA_CODE
			if( resX & 1 ) fprintf( stderr , "[ERROR] RegularGridFEM::RemapVertexIndex: Resolution is not even: %d\n" , resX ) , exit( 0 );
#endif // MISHA_CODE
			x += resX / 2;
			reflectY = !reflectY;
		}
		break;
	case BoundaryType::POLE_DIRICHLET:
	case BoundaryType::POLE_NEUMANN:
		_resY = 2 * (resY-1) + 1;	// The resolution of the sphere after boundary reflection
		y = MOD( y , 2 * (_resY-1) );
		if( y>=(int)_resY-1 )
		{
			y = 2*(_resY-1) - y - 1;
#if MISHA_CODE
			if( resX & 1 ) fprintf( stderr , "[ERROR] RegularGridFEM::RemapVertexIndex: Resolution is not even: %d\n" , resX ) , exit( 0 );
#endif // MISHA_CODE
			x += resX /2;
			reflectY = !reflectY;
		}
		break;
	case BoundaryType::DIRICHLET_POLE:
	case BoundaryType::NEUMANN_POLE:
		y += resY-1;
		_resY = 2 * (resY-1) + 1;	// The resolution of the sphere after boundary reflection
		y = MOD( y , 2 * (_resY-1) );
		if( y>=(int)_resY-1 )
		{
			y = 2*(_resY-1) - y - 1;
#if MISHA_CODE
			if( resX & 1 ) fprintf( stderr , "[ERROR] RegularGridFEM::RemapVertexIndex: Resolution is not even: %d\n" , resX ) , exit( 0 );
#endif // MISHA_CODE
			x += resX /2;
			reflectY = !reflectY;
		}
		y -= resY-1;
		break;
	default: fprintf( stderr , "[ERROR] RegularGridFEM::RemapIndex: Unrecognized y-type: %d\n" , gridType.yType() ) , exit( 0 );
	}
	if( y>=(int)resY-1 && !gridType.yPeriodic() )
	{
		y = 2 * ( resY - 1 ) - y - 1;
		reflectY = !reflectY;
		if( gridType.yDirichlet0() || gridType.yDirichlet1() ) negate = !negate;
	}
	if( y<0 )
	{
		y = -y-1;
		reflectY = !reflectY;
		if( gridType.yDirichlet0() || gridType.yDirichlet1() ) negate = !negate;
	}
	switch( gridType.xType() )
	{
	case BoundaryType::PERIODIC:
		x = MOD( x , resX );
		break;
	case BoundaryType::DIRICHLET_DIRICHLET:
	case BoundaryType::NEUMANN_NEUMANN:
		x = MOD( x , 2*(resX-1) );
		break;
	default: fprintf( stderr , "[ERROR] RegularGridFEM::RemapIndex: Unrecognized x-type: %d\n" , gridType.xType() ) , exit( 0 );
	}
	if( x>=(int)resX-1 && !gridType.xPeriodic() )
	{
		x = 2 * ( resX - 1 ) - x;
		reflectX = !reflectX;
		if( gridType.xDirichlet() ) negate = !negate;
	}
}
inline void RegularGridFEM::RemapParameter( double& x , double& y , unsigned int resX , unsigned int resY , GridType gridType , bool& reflectX , bool& reflectY , bool& negate )
{
	reflectX = reflectY = negate = false;
	unsigned int _resY;
	switch( gridType.yType() )
	{
	case BoundaryType::PERIODIC:
		y = MOD_F( y , resY );
		break;
	case BoundaryType::DIRICHLET_DIRICHLET:
	case BoundaryType::NEUMANN_NEUMANN:
		y = MOD_F( y , 2*(resY-1) );
		break;
	case BoundaryType::POLE_POLE:
		y = MOD_F( y , 2*(resY-1) );
		if( y>resY-1 ) y = 2*(resY-1) - y , x += (double)resX / 2 , reflectY = !reflectY;
		break;
	case BoundaryType::POLE_DIRICHLET:
	case BoundaryType::POLE_NEUMANN:
		_resY = 2 * (resY-1) + 1;	// The resolution of the sphere after boundary reflection
		y = MOD_F( y , 2*(_resY-1) );
		if( y>_resY-1 ) y = 2*(_resY-1) - y , x += (double)resX / 2 , reflectY = !reflectY;
		break;
	case BoundaryType::DIRICHLET_POLE:
	case BoundaryType::NEUMANN_POLE:
		y += resY-1;
		_resY = 2 * (resY-1) + 1;	// The resolution of the sphere after boundary reflection
		y = MOD_F( y , 2*(_resY-1) );
		if( y>_resY-1 ) y = 2*(_resY-1) - y , x += (double)resX / 2 , reflectY = !reflectY;
		y -= resY-1;
		break;
	default: fprintf( stderr , "[ERROR] RegularGridFEM::RemapParameter: Unrecognized y-type: %d\n" , gridType.yType() ) , exit( 0 );
	}
	if( y>resY-1 && !gridType.yPeriodic() )
	{
		y = 2 * ( resY - 1 ) - y;
		reflectY = !reflectY;
		if( gridType.yDirichlet0() || gridType.yDirichlet1() ) negate = !negate;
	}
	if( y<0 )
	{
		y = -y;
		reflectY = !reflectY;
		if( gridType.yDirichlet0() || gridType.yDirichlet1() ) negate = !negate;
	}
	switch( gridType.xType() )
	{
	case BoundaryType::PERIODIC:
		x = MOD_F( x , resX );
		break;
	case BoundaryType::DIRICHLET_DIRICHLET:
	case BoundaryType::NEUMANN_NEUMANN:
		x = MOD_F( x , 2*(resX-1) );
		break;
	default: fprintf( stderr , "[ERROR] RegularGridFEM::RemapParameter: Unrecognized x-type: %d\n" , gridType.xType() ) , exit( 0 );
	}
	if( x>resX-1 && !gridType.xPeriodic() )
	{
		x = 2 * ( resX - 1 ) - x;
		reflectX = !reflectX;
		if( gridType.xDirichlet() ) negate = !negate;
	}
}
unsigned int RegularGridFEM::Dim( unsigned int resX , unsigned int resY , GridType gridType )
{
	unsigned int dimX;

	switch( gridType.xType() )
	{
	case BoundaryType::NEUMANN_NEUMANN:
	case BoundaryType::PERIODIC:            dimX = resX   ; break;
	case BoundaryType::DIRICHLET_DIRICHLET: dimX = resX-2 ; break;
	default: fprintf( stderr , "[ERROR] RegularGridFEM::Dim: Unrecognized x-type: %d\n" , gridType.xType() ) , exit( 0 );
	}

	unsigned int poleDim = gridType.xType()==BoundaryType::DIRICHLET_DIRICHLET ? 0 : 1;
	switch( gridType.yType() )
	{
	case BoundaryType::NEUMANN_NEUMANN:
	case BoundaryType::PERIODIC:            return dimX *  resY                 ;
	case BoundaryType::DIRICHLET_DIRICHLET: return dimX * (resY-2)              ;
	case BoundaryType::POLE_POLE:           return dimX * (resY-2) + 2 * poleDim;
	case BoundaryType::POLE_DIRICHLET:
	case BoundaryType::DIRICHLET_POLE:      return dimX * (resY-2) + 1 * poleDim;
	case BoundaryType::POLE_NEUMANN:
	case BoundaryType::NEUMANN_POLE:        return dimX * (resY-1) + 1 * poleDim;
	default: fprintf( stderr , "[ERROR] RegularGridFEM::Dim: Unrecognized y-type: %d\n" , gridType.yType() ) , exit( 0 );
	}
}
unsigned int RegularGridFEM::DXDim( unsigned int resX , unsigned int resY , GridType gridType )
{
	unsigned int dimX;

	switch( gridType.xType() )
	{
	case BoundaryType::NEUMANN_NEUMANN:
	case BoundaryType::DIRICHLET_DIRICHLET: dimX = resX-1 ; break;
	case BoundaryType::PERIODIC:            dimX = resX   ; break;
	default: fprintf( stderr , "[ERROR] RegularGridFEM::DXDim: Unrecognized x-type: %d\n" , gridType.xType() ) , exit( 0 );
	}

	switch( gridType.yType() )
	{
	case BoundaryType::DIRICHLET_DIRICHLET:
	case BoundaryType::POLE_POLE:
	case BoundaryType::POLE_DIRICHLET:
	case BoundaryType::DIRICHLET_POLE:      return dimX * (resY-2);
	case BoundaryType::POLE_NEUMANN:
	case BoundaryType::NEUMANN_POLE:        return dimX * (resY-1);
	case BoundaryType::NEUMANN_NEUMANN:
	case BoundaryType::PERIODIC:            return dimX *  resY   ;
	default: fprintf( stderr , "[ERROR] RegularGridFEM::DXDim: Unrecognized y-type: %d\n" , gridType.yType() ) , exit( 0 );
	}
}
unsigned int RegularGridFEM::DYDim( unsigned int resX , unsigned int resY , GridType gridType )
{
	unsigned int dimX;

	switch( gridType.xType() )
	{
	case BoundaryType::NEUMANN_NEUMANN:
	case BoundaryType::PERIODIC:            dimX = resX   ; break;
	case BoundaryType::DIRICHLET_DIRICHLET: dimX = resX-2 ; break;
	default: fprintf( stderr , "[ERROR] RegularGridFEM::DYDim: Unrecognized x-type: %d\n" , gridType.xType() ) , exit( 0 );
	}

	switch( gridType.yType() )
	{
	case BoundaryType::NEUMANN_NEUMANN:
	case BoundaryType::DIRICHLET_DIRICHLET:
	case BoundaryType::POLE_POLE:
	case BoundaryType::POLE_DIRICHLET:
	case BoundaryType::DIRICHLET_POLE:
	case BoundaryType::POLE_NEUMANN:
	case BoundaryType::NEUMANN_POLE:        return dimX * (resY-1);
	case BoundaryType::PERIODIC:            return dimX *  resY   ;
	default: fprintf( stderr , "[ERROR] RegularGridFEM::DYDim: Unrecognized y-type: %d\n" , gridType.yType() ) , exit( 0 );
	}
}

bool RegularGridFEM::Factor( unsigned int index , unsigned int& x , unsigned int& y , unsigned int resX , unsigned int resY , GridType gridType )
{
	if( index>=Dim( resX , resY , gridType ) ) return false;
	unsigned int poleDim = gridType.xType()==BoundaryType::DIRICHLET_DIRICHLET ? 0 : 1;

	unsigned int dimX = resX;
	if( gridType.xDirichlet() ) dimX-=2;

	if( gridType.yPole0() && poleDim && index==0 ) x = y = 0;
	else
	{
		if( gridType.yPole0() ) index -= poleDim;
		y = index / dimX , x = index % dimX;
		if( gridType.yDirichlet0() || gridType.yPole0() ) y++;
	}
	return true;
}
unsigned int RegularGridFEM::Index( unsigned int x , unsigned int y , unsigned int resX , unsigned int resY , GridType gridType )
{
	unsigned int poleDim = gridType.xType()==BoundaryType::DIRICHLET_DIRICHLET ? 0 : 1;

	if( x>=resX || y>=resY ) return -1;
	if( gridType.xDirichlet()  && ( x==0 || x==resX-1 ) ) return -1;
	if( gridType.yDirichlet0() && y==0                  ) return -1;
	if( gridType.yDirichlet1() && y==resY-1             ) return -1;
	if( gridType.yPole0()      && y==0      && !poleDim ) return -1;
	if( gridType.yPole1()      && y==resY-1 && !poleDim ) return -1;

	if( gridType.yPole0() && y==0 ) return 0;

	unsigned int xDim = resX;
	if( gridType.xDirichlet() ) xDim -= 2  ,  x--;

	unsigned int index = 0;
	if( gridType.yPole0() ) index += poleDim;
	if( gridType.yDirichlet0() || gridType.yPole0() ) y--;
	if( gridType.yPole1() && y==resY-1 ) index += xDim * y;
	else                                 index += xDim * y + x;
	return index;
}
unsigned int RegularGridFEM::DXIndex( unsigned int x , unsigned int y , unsigned int resX , unsigned int resY , GridType gridType )
{
	const BoundaryType &xType = gridType.xBoundary() , &yType = gridType.yBoundary();

	if( (yType.dirichlet0() || yType.pole0() ) && y==0      ) return -1;
	if( (yType.dirichlet1() || yType.pole1() ) && y==resY-1 ) return -1;

	unsigned int dimX = resX;
	if( !xType.periodic() ) dimX--;
	if( x>=dimX ) return -1;
	if( y>=resY ) return -1;
	if( yType.dirichlet0() || yType.pole0() ) y--;
	return y * dimX  + x;
}
unsigned int RegularGridFEM::DYIndex( unsigned int x , unsigned int y , unsigned int resX , unsigned int resY , GridType gridType )
{
	const BoundaryType &xType = gridType.xBoundary() , &yType = gridType.yBoundary();

	if     ( xType.dirichlet0() && x==0      ) return -1;
	else if( xType.dirichlet1() && x==resX-1 ) return -1;

	unsigned int dimX = resX , dimY = resY;
	if( xType.dirichlet0() ) dimX-- , x--;
	if( xType.dirichlet1() ) dimX--;
	if( !yType.periodic() ) dimY--;
	if( x>=dimX ) return -1;
	if( y>=dimY ) return -1;
	return y * dimX  + x;
}
void RegularGridFEM::_SampleOffsetsAndWeights( double x , double y , unsigned int resX , unsigned int resY , GridType gridType , unsigned int offsets[2][2] , double weights[2][2] )
{
	int ix = (int)floor(x) , iy = (int)floor(y) , ix0 , ix1 , iy0 , iy1;
	double dx = x - ix , dy = y - iy;
	const BoundaryType &xType = gridType.xBoundary() , &yType = gridType.yBoundary();
	unsigned int poleDim = 1;

	// If the indices happen to fall exactly on the last band
	if( !xType.periodic() && ix==resX-1 ) ix = resX-2 , dx = 1.;
	if( !yType.periodic() && iy==resY-1 ) iy = resY-2 , dy = 1.;
	ix0 = ix , ix1 = ( ix0 + 1 ) % resX , iy0 = iy , iy1 = ( iy0+1 ) % resY;
	double dx1 = dx , dy1 = dy , dx0 = 1. - dx , dy0 = 1. - dy;
	// For Dirichlet boundary conditions, make sure we don't have an out-of-bounds index
	unsigned int dimX = resX;
	if( xType.type()==BoundaryType::DIRICHLET_DIRICHLET )
	{
		if( ix==0      ) ix0 = 1      , dx0 = 0.;
		if( ix==resX-2 ) ix1 = resX-2 , dx1 = 0.;
		dimX -= 2;
		// In the case that we have y-poles, those rows are also empty
		poleDim = 0;
		if( yType.pole0() && iy==0      ) iy0 = 1      , dy0 = 0.;
		if( yType.pole1() && iy==resY-2 ) iy1 = resY-2 , dy1 = 0.;
	}
	if( yType.dirichlet0() && iy==0      ) iy0 = 1      , dy0 = 0.;
	if( yType.dirichlet1() && iy==resY-2 ) iy1 = resY-2 , dy1 = 0.;

	if( xType.dirichlet0() ) ix0-- , ix1--;
	if( yType.dirichlet0() || yType.pole0() ) iy0-- , iy1--;
	switch( yType.type() )
	{
	case BoundaryType::NEUMANN_NEUMANN:
	case BoundaryType::DIRICHLET_DIRICHLET:
	case BoundaryType::PERIODIC:
		offsets[0][0] = ix0 + iy0 * dimX , offsets[1][0] = ix1 + iy0 * dimX , offsets[0][1] = ix0 + iy1 * dimX , offsets[1][1] = ix1 + iy1 * dimX;
		break;
	case BoundaryType::POLE_DIRICHLET:
	case BoundaryType::POLE_NEUMANN:
		if( iy==0 ) offsets[0][0] = offsets[1][0] = 0 , offsets[0][1] = poleDim + ix0 , offsets[1][1] = poleDim + ix1;
		else        offsets[0][0] = poleDim + ix0 + dimX * iy0 , offsets[1][0] = poleDim + ix1 + dimX * iy0 , offsets[0][1] = poleDim + ix0 + dimX * iy1 , offsets[1][1] = poleDim + ix1 + dimX * iy1;
		break;
	case BoundaryType::DIRICHLET_POLE:
	case BoundaryType::NEUMANN_POLE:
		if( iy==resY-2 ) offsets[0][0] = ix0 + dimX * iy0 , offsets[1][0] = ix1 + dimX * iy0 , offsets[0][1] = offsets[1][1] = dimX * iy1;
		else             offsets[0][0] = ix0 + dimX * iy0 , offsets[1][0] = ix1 + dimX * iy0 , offsets[0][1] = ix0 + dimX * iy1 , offsets[1][1] = ix1 + dimX * iy1;
		break;
	case BoundaryType::POLE_POLE:
		if     ( iy==     0 ) offsets[0][0] = offsets[1][0] = 0 , offsets[0][1] = poleDim + ix0 , offsets[1][1] = poleDim + ix1;
		else if( iy==resY-2 ) offsets[0][0] = poleDim + ix0 + dimX * iy0 , offsets[1][0] = poleDim + ix1 + dimX * iy0 , offsets[0][1] = offsets[1][1] = poleDim + iy1 * dimX;
		else                  offsets[0][0] = poleDim + ix0 + dimX * iy0 , offsets[1][0] = poleDim + ix1 + dimX * iy0 , offsets[0][1] = poleDim + ix0 + dimX * iy1 , offsets[1][1] = poleDim + ix1 + dimX * iy1;
		break;
	}
	weights[0][0] = dx0 * dy0;
	weights[1][0] = dx1 * dy0;
	weights[0][1] = dx0 * dy1;
	weights[1][1] = dx1 * dy1;
}
void RegularGridFEM::_SampleOffsetsAndWeights( double x , double y , unsigned int resX , unsigned int resY , GridType gridType , unsigned int xOffsets[2] , unsigned int yOffsets[2] , double xWeights[2] , double yWeights[2] )
{
	const BoundaryType &xType = gridType.xBoundary() , &yType = gridType.yBoundary();
	int ix = (int)floor(x) , iy = (int)floor(y) , ix0 , ix1 , iy0 , iy1;
	double dx = x - ix , dy = y - iy;

	// If the indices happen to fall exactly on the last band
	if( !xType.periodic() && ix==resX-1 ) ix = resX-2 , dx = 1.;
	if( !yType.periodic() && iy==resY-1 ) iy = resY-2 , dy = 1.;
	ix0 = ix , iy0 = iy , ix1 = ( ix0 + 1 ) % resX , iy1 = ( iy0+1 ) % resY;
	double dx1 = dx , dy1 = dy , dx0 = 1. - dx , dy0 = 1. - dy;
	// For Dirichlet boundary conditions, make sure we don't have an out-of-bounds index
	if( xType.type()==BoundaryType::DIRICHLET_DIRICHLET )
	{
		if( ix==0      ) ix0 = 1      , dx0 = 0.;
		if( ix==resX-2 ) ix1 = resX-2 , dx1 = 0.;
		if( yType.pole0() && iy==0      ) iy0 = 1      , dy0 = 0.;
		if( yType.pole1() && iy==resY-2 ) iy1 = resY-2 , dy1 = 0.;
	}
	if( ( yType.dirichlet0() || yType.pole0() ) && iy==0      ) iy0 = 1      , dy0 = 0.;
	if( ( yType.dirichlet1() || yType.pole1() ) && iy==resY-2 ) iy1 = resY-2 , dy1 = 0.;

	if( xType.dirichlet0() ) ix0-- , ix1--;
	if( yType.dirichlet0() || yType.pole0() ) iy0-- , iy1--;
	unsigned int ddimX = xType.periodic() ? resX : resX-1;
	unsigned int  dimX = xType.type()==BoundaryType::DIRICHLET_DIRICHLET ? resX-2 : resX;

	xWeights[0] = dy0 , xWeights[1] = dy1;
	yWeights[0] = dx0 , yWeights[1] = dx1;

	xOffsets[0] = ix  + iy0 * ddimX , xOffsets[1] = ix  + iy1 * ddimX;
	yOffsets[0] = ix0 + iy *   dimX , yOffsets[1] = ix1 + iy  *  dimX;
}

//////////////////////////////////
// RegularGridFEM::BoundaryType //
//////////////////////////////////
const int RegularGridFEM::BoundaryType::Types[] =
{
	( _0_PERIODIC  | _1_PERIODIC  ) ,
	( _0_DIRICHLET | _1_DIRICHLET ) ,
	( _0_NEUMANN   | _1_NEUMANN   ) ,
	( _0_POLE      | _1_POLE      ) ,
	( _0_POLE      | _1_DIRICHLET ) ,
	( _0_POLE      | _1_NEUMANN   ) ,
	( _0_DIRICHLET | _1_POLE      ) ,
	( _0_NEUMANN   | _1_POLE      )
};
const char* RegularGridFEM::BoundaryType::Names[] =
{
	"Periodic" ,
	"Dirichlet" ,
	"Neumann",
	"Pole-Pole" ,
	"Pole-Dirichlet" ,
	"Pole-Neumann" ,
	"Dirichlet-Pole" ,
	"Neumman-Pole"
};
////////////////////////////
// RegularGridFEM::Signal //
////////////////////////////
template< class Data , class Real >
RegularGridFEM::template Signal< Data , Real >::Signal( void ) { _resX = _resY = 0 , _gridType = GridType( GridType::TOROIDAL ) , _values = NullPointer< Data >(); }
template< class Data , class Real >
RegularGridFEM::template Signal< Data , Real >::Signal( unsigned int resX , unsigned int resY , GridType gridType , bool clear ){ _resX = _resY = 0 , _gridType = GridType( GridType::TOROIDAL ) , _values = NullPointer< Data >() ; resize( resX , resY , gridType , clear ); }
template< class Data , class Real >
RegularGridFEM::template Signal< Data , Real >::Signal( const Signal& s )
{
	resize( s._resX , s._resY , s._gridType );
	memcpy( _values , s._values , sizeof( Data ) * dim() );
}
template< class Data , class Real >
RegularGridFEM::template Signal< Data , Real >::~Signal( void ){ resize( 0 , 0 , GridType( GridType::TOROIDAL ) ); }
template< class Data , class Real >
RegularGridFEM::template Signal< Data , Real >& RegularGridFEM::template Signal< Data , Real >::operator = ( const RegularGridFEM::template Signal< Data , Real >& signal )
{
	unsigned int resX , resY;
	signal.resolution( resX , resY );
	resize( resX , resY , signal.gridType() );
	memcpy( _values , signal._values , sizeof( Data ) * dim() );
	return *this;
}
template< class Data , class Real > unsigned int RegularGridFEM::template Signal< Data , Real >::dim( void ) const { return RegularGridFEM::Dim( _resX , _resY , _gridType ); }
template< class Data , class Real > RegularGridFEM::GridType RegularGridFEM::template Signal< Data , Real >::gridType( void ) const { return _gridType; }
template< class Data , class Real >
void RegularGridFEM::template Signal< Data , Real >::resize( unsigned int resX , unsigned int resY , GridType gridType , bool clear )
{
	unsigned int dim = RegularGridFEM::Dim( resX , resY , gridType );
	if( resX==_resX && resY==_resY && gridType==_gridType )
	{
		if( clear ) memset( _values , 0 , sizeof(Data)*dim );
		return;
	}
	if( _values ) FreePointer( _values );
	_values = NullPointer< Data >();
	_resX = _resY = 0 , _gridType = GridType::TOROIDAL;
	if( !resX || !resY ) return;

#if !MISHA_CODE
	if( ( gridType.yPole0() || gridType.yPole1() ) && (resX&1) ) fprintf( stderr , "[ERROR] RegularGridFEM::Signal::resize: resX must be even\n" ) , exit( 0 );
#endif // !MISHA_CODE
	if( dim>resX*resY )  fprintf( stderr , "[ERROR] RegularGridFEM::Signal::resize: bad dimensions %d x %d -- %s %s\n" , resX , resY , gridType.xName() , gridType.yName() ) , exit( 0 );

	_values = AllocPointer< Data >( dim );
	if( !_values ) fprintf( stderr , "[ERROR] Failed to allocate RegularGridFEM::Signal::_values[ %d ] (%s %s)\n" , dim , gridType.xName() , gridType.yName() ) , exit( 0 );
	_resX = resX , _resY = resY , _gridType = gridType;
	if( clear )	memset( _values , 0 , sizeof( Data )*dim );
}
template< class Data , class Real > void RegularGridFEM::template Signal< Data , Real >::resolution( unsigned int& resX , unsigned int& resY ) const { resX = _resX , resY = _resY; }
template< class Data , class Real >      Pointer( Data ) RegularGridFEM::template Signal< Data , Real >::operator()( void )       { return _values; }
template< class Data , class Real > ConstPointer( Data ) RegularGridFEM::template Signal< Data , Real >::operator()( void ) const { return _values; }
template< class Data , class Real >       Data& RegularGridFEM::template Signal< Data , Real >::operator[]( unsigned int i )       { return _values[i]; }
template< class Data , class Real > const Data& RegularGridFEM::template Signal< Data , Real >::operator[]( unsigned int i ) const { return _values[i]; }
template< class Data , class Real >
Data& RegularGridFEM::template Signal< Data , Real >::operator()( int x , int y )
{
	return const_cast< Data& >( static_cast< const RegularGridFEM::template Signal< Data , Real >& >( *this )( x , y ) );
}
template< class Data , class Real >
const Data& RegularGridFEM::template Signal< Data , Real >::operator()( int x , int y ) const
{
	unsigned int idx = RegularGridFEM::Index( x , y , _resX , _resY , _gridType );
	if( idx==-1 ) fprintf( stderr , "[ERROR] RegularGridFEM::operator(): Bad index (%d,%d) in %d x %d (%s %s)\n" , x , y , _resX , _resY , _gridType.xName() , _gridType.yName() ) , exit( 0 );
	return _values[idx];
}

template< class Data , class Real >
Data RegularGridFEM::template Signal< Data , Real >::sample( double x , double y , bool remap ) const
{
	bool reflectX = false , reflectY = false , negate = false;
	if( remap ) RemapParameter( x ,  y , _resX , _resY , _gridType , reflectX , reflectY , negate );
	Real sign = (Real)( negate ? -1 : 1 );

	unsigned int offsets[2][2];
	double weights[2][2];
	_SampleOffsetsAndWeights( x , y , _resX , _resY , _gridType , offsets , weights );
	return
		(
		_values[ offsets[0][0] ] * (Real)weights[0][0] + _values[ offsets[1][0] ] * (Real)weights[1][0] +
		_values[ offsets[0][1] ] * (Real)weights[0][1] + _values[ offsets[1][1] ] * (Real)weights[1][1] 
		) * sign;
}
template< class Data , class Real >
void RegularGridFEM::template Signal< Data , Real >::splat( double x , double y , Data value , bool remap )
{
	bool reflectX , reflectY , negate;
	if( remap ) RemapParameter( x ,  y , _resX , _resY , _gridType , reflectX , reflectY , negate );

	Real sign = (Real)1.;
	if( _gridType.xDirichlet() && x>=(_resX-1) ) sign *= (Real)-1.;
	if( ( _gridType.yDirichlet0() || _gridType.yDirichlet1() ) && y>=(_resY-1) ) sign *= (Real)-1.;

	unsigned int offsets[2][2];
	double weights[2][2];
	_SampleOffsetsAndWeights( x , y , _resX , _resY , _gridType , offsets , weights );
	_values[ offsets[0][0] ] += value * (Real)weights[0][0] * sign;
	_values[ offsets[1][0] ] += value * (Real)weights[1][0] * sign;
	_values[ offsets[0][1] ] += value * (Real)weights[0][1] * sign;
	_values[ offsets[1][1] ] += value * (Real)weights[1][1] * sign;
}

template< class Data , class Real >
void RegularGridFEM::template Signal< Data , Real >::setFromFaceValues( ConstPointer( Data ) faceValues , bool add , int threads )
{
	// [WARNING] Why is OpenMP unhappy within a "switch" statement?
	// See: https://connect.microsoft.com/VisualStudio/feedback/details/879028/apparent-race-condition-in-generated-object-code-inside-openmp-loop-in-c-in-certain-circumstances
	int poleDim = _gridType.xDirichlet() ? 0 : 1;
	int fDimX = _resX , dimX = _resX;
	if( _gridType.xDirichlet() ) dimX -= 2;
	if( !_gridType.xPeriodic() ) fDimX -= 1;
#pragma omp parallel for num_threads( threads )
	for( int j=0 ; j<(int)_resY ; j++ )
	{
		if( j==0       && ( _gridType.yDirichlet0() || (_gridType.yPole0() && !poleDim) ) ) continue;
		if( j==_resY-1 && ( _gridType.yDirichlet1() || (_gridType.yPole1() && !poleDim) ) ) continue;

		int j0 = (j+_resY-1 ) % _resY , j1 = j;
		ConstPointer( Data ) fValues0 = faceValues + j0 * fDimX;
		ConstPointer( Data ) fValues1 = faceValues + j1 * fDimX;
		Pointer( Data ) values;

		if     ( _gridType.yPole0() )      values = _values + poleDim + (j-1)*dimX;
		else if( _gridType.yDirichlet0() ) values = _values +           (j-1)*dimX;
		else                               values = _values           +  j   *dimX;

		if( _gridType.xDirichlet() ) values--;

		if( j==0 && _gridType.yPole0() ) values = _values;

		if( j==0 && _gridType.yPole0() )
		{
			Data temp;
			memset( &temp , 0 , sizeof(Data) );
			for( int i=0 ; i<(int)fDimX ; i++ ) temp += fValues1[i];
			if( add ) values[0] += temp / (Real)fDimX;
			else      values[0]  = temp / (Real)fDimX;
		}
		else if( j==_resY-1 && _gridType.yPole1() )
		{
			Data temp;
			memset( &temp , 0 , sizeof(Data) );
			for( int i=0 ; i<(int)fDimX ; i++ ) temp += fValues0[i];
			if( add ) values[0] += temp / (Real)fDimX;
			else      values[0]  = temp / (Real)fDimX;
		}
		else if( j==0 && _gridType.yNeumann0() )
		{
			{
				int i=0;
				int i0 = (i+_resX-1) % _resX , i1 = i;
				if     ( _gridType.xPeriodic() )
					if( add ) values[i] += ( fValues1[i0] + fValues1[i1] ) * (Real) 0.5;
					else      values[i]  = ( fValues1[i0] + fValues1[i1] ) * (Real) 0.5;
				else if( _gridType.xNeumann() )
					if( add ) values[i] += (                fValues1[i1] ) * (Real) 1.0;
					else      values[i]  = (                fValues1[i1] ) * (Real) 1.0;
			}
			for( int i=1 ; i<(int)_resX-1 ; i++ )
			{
				int i0 = (i+_resX-1) % _resX , i1 = i;
				if( add ) values[i] += ( fValues1[i0] + fValues1[i1] ) * (Real) 0.5;
				else      values[i]  = ( fValues1[i0] + fValues1[i1] ) * (Real) 0.5;
			}
			{
				int i=_resX-1;
				int i0 = (i+_resX-1) % _resX , i1 = i;
				if     ( _gridType.xPeriodic() )
					if( add ) values[i] += ( fValues1[i0] + fValues1[i1] ) * (Real) 0.5;
					else      values[i]  = ( fValues1[i0] + fValues1[i1] ) * (Real) 0.5;
				else if( _gridType.xNeumann() )
					if( add ) values[i] += ( fValues1[i0]                ) * (Real) 1.0;
					else      values[i]  = ( fValues1[i0]                ) * (Real) 1.0;
			}
		}
		else if( j==_resY-1 && _gridType.yNeumann1() )
		{
			{
				int i=0;
				int i0 = (i+_resX-1) % _resX , i1 = i;
				if     ( _gridType.xPeriodic() ) 
					if( add ) values[i] += ( fValues0[i0] + fValues0[i1] ) * (Real) 0.5;
					else      values[i]  = ( fValues0[i0] + fValues0[i1] ) * (Real) 0.5;
				else if( _gridType.xNeumann() )
					if( add ) values[i] += (                fValues0[i1] ) * (Real) 1.0;
					else      values[i]  = (                fValues0[i1] ) * (Real) 1.0;
			}
			for( int i=1 ; i<(int)_resX-1 ; i++ )
			{
				int i0 = (i+_resX-1) % _resX , i1 = i;
				if( add ) values[i] += ( fValues0[i0] + fValues0[i1] ) * (Real) 0.5;
				else      values[i]  = ( fValues0[i0] + fValues0[i1] ) * (Real) 0.5;
			}
			{
				int i=_resX-1;
				int i0 = (i+_resX-1) % _resX , i1 = i;
				if     ( _gridType.xPeriodic() )
					if( add ) values[i]  = ( fValues0[i0] + fValues0[i1] ) * (Real) 0.5;
					else      values[i] += ( fValues0[i0] + fValues0[i1] ) * (Real) 0.5;
				else if( _gridType.xNeumann() )
					if( add ) values[i] += ( fValues0[i0]                ) * (Real) 1.0;
					else      values[i]  = ( fValues0[i0]                ) * (Real) 1.0;
			}
		}
		else
		{
			{
				int i=0;
				int i0 = (i+_resX-1) % _resX , i1 = i;
				if     ( _gridType.xPeriodic() )
					if( add ) values[i] += ( fValues0[i0] + fValues0[i1] + fValues1[i0] + fValues1[i1] ) * (Real) 0.25;
					else      values[i]  = ( fValues0[i0] + fValues0[i1] + fValues1[i0] + fValues1[i1] ) * (Real) 0.25;
				else if( _gridType.xNeumann() )
					if( add ) values[i] += (                fValues0[i1] +                fValues1[i1] ) * (Real) 0.5;
					else      values[i]  = (                fValues0[i1] +                fValues1[i1] ) * (Real) 0.5;
			}
			for( int i=1 ; i<(int)_resX-1 ; i++ )
			{
				int i0 = i-1 , i1 = i;
				if( add ) values[i] += ( fValues0[i0] + fValues0[i1] + fValues1[i0] + fValues1[i1] ) * (Real) 0.25;
				else      values[i]  = ( fValues0[i0] + fValues0[i1] + fValues1[i0] + fValues1[i1] ) * (Real) 0.25;
			}
			{
				int i=_resX-1;
				int i0 = i-1 , i1 = i;
				if     ( _gridType.xPeriodic() )
					if( add ) values[i] += ( fValues0[i0] + fValues0[i1] + fValues1[i0] + fValues1[i1] ) * (Real) 0.25;
					else      values[i]  = ( fValues0[i0] + fValues0[i1] + fValues1[i0] + fValues1[i1] ) * (Real) 0.25;
				else if( _gridType.xNeumann() )
					if( add ) values[i] += ( fValues0[i0] +                fValues1[i0]                ) * (Real) 0.5;
					else      values[i]  = ( fValues0[i0] +                fValues1[i0]                ) * (Real) 0.5;
			}
		}
	}
}
template< class Data , class Real >
void RegularGridFEM::template Signal< Data , Real >::read( FILE* fp )
{
	GridType gridType;
	unsigned int resX , resY;
	if( fread( &resX , sizeof( unsigned int ) , 1 , fp )!=1 ) fprintf( stderr , "[ERROR] RegularGridFEM::Signal::read: Failed to read resX\n" ) , exit( 0 );
	if( fread( &resY , sizeof( unsigned int ) , 1 , fp )!=1 ) fprintf( stderr , "[ERROR] RegularGridFEM::Signal::read: Failed to read resY\n" ) , exit( 0 );
	gridType.read( fp );
	resize( resX , resY , gridType );
	fread( _values , sizeof(Data) , dim() , fp );
}
template< class Data , class Real >
void RegularGridFEM::template Signal< Data , Real >::write( FILE* fp ) const
{
	fwrite( &_resX , sizeof( unsigned int ) , 1 , fp );
	fwrite( &_resY , sizeof( unsigned int ) , 1 , fp );
	_gridType.write( fp );
	fwrite( _values , sizeof(Data) , dim() , fp );
}

////////////////////////////////
// RegularGridFEM::Derivative //
////////////////////////////////
template< class Data , class Real , int Type >
RegularGridFEM::template Derivative< Data , Real , Type >::Derivative( void ) { _resX = _resY = 0 , _gridType = GridType( GridType::TOROIDAL ) , _values = NullPointer< Data >(); }
template< class Data , class Real , int Type >
RegularGridFEM::template Derivative< Data , Real , Type >::Derivative( unsigned int resX , unsigned int resY , GridType gridType , bool clear ){ _resX = _resY = 0 , _gridType = GridType( GridType::TOROIDAL ) , _values = NullPointer< Data >() ; resize( resX , resY , gridType , clear ); }
template< class Data , class Real , int Type >
RegularGridFEM::template Derivative< Data , Real , Type >::Derivative( const Derivative& d )
{
	resize( d._resX , d._resY , d._gridType );
	memcpy( _values , d._values , sizeof( Data ) * dim() );
}
template< class Data , class Real , int Type >
RegularGridFEM::template Derivative< Data , Real , Type >::~Derivative( void ){ resize( 0 , 0 , RegularGridFEM::GridType( RegularGridFEM::GridType::TOROIDAL ) ); }
template< class Data , class Real , int Type >
RegularGridFEM::template Derivative< Data , Real , Type >& RegularGridFEM::template Derivative< Data , Real , Type >::operator = ( const RegularGridFEM::template Derivative< Data , Real , Type >& derivative )
{
	unsigned int resX , resY;
	derivative.resolution( resX , resY );
	resize( resX , resY , derivative.gridType() );
	memcpy( _values , derivative._values , sizeof( Data ) * dim() );
	return *this;
}


template< class Data , class Real , int Type > unsigned int RegularGridFEM::template Derivative< Data , Real , Type >::dxDim( void ) const { return (Type==DERIVATIVE_X || Type==DERIVATIVE_BOTH) ? RegularGridFEM::DXDim( _resX , _resY , _gridType ) : 0; }
template< class Data , class Real , int Type > unsigned int RegularGridFEM::template Derivative< Data , Real , Type >::dyDim( void ) const { return (Type==DERIVATIVE_Y || Type==DERIVATIVE_BOTH) ? RegularGridFEM::DYDim( _resX , _resY , _gridType ) : 0; }
template< class Data , class Real , int Type > unsigned int RegularGridFEM::template Derivative< Data , Real , Type >::dim( void ) const { return dxDim() + dyDim(); }
template< class Data , class Real , int Type > RegularGridFEM::GridType RegularGridFEM::template Derivative< Data , Real , Type >::gridType( void ) const { return _gridType; }
template< class Data , class Real , int Type >
void RegularGridFEM::template Derivative< Data , Real , Type >::resize( unsigned int resX , unsigned int resY , GridType gridType , bool clear )
{
	unsigned int dim = 0;
	if( Type==DERIVATIVE_X || Type==DERIVATIVE_BOTH ) dim += RegularGridFEM::DXDim( resX , resY , gridType );
	if( Type==DERIVATIVE_Y || Type==DERIVATIVE_BOTH ) dim += RegularGridFEM::DYDim( resX , resY , gridType );
	if( resX==_resX && resY==_resY && gridType==_gridType )
	{
		if( clear ) memset( _values , 0 , sizeof(Data)*dim );
		return;
	}
	if( _values ) FreePointer( _values );
	_values = NullPointer< Data >();
	_resX = _resY = 0 , _gridType = GridType::TOROIDAL;
	if( !resX || !resY ) return;
#if !MISHA_CODE
	if( ( gridType.yPole0() || gridType.yPole1() ) && (resX&1) ) fprintf( stderr , "[ERROR] RegularGridFEM::Derivative::resize: resX must be even\n" ) , exit( 0 );
#endif // !MISHA_CODE
	if( dim>2*resX*resY )  fprintf( stderr , "[ERROR] RegularGridFEM::Derivative::resize: bad dimensions %d x %d -- %s %s\n" , resX , resY , gridType.xName() , gridType.yName() ) , exit( 0 );

	_values = AllocPointer< Data >( dim );
	if( !_values ) fprintf( stderr , "[ERROR] Failed to allocate RegularGridFEM::Derivative::_values[ %d ] (%s %s)\n" , dim , gridType.xName() , gridType.yName() ) , exit( 0 );
	_resX = resX , _resY = resY , _gridType = gridType;
	_xValues = _values , _yValues = _values + dxDim();
	if( clear )	memset( _values , 0 , sizeof( Data )*dim );
}
template< class Data , class Real , int Type > void RegularGridFEM::template Derivative< Data , Real , Type >::resolution( unsigned int& resX , unsigned int& resY ) const { resX = _resX , resY = _resY; }
template< class Data , class Real , int Type >      Pointer( Data ) RegularGridFEM::template Derivative< Data , Real , Type >::operator()( void )       { return _values; }
template< class Data , class Real , int Type > ConstPointer( Data ) RegularGridFEM::template Derivative< Data , Real , Type >::operator()( void ) const { return _values; }
template< class Data , class Real , int Type >      Pointer( Data ) RegularGridFEM::template Derivative< Data , Real , Type >::dx( void )       { return _xValues; }
template< class Data , class Real , int Type > ConstPointer( Data ) RegularGridFEM::template Derivative< Data , Real , Type >::dx( void ) const { return _xValues; }
template< class Data , class Real , int Type >      Pointer( Data ) RegularGridFEM::template Derivative< Data , Real , Type >::dy( void )       { return _yValues; }
template< class Data , class Real , int Type > ConstPointer( Data ) RegularGridFEM::template Derivative< Data , Real , Type >::dy( void ) const { return _yValues; }
template< class Data , class Real , int Type > Data& RegularGridFEM::template Derivative< Data , Real , Type >::dx( int x , int y )
{
	return const_cast< Data& >( static_cast< const RegularGridFEM::template Derivative< Data , Real , Type >* >( this )->dx( x , y ) );
}
template< class Data , class Real , int Type > const Data& RegularGridFEM::template Derivative< Data , Real , Type >::dx( int x , int y ) const
{
	unsigned int idx = RegularGridFEM::DXIndex( x , y , _resX , _resY , _gridType );
	if( idx==-1 ) fprintf( stderr , "[ERROR] RegularGridFEM::Derivative::dx: Bad index (%d,%d) in %d x %d (%s %s)\n" , x , y , _resX , _resY , _gridType.xName() , _gridType.yName() ) , exit( 0 );
	return _xValues[idx];
}
template< class Data , class Real , int Type > Data& RegularGridFEM::template Derivative< Data , Real , Type >::dy( int x , int y )
{
	return const_cast< Data& >( static_cast< const RegularGridFEM::template Derivative< Data , Real , Type >* >( this )->dy( x , y ) );
}
template< class Data , class Real , int Type > const Data&  RegularGridFEM::template Derivative< Data , Real , Type >::dy( int x , int y ) const
{
	unsigned int idx = RegularGridFEM::DYIndex( x , y , _resX , _resY , _gridType );
	if( idx==-1 ) fprintf( stderr , "[ERROR] RegularGridFEM::Derivative::dx: Bad index (%d,%d) in %d x %d (%s %s)\n" , x , y , _resX , _resY , _gridType.xName() , _gridType.yName() ) , exit( 0 );
	return _yValues[idx];
}
template< class Data , class Real , int Type >       Data& RegularGridFEM::template Derivative< Data , Real , Type >::operator[]( unsigned int i )       { return _values[i]; }
template< class Data , class Real , int Type > const Data& RegularGridFEM::template Derivative< Data , Real , Type >::operator[]( unsigned int i ) const { return _values[i]; }
template< class Data , class Real , int Type >
std::pair< Data , Data > RegularGridFEM::template Derivative< Data , Real , Type >::sample( double x , double y , bool remap ) const
{
	bool reflectX = false , reflectY = false , negate = false;
	if( remap ) RemapParameter( x ,  y , _resX , _resY , _gridType , reflectX , reflectY , negate );
	Real signX = (Real)( ( reflectX ^ negate ) ? -1 : 1 );
	Real signY = (Real)( ( reflectY ^ negate ) ? -1 : 1 );

	unsigned int xOffsets[2] , yOffsets[2];
	double xWeights[2] , yWeights[2];
	_SampleOffsetsAndWeights( x , y , _resX , _resY , _gridType , xOffsets , yOffsets , xWeights , yWeights);
	return std::pair< Data , Data >
		(
		( _xValues[ xOffsets[0] ] * (Real)xWeights[0] + _xValues[ xOffsets[1] ] * (Real)xWeights[1] ) * signX ,
		( _yValues[ yOffsets[0] ] * (Real)yWeights[0] + _yValues[ yOffsets[1] ] * (Real)yWeights[1] ) * signY
		);
}
template< class Data , class Real , int Type >
void RegularGridFEM::template Derivative< Data , Real , Type >::setFromFaceValues( ConstPointer( Data ) faceValuePairs , bool add , int threads )
{
	int fDimX = _resX , dimX = _resX , fDimY = _resY;
	if( _gridType.xDirichlet() ) dimX -= 2;
	if( !_gridType.xPeriodic() ) fDimX--;
	if( !_gridType.yPeriodic() ) fDimY--;

	if( Type==DERIVATIVE_X || Type==DERIVATIVE_BOTH )
#pragma omp parallel for num_threads( threads )
		for( int j=0 ; j<(int)_resY ; j++ )
		{
			if( j==0       && ( _gridType.yDirichlet0() || _gridType.yPole0() ) ) continue;
			if( j==_resY-1 && ( _gridType.yDirichlet1() || _gridType.yPole1() ) ) continue;

			int j0 = (j+_resY-1 ) % _resY , j1 = j;
			ConstPointer( Data ) fValues0 = faceValuePairs + j0 * fDimX * 2;
			ConstPointer( Data ) fValues1 = faceValuePairs + j1 * fDimX * 2;
			Pointer( Data ) values;
	
			if( _gridType.yPole0() || _gridType.yDirichlet0() ) values = _xValues + (j-1)*fDimX;
			else                                                values = _xValues +  j   *fDimX;

			if( add )
				if     ( j==0       && _gridType.yNeumann0() ) for( int i=0 ; i<fDimX ; i++ ) values[i] += (                      fValues1[(i<<1)|0] ) * (Real)1.0;
				else if( j==_resY-1 && _gridType.yNeumann1() ) for( int i=0 ; i<fDimX ; i++ ) values[i] += ( fValues0[(i<<1)|0]                      ) * (Real)1.0;
				else                                           for( int i=0 ; i<fDimX ; i++ ) values[i] += ( fValues0[(i<<1)|0] + fValues1[(i<<1)|0] ) * (Real)0.5;
			else
				if     ( j==0       && _gridType.yNeumann0() ) for( int i=0 ; i<fDimX ; i++ ) values[i]  = (                      fValues1[(i<<1)|0] ) * (Real)1.0;
				else if( j==_resY-1 && _gridType.yNeumann1() ) for( int i=0 ; i<fDimX ; i++ ) values[i]  = ( fValues0[(i<<1)|0]                      ) * (Real)1.0;
				else                                           for( int i=0 ; i<fDimX ; i++ ) values[i]  = ( fValues0[(i<<1)|0] + fValues1[(i<<1)|0] ) * (Real)0.5;
			
		}
	if( Type==DERIVATIVE_Y || Type==DERIVATIVE_BOTH )
#pragma omp parallel for num_threads( threads )
		for( int j=0 ; j<fDimY ; j++ )
		{
			ConstPointer( Data ) fValues = faceValuePairs + j * fDimX * 2;
			Pointer( Data ) values = _yValues + j * dimX;
			if( _gridType.xDirichlet() ) values--;

			{
				int i = 0;
				int i0 = (i+_resX-1) % _resX , i1 = i;
				if( _gridType.xPeriodic() )
					if( add ) values[i] += ( fValues[(i0<<1)|1] + fValues[(i1<<1)|1] ) * (Real)0.5;
					else      values[i]  = ( fValues[(i0<<1)|1] + fValues[(i1<<1)|1] ) * (Real)0.5;
				else if( _gridType.xNeumann() )
					if( add ) values[i] += (                      fValues[(i1<<1)|1] ) * (Real)1.0;
					else      values[i]  = (                      fValues[(i1<<1)|1] ) * (Real)1.0;
			}
			for( int i=1 ; i<(int)_resX-1 ; i++ )
			{
				int i0 = (i+_resX-1) % _resX , i1 = i;
				if( add ) values[i] += ( fValues[(i0<<1)|1] + fValues[(i1<<1)|1] ) * (Real)0.5;
				else      values[i]  = ( fValues[(i0<<1)|1] + fValues[(i1<<1)|1] ) * (Real)0.5;
			}
			{
				int i = _resX-1;
				int i0 = (i+_resX-1) % _resX , i1 = i;
				if     ( _gridType.xPeriodic() )
					if( add ) values[i] += ( fValues[(i0<<1)|1] + fValues[(i1<<1)|1] ) * (Real)0.5;
					else      values[i]  = ( fValues[(i0<<1)|1] + fValues[(i1<<1)|1] ) * (Real)0.5;
				else if( _gridType.xNeumann() )
					if( add ) values[i] += ( fValues[(i0<<1)|1]                      ) * (Real)1.0;
					else      values[i]  = ( fValues[(i0<<1)|1]                      ) * (Real)1.0;
			}
		}
}
template< class Data , class Real , int Type >
void RegularGridFEM::template Derivative< Data , Real , Type >::read( FILE* fp )
{
	GridType gridType;
	unsigned int resX , resY;
	if( fread( &resX , sizeof( unsigned int ) , 1 , fp )!=1 ) fprintf( stderr , "[ERROR] RegularGridFEM::Signal::read: Failed to read resX\n" ) , exit( 0 );
	if( fread( &resY , sizeof( unsigned int ) , 1 , fp )!=1 ) fprintf( stderr , "[ERROR] RegularGridFEM::Signal::read: Failed to read resY\n" ) , exit( 0 );
	gridType.read( fp );
	resize( resX , resY , gridType );
	fread( _values , sizeof(Data) , dim() , fp );
}
template< class Data , class Real , int Type >
void RegularGridFEM::template Derivative< Data , Real , Type >::write( FILE* fp ) const
{
	fwrite( &_resX , sizeof( unsigned int ) , 1 , fp );
	fwrite( &_resY , sizeof( unsigned int ) , 1 , fp );
	_gridType.write( fp );
	fwrite( _values , sizeof(Data) , dim() , fp );
}
////////////////////
// RegularGridFEM //
////////////////////
template< class Data , class Real , int Type >
void RegularGridFEM::Differentiate( const typename RegularGridFEM::template Signal< Data , Real >& signal , typename RegularGridFEM::template Derivative< Data , Real , Type >& derivative , int threads )
{
	unsigned int resX , resY;
	GridType gridType = signal.gridType();
	signal.resolution( resX , resY );
	derivative.resize( resX , resY , gridType );
	Pointer( Data ) _xValues = derivative.dx();
	Pointer( Data ) _yValues = derivative.dy();
	ConstPointer( Data ) _values = signal();
	unsigned int poleDim = gridType.xType()==BoundaryType::DIRICHLET_DIRICHLET ? 0 : 1;


	int fDimX = resX , vDimX = resX , fDimY = resY;
	if( gridType.xDirichlet() ) vDimX -= 2;
	if( !gridType.xPeriodic() ) fDimX -= 1;
	if( !gridType.yPeriodic() ) fDimY -= 1;

	// Partials in x
	if( Type==DERIVATIVE_X || Type==DERIVATIVE_BOTH )
#pragma omp parallel for num_threads( threads )
		for( int j=0 ; j<(int)resY ; j++ )
		{
			if( j==0      && (gridType.yDirichlet0() || gridType.yPole0() ) ) continue;
			if( j==resY-1 && (gridType.yDirichlet1() || gridType.yPole1() ) ) continue;

			ConstPointer( Data ) sValues;
			if     ( gridType.yPole0() )      sValues = signal() + poleDim + (j-1) * vDimX;
			else if( gridType.yDirichlet0() ) sValues = signal() +           (j-1) * vDimX;
			else                              sValues = signal() +            j    * vDimX;
			if( gridType.xDirichlet() ) sValues--;
			Pointer( Data ) dValues;
			if( gridType.yPole0() || gridType.yDirichlet0() ) dValues = derivative.dx() + (j-1) * fDimX;
			else                                              dValues = derivative.dx() +  j    * fDimX;

			{
				int i=0;
				if( gridType.xDirichlet() ) dValues[i] = sValues[(i+1)%resX]             ;
				else                        dValues[i] = sValues[(i+1)%resX] - sValues[i];
			}
			for( int i=1 ; i<fDimX-1 ; i++ ) dValues[i] = sValues[(i+1)%resX] - sValues[i];
			{
				int i=fDimX-1;
				if( gridType.xDirichlet() ) dValues[i] =                     - sValues[i];
				else                        dValues[i] = sValues[(i+1)%resX] - sValues[i];
			}
		}

	// Partials in y
	if( Type==DERIVATIVE_Y || Type==DERIVATIVE_BOTH )
#pragma omp parallel for num_threads( threads )
		for( int j=0 ; j<fDimY ; j++ )
		{
			int j0 = j , j1 = (j+1) % resY;
			ConstPointer( Data ) sValues0;
			ConstPointer( Data ) sValues1;
			if     ( gridType.yPole0() )      sValues0 = signal() + poleDim + (j0-1) * vDimX , sValues1 = signal() + poleDim + (j1-1) * vDimX;
			else if( gridType.yDirichlet0() ) sValues0 = signal() +           (j0-1) * vDimX , sValues1 = signal() +           (j1-1) * vDimX;
			else                              sValues0 = signal() +            j0    * vDimX , sValues1 = signal() +            j1    * vDimX;
			Pointer( Data ) dValues = derivative.dy() + j * vDimX;

			if( j==0 && gridType.yPole0() ) sValues0 = signal();

			if     ( j==0       && gridType.yPole0() && poleDim )                                    for( int i=0 ; i<vDimX ; i++ ) dValues[i] = sValues1[i] - sValues0[0];
			else if( j==0       && ( gridType.yDirichlet0() || ( gridType.yPole0() && !poleDim ) ) ) for( int i=0 ; i<vDimX ; i++ ) dValues[i] = sValues1[i];
			else if( j==fDimY-1 && gridType.yPole1() && poleDim )                                    for( int i=0 ; i<vDimX ; i++ ) dValues[i] = sValues1[0] - sValues0[i];
			else if( j==fDimY-1 && ( gridType.yDirichlet1() || ( gridType.yPole1() && !poleDim ) ) ) for( int i=0 ; i<vDimX ; i++ ) dValues[i] =             - sValues0[i];
			else                                                                                     for( int i=0 ; i<vDimX ; i++ ) dValues[i] = sValues1[i] - sValues0[i];
		}
}
