/*
Copyright (c) 2013, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#ifndef REGULAR_GRID_INCLUDED
#define REGULAR_GRID_INCLUDED
#define MISHA_CODE 1

// Toroidal:
// 1] There are _resX * _resY faces
// 2] There are _resX * _resY vertices
// 3] The domain of sampling is [0,_resX) x [0,_resY)
// Spherical:
// 1] There are _resX * (_resY-1) faces
// 2] There are _resX * (_resY-2) + 2 vertices
// 3] The domain of sampling is [0,_resX) x [0,_resY-1]
// Hemi-spherical (neumann): (y=0 is the pole)
// 1] There are _resX * (_resY-1) faces
// 2] There are _resX * (_resY-1) + 1 vertices
// 3] The domain of sampling is [0,_resX) x [0,_resY-1]
// Hemi-spherical (dirichlet): (y=0 is the boundary)
// 1] There are _resX * (_resY-1) faces
// 2] There are _resX * (_resY-2) + 1 vertices
// 3] The domain of sampling is [0,_resX) x [0,_resY-1]
// Cylindrical (neumann):
// 1] There are _resX * (_resY-1) faces
// 2] There are _resX * _resY vertices
// 3] The domain of sampling is [0,_resX) x [0,_resY-1]
// Cylindrical (dirichlet):
// 1] There are _resX * (_resY-1) faces
// 2] There are _resX * (_resY-2) vertices
// 3] The domain of sampling is [0,_resX) x [0,_resY-1]
// Planar (neumann):
// 1] There are (_resX-1) * (_resY-1) faces
// 2] There are _resX * _resY vertices
// 3] The domain of sampling is [0,_resX-1] x [0,_resY-1]
// Planar (neumann):
// 1] There are (_resX-1) * (_resY-1) faces
// 2] There are (_resX-2) * (_resY-2) vertices
// 3] The domain of sampling is [0,_resX-1] x [0,_resY-1]

struct RegularGridFEM
{
	struct BoundaryType
	{
protected:
		int _type;
		enum
		{
			_0_PERIODIC  =   1 ,
			_0_DIRICHLET =   2 ,
			_0_NEUMANN   =   4 ,
			_0_POLE      =   8 ,
			_1_PERIODIC  =  16 ,
			_1_DIRICHLET =  32 ,
			_1_NEUMANN   =  64 ,
			_1_POLE      = 128
		};
public:
		enum
		{
			PERIODIC ,
			DIRICHLET_DIRICHLET ,
			NEUMANN_NEUMANN ,
			POLE_POLE ,
			POLE_DIRICHLET ,
			POLE_NEUMANN ,
			DIRICHLET_POLE ,
			NEUMANN_POLE ,
			COUNT
		};
		static const int Types[];
		static const char* Names[];
		BoundaryType( void ){ _type = PERIODIC; }
		BoundaryType( int type )
		{
			if( type<0 || type>=COUNT ) fprintf( stderr , "[ERROR] Unknown boundary type: %d\n" , type ) , exit( 0 );
			_type = type;
		}
		int type( void ) const { return _type; }
		bool operator== ( const BoundaryType& t ) const { return t._type==_type; }
		bool operator!= ( const BoundaryType& t ) const { return t._type!=_type; }
		bool dirichlet0( void ) const { return ( Types[ _type ] & _0_DIRICHLET )!=0; }
		bool dirichlet1( void ) const { return ( Types[ _type ] & _1_DIRICHLET )!=0; }
		bool neumann0  ( void ) const { return ( Types[ _type ] & _0_NEUMANN   )!=0; }
		bool neumann1  ( void ) const { return ( Types[ _type ] & _1_NEUMANN   )!=0; }
		bool pole0     ( void ) const { return ( Types[ _type ] & _0_POLE      )!=0; }
		bool pole1     ( void ) const { return ( Types[ _type ] & _1_POLE      )!=0; }
		bool periodic  ( void ) const { return _type==PERIODIC; }
		const char* name( void ) const { return Names[_type]; }

		void read( FILE* fp )
		{
			if( fread( &_type , sizeof(int) , 1 , fp )!=1 ) fprintf( stderr , "[ERROR] RegularGridFEM::BoundaryType::read: Failed to read type\n" );
			if( _type<0 || _type>=COUNT ) fprintf( stderr , "[ERROR] RegularGridFEM::BoundaryType::read:Unknown boundary type: %d\n" , _type ) , exit( 0 );
		}
		void write( FILE* fp ) const { fwrite( &_type , sizeof(int) , 1 , fp ); }
	};
	struct GridType
	{
	protected:
		BoundaryType _xBoundary , _yBoundary;
	public:
		static const char* Names[];
		enum
		{
			SPHERICAL ,
			TOROIDAL ,
			HEMISPHERICAL_DIRICHLET ,
			HEMISPHERICAL_NEUMANN ,
			CYLINDRICAL_DIRICHLET ,
			CYLINDRICAL_NEUMANN ,
			COUNT
		};
		GridType( void ){ _xBoundary = _yBoundary = BoundaryType( BoundaryType::PERIODIC ); }
		GridType( BoundaryType xBoundary , BoundaryType yBoundary )
		{
			if( xBoundary.pole0() || xBoundary.pole1() ) fprintf( stderr , "[ERROR] GridType: x-boundary cannot have a pole\n" ) , exit( 0 );
			_xBoundary = xBoundary , _yBoundary = yBoundary;
		}
		GridType( int type )
		{
			switch( type )
			{
			case SPHERICAL:               _xBoundary = BoundaryType( BoundaryType::PERIODIC ) , _yBoundary = BoundaryType( BoundaryType::POLE_POLE           ) ; break;
			case TOROIDAL:                _xBoundary = BoundaryType( BoundaryType::PERIODIC ) , _yBoundary = BoundaryType( BoundaryType::PERIODIC            ) ; break;
			case HEMISPHERICAL_DIRICHLET: _xBoundary = BoundaryType( BoundaryType::PERIODIC ) , _yBoundary = BoundaryType( BoundaryType::DIRICHLET_POLE      ) ; break;
			case HEMISPHERICAL_NEUMANN:   _xBoundary = BoundaryType( BoundaryType::PERIODIC ) , _yBoundary = BoundaryType( BoundaryType::NEUMANN_POLE        ) ; break;
			case CYLINDRICAL_DIRICHLET:   _xBoundary = BoundaryType( BoundaryType::PERIODIC ) , _yBoundary = BoundaryType( BoundaryType::DIRICHLET_DIRICHLET ) ; break;
			case CYLINDRICAL_NEUMANN:     _xBoundary = BoundaryType( BoundaryType::PERIODIC ) , _yBoundary = BoundaryType( BoundaryType::NEUMANN_NEUMANN     ) ; break;
			default: fprintf( stderr , "[ERROR] GridType: Unrecognized type: %d\n" , type ) , exit( 0 );
			}
		}
		const BoundaryType& xBoundary( void ) const { return _xBoundary; }
		const BoundaryType& yBoundary( void ) const { return _yBoundary; }
		bool xDirichlet( void ) const { return _xBoundary.type()==BoundaryType::DIRICHLET_DIRICHLET; }
		bool xNeumann  ( void ) const { return _xBoundary.type()==BoundaryType::NEUMANN_NEUMANN; }
		bool xPeriodic ( void ) const { return _xBoundary.type()==BoundaryType::PERIODIC; }
		const char* xName( void ) const { return _xBoundary.name(); }
		int xType( void ) const { return _xBoundary.type(); }

		bool yDirichlet0( void ) const { return _yBoundary.dirichlet0(); }
		bool yDirichlet1( void ) const { return _yBoundary.dirichlet1(); }
		bool yNeumann0  ( void ) const { return _yBoundary.neumann0();   }
		bool yNeumann1  ( void ) const { return _yBoundary.neumann1();   }
		bool yPole0     ( void ) const { return _yBoundary.pole0();      }
		bool yPole1     ( void ) const { return _yBoundary.pole1();      }
		bool yPeriodic  ( void ) const { return _yBoundary.type()==BoundaryType::PERIODIC; }
		const char* yName( void ) const { return _yBoundary.name(); }
		int yType( void ) const { return _yBoundary.type(); }

		bool operator == ( const GridType& g ) const { return _xBoundary==g._xBoundary && _yBoundary==g._yBoundary; }
		bool operator != ( const GridType& g ) const { return _xBoundary!=g._xBoundary || _yBoundary!=g._yBoundary; }

		void read( FILE* fp ){ _xBoundary.read(fp) , _yBoundary.read(fp); }
		void write( FILE* fp ) const { _xBoundary.write(fp) , _yBoundary.write(fp); }
	};

	// [WARNING] These differ from the earlier "Remap" functions in that the x and y values may not correspond to an indexable position.
	// Specifically, if the grid has dirichlet/neumann boundaries, the x and y values can be in the range [0,2*res-1] (no [0,res] or [1,res-1])
	static void RemapVertexIndex(    int& x ,    int& y , unsigned int resX , unsigned int resY , GridType gridType , bool& reflectX , bool& reflectY , bool& negate );
	static void RemapFaceIndex  (    int& x ,    int& y , unsigned int resX , unsigned int resY , GridType gridType , bool& reflectX , bool& reflectY , bool& negate );
	static void RemapParameter  ( double& x , double& y , unsigned int resX , unsigned int resY , GridType gridType , bool& reflectX , bool& reflectY , bool& negate );

	static unsigned int   Dim( unsigned int resX , unsigned int resY , GridType gridType );
	static unsigned int DXDim( unsigned int resX , unsigned int resY , GridType gridType );
	static unsigned int DYDim( unsigned int resX , unsigned int resY , GridType gridType );

	static bool Factor( unsigned int index , unsigned int& x , unsigned int& y , unsigned int resX , unsigned int resY , GridType gridType );
	static unsigned int   Index( unsigned int x , unsigned int y , unsigned int resX , unsigned int resY , GridType gridType );
	static unsigned int DXIndex( unsigned int x , unsigned int y , unsigned int resX , unsigned int resY , GridType gridType );
	static unsigned int DYIndex( unsigned int x , unsigned int y , unsigned int resX , unsigned int resY , GridType gridType );
protected:
	// [NOTE] The assumption is that when this function gets called, the x and y values are within the appropriate bounds
	static void _SampleOffsetsAndWeights( double x , double y , unsigned int resX , unsigned int resY , GridType gridType , unsigned int offsets[2][2] , double weights[2][2] );
	static void _SampleOffsetsAndWeights( double x , double y , unsigned int resX , unsigned int resY , GridType gridType , unsigned int xOffsets[2] , unsigned int yOffsets[2] , double xWeights[2] , double yWeights[2] );
public:


	template< class Data , class Real >
	class Signal
	{
protected:
		GridType _gridType;
		unsigned int _resX , _resY;
		Pointer( Data ) _values;
public:
		~Signal( void );
		Signal( void );
		Signal( unsigned int resX , unsigned int resY , GridType gridType , bool clear=false );
		Signal( const Signal& s );
		Signal& operator = ( const Signal& signal );
		void resize( unsigned int resX , unsigned int resY , GridType gridType , bool clear=false );
		void resolution( unsigned int& resX , unsigned int& resY ) const;
		unsigned int dim( void ) const;
		GridType gridType( void ) const;
		Pointer( Data ) operator()( void );
		Data& operator[] ( unsigned int i );
		const Data& operator[] ( unsigned int i ) const;
		ConstPointer( Data ) operator()( void ) const;
		const Data& operator() ( int x , int y ) const;
		Data& operator() ( int x , int y );
		Data sample( double x , double y , bool remap=true ) const;
		void splat( double x , double y , Data value , bool remap=true );

		void setFromFaceValues( ConstPointer( Data ) faceValues , bool add , int threads=1 );

		void write( FILE* fp ) const;
		void read( FILE* fp );
	};

	enum
	{
		DERIVATIVE_X ,
		DERIVATIVE_Y ,
		DERIVATIVE_BOTH
	};
	template< class Data , class Real , int Type=DERIVATIVE_BOTH >
	class Derivative
	{
	protected:
		GridType _gridType;
		unsigned int _resX , _resY;
		Pointer( Data ) _values;
		Pointer( Data ) _xValues;
		Pointer( Data ) _yValues;
public:
		Derivative( void );
		~Derivative( void );
		Derivative( unsigned int resX , unsigned int resY , GridType gridType , bool clear=false );
		Derivative( const Derivative& d );
		Derivative& operator = ( const Derivative& derivative );
		void resize( unsigned int resX , unsigned int resY , GridType gridType , bool clear=false );
		void resolution( unsigned int& resX , unsigned int& resY ) const;
		unsigned int dxDim( void ) const;
		unsigned int dyDim( void ) const;
		unsigned int   dim( void ) const;
		GridType gridType( void ) const;
		Pointer( Data ) operator()( void );
		Pointer( Data ) dx( void );
		Pointer( Data ) dy( void );
		Data& dx( int x , int y );
		Data& dy( int x , int y );
		Data& operator[] ( unsigned int i );
		ConstPointer( Data ) operator()( void ) const;
		ConstPointer( Data ) dx( void ) const;
		ConstPointer( Data ) dy( void ) const;
		const Data& dx( int x , int y ) const;
		const Data& dy( int x , int y ) const;
		const Data& operator[] ( unsigned int i ) const;
		std::pair< Data , Data > sample( double x , double y , bool remap=true ) const;

		void setFromFaceValues( ConstPointer( Data ) faceValuePairs , bool add , int threads=1 );

		void write( FILE* fp ) const;
		void read( FILE* fp );
	};
	template< class Data , class Real , int DType >
	static void Differentiate( const Signal< Data , Real >& signal , Derivative< Data , Real , DType >& derivative , int threads=1 );
};
#include "RegularGridFEM.inl"
#endif // REGULAR_GRID_INCLUDED

