#ifndef CURVE_INCLUDED
#define CURVE_INCLUDED
#include <algorithm>
#include <Util/Geometry.h>

template< class Real >
struct Curve
{
	enum
	{
		CURVE_CLOSED ,
		CURVE_OPEN ,
		CURVE_CLOSED_REFLECTED ,
		CURVE_OPEN_REFLECTED ,
		CURVE_COUNT
	};
	static inline bool Reflected( int curveType ) { return curveType==CURVE_CLOSED_REFLECTED || curveType==CURVE_OPEN_REFLECTED; }
	static inline bool Closed( int curveType ) { return curveType==CURVE_CLOSED || curveType==CURVE_CLOSED_REFLECTED; }
	int type;
	std::vector< Point2D< Real > > points;

	Curve( void ){ type = CURVE_CLOSED; }
	bool read( const char* fileName )
	{
		FILE* fp = fopen( fileName , "r" );
		if( !fp ) return false;
		int s;
		fscanf( fp , " %d %d " , &type , &s );
		points.resize( s );
		float x , y;
		for( int i=0 ; i<points.size() ; i++ ) fscanf( fp , " %f %f " , &x , &y ) , points[i] = Point2D< Real >( (Real)x , (Real)y );
		fclose( fp );
		return true;
	}
	bool write( const char* fileName ) const
	{
		FILE* fp = fopen( fileName , "w" );
		if( !fp ) return false;
		fprintf( fp , "%d %d\n" , type , points.size() );
		for( int i=0 ; i<points.size() ; i++ ) fprintf( fp , "%f %f\n" , points[i][0] , points[i][1] );
		fclose( fp );
		return true;
	}
	void addPoint( Point2D< Real > p ){ points.push_back( p ); }
	void addPoint( Point2D< Real > p , int idx )
	{
		idx = std::max< int >( 0 , std::min< int >( idx , (int)points.size() ) );
		std::vector< Point2D< Real > > temp( points.size()+1 );
		for( int i=0 ; i<idx ; i++ ) temp[i] = points[i];
		temp[idx] = p;
		for( int i=idx ; i<points.size() ; i++ ) temp[i+1] = points[i];
		points = temp;
	}
	void deletePoint( int idx )
	{
		if( idx>=0 && idx<points.size() )
		{
			std::vector< Point2D< Real > > temp( points.size()-1 );
			for( int i=0 ; i<idx ; i++ ) temp[i] = points[i];
			for( int i=idx+1 ; i<points.size() ; i++ ) temp[i-1] = points[i];
			points = temp;
		}
	}
	void reset( void ){ points.resize( 0 ) , type = CURVE_CLOSED; }
	const int size( void ) const { return (int)( Curve< float >::Reflected( type ) ? points.size()*2 : points.size() ); }
	const Point2D< Real > operator[]( int idx ) const
	{
		int sz = size();
		while( idx<0 ) idx += sz;
		idx %= sz;
		if( !Curve< float >::Reflected( type ) || idx<points.size() ) return points[idx];
		else{ idx = sz-1-idx ; return Point2D< Real >( -points[idx][0] , points[idx][1] ); }
	}
	const Point2D< Real > operator() ( double t , bool linear=false ) const
	{
		Point2D< Real > p[4];
		Real w[4];
#if 1
		if( Curve< float >::Closed( type ) ) t *= size() , t-= 0.5;
		else t *= size()-1;
#else
		t *= size();
		t -= 0.5;
#endif
		int _t = (int)floor(t);
		double dt = t - _t;
		if( Closed( type ) ) for( int i=0 ; i<4 ; i++ ) p[i] = (*this)[_t-1+i];
		else for( int i=0 ; i<4 ; i++ ) p[i] = (*this)[std::max< int >( 0 , std::min< int >( _t-1+i , size()-1 ) ) ];
		if( linear ) return p[1]*(Real)(1.-dt) + p[2]*(Real)dt;
		else
		{
			double t1 = dt , t2 = t1*dt , t3 = t2*dt;
			w[0] = (Real)( -1./6 * t3 + 1./2 * t2 - 1./2 * t1 + 1./6 );
			w[1] = (Real)(  1./2 * t3 -        t2             + 2./3 );
			w[2] = (Real)( -1./2 * t3 + 1./2 * t2 + 1./2 * t1 + 1./6 );
			w[3] = (Real)(  1./6 * t3                                );
			return p[0]*w[0] + p[1]*w[1] + p[2]*w[2] + p[3]*w[3];
		}
	}
	Real Length( bool linear=false , int samples=1e6 ) const
	{
		Real l = (Real)0;
		for( int i=0 ; i<samples ; i++ )
		{
			Real x1 = (Real)i / samples , x2 = (Real)(i+1) / samples;
			if( Curve< float >::Reflected( type ) ) x1 /= 2 , x2 /= 2;
			Point2D< Real > p1 = (*this)( x1 , linear ) , p2 = (*this)( x2 , linear );
			l += (Real)sqrt( (p1-p2).squareNorm() );
		}
		return Curve< float >::Reflected( type ) ? 2*l : l;
	}
	void sample( std::vector< Point2D< Real > >& samples , int count , bool primal , double tStart=0. , double tEnd=1. , bool linear=false )
	{
		samples.resize( count );
		for( int i=0 ; i<count ; i++ )
		{
			Real t = (Real)( tStart + ( primal ? (Real)i / (count-1) : (Real)(i+0.5) / count ) * ( tEnd - tStart ) );
			samples[i] = (*this)( t , linear );
		}
	}
	void uniformSample( std::vector< Point2D< Real > >& samples , int count , bool primal , bool linear=false , int subSamples=1e6 )
	{
		samples.resize( count );
		Real length = Length( linear , subSamples );
		if( Curve< float >::Reflected( type ) ) length /= 2;
		int ii = 0;
		Real oldL = 0;
		Real dL;
		Point2D< Real > oldP = (*this)( (Real)0 , linear );
		Point2D< Real > newP = oldP;
		for( int i=0 ; i<count ; i++ )
		{
			Real l = ( primal ? (Real)i / (count-1) : (Real)(i+0.5) / count ) * length;
			if( l<oldL ) fprintf( stderr , "[ERROR] unsynched: %g < %g\n" , l , oldL ) , exit( 0 );
			Real d;
			if( !i && primal ) d=0;
			else
			{
				while( 1 )
				{
					Real x = (Real)(ii+1) / subSamples;
					if( Curve< float >::Reflected( type ) ) x /= 2;
					newP = (*this)( x , linear );
					dL = (Real)sqrt( (oldP-newP).squareNorm() );
					if( l<=oldL+dL ) break;
					else oldL += dL , oldP = newP , ii++;
				}
				// found ii such that: length[ii]<l<=length[ii+1]
				d = ( oldL + dL - l ) / dL;
			}
			samples[i] = oldP * d + newP * (1.-d);
		}
	}
};

#endif // CURVE_INCLUDED