#define MINIMAL_UI 1

#include <stdio.h>
#include <stdlib.h>
#include <omp.h>
#include <Visualization/SoRFlowVisualization.inl>
#include <Visualization/GeneratingCurveVisualization.inl>
#include <Util/Timer.h>
#include <Util/CmdLineParser.h>
#include <Util/SoRMetric.h>
#include <Util/Solvers.h>


cmdLineParameter< char* > In( "in" );
cmdLineParameter< int > Resolution( "res" , 1024 );
cmdLineParameter< int > CurveResolution( "cRes" , 512 );
cmdLineParameter< int > Threads( "threads" , omp_get_num_procs() );
cmdLineParameter< int > SubSteps( "steps" , 1 );
cmdLineParameter< float > StepSize( "stepSize" , 1.f );
cmdLineParameter< float > Viscosity( "viscosity" , (float)1e-3 );
cmdLineParameter< float > DisolveRate( "disolve" , 1.f );
cmdLineParameter< float > InkWidth( "width" , 0.1f );
cmdLineParameter< float > MaxStepSize( "maxStepSize" , -1.f );
cmdLineParameter< float > Angle( "angle" );
cmdLineReadable Pause( "pause" );
cmdLineReadable Single( "single" );
cmdLineReadable Vorticity( "vorticity" );
#if USE_DIRECT_SOLVER
cmdLineReadable UseDirectSolver( "direct" );
#endif // USE_DIRECT_SOLVER
cmdLineReadable* params[] =
{
	&In , &Resolution , &SubSteps , &MaxStepSize , &CurveResolution , &Threads , &StepSize , &Viscosity , &DisolveRate , &InkWidth , &Single , &Vorticity , &Pause , &Angle ,
#if USE_DIRECT_SOLVER
	&UseDirectSolver
#endif // USE_DIRECT_SOLVER
	NULL
};


void ShowUsage( const char* ex )
{
	printf( "Usage %s:\n" , ex );
#if !MINIMAL_UI
	printf( "\t --%s <curve name>\n" , In.name );
#endif // !MINIMAL_UI
	printf( "\t --%s <angular resolution>\n" , Resolution.name );
	printf( "\t[--%s <curve resolution>]\n" , CurveResolution.name );
	printf( "\t[--%s <parallelization threads>=%d]\n" , Threads.name , Threads.value );
#if !MINIMAL_UI
	printf( "\t[--%s <step size>=%f]\n" , StepSize.name , StepSize.value );
	printf( "\t[--%s <viscosity value>=%f]\n" , Viscosity.name , Viscosity.value );
	printf( "\t[--%s <ink disolve rate>=%f]\n" , DisolveRate.name , DisolveRate.value );
	printf( "\t[--%s <ink width>=%f]\n" , InkWidth.name , InkWidth.value );
	printf( "\t[--%s <maximum advection step-size>=%f]\n" , MaxStepSize.name , MaxStepSize.value );
	printf( "\t[--%s <advection sub-steps>=%d]\n" , SubSteps.name , SubSteps.value );
	printf( "\t[--%s <revolution angle (in degrees)>]\n" , Angle.name );
#endif // !MINIMAL_UI
#if USE_DIRECT_SOLVER
	printf( "\t[--%s]\n" , UseDirectSolver.name );
#endif // USE_DIRECT_SOLVER
	printf( "\t[--%s]\n" , Single.name );
}

template< class Real >
struct CurveAndFlowVisualizationViewer
{
	static Visualization* visualization;
	static Curve< float > curve;
	static SoRFlowVisualization< Real > fv;
	static GeneratingCurveVisualization gcv;
	static void Init( void );
	static void Idle        ( void );
	static void KeyboardFunc( unsigned char key , int x , int y );
	static void SpecialFunc ( int key, int x, int y );
	static void Display     ( void );
	static void Reshape     ( int w , int h );
	static void MouseFunc   ( int button , int state , int x , int y );
	static void MotionFunc  ( int x , int y );
	static void SetCurve( Visualization* , const char* );
	static void SetSoR( Visualization* , const char* );
	static void SetAngle( Visualization* , const char* );
	static void SetInfo( Visualization* , void* );
};

template< class Real > Visualization* CurveAndFlowVisualizationViewer< Real >::visualization = NULL;
template< class Real > Curve< float > CurveAndFlowVisualizationViewer< Real >::curve;
template< class Real > SoRFlowVisualization< Real> CurveAndFlowVisualizationViewer< Real >::fv;
template< class Real > GeneratingCurveVisualization CurveAndFlowVisualizationViewer< Real >::gcv( curve );
template< class Real > void CurveAndFlowVisualizationViewer< Real >::SetCurve( Visualization* , const char* )
{
	visualization = &gcv;
}
template< class Real > void CurveAndFlowVisualizationViewer< Real >::SetInfo( Visualization* v , void* )
{
	if( Angle.value ) v->addInfoString( "Angle: %.1f" , Angle.value );
	else              v->addInfoString( "Angle: %.1f" , 360.        );
}
template< class Real > void CurveAndFlowVisualizationViewer< Real >::SetAngle( Visualization* , const char* prompt )
{
	float v = (float)atof( prompt );
	if( strlen(prompt) && v>=-360.f && v<=360.f && v!=0.f )
	{
		Angle.set = true;
		Angle.value = v;
	}
	else Angle.set = false , Angle.value = 0.f;
}

template< class Real > void CurveAndFlowVisualizationViewer< Real >::SetSoR( Visualization* , const char* )
{
	int sz = CurveResolution.value;
	std::vector< Point2D< double > > _curve( CurveResolution.value );
	std::vector< Point2D< float > > samples;
	curve.sample( samples , sz , curve.type!=Curve< float >::CURVE_CLOSED , 0. , Curve< float >::Reflected( curve.type ) ? 0.5 : 1.0 , gcv.linear );
	for( int i=0 ; i<samples.size() ; i++ ) _curve[i] = Point2D< double >( samples[i][0] , samples[i][1] );
	if( curve.type==Curve< float >::CURVE_CLOSED_REFLECTED ) _curve[0][0] = _curve.back()[0] = 0;
	else if( curve.type==Curve< float >::CURVE_OPEN_REFLECTED ) _curve.back()[0] = 0;
	double scale = 0;
	for( int i=0 ; i<_curve.size() ; i++ ) scale = std::max< double >( scale , _curve[i].squareNorm() );
	scale = sqrt(scale);
	for( int i=0 ; i<_curve.size() ; i++ ) _curve[i] /= scale;

	RegularGridFEM::GridType gridType;
	RegularGridFEM::BoundaryType xType , yType;
	int resolution = Resolution.value;
	if( fabs( Angle.value )>360.f ) fprintf( stderr , "[WARNING] Setting to periodic\n" ) , Angle.set = false;
	if( Angle.set )
	{
		xType = RegularGridFEM::BoundaryType( RegularGridFEM::BoundaryType::DIRICHLET_DIRICHLET );
		fprintf( stderr , "[WARNING] CurveAndFlowVisualizationViewer::SetSoR: incrementing resolution for boundary constraints for efficiency\n" );
		resolution++;
	}
	else xType = RegularGridFEM::BoundaryType( RegularGridFEM::BoundaryType::PERIODIC );

	if     ( curve.type==Curve< float >::CURVE_CLOSED           ) yType = RegularGridFEM::BoundaryType( RegularGridFEM::BoundaryType::PERIODIC );
	else if( curve.type==Curve< float >::CURVE_CLOSED_REFLECTED ) yType = RegularGridFEM::BoundaryType( RegularGridFEM::BoundaryType::POLE_POLE );
	else if( curve.type==Curve< float >::CURVE_OPEN             ) yType = RegularGridFEM::BoundaryType( RegularGridFEM::BoundaryType::DIRICHLET_DIRICHLET );
	else if( curve.type==Curve< float >::CURVE_OPEN_REFLECTED   ) yType = RegularGridFEM::BoundaryType( RegularGridFEM::BoundaryType::DIRICHLET_POLE );
	gridType = RegularGridFEM::GridType( xType , yType );

#if USE_DIRECT_SOLVER
	fv.init( resolution , sz , gridType , GetPointer( _curve ) , true , UseDirectSolver.set , (Real)( fabs( Angle.value ) * 2. * PI / 360. ) );
#else // !USE_DIRECT_SOLVER
	fv.init( resolution , sz , gridType , GetPointer( _curve ) , true , (Real)( fabs( Angle.value ) * 2. * PI / 360. ) );
#endif // USE_DIRECT_SOLVER
	visualization = &fv;
}
template< class Real > void CurveAndFlowVisualizationViewer< Real >::Init( void )
{
	if( In.set ) curve.read( In.value );
	else
	{
		curve.points.resize( 0 );
		curve.type = Curve< float >::CURVE_OPEN;
		curve.addPoint( Point2D< float >(  0.75f, -0.75f ) );
		curve.addPoint( Point2D< float >(  0.75f,  0.75f ) );
	}

	fv.camera = Camera( Point3D< double >( 0 , 4 , 0 ) , Point3D< double >( 0 , -1 , 0 ) , Point3D< double >( 1 , 0 , 0 ) );
	gcv.camera = Camera( Point3D< double >( 0 , 0 , 0 ) , Point3D< double >( 0 , 0 , -1 ) , Point3D< double >( 0 , 1 , 0 ) );
	gcv.res = CurveResolution.value;
	fv.project      = true;
	fv.advect       = true;
	fv.harmonic     = true;
	fv.viscosity    = Viscosity.value;
	fv.disolve      = DisolveRate.value;
	fv.threads      = Threads.value;
	fv.maxStepSize  = MaxStepSize.value;
	fv.subSteps     = SubSteps.value;
	fv.stepSize     = StepSize.value;
	fv.useVorticity = Vorticity.set;
	fv.inkWidth     = InkWidth.value;
	if( Pause.set ) fv.advanceCount = 0;

	fv.keyboardCallBacks.push_back ( Visualization::KeyboardCallBack( NULL , 9 , "show curve" , SetCurve , false ) );
	gcv.keyboardCallBacks.push_back( Visualization::KeyboardCallBack( NULL , 'a' , "angle" , "Angle of Revolution" , SetAngle ) );
	gcv.keyboardCallBacks.push_back( Visualization::KeyboardCallBack( NULL , 9 , "show sor" , SetSoR , false ) );
	gcv.infoCallBacks.push_back( Visualization::InfoCallBack( SetInfo , NULL ) );
	if( In.set ) SetSoR  ( NULL , NULL );
	else         SetCurve( NULL , NULL );
}
template< class Real > void CurveAndFlowVisualizationViewer< Real >::Idle( void ){ visualization->Idle(); }
template< class Real > void CurveAndFlowVisualizationViewer< Real >::KeyboardFunc( unsigned char key , int x , int y ){ visualization->KeyboardFunc( key , x , y ); }
template< class Real > void CurveAndFlowVisualizationViewer< Real >::SpecialFunc( int key , int x , int y ){ visualization->SpecialFunc( key , x ,  y ); }
template< class Real > void CurveAndFlowVisualizationViewer< Real >::Display( void ){ visualization->Display(); }
template< class Real > void CurveAndFlowVisualizationViewer< Real >::Reshape( int w , int h ){ fv.Reshape( w , h ) , gcv.Reshape( w , h ); }
template< class Real > void CurveAndFlowVisualizationViewer< Real >::MouseFunc( int button , int state , int x , int y ){ visualization->MouseFunc( button , state , x , y ); }
template< class Real > void CurveAndFlowVisualizationViewer< Real >::MotionFunc( int x , int y ){ visualization->MotionFunc( x , y ); }
template< class Real >
int _main( int argc , char* argv[] )
{
	if( !CurveResolution.set ) CurveResolution.value = Resolution.value;
	int resX = Resolution.value , resY = CurveResolution.value;
	CurveAndFlowVisualizationViewer< Real >::Init( );
	glutInitDisplayMode( GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH );
	glutInitWindowSize( CurveAndFlowVisualizationViewer< Real >::visualization->screenWidth , CurveAndFlowVisualizationViewer< Real >::visualization->screenHeight );
	glutInit( &argc , argv );
	char windowName[1024];
	sprintf( windowName , "SoR Flow" );
	glutCreateWindow( windowName );

	if( glewInit()!=GLEW_OK ) fprintf( stderr , "[ERROR] glewInit failed\n" ) , exit( 0 );
	glutIdleFunc    ( CurveAndFlowVisualizationViewer< Real >::Idle );
	glutDisplayFunc ( CurveAndFlowVisualizationViewer< Real >::Display );
	glutReshapeFunc ( CurveAndFlowVisualizationViewer< Real >::Reshape );
	glutMouseFunc   ( CurveAndFlowVisualizationViewer< Real >::MouseFunc );
	glutMotionFunc  ( CurveAndFlowVisualizationViewer< Real >::MotionFunc );
	glutKeyboardFunc( CurveAndFlowVisualizationViewer< Real >::KeyboardFunc );
	glutSpecialFunc ( CurveAndFlowVisualizationViewer< Real >::SpecialFunc );

	glutMainLoop();
	return EXIT_SUCCESS;
}
int main( int argc , char* argv[] )
{
	cmdLineParse( argc-1 , argv+1 , params );
	if( !Resolution.set && !In.set ){ ShowUsage( argv[0] ) ; return EXIT_FAILURE; }
	if( Vorticity.set && !MaxStepSize.set ) MaxStepSize.value = 1.f;
	return Single.set ? _main< float >( argc , argv ) : _main< double >( argc , argv );
}