#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include "JPEG/jpeglib.h"

////////////////////////////////////////////////////////////////////
// Code adapted from example.c provided with libjpeg distribution //
////////////////////////////////////////////////////////////////////

struct my_error_mgr
{
	struct jpeg_error_mgr pub;    // "public" fields
	jmp_buf setjmp_buffer;        // for return to caller
};
typedef struct my_error_mgr * my_error_ptr;

METHODDEF(void)
my_error_exit (j_common_ptr cinfo)
{
	// cinfo->err really points to a my_error_mgr struct, so coerce pointer
	my_error_ptr myerr = (my_error_ptr) cinfo->err;
	
	// Always display the message.
	// We could postpone this until after returning, if we chose.
	(*cinfo->err->output_message) (cinfo);
	
	// Return control to the setjmp point
	longjmp(myerr->setjmp_buffer, 1);
}
struct JPEGReadInfo
{
	FILE* fp;
	struct jpeg_decompress_struct cInfo;
	struct my_error_mgr	jErr;
};
struct JPEGWriteInfo
{
	FILE* fp;
	struct jpeg_compress_struct cInfo;
	struct jpeg_error_mgr jErr;
};

void* JPEGInitWrite(char* fileName,int width,int height,int quality)
{
	JPEGWriteInfo *info=(JPEGWriteInfo*)malloc(sizeof(JPEGWriteInfo));

	info->fp=fopen(fileName,"wb");
	if(!info->fp)	fprintf(stderr,"Failed to open: %s\n",fileName),	exit(0);
	
	info->cInfo.err=jpeg_std_error(&info->jErr);
	jpeg_create_compress(&info->cInfo);
	
	jpeg_stdio_dest(&info->cInfo,info->fp);
	
	info->cInfo.image_width = width;    /* image width and height, in pixels */
	info->cInfo.image_height = height;
	info->cInfo.input_components = 3;           /* # of color components per pixel */
	info->cInfo.in_color_space = JCS_RGB;       /* colorspace of input image */
	
	jpeg_set_defaults(&info->cInfo);
	jpeg_set_quality(&info->cInfo, quality, TRUE);
	
	jpeg_start_compress(&info->cInfo, TRUE);
	return info;
}
void JPEGWriteRow(void* pixels,void* v)
{
	unsigned char* _pixels = (unsigned char*) pixels;
	JPEGWriteInfo* info=(JPEGWriteInfo*)v;
	JSAMPROW row_pointer[1];
	row_pointer[0] = _pixels;
	(void) jpeg_write_scanlines(&info->cInfo, row_pointer, 1);
}
void JPEGFinalizeWrite(void* v)
{
	JPEGWriteInfo* info=(JPEGWriteInfo*)v;
	jpeg_finish_compress(&info->cInfo);
	jpeg_destroy_compress(&info->cInfo);
	fclose(info->fp);
	free(info);
}

void* JPEGInitRead(char* fileName,int& width,int& height)
{
	JPEGReadInfo *info=(JPEGReadInfo*)malloc(sizeof(JPEGReadInfo));
	info->fp=fopen(fileName,"rb");
	if(!info->fp)	fprintf(stderr,"Failed to open: %s\n",fileName)	,	exit(0);

	info->cInfo.err = jpeg_std_error(&info->jErr.pub);
	info->jErr.pub.error_exit = my_error_exit;
	if (setjmp(info->jErr.setjmp_buffer))
	{
		jpeg_destroy_decompress(&info->cInfo);
		fprintf(stderr,"JPEG error occured\n");
		return 0;
	}
	

	jpeg_create_decompress(&info->cInfo);
	jpeg_stdio_src(&info->cInfo, info->fp);
	
	(void) jpeg_read_header(&info->cInfo, TRUE);
	(void) jpeg_start_decompress(&info->cInfo);

	if(info->cInfo.output_components!=3)
	{
		fprintf(stderr,"Only 3 components per pixel supported: %d != 3\n",info->cInfo.output_components);
		return NULL;
	}
	width=info->cInfo.output_width;
	height=info->cInfo.output_height;

	return info;
}
void JPEGReadRow(void* pixels,void* v)
{
	unsigned char* _pixels = (unsigned char*) pixels;
	JPEGReadInfo* info=(JPEGReadInfo*)v;
	if(info->cInfo.output_scanline >= info->cInfo.output_height)
	{
		fprintf(stderr,"Trying to read beyond the end of the jpeg file: %d >= %d\n",info->cInfo.output_scanline,info->cInfo.output_height);
		exit(0);
	}
	JSAMPROW row_pointers[1];
	row_pointers[0]=_pixels;
	int r=jpeg_read_scanlines(&info->cInfo, row_pointers, 1);
}

void JPEGFinalizeRead(void* v)
{
	JPEGReadInfo* info=(JPEGReadInfo*)v;
	(void) jpeg_finish_decompress(&info->cInfo);
	jpeg_destroy_decompress(&info->cInfo);
	fclose(info->fp);
	free(info);
}