/*
Copyright (c) 2008, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

//////////////////////
// WriteImageStream //
//////////////////////
template<class PReal,class BReal>
WriteImageStream<PReal,BReal>::WriteImageStream(void* (*IW)(char*,int,int,int),void (*WR)(void*,void*),void (*FW)(void*),					
												char* fileName,int width,int height,int quality,bool logRGB,bool clamp) 
{
	_InitWrite		= IW;
	_WriteRow		= WR;
	_FinalizeWrite	= FW;
	current=0;
	_q=quality;
	_cw=_w=width;
	_ch=_h=height;
	_logRGB=logRGB;
	_clamp=clamp;
	info=NULL;
	pixels=NULL;
	buffer=NULL;
	Init(fileName);
}
template<class PReal,class BReal>
WriteImageStream<PReal,BReal>::WriteImageStream(void* (*IW)(char*,int,int,int),void (*WR)(void*,void*),void (*FW)(void*),
												char* fileName,int width,int height,int clipWidth,int clipHeight,int quality,bool logRGB,bool clamp)
{
	_InitWrite		= IW;
	_WriteRow		= WR;
	_FinalizeWrite	= FW;
	current=0;
	_q=quality;
	_w=width;
	_h=height;
	_cw=clipWidth;
	_ch=clipHeight;
	_logRGB=logRGB;
	_clamp=clamp;
	info=NULL;
	pixels=NULL;
	buffer=NULL;
	Init(fileName);
}
template<class PReal,class BReal>
void WriteImageStream<PReal,BReal>::Init(char* fileName)
{
	average[0]=average[1]=average[2]=0;

	if(pixels)	free(pixels);
	pixels=(PReal*)malloc(sizeof(PReal)*_cw*3);
	buffer=(BReal*)malloc(sizeof(BReal)*_w*3);
	if(!buffer)	fprintf(stderr,"Failed to allocate buffer\n")	,	exit(0);
	if(!pixels)	fprintf(stderr,"Failed to allocate pixels\n")	,	exit(0);
	info=_InitWrite(fileName,_cw,_ch,_q);
}

template<class PReal,class BReal>
WriteImageStream<PReal,BReal>::~WriteImageStream(void)
{
	if(buffer)	free(buffer);
	if(pixels)	free(pixels);
	buffer=NULL;
	pixels=NULL;
	_FinalizeWrite(info);
	average[0]/=_cw;
	average[1]/=_cw;
	average[2]/=_cw;
	average[0]/=_ch;
	average[1]/=_ch;
	average[2]/=_ch;
	printf("Average: %f %f %f\n",average[0],average[1],average[2]);
}
template<class PReal,class BReal>
int WriteImageStream<PReal,BReal>::rows(void) const {return _h;}
template<class PReal,class BReal>
int WriteImageStream<PReal,BReal>::rowSize(void) const {return _w*3*sizeof(BReal);}
template<class PReal,class BReal>
void* WriteImageStream<PReal,BReal>::operator[] (int idx)
{
#if ASSERT_MEMORY_ACCESS
	if(idx!=current)	fprintf(stderr,"Index out of bounds: %d != %d\n",idx,current), exit(0);
#endif // ASSERT_MEMORY_ACCESS
	return buffer;
}
template<class PReal,class BReal>
void WriteImageStream<PReal,BReal>::advance(void)
{
	if(current<_ch)
	{
		for(int x=0;x<_cw;x++)
		{
			double _r=buffer[x+0*_w];
			double _g=buffer[x+1*_w];
			double _b=buffer[x+2*_w];
			if(_logRGB)
			{
				_r=exp(_r);
				_g=exp(_g);
				_b=exp(_b);
			}
			if(_clamp)
			{
				if		(_r<0)		_r=0;
				else if	(_r>255)	_r=255;
				if		(_g<0)		_g=0;
				else if	(_g>255)	_g=255;
				if		(_b<0)		_b=0;
				else if	(_b>255)	_b=255;
			}
			pixels[x*3+0]=PReal(_r);
			pixels[x*3+1]=PReal(_g);
			pixels[x*3+2]=PReal(_b);
			average[0]+=_r;
			average[1]+=_g;
			average[2]+=_b;
		}
		_WriteRow(pixels,info);
	}
	current++;
}
/////////////////////
// ReadImageStream //
/////////////////////
template<class PReal,class BReal>
ReadImageStream<PReal,BReal>::ReadImageStream(void* (*IR)(char*,int&,int&),void (*RR)(void*,void*),void (*FR)(void*),					
											  char* fileName,int& width,int& height,bool alpha)
{
	_InitRead		= IR;
	_ReadRow			= RR;
	_FinalizeRead	= FR;
	_a=alpha;
	current=0;

	info=_InitRead(fileName,width,height);
	_w=width;
	_h=height;

	if(_a)
	{
		pixels=(PReal*)malloc(sizeof(PReal)*_w*4);
		buffer=(BReal*)malloc(sizeof(BReal)*_w*4);
	}
	else
	{
		pixels=(PReal*)malloc(sizeof(PReal)*_w*3);
		buffer=(BReal*)malloc(sizeof(BReal)*_w*3);
	}
	if(!buffer)	fprintf(stderr,"Failed to allocate buffer\n")	,	exit(0);
	if(!pixels)	fprintf(stderr,"Failed to allocate pixels\n")	,	exit(0);

	_ReadRow(pixels,info);
	if(_a)
		for(int x=0;x<_w;x++)
		{
			buffer[x+0*_w]=PReal(pixels[x*4+0]);
			buffer[x+1*_w]=PReal(pixels[x*4+1]);
			buffer[x+2*_w]=PReal(pixels[x*4+2]);
			buffer[x+3*_w]=PReal(pixels[x*4+3]);
		}
	else
		for(int x=0;x<_w;x++)
		{
			buffer[x+0*_w]=PReal(pixels[x*3+0]);
			buffer[x+1*_w]=PReal(pixels[x*3+1]);
			buffer[x+2*_w]=PReal(pixels[x*3+2]);
		}
}
template<class PReal,class BReal>
ReadImageStream<PReal,BReal>::~ReadImageStream(void)
{
	if(buffer)	free(buffer);
	if(pixels)	free(pixels);
	buffer=NULL;
	pixels=NULL;
	_FinalizeRead(info);
}
template<class PReal,class BReal>
bool ReadImageStream<PReal,BReal>::hasAlpha(void) const {return _a;}
template<class PReal,class BReal>
int ReadImageStream<PReal,BReal>::rows(void) const {return _h;}
template<class PReal,class BReal>
int ReadImageStream<PReal,BReal>::rowSize(void) const
{
	if(_a)	return _w*3*sizeof(BReal);
	else	return _w*4*sizeof(BReal);
}
template<class PReal,class BReal>
void* ReadImageStream<PReal,BReal>::operator[] (int idx)
{
#if ASSERT_MEMORY_ACCESS
	if(idx!=current)	fprintf(stderr,"Index out of bounds: %d != %d\n",idx,current), exit(0);
#endif // ASSERT_MEMORY_ACCESS
	return buffer;
}
template<class PReal,class BReal>
void ReadImageStream<PReal,BReal>::advance(void)
{
	current++;
	if(current<_h)
	{
		_ReadRow(pixels,info);
		if(_a)
			for(int x=0;x<_w;x++)
			{
				buffer[x+0*_w]=BReal(pixels[x*4+0]);
				buffer[x+1*_w]=BReal(pixels[x*4+1]);
				buffer[x+2*_w]=BReal(pixels[x*4+2]);
				buffer[x+3*_w]=BReal(pixels[x*4+3]);
			}
		else
			for(int x=0;x<_w;x++)
			{
				buffer[x+0*_w]=BReal(pixels[x*3+0]);
				buffer[x+1*_w]=BReal(pixels[x*3+1]);
				buffer[x+2*_w]=BReal(pixels[x*3+2]);
			}
	}
}

//////////////////
// WImageStream //
//////////////////
template<class Real,class OutType>
WImageStream<Real,OutType>::WImageStream(char* fileName,int width,int height,int quality,bool logRGB) :
WriteImageStream(InitWrite<OutType>,WriteRow<OutType>,FinalizeWrite<OutType>,fileName,width,height,quality,logRGB,false)
{
}
template<class Real,class OutType>
WImageStream<Real,OutType>::WImageStream(char* fileName,int width,int height,int clipWidth,int clipHeight,int quality,bool logRGB) :
WriteImageStream(InitWrite<OutType>,WriteRow<OutType>,FinalizeWrite<OutType>,fileName,width,height,clipWidth,clipHeight,quality,logRGB,false)
{
}
/////////////////////
// BMPWImageStream //
/////////////////////
template<class Real>
BMPWImageStream<Real>::BMPWImageStream(char* fileName,int width,int height,int quality,bool logRGB) :
WriteImageStream(BMPInitWrite,BMPWriteRow,BMPFinalizeWrite,fileName,width,height,quality,logRGB,true)
{
}
template<class Real>
BMPWImageStream<Real>::BMPWImageStream(char* fileName,int width,int height,int clipWidth,int clipHeight,int quality,bool logRGB) :
WriteImageStream(BMPInitWrite,BMPWriteRow,BMPFinalizeWrite,fileName,width,height,clipWidth,clipHeight,quality,logRGB,true)
{
}
/////////////////////
// JPEGWImageSteam //
/////////////////////
template<class Real>
JPEGWImageStream<Real>::JPEGWImageStream(char* fileName,int width,int height,int quality,bool logRGB) :
WriteImageStream(JPEGInitWrite,JPEGWriteRow,JPEGFinalizeWrite,fileName,width,height,quality,logRGB,true)
{
}
template<class Real>
JPEGWImageStream<Real>::JPEGWImageStream(char* fileName,int width,int height,int clipWidth,int clipHeight,int quality,bool logRGB) :
WriteImageStream(JPEGInitWrite,JPEGWriteRow,JPEGFinalizeWrite,fileName,width,height,clipWidth,clipHeight,quality,logRGB,true)
{
}
////////////////////
// PNGWImageSteam //
////////////////////
template<class Real>
PNGWImageStream<Real>::PNGWImageStream(char* fileName,int width,int height,int quality,bool logRGB) :
WriteImageStream(PNGInitWrite,PNGWriteRow,PNGFinalizeWrite,fileName,width,height,quality,logRGB,true)
{
}
template<class Real>
PNGWImageStream<Real>::PNGWImageStream(char* fileName,int width,int height,int clipWidth,int clipHeight,int quality,bool logRGB) :
WriteImageStream(PNGInitWrite,PNGWriteRow,PNGFinalizeWrite,fileName,width,height,clipWidth,clipHeight,quality,logRGB,true)
{
}
/////////////////////
// WDPWImageStream //
/////////////////////
template<class Real>
WDPWImageStream<Real>::WDPWImageStream(char* fileName,int width,int height,int quality,bool logRGB) :
WriteImageStream(WDPInitWrite,WDPWriteRow,WDPFinalizeWrite,fileName,width,height,quality,logRGB,false)
{
}
template<class Real>
WDPWImageStream<Real>::WDPWImageStream(char* fileName,int width,int height,int clipWidth,int clipHeight,int quality,bool logRGB) :
WriteImageStream(WDPInitWrite,WDPWriteRow,WDPFinalizeWrite,fileName,width,height,clipWidth,clipHeight,quality,logRGB,false)
{
}
//////////////////
// RImageStream //
//////////////////
template<class Real,class InType>
RImageStream<Real,InType>::RImageStream(char* fileName,int& width,int& height) :
ReadImageStream(InitRead<InType>,ReadRow<InType>,FinalizeRead<InType>,fileName,width,height,false)
{
}
//////////////////////
// JPEGRImageStream //
//////////////////////
template<class Real>
JPEGRImageStream<Real>::JPEGRImageStream(char* fileName,int& width,int& height) :
ReadImageStream(JPEGInitRead,JPEGReadRow,JPEGFinalizeRead,fileName,width,height,false)
{
}
/////////////////////
// PNGRImageStream //
/////////////////////
template<class Real>
PNGRImageStream<Real>::PNGRImageStream(char* fileName,int& width,int& height) :
ReadImageStream(PNGInitRead,PNGReadRow,PNGFinalizeRead,fileName,width,height,true)
{
}
/////////////////////
// WDPRImageStream //
/////////////////////
template<class Real>
WDPRImageStream<Real>::WDPRImageStream(char* fileName,int& width,int& height) :
ReadImageStream(WDPInitRead,WDPReadRow,WDPFinalizeRead,fileName,width,height,false)
{
}
////////////////////////
// StreamingInputTile //
////////////////////////
template<class Real>
StreamingInputTile<Real>::StreamingInputTile(void)
{
	rc=1;
	data=NULL;
	w=h=0;
	sX=sY=0;
	rows=NULL;
	alpha=false;
}
template<class Real>
StreamingInputTile<Real>::~StreamingInputTile(void)
{
	if(data)	delete data;
	if(rows)
	{
		for(int i=0;i<rc;i++)	if(rows[i])	delete[] rows[i];
		delete[] rows;
	}
	data=NULL;
	rows=NULL;
	rc=0;
};

template<class Real>
void StreamingInputTile<Real>::Init(const char* tName,int rowCount,int startX,int startY)
{
	strcpy(tileName,tName);
	rc=rowCount;
	sX=startX;
	sY=startY;
	if(rowCount<1)	exit(0);
	rows=new Real*[rc];
	data = GetReadStream<Real>(tileName,w,h,alpha);
	for(int i=-rc;i<0;i++)	init(i);
}

template<class Real>
void StreamingInputTile<Real>::init(int idx)
{
	if(idx+rc-1==sY)
	{
		if(!data)	exit(0);
		if(alpha)	for(int i=0;i<rc;i++)	rows[i]=new Real[w*4];
		else		for(int i=0;i<rc;i++)	rows[i]=new Real[w*3];
		for(int i=0;i<rc-1;i++)
		{
			if(alpha)	memcpy(rows[i],(*data)[i],w*sizeof(Real)*4);
			else		memcpy(rows[i],(*data)[i],w*sizeof(Real)*3);
			data->advance();
		}
	}
	else if(idx==sY+h)
	{
		delete data;
		for(int i=0;i<rc;i++)	delete[] rows[i];
		delete[] rows;
		data=NULL;
		rows=NULL;
	}
	if(idx>=sY && data)
	{
		if(alpha)	memcpy(rows[(idx+rc-1-sY)%rc],(*data)[idx+rc-1-sY],w*4*sizeof(Real));
		else		memcpy(rows[(idx+rc-1-sY)%rc],(*data)[idx+rc-1-sY],w*3*sizeof(Real));
		data->advance();
	}
}
template<class Real>
Color<Real> StreamingInputTile<Real>::operator() (int i,int j)
{
	Color<Real> clr;
	Real* data=&rows[(j-sY)%rc][i-sX];
	clr[0]=data[0*w];
	clr[1]=data[1*w];
	clr[2]=data[2*w];
	return clr;
}
template<class Real>
Color<Real> StreamingInputTile<Real>::operator() (int i,int j,Real& a)
{
	Color<Real> clr;
	Real* data=&rows[(j-sY)%rc][i-sX];
	clr[0]=data[0*w];
	clr[1]=data[1*w];
	clr[2]=data[2*w];
	if(alpha)	a=data[3*w];
	return clr;
}
template<class Real>
int StreamingInputTile<Real>::width(void)	const	{return w;}
template<class Real>
int StreamingInputTile<Real>::height(void)	const	{return h;}
template<class Real>
int StreamingInputTile<Real>::startX(void) const	{return sX;}
template<class Real>
int StreamingInputTile<Real>::startY(void) const	{return sY;}
template<class Real>
bool StreamingInputTile<Real>::hasAlpha(void) const	{return alpha;}


////////////////////////////////
#include <Util/cmdLineParser.h>
template<class Real>
StreamingGrid* GetReadStream(char* fileName,int& width,int& height)
{
	bool hasAlpha;
	return GetReadStream<Real>(fileName,width,height,hasAlpha);
}
template<class Real>
StreamingGrid* GetReadStream(char* fileName,int& width,int& height,bool& hasAlpha)
{
	StreamingGrid* data=NULL;
	char* ext=GetFileExtension(fileName);
	if(!strcasecmp(ext,"jpeg") || !strcasecmp(ext,"jpg") )	data = new JPEGRImageStream	<Real>			(fileName,width,height),	hasAlpha=false;
	else if(!strcasecmp(ext,"wdp") )						data = new WDPRImageStream	<Real>			(fileName,width,height),	hasAlpha=false;
//	else if(!strcasecmp(ext,"bmp") )						data = new BMPRImageStream	<Real>			(fileName,width,height),	hasAlpha=false;
	else if(!strcasecmp(ext,"png") )						data = new PNGRImageStream	<Real>			(fileName,width,height),	hasAlpha=true;
	else if(!strcasecmp(ext,"int") )						data = new RImageStream		<Real,int>		(fileName,width,height),	hasAlpha=false;
	else if(!strcasecmp(ext,"int16") )						data = new RImageStream		<Real,__int16>	(fileName,width,height),	hasAlpha=false;
	else if(!strcasecmp(ext,"half") )						data = new RImageStream		<Real,half>		(fileName,width,height),	hasAlpha=false;
	else if(!strcasecmp(ext,"float") )						data = new RImageStream		<Real,float>	(fileName,width,height),	hasAlpha=false;
	else if(!strcasecmp(ext,"double") )						data = new RImageStream		<Real,double>	(fileName,width,height),	hasAlpha=false;
	else	fprintf(stderr,"Unsupported input file extension: %s\n",ext);
	delete[] ext;
	return data;
}
template<class Real>
StreamingGrid* GetWriteStream(char* fileName,const int& width,const int& height,const int& quality)
{
	StreamingGrid* data=NULL;
	char* ext=GetFileExtension(fileName);

	if(!strcasecmp(ext,"jpg") || !strcasecmp(ext,"jpeg"))	data=new JPEGWImageStream	<Real>			(fileName,width,height,quality);
	else if(!strcasecmp(ext,"wdp"))							data=new WDPWImageStream	<Real>			(fileName,width,height,quality);
	else if(!strcasecmp(ext,"bmp"))							data=new BMPWImageStream	<Real>			(fileName,width,height,quality);
	else if(!strcasecmp(ext,"png"))							data=new PNGWImageStream	<Real>			(fileName,width,height,quality);
	else if(!strcasecmp(ext,"int"))							data=new WImageStream		<Real,int>		(fileName,width,height,quality);
	else if(!strcasecmp(ext,"int16"))						data=new WImageStream		<Real,__int16>	(fileName,width,height,quality);
	else if(!strcasecmp(ext,"half"))						data=new WImageStream		<Real,half>		(fileName,width,height,quality);
	else if(!strcasecmp(ext,"float"))						data=new WImageStream		<Real,float>	(fileName,width,height,quality);
	else if(!strcasecmp(ext,"double"))						data=new WImageStream		<Real,double>	(fileName,width,height,quality);
	else	fprintf(stderr,"Unsupported output file extension: %s\n",ext);
	delete[] ext;
	return data;
}
