/*
Copyright (c) 2008, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#ifndef STREAMING_SOLVER_INCLUDED
#define STREAMING_SOLVER_INCLUDED

#define STREAMING_GRID_BUFFER_MULTIPLIER 2

#define USE_SSE_CODE 1


#if USE_SSE_CODE
#include <emmintrin.h>
#endif // USE_SSE_CODE
#include "LinearAlgebra/SparseMatrix.h"
#include "LaplacianMatrix/LaplacianMatrix1D.h"
#include "LaplacianMatrix/LaplacianMatrix2D.h"
#include "Util/GridStream.h"
#include "Util/MultiStreamIO.h"

template<class Real>
class RealType
{
public:
#if USE_SSE_CODE
	typedef __m128 WordClass;
#define ALIGNMENT 16
#else // !USE_SSE_CODE
	typedef Real WordClass;
#define ALIGNMENT 1
#endif // USE_SSE_CODE
	static const int RealPerWord=sizeof(WordClass)/sizeof(Real);
};

template<class Real,int Type,int Degree,int Channels>
class StreamingSolver : public RealType<Real>
{
	WordClass *_localR;
	WordClass *_XStream,*_BStream;
protected:
	bool symmetric,clearX,clearB;
	int xSize,bSize,rSize,iters,index,_iters;
	WordClass *localXPtrs[2*Degree+1];
	WordClass *_localXAccum[Degree],*localXAccum[Degree];
	Real* localBPtr;
	Real* localRPtr;
	int _major;
public:
	Real *XStream,*BStream,*localR;
	Real* GetXRow(int row,int channel);
	Real* GetBRow(int row,int channel);
	int major,minor;
protected:


	// Note: For appropriate indexing, the laplacian stencil needs to be indexed (minor,major)
#if USE_SSE_CODE
	class LaplacianTemplateSSE
	{
	public:
		__declspec (align(ALIGNMENT)) WordClass matrixValues[4][5];
		float diagonal[4];
	};
	LaplacianTemplateSSE *lapTemplates2,*_lapTemplates2;
	LaplacianTemplateSSE *lapTemplates3,*_lapTemplates3;	// Like lapTemplates2 but with the center term zeroed out
#else // !USE_SSE_CODE
	typename FiniteElements2D<double,Type,Degree>::FullMatrixStencil laplacianStencil;
#endif // USE_SSE_CODE

#if USE_SSE_CODE
	// Strictly Degree=2
	void SetDotSum(const WordClass mValues[],const WordClass* xPtrs[],size_t j,WordClass& dotSum) const;
	float GetLaplacianValue0(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	float GetLaplacianValue1(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	float GetLaplacianValue2(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	float GetLaplacianValue3(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;

	float GaussSeidelUpdate0(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	float GaussSeidelUpdate1(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	float GaussSeidelUpdate2(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	float GaussSeidelUpdate3(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	void SetInteriorDotSum(const WordClass mValues[],const WordClass* xPtrs[],size_t j,WordClass& dotSum) const;

	float InteriorGaussSeidelUpdate0(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	float InteriorGaussSeidelUpdate1(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	float InteriorGaussSeidelUpdate2(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	float InteriorGaussSeidelUpdate3(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;

	float GetInteriorLaplacianValue0(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	float GetInteriorLaplacianValue1(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	float GetInteriorLaplacianValue2(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
	float GetInteriorLaplacianValue3(const WordClass mValues[],const WordClass* xPtrs[],float& previousDotSum,int j) const;
#else // !USE_SSE_CODE
	Real GetLaplacianValue(const double lapValues[][2*Degree+1],int j);
	Real GaussSeidelUpdate(const double lapValues[][2*Degree+1],int j,int iB);
	Real GetInteriorLaplacianValue(const double lapValues[][2*Degree+1],int j);
	Real InteriorGaussSeidelUpdate(const double lapValues[][2*Degree+1],int j,int iB);
#endif // USE_SSE_CODE
public:
	Real laplacianScale;
	bool setResidual,laplacianRescale;
	double bSquareNorm,rSquareNorm;
	StreamingSolver(void);
	~StreamingSolver(void);

	void Init(const typename FiniteElements2D<double,Type,Degree>::FullMatrixStencil& lStencil,int major,int minor,bool symmetric);
	void Init(int major,int minor,bool symmetric);
	void SetIterations(int iters,int rSize=1);
	void SetIterations(int start,int iters,int bStart,int xStart,int bEnd,int xEnd,int rSize=1);
	void UnSetIterations(void);

	bool Increment(void);
	template<class StorageType>		bool UpdateXInput	(StreamingGrid* X);
	template<class StorageType>		bool UpdateBInput	(StreamingGrid* B);
	template<class StorageType>		bool UpdateXOutput	(StreamingGrid* X);
	template<class StorageType>		bool UpdateBOutput	(StreamingGrid* B);

	void Solve(void);

	void Solve(int idx);
	void SetResidual(int idx);
	void SolveInterior(int idx);
	void SetInteriorResidual(int idx);
	static MultiStreamIOServer server;
};
template<class Real>
class MultiGridRestrictionNode
{
public:
	virtual void InitRestriction	(bool symmetric){;}							// Generic Initializer
	virtual void SetRestrictionIterations	(int iters){;}						// Sets up interleaved dependencies
	virtual void SetRestriction		(Real* lB,int idx,int major2,int minor2){;}	// Set parent's row
	virtual void SolveRestriction	(void){;}
	virtual bool IterateRestriction	(void){return false;}
};

template<class Real,int Type,int Degree,int Channels,class StorageType=Real>
class MultiGridStreamingSolver : public StreamingSolver<Real,Type,Degree,Channels>, public MultiGridRestrictionNode<Real>
{
	typename FiniteElements1D<double,Type,Degree>::FullProlongationStencil majorProlongationStencil;
	typename FiniteElements1D<double,Type,Degree>::FullProlongationStencil minorProlongationStencil;
#if USE_SSE_CODE
	class ProlongationStencilSSE
	{
	public:
		__declspec (align(ALIGNMENT)) WordClass matrixValues[2][4];
	};
	ProlongationStencilSSE *prolongationStencil2,*_prolongationStencil2;
	class ProlongationStencilSSE2
	{
	public:
		__declspec (align(ALIGNMENT)) WordClass matrixValues[2];
	};
	ProlongationStencilSSE2 *prolongationStencil3,*_prolongationStencil3;
#endif // USE_SSE_CODE
	int prolongationOffset;
	int startProlongation,startRestriction;

	int restrictionBit;
	StreamingGrid *B,*X;
public:
// There is some messiness in terms of how in/out are defined and used (e.g. do they increment or do they set?)
	StreamingGrid *inX,*outX,*inB,*outB;

	MultiGridRestrictionNode *rChild;
	MultiGridStreamingSolver *parent,*pChild;


	MultiGridStreamingSolver(void);
	~MultiGridStreamingSolver(void);
	void Initialize(
		typename FiniteElements1D<double,Type,Degree>::DotProduct<Type,Degree>::FullDotProductStencil& dotMajor,
		typename FiniteElements1D<double,Type,Degree>::DotProduct<Type,Degree>::FullDotProductStencil& d2DotMajor,
		typename FiniteElements1D<double,Type,Degree>::DotProduct<Type,Degree>::FullDotProductStencil& dotMinor,
		typename FiniteElements1D<double,Type,Degree>::DotProduct<Type,Degree>::FullDotProductStencil& d2DotMinor,
		int major,int minor,bool symmetric,bool memoryMappedFile=false);
	void Initialize(int major,int minor,bool symmetric,bool memoryMappedFile=false);
	void InitProlongation	(bool symmetric);
	void InitRestriction	(bool symmetric);

	void SetProlongationIterations	(int iters);
	void SetRestrictionIterations	(int iters);
	void UnSetProlongationIterations(void);
	void UnSetRestrictionIterations	(void);
	bool IterateProlongation(void);
	bool IterateRestriction	(void);
	void SolveProlongation	(void);
	void SolveRestriction	(void);
	WordClass *localRPtrs[Degree+2];
	void SetRestriction(Real* lB,int idx,int major2,int minor2);
	void SetRestrictionDotSum(const WordClass mValues[],const WordClass* xPtrs[],size_t j,WordClass& dotSum) const;
	float RestrictionUpdate0(const WordClass mValues[],const WordClass* rPtrs[],float& previousDotSum,int j);
	float RestrictionUpdate1(const WordClass mValues[],const WordClass* rPtrs[],float& previousDotSum,int j);
	WordClass *_localRAccum,*localRAccum;
	void SetInteriorRestriction(Real* lowB,int idx,int major2,int minor2);
	void SetInteriorRestrictionDotSum(const WordClass mValues[],const WordClass* xPtrs[],size_t j,WordClass& dotSum) const;
	float InteriorRestrictionUpdate0(const WordClass mValues[],const WordClass* rPtrs[],float& previousDotSum,int j);
	float InteriorRestrictionUpdate1(const WordClass mValues[],const WordClass* rPtrs[],float& previousDotSum,int j);
	void SetRestrictionDotSum(const WordClass& mValues,const WordClass* xPtrs,size_t j,WordClass& dotSum) const;
	float RestrictionUpdate0(const WordClass& mValues,const WordClass* rPtrs,float& previousDotSum,int j);
	float RestrictionUpdate1(const WordClass& mValues,const WordClass* rPtrs,float& previousDotSum,int j);
	void ProlongationUpdate(int idx);

	static bool IsDownSamplable(const int& hMajor,const int& hMinor,int& lMajor,int& lMinor);
};



template<class Real,int Type,int Degree,int Channels,class PartialType=Real>
class StreamingDivergence : public MultiGridRestrictionNode<Real>, RealType<Real>
{
	typename FiniteElements1D<Real,Type,Degree>::DotProduct<Type,Degree>::FullDotProductStencil dotMajorStencil,dotMinorStencil;
	typename FiniteElements1D<Real,DERIVATIVE(Type),Degree-1>::DotProduct<Type,Degree>::FullDotProductStencil dDotMajorStencil,dDotMinorStencil;
	typename FiniteElements2D<Real,Type,Degree>::FullDivergenceStencil divergenceStencil;
	int startRestriction;
	int major,minor,_major,index,sMajor,sMinor,dSize;
	WordClass *_localDMajor,*_localDMinor;
	Real *localDMajor,*localDMinor;
	WordClass *_localDMajorAccum,*localDMajorAccum;
	WordClass *_localDMinorAccum,*localDMinorAccum;
public:
	double dXSquareNorm,dYSquareNorm,outputSquareNorm;

	StreamingGrid *dMajor,*dMinor;
	MultiGridRestrictionNode<Real> *rParent,*rChild;

	StreamingDivergence(void);
	~StreamingDivergence(void);
	void InitRestriction(int major,int minor,bool symmetric);
	void SetRestrictionIterations(int iters);
	void SetRestriction(Real* lB,int idx,int major2,int minor2);
	void SetInteriorRestriction(Real* lB,int idx,int major2,int minor2);
	void SolveRestriction(void);
	bool IterateRestriction(void);
};
#if USE_SSE_CODE
#include "StreamingSolver128.inl"
#endif // USE_SSE_CODE
#include "StreamingSolver.inl"

#endif // STREAMING_SOLVER_INCLUDED
