/*
Copyright (c) 2008, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "Util/CmdLineParser.h"
#include "Util/MultiStreamIO.h"
#include "Util/ImageStream.h"

#define NO_STREAM 0


typedef float Real;

cmdLineString In("in"),Labels("labels"),Positions("positions"),OutX("outX"),OutY("outY");
cmdLineReadable NoPad("noPad"),HarmonicFill("hFill"),Progress("progress");
cmdLineInt LScale("lScale",1);
cmdLineReadable* params[]=
{
	&In,&OutX,&OutY,&Labels,&Positions,&LScale,&NoPad,&HarmonicFill,&Progress
};

void ShowUsage(char* ex)
{
	printf("Usage %s:\n",ex);
	printf("\t--%s <input image list>\n",In.name);
	printf("\t--%s <output X gradient>\n",OutX.name);
	printf("\t--%s <output Y gradient>\n",OutY.name);
	printf("\t--%s <label image>\n",Labels.name);
	printf("\t[--%s <position list>]\n",Positions.name);
	printf("\t[--%s <label index scale>=%d]\n",LScale.name,LScale.value);
	printf("\t[--%s]\n",HarmonicFill.name);
	printf("\t[--%s]\n",NoPad.name);
	printf("\t[--%s]\n",Progress.name);
}

template<class Data>
Data Color2Data(const Color<Real>& c);

template<>
half			Color2Data<half>			(const Color<Real>& c)	{	return c.luminance();	}
template<>
float			Color2Data<float>			(const Color<Real>& c)	{	return c.luminance();	}
template<>
double			Color2Data<double>			(const Color<Real>& c)	{	return c.luminance();	}
template<>
__int16			Color2Data<__int16>			(const Color<Real>& c)	{	return c.luminance();	}
template<>
int				Color2Data<int>				(const Color<Real>& c)	{	return c.luminance();	}
template<>
Color<half>		Color2Data<Color<half> >	(const Color<Real>& c)	{	return c;	}
template<>
Color<float>	Color2Data<Color<float> >	(const Color<Real>& c)	{	return c;	}
template<>
Color<double>	Color2Data<Color<double> >	(const Color<Real>& c)	{	return c;	}
template<>
Color<__int16>	Color2Data<Color<__int16> >	(const Color<Real>& c)	{	return c;	}
template<>
Color<int>		Color2Data<Color<int> >		(const Color<Real>& c)	{	return c;	}

void PrintColor(const Color<Real>& c)	{	printf("%f %f %f",float(c[0]),float(c[1]),float(c[2]));	}
void PrintColor(const Real& c)			{	printf("%f",float(c));	}

template<class Real>
class MyStreamingInputTile : public StreamingInputTile<Real>
{
public:
	Color<Real> dflt;

	MyStreamingInputTile(void) : StreamingInputTile<Real>()	{dflt=0;}

	Color<Real> operator() (int i,int j)
	{
		if(i<startX() || i>=startX()+width() || j<startY() || j>=startY()+height())	return dflt;
		else																		return StreamingInputTile<Real>::operator ()(i,j);
	}
	
};

int CombinedIndex(int idx1,int idx2,int tCount)
{
	if(idx1>=0 && idx1<tCount)
		if(idx2>=0 && idx2<tCount)
			if(idx1==idx2)	return idx1;
			else			return -1;
		else				return idx1;
	else
		if(idx2>=0 && idx2<tCount)	return idx2;
		else						return -1;
}

int LabelIndex(const Color<int>& lColor)
{
	return lColor[0]/LScale.value;
}
template <class PixelData>
int Execute2(Color<double>& average,int& ww,int & hh)
{
	FILE* fp;
	int w,h;
	int tCount=0;
	char line[512];
	int *offX,*offY;
	long long pixelCount=0;
	MyStreamingInputTile<int> labels;
	MyStreamingInputTile<Real>* tiles;
#if NO_STREAM
	FILE *dX,*dY;
#else // !NO_STREAM
	MultiStreamIOClient *dX,*dY;
#endif // NO_STREAM

	fp=fopen(In.value,"r");
	if(!fp)
	{
		fprintf(stderr,"Failed to read: %s\n",In.value);
		return EXIT_FAILURE;
	}
	while(fscanf(fp," %s ",line)==1)	tCount++;
	fclose(fp);

	tiles = new MyStreamingInputTile<Real>[tCount];
	offX=new int[tCount];
	offY=new int[tCount];
	if(Positions.set)
	{
		fp=fopen(Positions.value,"r");
		if(!fp)
		{
			fprintf(stderr,"Failed to read: %s\n",Positions.value);
			return EXIT_FAILURE;
		}
		int temp=0;
		while(fscanf(fp," %d %d ",&offX[temp],&offY[temp])==2)	temp++;
		fclose(fp);
	}
	else	for(int i=0;i<tCount;i++) offX[i]=offY[i]=0;

	labels.Init(Labels.value,2);
	labels.dflt[0]=labels.dflt[1]=labels.dflt[2]=tCount*LScale.value;
	
	fp=fopen(In.value,"r");
	if(!fp)
	{
		fprintf(stderr,"Failed to read: %s\n",In.value);
		return EXIT_FAILURE;
	}
	tCount=0;
	while(fscanf(fp," %s ",line)==1)
	{
		tiles[tCount].Init(line,2,offX[tCount],offY[tCount]);
		tCount++;
	}
	fclose(fp);
	w=labels.width();
	h=labels.height();
	if(!NoPad.set)	ww=w+1,	hh=h+1;
	else			ww=w,	hh=h;
	printf("Input Size: %d x %d\n",w,h);

#if NO_STREAM
	dX=fopen(OutX.value,"wb");
	dY=fopen(OutY.value,"wb");
	if(!dX || !dY)	exit(0);
#else // !NO_STREAM
	dX = new MultiStreamIOClient(OutX.value,sizeof(PixelData)*(ww-1),hh,2,true);
	dY = new MultiStreamIOClient(OutY.value,sizeof(PixelData)*ww,(hh-1),2,true);
	dX->reset(false,1);
	dY->reset(false,1);
#endif // NO_STREAM

	int idx;
	Color<Real> clr1,clr2;
	int tIdx,idxX,idxY;
	PixelData *dxRow,*dyRow;
#if NO_STREAM
	dxRow=new PixelData[ww-1];
	dyRow=new PixelData[ww  ];
#endif // NO_STREAM
	for(int j=0;j<hh;j++)
	{
		if(Progress.set)	printf("[%.1f%%]\r",float(j)/float(hh-1)*100);
#if !NO_STREAM
		dxRow = (PixelData*)(*dX)[j];
#endif // !NO_STREAM
		memset(dxRow,0,sizeof(PixelData)*(ww-1));
		if(j<hh-1)
		{
#if !NO_STREAM
			dyRow = (PixelData*)(*dY)[j];
#endif // !NO_STREAM
			memset(dyRow,0,sizeof(PixelData)*ww);
		}
		labels.init(j);
		for(int t=0;t<tCount;t++)	tiles[t].init(j);

		for(int i=0;i<ww;i++)
		{
			idx=LabelIndex(labels(i,j));
			if(idx>=0 && idx<tCount)
			{
				average+=tiles[idx](i,j);
				pixelCount++;
			}
			else if (!HarmonicFill.set)	pixelCount++;

			if(!HarmonicFill.set)
			{
				// Partial X
				idxX=LabelIndex(labels(i+1,j));
				if(i<ww-1)
				{
					tIdx=CombinedIndex(idx,idxX,tCount);
					if(tIdx!=-1)
					{
						if(idx==tIdx)	clr1=tiles[tIdx](i,j);
						else			clr1=tiles[tIdx].dflt;
						if(idxX==tIdx)	clr2=tiles[tIdx](i+1,j);
						else			clr2=tiles[tIdx].dflt;
						dxRow[i]=Color2Data<PixelData>(clr2-clr1);
					}
				}
				// Partial Y
				idxY=LabelIndex(labels(i,j+1));
				if(j<hh-1)
				{
					tIdx=CombinedIndex(idx,idxY,tCount);
					if(tIdx!=-1)
					{
						if(idx==tIdx)	clr1=tiles[tIdx](i,j);
						else			clr1=tiles[tIdx].dflt;
						if(idxY==tIdx)	clr2=tiles[tIdx](i,j+1);
						else			clr2=tiles[tIdx].dflt;
						dyRow[i]=Color2Data<PixelData>(clr2-clr1);
					}
				}
			}
			else
			{
				if(idx<0 || idx>=tCount)	continue;
				// Partial X
				if(i<ww-1)
				{
#if 1
					if( LabelIndex(labels(i+1,j))==idx )	dxRow[i]=Color2Data<PixelData>(tiles[idx](i+1,j)-tiles[idx](i,j));
#else
					if(i==w-1)									dxRow[i]=Color2Data<PixelData>(tiles[idx].dflt-tiles[idx](i,j));
					else if( LabelIndex(labels(i+1,j))==idx )	dxRow[i]=Color2Data<PixelData>(tiles[idx](i+1,j)-tiles[idx](i,j));
#endif
				}
				// Partial Y
				if(j<hh-1)
				{
#if 1
					if( LabelIndex(labels(i,j+1))==idx )	dyRow[i]=Color2Data<PixelData>(tiles[idx](i,j+1)-tiles[idx](i,j));
#else
					if(j==h-1)									dyRow[i]=Color2Data<PixelData>(tiles[idx].dflt-tiles[idx](i,j));
					else if( LabelIndex(labels(i,j+1))==idx )	dyRow[i]=Color2Data<PixelData>(tiles[idx](i,j+1)-tiles[idx](i,j));
#endif
				}
			}
		}
#if NO_STREAM
		fwrite(dxRow,sizeof(PixelData),ww-1,dX);
		if(j<hh-1)	fwrite(dyRow,sizeof(PixelData),ww,dY);
#else // !NO_STREAM
		dX->advance();
		if(j<hh-1)	dY->advance();
#endif // NO_STREAM
	}
	if(Progress.set)	printf("\n");
	delete[] offX;
	delete[] offY;
	delete[] tiles;
#if NO_STREAM
	delete[] dxRow;
	delete[] dyRow;
	fclose(dX);
	fclose(dY);
#else // !NO_STREAM
	delete dX;
	delete dY;
#endif // NO_STREAM
	average/=pixelCount;
	return EXIT_SUCCESS;
}


template <class ChannelData>
int Execute1(int& w,int & h)
{
	int ret;
	Color<double> average=0;
	ret=Execute2<Color<ChannelData> >(average,w,h);
	printf("Average: %f %f %f\n",average[0],average[1],average[2]);
	return ret;
}
int main(int argc,char* argv[])
{
	int paramNum=sizeof(params)/sizeof(cmdLineReadable*);
	cmdLineParse(argc-1,&argv[1],paramNum,params,0);
	if(!In.set || !OutX.set || !OutY.set || !Labels.set)	
	{
		ShowUsage(argv[0]);
		return EXIT_FAILURE;
	}
	int w,h;

	int ret;
	char* extX=GetFileExtension(OutX.value);
	char* extY=GetFileExtension(OutY.value);
	if(strcasecmp(extX,extY))
	{
		fprintf(stderr,"Extensions on partials must be the same\n");
		delete[] extX;
		delete[] extY;
		return EXIT_FAILURE;
	}
	if(!strcasecmp(extX,"int16"))		ret=Execute1<__int16>	(w,h);
	else if(!strcasecmp(extX,"int"))	ret=Execute1<int>		(w,h);
	else if(!strcasecmp(extX,"half"))	ret=Execute1<half>		(w,h);
	else if(!strcasecmp(extX,"float"))	ret=Execute1<float>		(w,h);
	else if(!strcasecmp(extX,"double"))	ret=Execute1<double>	(w,h);
	else
	{
		fprintf(stderr,"Unrecognized extension: %s\n",extX);
		delete[] extX;
		delete[] extY;
		return EXIT_FAILURE;
	}

	delete[] extX;
	delete[] extY;
}