/*
Copyright (c) 2006, Michael Kazhdan and Matthew Bolitho
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

#include <float.h>


///////////////////
//  SparseMatrix //
///////////////////
///////////////////////////////////////
// SparseMatrix Methods and Memebers //
///////////////////////////////////////

template<class T>
SparseMatrix<T>::SparseMatrix()
{
	rowMajor=true;
	groups=0;
	groupSizes=NULL;
	m_ppElements=NULL;
}
template<class T>
bool SparseMatrix<T>::copyColumnMajor(SparseMatrix<T>& out,int inDim,int outDim)
{
	if(rowMajor && outDim!=groups)	return false;
	if(!rowMajor && inDim!=groups)	return false;
	out.Resize(inDim,false);
	if(rowMajor)
	{
		for(int i=0;i<groups;i++)
			for(int j=0;j<groupSizes[i];j++)
				if(0<=m_ppElements[i][j].N && m_ppElements[i][j].N<inDim)
					out.groupSizes[m_ppElements[i][j].N]++;

		for(int i=0;i<out.groups;i++)
		{
			int sz=out.groupSizes[i];
			out.groupSizes[i]=0;
			out.SetGroupSize(i,sz);
			out.groupSizes[i]=0;
		}
		for(int i=0;i<groups;i++)
			for(int j=0;j<groupSizes[i];j++)
			{
				int idx=m_ppElements[i][j].N;
				if(0<=idx && idx<inDim)
				{
					out.m_ppElements[idx][out.groupSizes[idx]].N=i;
					out.m_ppElements[idx][out.groupSizes[idx]].Value=m_ppElements[i][j].Value;
					out.groupSizes[idx]++;
				}
			}
	}
	else
		for(int i=0;i<groups;i++)
		{
			out.SetGroupSize(i,groupSizes[i]);
			memcpy(out.m_ppElements[i],m_ppElements[i],groupSizes[i]*sizeof(MatrixEntry<T>));
		}
	return true;
}
template<class T>
bool SparseMatrix<T>::copyRowMajor(SparseMatrix<T>& out,int inDim,int outDim)
{
	if(rowMajor && outDim!=groups)	return false;
	if(!rowMajor && inDim!=groups)	return false;

	out.Resize(outDim,true);
	if(rowMajor)
		for(int i=0;i<groups;i++)
		{
			out.SetGroupSize(i,groupSizes[i]);
			memcpy(out.m_ppElements[i],m_ppElements[i],groupSizes[i]*sizeof(MatrixEntry<T>));
		}
	else
	{
		for(int i=0;i<groups;i++)
			for(int j=0;j<groupSizes[i];j++)
				if(0<=m_ppElements[i][j].N && m_ppElements[i][j].N<outDim)
					out.groupSizes[m_ppElements[i][j].N]++;
		for(int i=0;i<out.groups;i++)
		{
			int sz=out.groupSizes[i];
			out.groupSizes[i]=0;
			out.SetGroupSize(i,sz);
			out.groupSizes[i]=0;
		}
		for(int i=0;i<groups;i++)
			for(int j=0;j<groupSizes[i];j++)
			{
				int idx=m_ppElements[i][j].N;
				if(0<=idx && idx<outDim)
				{
					out.m_ppElements[idx][out.groupSizes[idx]].N=i;
					out.m_ppElements[idx][out.groupSizes[idx]].Value=m_ppElements[i][j].Value;
					out.groupSizes[idx]++;
				}
			}
	}
	return true;
}

template<class T>
SparseMatrix<T>::SparseMatrix( int groups , bool rowMajor )
{
	rowMajor=true;
	groups=0;
	groupSizes=NULL;
	m_ppElements=NULL;
	Resize(groups,rowMajor);
}

template<class T>
SparseMatrix<T>::SparseMatrix( const SparseMatrix& M )
{
	rowMajor=true;
	groups=0;
	groupSizes=NULL;
	m_ppElements=NULL;
	Resize(M.groups,M.rowMajor);
	for (int i=0; i<groups; i++)
	{
		SetGroupSize(i,M.groupSizes[i]);
		for(int j=0;j<groupSizes[i];j++){m_ppElements[i][j]=M.m_ppElements[i][j];}
	}
}

template<class T>
double SparseMatrix<T>::SquareNorm(void)
{
	double l2=0;
	for(int i=0;i<groups;i++)
		for(int j=0;j<groupSizes[i];j++)
			l2+=m_ppElements[i][j].Value*m_ppElements[i][j].Value;
	return l2;
}
template<class T>
double SparseMatrix<T>::ASymmetricSquareNorm(void)
{
	double l2=0;
	for(int i=0;i<groups;i++)
		for(int j=0;j<groupSizes[i];j++)
		{
			double t1=0,t2=0;
			int N=m_ppElements[i][j].N;
			if(N==i)	continue;
			t1=m_ppElements[i][j].Value;
			for(int k=0;k<groupSizes[N];k++)
				if(m_ppElements[N][k].N==i)
				{
					t2=m_ppElements[N][k].Value;
					break;
				}
			l2+=(t1-t2)*(t1-t2);
		}
	return l2;
}

template<class T>
int SparseMatrix<T>::Entries(void) const {
	int e=0;
	for(int i=0;i<groups;i++){e+=int(groupSizes[i]);}
	return e;
}
template<class T>
SparseMatrix<T>& SparseMatrix<T>::operator = (const SparseMatrix<T>& M)
{
	Resize(M.groups,M.rowMajor);
	for (int i=0; i<groups; i++){
		SetGroupSize(i,M.groupSizes[i]);
		for(int j=0;j<groupSizes[i];j++){m_ppElements[i][j]=M.m_ppElements[i][j];}
	}
	return *this;
}

template<class T>
void SparseMatrix<T>::Transpose(void){rowMajor=!rowMajor;}

template<class T>
SparseMatrix<T>::~SparseMatrix()
{
	Resize(0,true);
}

template<class T>
void SparseMatrix<T>::Resize( int g, bool rm )
{
	rowMajor=rm;
	if(groups>0)
	{
		for(int i=0;i<groups;i++)
			if(groupSizes[i])
				free(m_ppElements[i]);
		free(m_ppElements);
		free(groupSizes);
		groupSizes=NULL;
		m_ppElements=NULL;
	}
	groups=g;
	if(g)
	{
		groupSizes=(int*)malloc(sizeof(int)*g);
		memset(groupSizes,0,sizeof(int)*g);
		m_ppElements=(MatrixEntry<T>**)malloc(sizeof(MatrixEntry<T>*)*g);
	}
}

template<class T>
void SparseMatrix<T>::SetGroupSize(int group,int count){
	if(group>=0 && group<groups)
	{
		if(groupSizes[group]) free(m_ppElements[group]);
		if(count>0)
		{
			m_ppElements[group]=(MatrixEntry<T>*)malloc(sizeof(MatrixEntry<T>)*count);
			memset( m_ppElements[group] , 0 , sizeof(MatrixEntry<T>)*count );
		}
		groupSizes[group]=count;
	}
}
template<class T>
template<class T2>
void SparseMatrix<T>::Multiply	( const Vector<T2>& In,Vector<T2>& Out) const
{
	if(rowMajor)
		for (int i=0; i<groups; i++)
		{
			T2 temp=T2();
			MatrixEntry<T>* elements = m_ppElements[i];
			for(int j=0;j<groupSizes[i];j++) temp+=In.m_pV[elements[j].N] * elements[j].Value;
			Out.m_pV[i]=temp;
		}
	else
		for (int i=0; i<groups; i++)
		{
			MatrixEntry<T>* elements = m_ppElements[i];
			for(int j=0;j<groupSizes[i];j++) Out.m_pV[elements[j].N]+=In.m_pV[i] * elements[j].Value;
		}
}


template<class T>
template<class T2>
void SparseMatrix<T>::MultiplyTranspose( const Vector<T2>& In , Vector<T2>& Out , bool checkIndex ) const
{
	if(!rowMajor)
		for (int i=0; i<groups; i++)
		{
			T2 temp=T2();
			MatrixEntry<T>* elements = m_ppElements[i];
			if(checkIndex)
			{
				for(int j=0;j<groupSizes[i];j++) if(elements[j].N>=0) temp+=In.m_pV[elements[j].N] * elements[j].Value;
			}
			else for(int j=0;j<groupSizes[i];j++) temp+=In.m_pV[elements[j].N] * elements[j].Value;
			Out.m_pV[i]+=temp;
		}
	else
		for (int i=0; i<groups; i++)
		{
			MatrixEntry<T>* elements = m_ppElements[i];
			if(checkIndex) 
			{
				for(int j=0;j<groupSizes[i];j++) if(elements[j].N>=0) Out.m_pV[elements[j].N]+=In.m_pV[i] * elements[j].Value;
			}
			else for(int j=0;j<groupSizes[i];j++) Out.m_pV[elements[j].N]+=In.m_pV[i] * elements[j].Value;
		}
}
template<class T>
template<class T2>
bool SparseMatrix<T>::Multiply ( const SparseMatrix<T>& M, const Vector<MatrixEntry<T> >& D,const Vector<T2>& In,Vector<T2>& Out,int startRow,int stopRow)
{
	if(!M.rowMajor)	return false;
	for (int i=startRow; i<stopRow; i++)
	{
		T2 temp=T2();
		for(int j=0;j<M.groupSizes[i];j++)	temp+=In.m_pV[M.m_ppElements[i][j].N] * M.m_ppElements[i][j].Value;
		Out.m_pV[i-startRow]=temp+In.m_pV[D[i].N]*D[i].Value;
	}
	return true;
}
template<class T>
template<class T2>
bool SparseMatrix<T>::Multiply ( const SparseMatrix<T>& M, const Vector<MatrixEntry2<T> >& D,const Vector<T2>& In,Vector<T2>& Out,int startRow,int stopRow)
{
	if(!M.rowMajor)	return false;
	for (int i=startRow; i<stopRow; i++)
	{
		T2 temp=T2();
		for(int j=0;j<M.groupSizes[i];j++)
			temp+=In.m_pV[M.m_ppElements[i][j].N] * M.m_ppElements[i][j].Value;
		Out.m_pV[i-startRow]=temp+In.m_pV[D[i].inN]*D[i].value;
	}
	return true;
}

template<class T>
template<class T2>
Vector<T2> SparseMatrix<T>::operator * (const Vector<T2>& V) const
{
	if(!rowMajor)
	{
		fprintf(stderr,"Unknown output matrix size for column-major matrix\n");
		exit(0);
	}
	Vector<T2> R( groups );
	for (int i=0; i<groups; i++)
	{
		T2 temp=T2();
		for(int ii=0;ii<groupSizes[i];ii++)	
			if( 0<=m_ppElements[i][ii].N && m_ppElements[i][ii].N<V.Dimensions())
				temp+=V.m_pV[m_ppElements[i][ii].N] * m_ppElements[i][ii].Value;
		R(i)=temp;
	}
	return R;
}
template<class T>
template<class T2>
int SparseMatrix<T>::SolveJacobi(const SparseMatrix<T>& M,const Vector<MatrixEntry<T> >& diagonal, const Vector<T2>& b,int iters,Vector<T2>& solution,bool reset)
{
	Vector<T2> Md;

	if(reset)
	{
		solution.Resize(b.Dimensions());
		solution.SetZero();
	}

	Md.Resize(M.groups);
	for(int i=0;i<iters;i++)
	{
		M.Multiply(solution,Md);
		for(int j=0;j<int(diagonal.Dimensions()); j++)
		{
			int idx = diagonal[j].N;
			solution[idx] = (b[idx]-Md[idx]) / diagonal[j].Value;
		}
	}
	return (int) iters;
}

template<class T>
template<class T2>
int SparseMatrix<T>::SolveGaussSeidel(const SparseMatrix<T>& M,const Vector<MatrixEntry<T> >& diagonal, const Vector<T2>& b,size_t iters,Vector<T2>& solution,double sorExtrapolation,bool zeroDiagonal)
{
	return SolveGaussSeidel(M,diagonal,b,iters,solution,0,int(diagonal.Dimensions()),sorExtrapolation,zeroDiagonal);
}
template<class T>
template<class T2>
int SparseMatrix<T>::SolveGaussSeidel( const SparseMatrix<T>& M , const Vector<MatrixEntry<T> >& diagonal , const Vector<T2>& b , size_t iters , Vector<T2>& solution ,
									   size_t start , size_t stop , double sorExtrapolation , bool zeroDiagonal , bool forward)
{
	if(!M.rowMajor)	return 0;
	if( sorExtrapolation!=1.0 )
		for(int i=0;i<iters;i++)
			if(forward)
				for( size_t j=start ; j<stop ; j++ )
				{
					T2 temp;
					memset(&temp,0,sizeof(T2));
					int idx = int(diagonal[j].N);
					T dValue = diagonal[j].Value;
					if(idx<0) continue;
					MatrixEntry<T>* el = M.m_ppElements[idx];
					MatrixEntry<T>* end = el+M.groupSizes[idx];
					for(  ; el!=end ; el++)	temp += solution[el->N]*el->Value;
					if(zeroDiagonal) solution[idx]  = solution[idx]*T(1.0-sorExtrapolation) + T2(b[idx]-temp) / T(dValue * sorExtrapolation);
					else			 solution[idx] += T2(b[idx]-temp) / T(dValue * sorExtrapolation);
				}
			else
				for( size_t j=stop-1 ; j>=start ; j-- )
				{
					T2 temp;
					memset(&temp,0,sizeof(T2));
					int idx = int(diagonal[j].N);
					T dValue = diagonal[j].Value;
					if(idx<0) continue;
					MatrixEntry<T>* el = M.m_ppElements[idx];
					MatrixEntry<T>* end = el+M.groupSizes[idx];
					for(  ; el!=end ; el++)	temp += solution[el->N]*el->Value;
					if(zeroDiagonal) solution[idx]  = solution[idx]*T(1.0-sorExtrapolation) + T2(b[idx]-temp) / T(dValue * sorExtrapolation);
					else			 solution[idx] += T2(b[idx]-temp) / T(dValue * sorExtrapolation);
				}
	else
		for(int i=0;i<iters;i++)
			if(forward)
				for( size_t j=start ; j<stop ; j++ )
				{
					T2 temp;
					memset(&temp,0,sizeof(T2));
					int idx = int(diagonal[j].N);
					T dValue = diagonal[j].Value;
					if(idx<0) continue;
					const MatrixEntry<T>* el = M.m_ppElements[idx];
					const MatrixEntry<T>* end = el+M.groupSizes[idx];
					for(  ; el!=end ; el++)	temp += solution[el->N]*el->Value;
					if(zeroDiagonal) solution[idx]  = T2(b[idx]-temp) / T(dValue);
					else			 solution[idx] += T2(b[idx]-temp) / T(dValue);
				}
			else
				for( size_t j=stop-1 ; j>=start ; j-- )
				{
					T2 temp;
					memset(&temp,0,sizeof(T2));
					int idx = int(diagonal[j].N);
					T dValue = diagonal[j].Value;
					if(idx<0) continue;
					MatrixEntry<T>* el = M.m_ppElements[idx];
					MatrixEntry<T>* end = el+M.groupSizes[idx];
					for(  ; el!=end ; el++)	temp += solution[el->N]*el->Value;
					if(zeroDiagonal) solution[idx]  = T2(b[idx]-temp) / T(dValue);
					else			 solution[idx] += T2(b[idx]-temp) / T(dValue);
				}
	return (int) iters;
}
template<class T>
template<class T2>
int SparseMatrix<T>::SolveJacobi( const SparseMatrix<T>& M , const Vector<MatrixEntry<T> >& diagonal , const Vector<T2>& b , int iters , Vector<T2>& solution ,
								 int start , int stop , double sorExtrapolation , bool zeroDiagonal )
{
	Vector<T2> temp;
	if(!M.rowMajor)	return 0;
	for(int i=0;i<iters;i++)
	{
		temp = M*solution;
		for( int j=start ; j<stop ; j++ )
		{
			int idx = int(diagonal[j].N);
			T dValue = diagonal[j].Value;
			if(idx<0) continue;
			if(!zeroDiagonal)	temp[j] -= solution[idx]*dValue;
			solution[idx] = solution[idx]*(1.0-sorExtrapolation) + (b[idx]-temp[j]) / dValue * sorExtrapolation;
		}
	}
	return (int) iters;
}
template<class T>
template<class T2>
int SparseMatrix<T>::SolveGaussSeidel(const SparseMatrix<T>& M,const Vector<MatrixEntry<T> >& diagonal, const Vector<T2>& b,size_t iters,Vector<T2>& solution,
									  size_t start,size_t stop,std::vector<double>& sorWeights,bool zeroDiagonal)
{
	if(!M.rowMajor)	return 0;
	for(int i=0;i<iters;i++)
		for( size_t j=start ; j<stop ; j++ )
		{
			T2 temp;
			memset(&temp,0,sizeof(T2));
			int idx = int(diagonal[j].N);
			T dValue = diagonal[j].Value;
			if(idx<0) continue;
			MatrixEntry<T>* el = M.m_ppElements[idx];
			MatrixEntry<T>* end = el+M.groupSizes[idx];
			for(  ; el!=end ; el++)	temp += solution[el->N]*el->Value;
			if(zeroDiagonal) solution[idx]  = solution[idx]*(1.0-sorWeights[idx]) + (b[idx]-temp) / dValue * sorWeights[idx];
			else			 solution[idx] += (b[idx]-temp) / dValue * sorWeights[idx];
		}
}

template<class T>
template<class T2>
int SparseMatrix<T>::SolveSuperGaussSeidel(const SparseMatrix<T>& M, const Vector<T2>& B,int iters,Vector<T2>& X)
{
	// Assuming that the matrix is symmetric!!!

	Vector<T2> R;

	for(int i=0;i<iters;i++)
	{
		if( !(i%10) ) R = M*X - B;
		for(int j=0 ; j<M.groups ; j++)
		{
			double squareColumnNorm = 0;
			T2 alpha , dot;
			memset(&dot,0,sizeof(T2));
			for(int k=0;k<M.groupSizes[j];k++)
			{
				squareColumnNorm += M.m_ppElements[j][k].Value*M.m_ppElements[j][k].Value;
				dot -= R[M.m_ppElements[j][k].N]*M.m_ppElements[j][k].Value;
			}
			if(squareColumnNorm <1e-8)	continue;
			alpha = dot/squareColumnNorm;
			X[j] += alpha;

			for(int k=0;k<M.groupSizes[j];k++) R[M.m_ppElements[j][k].N] += alpha * M.m_ppElements[j][k].Value;
		}
	}
	return (int) iters;
}


template<class T>
template<class T2>
int SparseMatrix<T>::SolveGaussSeidel2(const SparseMatrix<T>& M,const Vector<MatrixEntry<T> >& diagonal, const Vector<T2>& b,size_t iters,Vector<T2>& solution,bool reverseIndex)
{
	return SolveGaussSeidel2(M,diagonal,b,iters,solution,0,diagonal.Dimensions(),reverseIndex);
}
template<class T>
template<class T2>
int SparseMatrix<T>::SolveGaussSeidel2(const SparseMatrix<T>& M,const Vector<MatrixEntry<T> >& diagonal, const Vector<T2>& b,size_t iters,Vector<T2>& solution,size_t start,size_t stop,bool reverseIndex)
{
	int count=0;
	if(!M.rowMajor)	return 0;
	if(start>=stop)	return 0;

	for(int i=0;i<iters;i++)
		if(reverseIndex)
			for(size_t j=stop-1; j>=start ; j--)
			{
				T2 temp=0;
				int idx = diagonal[j].N;
				if(idx<0)	continue;
				for(int k=0;k<M.groupSizes[idx];k++)	temp+=solution[M.m_ppElements[idx][k].N]*M.m_ppElements[idx][k].Value;
				solution[idx] = (b[idx]-temp) / diagonal[j].Value;
				count++;
			}
		else
			for(size_t j=start;j<stop; j++)
			{
				T2 temp=0;
				int idx = diagonal[j].N;
				if(idx<0)	continue;
				for(int k=0;k<M.groupSizes[idx];k++)	temp+=solution[M.m_ppElements[idx][k].N]*M.m_ppElements[idx][k].Value;
				solution[idx] = (b[idx]-temp) / diagonal[j].Value;
				count++;
			}
	return count;
}

template <class Matrix,class Data>
static int SolveConjugateGradient(const Matrix& SPD,const Vector<Data>& b,const int& iters,Vector<Data>& solution,const double eps)
{
	Vector<Data> d,r,Md,temp;
	double alpha,beta,rDotR,oldRDotR;
	Md.Resize(b.Dimensions());

	temp.Resize(b.Dimensions());
	SPD.Multiply(solution,temp);
	d=r=b-temp;
	oldRDotR=rDotR=r.Dot(r);
	if(b.Dot(b)<=eps)
	{
		solution.SetZero();
		return 0;
	}
	int i;
	for(i=0;i<iters;i++)
	{
		double temp;
		SPD.Multiply(d,Md);
		temp=d.Dot(Md);
		if(temp<=eps)
		{
			break;
		}
		alpha=rDotR/temp;
		r.SubtractScaled(Md,alpha);
		temp=r.Dot(r);
		// BADNESS!!! How can the size of the residual increase?
		if(temp>2*oldRDotR)
		{
			break;
		}
		oldRDotR=rDotR;
		beta=temp/rDotR;
		solution.AddScaled(d,alpha);
		if(beta<=eps)
		{
			break;
		}
		rDotR=temp;
		Vector<Data>::Add(d,beta,r,d);
	}
	return i;
}
template <class Matrix,class IPS,class Real>
static int SolveConjugateGradient(const Matrix& SPD,const Vector<IPS>& b,const int& iters,Vector<IPS>& solution,const double eps)
{
	Vector<IPS> d,r,Md,temp;
	double alpha,beta,rDotR,oldRDotR;
	Md.Resize(b.Dimensions());

	temp.Resize(b.Dimensions());
	SPD.Multiply(solution,temp);
	d=r=b-temp;
	oldRDotR=rDotR=r.IPSDot(r);
	if(b.IPSDot(b)<=eps)
	{
		solution.SetZero();
		return 0;
	}
	int i;
	for(i=0;i<iters;i++)
	{
		double temp;
		SPD.Multiply(d,Md);
		temp=d.IPSDot(Md);
		if(temp<=eps)
		{
			break;
		}
		alpha=rDotR/temp;
		r.SubtractScaled(Md,alpha);
		temp=r.IPSDot(r);
		// BADNESS!!! How can the size of the residual increase?
		if(temp>2*oldRDotR)
		{
			break;
		}
		oldRDotR=rDotR;
		beta=temp/rDotR;
		solution.AddScaled(d,alpha);
		if(beta<=eps)
		{
			break;
		}
		rDotR=temp;
		Vector<IPS>::Add(d,beta,r,d);
	}
	return i;
}


template<class MType,class VType>
int SolveConjugateGradient(const SparseMatrix<MType>& A,const Vector<VType>& b,const int& iters,Vector<VType>& x,
						   Vector<VType> (*Multiply)(const SparseMatrix<MType>&,const Vector<VType>&))
{
	double eps=1e-16;
	Vector<VType> r = b - Multiply(A,x);
	Vector<VType> d = r;
	double delta_new = r.Dot(r);
	double delta_0 = delta_new;
	int i;
	for(i=0; i<iters && delta_new>eps*delta_0 ;i++)
	{
		Vector<VType> q = Multiply(A,d);
		double alpha = delta_new / d.Dot(q);
		x = x + d*alpha;
		if( !(i%50) )	r = b - Multiply(A,x);
		else			r = r - q*alpha;

		double delta_old = delta_new;
		delta_new = r.Dot(r);
		double beta = delta_new / delta_old;
		d = r + d*beta;
	}
	return i;
}


//////////////////////////
// PivotSymmetricMatrix //
//////////////////////////
template <class Matrix>
template <class Data>
Vector<Data> PivotSymmetricMatrix<Matrix>::operator * (const Vector<Data>& in) const
{
	Vector<Data> out;
	Multiply(in,out);
	return out;
}
template <class Matrix>
template <class Data>
void PivotSymmetricMatrix<Matrix>::Multiply(const Vector<Data>& in,Vector<Data>& out) const
{
	Vector<Data> tempIn,tempOut;
	tempIn=in;
	for(int i=rightMatrices.size()-1;i>=0;i--)
	{
		tempOut.Resize(rightMatrices[i].second.second);
		rightMatrices[i].first.Multiply(tempIn,tempOut);
		tempIn=tempOut;
	}
	tempOut.Resize(pivot.second.second);
	pivot.first.Multiply(tempIn,tempOut);
	tempIn=tempOut;

	for(int i=0;i<rightMatrices.size();i++)
	{
		tempOut.Resize(rightMatrices[i].second.first);
		rightMatrices[i].first.MultiplyTranspose(tempIn,tempOut);
		tempIn=tempOut;
	}
	out=tempIn;
}
template <class Matrix>
void PivotSymmetricMatrix<Matrix>::setPivot(const Matrix& M,int inDim,int outDim)
{
	pivot=std::pair<Matrix,std::pair<int,int> >(M,std::pair<int,int>(inDim,outDim));
}
template <class Matrix>
void PivotSymmetricMatrix<Matrix>::push(const Matrix& M,int inDim,int outDim)
{
	rightMatrices.push_back(std::pair<Matrix,std::pair<int,int> >(M,std::pair<int,int>(inDim,outDim)));
}
template <class Matrix>
void PivotSymmetricMatrix<Matrix>::pop(void)
{
	rightMatrices.pop_back();
}
