/*
Copyright (c) 2009, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

#ifndef POISSON_MESH_DATA_INCLUDED
#define POISSON_MESH_DATA_INCLUDED


#define MEMORY_ALLOCATOR_BLOCK_SIZE 1<<12

#include <vector>
#include "FunctionData.h"
#include "Octree.h"


class TriangleData
{
public:
	TriangleIndex tIndex;
	int parent;

	TriangleData( void ) { ; }
	TriangleData( const TriangleIndex& tIndex , const int& parent ) { this->tIndex = tIndex , this->parent = parent; }
};
class PolygonData
{
public:
	std::vector< int > polygon;
	int parent;
};
template< class Real >
class MeshTreeNodeData
{
public:
	int tStart , tEnd;
	int index , tempIndex;
	MeshTreeNodeData(void);
	~MeshTreeNodeData(void);
};
template< class Real >
class DownSampleMatrix
{
public:
	int inDim , outDim;
	SparseMatrix< Real > ds;
	template< class C > Vector< C > operator * ( const Vector< C >& v ) const { return ds * v; }
	template< class C > void MultiplyTranspose( const Vector< C >& in , Vector< C >& out , bool checkIndex=true ) const { ds.MultiplyTranspose( in , out , checkIndex ); }
	template< class C > void Multiply( const Vector< C >& in , Vector< C >& out ) const { ds.Multiply( in , out ); }
};

template< class Real , bool Primal >
class MeshOctree
{
public:
	static const int VERBOSE = 1;
	static const int FULL_VERBOSE = 2;
	typedef OctNode< MeshTreeNodeData< Real > , Real > TreeOctNode;
protected:
	typename OctNode< MeshTreeNodeData< Real > , Real >::NeighborKey3 neighborKey3;	
	typename OctNode< MeshTreeNodeData< Real > , Real >::NeighborKey5 neighborKey5;
	const TreeOctNode* getLeafNode( const Point3D<Real>& position ) const;
	TreeOctNode* getLeafNode( const Point3D<Real>& position , int maxDepth , bool init);
	void _getPointOffset( const Point3D< Real >& position , int depth , int offset[3] ) const;
	typename TreeOctNode::Neighbors3 _getFinestNodeNeighbors( const Point3D< Real >& position , int depth ,int offset[3] );


	template< class C >
	void _setNoiseCoefficients( const double& persistence , std::vector< C >& coefficients ) const;
	template< class C >
	void _setNoiseCoefficients( int minDepth , int maxDepth , const double& persistence , std::vector< C >& coefficients ) const;

	void _addSamples( const std::vector< OrientedPoint3D< Real > >& samples , const std::vector< int >& indices , int depth , bool progress );

	void _setTriangles( TreeOctNode* node , const std::vector< TriangleIndex >& triangles , const std::vector< int >& parents , int& offset );
	void _addTriangles( const std::vector< TriangleIndex >& triangles , int depth );
	void _addTriangles( TreeOctNode* node , const std::vector< TriangleIndex >& triangles , std::vector< PolygonData >& polygons , int depth ,
		hash_map< long long , int >& vTable , std::vector< TriangleIndex >& splitTriangles , std::vector< int >& splitParents );
	std::vector< int > _dimension;

	int _setPrimalTree( int depth , double cutOff );
	int _setDualTree  ( int depth , double cutOff );

	void _setPrimalMultiRes( void );
	void _setDualMultiRes( void );
	int _setAdjacency( int depth , SparseMatrix< Real >& matrix ) const;
	void _compressMatrix( SparseMatrix< Real >& matrix ) const;
	void _setFromPrimalSums( SparseMatrix< Real >& matrix, double sums[2][2][2][2][2][2] , int idx ) const;
	void _setFromDualSums( SparseMatrix< Real >& matrix, double sums[3][3][3][3][3][3] , int idx ) const;
	int _getPrimalDotProductMatrix( int depth , SparseMatrix<Real>& dotProduct , bool progress ) const;
	int _getDualDotProductMatrix( int depth , SparseMatrix<Real>& dotProduct , bool progress ) const;
	int _getPrimalLaplacianMatrix ( int depth , SparseMatrix<Real>& laplacian , bool progress ) const;
	int _getDualLaplacianMatrix ( int depth , SparseMatrix<Real>& laplacian , bool progress ) const;
	int _getPrimalConstrainedLaplacianMatrix ( int depth , SparseMatrix<Real>& laplacian , Real dotWeight , Real lapWeight , bool progress ) const;
	int _getDualConstrainedLaplacianMatrix ( int depth , SparseMatrix<Real>& laplacian , Real dotWeight , Real lapWeight , bool progress ) const;
	template< class V , class HighPrecisionV , class Function >
	void _getPrimalPreciseDotVector( const std::vector< Function >& functions , int depth , std::vector< V >& b , bool progress ) const;
	template< class V , class HighPrecisionV , class Function >
	void _getDualPreciseDotVector( const std::vector< Function >& functions , int depth , std::vector< V >& b ,  bool progress ) const;
	template< class V , class HighPrecisionV , class Function >
	int _getPrimalPreciseGradientDotGradientVector( const std::vector< Function >& gradientFunctions , int depth , std::vector< V >& b , bool progress , bool normalize ) const;
	template< class V , class HighPrecisionV , class Function >
	int _getDualPreciseGradientDotGradientVector( const std::vector< Function >& gradientFunctions , int depth , std::vector< V >& b , bool progress , bool normalize ) const;
	template< class V , class HighPrecisionV , class Function >
	int _getPrimalPreciseConstrainedGradientDotGradientVector( const std::vector< Function >& functions , Real dotWeight , Real lapWeight , int depth , std::vector< V >& b , bool progress , bool normalize ) const;
	template< class V , class HighPrecisionV , class Function >
	int _getDualPreciseConstrainedGradientDotGradientVector( const std::vector< Function >& functions , Real dotWeight , Real lapWeight , int depth , std::vector< V >& b , bool progress , bool normalize ) const;
	template< class V , class HighPrecisionV >
	V _getPrimalPreciseCoefficientIntegral( const std::vector< V >& coefficients , int depth ) const;
	template< class V , class HighPrecisionV >
	V _getDualPreciseCoefficientIntegral( const std::vector< V >& coefficients , int depth ) const;

	template< class Vertex , class V >
	void _getPrimalValues( const std::vector< Vertex >& points , const std::vector< V >& coefficients , std::vector< V >& values );
	template< class Vertex , class V >
	void _getDualValues( const std::vector< Vertex >& points , const std::vector< V >& coefficients , std::vector< V >& values );
	template< class OrientedVertex , class V >
	void _getPrimalGradientValues ( const std::vector< OrientedVertex >& points , const std::vector< V >& coefficients , std::vector< Gradient3D< V > >& gradients );
	template< class OrientedVertex , class V >
	void _getDualGradientValues ( const std::vector< OrientedVertex >& points , const std::vector< V >& coefficients , std::vector< Gradient3D< V > >& gradients );

	template< class C > void _primalDownSample( const Vector< C >& high , int depth , Vector< C >& low  ) const;
	template< class C > void _dualDownSample  ( const Vector< C >& high , int depth , Vector< C >& low  ) const;
	template< class C > void _primalUpSample  ( const Vector< C >& low  , int depth , Vector< C >& high ) const;
	template< class C > void _dualUpSample    ( const Vector< C >& low  , int depth , Vector< C >& high ) const;
	template< class C >	void _primalDownSample( const std::vector< C >& high , int depth , std::vector< C >& low  ) const;
	template< class C >	void _dualDownSample  ( const std::vector< C >& high , int depth , std::vector< C >& low  ) const;
	template< class C > void _primalUpSample  ( const std::vector< C >& low  , int depth , std::vector< C >& high ) const;
	template< class C > void _dualUpSample    ( const std::vector< C >& low  , int depth , std::vector< C >& high ) const;
	void _primalDownSample( const SparseMatrix< Real >& high , const DownSampleMatrix< Real >& hightToLow , int depth , SparseMatrix< Real >& low ) const;
	void _dualDownSample  ( const SparseMatrix< Real >& high , const DownSampleMatrix< Real >& hightToLow , int depth , SparseMatrix< Real >& low ) const;
	void _primalDownSample( const SparseMatrix< Real >& high , int depth , SparseMatrix< Real >& low ) const;
	void _dualDownSample(   const SparseMatrix< Real >& high , int depth , SparseMatrix< Real >& low ) const;
	void _getPrimalDownSampleMatrix( int high, DownSampleMatrix< Real >& highToLow ) const;
	void _getDualDownSampleMatrix  ( int high, DownSampleMatrix< Real >& highToLow ) const;

public:
	int dimension( int depth ) const;
	static double maxMemoryUsage;
	static double MemoryUsage(void);
	TreeOctNode tree;
	FunctionData< ( Primal ? 1 : 2 ) , double > fData;
	MeshOctree( void );

	template<class OrientedVertex>
	int setTree( const std::vector< OrientedVertex >& vertices , const std::vector< TriangleIndex >& triangles , int depth , double cutOff , bool progress=false );
	int setTree( const std::vector< OrientedPoint3D< Real > >& samples , const std::vector< int >& indices , int depth , double cutOff , bool progress=false );

	int getDotProductMatrix( int depth , SparseMatrix<Real>& dotProduct , bool progress ) const;
	int getLaplacianMatrix ( int depth , SparseMatrix<Real>& laplacian , bool progress ) const;
	int getConstrainedLaplacianMatrix ( int depth , SparseMatrix<Real>& laplacian , Real dotWeight , Real lapWeight , bool progress ) const;

	template< class Vertex , class V >
	void getValues( const std::vector< Vertex >& points , const std::vector< V >& coefficients , std::vector< V >& values );
	template< class OrientedVertex , class V >
	void getGradientValues ( const std::vector< OrientedVertex >& points , const std::vector< V >& coefficients , std::vector< Gradient3D< V > >& gradients );

	static void SetTangents( const Point3D< Real >& normal , Point3D< Real >& t1 , Point3D< Real >& t2 );

	////////////////////////////////////////////////////////
	// Computes the dot-products with the basis functions //
	// V / HighPrecisionV must be a vector space
	// The functions must take in a value of type Point3D< V::R > and return an object of type V
	template< class V , class HighPrecisionV , class Function >
	void getPreciseDotVector( const std::vector< Function >& functions , int depth , std::vector< V >& b , bool progress ) const;
	template< class V , class Function >
	void getDotVector( const std::vector< Function >& functions , int depth , std::vector< V >& b , bool progress ) const
	{ return getPreciseDotVector< V , V , Function >( functions , depth , b , progress ); }

	/////////////////////////////////////////////////////////////////////////////////////////
	// Computes the dot-products of the divergence of the samples with the basis functions //
	// V / HighPrecisionV must be a vector space
	// The functions must take in a value of type Point3D< V::R > and return an object of type Gradient3D< V >
	template< class V , class HighPrecisionV , class Function >
	int getPreciseGradientDotGradientVector( const std::vector< Function >& gradientFunctions , int depth , std::vector< V >& b , bool progress , bool normalize=true ) const;
	template< class V , class Function >
	int getGradientDotGradientVector( const std::vector< Function >& gradientFunctions , int depth , std::vector< V >& b , bool progress , bool normalize=true ) const
	{ return getPreciseGradientDotGradientVector< V , V , Function >( gradientFunctions , depth , b , progress , normalize ); }

	/////////////////////////////////////////////////////////////////////////////////////////
	// Computes the dot-products of the divergence of the samples with the basis functions //
	// V / HighPrecisionV must be a vector space
	// The functions must take in a value of type Point3D< V::R > and return an object of type Gradient3D< V >
	template< class V , class HighPrecisionV , class Function >
	int getPreciseConstrainedGradientDotGradientVector( const std::vector< Function >& functions , Real dotWeight , Real lapWeight , int depth , std::vector< V >& b , bool progress , bool normalize=true ) const;
	template< class V , class Function >
	int getConstrainedGradientDotGradientVector( const std::vector< Function >& functions , Real dotWeight , Real lapWeight , int depth , std::vector< V >& b , bool progress , bool normalize=true ) const
	{ return getPreciseConstrainedGradientDotGradientVector< V , V , Function >( functions , dotWeight , lapWeight , depth , b , progress , normalize ); }

	///////////////////////////////////////////
	// Computes the integral of the function //
	// V / HighPrecisionV must be vector spaces
	// The functions must take in a value of type Point3D< V::R > and return an object of type Gradient3D< V >
	template< class V , class HighPrecisionV >	V getPreciseCoefficientIntegral( const std::vector< V >& coefficients , int depth ) const;
	template< class V >	V getCoefficientIntegral( const std::vector< V >& coefficients , int depth ) const { return getPreciseIntegral< V , V >( coefficients , depth ); }
	template< class V , class HighPrecisionV , class Function >	V getPreciseIntegral( const std::vector< Function >& functions ) const;
	template< class V , class Function > V getIntegral( const std::vector< Function >& functions ) const { return getPreciseIntegral< V , V , Function >( functions ); }

	Real getArea( void ) const;

	template< class C > void downSample( const Vector< C >& high , int depth , Vector< C >& low  ) const;
	template< class C > void upSample  ( const Vector< C >& low  , int depth , Vector< C >& high ) const;
	template< class C >	void downSample( const std::vector< C >& high , int depth , std::vector< C >& low  ) const;
	template< class C > void upSample  ( const std::vector< C >& low  , int depth , std::vector< C >& high ) const;

	void downSample( const SparseMatrix< Real >& high , const DownSampleMatrix< Real >& hightToLow , int depth , SparseMatrix< Real >& low ) const;
	void downSample( const SparseMatrix< Real >& high , int depth , SparseMatrix< Real >& low ) const;
	void getDownSampleMatrix( int high, DownSampleMatrix< Real >& highToLow ) const;

	template< int Dim >
	void setNoiseCoefficients( const double& persistence , std::vector< Point< Real , Dim > >& coefficients ) const;
	template< int Dim >
	void setNoiseCoefficients( const int& minDepth , const int& maxDepth , const double& persistence , std::vector< Point< Real , Dim > >& coefficients ) const;

	void setNoiseCoefficients( const double& persistence , std::vector< Point3D< Real > >& coefficients ) const;
	void setNoiseCoefficients( const int& minDepth , const int& maxDepth , const double& persistence , std::vector< Point3D< Real > >& coefficients ) const;

	template< int Dim >
	void setIntegralCoefficients( std::vector< Point< Real , Dim > >& coefficients , bool allDepths ) const;
	void setCheckerBoardCoefficients( int depth , std::vector< Point3D< Real > >& coefficients ) const;

	template< class C >
	void solveLinearSystem( SparseMatrix< Real >& m , Vector< C >& b , Vector< C >& x,
		int minDepth , int iters , int vCycles , float sor , bool cascadic , int verbose , bool vCycle , bool solveBaseCG , double& normB , double& normR , double& normX ,
		double& vectorSetTime , double& matrixSetTime , double& solveTime ,
		float iterMultiplier=1 ) const;
	template< class C >
	void solveLinearSystem( SparseMatrix< Real >& m , Vector< C >& b , Vector< C >& x,
		int minDepth , int iters , int vCycles , float sor , bool cascadic , int verbose , bool vCycle , bool solveBaseCG , float iterMultiplier=1 ) const;

	template< class OrientedVertex , class Vertex >
	void EstimateMeanCurvature( const std::vector< OrientedVertex >& samples , int depth , int iters , const std::vector< Vertex >& vertices , std::vector< Real >& meanCurvature );

	int vertexCount( void ) const;
	const Point3D< Real >& vertex( int idx ) const;
	int triangleCount( void ) const;
	const TriangleIndex& triangle( int idx ) const;
	int triangleParent( int idx ) const;
private:
	bool _fromSamples;
	std::vector< OrientedPoint3D< Real > > _samples;
	std::vector< Point3D< Real > > _vertices , _baseNormals;
	std::vector< TriangleIndex > _triangles;
	std::vector< int > _parents;

	void _setDiagonal( const SparseMatrix< Real >& M , Vector< MatrixEntry< Real> >& D , bool verbose ) const;

	template< class C >
	void _solveRestriction(int minDepth, int maxDepth,
		const std::vector<  SparseMatrix< Real >* >& M,
		const std::vector< Vector< MatrixEntry< Real > > >& D,
		const std::vector< Vector< C >* >& B,
		const std::vector<  DownSampleMatrix< Real >* >& DS,
		const std::vector< Vector< C >* >& X,
		size_t iters , float sor , int verbose , double& solveTime , double& vectorSetTime , float iterMultiplier ) const;

	template< class C >
	void _solveProlongation(int minDepth, int maxDepth,
		const std::vector<  SparseMatrix< Real >* >& M,
		const std::vector< Vector< MatrixEntry< Real > > >& D,
		const std::vector< Vector< C >* >& B,
		const std::vector<  DownSampleMatrix< Real >* >& DS,
		const std::vector< Vector< C >* >& X,
		size_t iters , float sor ,  int verbose , double& solveTime , double& vectorSetTime , float iterMultiplier ) const;
	template< class C >
	int _solveConjugateGradient( const SparseMatrix<Real>& A , const Vector< C >& b , const size_t& iters , Vector< C >& x , size_t dim ) const;
	template< class C >
	int _solveConjugateGradient( const SparseMatrix<Real>& A , const Vector< C >& b , const Vector< Real >& precon , const size_t& iters , Vector< C >& x , size_t dim ) const;
};
template< class Real >
void SetTangents( const Point3D< Real >& normal , Point3D< Real >& t1 , Point3D< Real >& t2 );
template< class Real >
void SetCurvatureMatrix( const Point3D< Real >& normal , const Gradient3D< Point3D< Real > >& normalGradient ,
						Matrix< Real , 3 , 3 >& c , Point3D< Real >& t1 , Point3D< Real > &t2 );
#include "PoissonMeshData.inl"
#include "PrimalPoissonMeshData.inl"
#include "DualPoissonMeshData.inl"
#endif // POISSON_MESH_DATA_INCLUDED
