/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#ifndef TRIANGLE_INTEGRALS_INCLUDED
#define TRIANGLE_INTEGRALS_INCLUDED
#include "Geometry.h"

// Cubature weights taken from "Asymmetric cubature formulas for polynomial integration in the triangle and square", Taylor 2007
// and "A new 11 point degree 6 cubature formula for the triangle", May and Taylor, 2007
// Used to compute the integrals of (up to) 12-th degree polynomials over a unit right triangle.
// (UPDATED: "GAUSSIAN QUADRATURE FORMULAS FOR TRIANGLES", G. R. COWPER, 1972)
// (UPDATED: "Moderate Degree Symmetric Quadrature Rules for the Triangle", J. N. LYNESS and D. JESPERSEN, 1975)

template< int Samples >
class TriangleIntegrator
{
public:
	const static int SampleNum=Samples;
private:
	const static double _SamplePositionsAndWeights[];
	double _area;
	Point3D< double > _samplePositions[ Samples ];
	void _init( const Point3D< double >& p1 , const Point3D< double >& p2 , const Point3D< double >& p3 );
	void _init( const Point3D< float  >& p1 , const Point3D< float  >& p2 , const Point3D< float  >& p3 );
public:
	double weights[ Samples ];

	TriangleIntegrator( void );
	TriangleIntegrator( const Point3D< double >& p1 , const Point3D< double >& p2 , const Point3D< double >& p3 ) { _init( p1 , p2 , p3 ); }
	TriangleIntegrator( const Point3D< float  >& p1 , const Point3D< float  >& p2 , const Point3D< float  >& p3 ) { _init( p1 , p2 , p3 ); }
	TriangleIntegrator( const Point3D< double >* p ) { _init( p[0] , p[1] , p[2] ); }
	TriangleIntegrator( const Point3D< float  >* p ) { _init( p[0] , p[1] , p[2] ); }
	inline const Point3D< double >& operator [] ( int idx ) const;
	inline double area( void ) const;
	template< class VectorType > VectorType integrate( const VectorType* values ) const;
};

typedef TriangleIntegrator<  1 > TriangleIntegrator0;
typedef TriangleIntegrator<  3 > TriangleIntegratorMedian;
typedef TriangleIntegrator<  6 > TriangleIntegrator4;
typedef TriangleIntegrator< 11 > TriangleIntegrator6;
typedef TriangleIntegrator< 24 > TriangleIntegrator10;
typedef TriangleIntegrator< 27 > TriangleIntegrator11;
typedef TriangleIntegrator< 32 > TriangleIntegrator12;

template< class TIntegrator >
class TriangulatingPolygonIntegrator
{
	std::vector< TIntegrator > _tIntegrators;
public:
	static int SampleNum( int verts ) { return TIntegrator::SampleNum * (verts-2); }
	void set( const Point3D< double >* verts , int vCount );
	void set( const Point3D< float  >* verts , int vCount );
	inline const Point3D< double >& operator[] ( int idx ) const;
	inline double weight( int idx ) const;
	inline double area( void ) const;
	template< class VectorType > VectorType integrate( const VectorType* values ) const;
};

class MedianPolygonIntegrator
{
	std::vector< Point3D< double > > _samples;
	double _area;
public:
	static int SampleNum( int verts ) { return verts; }
	void set( const Point3D< double >* verts , int vCount );
	void set( const Point3D< float  >* verts , int vCount );
	inline const Point3D< double >& operator[] ( int idx ) const;
	inline double weight( int idx ) const;
	inline double area( void ) const;
	template< class VectorType > VectorType integrate( const VectorType* values ) const;
};
#include "TriangleIntegrals.inl"

#endif // TRIANGLE_INTEGRALS_INCLUDED
