/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
template< class T , class const_iterator > size_t SparseMatrixInterface< T , const_iterator >::Entries( void ) const
{
	size_t entries = 0;
	for( size_t i=0 ; i<Rows() ; i++ ) entries += RowSize( i );
	return entries;
}
template< class T , class const_iterator > double SparseMatrixInterface< T , const_iterator >::SquareNorm( void ) const
{
	double n=0;
	for( size_t i=0 ; i<Rows() ; i++ )
	{
		const_iterator e = end( i );
		for( const_iterator iter = begin( i ) ; iter!=e ; iter++ ) n += iter->Value * iter->Value;
	}
	return n;
}
template< class T , class const_iterator > double SparseMatrixInterface< T , const_iterator >::SquareASymmetricNorm( void ) const
{
	double n=0;
	for( size_t i=0 ; i<Rows() ; i++ )
	{
		const_iterator e = end( i );
		for( const_iterator iter1 = begin( i ) ; iter1!=e ; iter1++ )
		{
			int j = iter1->N;
			const_iterator e = end( j );
			double value = 0;
			for( const_iterator iter2 = begin( j ) ; iter2!=e ; iter2++ )
			{
				int k = iter2->N;
				if( k==i ) value += iter2->Value;
			}
			n += (iter1->Value-value) * (iter1->Value-value);
		}
	}
	return n;
}
template< class T , class const_iterator > double SparseMatrixInterface< T , const_iterator >::SquareASymmetricNorm( int& idx1 , int& idx2 ) const
{
	double n=0;
	double max=0;
	for( size_t i=0 ; i<Rows() ; i++ )
	{
		const_iterator e = end( i );
		for( const_iterator iter = begin( i ) ; iter!=e ; iter++ )
		{
			int j = iter->N;
			const_iterator e = end( j );
			double value = 0;
			for( const_iterator iter = begin( j ) ; iter!=e ; iter++ )
			{
				int k = iter->N;
				if( k==i ) value += iter->Value;
			}
			double temp = (iter->Value-value) * (iter->Value-value);
			n += temp;
			if( temp>=max ) idx1 = i , idx2 = j , max=temp;
		}
	}
	return n;
}
template< class T , class const_iterator >
template< class T2>
void SparseMatrixInterface< T , const_iterator >::Multiply( ConstPointer( T2 ) In , Pointer( T2 ) Out ) const
{
	ConstPointer( T2 ) in = In;
	for( size_t i=0 ; i<Rows() ; i++ )
	{
#if 1
		T2 temp = Out[i];
#else
		T2 temp;
		temp *= 0;
#endif
		ConstPointer( T2 ) _in = in;
		const_iterator e = end( i );
		for( const_iterator iter = begin( i ) ; iter!=e ; ++iter ) temp += _in[ iter->N ] * iter->Value;
#if 1
		Out[i] = temp;
#else
		Out[i] += temp;
#endif
	}
}
template< class T , class const_iterator >
template< class T2 >
void SparseMatrixInterface< T , const_iterator >::MultiplyParallel( ConstPointer( T2 ) In , Pointer( T2 ) Out , int threads , int multiplyFlag ) const
{
	ConstPointer( T2 ) in = In;
#pragma omp parallel for num_threads( threads )
	for( int i=0 ; i<Rows() ; i++ )
	{
		T2 temp;
		memset( &temp , 0 , sizeof(T2) );
		ConstPointer( T2 ) _in = in;
		const_iterator e = end( i );
		for( const_iterator iter = begin( i ) ; iter!=e ; iter++ ) temp += _in[ iter->N ] * iter->Value;
		switch( multiplyFlag )
		{
		case MULTIPLY_CLEAR:    Out[i]  = temp ; break;
		case MULTIPLY_ADD:      Out[i] += temp ; break;
		case MULTIPLY_SUBTRACT: Out[i] -= temp ; break;
		}
	}
}

template< class T , class const_iterator >
template< class T2 >
void SparseMatrixInterface< T , const_iterator >::MultiplyTranspose( ConstPointer( T2 ) In , Pointer( T2 ) Out ) const
{
	T2* out = &Out[0];
	for( size_t i=0 ; i<Rows() ; i++ )
	{
		T2* _out = out;
		const_iterator e = end( i );
		for( const_iterator iter = begin( i ) ; iter!=e ; ++iter ) _out[ iter->N ] += In[i] * iter->Value;
	}
}
template< class T , class const_iterator >
template< class T2 >
void SparseMatrixInterface< T , const_iterator >::BMinusMX( ConstPointer( T2 ) B , ConstPointer( T2 ) X , Pointer( T2 ) Out ) const
{
	ConstPointer( T2 ) x = X;
	for( size_t i=0 ; i<Rows() ; i++ )
	{
		T2 temp;
		temp *= 0;
		ConstPointer( T2 ) _x = x;
		const_iterator e = end( i );
		for( const_iterator iter = begin( i ) ; iter!=e ; ++iter ) temp += _x[ iter->N ] * iter->Value;
		Out[i] = B[i] - temp;
	}

}
template< class T , class const_iterator >
template< class T2 , class T3 , bool UnitDiagonal , bool StrippedDiagonal , bool UseSOR >
void SparseMatrixInterface< T , const_iterator >::SolveGaussSeidel( ConstPointer( T3 ) diagonal , ConstPointer( T2 ) b , int iters , Pointer( T2 ) solution , T sor ) const
{
	for( int i=0 ; i<iters ; i++ )
		if( UnitDiagonal )
			for( size_t j=0 ; j<Rows() ; j++ )
			{
				T2 temp;
				temp *= 0;
				Pointer( T2 ) _solution = solution;
				const_iterator e = end( j );
				for( const_iterator iter = begin( j ) ; iter!=e ; ++iter ) temp += _solution[ iter->N ] * iter->Value;
				if( UseSOR )
					if(StrippedDiagonal) solution[j]  = solution[j]*T(1.0-sor) + (b[j]-temp) * sor;
					else                 solution[j] +=                          (b[j]-temp) * sor;
				else
					if(StrippedDiagonal) solution[j]  = (b[j]-temp);
					else                 solution[j] += (b[j]-temp);
			}
		else
			for( size_t j=0 ; j<Rows() ; j++ )
			{
				T2 temp;
				temp *= 0;
				Pointer( T2 ) _solution = solution;
				T dValue = T(1.) / diagonal[j];
				const_iterator e = end( j );
				for( const_iterator iter = begin( j ) ; iter!=e ; ++iter ) temp += _solution[ iter->N ] * iter->Value;
				if( UseSOR )
					if(StrippedDiagonal) solution[j]  = solution[j]*T(1.0-sor) + (b[j]-temp) * dValue * sor;
					else                 solution[j] +=                          (b[j]-temp) * dValue * sor;
				else
					if(StrippedDiagonal) solution[j]  = (b[j]-temp) * dValue;
					else                 solution[j] += (b[j]-temp) * dValue;
			}
}

template< class T , class const_iterator >
template< class T2 , class T3 , bool UnitDiagonal , bool StrippedDiagonal , bool UseSOR >
void SparseMatrixInterface< T , const_iterator >::SolveGaussSeidelAndResidual( ConstPointer( T3 ) diagonal , ConstPointer( T2 ) b , int iters , Pointer( T2 ) solution , Pointer( T2 ) residual , T sor ) const
{
	for( int i=0 ; i<iters ; i++ )
		if( UnitDiagonal )
			for( size_t j=0 ; j<Rows() ; j++ )
			{
				T2 temp;
				temp *= 0;
				Pointer( T2 ) _solution = solution;
				const_iterator e = end( j );
				for( const_iterator iter = begin( j ) ; iter!=e ; ++iter ) temp += _solution[ iter->N ] * iter->Value;
				if( UseSOR )
					if( StrippedDiagonal ) solution[j]  = solution[j]*T(1.0-sor) + (b[j]-temp) * sor;
					else                   solution[j] +=                          (b[j]-temp) * sor;
				else
					if( StrippedDiagonal ) solution[j]  = (b[j]-temp);
					else                   solution[j] += (b[j]-temp);
			}
		else
			for( size_t j=0 ; j<Rows() ; j++ )
			{
				T2 temp;
				temp *= 0;
				Pointer( T2 ) _solution = solution;
				T dValue = T(1.) / diagonal[j];
				const_iterator e = end( j );
				for( const_iterator iter = begin( j ) ; iter!=e ; ++iter ) temp += _solution[ iter->N ] * iter->Value;
				if( UseSOR )
					if( StrippedDiagonal ) solution[j]  = solution[j]*T(1.0-sor) + (b[j]-temp) * dValue * sor;
					else                   solution[j] +=                          (b[j]-temp) * dValue * sor;
				else
					if( StrippedDiagonal ) solution[j]  = (b[j]-temp) * dValue;
					else                   solution[j] += (b[j]-temp) * dValue;
			}
	if( UnitDiagonal )
		for( size_t j=0 ; j<Rows() ; j++ )
		{
			T2 temp;
			temp *= 0;
			Pointer( T2 ) _solution = solution;
			const_iterator e = end( j );
			for( const_iterator iter = begin( j ) ; iter!=e ; ++iter ) temp += _solution[ iter->N ] * iter->Value;
			if( StrippedDiagonal ) residual[j] = b[j] - temp - solution[j];
			else                   residual[j] = b[j] - temp;
		}
	else
		for( size_t j=0 ; j<Rows() ; j++ )
		{
			T2 temp;
			temp *= 0;
			Pointer( T2 ) _solution = solution;
			T dValue = diagonal[j];
			const_iterator e = end( j );
			for( const_iterator iter = begin( j ) ; iter!=e ; ++iter ) temp += _solution[ iter->N ] * iter->Value;
			if( StrippedDiagonal ) residual[j] = b[j] - temp - ( solution[j] * dValue );
			else                   residual[j] = b[j] - temp;
		}
}


template< class T2 >
void ParallelSolution< T2 >::init( int iters , const std::vector< int >& entriesPerSlice , const std::vector< int >& slicesPerThread , int sliceDependence )
{
	_threads = slicesPerThread.size();
	if( _threads<1 ) fprintf( stderr , "Could not allocate ParallelSolution with no threads: %d\n" , _threads ) , exit( 0 );

	_iters = iters;
	_entriesPerSlice = &entriesPerSlice[0];
	_slicesPerThread = &slicesPerThread[0];
	_sliceDependence = sliceDependence;
	_sliceCount = entriesPerSlice.size();

	_pSolution.resize( _threads );
	_startEntry.resize( _sliceCount );
	_startEntry[0] = 0;
	for( int i=1 ; i<_sliceCount ; i++ ) _startEntry[i] = _startEntry[i-1] + entriesPerSlice[i-1];

	_sliceBounds.resize( _threads );
	_skirtBounds.resize( _threads );

	for( int t=0 ; t<_threads ; t++ )
	{
		// Compute the starting and ending slices associated with the current thread.
		int startSlice = 0 , endSlice;
		_sliceBounds[t].first = 0;
		for( int i=0 ; i<t ; i++ ) _sliceBounds[t].first += slicesPerThread[i];
		_sliceBounds[t].second = _sliceBounds[t].first + slicesPerThread[t];

		_skirtBounds[t].first  = _sliceBounds[t].first  - (iters+1) * sliceDependence;
		_skirtBounds[t].second = _sliceBounds[t].second + (iters+1) * sliceDependence;
		if( _skirtBounds[t].first<0 ) _skirtBounds[t].first = 0;
		if( _skirtBounds[t].second>entriesPerSlice.size() ) _skirtBounds[t].second = entriesPerSlice.size( );

		int skirtEntryStart = _startEntry[_skirtBounds[t].first];
		int      entryStart = _startEntry[_sliceBounds[t].first];
		int      entryEnd   = _startEntry[_sliceBounds[t].second-1] + _entriesPerSlice[_sliceBounds[t].second-1];
		int skirtEntryEnd   = _startEntry[_skirtBounds[t].second-1] + _entriesPerSlice[_skirtBounds[t].second-1];
		_pSolution[t] = AllocPointer< T2 >( skirtEntryEnd - skirtEntryStart );
		_pSolution[t] += entryStart-skirtEntryStart;
	}
}

template< class T2 >
ParallelSolution< T2 >::ParallelSolution( void )
{
	_threads = 0;
}
template< class T2 >
ParallelSolution< T2 >::ParallelSolution( int iters , const std::vector< int >& entriesPerSlice , const std::vector< int >& slicesPerThread , int sliceDependence )
{
	init( iters , entriesPerSlice , slicesPerThread , sliceDependence );
}
template< class T2 >
ParallelSolution< T2 >::~ParallelSolution( void )
{
	for( int t=0 ; t<_threads ; t++ )
	{
		int skirtEntryStart = _startEntry[_skirtBounds[t].first];
		int      entryStart = _startEntry[_sliceBounds[t].first];
		int      entryEnd   = _startEntry[_sliceBounds[t].second-1] + _entriesPerSlice[_sliceBounds[t].second-1];
		int skirtEntryEnd   = _startEntry[_skirtBounds[t].second-1] + _entriesPerSlice[_skirtBounds[t].second-1];
		_pSolution[t] -= entryStart-skirtEntryStart;
		FreePointer( _pSolution[t] );
	}
	_pSolution.clear();
}
template< class T2 >
void ParallelSolution< T2 >::SetFromArray( ConstPointer( T2 ) solution )
{
#pragma omp parallel for num_threads( _threads )
	for( int t=0 ; t<_threads ; t++ )
	{
		int skirtEntryStart = _startEntry[_skirtBounds[t].first];
		int      entryStart = _startEntry[_sliceBounds[t].first];
		int      entryEnd   = _startEntry[_sliceBounds[t].second-1] + _entriesPerSlice[_sliceBounds[t].second-1];
		int skirtEntryEnd   = _startEntry[_skirtBounds[t].second-1] + _entriesPerSlice[_skirtBounds[t].second-1];
		memcpy( _pSolution[t] - ( entryStart-skirtEntryStart ) , &solution[skirtEntryStart] , sizeof( T2 ) * ( skirtEntryEnd - skirtEntryStart ) );
	}
}
template< class T2 >
void ParallelSolution< T2 >::SetToArray( Pointer( T2 ) solution ) const
{
#pragma omp parallel for num_threads( _threads )
	for( int t=0 ; t<_threads ; t++ )
	{
		int skirtEntryStart = _startEntry[_skirtBounds[t].first];
		int      entryStart = _startEntry[_sliceBounds[t].first];
		int      entryEnd   = _startEntry[_sliceBounds[t].second-1] + _entriesPerSlice[_sliceBounds[t].second-1];
		int skirtEntryEnd   = _startEntry[_skirtBounds[t].second-1] + _entriesPerSlice[_skirtBounds[t].second-1];
		memcpy( solution + entryStart , _pSolution[t] , sizeof( T2 ) * ( entryEnd-entryStart ) );
	}
}
template< class T2 >
void ParallelSolution< T2 >::clear( void )
{
#pragma omp parallel for num_threads( _threads )
	for( int t=0 ; t<_threads ; t++ )
	{
		int skirtEntryStart = _startEntry[_skirtBounds[t].first];
		int      entryStart = _startEntry[_sliceBounds[t].first];
		int      entryEnd   = _startEntry[_sliceBounds[t].second-1] + _entriesPerSlice[_sliceBounds[t].second-1];
		int skirtEntryEnd   = _startEntry[_skirtBounds[t].second-1] + _entriesPerSlice[_skirtBounds[t].second-1];
		memset( _pSolution[t] - ( entryStart-skirtEntryStart ) , 0 , sizeof( T2 ) * ( skirtEntryEnd - skirtEntryStart ) );
	}
}
template< class T2 >
void ParallelSolution< T2 >::_clearSkirts( void )
{
	// This should not be parallelized (possibly not enough computation to be worth the overhead)
	for( int t=0 ; t<_threads ; t++ )
	{
		int skirtEntryStart = _startEntry[_skirtBounds[t].first];
		int      entryStart = _startEntry[_sliceBounds[t].first];
		int      entryEnd   = _startEntry[_sliceBounds[t].second-1] + _entriesPerSlice[_sliceBounds[t].second-1];
		int skirtEntryEnd   = _startEntry[_skirtBounds[t].second-1] + _entriesPerSlice[_skirtBounds[t].second-1];
		if( entryStart - skirtEntryStart ) memset( _pSolution[t] - ( entryStart-skirtEntryStart ) , 0 , sizeof( T2 ) * ( entryStart - skirtEntryStart ) );
		if( skirtEntryEnd - entryEnd )     memset( _pSolution[t] + ( entryEnd - entryStart ) , 0 , sizeof( T2 ) * ( skirtEntryEnd - entryEnd ) );
	}
}
// [WARNING] The synchronization and merging code is not completely correct. If the skirts are larger than the neighbors, there will be trouble.
template< class T2 >
void ParallelSolution< T2 >::_mergeSkirts( void )
{
	// This should not be parallelized (possibly not enough computation to be worth the overhead)
	for( int t=0 ; t<_threads-1 ; t++ )
	{
		int skirtEntryStart1 = _startEntry[_skirtBounds[t].first];
		int      entryStart1 = _startEntry[_sliceBounds[t].first];
		int      entryEnd1   = _startEntry[_sliceBounds[t].second-1] + _entriesPerSlice[_sliceBounds[t].second-1];
		int skirtEntryEnd1   = _startEntry[_skirtBounds[t].second-1] + _entriesPerSlice[_skirtBounds[t].second-1];
		int skirtEntryStart2 = _startEntry[_skirtBounds[t+1].first];
		int      entryStart2 = _startEntry[_sliceBounds[t+1].first];
		int      entryEnd2   = _startEntry[_sliceBounds[t+1].second-1] + _entriesPerSlice[_sliceBounds[t+1].second-1];
		int skirtEntryEnd2   = _startEntry[_skirtBounds[t+1].second-1] + _entriesPerSlice[_skirtBounds[t+1].second-1];

		int mergeSize;
		T2 *left , *right;

		// Merge the left side of the boundary
		mergeSize = entryStart2 - skirtEntryStart2;
		left  = _pSolution[t  ] + ( entryEnd1-entryStart1 ) - ( entryStart2-skirtEntryStart2 );
		right = _pSolution[t+1] - ( entryStart2-skirtEntryStart2 );
		for( int i=0 ; i<mergeSize ; i++ ) left[i] = right[i] = ( left[i]+right[i] );

		// Merge the right side of the boundary
		mergeSize = skirtEntryEnd1-entryEnd1;
		left  = _pSolution[t  ] + ( entryEnd1-entryStart1 );
		right = _pSolution[t+1];
		for( int i=0 ; i<mergeSize ; i++ ) left[i] = right[i] = ( left[i]+right[i] );
	}
}

template< class T2 >
void ParallelSolution< T2 >::_synchronize( void )
{
	// This should not be parallelized (possibly not enough computation to be worth the overhead)
	for( int t=0 ; t<_threads-1 ; t++ )
	{
		int skirtEntryStart1 = _startEntry[_skirtBounds[t].first];
		int      entryStart1 = _startEntry[_sliceBounds[t].first];
		int      entryEnd1   = _startEntry[_sliceBounds[t].second-1] + _entriesPerSlice[_sliceBounds[t].second-1];
		int skirtEntryEnd1   = _startEntry[_skirtBounds[t].second-1] + _entriesPerSlice[_skirtBounds[t].second-1];
		int skirtEntryStart2 = _startEntry[_skirtBounds[t+1].first];
		int      entryStart2 = _startEntry[_sliceBounds[t+1].first];
		int      entryEnd2   = _startEntry[_sliceBounds[t+1].second-1] + _entriesPerSlice[_sliceBounds[t+1].second-1];
		int skirtEntryEnd2   = _startEntry[_skirtBounds[t+1].second-1] + _entriesPerSlice[_skirtBounds[t+1].second-1];

		memcpy( _pSolution[t+1] - ( entryStart2-skirtEntryStart2 ) , _pSolution[t] + ( entryEnd1-entryStart1 ) - ( entryStart2-skirtEntryStart2 ) , sizeof( T2 ) * ( entryStart2-skirtEntryStart2 ) );
		memcpy( _pSolution[t] + ( entryEnd1-entryStart1 ) , _pSolution[t+1] , sizeof( T2 ) * ( skirtEntryEnd1-entryEnd1 ) );
	}
}
template< class T , class const_iterator >
template< class T2 >
void SparseMatrixInterface< T , const_iterator >::MultiplyParallel( ConstPointer( T2 ) in , ParallelSolution< T2 >* out , int multiplyFlag ) const
{
	const int threads = out->_threads;
	const int sliceDependence = out->_sliceDependence;
	const int sliceCount = out->_sliceCount;
	const int* startEntry = &out->_startEntry[0];
	const int* entriesPerSlice = out->_entriesPerSlice;
#pragma omp parallel for num_threads( threads )
	for( int t=0 ; t<threads ; t++ )
	{
		// Compute the starting and ending slices associated with the current thread.
		int outStartSlice = out->_sliceBounds[t].first;
		int   outEndSlice = out->_sliceBounds[t].second;
		int     outOffset = out->_startEntry[ outStartSlice ];

		Pointer( T2 ) _out = out->_pSolution[t] - outOffset;

		for( int i=outStartSlice ; i<outEndSlice ; i++ )
		{
			int start = startEntry[i];
			int stop  = startEntry[i] + entriesPerSlice[i];
			for( size_t j=start ; j<stop ; j++ )
			{
				T2 temp;
				temp *= 0;
				ConstPointer( T2 ) _in = in;
				const_iterator e = end( j );
				for( const_iterator iter = begin( j ) ; iter!=e ; iter++ ) temp += _in[ iter->N ] * iter->Value;
				switch( multiplyFlag )
				{
				case MULTIPLY_CLEAR:    _out[j]  = temp ; break;
				case MULTIPLY_ADD:      _out[j] += temp ; break;
				case MULTIPLY_SUBTRACT: _out[j] -= temp ; break;
				}
			}
		}
	}
}
// [WARNING]: In general, the next two function calls are not safe since they assume that in order to
// set out->_pSolution[t], we only need to know the values in in->_pSolution[t]!
template< class T , class const_iterator >
template< class T2 >
void SparseMatrixInterface< T , const_iterator >::MultiplyParallel( ParallelSolution< T2 >* in , ParallelSolution< T2 >* out , int multiplyFlag ) const
{
	in->_synchronize();
	const int threads = out->_threads;
	const int sliceDependence = out->_sliceDependence;
	const int sliceCount = out->_sliceCount;
	const int* startEntry = &out->_startEntry[0];
	const int* entriesPerSlice = out->_entriesPerSlice;
#pragma omp parallel for num_threads( threads )
	for( int t=0 ; t<threads ; t++ )
	{
		// Compute the starting and ending slices associated with the current thread.
		int  inStartSlice =  in->_sliceBounds[t].first;
		int    inEndSlice =  in->_sliceBounds[t].second;
		int outStartSlice = out->_sliceBounds[t].first;
		int   outEndSlice = out->_sliceBounds[t].second;
		int      inOffset =  in->_startEntry[  inStartSlice ];
		int     outOffset = out->_startEntry[ outStartSlice ];

		ConstPointer( T2 ) _in  =  in->_pSolution[t] -  inOffset;
		Pointer( T2 )      _out = out->_pSolution[t] - outOffset;

		for( int i=outStartSlice ; i<outEndSlice ; i++ )
		{
			int start = startEntry[i];
			int stop  = startEntry[i] + entriesPerSlice[i];
			for( int j=start ; j<stop ; j++ )
			{
				T2 temp;
				temp *= 0;
				ConstPointer( T2 ) in = _in;
				const_iterator e = end( j );
				for( const_iterator iter = begin( j ) ; iter!=e ; iter++ ) temp += in[ iter->N ] * iter->Value;
				switch( multiplyFlag )
				{
				case MULTIPLY_CLEAR:    _out[j]  = temp ; break;
				case MULTIPLY_ADD:      _out[j] += temp ; break;
				case MULTIPLY_SUBTRACT: _out[j] -= temp ; break;
				}
			}
		}
	}
}
template< class T , class const_iterator >
template< class T2 >
void SparseMatrixInterface< T , const_iterator >::MultiplyTransposeParallel( const ParallelSolution< T2 >* in , ParallelSolution< T2 >* out ) const
{
	const int threads = in->_threads;
	const int sliceDependence = in->_sliceDependence;
	const int sliceCount = in->_sliceCount;
	ConstPointer( int ) startEntry = GetPointer( in->_startEntry );
	const int* entriesPerSlice = in->_entriesPerSlice;

	out->_clearSkirts( );
#pragma omp parallel for num_threads( threads )
	for( int t=0 ; t<threads ; t++ )
	{
		// Compute the starting and ending slices associated with the current thread.
		int  inStartSlice =  in->_sliceBounds[t].first;
		int    inEndSlice =  in->_sliceBounds[t].second;
		int outStartSlice = out->_sliceBounds[t].first;
		int   outEndSlice = out->_sliceBounds[t].second;
		int      inOffset =  in->_startEntry[  inStartSlice ];
		int     outOffset = out->_startEntry[ outStartSlice ];

		ConstPointer( T2 ) _in  =  in->_pSolution[t] -  inOffset;
		T2*                _out = out->_pSolution[t] - outOffset;

		for( int i=inStartSlice ; i<inEndSlice ; i++ )
		{
			int start = startEntry[i];
			int stop  = startEntry[i] + entriesPerSlice[i];
			for( int j=start ; j<stop ; j++ )
			{
				T2 temp = _in[ j ];
				T2* out = _out;
				const_iterator e = end( j );
				for( const_iterator iter = begin( j ) ; iter!=e ; iter++ ) out[ iter->N ] += temp * iter->Value;
			}
		}
	}
	out->_mergeSkirts( );
}


template< class T , class const_iterator >
template< class T2 , class T3 , bool UnitDiagonal , bool StrippedDiagonal , bool UseSOR >
void SparseMatrixInterface< T , const_iterator >::SolveGaussSeidelParallel(	ConstPointer( T3 ) Diagonal , ConstPointer( T2 ) b , int iters , ParallelSolution< T2 >* Solution , T sor ) const
{
	Solution->_synchronize();
	const int threads = Solution->_threads;
	const int sliceDependence = Solution->_sliceDependence;
	const int sliceCount = Solution->_sliceCount;
	const int* startEntry = &Solution->_startEntry[0];
	const int* entriesPerSlice = Solution->_entriesPerSlice;
#pragma omp parallel for num_threads( threads )
	for( int t=0 ; t<threads ; t++ )
	{
		// Compute the starting and ending slices associated with the current thread.
		int startSlice = Solution->_sliceBounds[t].first;
		int   endSlice = Solution->_sliceBounds[t].second;
		int     offset = Solution->_startEntry[ startSlice ];

		Pointer( T2 ) solution = Solution->_pSolution[t] - offset;

		// When processing at index i, we will be relaxing on slices:
		// [ s , s + sliceDependence , ... , s + (iters-1)*sliceDependence ]
		for( int s=startSlice-(iters-1)*sliceDependence ; s<endSlice ; s++ )
		{

			// When processing at index j, we will be relaxing slice:
			// s + i*sliceDepdence for the (iters-i)-th time.
			for( int i=iters-1 ; i>=0 ; i-- ) 
			{
				int ss = s + i*sliceDependence;
				// Compute the starting and ending bounds of the skirt on which we are relaxing (keeping in mind that we will be doing a residual calcuation)
				int startBound = startSlice - i*sliceDependence;
				int   endBound =   endSlice + i*sliceDependence+1;
				if( startBound<0 ) startBound = 0;
				if(   endBound>sliceCount ) endBound = sliceCount;
				if( ss>=startBound && ss<endBound )
				{
					int start = startEntry[ss];
					int  stop = startEntry[ss] + entriesPerSlice[ss];
					for( int jj=start ; jj<stop ; jj++ )
					{
						T2 temp;
						temp *= 0;
						Pointer( T2 ) _solution = solution;
						const_iterator e = end( jj );
						for( const_iterator iter = begin( jj ) ; iter!=e ; iter++ ) temp += _solution[ iter->N ] * iter->Value;
						if( UnitDiagonal )
						{
							if( UseSOR )
								if(StrippedDiagonal) solution[jj]  = solution[jj]*T(1.0-sor) + (b[jj]-temp) * sor;
								else                 solution[jj] +=                           (b[jj]-temp) * sor;
							else
								if(StrippedDiagonal) solution[jj]  = (b[jj]-temp);
								else                 solution[jj] += (b[jj]-temp);
						}
						else
						{
							T dValue = T(1.) / Diagonal[jj];
							if( UseSOR )
								if(StrippedDiagonal) solution[jj]  = solution[jj]*T(1.0-sor) + (b[jj]-temp) * dValue * sor;
								else                 solution[jj] +=                           (b[jj]-temp) * dValue * sor;
							else
								if(StrippedDiagonal) solution[jj]  = (b[jj]-temp) * dValue;
								else                 solution[jj] += (b[jj]-temp) * dValue;
						}
					}
				}
			}
		}
	}
}
template< class T , class const_iterator >
template< class T2 , class T3 , bool UnitDiagonal , bool StrippedDiagonal , bool UseSOR >
void SparseMatrixInterface< T , const_iterator >::SolveGaussSeidelAndResidualParallel( ConstPointer( T3 ) Diagonal , ConstPointer( T2 ) b , int iters , ParallelSolution< T2 >* Solution , Pointer( T2 ) residual , T sor ) const
{
	Solution->_synchronize();
	const int threads = Solution->_threads;
	const int sliceDependence = Solution->_sliceDependence;
	const int sliceCount = Solution->_sliceCount;
	const int* startEntry = &Solution->_startEntry[0];
	const int* entriesPerSlice = Solution->_entriesPerSlice;
#pragma omp parallel for num_threads( threads )
	for( int t=0 ; t<threads ; t++ )
	{
		// Compute the starting and ending slices associated with the current thread.
		int startSlice = Solution->_sliceBounds[t].first;
		int   endSlice = Solution->_sliceBounds[t].second;
		int     offset = Solution->_startEntry[ startSlice ];

		Pointer( T2 ) solution = Solution->_pSolution[t] - offset;

		// When processing at index i, we will be relaxing on slices:
		// [ s , s + sliceDependence , ... , s + (iters-1)*sliceDependence ]

		for( int s=startSlice-iters*sliceDependence ; s<endSlice+sliceDependence ; s++ )
		{
			// When processing at index j, we will be relaxing slice:
			// s + i*sliceDepdence for the (iters-i)-th time.
			for( int i=iters-1 ; i>=0 ; i-- ) 
			{
				int ss = s + i*sliceDependence;
				// Compute the starting and ending bounds of the skirt on which we are relaxing (keeping in mind that we will be doing a residual calcuation)
				int startBound = startSlice - (i+1)*sliceDependence;
				int   endBound =   endSlice + (i+1)*sliceDependence+1;
				if( startBound<0 ) startBound = 0;
				if(   endBound>sliceCount ) endBound = sliceCount;
				if( ss>=startBound && ss<endBound )
				{
					int start = startEntry[ss];
					int  stop = startEntry[ss] + entriesPerSlice[ss];

					for( int jj=start ; jj<stop ; jj++ )
					{
						T2 temp;
						temp *= 0;
						Pointer( T2 ) _solution = solution;
						const_iterator e = end( jj );
						for( const_iterator iter=begin( jj ) ; iter!=e ; iter++ ) temp += _solution[ iter->N ] * iter->Value;
						if( UnitDiagonal )
						{
							if( UseSOR )
								if(StrippedDiagonal) solution[jj]  = solution[jj]*T(1.0-sor) + (b[jj]-temp) * sor;
								else                 solution[jj] +=                           (b[jj]-temp) * sor;
							else
								if(StrippedDiagonal) solution[jj]  = (b[jj]-temp);
								else                 solution[jj] += (b[jj]-temp);
						}
						else
						{
							T dValue = T(1.) / Diagonal[jj];
							if( UseSOR )
								if(StrippedDiagonal) solution[jj]  = solution[jj]*T(1.0-sor) + (b[jj]-temp) * dValue * sor;
								else                 solution[jj] +=                           (b[jj]-temp) * dValue * sor;
							else
								if(StrippedDiagonal) solution[jj]  = (b[jj]-temp) * dValue;
								else                 solution[jj] += (b[jj]-temp) * dValue;
						}
					}
				}
			}
			int ss = s-sliceDependence;
			if( ss>=startSlice && ss<endSlice )
			{
				int start = startEntry[ss];
				int  stop = startEntry[ss] + entriesPerSlice[ss];
				for( int jj=start ; jj<stop ; jj++ )
				{
					T2 temp;
					temp *= 0;
					Pointer( T2 ) _solution = solution;
					const_iterator e = end( jj );
					for( const_iterator iter=begin( jj ) ; iter!=e ; iter++ ) temp += _solution[ iter->N ] * iter->Value;
					if( UnitDiagonal )
					{
						if( StrippedDiagonal ) residual[jj] = b[jj] - temp - solution[jj];
						else                   residual[jj] = b[jj] - temp;
					}
					else
					{
						T dValue = Diagonal[jj];
						if( StrippedDiagonal ) residual[jj] = b[jj] - temp - ( solution[jj] * dValue );
						else                   residual[jj] = b[jj] - temp;
					}
				}
			}
		}
	}
}
