/* -*- C++ -*-
Copyright (c) 2006, Michael Kazhdan and Matthew Bolitho
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

#include <float.h>

///////////////////
//  SparseMatrix //
///////////////////
///////////////////////////////////////
// SparseMatrix Methods and Memebers //
///////////////////////////////////////

template< class T , class IndexType >
SparseMatrix< T , IndexType >::SparseMatrix()
{
	_contiguousMemory = false;
	rowMajor=true;
	groups=0;
	groupSizes  = NullPointer< int >( );
	m_ppElements= NullPointer< Pointer( MatrixEntry< T , IndexType > ) >( );
}
template< class T , class IndexType >
bool SparseMatrix< T , IndexType >::copyColumnMajor( SparseMatrix< T , IndexType >& out , int inDim , int outDim )
{
	if(rowMajor && outDim!=groups)	return false;
	if(!rowMajor && inDim!=groups)	return false;
	out.Resize(inDim,false);
	if(rowMajor)
	{
		for(int i=0;i<groups;i++)
			for(int j=0;j<groupSizes[i];j++)
				if(0<=m_ppElements[i][j].N && m_ppElements[i][j].N<inDim)
					out.groupSizes[m_ppElements[i][j].N]++;

		for(int i=0;i<out.groups;i++)
		{
			int sz=out.groupSizes[i];
			out.groupSizes[i]=0;
			out.SetGroupSize(i,sz);
			out.groupSizes[i]=0;
		}
		for(int i=0;i<groups;i++)
			for(int j=0;j<groupSizes[i];j++)
			{
				int idx=m_ppElements[i][j].N;
				if(0<=idx && idx<inDim)
				{
					out.m_ppElements[idx][out.groupSizes[idx]].N=i;
					out.m_ppElements[idx][out.groupSizes[idx]].Value=m_ppElements[i][j].Value;
					out.groupSizes[idx]++;
				}
			}
	}
	else
		for(int i=0;i<groups;i++)
		{
			out.SetGroupSize(i,groupSizes[i]);
			memcpy(out.m_ppElements[i],m_ppElements[i],groupSizes[i]*sizeof(MatrixEntry< T , IndexType >));
		}
	return true;
}
template< class T , class IndexType >
bool SparseMatrix< T , IndexType >::copyRowMajor(SparseMatrix< T , IndexType >& out , int inDim , int outDim )
{
	if(rowMajor && outDim!=groups)	return false;
	if(!rowMajor && inDim!=groups)	return false;

	out.Resize(outDim,true);
	if(rowMajor)
		for(int i=0;i<groups;i++)
		{
			out.SetGroupSize(i,groupSizes[i]);
			memcpy(out.m_ppElements[i],m_ppElements[i],groupSizes[i]*sizeof(MatrixEntry< T , IndexType >));
		}
	else
	{
		for(int i=0;i<groups;i++)
			for(int j=0;j<groupSizes[i];j++)
				if(0<=m_ppElements[i][j].N && m_ppElements[i][j].N<outDim)
					out.groupSizes[m_ppElements[i][j].N]++;
		for(int i=0;i<out.groups;i++)
		{
			int sz=out.groupSizes[i];
			out.groupSizes[i]=0;
			out.SetGroupSize(i,sz);
			out.groupSizes[i]=0;
		}
		for(int i=0;i<groups;i++)
			for(int j=0;j<groupSizes[i];j++)
			{
				int idx=m_ppElements[i][j].N;
				if(0<=idx && idx<outDim)
				{
					out.m_ppElements[idx][out.groupSizes[idx]].N=i;
					out.m_ppElements[idx][out.groupSizes[idx]].Value=m_ppElements[i][j].Value;
					out.groupSizes[idx]++;
				}
			}
	}
	return true;
}

template< class T , class IndexType >
SparseMatrix< T , IndexType >::SparseMatrix( int groups , bool rowMajor )
{
	_contiguousMemory = false;
	rowMajor=true;
	groups=0;
	groupSizes=NULL;
	m_ppElements=NULL;
	Resize( groups , rowMajor );
}

template< class T , class IndexType >
SparseMatrix< T , IndexType >::SparseMatrix( const SparseMatrix& M )
{
	_contiguousMemory = false;
	rowMajor=true;
	groups=0;
	groupSizes = NullPointer< int >( );
	m_ppElements = NullPointer< Pointer( MatrixEntry< T , IndexType > ) >( );
	Resize(M.groups,M.rowMajor);
	for (int i=0; i<groups; i++)
	{
		SetGroupSize(i,M.groupSizes[i]);
		for(int j=0;j<groupSizes[i];j++){m_ppElements[i][j]=M.m_ppElements[i][j];}
	}
}
template< class T , class IndexType >
template< class T2 , class IndexType2 >
SparseMatrix< T , IndexType >::SparseMatrix( const SparseMatrix< T2 , IndexType2 >& M )
{
	_contiguousMemory = false;
	rowMajor = true;
	groups = 0;
	groupSizes = NULL;
	m_ppElements = NULL;
	Resize( M.groups , M.rowMajor );
	for( int i=0 ; i<groups ; i++ )
	{
		SetGroupSize( i , M.groupSizes[i] );
		for( int j=0 ; j<groupSizes[i] ; j++ ) m_ppElements[i][j] = MatrixEntry< T , IndexType >( M.m_ppElements[i][j].N , T( M.m_ppElements[i][j].Value ) );
	}
}

template< class T , class IndexType >
double SparseMatrix< T , IndexType >::SquareNorm(void)
{
	double l2=0;
	for(int i=0;i<groups;i++)
		for(int j=0;j<groupSizes[i];j++)
			l2+=m_ppElements[i][j].Value*m_ppElements[i][j].Value;
	return l2;
}
template< class T , class IndexType >
double SparseMatrix< T , IndexType >::ASymmetricSquareNorm(void)
{
	double l2=0;
	for(int i=0;i<groups;i++)
		for(int j=0;j<groupSizes[i];j++)
		{
			double t1=0,t2=0;
			int N=m_ppElements[i][j].N;
			if(N==i)	continue;
			t1=m_ppElements[i][j].Value;
			for(int k=0;k<groupSizes[N];k++)
				if(m_ppElements[N][k].N==i)
				{
					t2=m_ppElements[N][k].Value;
					break;
				}
			l2+=(t1-t2)*(t1-t2);
		}
	return l2;
}
template< class T , class IndexType >
template< class T2 , class IndexType2 >
SparseMatrix< T , IndexType >& SparseMatrix< T , IndexType >::copy( const SparseMatrix< T2 , IndexType2 >& M )
{
	Resize( M.groups , M.rowMajor );
	for ( int i=0 ; i<groups ; i++)
	{
		SetGroupSize( i , M.groupSizes[i] );
		for( int j=0 ; j<groupSizes[i] ; j++ )
		{
			int idx = M.m_ppElements[i][j].N;
			m_ppElements[i][j] = MatrixEntry< T , IndexType >( idx , T( M[i][j].Value ) );
		}
	}
	return *this;
}

template< class T , class IndexType >
SparseMatrix< T , IndexType >& SparseMatrix< T , IndexType >::operator = ( const SparseMatrix< T , IndexType >& M )
{
	Resize(M.groups,M.rowMajor);
	for (int i=0; i<groups; i++){
		SetGroupSize(i,M.groupSizes[i]);
		for(int j=0;j<groupSizes[i];j++){m_ppElements[i][j]=M.m_ppElements[i][j];}
	}
	return *this;
}
template< class T , class IndexType >
template< class T2 , class IndexType2 >
SparseMatrix< T , IndexType >& SparseMatrix< T , IndexType >::operator = (const SparseMatrix< T2 , IndexType2 >& M)
{
	Resize( M.groups , M.rowMajor );
	for ( int i=0 ; i<groups ; i++)
	{
		SetGroupSize( i , M.groupSizes[i] );
		for( int j=0 ; j<groupSizes[i] ; j++ ) m_ppElements[i][j] = MatrixEntry< T , IndexType >( M.m_ppElements[i][j].N , T( M.m_ppElements[i][j].Value ) );
	}
	return *this;
}
template< class T , class IndexType >
template< class T2 >
Vector< T2 > SparseMatrix< T , IndexType >::operator * ( const Vector< T2 >& V ) const
{
	Vector< T2 > R( Rows() );
	Interface::Multiply( V , R );
	return R;
}


template< class T , class IndexType >
void SparseMatrix< T , IndexType >::Transpose(void){rowMajor=!rowMajor;}

template< class T , class IndexType >
SparseMatrix< T , IndexType >::~SparseMatrix()
{
	Resize( 0 , true );
}
template< class T , class IndexType >
bool SparseMatrix< T , IndexType >::isContiguous( void ) const { return _contiguousMemory; }
template< class T , class IndexType >
void SparseMatrix< T , IndexType >::MakeContiguous( void )
{
	if( _contiguousMemory ) return;
	Pointer( MatrixEntry< T , IndexType > ) entries = AllocPointer< MatrixEntry< T , IndexType > >( Entries() );

	for( int i=0 ; i<groups ; i++ )
	{
		memcpy( entries , m_ppElements[i] , sizeof( MatrixEntry< T , IndexType > ) * groupSizes[i] );
		FreePointer( m_ppElements[i] );
		m_ppElements[i] = entries;
		entries += groupSizes[i];
	}
	_contiguousMemory = true;
}

template< class T , class IndexType >
void SparseMatrix< T , IndexType >::Resize( int g, bool rm )
{
	rowMajor=rm;
	if( groups>0 )
	{
		if( !_contiguousMemory ){ for( int i=0 ; i<groups ; i++ ) FreePointer( m_ppElements[i] ); }
		else FreePointer( m_ppElements[0] );
		FreePointer( m_ppElements );
		FreePointer( groupSizes );
	}
	_contiguousMemory = false;
	groups = g;
	if( g )
	{
		groupSizes = AllocPointer< int >( g );
		memset( groupSizes , 0 , sizeof(int)*g );
		m_ppElements = AllocPointer< Pointer( MatrixEntry< T , IndexType > ) >( g );
		for( int i=0 ; i<g ; i++ ) m_ppElements[i] = NullPointer< MatrixEntry< T , IndexType > >( );
	}
}

template< class T , class IndexType >
void SparseMatrix< T , IndexType >::SetGroupSize(int group,int count)
{
	if( _contiguousMemory )
	{
		fprintf( stderr , "Cannot set row/column size in contiguous-memory mode\n" );
		return;
	}
	if(group>=0 && group<groups)
	{
		FreePointer( m_ppElements[group] );
		if(count>0)
		{
			m_ppElements[group] = AllocPointer< MatrixEntry< T , IndexType > >( count );
			memset( m_ppElements[group] , 0 , sizeof(MatrixEntry< T , IndexType >)*count );
		}
		groupSizes[group]=count;
	}
}

template< class T , class IndexType >
template< class A_T , class A_const_iterator , class B_T , class B_const_iterator >
bool SparseMatrix< T , IndexType >::Multiply( const SparseMatrixInterface< A_T , A_const_iterator >& A , const SparseMatrixInterface< B_T , B_const_iterator >& B , SparseMatrix< T , IndexType >& out , int threads )
{
	int aCols = 0 , aRows = A.Rows();
	int bCols = 0 , bRows = B.Rows();
	for( int i=0 ; i<A.Rows() ; i++ ) for( A_const_iterator iter=A.begin(i) ; iter!=A.end(i) ; iter++ ) if( aCols<iter->N ) aCols = iter->N+1;
	for( int i=0 ; i<B.Rows() ; i++ ) for( B_const_iterator iter=B.begin(i) ; iter!=B.end(i) ; iter++ ) if( bCols<iter->N ) bCols = iter->N+1;
	if( bRows<aCols )
	{
		fprintf( stderr , "[Error] Matrix sizes do not support multiplication %d x %d * %d x %d\n" , aRows , aCols , bRows , bCols );
		return false;
	}

	out.Resize( aRows );
#pragma omp parallel for num_threads( threads )
	for( int i=0 ; i<aRows ; i++ )
	{
		hash_map< IndexType , T > row;
		for( A_const_iterator iterA=A.begin(i) ; iterA!=A.end(i) ; iterA++ )
		{
			IndexType idx1 = iterA->N;
if( idx1==-1 ) continue;
			A_T AValue = iterA->Value;
if( AValue==A_T(0.) ) continue;
			if( idx1<0 ) continue;
			for( B_const_iterator iterB=B.begin(idx1) ; iterB!=B.end(idx1) ; iterB++ )
			{
				IndexType idx2 = iterB->N;
if( idx2==-1 ) continue;
				B_T BValue = iterB->Value;
if( BValue==B_T(0.) ) continue;
				T temp = T( BValue * AValue ); // temp = A( i , idx1 ) * B( idx1 , idx2 )
				typename hash_map< IndexType , T >::iterator iter = row.find(idx2);
				if( iter==row.end() ) row[idx2] = temp;
				else iter->second += temp;
			}
		}
		out.SetGroupSize( i , row.size() );
		out.groupSizes[i] = 0;
		for( typename hash_map< IndexType , T >::iterator iter=row.begin() ; iter!=row.end() ; iter++ )
			out[i][ out.groupSizes[i]++ ] = MatrixEntry< T , IndexType >( iter->first , iter->second );
	}
	return true;
}
template< class T , class IndexType >
template< class A_T , class A_const_iterator , class B_T , class B_const_iterator >
bool SparseMatrix< T , IndexType >::Multiply( const SparseMatrixInterface< A_T , A_const_iterator >& A , const Vector< A_T >& D , const SparseMatrixInterface< B_T , B_const_iterator >& B , SparseMatrix< T , IndexType >& out , int threads )
{
	int aCols = 0 , aRows = A.Rows();
	int bCols = 0 , bRows = B.Rows();
	for( int i=0 ; i<A.Rows() ; i++ ) for( A_const_iterator iter=A.begin(i) ; iter!=A.end(i) ; iter++ ) if( aCols<iter->N ) aCols = iter->N+1;
	for( int i=0 ; i<B.Rows() ; i++ ) for( B_const_iterator iter=B.begin(i) ; iter!=B.end(i) ; iter++ ) if( bCols<iter->N ) bCols = iter->N+1;
	if( aCols<D.Dimensions() ) aCols = D.Dimensions();
	if( bRows<aCols )
	{
		fprintf( stderr , "[Error] Matrix sizes do not support multiplication %d x %d * %d x %d\n" , aRows , aCols , bRows , bCols );
		return false;
	}
	if( D.Dimensions()!=aRows )
	{
		fprintf( stderr , "[Error] Diagonal dimension doesn't match: %d x %d * %d x %d * %d x %d\n" ,  aRows , aCols ,D.Dimensions() , D.Dimensions() , bRows , bCols );
		return false;
	}

	out.Resize( aRows );
#pragma omp parallel for num_threads( threads )
	for( int i=0 ; i<aRows ; i++ )
	{
		hash_map< IndexType , T > row;
		for( A_const_iterator iterA=A.begin(i) ; iterA!=A.end(i) ; iterA++ )
		{
			IndexType idx1 = iterA->N;
if( idx1==-1 ) continue;
			A_T AValue = iterA->Value;
if( AValue==A_T(0.) ) continue;
			if( idx1<0 ) continue;
			for( B_const_iterator iterB=B.begin(idx1) ; iterB!=B.end(idx1) ; iterB++ )
			{
				IndexType idx2 = iterB->N;
if( idx2==-1 ) continue;
				B_T BValue = iterB->Value;
if( BValue==B_T(0.) ) continue;
				T temp = T( BValue * AValue ); // temp = A( i , idx1 ) * B( idx1 , idx2 )
				typename hash_map< IndexType , T >::iterator iter = row.find(idx2);
				if( iter==row.end() ) row[idx2] = temp;
				else iter->second += temp;
			}
		}
		{
			IndexType idx1 = i;
if( idx1==-1 ) continue;
			A_T AValue = D[i];
if( AValue==A_T(0.) ) continue;
			if( idx1<0 ) continue;
			for( B_const_iterator iterB=B.begin(idx1) ; iterB!=B.end(idx1) ; iterB++ )
			{
				IndexType idx2 = iterB->N;
if( idx2==-1 ) continue;
				B_T BValue = iterB->Value;
if( BValue==B_T(0.) ) continue;
				T temp = T( BValue * AValue ); // temp = A( i , idx1 ) * B( idx1 , idx2 )
				typename hash_map< IndexType , T >::iterator iter = row.find(idx2);
				if( iter==row.end() ) row[idx2] = temp;
				else iter->second += temp;
			}
		}
		out.SetGroupSize( i , row.size() );
		out.groupSizes[i] = 0;
		for( typename hash_map< IndexType , T >::iterator iter=row.begin() ; iter!=row.end() ; iter++ )
			out[i][ out.groupSizes[i]++ ] = MatrixEntry< T , IndexType >( iter->first , iter->second );
	}
	return true;
}
template< class T , class IndexType >
template< class In_const_iterator >
bool SparseMatrix< T , IndexType >::Transpose( const SparseMatrixInterface< T , In_const_iterator >& At , SparseMatrix< T , IndexType >& A , T (*TransposeFunction)( const T& ) )
{
	int aRows = 0 , aCols = At.Rows();
	for( int i=0 ; i<At.Rows() ; i++ ) for( In_const_iterator iter=At.begin(i) ; iter!=At.end(i) ; iter++ ) if( aRows<=iter->N ) aRows = iter->N+1;

	A.Resize( aRows );
	for( int i=0 ; i<aRows ; i++ ) A.groupSizes[i] = 0;
//	for( int i=0 ; i<At.Rows() ; i++ ) for( In_const_iterator iter=At.begin(i) ; iter!=At.end(i) ; iter++ ) A.groupSizes[ iter->N ]++;
	for( int i=0 ; i<At.Rows() ; i++ ) for( In_const_iterator iter=At.begin(i) ; iter!=At.end(i) ; iter++ ) if( iter->N!=-1 ) A.groupSizes[ iter->N ]++;
	for( int i=0 ; i<A.groups ; i++ )
	{
		int t = A.groupSizes[i];
		A.groupSizes[i] = 0;
		A.SetGroupSize( i , t );
		A.groupSizes[i] = 0;
	}
	if( TransposeFunction )
		for( int i=0 ; i<At.Rows() ; i++ ) for( In_const_iterator iter=At.begin(i) ; iter!=At.end(i) ; iter++ )
		{
			int ii = iter->N;
			if( ii==-1 ) continue;
			A[ii][ A.groupSizes[ii]++ ] = MatrixEntry< T , IndexType >( i , TransposeFunction( iter->Value ) );
		}
	else
		for( int i=0 ; i<At.Rows() ; i++ ) for( In_const_iterator iter=At.begin(i) ; iter!=At.end(i) ; iter++ )
		{
			int ii = iter->N;
			if( ii==-1 ) continue;
			A[ii][ A.groupSizes[ii]++ ] = MatrixEntry< T , IndexType >( i , iter->Value );
		}
	return true;
}
template< class T , class IndexType >
template< class In_const_iterator >
bool SparseMatrix< T , IndexType >::Transpose( const SparseMatrixInterface< T , In_const_iterator >& At , SparseMatrix< T , IndexType >& A , int aRows , T (*TransposeFunction)( const T& ) )
{
	int _aRows = 0 , aCols = At.Rows();
	for( int i=0 ; i<At.Rows() ; i++ ) for( In_const_iterator iter=At.begin(i) ; iter!=At.end(i) ; iter++ ) if( aCols<=iter->N ) _aRows = iter->N+1;
	if( _aRows>aRows )
	{
		fprintf( stderr , "[Error] prescribed output dimension too low: %d < %d\n" , aRows , _aRows );
		return false;
	}

	A.Resize( aRows );
	for( int i=0 ; i<aRows ; i++ ) A.groupSizes[i] = 0;
	for( int i=0 ; i<At.Rows() ; i++ ) for( In_const_iterator iter=At.begin(i) ; iter!=At.end(i) ; iter++ ) A.rowSizes[ iter->N ]++;
	for( int i=0 ; i<A.rows ; i++ )
	{
		int t = A.groupSizes[i];
		A.groupSizes[i] = 0;
		A.SetGroupSize( i , t );
		A.groupSizes[i] = 0;
	}
	if( TransposeFunction )
		for( int i=0 ; i<At.Rows() ; i++ ) for( In_const_iterator iter=At.begin(i) ; iter!=At.end(i) ; iter++ )
		{
			int ii = iter->N;
			A[ii][ A.groupSizes[ii]++ ] = MatrixEntry< T , IndexType >( i , TransposeFunction( iter->Value ) );
		}
	else
		for( int i=0 ; i<At.Rows() ; i++ ) for( In_const_iterator iter=At.begin(i) ; iter!=At.end(i) ; iter++ )
		{
			int ii = iter->N;
			A[ii][ A.groupSizes[ii]++ ] = MatrixEntry< T , IndexType >( i , iter->Value );
		}
	return true;
}

#define DUMP_OUTPUT 0
template <class Matrix,class Data>
static int SolveConjugateGradient(const Matrix& SPD,const Vector<Data>& b,const int& iters,Vector<Data>& solution,const double eps)
{
	Vector<Data> d,r,Md,temp;
	double alpha,beta,rDotR,oldRDotR;
	Md.Resize(b.Dimensions());

	temp.Resize(b.Dimensions());
	SPD.Multiply(solution,temp);
	d=r=b-temp;
	oldRDotR=rDotR=r.Dot(r);
	if(b.Dot(b)<=eps)
	{
#if DUMP_OUTPUT
		printf("Badness0: %g %g\n",r.Dot(r),eps);
#endif // DUMP_OUTPUT
		solution.SetZero();
		return 0;
	}
	int i;
	for(i=0;i<iters;i++)
	{
		double temp;
		SPD.Multiply(d,Md);
		temp=d.Dot(Md);
		if(temp<=eps)
		{
#if DUMP_OUTPUT
			printf("Badness1: %g %g\n",temp,eps);
#endif // DUMP_OUTPUT
			break;
		}
		alpha=rDotR/temp;
		r.SubtractScaled(Md,alpha);
		temp=r.Dot(r);
		// BADNESS!!! How can the size of the residual increase?
		if(temp>2*oldRDotR)
		{
#if DUMP_OUTPUT
			printf("Badness1.5: %g %g\n",temp,oldRDotR);
#endif // DUMP_OUTPUT
			break;
		}
		oldRDotR=rDotR;
		if(temp/b.Dot(b)<=eps)
		{
#if DUMP_OUTPUT
//			printf("Badness2: %g %g\n",temp,eps);
#endif // DUMP_OUTPUT
//			break;
		}
		beta=temp/rDotR;
		solution.AddScaled(d,alpha);
		if(beta<=eps)
		{
#if DUMP_OUTPUT
			printf("Badness3: %g %g\n",beta,eps);
#endif // DUMP_OUTPUT
			break;
		}
		rDotR=temp;
		Vector<Data>::Add(d,beta,r,d);
	}
	return i;
}
template <class Matrix,class IPS,class Real>
static int SolveConjugateGradient(const Matrix& SPD,const Vector<IPS>& b,const int& iters,Vector<IPS>& solution,const double eps)
{
	Vector<IPS> d,r,Md,temp;
	double alpha,beta,rDotR,oldRDotR;
	Md.Resize(b.Dimensions());

	temp.Resize(b.Dimensions());
	SPD.Multiply(solution,temp);
	d=r=b-temp;
	oldRDotR=rDotR=r.IPSDot(r);
	if(b.IPSDot(b)<=eps)
	{
#if DUMP_OUTPUT
		printf("Badness0: %g %g\n",r.Dot(r),eps);
#endif // DUMP_OUTPUT
		solution.SetZero();
		return 0;
	}
	int i;
	for(i=0;i<iters;i++)
	{
		double temp;
		SPD.Multiply(d,Md);
		temp=d.IPSDot(Md);
		if(temp<=eps)
		{
#if DUMP_OUTPUT
			printf("Badness1: %g %g\n",temp,eps);
#endif // DUMP_OUTPUT
			break;
		}
		alpha=rDotR/temp;
		r.SubtractScaled(Md,alpha);
		temp=r.IPSDot(r);
		// BADNESS!!! How can the size of the residual increase?
		if(temp>2*oldRDotR)
		{
#if DUMP_OUTPUT
			printf("Badness1.5: %g %g\n",temp,oldRDotR);
#endif // DUMP_OUTPUT
			break;
		}
		oldRDotR=rDotR;
		if(temp/b.IPSDot(b)<=eps)
		{
#if DUMP_OUTPUT
//			printf("Badness2: %g %g\n",temp,eps);
#endif // DUMP_OUTPUT
//			break;
		}
		beta=temp/rDotR;
		solution.AddScaled(d,alpha);
		if(beta<=eps)
		{
#if DUMP_OUTPUT
			printf("Badness3: %g %g\n",beta,eps);
#endif // DUMP_OUTPUT
			break;
		}
		rDotR=temp;
		Vector<IPS>::Add(d,beta,r,d);
	}
	return i;
}


template <class Matrix,class IPS,class Real>
static int SolveConjugateGradient2(const Matrix& SPD,const Vector<IPS>& b,const int& iters,Vector<IPS>& x,const double eps)
{
	Vector<IPS> q,d,r;
	double delta_new,delta_old,delta_0,alpha,beta;
	q.Resize(b.Dimensions());
	SPD.Multiply(x,q);
	d=r=b-q;
	delta_0=delta_new=r.IPSDot(r);
printf("%f %f\n",x.IPSDot(x),delta_0);
	int i;
	for(i=0;i<iters && delta_new>eps*eps*delta_0;i++)
	{
		SPD.Multiply(d,q);
		alpha=delta_new/(d.IPSDot/*<double>*/(q));
printf("\t%d] %f\n",i,d.IPSDot(q));
printf("\t\talpha = %f\n",alpha);
		x.AddScaled(d,alpha);
		if(!(i%50))
		{
			SPD.Multiply(x,q);
			r=b-q;
		}
		else	r.SubtractScaled(q,alpha);
		delta_old=delta_new;
		delta_new=r.IPSDot/*<double>*/(r);
		beta=delta_new/delta_old;
printf("\t\t beta = %f\n",beta);
printf("\t\tresid = %f\n",delta_new/delta_0);
		Vector<IPS>::Add(d,beta,r,d);
	}
printf("Iters: %d / %d\n",i,iters);
//exit(0);
	return i;
}
template< class MType , class IndexType , class VType >
int SolveConjugateGradient( const SparseMatrix< MType , IndexType >& A , const Vector<VType>& b , const int& iters , Vector<VType>& x ,
						   Vector<VType> (*Multiply)(const SparseMatrix< MType , IndexType >& , const Vector<VType>& ) )
{
	double eps=1e-16;
	Vector<VType> r = b - Multiply(A,x);
	Vector<VType> d = r;
	double delta_new = r.Dot(r);
	double delta_0 = delta_new;
	int i;
	for(i=0; i<iters && delta_new>eps*delta_0 ;i++)
	{
		Vector<VType> q = Multiply(A,d);
		double alpha = delta_new / d.Dot(q);
		x = x + d*alpha;
		if( !(i%50) )	r = b - Multiply(A,x);
		else			r = r - q*alpha;

		double delta_old = delta_new;
		delta_new = r.Dot(r);
		double beta = delta_new / delta_old;
		d = r + d*beta;
	}
	return i;
}


