/*
Copyright (c) 2009, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#include "Ply.h"

///////////////
// RangeGrid //
///////////////
template<class Vertex,class Real>
bool RangeGrid<Vertex,Real>::Read( char* rangeGridFile , char* xFormFile )
{
	int fileType;
	vertices.clear();
	if( !PlyReadGrid( rangeGridFile , vertices , grid,Vertex::ReadProperties , Vertex::ReadComponents , fileType ) ) return false;
	if( xFormFile )
	{
		FILE* fp=fopen(xFormFile,"r");
		if(!fp)
		{
			fprintf(stderr,"Failed to open transform file: %s\n",xFormFile);
			return false;
		}
		for(int j=0;j<4;j++)
			for(int i=0;i<4;i++)
			{
				float d;
				if(fscanf(fp," %f ", &d)!=1)
				{
					fprintf(stderr,"Failed to read matrix coordinate: (%d, %d)\n",i,j);
					return false;
				}
				xForm.coords[i][j] = d;
			}
			fclose(fp);
	}
	else	xForm=XForm4x4<Real>::Identity();
	for(int i=0;i<vertices.size();i++)	vertices[i]=vertices[i].xForm(xForm);
	return true;
}
template<class Vertex,class Real>
size_t RangeGrid<Vertex,Real>::width() const	{return grid.width();}
template<class Vertex,class Real>
size_t RangeGrid<Vertex,Real>::height() const	{return grid.height();}
template<class Vertex,class Real>
int &RangeGrid<Vertex,Real>::operator ()		(size_t i,size_t j)			{return grid(int(i),int(j));}
template<class Vertex,class Real>
const int &RangeGrid<Vertex,Real>::operator ()	(size_t i,size_t j)	const	{return grid(int(i),int(j));}
template<class Vertex,class Real>
Vertex& RangeGrid<Vertex,Real>::operator []			(size_t i)				{return vertices[i];}
template<class Vertex,class Real>
const Vertex& RangeGrid<Vertex,Real>::operator []	(size_t i)	const		{return vertices[i];}
template<class Vertex,class Real>
XForm4x4<Real> RangeGrid<Vertex,Real>::backProjectionXForm(bool full) const
{
	////////////////////////////////////////////////////////////////////////////////////////////////
	// Get the matrix that best maps the points in the camera's coordinate frame to the scan grid //
	////////////////////////////////////////////////////////////////////////////////////////////////
	Point2D<Real> start,dX,dY;
	SquareMatrix<double,6> mat,iMat;
	Point<double,6> vec,iVec;
	Grid<Real> myMatrix;
	std::vector<Real> myVector;
	XForm4x4<Real> iXForm = xForm.inverse();
	int count=0;

	// Get the number of valid grid points
	for( int i=0 ; i<grid.width() ; i++ ) for( int j=0 ; j<grid.height() ; j++ ) if( grid(i,j)>=0 )	count++;

	// The matrix itself has six degrees of freedom -- the 2x2 affine transformation + the translation
	// Each vertex specifices two constraints -- the desired x and the desired y positions
	myMatrix.resize(6,count*2);
	myVector.resize(  count*2);
	count=0;
	for(int i=0;i<grid.width();i++)
		for(int j=0;j<grid.height();j++)
			if(grid(i,j)>=0)
			{
				int idx=grid(i,j);
				for( int c=0 ; c<2 ; c++ )
				{
					myMatrix( c+0 , count*2+c ) = Real( 1.);
					myMatrix( c+2 , count*2+c ) = Real( i );
					myMatrix( c+4 , count*2+c ) = Real( j );
					Point3D<float> p=iXForm( vertices[idx].point );
					myVector[count*2+c] = p.coords[c]/p.coords[2]; // The point in the camera plane to which a grid point gets mapped
				}
				count++;
			}

			// Set up the normal equation
			for(int i=0;i<6;i++)
				for(int j=0;j<6;j++)
					for(int k=0;k<count*2;k++)
						mat.coords[i][j]+=myMatrix(i,k)*myMatrix(j,k);
			for(int j=0;j<6;j++)
				for(int k=0;k<count*2;k++)
					vec.coords[j]+=myMatrix(j,k)*myVector[k];

			// Solve the normal equation
			iMat=mat.inverse();
			iVec=iMat*vec;

			for(int c=0;c<2;c++)
			{
				start.coords[c] = Real( iVec.coords[c+0] );
				dX.coords[c]    = Real( iVec.coords[c+2] );
				dY.coords[c]    = Real( iVec.coords[c+4] );
			}

			XForm4x4<Real> globalToLocal,coordinatesToTwoD;

			globalToLocal=xForm.inverse();
			coordinatesToTwoD=XForm4x4<Real>::Identity();

			for(int i=0;i<2;i++)
			{
				coordinatesToTwoD.coords[0][i]=dX.coords[i];
				coordinatesToTwoD.coords[1][i]=dY.coords[i];
				coordinatesToTwoD.coords[2][i]=start.coords[i];
			}
			if(full)	return coordinatesToTwoD.inverse()*globalToLocal;
			else		return coordinatesToTwoD.inverse();
}
template<class Vertex,class Real>
typename RangeGrid<Vertex,Real>::Sample RangeGrid<Vertex,Real>::sample(float x,float y)	const	{	return sample(double(x),double(y));	}
template<class Vertex,class Real>
typename RangeGrid<Vertex,Real>::Sample RangeGrid<Vertex,Real>::sample(double x,double y)	const
{
	Sample s;
	int x1 = (int) floor(x);
	int x2 = x1+1;
	int y1 = (int) floor(y);
	int y2 = y1+1;
	double dx=x-x1;
	double dy=y-y1;
	if( x1<0 || x1>=grid.width() || x2<0 || x2>=grid.width() || y1<0 || y1>=grid.height() || y2<0 || y2>=grid.height() ||
		grid(x1,y1)<0 || grid(x2,y2)<0 || grid(x1,y2)<0 || grid(x2,y1)<0 )
	{
		s.validSample=false;
		return s;
	}
	s.validSample=true;
	s.v =
		vertices[ grid( x1 , y1 ) ] * float( (1.0-dx)*(1.0-dy) ) +
		vertices[ grid( x1 , y2 ) ] * float( (1.0-dx)*(    dy) ) +
		vertices[ grid( x2 , y2 ) ] * float( (    dx)*(    dy) ) +
		vertices[ grid( x2 , y1 ) ] * float( (    dx)*(1.0-dy) );
	return s;
}
template<class Vertex,class Real>
Vertex RangeGrid<Vertex,Real>::cumulativeVertex(int& count)
{
	Vertex v;
	count=0;
	for(int i=0;i<grid.width();i++)
		for(int j=0;j<grid.height();j++)
			if(grid(i,j)>=0)
			{
				count++;
				v+=vertices[grid(i,j)];
			}
			return v;
}
