/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

/*
 * Construct the octree by intializing with either points or triangles.
 * This should initialize the tree and allocate the point samples used for integration.
 * Using the point samples, you should be able to compute the tracking information.
 * From either the point samples or the tracking point samples, you should be able to compute integral values.
 * From the integral values you should be able to construct the system matrices.
 */
/*
 * To Do:
 * 1] Fit mesh into subset of the cube and start solver at depth>0 to ensure partition of unity
 * 2] Make sure that Primal/Dual Float/Double all work.
*/
#ifndef POISSON_MESH_DATA_INCLUDED
#define POISSON_MESH_DATA_INCLUDED

#define FAST_QUADRATURE 1		// If this is enabled, we use a lower-degree quadrature than that dictated
								// by the the degree of the tensor product b-spline
#define SUPER_FAST_QUADRATURE 0	// If this is enabled, and even more liberal quadrature is used
								// This doesn't work terribly well for long-range MCF
#define USE_SSE_CODE 1
#define USE_ATOMICS 0
#define KEEP_MANIFOLD 0
#define MAJOR_CLEAN_UP 1

#include <vector>
#include <emmintrin.h>
#include "FunctionData.h"
#include "Octree.h"
#include "FastPoint3D.h"
#include "Multigrid.h"

#if USE_ATOMICS
inline void AtomicIncrement( float* ptr , float addend )
{
	int oldInt32 , newInt32;
	do
	{
		float oldValue = *ptr;
		float newValue = oldValue + addend;
		oldInt32 = *(int*)(&oldValue);
		newInt32 = *(int*)(&newValue);
	}
	while( InterlockedCompareExchange( (LONG*)ptr , newInt32 , oldInt32 )!=oldInt32 );
}
inline void AtomicIncrement( double* ptr , double addend )
{
	long long oldInt64 , newInt64;
	do
	{
		double oldValue = *ptr;
		double newValue = oldValue + addend;
		oldInt64 = *(long long*)(&oldValue);
		newInt64 = *(long long*)(&newValue);
	}
	while( InterlockedCompareExchange64( (LONGLONG*)ptr , newInt64 , oldInt64 )!=oldInt64 );
}
#else // !USE_ATOMICS
#define AtomicIncrement( ptr , addend ) ( *ptr += addend )
#endif // USE_ATOMICS


/************************************************
 * This datastructure is used to represent the
 * polygon that is obtained by repeatedly
 * clipping a triangle to x-, y-, and z- planes.
 ************************************************/
class PolygonData
{
public:
	static const int MAX_VERTICES = 9;
	int vertices[ MAX_VERTICES ];
	int count;
	int parent;
	const int& operator[] ( int idx ) const { return vertices[idx]; }
	int& operator[] ( int idx ) { return vertices[idx]; }
	PolygonData( void ) { count = 0 , parent = -1; }
};

template< class Real >
class MeshTreeNodeData
{
public:
	int tStart , tEnd;			// Start and end index of associated samples/triangles
	int geometryIndex;			// Index of the node within the array of geometry nodes
	int basisIndex;				// Index of the node within the array of basis nodes
	int tempIndex;				// Scratch space for processing -- not guaranteed to be preserved across function calls
	MeshTreeNodeData( void );
	~MeshTreeNodeData( void );
};
template< class Real , int size > inline Real Dot( const Real* v1 , const Real* v2 );
template< class Real , int size > inline void AddScaled( Real* out , const Real* in , Real s );


template< class Real , bool Primal >
class MeshOctree
{
	template< class _Real , bool _Primal > friend class MeshOctree;
public:
	/*******************************************
	/* Classes related to defining an FEM solver
	********************************************/
	template< class C >
	struct ElementVector;
	struct ElementMatrix;

	template< bool UnitEntries >
	struct ElementToBasisOperator;
	struct ElementRestrictionOperator;

	template< class C >
	struct ParallelSolver : public ParallelMultigridSolver< Real , C , int >
	{
		std::vector< ElementToBasisOperator< true > > e2b;		// Operators that transform element-based systems to function-based systems.
		std::vector< ElementRestrictionOperator > restriction;	// Operators that take high, element-based, systems to coarser ones.

		ElementMatrix& eM( void ) { return _eM.back(); }

		void Init( int blockSize , int dimensions , bool setDSFromElements );
		void ResetMatrix( int threads );						// Sets the function-based system from the element matrix
	private:
		std::vector< ElementMatrix > _eM;
	};

	struct MatrixIntegrationSample;

	template< class Vertex >
	struct TrackingData
	{
		enum
		{
			FLOW_LAPLACIAN,
			FLOW_CONFORMAL_LAPLACIAN,
			FLOW_AUTHALIC_LAPLACIAN
		};
		std::vector< int > sampleStart;
		std::vector< std::pair< Point3D< Real > , Point3D< Real > > > tangents;
		std::vector< MatrixIntegrationSample > mSamples;

		template< class C >
		void setConstrainedLaplacianAndVector( const ElementVector< Vertex >& embedding , const ElementVector< C >& coefficients , ElementMatrix& eM , typename ElementVector< C >& eV , Real dotM , Real lapM , Real dotV , Real lapV , int flowType , int threads ) const;
		template< class C >
		void setMatricesAndVectors( const ElementVector< Vertex >& embedding , const ElementVector< C >& coefficients , ElementMatrix& dotM , ElementMatrix& lapM , typename ElementVector< C >& dotV , typename ElementVector< C >& lapV , int flowType , int threads ) const;	
	};

	/**********************************************************
	/* Sets up the octree.
	/* Splits up the triangles to the nodes.
	/* Adds additional nodes for indexing basis/test functions.
	***********************************************************/
	template< class Vertex >
	int setTree( const std::vector< Vertex >& vertices , const std::vector< TriangleIndex >& triangles , int depth , double cutOff , bool progress , int threads );

	/********************************************************************
	 * Computes the system matrices (with respect to the basis functions)
	 ********************************************************************/
	int getDotProductMatrix          ( SparseMatrix< Real , int >& D                                   , bool progress ) const;
	int getLaplacianMatrix           ( SparseMatrix< Real , int >& L                                   , bool progress ) const;
	int getConstrainedLaplacianMatrix( SparseMatrix< Real , int >& M , Real dotWeight , Real lapWeight , bool progress ) const;

	/**********************************************************************
	 * Computes the system matrices (with respect to the element functions)
	 **********************************************************************/
	int getDotProductMatrix          ( ElementMatrix& D                                                                                          , bool progress ) const;
	int getLaplacianMatrix           ( ElementMatrix& L                                                                                          , bool progress ) const;
	int getConstrainedLaplacianMatrix( ElementMatrix& M , Real dotWeight , Real lapWeight                                                        , bool progress ) const;
	int getLaplacianMatrix           ( ElementMatrix& lMin , ElementMatrix& lMax , Vector< Real >& minCurvatures , Vector< Real >& maxCurvatures , bool progress ) const;

	/**************************************************************
	 * Computes the matrix which gives the evaluations of the basis
	 * functions at the prescribed points.
	 **************************************************************/
	template< class Vertex > void getEvaluationMatrix( const std::vector< Vertex >& points , SparseMatrix< Real , int >& E ) const;

	/**************************************************************************
	 * Computes the RHS constraint vector (with respect to the basis functions)
	 **************************************************************************/
	template< class V , class HighPrecisionV , class Function >
	void getPreciseDotVector( const std::vector< Function >& functions , std::vector< V >& b , bool progress ) const;
	template< class V , class HighPrecisionV , class Function >
	void getPreciseGradientDotGradientVector( const std::vector< Function >& gradientFunctions , std::vector< V >& b , bool progress ) const;
	template< class V , class HighPrecisionV , class Function >
	void getPreciseConstrainedGradientDotGradientVector( const std::vector< Function >& functions , std::vector< V >& b , Real dotWeight , Real lapWeight , bool progress ) const;

	template< class V , class Function >
	void getDotVector( const std::vector< Function >& functions , std::vector< V >& b , bool progress ) const
	{ getPreciseDotVector< V , V , Function >( functions , b , progress ); }
	template< class V , class Function >
	void getGradientDotGradientVector( const std::vector< Function >& gradientFunctions , std::vector< V >& b , bool progress ) const
	{ getPreciseGradientDotGradientVector< V , V , Function >( gradientFunctions , b , progress ); }
	template< class V , class Function >
	void getConstrainedGradientDotGradientVector( const std::vector< Function >& functions , std::vector< V >& b , Real dotWeight , Real lapWeight , bool progress ) const
	{ getPreciseConstrainedGradientDotGradientVector< V , V , Function >( functions , b , dotWeight , lapWeight , progress ); }

	/****************************************************************************
	 * Computes the RHS constraint vector (with respect to the element functions)
	 ****************************************************************************/
	template< class V , class HighPrecisionV , class Function >
	void getPreciseDotVector( const std::vector< Function >& functions , ElementVector< V >& b , bool progress ) const;
	template< class V , class HighPrecisionV , class Function >
	void getPreciseGradientDotGradientVector( const std::vector< Function >& gradientFunctions , ElementVector< V >& b , bool progress ) const;
	template< class V , class HighPrecisionV , class Function >
	void getPreciseConstrainedGradientDotGradientVector( const std::vector< Function >& functions , ElementVector< V >& b , Real dotWeight , Real lapWeight , bool progress ) const;

	template< class V , class Function >
	void getDotVector( const std::vector< Function >& functions , ElementVector< V >& b , bool progress ) const
	{ getPreciseDotVector< V , V , Function >( functions , b , progress ); }
	template< class V , class Function >
	void getGradientDotGradientVector( const std::vector< Function >& functions , ElementVector< V >& b , bool progress ) const
	{ getPreciseGradientDotGradientVector< V , V , Function >( functions , b , progress ); }
	template< class V , class Function >
	void getConstrainedGradientDotGradientVector( const std::vector< Function >& functions , ElementVector< V >& b , Real dotWeight , Real lapWeight , bool progress ) const
	{ getPreciseConstrainedGradientDotGradientVector< V , V , Function >( functions , b , dotWeight , lapWeight , progress ); }

	/************************
	 * Initializes the solver
	 ************************/
	template< class C > void setSolver(         MultigridSolver< Real , C , int >& solver                                                                     ) const;
	template< class C > void setSolver( ParallelMultigridSolver< Real , C , int >& solver , int pDepth , int threads , int blockSize                          ) const;
	template< class C > void setSolver(          ParallelSolver<        C       >& solver , int pDepth , int threads , int blockSize , bool setDSFromElements ) const;

	/*******************************************
	 * Gets information about quadrature points
	 * that can be used to re-compute the system
	 * as the surface evolves under flow.
	 *******************************************/
	template< class Vertex >
	void setMatrixTrackingData( TrackingData< Vertex >& tData , bool progress=false ) const;

	/********************************************
	 * Returns the indices of all basis functions
	 * whose support overlaps the edges
	 ********************************************/
	template< class Vertex >
	void setEdgeSupportingIndices( const std::vector< Vertex >& vertices , const std::vector< std::pair< int , int > >& edges , std::vector< int >& indices );

	/***********************************************************
	 * Return the number of test functions and element functions
	 ***********************************************************/
	int basisDimension( void ) const { return _basisDimension.back(); }
	int elements( void ) const { return _geometryDimension.back(); }
private:
	enum { MIN_CURVATURE = 0 , MAX_CURVATURE = 1 };
	typedef OctNode< MeshTreeNodeData< Real > , Real > Node;
	Node _tree;
	FunctionData< ( Primal ? 1 : 2 ) , double > _fData;
	Point3D< Real > _translation;
	Real _scale;
	int _depthOffset;
	std::vector< Point3D< Real > > _vertices , _baseNormals;
	std::vector< TriangleIndex > _baseTriangles;	// The original triangles
	std::vector< PolygonData > _polygons;			// The polygons resulting from clipping the input triangles to the voxels
	std::vector< int > _geometryDimension;			// The number of nodes at each depth that contain geometry
	std::vector< int > _basisDimension;				// The number of basis functions at each depth whose support overlaps a node with geometry

	int _threads;			// The number of threads to be run in parallel
	int _maxDepth;			// The actual depth of the tree (+1 for Primal)

	struct MatrixIntegrationSample
	{
	protected:
		/*
		 * The evaluation of the components of the supported basis functions
		 * and its derivatives on the sample.
		 * This uses the fact that the functions form a partition of unity.
		 * [Note that derivative values are stored pre-divided by the function values.]
		 */
		Real subValues[(Primal?2:3)-1][3] , subDValues[(Primal?2:3)-1][3];
	public:
		int index;		// Since many samples may come from the same triangle, we use an index to refer to the tangent directions
		double weight;	// The integration weight of each sample
		// These methods get/set the function and derivative values. Note that the internal storage always has the derivative divided by the value
		void setSubValues( const double sValues[Primal?2:3][3] , const double sDValues[Primal?2:3][3] , bool preDivided );
		void getSubValues(       double sValues[Primal?2:3][3] ,       double sDValues[Primal?2:3][3] , bool preDivided ) const;
	};

	template< class C >
	struct VectorIntegrals
	{
		C values[Primal?8:27];
		inline void clear( void );
		inline       C& operator[] ( int idx )       { return values[idx]; }
		inline const C& operator[] ( int idx ) const { return values[idx]; }
		inline VectorIntegrals& setScaled( const VectorIntegrals& integrals , Real s );
		inline VectorIntegrals& addScaled( const VectorIntegrals& integrals , Real s );
	};
	struct MatrixIntegrals
	{
		__declspec( align( 16 ) ) Real values[Primal?8:27][Primal?8:28];
		inline void clear( void );
		inline       Real* operator[] ( int idx )       { return values[idx]; }
		inline const Real* operator[] ( int idx ) const { return values[idx]; }
		inline MatrixIntegrals& setScaled( const MatrixIntegrals& integrals , Real s );
		inline MatrixIntegrals& addScaled( const MatrixIntegrals& integrals , Real s );
		inline MatrixIntegrals& addScaled( const MatrixIntegrals& integrals , Real s , int outRow , int inRow );
		static inline Real RowDot( const MatrixIntegrals& i1 , const Real*            i2 , int row );
		static inline Real RowDot( const MatrixIntegrals& i1 , const MatrixIntegrals& i2 , int row ) { return RowDot( i1 , i2[row] , row ); }
	};

	/*
	 * This structure stores a representation of a vector in terms of the element-function coefficients
	 */
	template< class C >	struct ElementVector : public std::vector< struct VectorIntegrals< C > >{ };

	/*
	 * These are the three matrix classes that together define a multigrid hierarchy
	 * for a multigrid fem solver
	 */
	struct ElementMatrixIterator;
	// As a matrix, this class is the linear operator defined on the elements
	struct ElementMatrix : public std::vector< MatrixIntegrals > , public SparseMatrixInterface< Real , ElementMatrixIterator >
	{
		/*************************************/
		/* Methods for SparseMatrixInterface */
		ElementMatrixIterator begin( int row ) const { return ElementMatrixIterator( row/(Primal?8:27) , (*this)[row/(Primal?8:27)][row%(Primal?8:27)] ); }
		ElementMatrixIterator end  ( int row ) const { return ElementMatrixIterator( row/(Primal?8:27) , (*this)[row/(Primal?8:27)][row%(Primal?8:27)] + (Primal?8:27) ); }
		size_t Rows( void ) const { return size() * (Primal?8:27); }
		size_t RowSize( size_t idx ) const { return Primal?8:27; }
		/*************************************/

		template< class C >
		void MultiplyParallel( const ElementVector< C >& in , ElementVector< C >& out , int threads , int multiplyFlag=MULTIPLY_ADD ) const;
	};
	struct ElementRestrictionOperatorIterator;
	// As a matrix, this class is the restriction operator, mapping from the higher resolution space to the lower.
	struct ElementRestrictionOperator : public SparseMatrixInterface< Real , ElementRestrictionOperatorIterator >
	{
		/*************************************/
		/* Methods for SparseMatrixInterface */
		ElementRestrictionOperatorIterator begin( int row ) const { return ElementRestrictionOperatorIterator( row%(Primal?8:27) , childInfo[row/(Primal?8:27)] ); }
		ElementRestrictionOperatorIterator end  ( int row ) const { return ElementRestrictionOperatorIterator( row%(Primal?8:27) , childInfo[row/(Primal?8:27)]+childCount[row/(Primal?8:27)] ); }
		size_t Rows( void )          const { return childCount.size()             * (Primal?8:27); }
		size_t RowSize( size_t idx ) const { return childCount[idx/(Primal?8:27)] * (Primal?8:27); }
		/*************************************/

		std::vector< std::pair< int , char >* > childInfo;		// The list of node-children (including their index and their corner index)
		std::vector< int > childCount;
		template< class C >
		void downSample( const ElementVector< C >& high , ElementVector< C >& low , int threads , bool resize );
		void downSample( const ElementMatrix&      high , ElementMatrix&      low , int threads , bool resize );
		ElementRestrictionOperator ( void );
		~ElementRestrictionOperator( void );
	protected:
		friend ElementRestrictionOperatorIterator;
		static __declspec( align( 16 ) ) Real  UpSampleValues    [Primal?8:27][8][Primal?8:28];
		static std::pair< int , Real >* NonZeroUpSampleValues    [Primal?8:27][8];
		static int                      NonZeroUpSampleValueCount[Primal?8:27][8];
	private:
		static bool _upSampleValuesSet;
	};

	static void BasisDownSampleMatrix( const SparseMatrix< Real , int >& high , const ElementRestrictionOperator& high2LowE , const SparseMatrix< Real , int >& low , SparseMatrix< Real , int >& high2LowB , int threads=1 );

	/*********************************************************************************************
	 * Data to support efficient transitioning from element-based systems to function-based system
	 * The template argument specifies whether or not we can assume that the expression for
	 * basis function as the linear combination of element functions only has 0/1 entries.
	 *********************************************************************************************/
	template< bool UnitEntries >
	struct ElementToBasisOperator
	{
		SparseMatrix< Real , int > e2b;
		int elementNum( void ) const { return _elementNum; }
		void init( int threads=1 );
		void resizeBasisMatrix( SparseMatrix< Real , int >& matrix ) const;
		void elementToBasisMatrix( const ElementMatrix& mIntegrals , SparseMatrix< Real , int >& matrix , int threads , bool resize ) const;

		template< class C > void MultiplyTranspose( const Vector< C >& in , ElementVector< C >& out , int threads , bool resize ) const;
		template< class C > void Multiply         ( const ElementVector< C >& in , Vector< C >& out , int threads , bool resize ) const;
	private:
		int _elementNum;
		std::vector< std::pair< int , Real > > _e2b;
		std::vector< int > _rowSize;
		std::vector< typename MeshOctree< int , Primal >::MatrixIntegrals > _offset;
	};

	/*
	 * Iterators for the different matrix classes
	 */
	struct ElementMatrixIterator : public MatrixEntry< Real , int >
	{
	public:
		using MatrixEntry<Real, int>::N;
		using MatrixEntry<Real, int>::Value;

		const Real* iValue;
		ElementMatrixIterator( int idx , const Real* i ){ iValue = i , N = idx*(Primal?8:27); }
		ElementMatrixIterator& operator ++ ( void ) { N++ ; iValue++ ; return *this; }
		bool operator == ( const ElementMatrixIterator& iter ) const { return iValue==iter.iValue; }
		bool operator != ( const ElementMatrixIterator& iter ) const { return iValue!=iter.iValue; }
		const MatrixEntry< Real , int >* operator -> ( void ) { Value = *iValue ; return this; }
	};
	struct ElementRestrictionOperatorIterator : public MatrixEntry< Real , int >
	{
		using MatrixEntry< Real , int >::N;
		using MatrixEntry< Real , int >::Value;
		int pIndex , cIndex;

		const std::pair< int , char >* entry;
		int count , base;
		const std::pair< int , Real >* values;
		ElementRestrictionOperatorIterator( int idx , const std::pair< int , char >* e ) { pIndex = idx , cIndex=0 , entry = e;	}
		ElementRestrictionOperatorIterator& operator ++ ( void ) { cIndex++ ; if( cIndex==count ) cIndex = 0 , entry++ ; return *this; }
		bool operator == ( const ElementRestrictionOperatorIterator& iter ) const { return entry==iter.entry && pIndex==iter.pIndex && cIndex==iter.cIndex; }
		bool operator != ( const ElementRestrictionOperatorIterator& iter ) const { return entry!=iter.entry || pIndex!=iter.pIndex || cIndex!=iter.cIndex; }
		const MatrixEntry< Real , int > *operator->( void )
		{
			// Note that we can only evaluate "entry" here since at construction time it might not be a valid pointer
			if( !cIndex ) count = ElementRestrictionOperator::NonZeroUpSampleValueCount[pIndex][entry->second] , values = ElementRestrictionOperator::NonZeroUpSampleValues[pIndex][entry->second] , base = entry->first * (Primal?8:27);
			N = base + values[cIndex].first;
			Value = values[cIndex].second;
			return this;
		}
	};
	template< class Integrator >
	int _maxSamplesPerNode( void ) const;

	template< class Vertex , class Integrator >
	void _setMatrixTrackingData( TrackingData< Vertex >& tData , bool progress ) const;

	void _setSolverData( std::vector< SparseMatrix< Real , int > >& DS ) const;
	void _setSolverData( std::vector< std::vector< int > >& slicesPerThread , std::vector< std::vector< int > >& entriesPerSlice , int pDepth , int threads ) const;
	void _setSolverData( std::vector< ElementRestrictionOperator >& DS ) const;
	void _setSolverData( std::vector< ElementToBasisOperator< true > >& e2b ) const;

	void _setElementToBasisMatrix( SparseMatrix< Real , int >& e2b , int depth ) const;
	void _setElementRestriction( ElementRestrictionOperator& restriction , int depth ) const;
	void _setBaseCurvatures( std::vector< Point2D< Real > >& baseCurvatureValues , std::vector< Point2D< Point3D< Real > > >& baseCurvatureDirections , double normalize=2. ) const;

	const Node* _getLeafNode( const Point3D<Real>& position ) const;
	const Node* _getLeafNode( const Point3D<Real>& position , int maxDepth ) const;
	Node*       _setLeafNode( const Point3D<Real>& position , int maxDepth );
	void _getPointOffset( const Point3D< Real >& position , int depth , int offset[3] ) const;
	typename Node::ConstNeighbors3 _getFinestNodeNeighbors( typename Node::ConstNeighborKey3& neighbors , const Point3D< Real >& position , int depth , int offset[3] ) const;

	template< class C > void _setNoiseCoefficients( const double& persistence , std::vector< C >& coefficients ) const;
	template< class C > void _setNoiseCoefficients( int minDepth , int maxDepth , const double& persistence , std::vector< C >& coefficients ) const;
	template< class C >	void _setIdentityCoefficients( Pointer( C ) coefficients ) const;

	void _addEdge( Node* node , Point3D< Real > v1 , Point3D< Real > v2 );
	bool _clipEdge( Node* node , Point3D< Real > in1 , Point3D< Real > in2 , Point3D< Real >& out1 , Point3D< Real >& out2 );

	void _addTriangles( const std::vector< TriangleIndex >& triangles , int depth );
	void _addPolygons( Node* node , const std::vector< PolygonData >& polygons , std::vector< PolygonData >& tempPolygons , int depth ,
		hash_map< long long , int >* vTable , std::vector< PolygonData >& splitPolygons );

	int _setTree( int depth , double cutOff , bool progress );

	template< class IndexType >	void _setAdjacency( int depth , SparseMatrix< byte , IndexType >& matrix , std::vector< const Node* >& nodes ) const;
	template< class IndexType >	void _setAdjacency( int depth , SparseMatrix< Real , IndexType >& matrix ) const;
	template< class IndexType >	void _compressMatrix( SparseMatrix< Real , IndexType >& matrix , bool resize=true ) const;

	template< class IndexType > void _setFromSums( SparseMatrix< Real , IndexType >& matrix , const SparseMatrix< byte , IndexType >& adjacency , double sums[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)][(Primal?2:3)][(Primal?2:3)][(Primal?2:3)] , int idx ) const;
	template< bool SetDValues > void _setSampleSubValues( const Node* node , ConstPointer( Point3D< double > ) samples , int sampleCount , Pointer( double ) values[(Primal?2:3)][3] , Pointer( double ) dValues[(Primal?2:3)][3] ) const;
	enum
	{
		DOT_MATRIX=1 ,
		LAP_MATRIX=2
	};

	// Templating the matrix type means that the compiler can remove parts of the function if the template argument ensures that they won't be called.
	template< class IndexType , int MatrixType , class Integrator >
	int _getSystemMatrix( int depth , SparseMatrix< Real , IndexType >& matrix , Real dotWeight , Real lapWeight , bool Progress ) const;

	template< int MatrixType , class Integrator >
	int _getMatrixIntegrals ( ElementMatrix& eM , bool progress , Real dWeight=Real(1.) , Real lWeight=Real(1.) ) const;

	template< class V , class HighPrecisionV , class Function , int MatrixType , class Integrator >
	int _getPreciseVector( const std::vector< Function >& functions , int depth , std::vector< V >& b , Real dotWeight , Real lapWeight , bool progress , bool normalize ) const;

	template< class V , class HighPrecisionV , class Function , int MatrixType , class Integrator >
	int _getElementVector( const std::vector< Function >& functions , ElementVector< V >& b , bool progress , Real dWeight=Real(1.) , Real lWeight=Real(1.) ) const;

	template< class V , class HighPrecisionV , class Function , class Integrator >
	V _getPreciseIntegral( const std::vector< Function >& functions ) const;

	template< class V , class HighPrecisionV , class Integrator >
	V _getPreciseCoefficientIntegral( const Vector< V >& coefficients , int depth ) const;

	template< class Vertex , class C > void _getValues( ConstPointer( Vertex ) points , int pCount , ConstPointer( C ) coefficients , Pointer( C ) pointValues );

	template< class MReal , class IndexType > void _getDownSampleMatrix( int high , SparseMatrix< MReal , IndexType >& highToLow ) const;

	template< class Integrator >
	int _getLaplacianMatrixIntegrals( ElementMatrix& minEM , ElementMatrix& maxEM , const std::vector< Point2D< Real > >& _baseCurvatureValues , const std::vector< Point2D< Point3D< Real > > > _baseCurvatureDirections , bool progress ) const;
	template< class  Integrator >
	int _getCurvatures( Vector< Real >& minCurvature , Vector< Real >& maxCurvature ,
		const std::vector< Point2D< Real > >& _baseCurvatureValues , const std::vector< Point2D< Point3D< Real > > > _baseCurvatureDirections , bool progress ) const;

	void _sortIndices( void );

	static void _FitVertices( std::vector< Point3D< Real > >& vertices , const Point3D< Real >& center , const Real& width , Point3D< Real >& translation , Real& scale );

public:
	static double maxMemoryUsage;
	static double MemoryUsage(void);
	MeshOctree( void );

	template< class Vertex , class C > void getValues( const std::vector< Vertex >& points , const Vector< C >& coefficients , std::vector< C >& pointValues );

	///////////////////////////////////////////
	// Computes the integral of the function //
	// V / HighPrecisionV must be vector spaces
	// The functions must take in a value of type Point3D< V::R > and return an object of type Gradient3D< V >
	template< class V >	V getCoefficientIntegral( const Vector< V >& coefficients ) const { return getPreciseCoefficientIntegral< V , V >( coefficients ); }
	template< class V , class Function > V getIntegral( const std::vector< Function >& functions ) const { return getPreciseIntegral< V , V , Function >( functions ); }
	template< class V , class HighPrecisionV >	V getPreciseCoefficientIntegral( const Vector< V >& coefficients ) const;
	template< class V , class HighPrecisionV , class Function >	V getPreciseIntegral( const std::vector< Function >& functions ) const;
	Real getArea( void ) const;

	template< int Dim >
	void setNoiseCoefficients( const double& persistence , std::vector< Point< Real , Dim > >& coefficients ) const;
	template< int Dim >
	void setNoiseCoefficients( const int& minDepth , const int& maxDepth , const double& persistence , std::vector< Point< Real , Dim > >& coefficients ) const;

	void setNoiseCoefficients( const double& persistence , std::vector< Point3D< Real > >& coefficients ) const;
	void setNoiseCoefficients( const int& minDepth , const int& maxDepth , const double& persistence , std::vector< Point3D< Real > >& coefficients ) const;

	template< class Point > void setIdentityCoefficients( Vector< Point >& coefficients ) const;

	void setCheckerBoardCoefficients( int depth , std::vector< Point3D< Real > >& coefficients ) const;

	/////////////////////////
	// CUDA Visualization  //
	/////////////////////////
	template< class Vertex > void getGeometryIndices( const std::vector< Vertex >& points, std::vector< int >& eIndices ) const;
	template< class Vertex > void getGeometryPositions( std::vector< Vertex >& positions ) const;
};
#include "PoissonMeshData.inl"
#include "PoissonMeshData.system.inl"
#include "PoissonMeshData.solver.inl"
#endif // POISSON_MESH_DATA_INCLUDED
