/* -*- C++ -*-
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

#include <set>
template< class Real , class C , class IndexType >
void MultigridSolver< Real , C , IndexType >::GetMatrixDiagonal( SparseMatrix< Real , IndexType >& matrix , Vector< Real >& diagonal , int threads )
{
	diagonal.Resize( matrix.groups );

#pragma omp parallel for num_threads( threads )
	for( int i=0 ; i<matrix.groups ; i++ )
	{
		diagonal[i] = 0;
		for( int j=matrix.groupSizes[i]-1 ; j>=0 ; j-- )
			if( matrix[i][j].N==i )
			{
				diagonal[i] += matrix[i][j].Value;
				matrix[i][j] = matrix[i][ --matrix.groupSizes[i] ];
			}
			for( int j=0 ; j<matrix.groupSizes[i] ; j++ ) matrix[i][j].Value /= diagonal[i];
	}
}
template< class Real , class C , class IndexType >
void MultigridSolver< Real , C , IndexType >::Multiply( const SparseMatrix< Real , IndexType >& matrix , const Vector< Real >& diagonal , const Vector< C >& in , Vector< C >& out , int threads , bool resize )
{
	if( resize ) out.Resize( matrix.groups );

	matrix.MultiplyParallel( in , out , threads , MULTIPLY_CLEAR );
#pragma omp parallel for num_threads( threads )
	for( int i=0 ; i<out.Dimensions() ; i++ ) out[i] = ( out[i] + in[i] ) * diagonal[i];
}
template< class Real , class C , class IndexType >
bool MultigridSolver< Real , C , IndexType >::DownSample( const SparseMatrix< Real , IndexType >* R , const SparseMatrix< Real , IndexType >& H , const SparseMatrix< Real , IndexType >* P , SparseMatrix< Real , IndexType >& L , int threads )
{
	if( !R && !P ) return false;
	SparseMatrix< Real , IndexType > temp;
	if( P ) SparseMatrix< Real , IndexType >::Multiply( H , *P , temp , threads );
	else
	{
		SparseMatrix< Real , IndexType > p;
		SparseMatrix< Real , IndexType >::Transpose( *R , p );
		SparseMatrix< Real , IndexType >::Multiply( H , p , temp , threads );
	}
	if( R ) SparseMatrix< Real , IndexType >::Multiply( *R , temp , L , threads );
	else
	{
		SparseMatrix< Real , IndexType > r;
		SparseMatrix< Real , IndexType >::Transpose( *P , r );
		SparseMatrix< Real , IndexType >::Multiply( r , temp , L , threads );
	}
	return true;
}
template< class Real , class C , class IndexType >
Vector< C > MultigridSolver< Real , C , IndexType >::operator * ( const Vector< C >& in ) const
{
	Vector< C > out;
	Multiply( _M.back() , _D.back() , in , out , threads , true );
	return out;
}
template< class Real , class C , class IndexType >
void MultigridSolver< Real , C , IndexType >::_SolveRestriction( int maxDepth , int gsIters )
{
	for( int d=maxDepth ; d>=0 ; d-- )
	{
		if( _fixedValues[d].size() ) for( int i=0 ; i<_fixedValues[d].size() ; i++ ) _b[d][_fixedValues[d][i].index] = _fixedValues[d][i].x;
		if( d<maxDepth ) _x[d].clear( );
		_M[d].SolveGaussSeidelAndResidual< C , Real >( _D[d] , _b[d] , gsIters , _x[d] , _r[d] , 1 , true , true );

		// Down-sample the residual
		if( d )
		{
			Vector< C >& BB = _b[d-1];
			Vector< C >& RR = _r[d];
#pragma omp parallel for num_threads( threads ) if( threads>0 )
			for( int i=0 ; i<RR.Dimensions() ; i++ ) RR[i] *= _D[d][i];
			DS[d].MultiplyParallel( _r[d] , _b[d-1] , threads , MULTIPLY_CLEAR );
#pragma omp parallel for num_threads( threads ) if( threads>0 )
			for( int i=0 ; i<BB.Dimensions() ; i++ ) if( _D[d-1][i]!=0 ) BB[i] /= _D[d-1][i] ; else BB[i] *= 0;
		}
	}
}
template< class Real , class C , class IndexType >
void ParallelMultigridSolver< Real , C , IndexType >::_SolveRestriction( int maxDepth , int gsIters )
{
	for( int d=maxDepth ; d>=0 ; d-- )
	{
		if( _fixedValues[d].size() ) for( int i=0 ; i<_fixedValues[d].size() ; i++ ) _b[d][_fixedValues[d][i].index] = _fixedValues[d][i].x;
		if( d<maxDepth )
			if( _pX[d].threads() ) _pX[d].clear();
			else                    _x[d].clear();
		if( _pX[d].threads() )
		{
			if( _gsBlockSize )
			{
				int count = (gsIters-1)/_gsBlockSize;
				if( count<0 ) count = 0;
				for( int i=0 ; i<count ; i++ ) _M[d].SolveGaussSeidelParallel< C , Real >( _D[d] , _b[d] , _gsBlockSize , &_pX[d] , 1 , true , true );
				int residual = gsIters - count * _gsBlockSize;
				if( residual<0 ) residual = 0;
				_M[d].SolveGaussSeidelAndResidualParallel< C , Real >( _D[d] , _b[d] , residual , &_pX[d] , _r[d] , 1 , true , true );
			}
			else
			{
				for( int i=0 ; i<gsIters ; i++ ) _M[d].SolveGaussSeidelParallel< C , Real >( _D[d] , _b[d] , 1 , &_pX[d] , 1 , true , true );
				_M[d].SolveGaussSeidelAndResidualParallel< C , Real , false >( _D[d] , _b[d] , 0 , &_pX[d] , _r[d] , 1 , true , true );
			}
		}
		else _M[d].SolveGaussSeidelAndResidual< C , Real >( _D[d] , _b[d] , gsIters , _x[d] , _r[d] , 1 , true , true );

		// Down-sample the residual
		if( d )
		{
			Vector< C >& BB = _b[d-1];
			Vector< C >& RR = _r[d];
#pragma omp parallel for num_threads( threads ) if( threads>0 )
			for( int i=0 ; i<RR.Dimensions() ; i++ ) RR[i] *= _D[d][i];
			DS[d].MultiplyParallel( _r[d] , _b[d-1] , threads , MULTIPLY_CLEAR );
#pragma omp parallel for num_threads( threads ) if( threads>0 )
			for( int i=0 ; i<BB.Dimensions() ; i++ ) if( _D[d-1][i]!=0 ) BB[i] /= _D[d-1][i] ; else BB[i] *= 0;
		}
	}
}
template< class Real , class C , class IndexType >
void MultigridSolver< Real , C , IndexType >::_SolveProlongation( int maxDepth , int gsIters )
{
	for( int d=0 ; d<=maxDepth ; d++ )
	{
		if( d ) _US[d-1].MultiplyParallel( _x[d-1] , _x[d] , threads , MULTIPLY_ADD );
		_M[d].SolveGaussSeidel< C , Real >( _D[d] , _b[d] , gsIters , _x[d] , 1 , true , true );
	}
}
template< class Real , class C , class IndexType >
void ParallelMultigridSolver< Real , C , IndexType >::_SolveProlongation( int maxDepth , int gsIters )
{
	for( int d=0 ; d<=maxDepth ; d++ )
	{
		if( d )
		{
			if( _pX[d-1].threads() && _pX[d].threads() ) _US[d-1].MultiplyParallel( &_pX[d-1] , &_pX[d] ,           MULTIPLY_ADD );
			else if( _pX[d].threads() )                  _US[d-1].MultiplyParallel(   _x[d-1] , &_pX[d] ,           MULTIPLY_ADD );
			else                                         _US[d-1].MultiplyParallel(   _x[d-1] ,   _x[d] , threads , MULTIPLY_ADD );
		}

		if( _pX[d].threads() )
		{
			if( _gsBlockSize )
			{
				int count = (gsIters-1)/_gsBlockSize;
				if( count<0 ) count = 0;
				for( int i=0 ; i<count ; i++ ) _M[d].SolveGaussSeidelParallel< C , Real >( _D[d] , _b[d] , _gsBlockSize , &_pX[d] , 1 , true , true );
				int residual = gsIters - count * _gsBlockSize;
				if( residual ) _M[d].SolveGaussSeidelParallel< C , Real >( _D[d] , _b[d] , residual , &_pX[d] , 1 , true , true );
			}
			else
			{
				for( int i=0 ; i<gsIters-1 ; i++ ) _M[d].SolveGaussSeidelParallel< C , Real >( _D[d] , _b[d] , 1 , &_pX[d] , 1 , true , true );
				if( gsIters ) _M[d].SolveGaussSeidelParallel< C , Real >( _D[d] , _b[d] , 1 , &_pX[d] , 1 , true , true );
			}
		}
		else _M[d].SolveGaussSeidel< C , Real >( _D[d] , _b[d] , gsIters , _x[d] , 1 , true , true );
	}
}
template< class Real , class C , class IndexType >
void MultigridSolver< Real , C , IndexType >::Init( int dimensions )
{
	_US.resize( DS.size() );
	_M.resize ( DS.size() );
	_D.resize ( DS.size() );
	_b.resize ( DS.size() );
	_x.resize ( DS.size() );
	_r.resize ( DS.size() );
	_fixedValues.resize( DS.size() );
	_b.back().Resize( dimensions ) , _x.back().Resize( dimensions ) , _r.back().Resize( dimensions );
	for( int d=1 ; d<DS.size() ; d++ ) _b[d-1].Resize( DS[d].groups ) , _x[d-1].Resize( DS[d].groups ) , _r[d-1].Resize( DS[d].groups );
	for( int i=0 ; i<DS.size()-1 ; i++ ) SparseMatrix< Real , IndexType >::Transpose( DS[i+1] , _US[i] );
}
template< class Real , class C , class IndexType >
void MultigridSolver< Real , C , IndexType >::ResetMatrix( void )
{
	for( int i=DS.size()-2 ; i>=0 ; i-- ) DownSample( &DS[i+1] , _M[i+1] , &_US[i] , _M[i] , threads );
	for( int i=0 ; i<DS.size() ; i++ ) GetMatrixDiagonal( _M[i] , _D[i] , threads );
}
template< class Real , class C ,  class IndexType >
void MultigridSolver< Real , C , IndexType >::PreDivide( void )
{
	Vector< Real >& D = _D.back();
	Vector< C >& b = _b.back();
#pragma omp parallel for num_threads( threads )
	for( int i=0 ; i<b.Dimensions() ; i++ )
		if( D[i] ) b[i] /= D[i];
		else       b[i] *=0;
}
template< class Real , class C ,  class IndexType >
void MultigridSolver< Real , C , IndexType >::PostDivide( void )
{
	const Vector< Real >& D = _D.back();
	Vector< C >& b = _b.back();
#pragma omp parallel for num_threads( threads )
	for( int i=0 ; i<b.Dimensions() ; i++ ) b[i] *= D[i];
}

template< class Real , class C , class IndexType >
void MultigridSolver< Real , C , IndexType >::SetFixedValues( const std::vector< std::pair< int , C > >& fixedValues )
{
	_fixedValues.back().resize( fixedValues.size() );
	for( int i=0 ; i<fixedValues.size() ; i++ ) _fixedValues.back()[i].index = fixedValues[i].first , _fixedValues.back()[i].x = fixedValues[i].second;
	for( int d=DS.size()-1 ; d>0 ; d-- )
	{
		std::set< int > fixedSet;
		for( int i=0 ; i<_fixedValues[d].size() ; i++ ) 
			for( int j=0 ; j<_US[d-1].groupSizes[ _fixedValues[d][i].index ] ; j++ )
				fixedSet.insert( _US[d-1][_fixedValues[d][i].index][j].N );
		_fixedValues[d-1].resize( fixedSet.size() );
		int i=0;
		C zeroValue;
		zeroValue *= 0;
		for( std::set< int >::iterator iter=fixedSet.begin() ; iter!=fixedSet.end() ; iter++ , i++ )
			_fixedValues[d-1][i].index = (*iter) , _fixedValues[d-1][i].x = zeroValue;
	}
}
template< class Real , class C , class IndexType >
void MultigridSolver< Real , C , IndexType >::_PreSolve( void )
{
	for( int d=0 ; d<DS.size() ; d++ )
	{
		Vector< C >& b = _b[d];
		Vector< Real >& D = _D[d];
		SparseMatrix< Real , IndexType >& M = _M[d];
#pragma omp parallel for num_threads( threads )
		for( int i=0 ; i<_fixedValues[d].size() ; i++ )
		{
			int idx = _fixedValues[d][i].index;
			// Get the old constraint, diagonal value, and diagonal count
			_fixedValues[d][i].b = b[ idx ];
			_fixedValues[d][i].D = D[ idx ];
			_fixedValues[d][i].groupSize = M.groupSizes[idx];
			// Update the diagonal value and diagonal count
			D[idx] = Real(1.);
			M.groupSizes[idx] = 0;
		}
	}
}
template< class Real , class C , class IndexType >
void MultigridSolver< Real , C , IndexType >::_PostSolve( void )
{
	for( int d=0 ; d<DS.size() ; d++ )
	{
		Vector< C >& b = _b[d];
		Vector< Real >& D = _D[d];
		SparseMatrix< Real , IndexType >& M = _M[d];
#pragma omp parallel for num_threads( threads )
		for( int i=0 ; i<_fixedValues[d].size() ; i++ )
		{
			int idx = _fixedValues[d][i].index;
			b[idx] = _fixedValues[d][i].b;
			D[idx] = _fixedValues[d][i].D;
			M.groupSizes[idx] = _fixedValues[d][i].groupSize;
		}
	}
}
template< class Real , class C , class IndexType >
void MultigridSolver< Real , C , IndexType >::Solve( bool preDivided )
{
	if( !preDivided ) PreDivide();
	_PreSolve();
	// Solve the linear system
	int rGSIters = cascadic ? 0 : gsIters , pGSIters = gsIters;
	for( int c=0 ; c<cycles ; c++ )
	{
		// Perform multiple V/W-cycle of the multi-grid solver
		if( vCycle )
		{
			_SolveRestriction ( _M.size()-1 , rGSIters );
			_SolveProlongation( _M.size()-1 , pGSIters );
		}
		else
		{
			_SolveRestriction( _M.size()-1 , 0 );
			for( int d=0 ; d<_M.size() ; d++ )
			{
				_SolveRestriction ( d , rGSIters );
				_SolveProlongation( d , pGSIters );
				if( d<_M.size()-1 ) _US[d].MultiplyParallel( _x[d] , _x[d+1] , threads , MULTIPLY_ADD );
			}
		}
	}
	_PostSolve();
	if( !preDivided ) PostDivide();
}
template< class Real , class C , class IndexType >
void ParallelMultigridSolver< Real , C , IndexType >::Init( int blockSize , int dimensions )
{
	_gsBlockSize = blockSize;
	MultigridSolver::Init( dimensions );

	_pX.resize( DS.size() );
	for( int i=0 ; i<DS.size() ; i++ ) if( slicesPerThread[i].size() ) _pX[i].init( blockSize , entriesPerSlice[i] , slicesPerThread[i] , sliceDependence );
}
template< class Real , class C , class IndexType >
void ParallelMultigridSolver< Real , C , IndexType >::Solve( bool setSerial , bool preDivided )
{
	if( setSerial ) _pX.back().SetFromArray( _x.back() );

	if( !preDivided ) PreDivide();
	_PreSolve();
	int rGSIters = cascadic ? 0 : gsIters , pGSIters = gsIters;
	for( int c=0 ; c<cycles ; c++ )
	{
		// Perform multiple V/W-cycle of the multi-grid solver
		if( vCycle )
		{
			_SolveRestriction ( _M.size()-1 , rGSIters );
			_SolveProlongation( _M.size()-1 , pGSIters );
		}
		else
		{
			_SolveRestriction( _M.size()-1 , 0 );
			for( int d=0 ; d<_M.size() ; d++ )
			{
				_SolveRestriction ( d , rGSIters );
				_SolveProlongation( d , pGSIters );
				if( d<_M.size()-1 )
					if( _pX[d].threads() && _pX[d+1].threads() ) _US[d].MultiplyParallel( &_pX[d] , &_pX[d+1] ,           MULTIPLY_ADD );
					else if( _pX[d+1].threads() )                _US[d].MultiplyParallel(   _x[d] , &_pX[d+1] ,           MULTIPLY_ADD );
					else                                         _US[d].MultiplyParallel(   _x[d] ,   _x[d+1] , threads , MULTIPLY_ADD );
			}
		}
	}
	_PostSolve();
	if( !preDivided ) PostDivide();

	if( setSerial ) _pX.back().SetToArray( _x.back() );
}
