/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#ifndef MULTIGRID_INCLUDED
#define MULTIGRID_INCLUDED

template< class Real , class C , class IndexType >
struct MultigridSolver
{
	////////////////////
	// Static Methods //
	////////////////////
	/************************************************************
	 * This method down-samples a matrix. (Either the restriction
	 * or the prolongation must be defined.)
	 ************************************************************/
	static bool DownSample( const SparseMatrix< Real , IndexType >* R , const SparseMatrix< Real , IndexType >& H , const SparseMatrix< Real , IndexType >* P , SparseMatrix< Real , IndexType >& L , int threads );

	/*************************************************************
	 * This method multiplies a vector by a matrix with a stripped
	 * out diagonal.
	 **************************************************************/
	static void Multiply( const SparseMatrix< Real , IndexType >& M , const Vector< Real >& D , const Vector< C >& in , Vector< C >& out , int threads , bool resize );

	/***********************************************************
	 * These methods extracts the diagonal entries from a matrix 
	 * and will remove the diagonal entry from the matrix and
	 * divide the entries by the diagonal
	 **********************************************************/
	static void GetMatrixDiagonal( SparseMatrix< Real , IndexType >& matrix , Vector< Real >& diagonal , int threads );


	///////////////////
	// Class Methods //
	///////////////////
	/*****************************************************
	 * Down-sampling data required for the multigrid solve
	 *****************************************************/
	std::vector< SparseMatrix< Real , IndexType > > DS;

	/********************************************************
	 * This method needs to be called after the down-sampling
	 * matrices are set with the dimension of the finest
	 * resolution system.
	 ********************************************************/
	void Init( int dimensions );

	/***************************
	 * Parameters for the solver
	 ***************************/
	bool   vCycle;	// Should the solver use V-cycles or W-cycles?
	int    cycles;	// The number of V/W-cycles used in the solver
	bool cascadic;	// Should relaxation only be performed on up-sampling?
	int   gsIters;	// The number of GS iterations
	int   threads;	// The number of threads for parallelization

	SparseMatrix< Real , IndexType >& M( void ) { return _M.back(); }
	Vector< Real >&                   D( void ) { return _D.back(); }
	Vector< C >&                      b( void ) { return _b.back(); }
	Vector< C >&                      x( void ) { return _x.back(); }
	Vector< C > operator * ( const Vector< C >& x ) const;

	/*****************************************
	 * This method sets up the system matrices
	 * Note that this will divide each row by
	 * the diagonal entry and strip off the
	 * diagonal
	 *****************************************/
	void ResetMatrix( void );

	/*****************************************
	 * This method divides the constraint
	 * vector by the diagonal, allowing
	 * Solve to be called with preDivided=true
	 *****************************************/
	void PreDivide( void );
	/**************************************
	 * This method undoes the division done
	 * by PreDivide
	 **************************************/
	void PostDivide( void );

	/*****************************************
	 * This method perform the multigrid solve
	 * If PreDivide has been called and neither
	 * the system matrix nor the constraints
	 * have been modivide, then preDivided
	 * can be set to "true".
	 *****************************************/
	void Solve( bool preDivided=false ); 

	/***************************************************
	 * This methods specifies that specific coefficients
	 * should have prescribed values
	 ***************************************************/
	void SetFixedValues( const std::vector< std::pair< int , C > >& fixedValues );

	MultigridSolver( void ) { cycles = 1 , vCycle = true , cascadic = false , gsIters = 2 , threads = omp_get_num_procs(); }
protected:
	void _SolveRestriction ( int maxDepth , int gsIters );
	void _SolveProlongation( int maxDepth , int gsIters );
	void _PreSolve( void );
	void _PostSolve( void );
	std::vector< SparseMatrix< Real , IndexType > > _US , _M;
	std::vector< Vector< Real > > _D;
	std::vector< Vector< C > > _b , _x , _r;
	struct _FixedValueData
	{
		int index;
		C x , b;
		Real D;
		int groupSize;
	};
	std::vector< std::vector< _FixedValueData > > _fixedValues;
};

template< class Real , class C , class IndexType >
struct ParallelMultigridSolver : public MultigridSolver< Real , C , IndexType >
{
	/**********************************************
	 * Information describing the ordering of nodes
	 * and their dependence for parallelization
	 **********************************************/
	int sliceDependence;									// The maximum number of slices between which there can be dependence
	std::vector< std::vector< int > > slicesPerThread;		// The number of slices associate to each thread (per depth)
	std::vector< std::vector< int > > entriesPerSlice;		// The number of entries associated to each slice (per depth)

	/*************************************************************
	 * This method needs to be called after the down-sampling
	 * matrices are set with the number of Gauss-Seidel iterations
	 * that are to be performed within a streaming pass and the
	 * dimension of the finest resolution system.
	 *************************************************************/
	void Init( int blockSize , int dimensions );

	ParallelSolution< C >& pX( void ) { return _pX.back(); }

	/******************************************
	 * This method perform the multigrid solve.
	 * If the setSerial flag is set to false,
	 * the solution is not moved from pX to x.
	 ******************************************/
	void Solve( bool setSerial=true , bool preDivided=false );
private:
	void _SolveRestriction ( int maxDepth , int gsIters );
	void _SolveProlongation( int maxDepth , int gsIters );
	std::vector< ParallelSolution< C > > _pX;
	int _gsBlockSize;
};

#include "Multigrid.inl"
#endif // MULTIGRID_INCLUDED