/*
Copyright (c) 2009, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#include "Hash.h"

template< class Real , class Vertex>
Real Area( const std::vector< Vertex >& vertices , const std::vector< TriangleIndex >& triangles )
{
	Real area = 0;
	for( int i=0 ; i<triangles.size() ; i++ )
	{
		Point3D< Real > v1 = Point3D< Real >( vertices[ triangles[i][1] ] ) - Point3D< Real >( vertices[ triangles[i][0] ] );
		Point3D< Real > v2 = Point3D< Real >( vertices[ triangles[i][2] ] ) - Point3D< Real >( vertices[ triangles[i][0] ] );
		area += sqrt( Point3D< Real >::SquareNorm( Point3D< Real >::CrossProduct( v1 , v2 ) ) );
	}
	return area / 2;
}

#if 1
template< class Real , class Vertex >
void FitVertices( std::vector< Vertex >& vertices , const Point3D< Real >& center , const Real& width , Point3D< Real >& translation , Real& scale )
{
	Point3D< Real > min , max;
	for( size_t i=0 ; i<vertices.size() ; i++ )
		for( int c=0 ; c<3 ;c++ )
		{
			if( !i || vertices[i].point[c]<min[c] )	min[c] = vertices[i].point[c];
			if( !i || vertices[i].point[c]>max[c] )	max[c] = vertices[i].point[c];
		}

	for( int c=0 ; c<3 ; c++ )
		if( !c || scale<max[c]-min[c] ) scale = max[c]-min[c];
	scale /= width;
	for( int c=0 ; c<3 ; c++ ) translation[c] = (max[c]+min[c])/2 - center[c]*scale;

	for( size_t s=0 ; s<vertices.size() ; s++ ) vertices[s].point = ( Point3D< Real >( vertices[s].point ) - translation ) / scale;
}
#else
template< class Real , class Vertex >
void FitVertices( std::vector< Vertex >& vertices , const Real& scaleFactor , Point3D< Real >& center , Real& scale )
{
	Point3D< Real > min , max;
	for( size_t i=0 ; i<vertices.size() ; i++ )
		for(int c=0;c<3;c++)
		{
			if(!i || vertices[i].point.coords[c]<min.coords[c])	min.coords[c] = vertices[i].point[c];
			if(!i || vertices[i].point.coords[c]>max.coords[c])	max.coords[c] = vertices[i].point[c];
		}

	for(int c=0;c<3;c++)
	{
		if( !c || scale<max.coords[c]-min.coords[c] ) scale = max.coords[c]-min.coords[c];
		center.coords[c] = (max.coords[c]+min.coords[c])/2;
	}
	scale *= scaleFactor;
	for(int c=0;c<3;c++) center.coords[c] -= scale/2;

	for( size_t s=0 ; s<vertices.size() ; s++ ) vertices[s].point = ( Point3D< Real >( vertices[s].point ) - center ) / scale;
}
#endif

template< class Vertex , class Real >
void SetVertexAreas( std::vector< Vertex >& vertices , const std::vector< std::vector< int> >& polygons , std::vector< Real >& areas )
{
	areas.resize( vertices.size() );
	for( size_t i=0 ; i<vertices.size() ; i++ ) areas[i] = 0;
	for( size_t i=0 ; i<polygons.size() ; i++ )
	{
		Point3D< double > center , normal;
		for( size_t j=0 ; j<polygons[i].size() ; j++) center += vertices[polygons[i][j]].point;
		center /= double( polygons[i].size() );

		for( size_t j=0 ; j<polygons[i].size() ; j++ )
		{
			size_t j2 =  (j+1)%polygons[i].size();
			Point3D< double > temp;
			CrossProduct< double >(vertices[ polygons[i][j] ].point-center , vertices[ polygons[i][j2] ].point-center , temp );
			double area = Length( temp ) / 2;
			areas[ polygons[i][j]  ] += Real( area );
			areas[ polygons[i][j2] ] += Real( area );
		}
	}
}
template< class Vertex , class Real >
void SetVertexAreas( std::vector< Vertex >& vertices , const std::vector< TriangleIndex >& triangles , std::vector< Real >& areas )
{
	areas.resize( vertices.size() );
	for( size_t i=0 ; i<vertices.size() ; i++ ) areas[i] = 0;
	for( size_t i=0 ; i<triangles.size() ; i++ )
	{
		Point3D< double > verts[] = { Point3D< double >( vertices[triangles[i][0]] ) , Point3D< double >( vertices[triangles[i][1]] ) , Point3D< double >( vertices[triangles[i][2]] ) };
		Point3D< double > normal = Point3D< double >::CrossProduct( verts[1]-verts[0] , verts[2]-verts[0] );
		double area = Length( normal ) / 3;
		for( int j=0 ; j<3 ; j++ ) areas[ triangles[i][j] ] += Real( area );
	}
}
template< class Vertex , class Real >
void SetVertexNormals( std::vector< Vertex >& vertices , const std::vector< std::vector< int> >& polygons , std::vector< Point3D< Real > >& normals , bool normalize )
{
	normals.resize( vertices.size() );
	for( size_t i=0 ; i<vertices.size() ; i++ ) normals[i] *= 0;
	for( size_t i=0 ; i<polygons.size() ; i++ )
	{
		Point3D< Real > center , normal;
		for( size_t j=0 ; j<polygons[i].size() ; j++) center += Point3D< Real >( vertices[polygons[i][j]] );
		center /= float( polygons[i].size() );

		for( size_t j=0 ; j<polygons[i].size() ; j++ )
		{
			size_t j2 =  (j+1)%polygons[i].size();
			Point3D< Real > temp;
			CrossProduct< Real >( Point3D< Real >( vertices[polygons[i][j]] ) - center , Point3D< Real >( vertices[polygons[i][j2]] ) - center , temp );
			normal += temp;
		}
		for( size_t j=0 ; j<polygons[i].size() ; j++ ) normals[ polygons[i][j] ] += normal;
	}
	if( normalize ) for( size_t i=0 ; i<vertices.size() ; i++ ) normals[i] /= sqrt( Point3D< Real >::SquareNorm( Point3D< Real >( normals[i] ) ) );
}
template< class Vertex , class Real >
void SetVertexNormals( std::vector< Vertex >& vertices , const std::vector< TriangleIndex >& triangles , std::vector< Point3D< Real > >& normals , bool normalize )
{
	normals.resize( vertices.size() );
	for( size_t i=0 ; i<vertices.size() ; i++ ) normals[i] *= 0;
	for( size_t i=0 ; i<triangles.size() ; i++ )
	{
		Point3D< double > verts[] = { Point3D< double >( vertices[triangles[i][0]] ) , Point3D< double >( vertices[triangles[i][1]] ) , Point3D< double >( vertices[triangles[i][2]] ) };
		Point3D< double > normal = Point3D< double >::CrossProduct( verts[1]-verts[0] , verts[2]-verts[0] );
		for( int j=0 ; j<3 ; j++ ) normals[triangles[i][j]] += Point3D< Real >( normal );
	}
	if( normalize ) for( size_t i=0 ; i<vertices.size() ; i++ ) normals[i] /= sqrt( Point3D< Real >::SquareNorm( Point3D< Real >( normals[i] ) ) );
}
template<class OrientedVertex>
void SetVertexNormals( std::vector<OrientedVertex>& vertices , const std::vector< std::vector< int> >& polygons , bool normalize )
{
	for( size_t i=0 ; i<vertices.size() ; i++ ) vertices[i].normal *= 0;
	for( size_t i=0 ; i<polygons.size() ; i++ )
	{
		Point3D<float> center , normal;
		for( size_t j=0 ; j<polygons[i].size() ; j++) center += vertices[polygons[i][j]].point;
		center /= float( polygons[i].size() );

		for( size_t j=0 ; j<polygons[i].size() ; j++ )
		{
			size_t j2 =  (j+1)%polygons[i].size();
			Point3D<float> temp;
			CrossProduct<float>(vertices[polygons[i][j]].point-center,vertices[polygons[i][j2]].point-center,temp);
			normal += temp;
		}
		for( size_t j=0 ; j<polygons[i].size() ; j++ ) vertices[polygons[i][j]].normal += normal;
	}
	if( normalize ) for( size_t i=0 ; i<vertices.size() ; i++ ) vertices[i].normal /= sqrt( Point3D<float>::SquareNorm( Point3D<float>( vertices[i].normal ) ) );
}
template<class OrientedVertex>
void SetVertexNormals( std::vector<OrientedVertex>& vertices , const std::vector< TriangleIndex >& triangles , bool normalize )
{
	for( size_t i=0 ; i<vertices.size() ; i++ ) vertices[i].normal *= 0;
	for( size_t i=0 ; i<triangles.size() ; i++ )
	{
		Point3D< double > verts[] = { Point3D< double >( vertices[triangles[i][0]] ) , Point3D< double >( vertices[triangles[i][1]] ) , Point3D< double >( vertices[triangles[i][2]] ) };
		Point3D< double > normal = Point3D< double >::CrossProduct( verts[1]-verts[0] , verts[2]-verts[0] );
		for( int j=0 ; j<3 ; j++ ) vertices[triangles[i][j]].normal += normal;
	}
	if( normalize )
		for( size_t i=0 ; i<vertices.size() ; i++ )
		{
			double l = Point3D<float>::SquareNorm( Point3D<float>( vertices[i].normal ) );
			if( l!=0 ) vertices[i].normal /= sqrt( l );
		}
}
template<class OrientedVertex>
void SetVertexNormals( const std::vector< Point3D<float> >& points , const std::vector< TriangleIndex >& triangles , std::vector< Point3D<float> >& normals , bool normalize )
{
	normals.clear();
	normals.resize( points.size() );
	for( size_t i=0 ; i<triangles.size() ; i++ )
	{
		Point3D< double > verts[] = { Point3D< double >( points[triangles[i][0]] ) , Point3D< double >( points[triangles[i][1]] ) , Point3D< double >( points[triangles[i][2]] ) };
		Point3D< double > normal = Point3D< double >::CrossProduct( verts[1]-verts[0] , verts[2]-verts[0] );
		for( int j=0 ; j<3 ; j++ ) normals[triangles[i][j]] += normal;
	}
	if( normalize ) for( size_t i=0 ; i<normals.size() ; i++ ) normals[i] /= sqrt( Point3D<float>::SquareNorm( Point3D<float>( normals[i] ) ) );
}

template< class Vertex >
void TrimHangingVertices( std::vector< Vertex >& vertices , std::vector< TriangleIndex >& triangles )
{
	std::vector< int > old2new;
	old2new.resize( vertices.size() );
	for( int i=0 ; i<vertices.size() ; i++ ) old2new[i] = -1;

	int sz = 0;
	for( int i=0 ; i<triangles.size() ; i++ ) for( int j=0 ; j<3 ; j++ )
		if( old2new[ triangles[i][j] ]==-1 ) old2new[ triangles[i][j] ] = sz++;
	for( int i=0 ; i<triangles.size() ; i++ ) for( int j=0 ; j<3 ; j++ ) triangles[i][j] = old2new[ triangles[i][j] ];
	std::vector< Vertex > nV;
	nV.resize( sz );
	for( int i=0 ; i<old2new.size() ; i++ ) if( old2new[i]!=-1 ) nV[ old2new[i] ] = vertices[i];
	vertices.resize( nV.size() );
	for( int i=0 ; i<nV.size() ; i++ ) vertices[i] = nV[i];
}
///////////////////
// Sampling Code //
///////////////////
template<class Vertex>
double TriangleArea(const std::vector<Vertex>& vertices,const TriangleIndex& triangle )
{
	Point3D<float> d[2];
	d[0] = Point3D< float >( vertices[ triangle[1] ].point-vertices[ triangle[0] ].point );
	d[1] = Point3D< float >( vertices[ triangle[2] ].point-vertices[ triangle[0] ].point );
	return Length( Point3D< float >::CrossProduct( d[0] , d[1] ) );
}
template< class Real >
double TriangleArea(const std::vector< Point3D< Real > >& vertices,const TriangleIndex& triangle )
{
	Point3D<float> d[2];
	d[0] = Point3D< float >( vertices[ triangle[1] ]-vertices[ triangle[0] ] );
	d[1] = Point3D< float >( vertices[ triangle[2] ]-vertices[ triangle[0] ] );
	return Length( Point3D< float >::CrossProduct( d[0] , d[1] ) );
}

template<class Vertex>
double PolygonArea(const std::vector<Vertex>& vertices,const std::vector<int>& polygon)
{
	if(polygon.size()<3)	return 0;
	else if(polygon.size()==3)
	{
		Point3D<float> d[2],n;
		d[0] = Point3D<float>(vertices[polygon[1]].point-vertices[polygon[0]].point);
		d[1] = Point3D<float>(vertices[polygon[2]].point-vertices[polygon[0]].point);
		CrossProduct(d[0],d[1],n);
		return Length(n);
	}
	else
	{
		double area=0;
		Point3D<float> center;
		for( size_t p=0 ; p<polygon.size() ; p++) center += Point3D<float>( vertices[polygon[p]].point / float(polygon.size()) );
		Point3D<float> d[2],n;
		for( size_t p=0 ; p<polygon.size() ; p++)
		{
			d[0] = Point3D<float>( vertices[polygon[(p  )               ]].point - center );
			d[1] = Point3D<float>( vertices[polygon[(p+1)%polygon.size()]].point - center );
			CrossProduct( d[0] , d[1],n );
			area+=Length(n);
		}
		return area;
	}
}

template< class Real >
double PolygonArea( const std::vector< Point3D< Real > >& vertices , const std::vector< int >& polygon )
{
	if( polygon.size()<3 )	return 0;
	else if( polygon.size()==3 )
	{
		Point3D<float> d[2],n;
		d[0] = Point3D< float >( vertices[polygon[1]]-vertices[polygon[0]] );
		d[1] = Point3D< float >( vertices[polygon[2]]-vertices[polygon[0]] );
		CrossProduct(d[0],d[1],n);
		return Length(n);
	}
	else
	{
		double area=0;
		Point3D<float> center;
		for( size_t p=0 ; p<polygon.size() ; p++) center += Point3D<float>( vertices[polygon[p]] / float(polygon.size()) );
		Point3D<float> d[2],n;
		for( size_t p=0 ; p<polygon.size() ; p++)
		{
			d[0] = Point3D<float>( vertices[polygon[(p  )               ]] - center );
			d[1] = Point3D<float>( vertices[polygon[(p+1)%polygon.size()]] - center );
			CrossProduct( d[0] , d[1],n );
			area+=Length(n);
		}
		return area;
	}
}
template< class Real , class OrientedVertex >
void GetRandomSample(const OrientedVertex& v1 , const OrientedVertex& v2 , const OrientedVertex& v3 , OrientedVertex& sample , bool flatNormal )
{
	double r1=Random2< double >();
	double r2=Random2< double >();
	if( r1+r2>1 )
	{
		r1=1-r1;
		r2=1-r2;
	}
	sample.position = Point3D< Real >( v1 ) + Point3D< Real >( v2-v1 ) * Real( r1 ) + Point3D< Real >( v3-v1 ) * Real( r2 );
	if( flatNormal ) sample.normal = Point3D< float >( Point3D<Real>::CrossProduct( Point3D<Real>( v2-v1 ) , Point3D<Real>( v3-v1 ) ) );
}
template< class Real , class Vertex >
void GetRandomSample(const Vertex& v1 , const Vertex& v2 , const Vertex& v3 , OrientedPoint3D< Real >& sample )
{
	double r1=Random2< double >();
	double r2=Random2< double >();
	if( r1+r2>1 )
	{
		r1=1-r1;
		r2=1-r2;
	}
	sample.position = Point3D< float >( Point3D< Real >( v1 ) + Point3D< Real >(v2-v1) * Real( r1 ) + Point3D< Real >(v3-v1) * Real( r2 ) );
	sample.normal = Point3D< float >( Point3D< Real >::CrossProduct( Point3D< Real >( v2-v1 ) , Point3D< Real >( v3-v1 ) ) );
}
template< class Real , class Vertex >
void GetRandomSample( const Vertex& v1 , const Vertex& v2 , const Vertex& v3 , Point3D< Real >& sample )
{
	double r1=Random2< double >();
	double r2=Random2< double >();
	if( r1+r2>1 )
	{
		r1=1-r1;
		r2=1-r2;
	}
	sample = Point3D< Real >( v1 ) + Point3D< Real >( v2-v1 ) * Real(r1) + Point3D< Real >( v3-v1 )*Real(r2);
}

template< class Real , class OrientedVertex >
void GetSample( const std::vector<OrientedVertex>& vertices , const TriangleIndex& tIndex , OrientedVertex& sample , bool flatNormal )
{
	GetRandomSample( vertices[ tIndex[0] ] , vertices[ tIndex[1] ] , vertices[ tIndex[2] ] , sample , flatNormal );
}
template< class Real , class Vertex >
void GetSample( const std::vector< Vertex >& vertices , const TriangleIndex& tIndex , Point3D< Real >& sample )
{
	GetRandomSample( vertices[ tIndex[0] ] , vertices[ tIndex[1] ] , vertices[ tIndex[2] ] , sample );
}
template< class Real , class Vertex >
void GetSample( const std::vector< Vertex >& vertices , const TriangleIndex& tIndex , OrientedPoint3D< Real >& sample )
{
	GetRandomSample( vertices[ tIndex[0] ] , vertices[ tIndex[1] ] , vertices[ tIndex[2] ] , sample );
}
template< class Real , class OrientedVertex >
void GetSample( const std::vector< OrientedVertex >& vertices , const std::vector<int>& polygon , OrientedVertex& sample , bool flatNormal )
{
	if(polygon.size()==3)
	{
		GetRandomSample( vertices[ polygon[0] ] , vertices[ polygon[1] ] , vertices[ polygon[2] ] , sample , flatNormal );
		return;
	}
	else
	{
		double area = PolygonArea( vertices , polygon );
		double a = area * Random2< double >();
		OrientedVertex center;
		for( size_t p=0 ; p<polygon.size() ; p++ ) center += vertices[polygon[p]] / float( polygon.size() );
				
		Point3D<float> d[2],n;
		for( size_t p=0 ; p<polygon.size() ; p++ )
		{
			d[0] = Point3D<float>( vertices[polygon[(p  )%polygon.size()]].point-center.point );
			d[1] = Point3D<float>( vertices[polygon[(p+1)%polygon.size()]].point-center.point );
			a -= Length( Point3D< float >::CrossProduct( d[0] , d[1] ) );
			if( a<=0 )
			{
				GetRandomSample( vertices[ polygon[ p ] ] , vertices[ polygon[ (p+1)%polygon.size() ] ] , center , sample , flatNormal );
				return;
			}
		}
	}
}
template< class Real , class Vertex >
void GetSample( const std::vector< Vertex >& vertices , const std::vector<int>& polygon , OrientedPoint3D< Real > & sample )
{
	if(polygon.size()==3)
	{
		GetRandomSample( vertices[ polygon[0] ] , vertices[ polygon[1] ] , vertices[ polygon[2] ] , sample , true );
		return;
	}
	else
	{
		double area = PolygonArea( vertices , polygon );
		double a = area * Random2< double >();
		Vertex center;
		for( size_t p=0 ; p<polygon.size() ; p++ ) center += vertices[polygon[p]] / float( polygon.size() );
				
		Point3D<float> d[2],n;
		for( size_t p=0 ; p<polygon.size() ; p++ )
		{
			d[0] = Point3D<float>( vertices[polygon[(p  )%polygon.size()]].point-center.point );
			d[1] = Point3D<float>( vertices[polygon[(p+1)%polygon.size()]].point-center.point );
			a -= Length( Point3D< float >::CrossProduct( d[0] , d[1] ) );
			if( a<=0 )
			{
				GetRandomSample( vertices[ polygon[ p ] ] , vertices[ polygon[ (p+1)%polygon.size() ] ] , center , sample , true );
				return;
			}
		}
	}
}

template< class Real , class OrientedVertex >
double SampleOrientedTriangles( const std::vector<OrientedVertex>& vertices , const std::vector< TriangleIndex >& triangles , int sampleNum,
							    std::vector<OrientedVertex>& samples , bool flatNormal )
{
	std::vector<int> pIndices;
	return SampleOrientedTriangles< Real , OrientedVertex >( vertices , triangles , sampleNum , samples , pIndices , flatNormal );
}
template< class Real , class OrientedVertex >
double SampleOrientedTriangles( const std::vector<OrientedVertex>& vertices , const std::vector< TriangleIndex >& triangles , int sampleNum ,
							    std::vector<OrientedVertex>& samples , std::vector<int>& pIndices , bool flatNormal )
{
	double area;
	double* areas=new double[triangles.size()+1];
	samples.resize(sampleNum);
	pIndices.resize(sampleNum);
	areas[0]=0;
	for( size_t i=0 ; i<triangles.size() ; i++ )
	{
		areas[i+1]=TriangleArea( vertices , triangles[i] );
		areas[i+1]+=areas[i];
	}
	area = areas[ triangles.size() ];
	double dA=areas[triangles.size()]/sampleNum;
	double areaSum=0;
	int pIndex=1;
	for( size_t i=0 ; i<size_t(sampleNum) ; i++ )
	{
		while(areaSum>areas[pIndex])	pIndex++;
		GetSample<Real>( vertices , triangles[pIndex-1] , samples[i] , flatNormal );
		samples[i].normal *= dA / Length( samples[i].normal );
		pIndices[i] = pIndex-1;
		areaSum+=dA;
	}
	delete[] areas;
	return area;
}

template< class Real , class Vertex >
double SampleTriangles( const std::vector< Vertex >& vertices , const std::vector< TriangleIndex >& triangles , int sampleNum , std::vector< OrientedPoint3D< Real > >& samples )
{
	std::vector<int> pIndices;
	return SampleTriangles< Real , Vertex >( vertices , triangles , sampleNum , samples , pIndices );
}
template< class Real , class Vertex >
double SampleTriangles( const std::vector< Vertex >& vertices , const std::vector< TriangleIndex >& triangles , int sampleNum , std::vector< OrientedPoint3D< Real > >& samples , std::vector<int>& pIndices )
{
	double area;
	double* areas=new double[triangles.size()+1];
	samples.resize(sampleNum);
	pIndices.resize(sampleNum);
	areas[0]=0;
	for( size_t i=0 ; i<triangles.size() ; i++ )
	{
		areas[i+1]=TriangleArea( vertices , triangles[i] );
		areas[i+1]+=areas[i];
	}
	area = areas[ triangles.size() ];
	double dA=areas[triangles.size()]/sampleNum;
	double areaSum=0;
	int pIndex=1;
	for( size_t i=0 ; i<size_t(sampleNum) ; i++ )
	{
		while(areaSum>areas[pIndex])	pIndex++;
		GetSample< Real , Vertex >( vertices , triangles[pIndex-1] , samples[i] );
		samples[i].normal *= Real( dA / Length( Point3D< double >( samples[i].normal ) ) );
		pIndices[i] = pIndex-1;
		areaSum+=dA;
	}
	delete[] areas;
	return area;
}

template< class Real , class Vertex >
double SampleTriangles( const std::vector< Vertex >& vertices , const std::vector< TriangleIndex >& triangles , int sampleNum , std::vector< Point3D< Real > >& samples )
{
	std::vector< int > pIndices;
	return SampleTriangles( vertices , triangles , sampleNum , samples , pIndices );
}
template< class Real , class Vertex >
double SampleTriangles( const std::vector< Vertex >& vertices , const std::vector< TriangleIndex >& triangles , int sampleNum , std::vector< Point3D< Real > >& samples , std::vector< int >& pIndices )
{
	double area;
	double* areas=new double[triangles.size()+1];
	samples.resize(sampleNum);
	pIndices.resize(sampleNum);
	areas[0]=0;
	for( size_t i=0 ; i<triangles.size() ; i++ )
	{
		areas[i+1]=TriangleArea( vertices , triangles[i] );
		areas[i+1]+=areas[i];
	}
	area = areas[ triangles.size() ];
	double dA=areas[triangles.size()] / sampleNum;
	double areaSum=0;
	int pIndex=1;
	for( size_t i=0 ; i<size_t(sampleNum) ; i++ )
	{
		while(areaSum>areas[pIndex])	pIndex++;
		GetSample< Real , Vertex >( vertices , triangles[pIndex-1] , samples[i] );
		pIndices[i] = pIndex-1;
		areaSum+=dA;
	}
	delete[] areas;
	return area;
}


template< class Real , class OrientedVertex >
double SampleOrientedPolygons( const std::vector<OrientedVertex>& vertices , const std::vector<std::vector<int> >& polygons , int sampleNum ,
							   std::vector<OrientedVertex>& samples , bool flatNormal )
{
	std::vector<int> pIndices;
	return SampleOrientedPolygons< Real , OrientedVertex >( vertices , polygons , sampleNum , samples , pIndices , flatNormal );
}
template< class Real , class OrientedVertex >
double SampleOrientedPolygons( const std::vector<OrientedVertex>& vertices , const std::vector<std::vector<int> >& polygons , int sampleNum,
							   std::vector<OrientedVertex>& samples , std::vector<int>& pIndices , bool flatNormal )
{
	double area;
	double* areas=new double[polygons.size()+1];
	samples.resize(sampleNum);
	pIndices.resize(sampleNum);
	areas[0]=0;
	for( size_t i=0 ; i<polygons.size() ; i++ )
	{
		areas[i+1]=PolygonArea( vertices , polygons[i] );
		areas[i+1]+=areas[i];
	}
	area = areas[ polygons.size() ];
	double dA=areas[polygons.size()]/sampleNum;
	double areaSum=0;
	int pIndex=1;
	for( size_t i=0 ; i<size_t(sampleNum) ; i++ )
	{
		while(areaSum>areas[pIndex])	pIndex++;
		GetSample<Real>( vertices , polygons[pIndex-1] , samples[i] , flatNormal );
		samples[i].normal *= dA / Length( samples[i].normal );
		pIndices[i] = pIndex-1;
		areaSum+=dA;
	}
	delete[] areas;
	return area;
}


long long VertexKey( int idx1 , int idx2 )
{
	long long key;
	if(idx1<idx2)
	{
		key = idx1;
		key <<= 32;
		key |= idx2;
	}
	else
	{
		key = idx2;
		key <<= 32;
		key |= idx1;
	}
	return key;
}
template<class Vertex>
void SubdividePolygon( std::vector<Vertex>& vertices , std::vector<std::vector<int> >& polygons )
{
	hash_map<long long,int> vTable;

	size_t pCount = polygons.size();
	for( size_t i=0 ; i<pCount ; i++ )
	{
		size_t sz=polygons[i].size();
		// Generate the vertices
		for( size_t j=0 ; j<sz ; j++ )
		{
			long long key=VertexKey(polygons[i][j],polygons[i][(j+1)%sz]);
			if(vTable.find(key)==vTable.end())
			{
				vertices.push_back((vertices[polygons[i][j]]+vertices[polygons[i][(j+1)%sz]])/2);
				vTable[key]=int( vertices.size()-1 );
			}
		}
		// Generate the ear triangles
		for( size_t j=0 ; j<sz ; j++ )
		{
			size_t pIndex=polygons.size();
			polygons.resize(pIndex+1);
			polygons[pIndex].resize(3);
			int v1,v2,v3;
			v1=polygons[i][(j+sz-1)%sz];
			v2=polygons[i][j];
			v3=polygons[i][(j+sz+1)%sz];
			polygons[pIndex][0]=vTable[VertexKey(v1,v2)];
			polygons[pIndex][1]=v2;
			polygons[pIndex][2]=vTable[VertexKey(v2,v3)];
		}
		// Generate the interior polygon
		std::vector<int> poly;
		poly.resize(sz);
		for( size_t j=0 ; j<sz ; j++ )	poly[j]=vTable[VertexKey(polygons[i][j],polygons[i][(j+1)%sz])];
		for( size_t j=0 ; j<sz ; j++ )	polygons[i][j]=poly[j];
	}
}
template<class Vertex>
void SubdivideTriangle( std::vector<Vertex>& vertices , std::vector< TriangleIndex >& triangles )
{
	hash_map<long long,int> vTable;

	size_t pCount = triangles.size();
	for( size_t i=0 ; i<pCount ; i++ )
	{
		// Generate the vertices
		for( int j=0 ; j<3 ; j++ )
		{
			long long key=VertexKey(triangles[i][j],triangles[i][(j+1)%3]);
			if(vTable.find(key)==vTable.end())
			{
				vertices.push_back((vertices[triangles[i][j]]+vertices[triangles[i][(j+1)%3]])/2);
				vTable[key]=int( vertices.size()-1 );
			}
		}
		// Generate the ear triangles
		for( int j=0 ; j<3 ; j++ )
		{
			size_t tIndex=triangles.size();
			triangles.resize(tIndex+1);
			int v1,v2,v3;
			v1=triangles[i][(j+3-1)%3];
			v2=triangles[i][j];
			v3=triangles[i][(j+3+1)%3];
			triangles[tIndex][0] = vTable[ VertexKey(v1,v2) ];
			triangles[tIndex][1] = v2;
			triangles[tIndex][2] = vTable[ VertexKey(v2,v3) ];
		}
		// Generate the interior polygon
		TriangleIndex tri;
		for( int j=0 ; j<3 ; j++ ) tri[j] = vTable[VertexKey(triangles[i][j],triangles[i][(j+1)%3])];
		triangles[i] = tri;
	}
}
#if 1
template<class Vertex>
void SubdivideTriangles( std::vector<Vertex>& vertices , std::vector< TriangleIndex >& triangles , std::vector< int >& oldIndices )
{
	std::vector< std::pair< int , int > > tempEdges;
	std::vector< int > tempIndices;
	SubdivideTrianglesAndEdges( vertices , triangles , tempEdges , oldIndices , tempIndices );
}
template<class Vertex>
void SubdivideEdges( std::vector<Vertex>& vertices , std::vector< std::pair< int , int > >& edges , std::vector< int >& oldIndices )
{
	std::vector< TriangleIndex > tempTriangles;
	std::vector< int > tempIndices;
	SubdivideTrianglesAndEdges( vertices , tempTriangles , edges , tempIndices , oldIndices );
}
#else
template<class Vertex>
void SubdivideTriangle( std::vector<Vertex>& vertices , std::vector< TriangleIndex >& triangles , std::vector< int >& oldIndices )
{
	hash_map<long long,int> vTable;

	size_t pCount = triangles.size();
	oldIndices.resize( pCount );
	for( int i=0 ; i<pCount ; i++ ) oldIndices[i] = i;
	for( size_t i=0 ; i<pCount ; i++ )
	{
		// Generate the vertices
		for( int j=0 ; j<3 ; j++ )
		{
			long long key = VertexKey( triangles[i][j] , triangles[i][(j+1)%3] );
			if(vTable.find(key)==vTable.end())
			{
				vertices.push_back((vertices[triangles[i][j]]+vertices[triangles[i][(j+1)%3]])/2);
				vTable[key]=int( vertices.size()-1 );
			}
		}
		// Generate the ear triangles
		for( int j=0 ; j<3 ; j++ )
		{
			size_t tIndex=triangles.size();
			triangles.resize(tIndex+1);
			oldIndices.resize(tIndex+1);
			int v1,v2,v3;
			v1=triangles[i][(j+3-1)%3];
			v2=triangles[i][j];
			v3=triangles[i][(j+3+1)%3];
			triangles[tIndex][0] = vTable[ VertexKey(v1,v2) ];
			triangles[tIndex][1] = v2;
			triangles[tIndex][2] = vTable[ VertexKey(v2,v3) ];
			oldIndices[tIndex] = (int)i;
		}
		// Generate the interior polygon
		TriangleIndex tri;
		for( int j=0 ; j<3 ; j++ ) tri[j] = vTable[VertexKey(triangles[i][j],triangles[i][(j+1)%3])];
		triangles[i] = tri;
	}
}
#endif
template<class Vertex>
void SubdivideTrianglesAndEdges( std::vector<Vertex>& vertices , std::vector< TriangleIndex >& triangles , std::vector< std::pair< int , int > >& edges , std::vector< int >& oldTIndices , std::vector< int >& oldEIndices )
{
	hash_map<long long,int> vTable;

	size_t pCount = triangles.size();
	oldTIndices.resize( triangles.size() );
	oldEIndices.resize( edges.size() );
	for( int i=0 ; i<triangles.size() ; i++ ) oldTIndices[i] = i;
	for( int i=0 ; i<edges.size()     ; i++ ) oldEIndices[i] = i;

	// Generate the vertices
	for( size_t i=0 ; i<triangles.size() ; i++ )
		for( int j=0 ; j<3 ; j++ )
		{
			long long key = VertexKey( triangles[i][j] , triangles[i][(j+1)%3] );
			if(vTable.find(key)==vTable.end())
			{
				vertices.push_back((vertices[triangles[i][j]]+vertices[triangles[i][(j+1)%3]])/2);
				vTable[key]=int( vertices.size()-1 );
			}
		}

	for( size_t i=0 ; i<edges.size() ; i++ )
	{
		long long key = VertexKey( edges[i].first , edges[i].second );
		if( vTable.find(key)==vTable.end() )
		{
			vertices.push_back( (vertices[edges[i].first]+vertices[edges[i].second])/2);
			vTable[key] = int( vertices.size()-1 );
		}
	}

	// Generate the refined triangles
	size_t tCount = triangles.size();
	for( size_t i=0 ; i<tCount ; i++ )
	{
		// Generate the ear triangles
		for( int j=0 ; j<3 ; j++ )
		{
			size_t tIndex=triangles.size();
			triangles.resize(tIndex+1);
			oldTIndices.resize(tIndex+1);
			int v1,v2,v3;
			v1=triangles[i][(j+3-1)%3];
			v2=triangles[i][j];
			v3=triangles[i][(j+3+1)%3];
			triangles[tIndex][0] = vTable[ VertexKey(v1,v2) ];
			triangles[tIndex][1] = v2;
			triangles[tIndex][2] = vTable[ VertexKey(v2,v3) ];
			oldTIndices[tIndex] = (int)i;
		}
		// Generate the interior polygon
		TriangleIndex tri;
		for( int j=0 ; j<3 ; j++ ) tri[j] = vTable[VertexKey(triangles[i][j],triangles[i][(j+1)%3])];
		triangles[i] = tri;
	}

	// Generate the refined edges
	size_t eCount = edges.size();
	for( size_t i=0 ; i<eCount ; i++ )
	{
		int i1 = i , i2 = edges.size();
		edges.resize(i2+1);
		oldEIndices.resize(i2+1);

		int v1 , v2 , v3;
		v1 = edges[i].first;
		v2 = vTable[ VertexKey( edges[i].first , edges[i].second ) ];
		v3 = edges[i].second;

		edges[i1].first = v1 , edges[i1].second = v2;
		edges[i2].first = v2 , edges[i2].second = v3;
		oldEIndices[i1] = oldEIndices[i2] = (int)i;
	}
}



long long SetEdgeKey( int idx1 , int idx2 , bool sort=true )
{
	unsigned long long i1 , i2;
	i1 = idx1;
	i2 = idx2;
	if( !sort || i1>i2 ) return (i1<<32) | i2;
	else                 return (i2<<32) | i1;
}
void FactorEdgeKey( long long key , int& idx1 , int& idx2 )
{
	long long i1 , i2;
	i1 = key>>32;
	i2 = (key<<32)>>32;
	idx1 = int( i1 );
	idx2 = int( i2 );
}

void SetBoundaryEdges( const std::vector< std::vector< int > >& polygons , std::vector< std::pair< int , int > >& edges )
{
	stdext::hash_map< long long , int > eTable;
	for( size_t i=0 ; i<polygons.size() ; i++ )
		for( size_t j=0 ; j<polygons[i].size() ; j++ )
		{
			long long key = SetEdgeKey( polygons[i][j] , polygons[i][(j+1)%polygons[i].size()] );
			if( eTable.find( key )==eTable.end() ) eTable[key] = 0;
			else                                   eTable[key]++;
		}
	int count=0;
	for( stdext::hash_map< long long , int >::iterator iter=eTable.begin() ; iter!=eTable.end() ; iter++ ) if( iter->second&1 ) count++;
	edges.resize( count );
	count=0;
	for( stdext::hash_map< long long , int >::iterator iter=eTable.begin() ; iter!=eTable.end() ; iter++ )
		if( iter->second&1 ) FactorEdgeKey( iter->first , edges[count].first , edges[count].second ) , count++;
}
void SetBoundaryEdges( const std::vector< TriangleIndex >& triangles , std::vector< std::pair< int , int > >& edges )
{
	stdext::hash_map< long long , int > eTable;
	for( int i=0 ; i<triangles.size() ; i++ )
		for( int j=0 ; j<3 ; j++ )
		{
			long long key = SetEdgeKey( triangles[i][j] , triangles[i][(j+1)%3] );
			if( eTable.find( key )==eTable.end() ) eTable[key] = 1;
			else                                   eTable[key]++;
		}
	int count=0;
	for( stdext::hash_map< long long , int >::iterator iter=eTable.begin() ; iter!=eTable.end() ; iter++ ) if( iter->second&1 ) count++;
	edges.resize( count );
	count=0;
	for( stdext::hash_map< long long , int >::iterator iter=eTable.begin() ; iter!=eTable.end() ; iter++ )
		if( iter->second&1 ) FactorEdgeKey( iter->first , edges[count].first , edges[count].second ) , count++;
}
template< class Real >
void WriteTriangles( char* fileName , const std::vector< Point3D< Real > >& vertices , const std::vector< TriangleIndex >& triangles , int fileType )
{
	std::vector< PlyVertex< Real > > _vertices( vertices.size() );
	for( int i=0 ; i<vertices.size() ; i++ ) _vertices[i].point = vertices[i];
	PlyWriteTriangles( fileName , _vertices , triangles , PlyVertex< Real >::WriteProperties , PlyVertex< Real >::WriteComponents , fileType );
}
