/*
Copyright (c) 2006, Michael Kazhdan and Matthew Bolitho
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#ifndef LAPLACIAN_MATRIX_1D_INCLUDED
#define LAPLACIAN_MATRIX_1D_INCLUDED
#include "SparseMatrix.h"
#include "PPolynomial.h"

enum
{
	ZERO_VALUE,
	ZERO_DERIVATIVE,
	PERIODIC_BOUNDARY,
	TOTAL_BOUNDARY_TYPES
};
#define DERIVATIVE(Type) ( ((~(Type&1))&1) | (Type&2) )

char* BoundaryTypes[]=
{
	"Zero Boundary Values",
	"Zero Boundary Derivatives",
	"Periodic Boundary"
};

//////////////////////////
// BinomialCoefficients //
//////////////////////////
template<int Degree>
class BinomialCoefficients
{
public:
	int coeffs[Degree+1];
	BinomialCoefficients(void);
};

/////////////////////////
// FiniteDifferences1D //
/////////////////////////
template<class Real,int Type>
class FiniteDifferences1D
{
	template< class IndexType >
	static void SetValues( int dim , int i , MatrixEntry< Real , IndexType >* values , bool makePositive );
public:
	template< class IndexType >
	static void GetMatrix		( const int& dim , SparseMatrix< Real , IndexType >& m , bool makePositive );
	template< class IndexType >
	static void StripDiagonal	( const SparseMatrix< Real , IndexType >& M , Vector< MatrixEntry< Real , IndexType > >& D , bool clearDiagonal );
};


//////////////////////
// FiniteElements1D //
//////////////////////
template<class Real,int Type,int Degree>
class FiniteElements1D
{
	template<class Data>
	static bool UpSample(Vector<Data>& in,Vector<Data>& out,const Real* weights,int weightCount);
	template<class Data>
	static bool DownSample(Vector<Data>& in,Vector<Data>& out,const Real* weights,int weightCount);
	template< class IndexType >
	static bool UpSampleMatrix( SparseMatrix< Real , IndexType >& M,int lowD,const Real* weights,int weightCount,int& highD);
	template< class IndexType >
	static bool DownSampleMatrix( SparseMatrix< Real , IndexType >& M,int highD,const Real* weights,int weightCount,int& lowD);
public:
	static int DimensionOffset(void);
	static int DomainSize(int dim);
	static int Dimension(int domainSize);

	template< class IndexType > static int DerivativeMatrix(int dim,SparseMatrix< Real , IndexType >& m);

	template<class Data>
	static bool UpSample(Vector<Data>& in,Vector<Data>& out);
	template<class Data>
	static bool DownSample(Vector<Data>& in,Vector<Data>& out);

	static bool IsDownSamplable(int highD,int& lowD);
	static bool IsUpSamplable(int lowD,int& highD);
	template< class IndexType >
	static bool UpSampleMatrix( SparseMatrix< Real , IndexType >& M,int lowD,int& highD);
	template< class IndexType >
	static bool DownSampleMatrix( SparseMatrix< Real , IndexType >& M,int highD,int& lowD);

	template< class IndexType >
	static bool IdentityMatrix(int dim,SparseMatrix< Real , IndexType >& m,bool rowMajor=true);
	template< class IndexType >
	static bool DotProductMatrix(int dim,SparseMatrix< Real , IndexType >& m);
	template< class IndexType >
	static bool DotProductMatrix(int dim,SparseMatrix< Real , IndexType >& m,Real min,Real max);
	template< class IndexType >
	static bool LaplacianMatrix(int dim,SparseMatrix< Real , IndexType >& m,bool weakForm,bool negate=false);
	template< class IndexType >
	static bool LaplacianMatrix(int dim,SparseMatrix< Real , IndexType >& m,Real min,Real max,bool weakForm,bool negate=false);
	template< class IndexType >
	static void StripDiagonal( const SparseMatrix< Real , IndexType >& M , Vector< MatrixEntry< Real , IndexType > >& D , bool clearDiagonal);


	template<int DotType=Type,int DotDegree=Degree>
	class DotProduct
	{
	public:
		class Helper
		{
			double shift,dotShift;
			static const int Start,Stop;
			PPolynomial<Degree   > F1;
			PPolynomial<DotDegree> F2;
			PPolynomial<Degree+DotDegree> F1F2[Degree+DotDegree+1];
			double min[Degree+DotDegree+1],max[Degree+DotDegree+1];
			double fullValues[Degree+DotDegree+1];
		public:
			Helper(void);
			void setDerivatives(int d1,int d2);
			double GetValue(int i,int j,double min,double max) const;
			template< class IndexType >
			void SetValues( int dim , int i , MatrixEntry< Real , IndexType >* values , Real min , Real max );

			static int StartOffset(void);
			static int StopOffset(void);
		};
		template< class IndexType >
		static bool DerivativeMatrix(int dim,SparseMatrix< Real , IndexType >& m,int d1,int d2,bool negate=false);
		template< class IndexType >
		static bool DerivativeMatrix(int dim,SparseMatrix< Real , IndexType >& m,Real min,Real max,int d1,int d2,bool negate=false);
		class FullDotProductStencil
		{
		public:
			class DotProductStencil
			{
			public:
				Real values[Degree+DotDegree+1];
			};
			DotProductStencil caseTable[2*DotDegree+1];
		};
		static bool DotProductStencil(int dim,FullDotProductStencil& s,int d1,int d2,bool negate=false);
	};
	class FullProlongationStencil
	{
	public:
		class ProlongationStencil
		{
		public:
			Real values[Degree+2];
		};
		ProlongationStencil caseTable[2*Degree+1];
		void Normalize(int lowD);
	};
	static bool ProlongationStencil(int lowD,FullProlongationStencil &s,int& highD);

	class FullRestrictionStencil
	{
	public:
		class RestrictionStencil
		{
		public:
			Real values[(Degree+3)>>1];
		};
		RestrictionStencil caseTable[2*Degree+2];
	};
	static bool RestrictionStencil(int highD,FullRestrictionStencil &s,int& lowD);
};

#ifdef WIN32
template<class Real,int Type,int Degree>
void CombineStencils(const typename FiniteElements1D<Real,Type,Degree>::DotProduct< Type , Degree >::FullDotProductStencil& dStencil,
					 const typename FiniteElements1D<Real,Type,Degree>::FullProlongationStencil& pStencil,int dim,
					 typename FiniteElements1D<Real,Type,Degree>::DotProduct< Type , Degree >::FullDotProductStencil& newDStencil);
template<class Real,int Type,int Degree>
void CombineStencils(const typename FiniteElements1D<Real,Type,Degree>::DotProduct< Type , Degree >::FullDotProductStencil& dStencil,
					 const typename FiniteElements1D<Real,Type,Degree>::FullProlongationStencil& pStencil,int lowDim,int highDim,
					 typename FiniteElements1D<Real,Type,Degree>::DotProduct< Type , Degree >::FullDotProductStencil& newDStencil);
template<class Real,int Type,int Degree>
void CombineStencils(const typename FiniteElements1D<Real,Type,Degree>::FullDotProductStencil& dStencil,
					 const typename FiniteElements1D<Real,Type,Degree-1>::FullProlongationStencil& pStencil,int dim,
					 typename FiniteElements1D<Real,Type,Degree>::DotProduct< Type , Degree >::FullDotProductStencil& newDStencil);
template<class Real,int Type,int Degree>
void CombineStencils(const typename FiniteElements1D<Real,Type,Degree>::FullDotProductStencil& dStencil,
					 const typename FiniteElements1D<Real,Type,Degree+1>::FullProlongationStencil& pStencil,int dim,
					 typename FiniteElements1D<Real,Type,Degree>::DotProduct< Type , Degree >::FullDotProductStencil& newDStencil);
#else // !WIN32
template<class Real,int Type,int Degree>
void CombineStencils(const typename FiniteElements1D<Real,Type,Degree>::DotProduct::FullDotProductStencil& dStencil,
					 const typename FiniteElements1D<Real,Type,Degree>::FullProlongationStencil& pStencil,int dim,
					 typename FiniteElements1D<Real,Type,Degree>::DotProduct::FullDotProductStencil& newDStencil);
template<class Real,int Type,int Degree>
void CombineStencils(const typename FiniteElements1D<Real,Type,Degree>::DotProduct::FullDotProductStencil& dStencil,
					 const typename FiniteElements1D<Real,Type,Degree>::FullProlongationStencil& pStencil,int lowDim,int highDim,
					 typename FiniteElements1D<Real,Type,Degree>::DotProduct::FullDotProductStencil& newDStencil);
template<class Real,int Type,int Degree>
void CombineStencils(const typename FiniteElements1D<Real,Type,Degree>::FullDotProductStencil& dStencil,
					 const typename FiniteElements1D<Real,Type,Degree-1>::FullProlongationStencil& pStencil,int dim,
					 typename FiniteElements1D<Real,Type,Degree>::DotProduct::FullDotProductStencil& newDStencil);
template<class Real,int Type,int Degree>
void CombineStencils(const typename FiniteElements1D<Real,Type,Degree>::FullDotProductStencil& dStencil,
					 const typename FiniteElements1D<Real,Type,Degree+1>::FullProlongationStencil& pStencil,int dim,
					 typename FiniteElements1D<Real,Type,Degree>::DotProduct::FullDotProductStencil& newDStencil);
#endif
#include "LaplacianMatrix1D.inl"
#endif // LAPLACIAN_MATRIX_INCLUDED
