/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#ifndef HALF_EDGE_INCLUDED
#define HALF_EDGE_INCLUDED

template< class VertexData >
struct Vertex
{
	int inEdgePointer;		 	// Index of a half-edge entering the vertex. If the vertex is on the boundary, the edge will be a boundary edge.
	VertexData data;			// Additional vertex data.
};
template< class EdgeData >
struct HalfEdge
{
	int nextEdgePointer;		// Index of the previous edge on the face.
	int facePointer;			// Index of the face tied to this half-edge or -1 if this is a boundary edge.
	int startVertexPointer;		// Index of the vertex at the beginning of the edge.
	int oppositeEdgePointer;	// Index of the opposite edge.
	EdgeData data;				// Additional edge data.
};
// Assume that the face has a polygon boundary that is a single connected component.
template< class FaceData >
struct Face
{
	int edgePointer;			// Index of an edge on the face
	FaceData data;				// Additional data for the face
};
template< class FaceData >
struct Triangle
{
	int vIndices[3];			// The indices for the three vertices of the triangle
	FaceData data;				// Additional data for the face.

	// Support for easier access to the vertex indices
	inline       int& operator[] ( int idx )       { return vIndices[idx]; }
	inline const int& operator[] ( int idx ) const { return vIndices[idx]; }
};

template< class VertexData , class EdgeData , class FaceData >
class HalfEdgeInterface
{
public:
	typedef Vertex< VertexData > V;
	typedef Edge  < EdgeData   > E;
	typedef Face  < FaceData   > F;

	// Implement with circulators

	// Returns the vertices in the one ring-neighborhood of a vertex
	static std::vector< int > VertexAdjacentVertices( const std::vector< V >& vertices , const std::vector< E >& edges , const std::vector< F >& faces , int vIndex );

	// Returns the faces in the one-ring neighborhood of a vertex
	static std::vector< int > VertexAdjacentFaces   ( const std::vector< V >& vertices , const std::vector< E >& edges , const std::vector< F >& faces , int vIndex );

	// Returns the edges in the one-ring neighborhood of a vertex
	static std::vector< int > VertexAdjacentEdges   ( const std::vector< V >& vertices , const std::vector< E >& edges , const std::vector< F >& faces , int vIndex );

	// Return the vertices on the face
	static std::vector< int > FaceVertices( const std::vector< V >& vertices , const std::vector< E >& edges , const std::vector< F >& faces , int fIndex );

	// Return the vertices on an edge
	std::pair< int , int > EdgeVertices( const std::vector< V >& vertices , const std::vector< E >& edges , const std::vector< F >& faces , int eIndex );

	// Transforms a triangle mesh into a half-edge data structure
	static bool SetHalfEdgeData( const std::vector< Vertex >& triangleVertices , const std::vector< Triangle< FaceData > >& triangles ,
		std::vector< V >& vertices , std::vector< E >& edges , std::vector< F >& faces );

	// Opens up the mesh by removing opposite connection and duplicating vertices as necessary.
	static bool SplitOpenMesh( const std::vector< V >& oldVertices , const std::vector< E >& oldEdges , const std::vector< std::pair< int , int > >& seamEdges ,
		std::vector< V >& newVertices , std::vector< E >& newEdges );
};
#include "HalfEdge.inl"
#endif // HALF_EDGE_INCLUDED

