/* -*- C++ -*-
Copyright (c) 2006, Michael Kazhdan and Matthew Bolitho
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

//////////////////
// FunctionData //
//////////////////
template<int Degree,class Real> const int FunctionData<Degree,Real>::D0_FLAG=1;
template<int Degree,class Real> const int FunctionData<Degree,Real>::D1_FLAG=2;
template<int Degree,class Real> const int FunctionData<Degree,Real>::D2_FLAG=4;
template<int Degree,class Real> const int FunctionData<Degree,Real>::DOT_FLAG=1;
template<int Degree,class Real> const int FunctionData<Degree,Real>::D_DOT_FLAG=2;
template<int Degree,class Real> const int FunctionData<Degree,Real>::D2_DOT_FLAG=4;
template<int Degree,class Real>
const int FunctionData<Degree,Real>::VALUE_FLAG=1;
template<int Degree,class Real>
const int FunctionData<Degree,Real>::D_VALUE_FLAG=2;

template< int Degree , class Real >
FunctionData< Degree , Real >::FunctionData( void )
{
	fCount = vCount = 0;
#if NEW_FUNCTION_MEMORY
	dotTable = dDotTable = d2DotTable = NullPointer< Real >( );
	valueTables = dValueTables = NullPointer< Real >( );
	baseFunctions   = NullPointer< PPolynomial<  Degree                   > >( );
	dBaseFunctions  = NullPointer< PPolynomial< (Degree>0 ? Degree-1 : 0) > >( );
	d2BaseFunctions = NullPointer< PPolynomial< (Degree>1 ? Degree-2 : 0) > >( );
#else // !NEW_FUNCTION_MEMORY
	dotTable=dDotTable=d2DotTable=NULL;
	valueTables=dValueTables=NULL;
	baseFunctions = NULL;
	dBaseFunctions = NULL;
	d2BaseFunctions = NULL;
#endif // NEW_FUNCTION_MEMORY
}

template<int Degree,class Real>
FunctionData<Degree,Real>::~FunctionData(void){
	if( fCount )
	{
#if NEW_FUNCTION_MEMORY
		DeletePointer(     dotTable );
		DeletePointer(    dDotTable );
		DeletePointer(   d2DotTable );
		DeletePointer(  valueTables );
		DeletePointer( dValueTables );
#else // !NEW_FUNCTION_MEMORY
		if(    dotTable )  delete[]    dotTable;
		if(   dDotTable )  delete[]   dDotTable;
		if(  d2DotTable )  delete[]  d2DotTable;
		if(  valueTables ) delete[]  valueTables;
		if( dValueTables ) delete[] dValueTables;
#endif // NEW_FUNCTION_MEMORY
	}
#if NEW_FUNCTION_MEMORY
	dotTable = dDotTable = d2DotTable = NullPointer< Real >( );
	valueTables = dValueTables = NullPointer< Real >( );
#else // !NEW_FUNCTION_MEMORY
	dotTable=dDotTable=d2DotTable=NULL;
	valueTables=dValueTables=NULL;
#endif // NEW_FUNCTION_MEMORY
	fCount = vCount = 0;
#if NEW_FUNCTION_MEMORY
	DeletePointer(   baseFunctions );
	DeletePointer(  dBaseFunctions );
	DeletePointer( d2BaseFunctions );
#else // !NEW_FUNCTION_MEMORY
	if(   baseFunctions ) delete[]   baseFunctions ,   baseFunctions = NULL;
	if(  dBaseFunctions ) delete[]  dBaseFunctions ,  dBaseFunctions = NULL;
	if( d2BaseFunctions ) delete[] d2BaseFunctions , d2BaseFunctions = NULL;
#endif // NEW_FUNCTION_MEMORY
}
template<int Degree,class Real>
void FunctionData<Degree,Real>::reset(const int& maxDepth)
{
	depth = maxDepth;
	if( primal ) fCount = BinaryNode< double >::CumulativeCornerCount( depth );
	else         fCount = BinaryNode< double >::CumulativeCenterCount( depth );
	vCount = BinaryNode< double >::CornerCount( depth+1 );
}
template<int Degree,class Real>
void FunctionData<Degree,Real>::set( const int& maxDepth , const PPolynomial<Degree>& F , const int& normalize , const int& useDotRatios , bool symmetrize , int flag , bool primal , Real start , Real end )
{
	this->normalize = normalize;
	this->useDotRatios = useDotRatios;
	this->primal = primal;
	this->start = start;
	this->end = end;

	reset( maxDepth );
#if NEW_FUNCTION_MEMORY
	DeletePointer(   baseFunctions );
	DeletePointer(  dBaseFunctions );
	DeletePointer( d2BaseFunctions );
	if( flag & D0_FLAG )   baseFunctions = NewPointer< PPolynomial<  Degree                   > >( fCount );
	if( flag & D1_FLAG )  dBaseFunctions = NewPointer< PPolynomial< (Degree>0 ? Degree-1 : 0) > >( fCount );
	if( flag & D2_FLAG ) d2BaseFunctions = NewPointer< PPolynomial< (Degree>1 ? Degree-2 : 0) > >( fCount );
#else // !NEW_FUNCTION_MEMORY
	if(   baseFunctions ) delete[]   baseFunctions ,   baseFunctions = NULL;
	if(  dBaseFunctions ) delete[]  dBaseFunctions ,  dBaseFunctions = NULL;
	if( d2BaseFunctions ) delete[] d2BaseFunctions , d2BaseFunctions = NULL;
	if( flag & D0_FLAG )   baseFunctions = new PPolynomial< Degree   >[fCount];
	if( flag & D1_FLAG )  dBaseFunctions = new PPolynomial< (Degree>0 ? Degree-1 : 0) >[fCount];
	if( flag & D2_FLAG ) d2BaseFunctions = new PPolynomial< (Degree>1 ? Degree-2 : 0) >[fCount];
#endif // NEW_FUNCTION_MEMORY
	// Scale the function so that it has:
	// 0] Unmodified
	// 1] Value 1 at 0
	// 2] Integral equal to 1
	// 3] Square integral equal to 1
	switch(normalize)
	{
		case 3:
			baseFunction=F/sqrt((F*F).integral(F.polys[0].start,F.polys[F.polyCount-1].start));
			break;
		case 2:
			baseFunction=F/F.integral(F.polys[0].start,F.polys[F.polyCount-1].start);
			break;
		case 1:
			baseFunction=F/F(0);
			break;
		default:
			baseFunction=F;
	}
	dBaseFunction  =  baseFunction.derivative();
	d2BaseFunction = dBaseFunction.derivative();
	double c1 , w1;
	for( int i=0 ; i<fCount ; i++ )
	{
		if( primal ) BinaryNode<double>::CornerAndWidth( i , c1 , w1 , start , end );
		else         BinaryNode<double>::CenterAndWidth( i , c1 , w1 , start , end );
		PPolynomial< Degree > bf = baseFunction.scale(w1).shift(c1);
		// Scale the function so that it has L2-norm equal to one
		switch(normalize)
		{
			case 3:
				bf /= sqrt(w1);
				break;
			case 2:
				bf /= w1;
				break;
		}
		if( symmetrize && !primal )
		{
			if( bf.polys[0].start<start && bf.polys[bf.polyCount-1].start>end ) bf = bf + bf.shift( -2*(c1-start) ) + bf.shift( 2*(end-c1) );
			else if( bf.polys[0].start<start ) bf = bf + bf.shift( -2*(c1-start) );
			else if( bf.polys[bf.polyCount-1].start>end ) bf = bf + bf.shift( 2*(end-c1) );
	    }
		if( flag&D0_FLAG )   baseFunctions[i] = bf;
		if( flag&D1_FLAG )  dBaseFunctions[i] = bf.derivative();
		if( flag&D2_FLAG ) d2BaseFunctions[i] = bf.derivative().derivative();
	}
}
template<int Degree,class Real>
void FunctionData<Degree,Real>::setDotTables( const int& flags )
{
	clearDotTables(flags);
	int size;
	size=(fCount*fCount+fCount)>>1;
	if(flags & DOT_FLAG)
	{
#if NEW_FUNCTION_MEMORY
		dotTable = NewPointer< Real >( size );
#else // !NEW_FUNCTION_MEMORY
		dotTable=new Real[size];
#endif // NEW_FUNCTION_MEMORY
		memset(dotTable,0,sizeof(Real)*size);
	}
	if(flags & D_DOT_FLAG)
	{
#if NEW_FUNCTION_MEMORY
		dDotTable = NewPointer< Real >( size );
#else // !NEW_FUNCTION_MEMORY
		dDotTable=new Real[size];
#endif // NEW_FUNCTION_MEMORY
		memset(dDotTable,0,sizeof(Real)*size);
	}
	if(flags & D2_DOT_FLAG)
	{
#if NEW_FUNCTION_MEMORY
		d2DotTable = NewPointer< Real >( size );
#else // !NEW_FUNCTION_MEMORY
		d2DotTable=new Real[size];
#endif // NEW_FUNCTION_MEMORY
		memset(d2DotTable,0,sizeof(Real)*size);
	}
	double t1,t2;
	t1=baseFunction.polys[0].start;
	t2=baseFunction.polys[baseFunction.polyCount-1].start;
	for( int i=0 ; i<fCount ; i++ )
	{
		double c1 , c2 , w1 , w2;
		if( primal ) BinaryNode<double>::CornerAndWidth( i , c1 , w1 , start , end );
		else         BinaryNode<double>::CenterAndWidth( i , c1 , w1 , start , end );
		double start1 = t1*w1+c1;
		double end1   = t2*w1+c1;
		for(int j=0;j<=i;j++)
		{
			if( primal ) BinaryNode<double>::CornerAndWidth( j , c2 , w2 , start , end );
			else         BinaryNode<double>::CenterAndWidth( j , c2 , w2 , start , end );
			int idx=SymmetricIndex(i,j);

			double start2 = t1*w2+c2;
			double end2   = t2*w2+c2;

			if(start2<start1) start2=start1;
			if(end2>end1)     end2=end1;
			if(start2>=end2) continue;

			if( primal ) BinaryNode<double>::CornerAndWidth( j , c2 , w2 , start , end );
			else         BinaryNode<double>::CenterAndWidth( j , c2 , w2 , start , end );
			Real dot=dotProduct( c1 , w1 , c2 , w2 );
			if(fabs(dot)<1e-15) continue; 
			if(flags & DOT_FLAG) dotTable[idx]=dot;
			if(useDotRatios)
			{
				if(flags &  D_DOT_FLAG) dDotTable [idx] = -dDotProduct(c1,w1,c2,w2)/dot;
				if(flags & D2_DOT_FLAG) d2DotTable[idx] = d2DotProduct(c1,w1,c2,w2)/dot;
			}
			else
			{
				if(flags &  D_DOT_FLAG) dDotTable [idx] =  dDotProduct(c1,w1,c2,w2);
				if(flags & D2_DOT_FLAG) d2DotTable[idx] = d2DotProduct(c1,w1,c2,w2);
			}
		}
	}
}
template<int Degree,class Real>
void FunctionData<Degree,Real>::clearDotTables(const int& flags)
{
	if((flags & DOT_FLAG) && dotTable)
	{
#if NEW_FUNCTION_MEMORY
		DeletePointer( dotTable );
#else // !NEW_FUNCTION_MEMORY
		delete[] dotTable;
		dotTable=NULL;
#endif // NEW_FUNCTION_MEMORY
	}
	if((flags & D_DOT_FLAG) && dDotTable)
	{
#if NEW_FUNCTION_MEMORY
		DeletePointer( dDotTable );
#else // !NEW_FUNCTION_MEMORY
		delete[] dDotTable;
		dDotTable=NULL;
#endif // NEW_FUNCTION_MEMORY
	}
	if((flags & D2_DOT_FLAG) && d2DotTable)
	{
#if NEW_FUNCTION_MEMORY
		DeletePointer( d2DotTable );
#else // !NEW_FUNCTION_MEMORY
		delete[] d2DotTable;
		d2DotTable=NULL;
#endif // NEW_FUNCTION_MEMORY
	}
}
template<int Degree,class Real>
void FunctionData<Degree,Real>::setValueTables(const int& flags,const double& smooth){
	clearValueTables();
#if NEW_FUNCTION_MEMORY
	if(flags &   VALUE_FLAG)  valueTables = NewPointer< Real >( fCount*vCount );
	if(flags & D_VALUE_FLAG) dValueTables = NewPointer< Real >( fCount*vCount );
#else // !NEW_FUNCTION_MEMORY
	if(flags &   VALUE_FLAG)  valueTables=new Real[fCount*vCount];
	if(flags & D_VALUE_FLAG) dValueTables=new Real[fCount*vCount];
#endif // NEW_FUNCTION_MEMORY
	PPolynomial<Degree+1> function;
	PPolynomial<Degree>  dFunction;
	for(int i=0;i<fCount;i++){
		if(smooth>0)
		{
			function =baseFunctions[i].MovingAverage(smooth);
			dFunction=baseFunctions[i].derivative().MovingAverage(smooth);
		}
		else
		{
			function =baseFunctions[i];
			dFunction=baseFunctions[i].derivative();
		}
		for(int j=0;j<vCount;j++)
		{
			double x=double(j)/(vCount-1);
			x *= (end-start);
			x += start;
			if(flags &   VALUE_FLAG)  valueTables[j*fCount+i]=Real( function(x));
			if(flags & D_VALUE_FLAG) dValueTables[j*fCount+i]=Real(dFunction(x));
		}
	}
}
template<int Degree,class Real>
void FunctionData<Degree,Real>::setValueTables(const int& flags,const double& valueSmooth,const double& normalSmooth){
	clearValueTables();
#if NEW_FUNCTION_MEMORY
	if(flags &   VALUE_FLAG)  valueTables = NewPointer< Real >( fCount*vCount );
	if(flags & D_VALUE_FLAG) dValueTables = NewPointer< Real >( fCount*vCount );
#else // !NEW_FUNCTION_MEMORY
	if(flags &   VALUE_FLAG)  valueTables=new Real[fCount*vCount];
	if(flags & D_VALUE_FLAG) dValueTables=new Real[fCount*vCount];
#endif // NEW_FUNCTION_MEMORY
	PPolynomial<Degree+1> function;
	PPolynomial<Degree>  dFunction;
	for(int i=0;i<fCount;i++)
	{
		if(valueSmooth>0) function=baseFunctions[i].MovingAverage(valueSmooth);
		else              function=baseFunctions[i];
		if(normalSmooth>0) dFunction=baseFunctions[i].derivative().MovingAverage(normalSmooth);
		else               dFunction=baseFunctions[i].derivative();

		for(int j=0;j<vCount;j++)
		{
			double x=double(j)/(vCount-1);
			x *= (end-start);
			x += start;
			if(flags &   VALUE_FLAG)  valueTables[j*fCount+i]=Real( function(x));
			if(flags & D_VALUE_FLAG) dValueTables[j*fCount+i]=Real(dFunction(x));
		}
	}
}


template<int Degree,class Real>
void FunctionData<Degree,Real>::clearValueTables(void)
{
#if NEW_FUNCTION_MEMORY
	DeletePointer(  valueTables );
	DeletePointer( dValueTables );
#else // !NEW_FUNCTION_MEMORY
	if( valueTables){delete[]  valueTables;}
	if(dValueTables){delete[] dValueTables;}
	valueTables=dValueTables=NULL;
#endif // NEW_FUNCTION_MEMORY
}
template<int Degree,class Real>
Real FunctionData<Degree,Real>::dotProduct(const double& center1,const double& width1,const double& center2,const double& width2) const
{
	double r=fabs(baseFunction.polys[0].start);
	switch(normalize){
		case 2:
			return Real((baseFunction*baseFunction.scale(width2/width1).shift((center2-center1)/width1)).integral(-2*r,2*r)*width1/sqrt(width1*width2));
		case 1:
			return Real((baseFunction*baseFunction.scale(width2/width1).shift((center2-center1)/width1)).integral(-2*r,2*r)*width1/(width1*width2));
		default:
			return Real((baseFunction*baseFunction.scale(width2/width1).shift((center2-center1)/width1)).integral(-2*r,2*r)*width1);
	}
}
template<int Degree,class Real>
Real FunctionData<Degree,Real>::dDotProduct(const double& center1,const double& width1,const double& center2,const double& width2) const
{
	double r=fabs(baseFunction.polys[0].start);
	switch(normalize){
		case 2:
			return Real((dBaseFunction*baseFunction.scale(width2/width1).shift((center2-center1)/width1)).integral(-2*r,2*r)/sqrt(width1*width2));
		case 1:
			return Real((dBaseFunction*baseFunction.scale(width2/width1).shift((center2-center1)/width1)).integral(-2*r,2*r)/(width1*width2));
		default:
			return Real((dBaseFunction*baseFunction.scale(width2/width1).shift((center2-center1)/width1)).integral(-2*r,2*r));
	}
}
template<int Degree,class Real>
Real FunctionData<Degree,Real>::d2DotProduct(const double& center1,const double& width1,const double& center2,const double& width2) const{
	double r=fabs(baseFunction.polys[0].start);
	switch(normalize){
		case 2:
			return Real((dBaseFunction*dBaseFunction.scale(width2/width1).shift((center2-center1)/width1)).integral(-2*r,2*r)/width2/sqrt(width1*width2));
		case 1:
			return Real((dBaseFunction*dBaseFunction.scale(width2/width1).shift((center2-center1)/width1)).integral(-2*r,2*r)/width2/(width1*width2));
		default:
			return Real((dBaseFunction*dBaseFunction.scale(width2/width1).shift((center2-center1)/width1)).integral(-2*r,2*r)/width2);
	}
}
template<int Degree,class Real>
inline int FunctionData<Degree,Real>::SymmetricIndex(const int& i1,const int& i2)
{
	if(i1>i2) return ((i1*i1+i1)>>1)+i2;
	else      return ((i2*i2+i2)>>1)+i1;
}
template<int Degree,class Real>
inline int FunctionData<Degree,Real>::SymmetricIndex(const int& i1,const int& i2,int& index)
{
	if(i1<i2)
	{
		index=((i2*i2+i2)>>1)+i1;
		return 1;
	}
	else
	{
		index=((i1*i1+i1)>>1)+i2;
		return 0;
	}
}
