/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#include "mouse.h"
#include "shaders.inl"

int window_width  = 800;
int window_height = 600;
Point3D< float > clearColor( 180./255 , 180./255 , 180./255 );
Mouse mouse;

unsigned int timer = 0;
const int FPS_FRAMES_TO_REFRESH = 20;
double fps;
double currentTime, lastTime;
int frameCount=0;

// Values locked on the host
int coefficientMemSize;
float* host_coefficientValues;
// Values stored on the GPU
float* gpu_evaluationValues;
float* gpu_coefficientValues;
unsigned int* gpu_evaluationIndices;

/////////////////////////
// GL Shading Language //
/////////////////////////
GLuint ShaderProgram;
GLuint fShader;
GLuint vShader;

bool pauseSolver = true;
class Event
{
public:
	enum
	{
		SOLVE_EVENT ,
		WRITE_EVENT ,
		EXIT_EVENT
	};
	int eventType;
	char fileName[512];
};
std::vector< Event > eventList;

template< class Real , bool Primal >
void cpuUpdateVBO( )
{
	const int supportCount = Tree< Real , Primal >::supportCount;
	int threads = Tree< Real , Primal >::solver.pX().threads();
	Vector< FastPoint3D< Real > >& X = Tree< Real , Primal >::solver.x();
	Tree< Real , Primal >::solver.pX().SetToArray( X );
#pragma omp parallel for num_threads( threads )
	for( int i=0 ; i<myStage.vertices.size() ; i++ )
	{
		const SparseMatrix< Real , int >& E = Tree< Real , Primal >::E;
		myStage.vertices[i] = Point3D< float >( );
		for( int j=0 ; j<supportCount ; j++ )
			if( E[i][j].Value>0 && E[i][j].N>=0 ) myStage.vertices[i] += Point3D< Real >( X[E[i][j].N] * Real( E[i][j].Value ) );
	}
	myStage.setVertexBufferObject( );
}
template< class Real , bool Primal >
void gpuUpdateVBO( void )
{
	double t;
	if( FullVerbose.set ) t=Time();
#pragma omp parallel for num_threads( Tree< Real , Primal >::solver.pX().threads() )
	for( int i=0 ; i<Tree< Real , Primal >::solver.pX().threads() ; i++ )
	{
		int ii=0;
		for( int j=0 ; j<i ; j++ ) ii += Tree< Real , Primal >::solver.pX().size( j );
		ConstPointer( FastPoint3D< Real > ) _X = Tree< Real , Primal >::solver.pX()[i];
		for( int j=0 ; j<Tree< Real , Primal >::solver.pX().size( i ) ; j++ , ii++ )
		{
			Point3D< Real > x = Point3D< Real >( _X[j] );
			host_coefficientValues[ii*4+0] = (float)x[0];
			host_coefficientValues[ii*4+1] = (float)x[1];
			host_coefficientValues[ii*4+2] = (float)x[2];
			host_coefficientValues[ii*4+3] = 0;
		}
	}
	float4 *dptr;
	cutilSafeCall( cudaMemcpy( gpu_coefficientValues , host_coefficientValues , coefficientMemSize, cudaMemcpyHostToDevice) );
	cutilSafeCall( cudaGLMapBufferObject( (void**)&dptr , GLuint(myStage.vbo) ) );
	Tree< Real , Primal >::cuKernel( dptr, gpu_evaluationIndices, gpu_evaluationValues, gpu_coefficientValues, (GLuint)Tree< Real , Primal >::subdividedVertices.size(), Tree< Real , Primal >::solver.pX().size() );
	cutilSafeCall( cudaGLUnmapBufferObject( myStage.vbo ) );
	if( FullVerbose.set ) printf( "VBO updated in %fs \n" , Time()-t );
}

template< class Real , bool Primal >
void updateVBO( void )
{
	if( CPUEvaluation.set ) cpuUpdateVBO< Real , Primal >( );
	else                    gpuUpdateVBO< Real , Primal >( );
}

void setShaders( void ) 
{
	const char * vv = vertex_code;
	const char * ff = fragment_code;
	vShader = glCreateShader(GL_VERTEX_SHADER);
	fShader = glCreateShader(GL_FRAGMENT_SHADER);	
	glShaderSource(vShader, 1, &vv, NULL);
	glShaderSource(fShader, 1, &ff, NULL);

	GLint success;
	GLchar infoLog[1024];
	glCompileShader(vShader);
	glGetShaderiv(vShader, GL_COMPILE_STATUS, &success);
	if( !success )
	{
	   glGetShaderInfoLog(vShader, 1024, NULL, infoLog);
	   fprintf(stderr, "Error in the vertex shader compilation:\n %s\n", infoLog);
	}

	glCompileShader(fShader);
	glGetShaderiv(fShader, GL_COMPILE_STATUS, &success);
	if( !success )
	{
	   glGetShaderInfoLog(fShader, 1024, NULL, infoLog);
	   fprintf(stderr, "Error in the fragment shader compilation:\n %s\n", infoLog);
	}

	ShaderProgram = glCreateProgram();		
	glAttachShader(ShaderProgram,vShader);
	glAttachShader(ShaderProgram,fShader);

	glLinkProgram(ShaderProgram);
	glGetProgramiv(ShaderProgram, GL_LINK_STATUS, &success);
	if (!success)
	{
		glGetProgramInfoLog(ShaderProgram, 1024, NULL, infoLog);
		fprintf(stderr, "Error in th shader linking:\n %s\n", infoLog);
	}
	glUseProgram(ShaderProgram);
}
template< class Real , bool Primal >
void OutputEditedPLY( char* filename )
{
	std::vector< PlyVertex< Real > > outVertices( Tree< Real , Primal >::subdividedVertices.size() );
	std::vector< TriangleIndex > outTriangles = myStage.triangles;
	for( int i=0 ; i<outVertices.size() ; i++ ) outVertices[i].point = Tree< Real , Primal >::subdividedVertices[i].point;

	Vector< FastPoint3D< Real > > vPositions;
	vPositions = Tree< Real , Primal >::E * Tree< Real , Primal >::solver.x();

	for( int i=0 ; i<vPositions.Dimensions() ; i++ )
		outVertices[i].point = Point3D< Real >( vPositions[i] ) * Tree< Real , Primal >::scale + Tree< Real , Primal >::center;

	PlyWriteTriangles( filename , outVertices , outTriangles , PlyVertex< Real >::ReadProperties , PlyVertex< Real >::ReadComponents , file_type , comments , commentNum );
}
void drawTimer( void )
{
	frameCount++;
	if( frameCount==FPS_FRAMES_TO_REFRESH )
	{
		currentTime=Time();
		unsigned int free, total;
		cuMemGetInfo(&free,&total);
		free /=1<<20;
		total/=1<<20;
		char message[64];
		fps = float(frameCount)/(currentTime-lastTime);
		sprintf( message, "%u/%u mb, %.2f fps", total-free, total, fps );	
		glutSetWindowTitle(message);
		frameCount=0;
		lastTime=Time();
	}
}

template< class Real , bool Primal >
void DisplayFunction( void )
{
	if( eventList.size() )
	{
		Event& e = eventList[0];
		switch( e.eventType )
		{
		case Event::EXIT_EVENT:	exit( 0 ) ; break;
		case Event::SOLVE_EVENT: KeyboardFunction< Real , Primal >( '+' , 0 , 0 ) ; break;
		case Event::WRITE_EVENT: OutputEditedPLY< Real , Primal >( e.fileName ) ; printf( "Output mesh: %s\n" , e.fileName ) ; break;
		default: fprintf( stderr , "Unknown event type: %d\n" , e.eventType );
		}
		for( int i=1 ; i<eventList.size() ; i++ ) eventList[i-1] = eventList[i];
		eventList.pop_back( );
	}
	else if( !pauseSolver ) KeyboardFunction< Real , Primal >( '+' , 0 , 0 );
	updateVBO< Real , Primal >( );
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
	myStage.drawOpenGL();
	drawTimer();
	glutSwapBuffers();
}
template< class Real , bool Primal >
void IdleFunction( void )
{
	DisplayFunction< Real , Primal >( );
}
void MouseFunction( int button, int state, int x, int y )
{
	mouse.update( button, state, x, window_height-y );
	glutPostRedisplay();
}
void MotionFunction( int x, int y )
{
	mouse.move( x , window_height-y );
	myStage.MotionFunction( mouse );
	glutPostRedisplay();
}
void PassiveMotionFunction( int x, int y )
{
	mouse.move( x , window_height-y );
	glutPostRedisplay();
}
template< class Real , bool Primal >
void KeyboardFunction( unsigned char c, int x, int y )
{
	switch( c )
	{
		case '\033':
		{
			Sleep(1000);
			exit( 0 );
		}
		break;
		case 'O':
		{
			char filename[256] , temp[256];
			printf( "File Name: " );
			gets( filename );
			OutputEditedPLY< Real , Primal >( filename );
			pauseSolver = true;
		}
		break;
		case ' ':
			pauseSolver = !pauseSolver;
			break;
		case '+':
		{
			int iteration = 0;
			double iterationTime = Time();
			int                      flowType = MeshOctree< Real , Primal >::TrackingData< FastPoint3D< Real > >::FLOW_LAPLACIAN;
			if     ( Conformal.set ) flowType = MeshOctree< Real , Primal >::TrackingData< FastPoint3D< Real > >::FLOW_CONFORMAL_LAPLACIAN;
			else if(  Authalic.set ) flowType = MeshOctree< Real , Primal >::TrackingData< FastPoint3D< Real > >::FLOW_AUTHALIC_LAPLACIAN;

			double setTime = Time();
			// Convert the solution to an element based solution
			Tree< Real , Primal >::solver.e2b.back().MultiplyTranspose( Tree< Real , Primal >::solver.x() , Tree< Real , Primal >::X , Threads.value , false );

			// Get the updated system
			Tree< Real , Primal >::trackingData.setConstrainedLaplacianAndVector( Tree< Real , Primal >::X , Tree< Real , Primal >::X , Tree< Real , Primal >::solver.eM() , Tree< Real , Primal >::B , 1./StepSize.value , 1. , 1./StepSize.value , 0. , flowType , Threads.value );
			Tree< Real , Primal >::solver.ResetMatrix( Threads.value );
			Tree< Real , Primal >::solver.e2b.back().Multiply( Tree< Real , Primal >::B , Tree< Real , Primal >::solver.b() , Threads.value , false );
			setTime = Time() - setTime;

			double normX = 0 , normB = 0 , normR = 0 , _normB = 0;
			 
			if( Clear.set ) for( int i=0 ; i<Tree< Real , Primal >::solver.x().Dimensions() ; i++ ) Tree< Real , Primal >::solver.x()[i] *= 0;

			if( ShowError.set )
			{
				Vector< FastPoint3D< Real > > temp = Tree< Real , Primal >::solver * Tree< Real , Primal >::solver.x();
				for( int i=0 ; i<Tree< Real , Primal >::solver.b().Dimensions() ; i++ ) normX += FastPoint3D< Real >::SquareNorm( Tree< Real , Primal >::solver.x()[i] );
				for( int i=0 ; i<Tree< Real , Primal >::solver.b().Dimensions() ; i++ )
				{
					normB  += FastPoint3D< Real >::SquareNorm( Tree< Real , Primal >::solver.b()[i] );
					_normB += FastPoint3D< Real >::SquareNorm( Tree< Real , Primal >::solver.b()[i]-temp[i] );
				}
			}
			double solveTime = Time();
			Tree< Real , Primal >::solver.Solve( true );
			solveTime = Time() - solveTime;
			if( ShowError.set )
			{
				Vector< FastPoint3D< Real > > temp = Tree< Real , Primal >::solver * Tree< Real , Primal >::solver.x();
				for( int i=0 ; i<Tree< Real , Primal >::solver.b().Dimensions() ; i++ ) normR += FastPoint3D< Real >::SquareNorm( Tree< Real , Primal >::solver.b()[i]-temp[i] );
				printf( "Error: (%g )\t%g -> %g -> %g ( %g = %g * %g )\n" , sqrt( normX ) , sqrt( normB ) , sqrt( _normB ) , sqrt( normR ) , sqrt( normR/normB ) , sqrt( _normB/normB ) , sqrt( normR/_normB) );
				printf( "%d] set / solve (sec): %.3f = %.3f + %.3f\n", ++iteration , Time()-iterationTime , setTime , solveTime );
			}
			else
				printf( "%4d] set / solve (sec): %.3f = %.3f + %.3f             \r" , ++iteration , Time()-iterationTime , setTime , solveTime );
		}
		break;
		default: myStage.KeyboardFunction( c );
	}
}
template< class Real , bool Primal >
void SpecialFunction( int key, int x, int y )
{
	switch( key )
	{
	case GLUT_KEY_F12:
		{
			char str[512];
			printf( "Steps: " );
			gets( str );
			Event e;
			int s = atoi( str );
			if( s>=0 ) 
			{
				e.eventType = Event::SOLVE_EVENT;
				for( int s=0 ; s<atoi( str ) ; s++ ) eventList.push_back( e );
				pauseSolver = true;
			}
			else pauseSolver = false;
		}
		break;
	default: myStage.SpecialFunction( key );
	}
	glutPostRedisplay();
}
void ReshapeFunction( int width , int height )
{
	window_width = width , window_height = height;
	myStage.Reshape( width , height );
	glutPostRedisplay( );
}


template< class Real , bool Primal >
int Visualize(int argc,char* argv[])
{
	if( Out.set )
	{
		int passes = atoi( Out.values[1] );
		int steps  = atoi( Out.values[2] );
		Event e;
		for( int i=0 ; i<=passes ; i++ )
		{
			if( !(i%steps) )
			{
				e.eventType = Event::WRITE_EVENT;
				sprintf( e.fileName , "%s.%d.ply" , Out.values[0] , i );
				eventList.push_back( e );
			}
			e.eventType = Event::SOLVE_EVENT;
			eventList.push_back( e );
		}
		e.eventType = Event::EXIT_EVENT;
		eventList.push_back( e );
	}

	/////////////////////////
	// GLUT Initialization //
	/////////////////////////
	glutInit(&argc,argv);	
	glutInitDisplayMode( GLUT_RGBA | GLUT_DOUBLE | GLUT_DEPTH);
    glutInitWindowSize( window_width , window_height );
	glutCreateWindow( "Mean-Curvature Flow" ); 
	glClearColor( clearColor[0] , clearColor[1] , clearColor[2] , 1.0);

	glutDisplayFunc			( DisplayFunction< Real , Primal >  );
	glutIdleFunc			( IdleFunction   < Real , Primal >  );
	glutMouseFunc			( MouseFunction                     );
	glutMotionFunc			( MotionFunction                    );
	glutPassiveMotionFunc	( PassiveMotionFunction             );
	glutKeyboardFunc		( KeyboardFunction< Real , Primal > );
	glutSpecialFunc			( SpecialFunction< Real , Primal >  );
	glutReshapeFunc			( ReshapeFunction                   );
	 

	glEnable(GL_DEPTH_TEST);
	glEnable(GL_NORMALIZE);
	glDepthMask(GL_TRUE);
	glDisable(GL_BLEND);
	glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
	glCullFace( GL_BACK );
	glEnable(GL_CULL_FACE);

	/////////////////////////
	// Camera and Lighting //
	/////////////////////////
	myStage.center[0] = myStage.center[1] = myStage.center[2] = 0.5;
	myStage.radius = 1.f;
	myStage.Reshape( window_width , window_height );

	GLfloat lightPosition[] = { 1.0f, 100.0f, 100.0f, 1.0 };
	GLfloat LightColor[] = { 0.8f, 0.8f, 0.8f, 1.0 };
	GLfloat White[] = { 1.0f, 1.0f, 1.0f, 1.0 }; 
	GLfloat Gray[]  = { 0.8f, 0.8f, 0.8f, 1.0 };
	glLightfv(GL_LIGHT0, GL_POSITION, lightPosition);
	glLightfv(GL_LIGHT0, GL_DIFFUSE,  LightColor);
	glLightfv(GL_LIGHT0, GL_SPECULAR, LightColor);
	glColorMaterial(GL_FRONT, GL_DIFFUSE);
	glMaterialfv(GL_FRONT, GL_SPECULAR, White);
	glMaterialfv(GL_FRONT, GL_DIFFUSE,  White);
	glMateriali (GL_FRONT, GL_SHININESS,100.0);
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glEnable(GL_COLOR_MATERIAL);

	//////////////////////////
	// Vertex Object Buffer //
	//////////////////////////
	glewInit();
	if( !glewIsSupported("GL_VERSION_2_0 ") ) printf("ERROR: Support for necessary OpenGL extensions missing.");
	cudaGLSetGLDevice( cutGetMaxGflopsDeviceId() );
	cutilCheckError( cutCreateTimer( &timer ) );

	myStage.setTriangleBufferObject( );
	cutilSafeCall( cudaGLRegisterBufferObject( myStage.tbo ) );
	myStage.setVertexBufferObject( );
	cutilSafeCall( cudaGLRegisterBufferObject( myStage.vbo ) );

	int memSize;

	memSize = Tree< Real , Primal >::subdividedVertices.size() * sizeof(float) * Tree< Real , Primal >::supportCount;
	float* eValues = new float[ memSize ];
	for( int i=0 ; i<Tree< Real , Primal >::E.groups ; i++ )
		for( int j=0 ; j<Tree< Real , Primal >::supportCount ; j++ )
			eValues[ i*Tree< Real , Primal >::supportCount + j ] = Tree< Real , Primal >::E[i][j].Value;
	cutilSafeCall( cudaMalloc((void**) &gpu_evaluationValues, memSize) );
    cutilSafeCall( cudaMemcpy(gpu_evaluationValues, eValues, memSize, cudaMemcpyHostToDevice) );
	delete[] eValues;

	memSize = Tree< Real , Primal >::E.groups * Tree< Real , Primal >::supportCount * sizeof( unsigned int );
	unsigned int* eIndicesTemp = (unsigned int*) malloc( memSize );
	// A void basis has index "-1", but since its contribution is "0.0", we don't really care.
	unsigned int LastNonZeroIndex=0;
	for( int i=0 ; i<Tree< Real , Primal >::E.groups ; i++ )
		for( int j=0 ; j<Tree< Real , Primal >::supportCount ; j++ )
			eIndicesTemp[ i*Tree< Real , Primal >::supportCount+j ] = LastNonZeroIndex = unsigned int( Tree< Real , Primal >::E[i][j].N );
	cutilSafeCall( cudaMalloc((void**) &gpu_evaluationIndices, memSize) );
    cutilSafeCall( cudaMemcpy(gpu_evaluationIndices, eIndicesTemp, memSize, cudaMemcpyHostToDevice) );
	delete eIndicesTemp;

	coefficientMemSize = sizeof(float)*4*Tree< Real , Primal >::solver.pX().size( );
	cudaMallocHost( (void**)&host_coefficientValues , coefficientMemSize );
	cutilSafeCall( cudaMalloc((void**) &gpu_coefficientValues , coefficientMemSize) );

	setShaders();
 	glutMainLoop();
    cudaThreadExit();
    cutilExit( argc , argv );
	return 1;
}